/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *                   Copyright (C) 2000-2002 by OpenSG Forum                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of th4e GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

#include <sys/types.h>

#include <OSGBaseFunctions.h>

OSG_BEGIN_NAMESPACE

inline
BinaryDataHandler::ReadError::ReadError(const Char8 *reason) : 
    Exception() 
{
    _what += "BinaryDataHandler ReadError: ";
    _what += reason;
}

inline
BinaryDataHandler::WriteError::WriteError(const Char8 *reason) : 
    Exception() 
{
    _what += "BinaryDataHandler WriteError: ";
    _what += reason;
}

inline
void BinaryDataHandler::putValueSize(const SizeT &value)
{
    if(osgUseOSB64())
    {
        UInt64 val64 = value;
        putValue(val64);
    }
    else
    {
        UInt32 val32 = (UInt32)value;
        putValue(val32);
    }
}

template<typename T>
inline void BinaryDataHandler::put(const T &data)
{
    // nice speed improvement, this method is inlined
    // and sizeof(T) is known at compile time so
    // the memcpy is replaced with some optiomal code by the c++ compiler.
    if(!_noBuffer)
    {
        if(_currentWriteBuffer == writeBufEnd())
            pushBuffer();

        if((_currentWriteBuffer->getSize() - _currentWriteBufferPos) >= sizeof(T))
        {
            memcpy(_currentWriteBuffer->getMem() + _currentWriteBufferPos, &data, sizeof(T));
            //*reinterpret_cast<T *>(_currentWriteBuffer->getMem() + _currentWriteBufferPos) = data;
            _currentWriteBufferPos += sizeof(T);
            return;
        }
    }
    put(&data, sizeof(T));
}

inline 
void BinaryDataHandler::putValue(const bool &value)
{
    // on Mac OS X a bool is four bytes long on all other
    // platfroms it is one byte long. So we write now always
    // one byte out.
    // put(&value, sizeof(bool));
    UInt8 temp = UInt8(value);
    put(temp);
}

inline 
void BinaryDataHandler::putValue(const UInt8 &value)
{
    put(value);
}

inline 
void BinaryDataHandler::putValue(const UInt16 &value)
{
    UInt16 z = osghtons(value);
    put(z);
}

inline 
void BinaryDataHandler::putValue(const UInt32 &value)
{
    UInt32 z = osghtonl(value);
    put(z);
}

inline 
void BinaryDataHandler::putValue(const UInt64 &value)
{
    UInt64 z = osghtonll(value);
    put(z);
}

inline 
void BinaryDataHandler::putValue(const UInt128 &value)
{
    UInt128 z = osghtonllll(value);
    put(z);
}

inline 
void BinaryDataHandler::putValue(const Int8 &value)
{
    put(value);
}

inline 
void BinaryDataHandler::putValue(const Int16 &value)
{
    Int16 z = osghtons(value);
    put(z);
}

inline 
void BinaryDataHandler::putValue(const Int32 &value)
{
    Int32 z = osghtonl(value);
    put(z);
}

inline 
void BinaryDataHandler::putValue(const Int64 &value)
{
    Int64 z = osghtonll(value);
    put(z);
}

inline 
void BinaryDataHandler::putValue(const Int128 &value)
{
    Int128 z = osghtonllll(value);
    put(z);
}

inline 
void BinaryDataHandler::putValue(const Real16 &value)
{
    UInt16 v = osghtons(value.bits());
    put(v);
}

inline 
void BinaryDataHandler::putValue(const Real32 &value)
{
    UInt32 v = osghtonl( *(reinterpret_cast<const UInt32 *>(&value)) );
    put(v);
}

inline 
void BinaryDataHandler::putValue(const Real64 &value)
{
    UInt64 v = osghtonll( *(reinterpret_cast<const UInt64 *>(&value)) );
    put(v);
}

inline 
void BinaryDataHandler::putValue(const Real128 &value)
{
    UInt64 v = osghtonll( *( reinterpret_cast<const UInt64 *>(&value)) );
    UInt64 w = osghtonll( *((reinterpret_cast<const UInt64 *>(&value)) + 1) );

#if BYTE_ORDER == LITTLE_ENDIAN
    put(w);
    put(v);
#else
    put(v);
    put(w);
#endif
}

inline 
void BinaryDataHandler::putValue(const std::string &value)
{
    const UInt32 len = value.length() + 1;

    putValue(len);

    if(len == 0)
        return;

    if(!_noBuffer)
    {
        if(_currentWriteBuffer == writeBufEnd())
            pushBuffer();
        if((_currentWriteBuffer->getSize() - _currentWriteBufferPos) >= len)
        {
            memcpy(_currentWriteBuffer->getMem() + _currentWriteBufferPos, value.c_str(), len);
            _currentWriteBufferPos += len;
            return;
        }
    }

    put(value.c_str(), len);
}


inline 
void BinaryDataHandler::putValues(const bool *value, UInt64 size)
{
    for(UInt64 i = 0; i < size; ++i)
        putValue(value[i]);
}

inline 
void BinaryDataHandler::putValues(const UInt8 *value, UInt64 size)
{
    put(value, size * sizeof(UInt8));
}

inline 
void BinaryDataHandler::putValues(const UInt16 *value, UInt64 size)
{
#if BYTE_ORDER == LITTLE_ENDIAN

    if(_networkOrder == true)
    {
        for(UInt64 i = 0; i < size; ++i)
        {
            putValue(value[i]);
        }
    }
    else
#endif
    {
        put(value, size * sizeof(UInt16));
    }
}

inline 
void BinaryDataHandler::putValues(const UInt32 *value, UInt64 size)
{
#if BYTE_ORDER == LITTLE_ENDIAN
    if(_networkOrder == true)
    {
        for(UInt64 i = 0; i < size; ++i)
        {
            putValue(value[i]);
        }
    }
    else
#endif
    {
        put(value, size * sizeof(UInt32));
    }
}

inline 
void BinaryDataHandler::putValues(const UInt64 *value, UInt64 size)
{
#if BYTE_ORDER == LITTLE_ENDIAN
    if(_networkOrder == true)
    {
        for(UInt64 i = 0; i < size; ++i)
        {
            putValue(value[i]);
        }
    }
    else
#endif
    {
        put(value, size * sizeof(UInt64));
    }
}

inline 
void BinaryDataHandler::putValues(const UInt128 *value, UInt64 size)
{
#if BYTE_ORDER == LITTLE_ENDIAN
    if(_networkOrder == true)
    {
        for(UInt64 i = 0; i < size; ++i)
        {
            putValue(value[i]);
        }
    }
    else
#endif
    {
        put(value, size * sizeof(UInt128));
    }
}

inline 
void BinaryDataHandler::putValues(const Int8 *value, UInt64 size)
{
    put(value, size * sizeof(Int8));
}

inline 
void BinaryDataHandler::putValues(const Int16 *value, UInt64 size)
{
#if BYTE_ORDER == LITTLE_ENDIAN
    if(_networkOrder == true)
    {
        for(UInt64 i = 0; i < size; ++i)
        {
            putValue(value[i]);
        }
    }
    else
#endif
    {
        put(value, size * sizeof(Int16));
    }
}

inline 
void BinaryDataHandler::putValues(const Int32 *value, UInt64 size)
{
#if BYTE_ORDER == LITTLE_ENDIAN
    if(_networkOrder == true)
    {
        for(UInt64 i = 0; i < size; ++i)
        {
            putValue(value[i]);
        }
    }
    else
#endif
    {
        put(value, size * sizeof(Int32));
    }
}

inline 
void BinaryDataHandler::putValues(const Int64 *value, UInt64 size)
{
#if BYTE_ORDER == LITTLE_ENDIAN
    if(_networkOrder == true)
    {
        for(UInt64 i = 0; i < size; ++i)
        {
            putValue(value[i]);
        }
    }
    else
#endif
    {
        put(value, size * sizeof(Int64));
    }
}

inline 
void BinaryDataHandler::putValues(const Int128 *value, UInt64 size)
{
#if BYTE_ORDER == LITTLE_ENDIAN
    if(_networkOrder == true)
    {
        for(UInt64 i = 0; i < size; ++i)
        {
            putValue(value[i]);
        }
    }
    else
#endif
    {
        put(value, size * sizeof(Int128));
    }
}

inline 
void BinaryDataHandler::putValues(const Real16 *value, UInt64 size)
{
#if BYTE_ORDER == LITTLE_ENDIAN
    if(_networkOrder == true)
    {
        for(UInt64 i = 0; i < size; ++i)
        {
            putValue(value[i]);
        }
    }
    else
#endif
    {
        put(value, size * sizeof(Real16));
    }
}

inline 
void BinaryDataHandler::putValues(const Real32 *value, UInt64 size)
{
#if BYTE_ORDER == LITTLE_ENDIAN
    if(_networkOrder == true)
    {
        for(UInt64 i = 0; i < size; ++i)
        {
            putValue(value[i]);
        }
    }
    else
#endif
    {
        put(value, size * sizeof(Real32));
    }
}

inline 
void BinaryDataHandler::putValues(const Real64 *value, UInt64 size)
{
#if BYTE_ORDER == LITTLE_ENDIAN
    if(_networkOrder == true)
    {
        for(UInt64 i = 0; i < size; ++i)
        {
            putValue(value[i]);
        }
    }
    else
#endif
    {
        put(value, size * sizeof(Real64));
    }
}

inline 
void BinaryDataHandler::putValues(const Real128 *value, UInt64 size)
{
#if BYTE_ORDER == LITTLE_ENDIAN
    if(_networkOrder == true)
    {
        for(UInt64 i = 0; i < size; ++i)
        {
            putValue(value[i]);
        }
    }
    else
#endif
    {
        put(value, size * sizeof(Real128));
    }
}

inline 
void BinaryDataHandler::putValues(const std::string *value, UInt64 size)
{
    for(UInt64 i = 0; i<size; ++i)
    {
        putValue(value[i]);    
    }
}

inline
void BinaryDataHandler::getValueSize(SizeT &value)
{
    if(osgUseOSB64())
    {
        UInt64 val;
        getValue(val);
        value = val;
    }
    else
    {
        UInt32 val;
        getValue(val);
        value = val;
    }
}

template<typename T>
inline void BinaryDataHandler::get(T &data)
{
    // nice speed improvement, this method is inlined
    // and sizeof(T) is known at compile time so
    // the memcpy is replaced with some optiomal code by the c++ compiler.
    if(!_noBuffer)
    {
        if(_currentReadBuffer == readBufEnd())
            pullBuffer();

        if((_currentReadBuffer->getDataSize() - _currentReadBufferPos) >= sizeof(T))
        {
            memcpy(&data, _currentReadBuffer->getMem() + _currentReadBufferPos, sizeof(T));
            _currentReadBufferPos += sizeof(T);
            return;
        }
    }
    get(&data, sizeof(T));
}

inline
void BinaryDataHandler::getValue(bool &value)
{
    //get(&value, sizeof(bool));
    UInt8 temp;
    get(temp);
    value = (temp!=0);
}

inline 
void BinaryDataHandler::getValue(UInt8 &value)
{
    get(value);
}

inline 
void BinaryDataHandler::getValue(UInt16 &value)
{
    get(value);

    value = osgntohs(value);
}

inline 
void BinaryDataHandler::getValue(UInt32 &value)
{
    get(value);

    value = osgntohl(value);
}

inline 
void BinaryDataHandler::getValue(UInt64 &value)
{
    get(value);

    value = osgntohll(value);
}

inline 
void BinaryDataHandler::getValue(UInt128 &value)
{
    get(value);

    value = osgntohllll(value);
}

inline 
void BinaryDataHandler::getValue(Int8 &value)
{
    get(value);
}

inline 
void BinaryDataHandler::getValue(Int16 &value)
{
    get(value);

    value = osgntohs(value);
}

inline 
void BinaryDataHandler::getValue(Int32 &value)
{
    get(value);

    value = osgntohl(value);
}

inline 
void BinaryDataHandler::getValue(Int64 &value)
{
    get(value);

    value = osgntohll(value);
}

inline 
void BinaryDataHandler::getValue(Int128 &value)
{
    get(value);

    value = osgntohllll(value);
}

inline 
void BinaryDataHandler::getValue(Real16 &value)
{
    UInt16 v;

    get(v);

    v     = osgntohs(v);
    value.setBits(v);
}

inline 
void BinaryDataHandler::getValue(Real32 &value)
{
    get(value);

    value = osgntohf(value);
}

inline 
void BinaryDataHandler::getValue(Real64 &value)
{
    get(value);

    value = osgntohd(value);
}

inline 
void BinaryDataHandler::getValue(Real128 &value)
{
    get(value);

    value = osgntohdd(value);
}

inline 
void BinaryDataHandler::getValue(std::string &value)
{
    UInt32  len;

    getValue(len);

    if(len != 0)
    {
        value.resize(len);
        get(&value[0], len);
        // remove terminating 0
        value.resize(len - 1);
    }
    else
    {
        value.erase();
    }
}

inline 
void BinaryDataHandler::getValues(bool *value, UInt64 size)
{
    for(UInt64 i = 0; i < size; ++i)
        getValue(value[i]);
}

inline 
void BinaryDataHandler::getValues(UInt8 *value, UInt64 size)
{
    get(value, size * sizeof(UInt8));
}

inline 
void BinaryDataHandler::getValues(UInt16 *value, UInt64 size)
{
    get(value, size * sizeof(UInt16));

#if BYTE_ORDER == LITTLE_ENDIAN
    if(_networkOrder == true)
    {
        for(UInt64 i = 0; i < size; ++i)
        {
            value[i] = osgntohs(value[i]);
        }
    }
#endif
}

inline 
void BinaryDataHandler::getValues(UInt32 *value, UInt64 size)
{
    get(value, size * sizeof(UInt32));

#if BYTE_ORDER == LITTLE_ENDIAN
    if(_networkOrder == true)
    {
        for(UInt64 i = 0; i < size; ++i)
        {
            value[i] = osgntohl(value[i]);
        }
    }
#endif
}

inline 
void BinaryDataHandler::getValues(UInt64 *value, UInt64 size)
{
    get(value, size * sizeof(UInt64));

#if BYTE_ORDER == LITTLE_ENDIAN
    if(_networkOrder == true)
    {
        for(UInt64 i = 0; i < size; ++i)
        {
            value[i] = osgntohll(value[i]);
        }
    }
#endif
}

inline 
void BinaryDataHandler::getValues(UInt128 *value, UInt64 size)
{
    get(value, size * sizeof(UInt128));

#if BYTE_ORDER == LITTLE_ENDIAN
    if(_networkOrder == true)
    {
        for(UInt64 i = 0; i < size; ++i)
        {
            value[i] = osgntohllll(value[i]);
        }
    }
#endif
}

inline 
void BinaryDataHandler::getValues(Int8 *value, UInt64 size)
{
    get(value, size * sizeof(Int8));
}

inline 
void BinaryDataHandler::getValues(Int16 *value, UInt64 size)
{
    get(value, size * sizeof(Int16));

#if BYTE_ORDER == LITTLE_ENDIAN
    if(_networkOrder == true)
    {
        for(UInt64 i = 0; i < size; ++i)
        {
            value[i] = osgntohs(value[i]);
        }
    }
#endif
}

inline 
void BinaryDataHandler::getValues(Int32 *value, UInt64 size)
{
    get(value, size * sizeof(Int32));

#if BYTE_ORDER == LITTLE_ENDIAN
    if(_networkOrder == true)
    {
        for(UInt64 i = 0; i < size; ++i)
        {
            value[i] = osgntohl(value[i]);
        }
    }
#endif
}

inline 
void BinaryDataHandler::getValues(Int64 *value, UInt64 size)
{
    get(value, size * sizeof(Int64));

#if BYTE_ORDER == LITTLE_ENDIAN
    if(_networkOrder == true)
    {
        for(UInt64 i = 0; i < size; ++i)
        {
            value[i] = osgntohll(value[i]);
        }
    }
#endif
}

inline 
void BinaryDataHandler::getValues(Int128 *value, UInt64 size)
{
    get(value, size * sizeof(Int128));

#if BYTE_ORDER == LITTLE_ENDIAN
    if(_networkOrder == true)
    {
        for(UInt64 i = 0; i < size; ++i)
        {
            value[i] = osgntohllll(value[i]);
        }
    }
#endif
}

inline 
void BinaryDataHandler::getValues(Real16 *value, UInt64 size)
{
    get(value, size * sizeof(Real16));

#if BYTE_ORDER == LITTLE_ENDIAN
    if(_networkOrder == true)
    {
        UInt16 *intValue = reinterpret_cast<UInt16 *>(value);

        for(UInt64 i = 0; i < size; ++i)
        {
            value[i].setBits(osgntohs(intValue[i]));
        }
    }
#endif
}

inline 
void BinaryDataHandler::getValues(Real32 *value, UInt64 size)
{
    get(value, size * sizeof(Real32));

#if BYTE_ORDER == LITTLE_ENDIAN
    if(_networkOrder == true)
    {
        UInt32 *intValue = reinterpret_cast<UInt32 *>(value);

        for(UInt64 i = 0; i < size; ++i)
        {
            intValue[i] = osgntohl(intValue[i]);
        }
    }
#endif
}

inline 
void BinaryDataHandler::getValues(Real64 *value, UInt64 size)
{
    get(value, size * sizeof(Real64));

#if BYTE_ORDER == LITTLE_ENDIAN
    if(_networkOrder == true)
    {
        UInt64 *longValue = reinterpret_cast<UInt64 *>(value);

        for(UInt64 i = 0; i < size; ++i)
        {
            longValue[i] = osgntohll(longValue[i]);
        }
    }
#endif
}

inline 
void BinaryDataHandler::getValues(Real128 *value, UInt64 size)
{
    get(value, size * sizeof(UInt64) * 2);

#if BYTE_ORDER == LITTLE_ENDIAN
    if(_networkOrder == true)
    {
        UInt64 *longValue = reinterpret_cast<UInt64 *>(value);

        for(UInt64 i = 0; i < size; i += 2)
        {
            UInt64 l = longValue[i];
            longValue[i]     = osgntohll(longValue[i + 1]);
            longValue[i + 1] = osgntohll(longValue[l    ]);
        }
    }
#endif
}

inline 
void BinaryDataHandler::getValues(std::string *value, UInt64 size)
{
    for(UInt64 i = 0; i < size; ++i)
    {
        getValue(value[i]);
    }
}

inline
BinaryDataHandler::MemoryBlock::MemoryBlock(MemoryHandle m,
                                            SizeT        s,
                                            SizeT        ds) :
    _mem     (m ),
    _size    (s ),
    _dataSize(ds)
{
}

inline
MemoryHandle BinaryDataHandler::MemoryBlock::getMem(void)
{
    return _mem;
}

inline
void BinaryDataHandler::MemoryBlock::setMem(MemoryHandle mem)
{
    _mem = mem;
}

inline
SizeT BinaryDataHandler::MemoryBlock::getSize(void)
{
    return _size;
}

inline
void BinaryDataHandler::MemoryBlock::setSize(SizeT size)
{
    _size = size;
}

inline
SizeT BinaryDataHandler::MemoryBlock::getDataSize(void)
{
    return _dataSize;
}

inline
void BinaryDataHandler::MemoryBlock::setDataSize(SizeT dataSize)
{
    _dataSize = dataSize;
}

inline
BinaryDataHandler::BuffersT::iterator BinaryDataHandler::readBufBegin(void)
{
    return _readBuffers.begin();
}

inline
BinaryDataHandler::BuffersT::iterator BinaryDataHandler::readBufEnd(void)
{
    return _readBuffers.end();
}

inline
BinaryDataHandler::BuffersT::iterator BinaryDataHandler::writeBufBegin(void)
{
    return _writeBuffers.begin();
}

inline
BinaryDataHandler::BuffersT::iterator BinaryDataHandler::writeBufEnd(void)
{
    return _writeBuffers.end();
}

OSG_END_NAMESPACE

#define OSGBINARYDATAHANDLER_INLINE_CVSID "@(#)$Id: $"

