#ifndef _OSGBITVECTOR128_H_
#define _OSGBITVECTOR128_H_

#include <cstdint>
#include <stdexcept>
#include <iostream>
#include <string>

#include <OSGConfig.h>

OSG_BEGIN_NAMESPACE

class OSG_BASE_DLLMAPPING BitVector128
{
public:

    constexpr BitVector128() :
        _lo(0),
        _hi(0)
    {
    }

    constexpr BitVector128(uint64_t value) :
        _lo(value),
        _hi(0)
    {
    }

    constexpr explicit BitVector128(uint64_t hi, uint64_t lo) :
        _lo(lo),
        _hi(hi)
    {
    }

    constexpr BitVector128(const BitVector128 &value) :
        _lo(value._lo),
        _hi(value._hi)
    {
    }

    constexpr bool isOne() const
    {
        return (_hi == 0 && _lo == 1);
    }

    bool operator==(const BitVector128 &o) const
    {
        return (_hi == o._hi) && (_lo == o._lo);
    }

    bool operator!=(const BitVector128 &o) const
    {
        return (_hi != o._hi) || (_lo != o._lo);
    }

    bool operator&(const BitVector128 &o) const
    {
        return (_lo & o._lo) || (_hi & o._hi);
    }

    BitVector128 &operator&=(const BitVector128 &b)
    {
        _hi &= b._hi;
        _lo &= b._lo;
        return *this;
    }

    BitVector128 &operator|=(const BitVector128 &b)
    {
        _hi |= b._hi;
        _lo |= b._lo;
        return *this;
    }

    inline std::string toString() const
    {
        std::string r;
        r.reserve(128);

        uint64_t s = 1;
        for(int32_t i=63;i>=0;--i)
            r += (((_hi & (s << i)) == 0) ? "0" : "1");
        s = 1;
        for(int32_t i=63;i>=0;--i)
            r += (((_lo & (s << i)) == 0) ? "0" : "1");
        return r;
    }

    inline uint64_t getLow() const
    {
        return _lo;
    }

    inline uint64_t getHigh() const
    {
        return _hi;
    }

private:

    uint64_t _lo;
    uint64_t _hi;
};

// this operator works only with BitVector128 value 1 so "BitVector128(1) << 12" is ok but "BitVector128(2) << 12" is not allowed!
constexpr BitVector128 operator<<(BitVector128 const &a, uint32_t n)
{
    return a.isOne() ? (n < 64) ? BitVector128(uint64_t(0), uint64_t(1) << n) : BitVector128(uint64_t(1) << (n - 64), uint64_t(0)) : throw std::logic_error("BitVector128 operator<< works only for BitVector128(1)!");
}

inline std::ostream &operator<<(std::ostream &o, const BitVector128 &v)
{
    o << v.toString();
    return o;
}

OSG_END_NAMESPACE

#endif
