/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright (C) 2000-2002 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/


#include <stdio.h>
#include <stdlib.h>
#include "OSGCRC32.h"

OSG_USING_NAMESPACE

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

CRC32::CRC32(void) :
    _ulTable()
{
    init();
}


///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

CRC32::~CRC32(void)
{
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
    This function initializes "CRC Lookup Table". You only need to call it once to
        initalize the table before using any of the other CRC32 calculation functions.
*/

void CRC32::init(void)
{
    //0x04C11DB7 is the official polynomial used by PKZip, WinZip and Ethernet.
    UInt32 ulPolynomial = 0x04C11DB7;

    //memset(&_ulTable, 0, sizeof(_ulTable));

    // 256 values representing ASCII character codes.
    for(int iCodes = 0; iCodes <= 0xFF; iCodes++)
    {
        _ulTable[iCodes] = reflect(iCodes, 8) << 24;

        for(int iPos = 0; iPos < 8; iPos++)
        {
            _ulTable[iCodes] = (_ulTable[iCodes] << 1) ^ ((_ulTable[iCodes] & (1 << 31)) ? ulPolynomial : 0);
        }

        _ulTable[iCodes] = reflect(_ulTable[iCodes], 32);
    }
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
    Reflection is a requirement for the official CRC-32 standard.
    You can create CRCs without it, but they won't conform to the standard.
*/

UInt32 CRC32::reflect(UInt32 ulReflect, const char cChar)
{
    UInt32 ulValue = 0;

    // Swap bit 0 for bit 7, bit 1 For bit 6, etc....
    for(int iPos = 1; iPos < (cChar + 1); iPos++)
    {
        if(ulReflect & 1)
        {
            ulValue |= (1 << (cChar - iPos));
        }
        ulReflect >>= 1;
    }

    return ulValue;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
    Calculates the CRC32 by looping through each of the bytes in sData.

    Note: For Example usage example, see FileCRC().
*/

void CRC32::partialCRC(UInt32 *ulCRC, const UInt8 *sData, UInt32 ulDataLength)
{
    while(ulDataLength--)
    {
        //If your compiler complains about the following line, try changing each
        //    occurrence of *ulCRC with "((UInt32)*ulCRC)" or "*(UInt32 *)ulCRC".

         *(UInt32 *)ulCRC =
            ((*(UInt32 *)ulCRC) >> 8) ^ _ulTable[((*(UInt32 *)ulCRC) & 0xFF) ^ *sData++];
    }
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
    Returns the calculated CRC32 (through ulOutCRC) for the given string.
*/
void CRC32::fullCRC(const UInt8 *sData, UInt32 ulDataLength, UInt32 *ulOutCRC)
{
    *(UInt32 *)ulOutCRC = 0xffffffff; //Initilaize the CRC.
    partialCRC(ulOutCRC, sData, ulDataLength);
    *(UInt32 *)ulOutCRC ^= 0xffffffff; //Finalize the CRC.
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
    Returns the calculated CRC23 for the given string.
*/
UInt32 CRC32::fullCRC(const UInt8 *sData, UInt32 ulDataLength)
{
    UInt32 ulCRC = 0xffffffff; //Initilaize the CRC.
    partialCRC(&ulCRC, sData, ulDataLength);
    return(ulCRC ^ 0xffffffff); //Finalize the CRC and return.
}
