
#ifndef OSG_FAST_MAP_H
#define OSG_FAST_MAP_H

namespace osg
{

template <typename Mapped>
class FastMap
{
public:

    class const_iterator
    {
    public:
        const_iterator(const std::vector<std::pair<std::vector<Mapped> *, unsigned int> > *buckets,
                 unsigned int bi, unsigned int i) :
            first((bi << 8) | i),
            second(),
            _buckets(buckets),
            _bi(bi),
            _i(i)
        {
            if(i != 0xffffffff)
                second = _buckets->at(bi).first->at(i);
        }

        const_iterator(const const_iterator &other) :
            first(other.first),
            second(other.second),
            _buckets(other._buckets),
            _bi(other._bi),
            _i(other._i)
        {
        }

        const_iterator& operator=(const const_iterator &other)
        {
            first = other.first;
            second = other.second;
            _buckets = other._buckets;
            _bi = other._bi;
            _i = other._i;
            return *this;
        }

        const_iterator &operator++()
        {
            ++_i;
            if(_i > 0xff)
            {
                _i = 0;
                ++_bi;
            }

            if(_bi >= _buckets->size())
            {
                _bi = 0;
                _i = 0xffffffff;
                second = Mapped();
                return *this;
            }
            if(_i >= _buckets->at(_bi).first->size())
            {
                _bi = 0;
                _i = 0xffffffff;
                second = Mapped();
                return *this;
            }

            if(_i != 0xffffffff)
                second = _buckets->at(_bi).first->at(_i);
            else
                second = Mapped();

            return *this;
        }

        const_iterator *operator*() const
        {
            return const_cast<const_iterator *>(this);
        }

        const_iterator *operator->() const
        {
            return const_cast<const_iterator *>(this);
        }

        bool operator==(const const_iterator &other) const
        {
            return ((_bi == other._bi) && (_i == other._i));
        }

        bool operator!=(const const_iterator &other) const
        {
            return !(*this==other);
        }

        unsigned int first;
        Mapped second;

    private:
        const std::vector<std::pair<std::vector<Mapped> *, unsigned int> > *_buckets;
        unsigned int _bi;
        unsigned int _i;
    };

    FastMap(void)
    {
        // faster but needs more memory!
        //_buckets.resize(0x10000);
    }

    virtual ~FastMap()
    {
    }

    void insert(const std::pair<unsigned int, const Mapped> &p)
    {
        unsigned int key = p.first;
        const Mapped &value = p.second;
        //unsigned int bi = ((key >> 16) & 0xffff);
        //unsigned int i = (key & 0xffff);
        unsigned int bi = ((key >> 8) & 0xffffff);
        unsigned int i = (key & 0xff);

        _buckets.resize(bi + 1, std::make_pair((std::vector<Mapped> *) NULL, 0));
        std::vector<Mapped> *values = NULL;
        if(_buckets[bi].first == NULL)
        {
            values = new std::vector<Mapped>;
            _buckets[bi].first = values;
            _buckets[bi].second = 0;
        }
        else
        {
            values = _buckets[bi].first;
        }
        values->resize(i + 1);
        values->at(i) = value;
        _buckets[bi].second++;
    }

    const_iterator find(unsigned int key) const
    {
        //unsigned int bi = ((key >> 16) & 0xffff);
        //unsigned int i = (key & 0xffff);
        unsigned int bi = ((key >> 8) & 0xffffff);
        unsigned int i = (key & 0xff);

        if(bi >= _buckets.size())
            return const_iterator(&_buckets, 0, 0xffffffff);
        if(_buckets[bi].first == NULL || i >=  _buckets[bi].first->size())
            return const_iterator(&_buckets, 0, 0xffffffff);

        return const_iterator(&_buckets, bi, i);
    }

    void erase(unsigned int key)
    {
        //unsigned int bi = ((key >> 16) & 0xffff);
        //unsigned int i = (key & 0xffff);
        unsigned int bi = ((key >> 8) & 0xffffff);
        unsigned int i = (key & 0xff);

        if(bi >= _buckets.size())
            return;
        if(i >=  _buckets[bi].first->size())
            return;

        _buckets[bi].first->at(i) = Mapped();
        _buckets[bi].second--;
        if(_buckets[bi].second == 0)
        {
            //printf("delete bucket %u\n", bi);
            delete _buckets[bi].first;
            _buckets[bi].first = NULL;
        }
    }

    void erase(const const_iterator &it)
    {
        if(it == end())
            return;

        erase(it->first);
    }

    const_iterator begin(void) const
    {
        return const_iterator(&_buckets, 0, 0);
    }

    const_iterator end(void) const
    {
        return const_iterator(&_buckets, 0, 0xffffffff);
    }

    unsigned int size(void) const
    {
        unsigned int s = 0;
        for(size_t i = 0;i < _buckets.size();++i)
            s += _buckets[i].second;
        return s;
    }


private:

    std::vector<std::pair<std::vector<Mapped> *, unsigned int> > _buckets;
};

}

#endif
