/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright (C) 2000-2002 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

#ifndef _OSGGL_H_
#define _OSGGL_H_
#pragma once

#include <OSGConfig.h>

#include "glad/glad.h"

// Need to define a definitely unused constant
// OpenGL doesn't provide one... :(

#define OSG_GL_UNUSED 0xffff


#include <map>
#include <iostream>
#include <vector>
#include <functional>

#ifdef OSG_DEBUG
#define glErr(text)                                 \
{                                                   \
    OSGGLenum glerr;                                   \
                                                    \
    while((glerr = glGetError()) != GL_NO_ERROR)    \
    {                                               \
        FWARNING(("(%s,%d): %s failed: %s (%#x)\n", \
                __FILE__, __LINE__,                 \
                (text),                             \
                gluErrorString(glerr),              \
                glerr));                            \
    }                                               \
}
#else
#define glErr(text)
#endif

#ifndef OSG_GL_ATTRIB_STACK_DEPTH
#define OSG_GL_ATTRIB_STACK_DEPTH 64
#endif





struct OSG_BASE_DLLMAPPING ViewportStack
{
    typedef struct
    {
        GLint x;
        GLint y;
        GLsizei width;
        GLsizei height;
    } ViewportEntry;
    ViewportStack() : m_currentIdx(0)
    {
        m_stack[0].x = 0;
        m_stack[0].y = 0;
        m_stack[0].width = 1;
        m_stack[0].height = 1;
    }
    ViewportEntry   m_stack[OSG_GL_ATTRIB_STACK_DEPTH];
    unsigned int    m_currentIdx;
};

struct OSG_BASE_DLLMAPPING BitFlagStack
{
    typedef struct
    {
        unsigned long long flags;
    } BitFlagEntry;
    BitFlagStack() : m_currentIdx(0)
    {
        unsigned long long oneBit = 1;
        m_flags[GL_BLEND] = oneBit << 0;
        m_flags[GL_CLIP_DISTANCE0] = oneBit << 1;
        m_flags[GL_CLIP_DISTANCE1] = oneBit << 2;
        m_flags[GL_CLIP_DISTANCE2] = oneBit << 3;
        m_flags[GL_CLIP_DISTANCE3] = oneBit << 4;
        m_flags[GL_CLIP_DISTANCE4] = oneBit << 5;
        m_flags[GL_CLIP_DISTANCE5] = oneBit << 6;
        m_flags[GL_PROGRAM_POINT_SIZE] = oneBit << 7;
        m_flags[GL_COLOR_LOGIC_OP] = oneBit << 8;
        m_flags[GL_CULL_FACE] = oneBit << 9;
        m_flags[GL_DEBUG_OUTPUT] = oneBit << 10;
        m_flags[GL_DEBUG_OUTPUT_SYNCHRONOUS] = oneBit << 11;
        m_flags[GL_DEPTH_CLAMP] = oneBit << 12;
        m_flags[GL_DEPTH_TEST] = oneBit << 13;
        m_flags[GL_DITHER] = oneBit << 14;
        m_flags[GL_FRAMEBUFFER_SRGB] = oneBit << 15;
        m_flags[GL_LINE_SMOOTH] = oneBit << 16;
        m_flags[GL_MULTISAMPLE] = oneBit << 17;
        m_flags[GL_POLYGON_OFFSET_FILL] = oneBit << 18;
        m_flags[GL_POLYGON_OFFSET_LINE] = oneBit << 19;
        m_flags[GL_POLYGON_OFFSET_POINT] = oneBit << 20;
        m_flags[GL_POLYGON_SMOOTH] = oneBit << 21;
        m_flags[GL_PRIMITIVE_RESTART] = oneBit << 22;
        m_flags[GL_PRIMITIVE_RESTART_FIXED_INDEX] = oneBit << 23;
        m_flags[GL_RASTERIZER_DISCARD] = oneBit << 24;
        m_flags[GL_SAMPLE_ALPHA_TO_COVERAGE] = oneBit << 25;
        m_flags[GL_SAMPLE_ALPHA_TO_ONE] = oneBit << 26;
        m_flags[GL_SAMPLE_COVERAGE] = oneBit << 27;
        m_flags[GL_SAMPLE_SHADING] = oneBit << 28;
        m_flags[GL_SAMPLE_MASK] = oneBit << 29;
        m_flags[GL_SCISSOR_TEST] = oneBit << 30;
        m_flags[GL_STENCIL_TEST] = oneBit << 31;
        m_flags[GL_TEXTURE_CUBE_MAP_SEAMLESS] = oneBit << 32;
        m_flags[GL_SHADING_RATE_IMAGE_NV] = oneBit << 33;
        m_flags[GL_POINT_SPRITE] = oneBit << 34;
        m_stack[0].flags = m_flags[GL_DITHER] | m_flags[GL_MULTISAMPLE]; // everything off by default according to the OpenGL standard except for MULTISAMPLE and DITHER
        m_currentFlagState = m_stack[0].flags;
    }
    BitFlagEntry    m_stack[OSG_GL_ATTRIB_STACK_DEPTH];
    unsigned long long                  m_currentFlagState;
    unsigned int    m_currentIdx;
    std::map<GLenum, unsigned long long> m_flags;
};

struct OSG_BASE_DLLMAPPING FramebufferStack
{
    typedef struct
    {
        GLenum  target;
        GLuint  buffer;
    } FramebufferEntry;
    FramebufferStack(GLuint defaultFrameBuffer = 0) : m_currentIdx(0)
    {
        m_stack[0].target = GL_FRAMEBUFFER;
        m_stack[0].buffer = defaultFrameBuffer;
    }
    FramebufferEntry         m_stack[OSG_GL_ATTRIB_STACK_DEPTH];
    unsigned int             m_currentIdx;
};

struct OSG_BASE_DLLMAPPING BufferStack
{
    BufferStack(GLenum defaultBuffer) : m_currentIdx(0)
    {
        m_stack[0].push_back(defaultBuffer);
    }
    std::vector<GLenum>      m_stack[OSG_GL_ATTRIB_STACK_DEPTH];
    unsigned int             m_currentIdx;
};

struct OSG_BASE_DLLMAPPING ColorMaskStack
{
    typedef struct
    {
        GLboolean redMask;
        GLboolean greenMask;
        GLboolean blueMask;
        GLboolean alphaMask;
    } ColorMaskEntry;

    ColorMaskStack() : m_currentIdx(0)
    {
        m_stack[0].redMask = GL_TRUE;
        m_stack[0].greenMask = GL_TRUE;
        m_stack[0].blueMask = GL_TRUE;
        m_stack[0].alphaMask = GL_TRUE;

    }
    ColorMaskEntry   m_stack[OSG_GL_ATTRIB_STACK_DEPTH];
    unsigned int    m_currentIdx;
};

struct OSG_BASE_DLLMAPPING DepthMaskStack
{
    typedef struct
    {
        GLboolean depthMask;
    } DepthMaskEntry;

    DepthMaskStack() : m_currentIdx(0)
    {
        m_stack[0].depthMask = GL_TRUE;
    }

    DepthMaskEntry  m_stack[OSG_GL_ATTRIB_STACK_DEPTH];
    unsigned int    m_currentIdx;
};

struct OSG_BASE_DLLMAPPING DepthFuncStack
{
    typedef struct
    {
        GLenum depthFunc;
    } DepthFuncEntry;

    DepthFuncStack() : m_currentIdx(0)
    {
        m_stack[0].depthFunc = GL_LESS;
    }

    DepthFuncEntry  m_stack[OSG_GL_ATTRIB_STACK_DEPTH];
    unsigned int    m_currentIdx;
};

struct OSG_BASE_DLLMAPPING ClearDepthStack
{
    typedef struct
    {
        float depth;
    } ClearDepthEntry;

    ClearDepthStack() : m_currentIdx(0)
    {
        m_stack[0].depth = 1.0f;
    }

    ClearDepthEntry m_stack[OSG_GL_ATTRIB_STACK_DEPTH];
    unsigned int    m_currentIdx;
};

struct OSG_BASE_DLLMAPPING ClipControlStack
{
    typedef struct
    {
        GLenum origin;
        GLenum depth;
    } ClipControlEntry;

    ClipControlStack() : m_currentIdx(0)
    {
        m_stack[0].origin = GL_LOWER_LEFT;
        m_stack[0].depth = GL_NEGATIVE_ONE_TO_ONE;
    }

    ClipControlEntry m_stack[OSG_GL_ATTRIB_STACK_DEPTH];
    unsigned int    m_currentIdx;
};

OSG_BASE_DLLMAPPING GLuint* getDefaultLeftFramebufferHandle();
OSG_BASE_DLLMAPPING GLuint* getDefaultRightFramebufferHandle();
OSG_BASE_DLLMAPPING ViewportStack* getScissorStack();
OSG_BASE_DLLMAPPING ViewportStack* getViewportStack();
OSG_BASE_DLLMAPPING BitFlagStack* getBitFlagStack();
OSG_BASE_DLLMAPPING FramebufferStack* getFramebufferStack();
OSG_BASE_DLLMAPPING BufferStack* getDrawBufferStack();
OSG_BASE_DLLMAPPING BufferStack* getReadBufferStack();
OSG_BASE_DLLMAPPING ColorMaskStack* getColorMaskStack();
OSG_BASE_DLLMAPPING DepthMaskStack* getDepthMaskStack();
OSG_BASE_DLLMAPPING DepthFuncStack* getDepthFuncStack();
OSG_BASE_DLLMAPPING ClearDepthStack* getClearDepthStack();
OSG_BASE_DLLMAPPING ClipControlStack* getClipControlStack();
OSG_BASE_DLLMAPPING void registerMakeCurrentCB(std::function<bool(void)> callback);
OSG_BASE_DLLMAPPING void registerDoneCurrentCB(std::function<void(void)> callback);
OSG_BASE_DLLMAPPING bool osgMakeCurrent();
OSG_BASE_DLLMAPPING void osgDoneCurrent();



inline void setDefaultFramebuffer(GLuint bufferId)
{
    (*getDefaultLeftFramebufferHandle()) = bufferId;
}

inline GLuint getDefaultFramebuffer()
{
    return *getDefaultLeftFramebufferHandle();
}

inline void setDefaultLeftFramebuffer(GLuint bufferId)
{
    setDefaultFramebuffer (bufferId);
}

inline GLuint getDefaultLeftFramebuffer()
{
    return getDefaultFramebuffer();
}

inline void setDefaultRightFramebuffer(GLuint bufferId)
{
    (*getDefaultRightFramebufferHandle()) = bufferId;
}

inline GLuint getDefaultRightFramebuffer()
{
    return *getDefaultRightFramebufferHandle();
}

inline void setGLViewport(GLint x, GLint y, GLsizei width, GLsizei height)
{
    // need to store the value on the stack whenever we set it
    static ViewportStack* viewportStack = getViewportStack();
    const unsigned int currentIdx = viewportStack->m_currentIdx;
    viewportStack->m_stack[currentIdx].x = x;
    viewportStack->m_stack[currentIdx].y = y;
    viewportStack->m_stack[currentIdx].width = width;
    viewportStack->m_stack[currentIdx].height = height;

    glViewport(x, y, width, height);
}

inline void pushGLViewport()
{
    static ViewportStack* viewportStack = getViewportStack();
    const unsigned int oldIdx = viewportStack->m_currentIdx;
    viewportStack->m_currentIdx++;
    const unsigned int newIdx = viewportStack->m_currentIdx;

    viewportStack->m_stack[newIdx].x = viewportStack->m_stack[oldIdx].x;
    viewportStack->m_stack[newIdx].y = viewportStack->m_stack[oldIdx].y;
    viewportStack->m_stack[newIdx].width = viewportStack->m_stack[oldIdx].width;
    viewportStack->m_stack[newIdx].height = viewportStack->m_stack[oldIdx].height;
}

inline void popGLViewport()
{
    static ViewportStack* viewportStack = getViewportStack();
    viewportStack->m_currentIdx--;
    const unsigned int currentIdx = viewportStack->m_currentIdx;
    glViewport(viewportStack->m_stack[currentIdx].x, viewportStack->m_stack[currentIdx].y, viewportStack->m_stack[currentIdx].width, viewportStack->m_stack[currentIdx].height);
}

inline ViewportStack::ViewportEntry getGLCurrentViewport()
{
    static ViewportStack* viewportStack = getViewportStack();
    return viewportStack->m_stack[viewportStack->m_currentIdx];
}



inline void setGLScissor(GLint x, GLint y, GLsizei width, GLsizei height)
{
    // need to store the value on the stack whenever we set it
    static ViewportStack* scissorStack = getScissorStack();
    const unsigned int currentIdx = scissorStack->m_currentIdx;
    scissorStack->m_stack[currentIdx].x = x;
    scissorStack->m_stack[currentIdx].y = y;
    scissorStack->m_stack[currentIdx].width = width;
    scissorStack->m_stack[currentIdx].height = height;

    glScissor(x, y, width, height);
}

inline void pushGLScissor()
{
    static ViewportStack* scissorStack = getScissorStack();
    const unsigned int oldIdx = scissorStack->m_currentIdx;
    scissorStack->m_currentIdx++;
    const unsigned int newIdx = scissorStack->m_currentIdx;
    scissorStack->m_stack[newIdx].x = scissorStack->m_stack[oldIdx].x;
    scissorStack->m_stack[newIdx].y = scissorStack->m_stack[oldIdx].y;
    scissorStack->m_stack[newIdx].width = scissorStack->m_stack[oldIdx].width;
    scissorStack->m_stack[newIdx].height = scissorStack->m_stack[oldIdx].height;
}

inline void popGLScissor()
{
    static ViewportStack* scissorStack = getScissorStack();
    scissorStack->m_currentIdx--;
    const unsigned int currentIdx = scissorStack->m_currentIdx;
    glScissor(scissorStack->m_stack[currentIdx].x, scissorStack->m_stack[currentIdx].y, scissorStack->m_stack[currentIdx].width, scissorStack->m_stack[currentIdx].height);
}


inline void enableGLFlag(GLenum cap)
{
    // need to store the value on the stack whenever we set it
    static BitFlagStack* bitflagStack = getBitFlagStack();
    const unsigned int currentIdx = bitflagStack->m_currentIdx;
    //std::cout << "enableGLFlag " << cap << " bitflagStack->m_currentIdx: " << bitflagStack->m_currentIdx << " Before Flags : " << bitflagStack->m_stack[currentIdx].flags << std::endl;
    std::map<GLenum, unsigned long long>::iterator iter = bitflagStack->m_flags.find(cap);
    if (iter == bitflagStack->m_flags.end())
        std::cout << "Enable Unknown Flag " << cap << std::endl;
    else
    {
        const unsigned long long bit = iter->second;
        if (!(bitflagStack->m_currentFlagState & bit))
        {
            bitflagStack->m_stack[currentIdx].flags |= bit;
            bitflagStack->m_currentFlagState |= bit; // remember the current flag state
            glEnable(cap);
        }
    }
}

inline void disableGLFlag(GLenum cap)
{
    // need to store the value on the stack whenever we set it
    static BitFlagStack* bitflagStack = getBitFlagStack();
    const unsigned int currentIdx = bitflagStack->m_currentIdx;
    std::map<GLenum, unsigned long long>::iterator iter = bitflagStack->m_flags.find(cap);
    if (iter == bitflagStack->m_flags.end())
    {
        std::cout << "Disable Unknown Flag " << cap << std::endl;
    }
    else
    {
        const unsigned long long bit = iter->second;
        if (bitflagStack->m_currentFlagState & bit)
        {
            bitflagStack->m_stack[currentIdx].flags &= ~bit;
            bitflagStack->m_currentFlagState &= ~bit;
            glDisable(cap);
        }
    }

}

inline bool isGLFlagEnabled(GLenum cap)
{
    static BitFlagStack* bitflagStack = getBitFlagStack();
    //const unsigned int currentIdx = bitflagStack->m_currentIdx;
    std::map<GLenum, unsigned long long>::iterator iter = bitflagStack->m_flags.find(cap);
    if (iter == bitflagStack->m_flags.end())
    {
        std::cout << "IsEnabled Unknown Flag " << cap << std::endl;
    }
    else
    {
        const unsigned long long bit = iter->second;
        if (bitflagStack->m_currentFlagState & bit)
        {
            return true;
        }
    }
    return false;
}


inline void pushGLFlags()
{
    static BitFlagStack* bitflagStack = getBitFlagStack();
    const unsigned int oldIdx = bitflagStack->m_currentIdx;
    bitflagStack->m_currentIdx++;
    const unsigned int newIdx = bitflagStack->m_currentIdx;
    bitflagStack->m_stack[newIdx].flags = bitflagStack->m_stack[oldIdx].flags;
}

inline void popGLFlags()
{
    static BitFlagStack* bitflagStack = getBitFlagStack();
    bitflagStack->m_currentIdx--;
    const unsigned int currentIdx = bitflagStack->m_currentIdx;
    const unsigned long long bitMask = bitflagStack->m_stack[currentIdx].flags;
    for (auto&& flag : bitflagStack->m_flags)
    {
        if (bitMask & flag.second)
        {
            // only enable the flag if it is not already enabled
            if (!(bitflagStack->m_currentFlagState & flag.second))
                glEnable(flag.first);
        }
        else
        {
            // only disable the flag if it is enabled at the moment
            if (bitflagStack->m_currentFlagState & flag.second)
                glDisable(flag.first);
        }
    }
    bitflagStack->m_currentFlagState = bitMask;
}

inline void bindGLFramebuffer(GLenum target, GLuint buffer)
{
    // need to store the value on the stack whenever we set it
    static FramebufferStack* framebufferStack = getFramebufferStack();
    const unsigned int currentIdx = framebufferStack->m_currentIdx;
    framebufferStack->m_stack[currentIdx].target = target;
    framebufferStack->m_stack[currentIdx].buffer = buffer;
    glBindFramebuffer(target, buffer);
}

inline void pushGLFramebuffer()
{
    static FramebufferStack* framebufferStack = getFramebufferStack();
    framebufferStack->m_stack[framebufferStack->m_currentIdx + 1].target = framebufferStack->m_stack[framebufferStack->m_currentIdx].target;
    framebufferStack->m_stack[framebufferStack->m_currentIdx + 1].buffer = framebufferStack->m_stack[framebufferStack->m_currentIdx].buffer;
    framebufferStack->m_currentIdx++;
}

inline void popGLFramebuffer()
{
    static FramebufferStack* framebufferStack = getFramebufferStack();
    framebufferStack->m_currentIdx--;
    const unsigned int idx = framebufferStack->m_currentIdx;
    glBindFramebuffer(framebufferStack->m_stack[idx].target, framebufferStack->m_stack[idx].buffer);
}

inline FramebufferStack::FramebufferEntry getGLCurrentFramebuffer()
{
    static FramebufferStack* framebufferStack = getFramebufferStack();
    return framebufferStack->m_stack[framebufferStack->m_currentIdx];
}


inline void setGLDrawBuffer(GLenum buffer)
{
    // need to store the value on the stack whenever we set it
    static BufferStack* drawBufferStack = getDrawBufferStack();
    const unsigned int currentIdx = drawBufferStack->m_currentIdx;
    drawBufferStack->m_stack[currentIdx].clear();
    drawBufferStack->m_stack[currentIdx].push_back(buffer);
    glDrawBuffer(buffer);
}
inline void setGLDrawBuffers(GLsizei count, const GLenum* buffers)
{
    // need to store the value on the stack whenever we set it
    static BufferStack* drawBufferStack = getDrawBufferStack();
    const unsigned int currentIdx = drawBufferStack->m_currentIdx;
    drawBufferStack->m_stack[currentIdx].clear();
    for (GLsizei i = 0; i < count; ++i)
        drawBufferStack->m_stack[currentIdx].push_back(buffers[i]);

    glDrawBuffers(count, buffers);
}

inline void pushGLDrawBuffer()
{
    static BufferStack* drawBufferStack = getDrawBufferStack();
    const unsigned int oldIdx = drawBufferStack->m_currentIdx;
    drawBufferStack->m_currentIdx++;
    const unsigned int currentIdx = drawBufferStack->m_currentIdx;
    drawBufferStack->m_stack[currentIdx].clear();
    for (size_t i = 0; i < drawBufferStack->m_stack[oldIdx].size(); ++i)
        drawBufferStack->m_stack[currentIdx].push_back(drawBufferStack->m_stack[oldIdx][i]);
}

inline void popGLDrawBuffer()
{
    static BufferStack* drawBufferStack = getDrawBufferStack();
    drawBufferStack->m_currentIdx--;
    const unsigned int currentIdx = drawBufferStack->m_currentIdx;
    /* if (drawBufferStack->m_stack[currentIdx].size() == 1)
    {
    // if ((drawBufferStack->m_stack[currentIdx][0] >= OSG_GL_COLOR_ATTACHMENT0) && (drawBufferStack->m_stack[currentIdx][0] <= OSG_GL_COLOR_ATTACHMENT15))
    glDrawBuffers((GLsizei)drawBufferStack->m_stack[currentIdx].size(), &(drawBufferStack->m_stack[currentIdx][0]));
    // else
    //     glDrawBuffer(drawBufferStack->m_stack[currentIdx][0]);
    }
    else*/
    glDrawBuffers((GLsizei)drawBufferStack->m_stack[currentIdx].size(), &(drawBufferStack->m_stack[currentIdx][0]));
}


inline void setGLReadBuffer(GLenum buffer)
{
    // need to store the value on the stack whenever we set it
    static BufferStack* readBufferStack = getReadBufferStack();
    const unsigned int currentIdx = readBufferStack->m_currentIdx;
    readBufferStack->m_stack[currentIdx].clear();
    readBufferStack->m_stack[currentIdx].push_back(buffer);
    glReadBuffer(buffer);
}

inline void pushGLReadBuffer()
{
    static BufferStack* readBufferStack = getReadBufferStack();
    const unsigned int oldIdx = readBufferStack->m_currentIdx;
    readBufferStack->m_currentIdx++;
    const unsigned int currentIdx = readBufferStack->m_currentIdx;
    readBufferStack->m_stack[currentIdx].clear();
    for (size_t i = 0; i < readBufferStack->m_stack[oldIdx].size(); ++i)
        readBufferStack->m_stack[currentIdx].push_back(readBufferStack->m_stack[oldIdx][i]);
}

inline void popGLReadBuffer()
{
    static BufferStack* readBufferStack = getReadBufferStack();
    readBufferStack->m_currentIdx--;
    const unsigned int currentIdx = readBufferStack->m_currentIdx;

    glReadBuffer(readBufferStack->m_stack[currentIdx][0]);
}

inline void setGLColorMask(GLboolean redMask, GLboolean greenMask, GLboolean blueMask, GLboolean alphaMask)
{
    // need to store the value on the stack whenever we set it
    static ColorMaskStack* colorMaskStack = getColorMaskStack();

    const unsigned int currentIdx = colorMaskStack->m_currentIdx;
    colorMaskStack->m_stack[currentIdx].redMask = redMask;
    colorMaskStack->m_stack[currentIdx].greenMask = greenMask;
    colorMaskStack->m_stack[currentIdx].blueMask = blueMask;
    colorMaskStack->m_stack[currentIdx].alphaMask = alphaMask;

    glColorMask(redMask, greenMask, blueMask, alphaMask);
}

inline void setGLColorMaskBuffer(GLuint buffer, GLboolean redMask, GLboolean greenMask, GLboolean blueMask, GLboolean alphaMask)
{
    // need to store the value on the stack whenever we set it
    static ColorMaskStack* colorMaskStack = getColorMaskStack();

    const unsigned int currentIdx = colorMaskStack->m_currentIdx;
    colorMaskStack->m_stack[currentIdx].redMask = redMask;
    colorMaskStack->m_stack[currentIdx].greenMask = greenMask;
    colorMaskStack->m_stack[currentIdx].blueMask = blueMask;
    colorMaskStack->m_stack[currentIdx].alphaMask = alphaMask;
    for( unsigned int i = 0; i < 16; i++)
    {
        if( buffer & (1 << i))
            glColorMaski(i, redMask, greenMask, blueMask, alphaMask);
    }
}

inline void pushGLColorMask()
{
    static ColorMaskStack* colorMaskStack = getColorMaskStack();
    const unsigned int oldIdx = colorMaskStack->m_currentIdx;
    colorMaskStack->m_currentIdx++;
    const unsigned int newIdx = colorMaskStack->m_currentIdx;
    colorMaskStack->m_stack[newIdx].redMask = colorMaskStack->m_stack[oldIdx].redMask;
    colorMaskStack->m_stack[newIdx].greenMask = colorMaskStack->m_stack[oldIdx].greenMask;
    colorMaskStack->m_stack[newIdx].blueMask = colorMaskStack->m_stack[oldIdx].blueMask;
    colorMaskStack->m_stack[newIdx].alphaMask = colorMaskStack->m_stack[oldIdx].alphaMask;
}

inline void popGLColorMask()
{
    static ColorMaskStack* colorMaskStack = getColorMaskStack();
    colorMaskStack->m_currentIdx--;
    const unsigned int currentIdx = colorMaskStack->m_currentIdx;
    glColorMask(colorMaskStack->m_stack[currentIdx].redMask, colorMaskStack->m_stack[currentIdx].greenMask, colorMaskStack->m_stack[currentIdx].blueMask, colorMaskStack->m_stack[currentIdx].alphaMask);
}

inline void setGLDepthMask(GLboolean mask)
{
    // need to store the value on the stack whenever we set it
    static DepthMaskStack* depthMaskStack = getDepthMaskStack();
    const unsigned int currentIdx = depthMaskStack->m_currentIdx;
    if(depthMaskStack->m_stack[currentIdx].depthMask != mask)
    {
        depthMaskStack->m_stack[currentIdx].depthMask = mask;
        glDepthMask(mask);
    }
}

inline void pushGLDepthMask()
{
    static DepthMaskStack* depthMaskStack = getDepthMaskStack();
    const unsigned int oldIdx = depthMaskStack->m_currentIdx;
    depthMaskStack->m_currentIdx++;
    const unsigned int newIdx = depthMaskStack->m_currentIdx;
    depthMaskStack->m_stack[newIdx].depthMask = depthMaskStack->m_stack[oldIdx].depthMask;
}

inline void popGLDepthMask()
{
    static DepthMaskStack* depthMaskStack = getDepthMaskStack();
    depthMaskStack->m_currentIdx--;
    const unsigned int currentIdx = depthMaskStack->m_currentIdx;
    glDepthMask(depthMaskStack->m_stack[currentIdx].depthMask);
}

inline void setGLDepthFunc(GLenum func)
{
    // need to store the value on the stack whenever we set it
    static DepthFuncStack* depthFuncStack = getDepthFuncStack();
    const unsigned int currentIdx = depthFuncStack->m_currentIdx;
    if(depthFuncStack->m_stack[currentIdx].depthFunc != func)
    {
        depthFuncStack->m_stack[currentIdx].depthFunc = func;
        glDepthFunc(func);
    }
}

inline void pushGLDepthFunc()
{
     static DepthFuncStack* depthFuncStack = getDepthFuncStack();
    const unsigned int oldIdx = depthFuncStack->m_currentIdx;
    depthFuncStack->m_currentIdx++;
    const unsigned int newIdx = depthFuncStack->m_currentIdx;
    depthFuncStack->m_stack[newIdx].depthFunc = depthFuncStack->m_stack[oldIdx].depthFunc;
}

inline void popGLDepthFunc()
{
     static DepthFuncStack* depthFuncStack = getDepthFuncStack();
    depthFuncStack->m_currentIdx--;
    const unsigned int currentIdx = depthFuncStack->m_currentIdx;
    glDepthFunc(depthFuncStack->m_stack[currentIdx].depthFunc);
}

inline void setGLClearDepth(float depth)
{
    // need to store the value on the stack whenever we set it
    static ClearDepthStack* clearDepthStack = getClearDepthStack();
    const unsigned int currentIdx = clearDepthStack->m_currentIdx;
    if(clearDepthStack->m_stack[currentIdx].depth != depth)
    {
        clearDepthStack->m_stack[currentIdx].depth = depth;
        glClearDepth(depth);
    }
}

inline void pushGLClearDepth()
{
    static ClearDepthStack* clearDepthStack = getClearDepthStack();
    const unsigned int oldIdx = clearDepthStack->m_currentIdx;
    clearDepthStack->m_currentIdx++;
    const unsigned int newIdx = clearDepthStack->m_currentIdx;
    clearDepthStack->m_stack[newIdx].depth = clearDepthStack->m_stack[oldIdx].depth;
}

inline void popGLClearDepth()
{
    static ClearDepthStack* clearDepthStack = getClearDepthStack();
    clearDepthStack->m_currentIdx--;
    const unsigned int currentIdx = clearDepthStack->m_currentIdx;
    glClearDepth(clearDepthStack->m_stack[currentIdx].depth);
}

inline void setGLClipControl(GLenum origin, GLenum depth)
{
    // need to store the value on the stack whenever we set it
    static ClipControlStack* clipControlStack = getClipControlStack();
    const unsigned int currentIdx = clipControlStack->m_currentIdx;
    if(clipControlStack->m_stack[currentIdx].origin != origin || clipControlStack->m_stack[currentIdx].depth != depth)
    {
        clipControlStack->m_stack[currentIdx].origin = origin;
        clipControlStack->m_stack[currentIdx].depth = depth;
        if(GLAD_GL_VERSION_4_5 || GLAD_GL_ARB_clip_control)
        {
            glClipControl(origin, depth);
        }
    }
}

inline void pushGLClipControl()
{
    static ClipControlStack* clipControlStack = getClipControlStack();
    const unsigned int oldIdx = clipControlStack->m_currentIdx;
    clipControlStack->m_currentIdx++;
    const unsigned int newIdx = clipControlStack->m_currentIdx;
    clipControlStack->m_stack[newIdx].origin = clipControlStack->m_stack[oldIdx].origin;
    clipControlStack->m_stack[newIdx].depth = clipControlStack->m_stack[oldIdx].depth;
}

inline void popGLClipControl()
{
    static ClipControlStack* clipControlStack = getClipControlStack();
    clipControlStack->m_currentIdx--;
    const unsigned int currentIdx = clipControlStack->m_currentIdx;
    if(GLAD_GL_VERSION_4_5 || GLAD_GL_ARB_clip_control)
    {
        glClipControl(clipControlStack->m_stack[currentIdx].origin, clipControlStack->m_stack[currentIdx].depth);
    }
}
#define OSGGL_HEADER_CVSID "@(#)$Id: $"

#endif /* _OSGGL_H_ */
