#ifndef __ALIGNED_ALLOCATOR_H__
#define __ALIGNED_ALLOCATOR_H__

#ifndef __APPLE__
#include "malloc.h"
#endif
#include "xmmintrin.h"

template <class T> class aligned_allocator
{
public:
    typedef T                 value_type;
    typedef value_type*       pointer;
    typedef const value_type* const_pointer;
    typedef value_type&       reference;
    typedef const value_type& const_reference;
    typedef std::size_t       size_type;
    typedef std::ptrdiff_t    difference_type;

    template <class U> 
    struct rebind { typedef aligned_allocator<U> other; };

    aligned_allocator() {}
    aligned_allocator(const aligned_allocator&) {}
    template <class U> 
    aligned_allocator(const aligned_allocator<U>&) {}
    ~aligned_allocator() {}

    pointer address(reference x) const 
    { 
        return &x; 
    }
    const_pointer address(const_reference x) const 
    {
        return x;
    }

    pointer allocate(size_type n, const_pointer = 0) 
    {
        void* p = _mm_malloc(n * sizeof(T), 32); // align to 32 byte for AVX
        if (!p)
            throw std::bad_alloc();
        return static_cast<pointer>(p);
    }

    void deallocate(pointer p, size_type) 
    { 
        _mm_free(p);
    }

    size_type max_size() const 
    {
        return static_cast<size_type>(-1) / sizeof(T);
    }

    void construct(pointer p, const value_type& x) 
    {
        new(p) value_type(x);
    }
    
    void construct(pointer p) 
    {
        new(p) value_type;
    }
    
    void destroy(pointer p) 
    { 
        p->~value_type();
    }

private:
    void operator=(const aligned_allocator&);
};

template<> class aligned_allocator<void>
{
    typedef void        value_type;
    typedef void*       pointer;
    typedef const void* const_pointer;

    template <class U> 
    struct rebind { typedef aligned_allocator<U> other; };
};


template <class T>
inline bool operator==(const aligned_allocator<T>&, 
    const aligned_allocator<T>&) {
        return true;
}

template <class T>
inline bool operator!=(const aligned_allocator<T>&, 
    const aligned_allocator<T>&) {
        return false;
}

#endif
