/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright (C) 2000-2002 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

#ifdef OSG_USE_RSOCKETS
#define WIN32_LEAN_AND_MEAN
#endif

#include "OSGConfig.h"

#ifdef WIN32
#ifdef WIN32_LEAN_AND_MEAN
#include <winsock2.h>
#endif
#include <io.h>
#else
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/time.h>
#endif

#include <errno.h>
#include <stdio.h>
#include <math.h>
#include <map>
#include <OSGBase.h>
#include <OSGLog.h>
#include <OSGSocketAddress.h>
#include <OSGStreamSocket.h>

#ifndef AF_INET_SDP
#define AF_INET_SDP 27
#endif

OSG_USING_NAMESPACE

/** \class osg::StreamSocket
 *  \ingroup GrpBaseNetwork
 *  \brief Stream socket handler
 *
 * This class is a Handler to connection oriented sockets. A call to
 * <EM>open</EM> will assing a stream socket and <EM>close</EM>
 * releases the socket.
 *
 * Client example
 * <PRE>
 * char buffer[100];
 * StreamSocket s;
 * s.open();
 * s.connect(Address("serverhost.com",4567);
 * s.send(buffer,100);
 * s.close();
 * </PRE>
 *
 * Server example
 * <PRE>
 * char buffer[100];
 * StreamSocket s;
 * s.open();
 * s.bind(AnySocketAddress(4567);
 * c=s.accept();               // accept incomming client
 * c.recv(buffer,100);         // read client message
 * c.close();
 * s.close();
 * </PRE>
 **/

#ifdef OSG_USE_RSOCKETS
static const GUID rsProviderGuid = { //D478E78B-A803-4a25-A5E4-83BFB7EAF4A7
    0xd478e78b,
    0xa803,
    0x4a25,
    { 0xa5, 0xe4, 0x83, 0xbf, 0xb7, 0xea, 0xf4, 0xa7 }
};

static WSAPROTOCOL_INFO rsProtocolInfo = {0};

/**
 * \brief			Get RSockets Winsock provider's WSAPROTOCOL_INFO structure.
 *
 * \param lpStatus	Pointer to status variable to be returned. Can be NULL if not required.
 *
 * \return			Pointer to the RSockets Winsock provider's WSAPROTOCOL_INFO structure
 *					(NULL if the RSockets provider is not found or another error occured).
 */
static LPWSAPROTOCOL_INFO rsGetProtocolInfo (LPINT lpStatus)
{
    int                Status			= ERROR_SUCCESS;
    LPWSAPROTOCOL_INFO lpProtocolBuffer	= NULL;
    LPWSAPROTOCOL_INFO lpReturn			= NULL;
    DWORD              BufferLength		= 0;
    DWORD              i;

    WSAEnumProtocols (NULL, NULL, &BufferLength); // Should always return the BufferLength

    if (NULL == (lpProtocolBuffer = (LPWSAPROTOCOL_INFO)malloc (BufferLength)))
    {
        Status = ERROR_NOT_ENOUGH_MEMORY;
        goto cleanup;
    }

    if (SOCKET_ERROR == WSAEnumProtocols (NULL, lpProtocolBuffer, &BufferLength))
    {
        Status = WSAGetLastError();
        goto cleanup;
    }

    for (i = 0; i < BufferLength / sizeof(*lpProtocolBuffer); i++)
    {
        if (0 == memcmp (&lpProtocolBuffer[i].ProviderId, &rsProviderGuid, sizeof(rsProviderGuid)))
        {
            rsProtocolInfo	= lpProtocolBuffer[i];
            lpReturn		= &rsProtocolInfo;
            break;
        }
    }

    cleanup:
    if (lpProtocolBuffer)
        free (lpProtocolBuffer);

    if (lpStatus)
        *lpStatus = Status;

    return lpReturn;
}
#endif

bool StreamSocket::_sdp = false;

/*-------------------------------------------------------------------------*/
/*                            constructor destructor                       */

/*! Constructor. Use open to assign a system socket. No system socket is assigned by
    the constructor.
    \see StreamSocket::open
 */
StreamSocket::StreamSocket():
    Socket()
{
}

/** \brief Copy constructor
 */
StreamSocket::StreamSocket(const StreamSocket &source):
    Socket(source)
{
}

/*-------------------------------------------------------------------------*/
/*                           Socket functionaliy                           */

/*! Assign a socket. <CODE>Open</CODE> assignes a system socket to the
    StreamSocket.
    \see Socket::close
 */
void StreamSocket::open(void)
{
#ifdef OSG_USE_RSOCKETS
    _sd = WSASocket(AF_INET, SOCK_STREAM, IPPROTO_TCP, rsGetProtocolInfo(NULL), 0, 0);
#else
    // Socket Direct Protocol is only supported on linux!
#ifdef __linux
    if(_sdp)
    {
        _sd = ::socket(AF_INET_SDP, SOCK_STREAM, 0);
        if(_sd < 0)
        {
            _sd = ::socket(AF_INET, SOCK_STREAM, 0);
            FFATAL(("Socket Direct Protocol is not supported using standard Socket Protocol."));
        }
    }
    else
#endif
    {
        _sd = ::socket(AF_INET, SOCK_STREAM, 0);
    }
#endif

    if(_sd < 0)
    {
        throw SocketError("socket()");
    }
    struct linger li;
    li.l_onoff = 1;
    li.l_linger = 1;
    int rc = setsockopt(_sd,
                        SOL_SOCKET,
                        SO_LINGER,
                        reinterpret_cast<SocketOptT *>(&li),
                        sizeof(li));
}

/*! close socket
 */
void StreamSocket::close(void)
{
    Inherited::close();
}

/*! Accept incomming connection. Use the returned StreamSocket to
    communicate over the accepted communication. If the new StreamSocket
    is no longer used, you have to close it. A new StreamSocket is
    returned to communicate with the accepted client.
 */
StreamSocket StreamSocket::acceptFrom(SocketAddress &address)
{
    StreamSocket client;
    SocketLenT len;

    len = address.getSockAddrSize();
    client._sd =::accept(_sd,
                         address.getSockAddr(),
                         &len);
    if(client._sd < 0)
    {
        throw SocketError("accept()");
    }
    return client;
}

/*! Accept incomming connection. Use the returned StreamSocket to
    communicate over the accepted communication. If the new StreamSocket
    is no longer used, you have to close it.
 */
StreamSocket StreamSocket::accept()
{
    SocketAddress addr;
    return acceptFrom(addr);
}

/*! A Stream socket doesen't send data immediately. Only if the internal
    buffer contains enough data, an immediate write is forced. If
    delay is set to false, then data is written always immediately.
 */
void StreamSocket::setDelay(bool value)
{
    int rc, on;
    on = !value;
    rc = setsockopt(_sd,
                    IPPROTO_TCP,
                    TCP_NODELAY,
                    reinterpret_cast<SocketOptT *>(&on),
                    sizeof(on));
    if(rc < 0)
    {
        throw SocketError("setsockopt(,SOCK_STREAM,TCP_NODELAY)");
    }
}

/*-------------------------------------------------------------------------*/
/*                              assignment                                 */

/*! assignment
 */
const StreamSocket &StreamSocket::operator =(const StreamSocket &source)
{
    _sd = source._sd;
    return *this;
}

void StreamSocket::setSDP(bool s)
{
    _sdp = s;
}

bool StreamSocket::getSDP(void)
{
    return _sdp;
}

/*-------------------------------------------------------------------------*/
/*                              cvs id's                                   */

#ifdef __sgi
#pragma set woff 1174
#endif

#ifdef OSG_LINUX_ICC
#pragma warning( disable : 177 )
#endif

namespace
{
    static Char8 cvsid_cpp       [] = "@(#)$Id: $";
    static Char8 cvsid_hpp       [] = OSG_STREAMSOCKET_HEADER_CVSID;
}
