/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright (C) 2000-2002 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

#include <stdlib.h>
#include <stdio.h>

#include "OSGConfig.h"

#include <OSGLog.h>
#include <OSGFieldContainer.h>
#include <OSGFieldContainerPtr.h>
#include <OSGNode.h>
#include <OSGNodeCore.h>
#include <OSGAction.h>
#include <OSGRenderAction.h>
#include <OSGWindow.h>
#include <OSGCamera.h>
#include <OSGViewport.h>
#include <OSGBackground.h>
#include <OSGGLEXT.h>
#include <OSGTransform3D.h>

#include <OSGBaseFunctions.h>

#include <OSGMaterial.h>
#include <OSGMultiPassMaterial.h>
#include <OSGSwitchMaterial.h>
#include <OSGSimpleMaterial.h>

#include <OSGGeometry.h>
#include <OSGSwitch.h>
#include <OSGLog.h>

#include <OSGLight.h>
#include <OSGLightEnv.h>

#include <OSGClipPlane.h>

#include <OSGGL.h>
#include <OSGVolumeDraw.h>
#include <OSGName.h>

OSG_USING_NAMESPACE

/***************************************************************************\
 *                            Description                                  *
\***************************************************************************/

/*! \class osg::RenderAction
    \ingroup GrpSystemAction

    The render action class.

*/

/***************************************************************************\
 *                               Types                                     *
\***************************************************************************/

/***************************************************************************\
 *                           Class variables                               *
\***************************************************************************/

char RenderAction::cvsid[] = "@(#)$Id: $";


/*! \brief the prototype which is copied to create new actions
 */

RenderAction *RenderAction::_pPrototype = NULL;

/*! \brief Default functors for instantiation
 */

std::vector<Action::Functor> *RenderAction::_vDefaultEnterFunctors = NULL;
std::vector<Action::Functor> *RenderAction::_vDefaultLeaveFunctors = NULL;


StatElemDesc<StatTimeElem> RenderAction::statDrawTime("drawTime", 
"time for draw tree traversal");
StatElemDesc<StatUInt64Elem> RenderAction::statNMaterials("NMaterials",
"number of material changes");
StatElemDesc<StatUInt64Elem> RenderAction::statNMatrices("NMatrices",
"number of matrix changes");
StatElemDesc<StatUInt64Elem> RenderAction::statNLights("NLights",
"number of light changes");
StatElemDesc<StatUInt64Elem> RenderAction::statNGeometries("NGeometries",
"number of Geometry nodes");
StatElemDesc<StatUInt64Elem> RenderAction::statNTransGeometries("NTransGeometries",
"number of transformed Geometry nodes");
StatElemDesc<StatIntOnceElem> RenderAction::statNTextures("NTextures",
"number of texture changes");
StatElemDesc<StatIntOnceElem> RenderAction::statNTexBytes("NTexBytes",
"sum of all used textures' sizes (approx., in bytes)");
StatElemDesc<StatStringElem> RenderAction::statNOcclusionMode("OcclusionMode",
"occlusion culling mode");
StatElemDesc<StatUInt64Elem> RenderAction::statNOcclusionTests("NOcclusionTests",
"number of occlusion tests");
StatElemDesc<StatUInt64Elem> RenderAction::statNOcclusionCulled("NOcclusionCulled",
"number of objects culled via occlusion culling");

const Int32 RenderAction::OcclusionCullingAccurate = 0;
const Int32 RenderAction::OcclusionCullingFast = 1;

/***************************************************************************\
 *                           Class methods                                 *
\***************************************************************************/



/*-------------------------------------------------------------------------*\
 -  public                                                                 -
\*-------------------------------------------------------------------------*/

/*! \brief Default registration. static, so it can be called during static init
 */

void RenderAction::registerEnterDefault(const FieldContainerType &type, 
                                        const Action::Functor    &func)
{
    if(_vDefaultEnterFunctors == NULL)
    {
        _vDefaultEnterFunctors = new std::vector<Action::Functor>;
    }

    while(type.getId() >= _vDefaultEnterFunctors->size())
    {
        _vDefaultEnterFunctors->push_back(&Action::_defaultEnterFunction);
    }
    
    (*_vDefaultEnterFunctors)[type.getId()] = func;
}

void RenderAction::registerLeaveDefault(const FieldContainerType &type, 
                                        const Action::Functor    &func)
{
    if(_vDefaultLeaveFunctors == NULL)
    {
        _vDefaultLeaveFunctors = new std::vector<Action::Functor>;
    }

    while(type.getId() >= _vDefaultLeaveFunctors->size())
    {
        _vDefaultLeaveFunctors->push_back(&Action::_defaultLeaveFunction);
    }
    
    (*_vDefaultLeaveFunctors)[type.getId()] = func;
}


/*! \brief  prototype access
 *  after setting the prototype all new DrawActions are clones of it
 */

void RenderAction::setPrototype(RenderAction *pPrototype)
{
    _pPrototype = pPrototype;
}

RenderAction *RenderAction::getPrototype(void)
{
    return _pPrototype;
}

/*-------------------------------------------------------------------------*\
 -  protected                                                              -
\*-------------------------------------------------------------------------*/


/*-------------------------------------------------------------------------*\
 -  private                                                                -
\*-------------------------------------------------------------------------*/



/***************************************************************************\
 *                           Instance methods                              *
\***************************************************************************/

/*-------------------------------------------------------------------------*\
 -  public                                                                 -
\*-------------------------------------------------------------------------*/

/*------------- constructors & destructors --------------------------------*/

/** \brief Constructor
 */

RenderAction::RenderAction(void) :
     Inherited           (),

    _currMatrix          (),
    _pActiveState        (NULL),

    _uiNumMaterialChanges(0),
    _uiNumMatrixChanges  (0),
    _uiNumLightChanges   (0),
    _uiNumGeometries     (0),
    _uiNumOcclusionTests (0),
    _uiNumOcclusionCulled(0),
    _uiNumTransGeometries(0),

    _bSortTrans               (true),
    _bZWriteTrans             (false),
    _bLocalLights             (false),
    _bOcclusionCulling        (false),
    _occlusionCullingMode     (OcclusionCullingAccurate),
    _occlusionCullingThreshold(64),
    _currentOcclusionQueryIndex(0),
    _occluded_nodes            (),
    _hier_occlusions           (),
    _occ_bb_dl                 (0),


    _useGLFinish            (false),
    _depth_only_pass        (false),

    _selection_mode         (0),
    _polygon_mode           (GL_FILL),

    _hdrReferenceWhite(80.0f),

    _clipPlanesState(0),
    _clipPlanesTable(),
    _clipPlanesPath(),


    _stateSorting(true),


    _occlusionQuery         (0),
    _occlusionQueriesPool   (),
    _occlusionQueries       (),

    _glGenQueriesARB        (NULL),
    _glDeleteQueriesARB     (NULL),
    _glBeginQueryARB        (NULL),
    _glEndQueryARB          (NULL),
    _glGetQueryObjectuivARB (NULL)
{
    if(_vDefaultEnterFunctors != NULL)
        _enterFunctors = *_vDefaultEnterFunctors;

    if(_vDefaultLeaveFunctors != NULL)
        _leaveFunctors = *_vDefaultLeaveFunctors;

}


/** \brief Copy-Constructor
 */

RenderAction::RenderAction(const RenderAction &source) :
     Inherited           (source),


    _currMatrix          (source._currMatrix),
    _pActiveState        (source._pActiveState),

    _uiNumMaterialChanges(source._uiNumMaterialChanges),
    _uiNumMatrixChanges  (source._uiNumMatrixChanges),
    _uiNumLightChanges   (source._uiNumLightChanges),
    _uiNumGeometries     (source._uiNumGeometries),
    _uiNumOcclusionTests (source._uiNumOcclusionTests),
    _uiNumOcclusionCulled(source._uiNumOcclusionCulled),
    _uiNumTransGeometries(source._uiNumTransGeometries),

    _bSortTrans               (source._bSortTrans),
    _bZWriteTrans             (source._bZWriteTrans),
    _bLocalLights             (source._bLocalLights),
    _bOcclusionCulling        (source._bOcclusionCulling),
    _occlusionCullingMode     (source._occlusionCullingMode),
    _occlusionCullingThreshold(source._occlusionCullingThreshold),
    _currentOcclusionQueryIndex(source._currentOcclusionQueryIndex),
    _occluded_nodes            (source._occluded_nodes),
    _hier_occlusions           (source._hier_occlusions),
    _occ_bb_dl                 (source._occ_bb_dl),


    _useGLFinish            (source._useGLFinish),
    _depth_only_pass        (source._depth_only_pass),

    _selection_mode         (source._selection_mode),
    _polygon_mode(source._polygon_mode),

    _clipPlanesState        (source._clipPlanesState),
    _clipPlanesTable        (source._clipPlanesTable),
    _clipPlanesPath         (source._clipPlanesPath),

    _stateSorting           (source._stateSorting),

    _occlusionQuery         (source._occlusionQuery),
    _occlusionQueriesPool   (source._occlusionQueriesPool),
    _occlusionQueries       (source._occlusionQueries)
{

}

/** \brief create a new DrawAction by cloning the prototype
 */

RenderAction * RenderAction::create(void)
{
    RenderAction *returnValue;
    
    if(_pPrototype)
    {
        returnValue = new RenderAction(*_pPrototype);
    }
    else
    {
        returnValue = new RenderAction();
    }

    return returnValue;
}


/** \brief Destructor
 */

RenderAction::~RenderAction(void)
{
    if(_occ_bb_dl != 0)
        glDeleteLists(_occ_bb_dl, 1);

    if(_occlusionQuery != 0)
        _glDeleteQueriesARB(1, &_occlusionQuery);
    deleteOcclusionQueriesPool();
}

/*------------------------------ access -----------------------------------*/

/*---------------------------- properties ---------------------------------*/

void RenderAction::getMaterialStates(Material *mat, std::vector<State *> &states)
{
    if(!mat->isMultiPass())
    {
        states.push_back(getCPtr(mat->getState()));
    }
    else
    {
        // HACK need to find a nicer solution.
        MultiPassMaterial *mmat = dynamic_cast<MultiPassMaterial *>(mat);
        if(mmat != NULL)
        {
            // first check for a real multipass material.
            UInt32 passes = mmat->getMFMaterials()->getSize();
            if(passes > 0)
            {
                for(UInt32 i=0;i<passes;++i)
                {
                    getMaterialStates(getCPtr(mmat->getMaterials(i)), states);
                }
            }
            else
            {
                // could be a derived multipass material like CGFXMaterial which overrides
                // only some virtual methods.
                passes = mmat->getNPasses();
                for(UInt32 i=0;i<passes;++i)
                    states.push_back(getCPtr(mmat->getState(i)));
            }
        }
        else
        {
            SwitchMaterial *swmat = dynamic_cast<SwitchMaterial *>(mat);
            if(swmat != NULL)
            {
                getMaterialStates(getCPtr(swmat->getCurrentMaterial()), states);
            }
            else
            {
                UInt32 passes = mat->getNPasses();
                for(UInt32 i=0;i<passes;++i)
                    states.push_back(getCPtr(mat->getState(i)));
            }
        }
    }
}

void RenderAction::setPreselectionMode(UInt32 m)
{
}

void RenderAction::setSelectionMode(UInt32 m)
{
    _selection_mode = m;
}

UInt32 RenderAction::getSelectionMode(void) const
{
    return _selection_mode;
}

void RenderAction::setPolygonMode(const OSGGLenum& m)
{
    _polygon_mode = m;
}

const OSGGLenum RenderAction::getPolygonMode(void) const
{
    return _polygon_mode;
}


bool RenderAction::isVisible( Node* node )
{
  
    return true;
}


void RenderAction::deleteOcclusionQueriesPool(void)
{
    for(std::vector<OSGGLuint>::iterator occIt = _occlusionQueriesPool.begin();
        occIt != _occlusionQueriesPool.end();++occIt)
    {
        OSGGLuint occlusionQuery = (*occIt);
        glDeleteQueries(1, &occlusionQuery);
    }
    _occlusionQueriesPool.clear();

    for(std::map<UInt32, OSGGLuint>::iterator it=_occlusionQueries.begin();
        it!=_occlusionQueries.end();++it)
    {
        NodePtr node = NodePtr::dcast(FieldContainerFactory::the()->getContainer((*it).first));
        if(node != NullFC)
            setOcclusionMask(node, 0);
    }
    _occlusionQueries.clear();
    _occluded_nodes.clear();

    for(std::set<UInt32>::iterator it=_hier_occlusions.begin();
        it!=_hier_occlusions.end();++it)
    {
        NodePtr node = NodePtr::dcast(FieldContainerFactory::the()->getContainer(*it));
        if(node != NullFC)
            setOcclusionMask(node, 0);
    }
    _hier_occlusions.clear();

#if 0
    // reset all occlusion masks.
    Viewport *vp = getViewport();
    if(vp != NULL)
    {
        NodePtr root = vp->getRoot();
        setOcclusionMask(root, 0);
    }
#endif
}

void RenderAction::resetOcclusionQueryIndex(void)
{
    _currentOcclusionQueryIndex = 0;
}

void RenderAction::setOcclusionMask(NodePtr node, UInt8 mask)
{
    if(node == NullFC)
        return;

    node->setOcclusionMask(mask);

    for(UInt32 i=0;i<node->getNChildren();++i)
        setOcclusionMask(node->getChild(i), mask);
}

bool RenderAction::hasGeometryChild(NodePtr node)
{
    if(node == NullFC)
        return false;

    if(GeometryPtr::dcast(node->getCore()) != NullFC)
        return true;

    for(UInt32 i=0;i<node->getNChildren();++i)
    {
        if(hasGeometryChild(node->getChild(i)))
            return true;
    }

    return false;
}

OSGGLuint RenderAction::getOcclusionQuery(void)
{
    OSGGLuint occlusionQuery = 0;
    if(_currentOcclusionQueryIndex >= _occlusionQueriesPool.size())
    {
        // ok we re-use already created occlusion query objects.
        _glGenQueriesARB(1, &occlusionQuery);
        _occlusionQueriesPool.push_back(occlusionQuery);
    }
    else
    {
        occlusionQuery = _occlusionQueriesPool[_currentOcclusionQueryIndex];
    }

    ++_currentOcclusionQueryIndex;

    return occlusionQuery;
}

OSGGLuint RenderAction::getOcclusionQuery(NodePtr node)
{
    if(node == NullFC)
        return 0;

    std::map<UInt32, OSGGLuint>::iterator it = _occlusionQueries.find(getContainerId(node));

    if(it != _occlusionQueries.end())
        return (*it).second;

    return 0;
}

void RenderAction::setOcclusionQuery(NodePtr node, OSGGLuint occlusionQuery)
{
    if(node == NullFC)
        return;

    _occlusionQueries.insert(std::make_pair(getContainerId(node), occlusionQuery));
}



void RenderAction::setSortTrans(bool bVal)
{
    _bSortTrans = bVal;
}

bool RenderAction::getSortTrans(void) const
{
    return _bSortTrans;
}

void RenderAction::setZWriteTrans(bool bVal)
{
    _bZWriteTrans = bVal;
}

bool RenderAction::getZWriteTrans(void) const
{
    return _bZWriteTrans;
}

void RenderAction::setLocalLights(bool bVal)
{
    _bLocalLights = bVal;
}

bool RenderAction::getLocalLights(void) const
{
    return _bLocalLights;
}

void RenderAction::setOcclusionCulling(bool bVal)
{
    if(_bOcclusionCulling == bVal)
        return;

    deleteOcclusionQueriesPool();
    _bOcclusionCulling = bVal;
}

bool RenderAction::getOcclusionCulling(void) const
{
    return _bOcclusionCulling;
}

void RenderAction::setOcclusionCullingMode(Int32 mode)
{
    if(_occlusionCullingMode == mode)
        return;

    deleteOcclusionQueriesPool();
    _occlusionCullingMode = mode;
}

Int32 RenderAction::getOcclusionCullingMode(void) const
{
    return _occlusionCullingMode;
}

void RenderAction::setOcclusionCullingThreshold(UInt32 threshold)
{
    _occlusionCullingThreshold = threshold;
}

UInt32 RenderAction::getOcclusionCullingThreshold(void) const
{
    return _occlusionCullingThreshold;
}

void RenderAction::setUseGLFinish(bool s)
{
    _useGLFinish = s;
}

bool RenderAction::getUseGLFinish(void) const
{
    return _useGLFinish;
}

void RenderAction::setDepthOnlyPass(bool s)
{
    _depth_only_pass = s;
}

bool RenderAction::getDepthOnlyPass(void) const
{
    return _depth_only_pass;
}

void RenderAction::setHdrReferenceWhite(const float nits)
{
    _hdrReferenceWhite = nits;
}
    
float RenderAction::getHdrReferenceWhite() const
{
    return _hdrReferenceWhite;
}



void RenderAction::setUseMulticastSLI(bool on)
{
}

bool RenderAction::getUseMulticastSLI() const
{
    return false;
}



/*-------------------------- assignment -----------------------------------*/

/** \brief assignment
 */

/*-------------------------- comparison -----------------------------------*/


/*-------------------------------------------------------------------------*\
 -  protected                                                              -
\*-------------------------------------------------------------------------*/


std::vector<RenderAction::Functor> *RenderAction::getDefaultEnterFunctors(void)
{
    return _vDefaultEnterFunctors;
}

std::vector<RenderAction::Functor> *RenderAction::getDefaultLeaveFunctors(void)
{
    return _vDefaultLeaveFunctors;
}

/*-------------------------------------------------------------------------*\
 -  private                                                                -
\*-------------------------------------------------------------------------*/



///---------------------------------------------------------------------------
///  FUNCTION: 
///---------------------------------------------------------------------------
//:  Example for the head comment of a function
///---------------------------------------------------------------------------
///
//p: Paramaters: 
//p: 
///
//g: GlobalVars:
//g: 
///
//r: Return:
//r: 
///
//c: Caution:
//c: 
///
//a: Assumptions:
//a: 
///
//d: Description:
//d: 
///
//s: SeeAlso:
//s: 
///---------------------------------------------------------------------------

