/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright (C) 2000-2002 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

#include <stdlib.h>
#include <stdio.h>

#include <OSGConfig.h>
#include <OSGGLEXT.h>

#include "OSGRenderOptions.h"
#include "OSGRenderAction.h"

// the following tokens are not defined under WIN
#ifndef GL_NV_multisample_filter_hint
# define GL_MULTISAMPLE_FILTER_HINT_NV     0x8534
#endif

#ifndef GL_VERSION_1_3
# define GL_MULTISAMPLE                    0x809D
# define GL_SAMPLE_ALPHA_TO_COVERAGE       0x809E
# define GL_SAMPLE_ALPHA_TO_ONE            0x809F
# define GL_SAMPLE_COVERAGE                0x80A0
# define GL_SAMPLE_BUFFERS                 0x80A8
# define GL_SAMPLES                        0x80A9
# define GL_SAMPLE_COVERAGE_VALUE          0x80AA
# define GL_SAMPLE_COVERAGE_INVERT         0x80AB
# define GL_MULTISAMPLE_BIT                0x20000000
#endif

OSG_USING_NAMESPACE

namespace
{
    // Similar code in OSGUberMaterial.cpp. Do not forget to adapt when changes were made.
    const UInt32 analytic_mode = 1 << 0;
    const UInt32 realistic_mode = 1 << 1;
    const UInt32 raytracing_mode = 1 << 2;
    const UInt32 no_shader_mode = 1 << 3;
    const UInt32 texture_compression = 1 << 4;
    const UInt32 ao_on = 1 << 5;
    const UInt32 indirect_light = 1 << 6;
    const UInt32 high_quality = 1 << 7;
    const UInt32 npr_mode = 1 << 8;
    const UInt32 ssao_on = 1 << 9;
    const UInt32 lens_matched_shading = 1 << 10;
    const UInt32 use_geometry_lights = 1 << 11;
    const UInt32 evaluate_clipping_planes = 1 << 12;
    const UInt32 fresnel_respects_glossy_color = 1 << 13;
    const UInt32 use_ggx_brdf = 1 << 14;
    const UInt32 kDefaultRenderingMode = high_quality | indirect_light | ao_on | realistic_mode | evaluate_clipping_planes | fresnel_respects_glossy_color | use_ggx_brdf;
}


/*!
 * \brief
 * \param
 * \return
 */
RenderOptions::RenderOptions(void) :
    Inherited(),
    _changed(0),
    _last_changed(0),
    _gl_version(0.0f),
    _polygon_mode(GL_FILL),
    _backface_culling(false),
    _two_sided_lighting(false),
    _spec_tex_lighting(false)
{
    setRenderingMode(kDefaultRenderingMode);
}

/*!
 * \brief
 * \param
 * \return
 */
RenderOptions::RenderOptions(const RenderOptions &source) :
    Inherited(source),
    _changed(source._changed),
    _last_changed(source._last_changed),
    _gl_version(source._gl_version),
    _polygon_mode(source._polygon_mode),
    _backface_culling(source._backface_culling),
    _two_sided_lighting(source._two_sided_lighting),
    _spec_tex_lighting(source._spec_tex_lighting)
{
}

/*!
 * \brief
 * \param
 * \return
 */
RenderOptions::~RenderOptions(void)
{
}

/*!
 * \brief
 * \param
 * \return
 */
void RenderOptions::initMethod (void)
{
}

/*!
 * \brief
 * \param
 * \return
 */
void RenderOptions::changed(BitVector whichField, UInt32 origin)
{
    _changed |= whichField;
    Inherited::changed(whichField, origin);
}

void RenderOptions::setWireframe( bool value )
{
    if(value)
        setPolygonMode(GL_LINE);
    else
        setPolygonMode(GL_FILL);
}

bool RenderOptions::getWireframe( void )
{
    return getPolygonMode() == GL_LINE;
}

BitVector RenderOptions::getChanged( void )
{
    return _changed;
}

BitVector RenderOptions::getLastChanged( void )
{
    return _last_changed;
}

void RenderOptions::activateOptions(RenderAction *action)
{
    if(_gl_version == 0.0f)
    {
        // detect OpenGL version.
        std::string vstr = 
            reinterpret_cast<const char *>(glGetString(GL_VERSION));

        // remove last .x
        vstr = vstr.substr(0, 3);
        _gl_version = atof(vstr.c_str());
    }

    if(_changed & PolygonModeFieldMask)
        _polygon_mode = getPolygonMode();

    if(_changed & BackfaceCullingFieldMask)
        _backface_culling = getBackfaceCulling();
    
    if(_changed & SortTransFieldMask)
        action->setSortTrans(getSortTrans());

    if(_changed & ZWriteTransFieldMask)
        action->setZWriteTrans(getZWriteTrans());

    if(_changed & LocalLightsFieldMask)
        action->setLocalLights(getLocalLights());

    if(_changed & OcclusionCullingFieldMask)
        action->setOcclusionCulling(getOcclusionCulling());

   
    if (_changed & OcclusionCullingModeFieldMask)
    {
#if 1
        action->setOcclusionCullingMode(0); // fast occlusion culling mode does not work correctly in cluster mode so we force it to accurate for now
#else
        action->setOcclusionCullingMode(getOcclusionCullingMode());
#endif
    }

    if(_changed & FrustumCullingFieldMask)
        action->setFrustumCulling(getFrustumCulling());

    if(_changed & DepthOnlyPassFieldMask)
        action->setDepthOnlyPass(getDepthOnlyPass());

    if(_changed & SelectionFieldMask)
        action->setSelectionMode(getSelection());

    if(_changed & PreSelectionFieldMask)
        action->setPreselectionMode(getPreSelection());

    action->setPolygonMode(_polygon_mode);
    // we update the gl stuff each frame.
    if(_backface_culling)
    {
        enableGLFlag(GL_CULL_FACE);
        glCullFace(GL_BACK);
    }
    else
    {
        disableGLFlag(GL_CULL_FACE);
        glCullFace(GL_BACK);
    } 

    _last_changed = _changed;
    _changed = 0;
}
