/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright (C) 2000-2002 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------


#include <stdlib.h>
#include <stdio.h>

#include <sstream>

#include <OSGConfig.h>
#include <OSGSystemDef.h>

#include "OSGRenderAction.h"

#include "OSGDisplayCalibration.h"
#include "OSGRenderOptions.h"

#include "OSGDisplayFilterForeground.h"
#include "OSGColorDisplayFilter.h"
#include "OSGDistortionDisplayFilter.h"
#include "OSGResolutionDisplayFilter.h"

#include "OSGClusterWindow.h"

OSG_USING_NAMESPACE

/** \class osg::ClusterWindow
 *  \ingroup GrpSystemCluster
 *  \brief Abstract base class for cluster configurations
 *
 * A ClusterWindow describes a clustering algorithm. A ClusterWindow
 * inherits the ability to connect rendering servers and initiate
 * remote rendering. By configuring the cluster algorithm with an
 * OpenSG Window structure, it is possible to use cluster rendering
 * in the same way as rendering in a GLUT or Qt window.
 *
 **/

StatElemDesc<StatTimeElem> ClusterWindow::statActivateTime
  ("statActivateTime", "time to activate remote window");

StatElemDesc<StatTimeElem> ClusterWindow::statFrameInitTime
  ("statFrameInitTime", "time to frameInit remote window");

StatElemDesc<StatTimeElem> ClusterWindow::statRAVTime
  ("statRAVTime", "time to RAV remote window");

StatElemDesc<StatTimeElem> ClusterWindow::statSwapTime
  ("statSwapTime", "time to swap remote window");

StatElemDesc<StatTimeElem> ClusterWindow::statFrameExitTime
  ("statFrameExitTime", "time to frameExit remote window");

/*-------------------------------------------------------------------------*/
/*                          window functions                               */

//! react to field changes

void ClusterWindow::changed(BitVector whichField, UInt32 origin)
{
    Inherited::changed(whichField, origin);
}

//! output the instance for debug purposes

void ClusterWindow::dump(      UInt32    ,
                         const BitVector ) const
{
    SLOG << "Dump ClusterWindow NI" << std::endl;
}

void (*ClusterWindow::getFunctionByName ( const Char8 * ))()
{
    return NULL;
}

/*! init cluster window. connect to all servers
 */

void ClusterWindow::init( void )
{
}

void ClusterWindow::render(RenderActionBase *action)
{
    // activate
    if(_statistics)
        _statistics->getElem(statActivateTime)->start();
    activate();
    if(_statistics)
        _statistics->getElem(statActivateTime)->stop();

    // frameInit
    if(_statistics)
        _statistics->getElem(statFrameInitTime)->start();
    frameInit();
    if(_statistics)
        _statistics->getElem(statFrameInitTime)->stop();

    // RenderAllViewports
    if(_statistics)
        _statistics->getElem(statRAVTime)->start();
    renderAllViewports(action);
    if(_statistics)
        _statistics->getElem(statRAVTime)->stop();

    // swap
    if(_statistics)
        _statistics->getElem(statSwapTime)->start();
    swap();
    if(_statistics)
        _statistics->getElem(statSwapTime)->stop();

    // frameExit
    if(_statistics)
        _statistics->getElem(statFrameExitTime)->start();
    frameExit();
    if(_statistics)
        _statistics->getElem(statFrameExitTime)->stop();
}

void ClusterWindow::activate( void )
{
}

void ClusterWindow::deactivate( void )
{
}

void ClusterWindow::swap( void )
{
}

void ClusterWindow::renderAllViewports(RenderActionBase *action)
{
}

void ClusterWindow::frameInit(void)
{
}

void ClusterWindow::frameExit(void)
{
}

/*-------------------------------------------------------------------------*/
/*                          statistics                                     */

void ClusterWindow::setStatistics(StatCollector *statistics)
{
    _statistics = statistics;
}

/*-------------------------------------------------------------------------*/
/*                         client methods                                  */

void ClusterWindow::postInit(void)
{
}

/*! init client window. In a derived cluster window this method is called
 *  before the first sync with the rendering servers. There is no default
 *  action.
 */
void ClusterWindow::clientInit( void )
{
}

/** client frame before sync
 *
 * In a derived cluster window this method is called before
 * sync with the rendering servers. Default aciton is to activate
 * and init the client window.
 **/

void ClusterWindow::clientPreSync( void )
{
    if(getClientWindow() != NullFC)
    {
        getClientWindow()->activate();
        getClientWindow()->frameInit();
    }
}

/** initiate client rendering
 *
 * In a derived cluster window this method is called after the
 * sync with all rendering servers. Default aciton is to render all
 * viewports of the client window.
 **/

void ClusterWindow::clientRender(RenderActionBase *action)
{
    if(getClientWindow() != NullFC)
    {
        getClientWindow()->renderAllViewports( action );
    }
}

/** swap client window
 *
 * In a derived cluster window this method is called after rendering
 * Default action is to swap the local client window.
 **/

void ClusterWindow::clientSwap( void )
{
    if(getClientWindow() != NullFC)
    {
        getClientWindow()->swap( );
        getClientWindow()->frameExit();
    }

    if (getDirty())
    {
        ClusterWindowPtr ptr(this);

        beginEditCP(ptr, DirtyFieldMask);
            setDirty(false);
        endEditCP(ptr, DirtyFieldMask);
    }
}

bool ClusterWindow::updateFilter(WindowPtr window, UInt32 id,
                                 RenderActionBase *action)
{
    bool found = false;

    if (!getMFFilter()->empty() && getDirty())
    {
        UInt32 c, p;

        ClusterWindowPtr ptr(this);

        beginEditCP(ptr, DirtyFieldMask);
            setDirty(false);
        endEditCP(ptr, DirtyFieldMask);

        // for all viewports
        for(p=0; p<window->getMFPort()->size(); ++p)
        {
            // search filter foregrounds
            for(c=0; c<getMFFilter()->size(); ++c)
            {
                std::string name = getServers(id);
                char portName[64];

                if(window->getMFPort()->size() > 1)
                {
                    sprintf(portName,"[%d]",p);
                    name = name + portName;
                }

                DisplayFilterForegroundPtr filterFgnd = getFilter(c);

                if(filterFgnd->getServer() == name)
                {
                    beginEditCP(window->getPort(p), Viewport::ForegroundsFieldMask);

                    // first remove old filters, if any
                    for (Int32 n=window->getPort(p)->getMFForegrounds()->size(), j=n-1;
                            j>=0; j--)
                    {
                        MFForegroundPtr::iterator fgndIt =
                            window->getPort(p)->editMFForegrounds()->begin() + j;

                        if ( (*fgndIt) == filterFgnd )
                            window->getPort(p)->editMFForegrounds()->erase(fgndIt);
                    }

                    // then add new one
                    window->getPort(p)->editMFForegrounds()->push_back(filterFgnd);

                    endEditCP(window->getPort(p), Viewport::ForegroundsFieldMask);

                    found = true;
                    break;
                }
            }
        }
    }

    return found;
}

/** initialise the cluster window on the server side
 *
 * This method is called after the first sync.
 *
 * \param window     server render window
 * \param id         server id
 **/

void ClusterWindow::serverInit( WindowPtr ,
                                UInt32 )
{
}

/** render server window
 *
 * This method is called after synchronisation of all changes with the
 * rendering client. Default action is to render all viewports with the
 * given action
 *
 * !param window     server render window
 * !param id         server id
 * !param action     action
 **/

void ClusterWindow::serverRender( WindowPtr window,
                                  UInt32 id,
                                  RenderActionBase *action )
{
    updateFilter(window, id, action);

    RenderOptionsPtr ro;

    window->activate();
    window->frameInit();

    RenderAction *ract = dynamic_cast<RenderAction *>(action);
    bool useMulticastSLI = false;
    if (m_getUseMulticastSLIBufferCallback)
    {
        useMulticastSLI = m_getUseMulticastSLIBufferCallback();
        if (window->getMFPort()->size() != 2) // multicast only works if we have excactly two viewports.
            useMulticastSLI = false;

        ract->setUseMulticastSLI(useMulticastSLI);
        useMulticastSLI = ract->getUseMulticastSLI();
    }
    else
        ract->setUseMulticastSLI(false);
    
    if(ract != NULL)
    {
        MFViewportPtr::const_iterator portIt  = window->getMFPort()->begin();
        MFViewportPtr::const_iterator portEnd = window->getMFPort()->end();
        // try to find option as an attachment of window
        osg::RenderOptionsPtr winRo = osg::RenderOptionsPtr::dcast(
            window->findAttachment(osg::RenderOptions::getClassType()));
        ract->setWindow(getCPtr(window));

        if (m_initBufferCallback)
            m_initBufferCallback(window->getWidth(), window->getHeight());

        int eye = 0;
        while(portIt != portEnd)
        {
            // Removed RenderOptions activation here (ro->activateOptions).
            // It is now handled in vrNetWidget for all widget.

            if (m_bindBufferCallback)
                m_bindBufferCallback(eye);

            (*portIt)->render(ract);

            if (m_unbindBufferCallback)
                m_unbindBufferCallback(eye);

            eye = (eye + 1) % 2;

            ++portIt;

            if (useMulticastSLI)
                portIt = portEnd;
        }
    } else {
        if(action)
            window->renderAllViewports(action);
    }

    if (m_copyBufferCallback)
        m_copyBufferCallback(ract, useMulticastSLI);

    // do calibration
    DisplayCalibrationPtr calibPtr=NullFC;
    UInt32 c, p;

    // for all viewports
    for(p = 0 ; p<window->getMFPort()->size() ; ++p)
    {
        // search calibration
        for(c=0 ; c<getMFCalibration()->size() ; ++c)
        {
            std::string name = getServers(id);
            char portName[64];
            if(window->getMFPort()->size() > 1)
            {
                sprintf(portName,"[%d]",p);
                name = name + portName;
            }
            if(getCalibration(c)->getServer() == name)
            {
                calibPtr = getCalibration(c);
                calibPtr->calibrate(window->getPort(p),action);
                break;
            }
        }
    }
}

/** swap server window
 *
 * <code>serverSwap</code> is called after rendering. Default action is
 * to swap the rendering window.
 *
 * !param window     server render window
 * !param id         server id
 * !param connection connection to client
 **/
void ClusterWindow::serverSwap( WindowPtr window,
                                UInt32)
{
    window->swap();
    window->frameExit();
}

/*-------------------------------------------------------------------------*/
/*                         constructor / destructor                        */

//! Constructor

ClusterWindow::ClusterWindow(void) :
     Inherited()/*,
    _firstFrame(true),
    _statistics(NULL),
    _connectionFP(NULL),
    _network(NULL)*/
{
}

//! Copy Constructor

ClusterWindow::ClusterWindow(const ClusterWindow &source) :
    Inherited(source)/*,
    _firstFrame(true),
    _statistics(NULL),
    _connectionFP(source._connectionFP),
    _network(NULL)*/
{
}

//! Destructor

ClusterWindow::~ClusterWindow(void)
{
}

/*! initialize the static features of the class, e.g. action callbacks
 */
void ClusterWindow::initMethod (void)
{
}

void ClusterWindow::registerGetUseMulticastSLI(std::function<bool(void)>& callback)
{
	m_getUseMulticastSLIBufferCallback = callback;
}

void ClusterWindow::registerInitBufferCallback(std::function<void(const unsigned int, const unsigned int)>& callback)
{
    m_initBufferCallback = callback;
}

void ClusterWindow::registerBindBufferCallback(std::function<void(const unsigned int)>& callback)
{
    m_bindBufferCallback = callback;
}

void ClusterWindow::registerUnbindBufferCallback(std::function<void(const unsigned int)>& callback)
{
    m_unbindBufferCallback = callback;
}

void ClusterWindow::registerCopyBufferCallback(std::function<void(osg::DrawActionBase*, const bool)>& callback)
{
    m_copyBufferCallback = callback;
}
