/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *                     Copyright 2000-2002 by OpenSG Forum                   *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

#ifndef _OSGFIELDCONTAINERIMPL_H_
#define _OSGFIELDCONTAINERIMPL_H_
#pragma once

#ifdef OSG_DOC_FILES_IN_MODULE
/*! \file OSGFieldContainerImpl.h
    \ingroup GrpSystemFieldContainer
 */
#endif

#include <functional>
#include <vector>
#include <memory>
#include <atomic>
#include <set>
#include <string>

#include <OSGSystemDef.h>
#include <OSGIDStringLink.h>

#include <OSGFieldContainerBase.h>

#if defined(OSG_GV_BETA) && defined(OSG_DBG_MEM)
#include <OSGTime.h>
#include <OSGThread.h>
#endif

#define USE_CUSTOM_NEW_DELETE

#ifdef USE_CUSTOM_NEW_DELETE

#ifdef __GNUC__

#include <inttypes.h>
#else

#endif

#include "oneapi/tbb/scalable_allocator.h"

#endif

typedef std::function<void(osg::UInt32)> OnDestroyCallback;
typedef std::function<void(osg::UInt32, osg::BitVector, osg::UInt32)> ChangedCallback;

OSG_BEGIN_NAMESPACE

class Field;
class BinaryDataHandler;

struct ChannelAttributes
{
    ChannelAttributes(void)
        : _fcids(), _fieldName(""), _channelName("")
    {
    }

    ChannelAttributes(UInt32 fcid, const std::string& fieldName)
        : _fcids(1, fcid), _fieldName(fieldName), _channelName(fieldName)
    {
    }

    ChannelAttributes(UInt32 fcid, const std::string& fieldName,
        const std::string& channelName)
        : _fcids(1, fcid), _fieldName(fieldName), _channelName(channelName)
    {
    }

    ChannelAttributes(UInt32 fcid, const std::string& fieldName,
        const std::string& channelName,
        UInt32 affectedFcid)
        : _fieldName(fieldName), _channelName(channelName)
    {
        _fcids.push_back(fcid);
        _fcids.push_back(affectedFcid);
    }


    osg::UInt32 getAim() const { return (_fcids.empty() ? 0 : _fcids[0]); }

    // first item of _fcids is the field container that contains the animated field, 
    // next items are field containers that must be edited when changing that field
    std::vector<osg::UInt32> _fcids;
    std::string _fieldName;
    std::string _channelName;
};
typedef std::vector<ChannelAttributes> ChannelData;

/*! \ingroup GrpSystemFieldContainer
 */
#ifdef USE_CUSTOM_NEW_DELETE
class alignas(32) OSG_SYSTEMLIB_DLLMAPPING FieldContainer
#else
class OSG_SYSTEMLIB_DLLMAPPING FieldContainer
#endif
{
    /*==========================  PUBLIC  =================================*/

  public:

    enum { NextFieldId = 1 };

    static const BitVector NextFieldMask   = (TypeTraits<BitVector>::One << NextFieldId);
    static const BitVector MTInfluenceMask = 0x00;

    /*---------------------------------------------------------------------*/
    /*! \name             Get Class Type Information                       */
    /*! \{                                                                 */

    static FieldContainerType &getClassType   (void);
    static UInt32              getClassTypeId (void);
    static UInt16              getClassGroupId(void);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name             Get Instance Type Information                    */
    /*! \{                                                                 */

    
    virtual       FieldContainerType &getType    (void);

    
    virtual const FieldContainerType &getType    (void) const;

                  
                  UInt32              getTypeId  (void) const;

                  
                  UInt16              getGroupId (void) const;

            
            const Char8              *getTypeName(void) const;

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                    Container Size                            */
    /*! \{                                                                 */

    
    virtual UInt32  getContainerSize(void) const = 0;

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                                                              */
    /*! \{                    Generic Field Access                         */

            
            Field  *getField        (      UInt32 fieldId  );

            
            Field  *getField        (const Char8 *fieldName);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                       Clone                                  */
    /*! \{                                                                 */

    virtual FieldContainerPtr shallowCopy(void) const = 0;

    
    virtual void onDeepClone(void);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                   Binary Access                              */
    /*! \{                                                                 */

    
    virtual SizeT   getBinSize (const BitVector         &whichField);

    
    virtual void    copyToBin  (      BinaryDataHandler &pMem,
                                const BitVector         &whichField);

    
    virtual void    copyFromBin(      BinaryDataHandler &pMem,
                                const BitVector         &whichField);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                        Changed                               */
    /*! \{                                                                 */

    
    virtual void changed        (BitVector whichField, UInt32 origin);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                        Dump                                  */
    /*! \{                                                                 */

    
    virtual void dump(      UInt32    uiIndent = 0,
                      const BitVector bvFlags  = 0) const = 0;
    /*! \}                                                                 */

    /*---------------------------------------------------------------------*/
    /*! \name                comparison                                    */
    /*! \{                                                                 */
    
    struct IsEqualOptions
    {
        /*! Field names that are skipped during the comparison. */
        std::set<std::string> skipFields;

        /*! If not null, images with these attributes are skipped during the comparison. */
        int skipImageAttributes { 0};
        
        /*! Optional function to skip a field during comparison of two field containers.
            Arguments: the two field containers to compare and the field id to skip. Return true if the field should be skipped. */
        std::function<bool(const osg::FieldContainerPtr&, const osg::FieldContainerPtr&, unsigned int)> skipFieldIdCallback;
    };
    /*! \brief Compares this with another field container
        \param fc The field container to compare with
        \param options options for the comparison
        \param equalFCs a map of field container -> field containers that are equal. It is filled during the comparison.
        \return true if equal.
    */
    virtual bool equals(const osg::FieldContainerPtr & fc, const IsEqualOptions &options, std::map<osg::FieldContainerPtr, std::set<osg::FieldContainerPtr> > &equalFCs) const;
    static bool isEqual(const osg::FieldContainerPtr &a, const osg::FieldContainerPtr &b);
    static bool isEqual(const osg::FieldContainerPtr &a, const osg::FieldContainerPtr &b, const IsEqualOptions &options, std::map<osg::FieldContainerPtr, std::set<osg::FieldContainerPtr> > &equalFCs);
    /*! \}                                                                 */


    /*---------------------------------------------------------------------*/
    /*! \name                animation                                     */
    /*! \{                                                                 */

    virtual bool supportsAnimChannels() const;

    virtual void getAnimChannels(ChannelData& data);
    /*! \}                                                                 */


    /*---------------------------------------------------------------------*/
    /*! \name                CoredNodePtr helper                           */
    /*! \{                                                                 */

    /*! \ingroup GrpSystemFieldContainer
        Dummy class for CoredNodePtr instantiations on classes that are not
        NodeCores
    */

    class attempt_to_create_CoredNodePtr_on_non_NodeCore_FC
    {
        attempt_to_create_CoredNodePtr_on_non_NodeCore_FC() {}
    };


    
    void registerOnDestroyCallback( std::shared_ptr<OnDestroyCallback> const& callback);

    
    void unregisterOnDestroyCallback( std::shared_ptr<OnDestroyCallback> const& callback);

    
    void unregisterOnDestroyCallbacks();

    
    void registerChangedCallback( std::shared_ptr<ChangedCallback> const& callback);

    
    void unregisterChangedCallback( std::shared_ptr<ChangedCallback> const& callback);

    
    void unregisterChangedCallbacks();

#ifdef USE_CUSTOM_NEW_DELETE
    void* operator new( size_t size) throw ()
    {
        size = ( size == 0) ? 1 : size;
        return scalable_aligned_malloc( size, 32);
    }

    void* operator new[]( size_t size) throw()
    {
        return operator new(size);
    }

    void* operator new( size_t size, const std::nothrow_t&) throw()
    {
        if( size == 0)
            size = 1;

        return scalable_aligned_malloc( size, 32);

    }
    void* operator new[]( size_t size, const std::nothrow_t&) throw()
    {
        return operator new(size, std::nothrow);
    }

    void operator delete( void* ptr) throw()
    {
        if( ptr != 0)
            scalable_aligned_free( ptr);
    }

    void operator delete[]( void* ptr) throw()
    {
        operator delete( ptr);
    }

    void operator delete( void* ptr, const std::nothrow_t&) throw()
    {
        if( ptr != 0)
            scalable_aligned_free( ptr);
    }

    void operator delete[]( void* ptr, const std::nothrow_t&) throw()
    {
        operator delete( ptr, std::nothrow);
    }
#endif
    static const bool isNodeCore = false;



    /*! \}                                                                 */
    /*=========================  PROTECTED  ===============================*/

  protected:

    /*---------------------------------------------------------------------*/
    /*! \name                  Type information                            */
    /*! \{                                                                 */

    static FieldContainerType _type;

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                      Member                                  */
    /*! \{                                                                 */

    UInt32 _id;
    std::atomic<Int32> _refCount;

    std::vector< void*>*                                            _callbacks;
    void initCallbacks();
    void deleteCallbacks();
    std::vector< std::weak_ptr<OnDestroyCallback> >* getOnDestroyCallbacks() const;
    std::vector< std::weak_ptr<ChangedCallback> >* getChangeCallbacks() const;


    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                   Constructors                               */
    /*! \{                                                                 */

    
    FieldContainer(void);

    
    FieldContainer(const FieldContainer &obj);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                   Destructor                                 */
    /*! \{                                                                 */

    
    virtual ~FieldContainer (void);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name              Create Fieldcontainer                           */
    /*! \{                                                                 */

    template <class ObjectPtrT>
    static void newPtr(
                       ObjectPtrT                   &result,
        const typename ObjectPtrT::StoredObjectType *prototypeP);

    template <class ObjectPtrT>
    static void newPtr(ObjectPtrT &result);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                    MT Edit                                   */
    /*! \{                                                                 */

    template<class FieldTypeT>
    void beginEditX(const BitVector &whichField, FieldTypeT &field);

    template<class FieldTypeT>
    void endEditX  (const BitVector &whichField, FieldTypeT &field);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                MT Construction                               */
    /*! \{                                                                 */

    
    void onCreate      (const FieldContainer *source = NULL);

    
    void onCreateAspect(const FieldContainer *aspect0,
                        const FieldContainer *source = NULL);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                MT Destruction                                */
    /*! \{                                                                 */

    
    virtual void onDestroy      (void);

#if defined(OSG_FIXED_MFIELDSYNC)
    
    virtual void onDestroyAspect(UInt32 uiId, UInt32 uiAspect);
#endif

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                     Sync                                     */
    /*! \{                                                                 */
            void notifyChangeCallbacks(BitVector whichField, UInt32 origin);

#if !defined(OSG_FIXED_MFIELDSYNC)
    
    virtual void executeSync    (      FieldContainer &other,
                                 const BitVector      &whichField) = 0;

            
            void executeSyncImpl(      FieldContainer *pOther,
                                 const BitVector      &whichField);
#else
    
    virtual void executeSync       (      FieldContainer &other,
                                    const BitVector      &whichField,
                                    const SyncInfo       &iSinfo    ) = 0;

            
            void executeSyncImpl   (      FieldContainer *pOther,
                                    const BitVector      &whichField,
                                    const SyncInfo       &sInfo     );

    
    virtual void execBeginEdit     (const BitVector &whichField,
                                          UInt32     uiAspect,
                                          UInt32     uiContainerSize) = 0;

    
            void execBeginEditImpl (const BitVector &whichField,
                                          UInt32     uiAspect,
                                          UInt32     uiContainerSize);
#endif

    /*! \}                                                                 */
    /*==========================  PRIVATE  ================================*/

  private:

    friend class FieldContainerPtrBase;
    friend class FieldContainerPtr;

    /*!\brief prohibit default functions (move to 'public' if needed) */
    FieldContainer &operator = (const FieldContainer &other);
};

//---------------------------------------------------------------------------
//   Exported Types
//---------------------------------------------------------------------------

#define OSG_FC_TMPL_VAR_INL

#if 0
#ifdef FMSCOMPILER

// Macros used to derive a new fieldcontainer

/*! \ingroup GrpSystemFieldContainerX
 *  \relates FieldContainer
 *  \brief declare the first field
 */

#define OSG_FC_FIRST_FIELD_IDM_DECL(OSG_FIELDNAME)                            \
    static const osg::UInt32    OSG_FIELDNAME##Id;                            \
    static const osg::BitVector OSG_FIELDNAME##Mask;

/*! \ingroup GrpSystemFieldContainerX
 *  \relates FieldContainer
 *  \brief define the first field
 */

#define OSG_FC_FIRST_FIELD_IDM_DEF(OSG_CLASS, OSG_FIELDNAME)                  \
const osg::UInt32    OSG_CLASS::OSG_FIELDNAME##Id   =                         \
    Inherited::NextFieldId;                                                   \
const osg::BitVector OSG_CLASS::OSG_FIELDNAME##Mask =                         \
    Inherited::NextFieldMask;

/*! \ingroup GrpSystemFieldContainerX
 *  \relates FieldContainer
 *  \brief define the first field in template
 */

#define OSG_FC_FIRST_FIELD_IDM_INL_TMPL_DEF(OSG_CLASS,                        \
                                            OSG_TMPL_PARAM,                   \
                                            OSG_FIELDNAME)                    \
template <class OSG_TMPL_PARAM> OSG_FC_TMPL_VAR_INL                           \
const osg::UInt32    OSG_CLASS<OSG_TMPL_PARAM>::OSG_FIELDNAME##Id    =        \
    Inherited::NextFieldId;                                                   \
template <class OSG_TMPL_PARAM> OSG_FC_TMPL_VAR_INL                           \
const osg::BitVector OSG_CLASS<OSG_TMPL_PARAM>::OSG_FIELDNAME##Mask  =        \
    Inherited::NextFieldMask;

/*! \ingroup GrpSystemFieldContainerX
 *  \relates FieldContainer
 *  \brief declare a field
 */

#define OSG_FC_FIELD_IDM_DECL(OSG_FIELDNAME)                                  \
    static const osg::UInt32    OSG_FIELDNAME##Id;                            \
    static const osg::BitVector OSG_FIELDNAME##Mask;

/*! \ingroup GrpSystemFieldContainerX
 *  \relates FieldContainer
 *  \brief define a field
 */

#define OSG_FC_FIELD_IDM_DEF(OSG_CLASS, OSG_FIELDNAME, OSG_PREV_FIELDNAME)    \
const osg::UInt32    OSG_CLASS::OSG_FIELDNAME##Id    =                        \
    OSG_PREV_FIELDNAME##Id + 1;                                               \
const osg::BitVector OSG_CLASS::OSG_FIELDNAME##Mask  =                        \
    OSG_PREV_FIELDNAME##Mask << 1;

/*! \ingroup GrpSystemFieldContainerX
 *  \relates FieldContainer
 *  \brief define a field in template
 */

#define OSG_FC_FIELD_IDM_INL_TMPL_DEF(OSG_CLASS,                              \
                                      OSG_TMPL_PARAM,                         \
                                      OSG_FIELDNAME,                          \
                                      OSG_PREV_FIELDNAME)                     \
template <class OSG_TMPL_PARAM> OSG_FC_TMPL_VAR_INL                           \
const osg::UInt32    OSG_CLASS<OSG_TMPL_PARAM>::OSG_FIELDNAME##Id    =        \
    OSG_PREV_FIELDNAME##Id + 1;                                               \
template <class OSG_TMPL_PARAM> OSG_FC_TMPL_VAR_INL                           \
const osg::BitVector OSG_CLASS<OSG_TMPL_PARAM>::OSG_FIELDNAME##Mask  =        \
    OSG_PREV_FIELDNAME##Mask << 1;

/*! \ingroup GrpSystemFieldContainerX
 *  \brief declare the end of fields
 */

#define OSG_FC_LAST_FIELD_IDM_DECL                                            \
    static const osg::UInt32    NextFieldId;                                  \
    static const osg::BitVector NextFieldMask;

/*! \ingroup GrpSystemFieldContainerX
 *  \brief define the end of fields
 */

#define OSG_FC_LAST_FIELD_IDM_DEF(OSG_CLASS, OSG_PREV_FIELDNAME)              \
const osg::UInt32    OSG_CLASS::NextFieldId   =                               \
    OSG_PREV_FIELDNAME##Id + 1;                                               \
const osg::BitVector OSG_CLASS::NextFieldMask =                               \
    OSG_PREV_FIELDNAME##Mask << 1;

/*! \ingroup GrpSystemFieldContainerX
 *  \brief define the end of fields in template
 */

#define OSG_FC_LAST_FIELD_IDM_INL_TMPL_DEF(OSG_CLASS,                         \
                                           OSG_TMPL_PARAM,                    \
                                           OSG_PREV_FIELDNAME)                \
template <class OSG_TMPL_PARAM> OSG_FC_TMPL_VAR_INL                           \
const osg::UInt32    OSG_CLASS<OSG_TMPL_PARAM>::NextFieldId   =               \
    OSG_PREV_FIELDNAME##Id + 1;                                               \
template <class OSG_TMPL_PARAM> OSG_FC_TMPL_VAR_INL                           \
const osg::BitVector OSG_CLASS<OSG_TMPL_PARAM>::NextFieldMask =               \
    OSG_PREV_FIELDNAME##Mask << 1;

/*! \ingroup GrpSystemFieldContainerX
 *  \brief put the field information in the fielddescription constructor
 */
#else
#define OSG_FC_FIRST_FIELD_IDM_DECL(OSG_ELEMNAME)                             \
  static const osg::UInt32    OSG_ELEMNAME##Id   = Inherited::NextFieldId;    \
  static const osg::BitVector OSG_ELEMNAME##Mask = Inherited::NextFieldMask;

#define OSG_FC_FIELD_IDM_DECL(OSG_ELEMNAME, OSG_PREV_ELEMNAME)                \
  static const osg::UInt32    OSG_ELEMNAME##Id   = OSG_PREV_ELEMNAME##Id + 1; \
  static const osg::BitVector OSG_ELEMNAME##Mask = OSG_PREV_ELEMNAME##Mask <<1;

#define OSG_FC_LAST_FIELD_IDM_DECL(OSG_PREV_ELEMNAME)                         \
  static const osg::UInt32    NextFieldId   = OSG_PREV_ELEMNAME##Id + 1;      \
  static const osg::BitVector NextFieldMask = OSG_PREV_ELEMNAME##Mask << 1;


#define OSG_FC_FIRST_FIELD_IDM_DEF(OSG_CLASS, OSG_ELEMNAME)                   \
  const osg::UInt32    OSG_CLASS::OSG_ELEMNAME##Id;                           \
  const osg::BitVector OSG_CLASS::OSG_ELEMNAME##Mask;

#define OSG_FC_FIRST_FIELD_IDM_INL_TMPL_DEF(OSG_CLASS,                        \
                                            OSG_TMPL_PARAM,                   \
                                            OSG_ELEMNAME)                     \
  template <class OSG_TMPL_PARAM> OSG_FC_TMPL_VAR_INL                         \
  const osg::UInt32    OSG_CLASS<OSG_TMPL_PARAM>::OSG_ELEMNAME##Id;           \
  template <class OSG_TMPL_PARAM> OSG_FC_TMPL_VAR_INL                         \
  const osg::BitVector OSG_CLASS<OSG_TMPL_PARAM>::OSG_ELEMNAME##Mask;


#define OSG_FC_FIELD_IDM_DEF(OSG_CLASS, OSG_ELEMNAME, OSG_PREV_ELEMNAME)      \
  const osg::UInt32    OSG_CLASS::OSG_ELEMNAME##Id;                           \
  const osg::BitVector OSG_CLASS::OSG_ELEMNAME##Mask;

#define OSG_FC_FIELD_IDM_INL_TMPL_DEF(OSG_CLASS,                              \
                                      OSG_TMPL_PARAM,                         \
                                      OSG_ELEMNAME,                           \
                                      OSG_PREV_ELEMNAME)                      \
  template <class OSG_TMPL_PARAM> OSG_FC_TMPL_VAR_INL                         \
  const osg::UInt32    OSG_CLASS<OSG_TMPL_PARAM>::OSG_ELEMNAME##Id;           \
  template <class OSG_TMPL_PARAM> OSG_FC_TMPL_VAR_INL                         \
  const osg::BitVector OSG_CLASS<OSG_TMPL_PARAM>::OSG_ELEMNAME##Mask;

#define OSG_FC_LAST_FIELD_IDM_DEF(OSG_CLASS, OSG_PREV_ELEMNAME)               \
  const osg::UInt32    OSG_CLASS::NextFieldId;                                \
  const osg::BitVector OSG_CLASS::NextFieldMask;

#define OSG_FC_LAST_FIELD_IDM_INL_TMPL_DEF(OSG_CLASS,                         \
                                           OSG_TMPL_PARAM,                    \
                                           OSG_PREV_FIELDNAME)                \
  template <class OSG_TMPL_PARAM> OSG_FC_TMPL_VAR_INL                         \
  const osg::UInt32    OSG_CLASS<OSG_TMPL_PARAM>::NextFieldId;                \
  template <class OSG_TMPL_PARAM> OSG_FC_TMPL_VAR_INL                         \
  const osg::BitVector OSG_CLASS<OSG_TMPL_PARAM>::NextFieldMask;

#endif
#endif

#define OSG_FC_FIRST_FIELD_IDM_DECL(OSG_ELEMNAME)                             \
  enum { OSG_ELEMNAME##Id   = Inherited::NextFieldId };                       \
  static const osg::BitVector OSG_ELEMNAME##Mask;

#define OSG_FC_FIELD_IDM_DECL(OSG_ELEMNAME, OSG_PREV_ELEMNAME)                \
  enum { OSG_ELEMNAME##Id   = OSG_PREV_ELEMNAME##Id + 1 };                    \
  static const osg::BitVector OSG_ELEMNAME##Mask;

#define OSG_FC_LAST_FIELD_IDM_DECL(OSG_PREV_ELEMNAME)                         \
  enum { NextFieldId   = OSG_PREV_ELEMNAME##Id + 1 };                         \


#define OSG_FC_FIRST_FIELD_IDM_DEF(OSG_CLASS, OSG_ELEMNAME)                   \
  const osg::BitVector OSG_CLASS::OSG_ELEMNAME##Mask =                        \
      (TypeTraits<BitVector>::One << OSG_CLASS::OSG_ELEMNAME##Id);

#define OSG_FC_FIRST_FIELD_IDM_INL_TMPL_DEF(OSG_CLASS,                        \
                                            OSG_TMPL_PARAM,                   \
                                            OSG_ELEMNAME)                     \
  template <class OSG_TMPL_PARAM> OSG_FC_TMPL_VAR_INL                         \
  const osg::BitVector OSG_CLASS<OSG_TMPL_PARAM>::OSG_ELEMNAME##Mask =        \
    (TypeTraits<BitVector>::One <<                                \
        OSG_CLASS<OSG_TMPL_PARAM>::OSG_ELEMNAME##Id);


#define OSG_FC_FIELD_IDM_DEF(OSG_CLASS, OSG_ELEMNAME, OSG_PREV_ELEMNAME)      \
  const osg::BitVector OSG_CLASS::OSG_ELEMNAME##Mask =                        \
    (TypeTraits<BitVector>::One << OSG_CLASS::OSG_ELEMNAME##Id);

#define OSG_FC_FIELD_IDM_INL_TMPL_DEF(OSG_CLASS,                              \
                                      OSG_TMPL_PARAM,                         \
                                      OSG_ELEMNAME,                           \
                                      OSG_PREV_ELEMNAME)                      \
  template <class OSG_TMPL_PARAM> OSG_FC_TMPL_VAR_INL                         \
  const osg::BitVector OSG_CLASS<OSG_TMPL_PARAM>::OSG_ELEMNAME##Mask =        \
    (TypeTraits<BitVector>::One <<                                \
        OSG_CLASS<OSG_TMPL_PARAM>::OSG_ELEMNAME##);

#define OSG_FC_LAST_FIELD_IDM_DEF(OSG_CLASS, OSG_PREV_ELEMNAME)

#define OSG_FC_LAST_FIELD_IDM_INL_TMPL_DEF(OSG_CLASS,                         \
                                           OSG_TMPL_PARAM,                    \
                                           OSG_PREV_FIELDNAME)


#define OSG_FC_FIELD_IDM_DESC(OSG_FIELDNAME)                                  \
    OSG_FIELDNAME##Id, OSG_FIELDNAME##Mask

/*---------------------------- get type -------------------------------------*/

#define OSG_FC_GET_TYPE_DECL                                                  \
    virtual       osg::FieldContainerType &getType(void);                     \
    virtual const osg::FieldContainerType &getType(void) const;

#define OSG_FC_GET_TYPE_DEF(OSG_CLASS)                                        \
    osg::FieldContainerType &OSG_CLASS::getType(void)                         \
    {                                                                         \
        return _type;                                                         \
    }                                                                         \
    const osg::FieldContainerType &OSG_CLASS::getType(void) const             \
    {                                                                         \
        return _type;                                                         \
    }

#define OSG_FC_GET_TYPE_INL_TMPL_DEF(OSG_TMPL_PARAM, OSG_CLASS)               \
  template <class OSG_TMPL_PARAM> inline                                      \
  osg::FieldContainerType &OSG_CLASS<OSG_TMPL_PARAM>::getType(void)           \
  {                                                                           \
      return _type;                                                           \
  }                                                                           \
  template <class OSG_TMPL_PARAM> inline                                      \
  const osg::FieldContainerType &OSG_CLASS<OSG_TMPL_PARAM>::getType(          \
      void) const                                                             \
  {                                                                           \
      return _type;                                                           \
  }


#define OSG_FC_GET_STATIC_TYPE_DECL                                           \
    static osg::FieldContainerType & getClassType(void);

#define OSG_FC_GET_STATIC_TYPE_INL_DEF(OSG_CLASS)                             \
    inline                                                                    \
    osg::FieldContainerType &OSG_CLASS::getClassType(void)                    \
    {                                                                         \
        return _type;                                                         \
    }

#define OSG_FC_GET_STATIC_TYPE_INL_TMPL_DEF(OSG_TMPL_PARAM,                   \
                                            OSG_CLASS)                        \
    template <class OSG_TMPL_PARAM> inline                                    \
    osg::FieldContainerType &OSG_CLASS<OSG_TMPL_PARAM>::getClassType(void)    \
    {                                                                         \
        return _type;                                                         \
    }

#define OSG_FC_GET_STATIC_TYPE_ID_DECL                                        \
    static osg::UInt32 getClassTypeId(void);

#define OSG_FC_GET_STATIC_TYPE_ID_INL_DEF(OSG_CLASS)                          \
    inline                                                                    \
    osg::UInt32 OSG_CLASS::getClassTypeId(void)                               \
    {                                                                         \
        return _type.getId();                                                 \
    }

#define OSG_FC_GET_STATIC_TYPE_ID_INL_TMPL_DEF(OSG_TMPL_PARAM,                \
                                               OSG_CLASS)                     \
    template <class OSG_TMPL_PARAM> inline                                    \
    osg::UInt32 OSG_CLASS<OSG_TMPL_PARAM>::getClassTypeId(void)               \
    {                                                                         \
        return _type.getId();                                                 \
    }

#define OSG_FC_TYPE_FUNCTIONS_DECL                                            \
    OSG_FC_GET_TYPE_DECL                                                      \
    OSG_FC_GET_STATIC_TYPE_DECL                                               \
    OSG_FC_GET_STATIC_TYPE_ID_DECL

#define OSG_FC_TYPE_FUNCTIONS_DEF(OSG_CLASS)                                  \
    OSG_FC_GET_TYPE_DEF(OSG_CLASS)

#define OSG_FC_TYPE_FUNCTIONS_INL_DEF(OSG_CLASS)                              \
    OSG_FC_GET_STATIC_TYPE_INL_DEF   (OSG_CLASS)                              \
    OSG_FC_GET_STATIC_TYPE_ID_INL_DEF(OSG_CLASS)


#define OSG_FC_TYPE_FUNCTIONS_INL_TMPL_DEF(OSG_TMPL_PARAM, OSG_CLASS)         \
    OSG_FC_GET_TYPE_INL_TMPL_DEF          (OSG_TMPL_PARAM, OSG_CLASS)         \
    OSG_FC_GET_STATIC_TYPE_INL_TMPL_DEF   (OSG_TMPL_PARAM, OSG_CLASS)         \
    OSG_FC_GET_STATIC_TYPE_ID_INL_TMPL_DEF(OSG_TMPL_PARAM, OSG_CLASS)

#define OSG_FC_ST_TYPE_FUNCTIONS_INL_TMPL_DEF(OSG_TMPL_PARAM, OSG_CLASS)      \
    OSG_FC_GET_STATIC_TYPE_INL_TMPL_DEF   (OSG_TMPL_PARAM, OSG_CLASS)         \
    OSG_FC_GET_STATIC_TYPE_ID_INL_TMPL_DEF(OSG_TMPL_PARAM, OSG_CLASS)


/*---------------------------- create ---------------------------------------*/

#define OSG_FC_CREATE_DECL(OSG_CLASS_PTR)                                     \
    static OSG_CLASS_PTR create(void);

#define OSG_FC_CREATE_INL_DEF(OSG_CLASS, OSG_CLASS_PTR)                       \
    inline                                                                    \
    OSG_CLASS_PTR OSG_CLASS::create(void)                                     \
    {                                                                         \
        OSG_CLASS_PTR fc;                                                     \
                                                                              \
        if(getClassType().getPrototype() != osg::NullFC)                      \
         fc = OSG_CLASS_PTR::dcast(getClassType().getPrototype()->            \
                 shallowCopy());                                              \
                                                                              \
        return fc;                                                            \
    }

#define OSG_FC_CREATE_INL_TMPL_DEF(OSG_TMPL_PARAM,                            \
                                   OSG_CLASS,                                 \
                                   OSG_CLASS_PTR)                             \
    template <class OSG_TMPL_PARAM> inline                                    \
    typename OSG_CLASS<OSG_TMPL_PARAM>::OSG_CLASS_PTR                         \
        OSG_CLASS<OSG_TMPL_PARAM>::create(void)                               \
    {                                                                         \
        OSG_CLASS_PTR fc;                                                     \
                                                                              \
        if(getClassType().getPrototype() != osg::NullFC)                      \
         fc = OSG_CLASS_PTR::dcast(getClassType().getPrototype()->            \
                  shallowCopy());                                             \
                                                                              \
        return fc;                                                            \
    }

#define OSG_FC_CREATE_EMPTY_DECL(OSG_CLASS_PTR)                         \
    static OSG_CLASS_PTR createEmpty(void);

#define OSG_FC_CREATE_EMPTY_INL_DEF(OSG_CLASS, OSG_CLASS_PTR)           \
    inline                                                              \
    OSG_CLASS_PTR OSG_CLASS::createEmpty(void)                          \
    {                                                                   \
        OSG_CLASS_PTR returnValue;                                      \
                                                                        \
        newPtr(returnValue);                                            \
                                                                        \
        return returnValue;                                             \
    }

#define OSG_FC_CREATE_EMPTY_INL_TMPL_DEF(OSG_TMPL_PARAM,                \
                                         OSG_CLASS,                     \
                                         OSG_CLASS_PTR)                 \
    template <class OSG_TMPL_PARAM> inline                              \
    typename OSG_CLASS<OSG_TMPL_PARAM>::OSG_CLASS_PTR                   \
        OSG_CLASS<OSG_TMPL_PARAM>::createEmpty(void)                    \
    {                                                                   \
        OSG_CLASS_PTR returnValue;                                      \
                                                                        \
        Self::newPtr(returnValue);                                      \
                                                                        \
        return returnValue;                                             \
    }


#define OSG_FC_SHALLOWCOPY_DECL                                         \
    virtual osg::FieldContainerPtr shallowCopy(void) const;

#define OSG_FC_SHALLOWCOPY_DEF(OSG_CLASS, OSG_CLASS_PTR)                \
    osg::FieldContainerPtr OSG_CLASS::shallowCopy(void) const           \
    {                                                                   \
        OSG_CLASS_PTR returnValue;                                      \
                                                                        \
        newPtr(returnValue, this);                                      \
                                                                        \
        return returnValue;                                             \
    }

#define OSG_FC_SHALLOWCOPY_INL_TMPL_DEF(OSG_TMPL_PARAM,                       \
                                        OSG_CLASS,                            \
                                        OSG_CLASS_PTR)                        \
    template <class OSG_TMPL_PARAM> inline                                    \
    osg::FieldContainerPtr OSG_CLASS<OSG_TMPL_PARAM>::shallowCopy(void) const \
    {                                                                         \
        OSG_CLASS_PTR returnValue;                                            \
                                                                              \
        Self::newPtr(returnValue, this);                                      \
                                                                              \
        return returnValue;                                                   \
    }


#define OSG_FC_CREATE_FUNCTIONS_DECL(OSG_CLASS_PTR)                     \
    OSG_FC_CREATE_DECL      (OSG_CLASS_PTR)                             \
    OSG_FC_CREATE_EMPTY_DECL(OSG_CLASS_PTR)                             \
    OSG_FC_SHALLOWCOPY_DECL

#define OSG_FC_CREATE_FUNCTIONS_DEF(OSG_CLASS, OSG_CLASS_PTR)           \
    OSG_FC_SHALLOWCOPY_DEF(OSG_CLASS, OSG_CLASS_PTR)

#define OSG_FC_CREATE_FUNCTIONS_INL_DEF(OSG_CLASS, OSG_CLASS_PTR)       \
    OSG_FC_CREATE_INL_DEF      (OSG_CLASS, OSG_CLASS_PTR)               \
    OSG_FC_CREATE_EMPTY_INL_DEF(OSG_CLASS, OSG_CLASS_PTR)

#define OSG_FC_CREATE_FUNCTIONS_INL_TMPL_DEF(OSG_TMPL_PARAM,            \
                                             OSG_CLASS,                 \
                                             OSG_CLASS_PTR)             \
    OSG_FC_SHALLOWCOPY_INL_TMPL_DEF       (OSG_TMPL_PARAM,              \
                                     OSG_CLASS,                         \
                                     OSG_CLASS_PTR)                     \
    OSG_FC_CREATE_INL_TMPL_DEF      (OSG_TMPL_PARAM,                    \
                                     OSG_CLASS,                         \
                                     OSG_CLASS_PTR)                     \
    OSG_FC_CREATE_EMPTY_INL_TMPL_DEF(OSG_TMPL_PARAM,                    \
                                     OSG_CLASS,                         \
                                     OSG_CLASS_PTR)

/*-------------------------- get size ---------------------------------------*/


#define OSG_FC_GET_SIZE_DECL                                           \
    virtual osg::UInt32 getContainerSize(void) const;

#define OSG_FC_GET_SIZE_DEF(OSG_CLASS)                                 \
    osg::UInt32 OSG_CLASS::getContainerSize(void) const                \
    {                                                                  \
        return sizeof(OSG_CLASS);                                      \
    }

#define OSG_FC_GET_SIZE_INL_TMPL_DEF(OSG_TMPL_PARAM, OSG_CLASS)         \
    template <class OSG_TMPL_PARAM> inline                              \
    osg::UInt32 OSG_CLASS<OSG_TMPL_PARAM>::getContainerSize(void) const \
    {                                                                   \
        return sizeof(OSG_CLASS);                                       \
    }

#define OSG_FC_SIZE_FUNCTIONS_DECL                                     \
    OSG_FC_GET_SIZE_DECL

#define OSG_FC_SIZE_FUNCTIONS_DEF(OSG_CLASS)                           \
    OSG_FC_GET_SIZE_DEF(OSG_CLASS)

#define OSG_FC_SIZE_FUNCTIONS_INL_TMPL_DEF(OSG_TMPL_PARAM, OSG_CLASS)  \
    OSG_FC_GET_SIZE_INL_TMPL_DEF(OSG_TMPL_PARAM, OSG_CLASS)


/*------------------------ decl and def macros ------------------------------*/

#define OSG_FIELD_CONTAINER_DECL(OSG_CLASS_PTR)               \
    OSG_FC_TYPE_FUNCTIONS_DECL                                \
    OSG_FC_CREATE_FUNCTIONS_DECL(OSG_CLASS_PTR)               \
    OSG_FC_SIZE_FUNCTIONS_DECL

#define OSG_FIELD_CONTAINER_DEF(OSG_CLASS, OSG_CLASS_PTR)     \
    OSG_FC_TYPE_FUNCTIONS_DEF  (OSG_CLASS)                    \
    OSG_FC_CREATE_FUNCTIONS_DEF(OSG_CLASS, OSG_CLASS_PTR)     \
    OSG_FC_SIZE_FUNCTIONS_DEF  (OSG_CLASS)

#define OSG_FIELD_CONTAINER_INL_DEF(OSG_CLASS, OSG_CLASS_PTR) \
    OSG_FC_TYPE_FUNCTIONS_INL_DEF  (OSG_CLASS)                \
    OSG_FC_CREATE_FUNCTIONS_INL_DEF(OSG_CLASS, OSG_CLASS_PTR)

/*--------------------- abstr decl and def macros --------------------------*/

#define OSG_ABSTR_FIELD_CONTAINER_DECL(OSG_CLASS_PTR)               \
    OSG_FC_TYPE_FUNCTIONS_DECL                                      \
    OSG_FC_SIZE_FUNCTIONS_DECL

#define OSG_ABSTR_FIELD_CONTAINER_DEF(OSG_CLASS, OSG_CLASS_PTR)     \
    OSG_FC_TYPE_FUNCTIONS_DEF  (OSG_CLASS)                          \
    OSG_FC_SIZE_FUNCTIONS_DEF  (OSG_CLASS)

#define OSG_ABSTR_FIELD_CONTAINER_INL_DEF(OSG_CLASS, OSG_CLASS_PTR) \
    OSG_FC_TYPE_FUNCTIONS_INL_DEF  (OSG_CLASS)

/*------------------ template decl and def macros --------------------------*/

#define OSG_FIELD_CONTAINER_TMPL_DECL(OSG_CLASS_PTR)                 \
    OSG_FC_TYPE_FUNCTIONS_DECL                                       \
    OSG_FC_CREATE_FUNCTIONS_DECL(OSG_CLASS_PTR)                      \
    OSG_FC_SIZE_FUNCTIONS_DECL

#ifdef WIN32
#define OSG_FIELD_CONTAINER_INL_TMPL_DEF(OSG_CLASS,                  \
                                         OSG_TMPL_PARAM,             \
                                         OSG_CLASS_PTR)              \
    OSG_FC_SIZE_FUNCTIONS_INL_TMPL_DEF  (OSG_TMPL_PARAM, OSG_CLASS)  \
    OSG_FC_CREATE_FUNCTIONS_INL_TMPL_DEF(OSG_TMPL_PARAM,             \
                                         OSG_CLASS,                  \
                                         OSG_CLASS_PTR)
#else
#define OSG_FIELD_CONTAINER_INL_TMPL_DEF(OSG_CLASS,                  \
                                         OSG_TMPL_PARAM,             \
                                         OSG_CLASS_PTR)              \
    OSG_FC_TYPE_FUNCTIONS_INL_TMPL_DEF  (OSG_TMPL_PARAM, OSG_CLASS)  \
    OSG_FC_SIZE_FUNCTIONS_INL_TMPL_DEF  (OSG_TMPL_PARAM, OSG_CLASS)  \
    OSG_FC_CREATE_FUNCTIONS_INL_TMPL_DEF(OSG_TMPL_PARAM,             \
                                         OSG_CLASS,                  \
                                         OSG_CLASS_PTR)
#endif


#ifdef WIN32
#define OSG_FIELD_CONTAINER_ST_INL_TMPL_DEF(OSG_CLASS,                  \
                                            OSG_TMPL_PARAM,             \
                                            OSG_CLASS_PTR)              \
    OSG_FC_SIZE_FUNCTIONS_INL_TMPL_DEF     (OSG_TMPL_PARAM, OSG_CLASS)  \
    OSG_FC_CREATE_FUNCTIONS_INL_TMPL_DEF   (OSG_TMPL_PARAM,             \
                                            OSG_CLASS,                  \
                                            OSG_CLASS_PTR)
#else
#define OSG_FIELD_CONTAINER_ST_INL_TMPL_DEF(OSG_CLASS,                  \
                                            OSG_TMPL_PARAM,             \
                                            OSG_CLASS_PTR)              \
    OSG_FC_ST_TYPE_FUNCTIONS_INL_TMPL_DEF  (OSG_TMPL_PARAM, OSG_CLASS)  \
    OSG_FC_SIZE_FUNCTIONS_INL_TMPL_DEF     (OSG_TMPL_PARAM, OSG_CLASS)  \
    OSG_FC_CREATE_FUNCTIONS_INL_TMPL_DEF   (OSG_TMPL_PARAM,             \
                                            OSG_CLASS,                  \
                                            OSG_CLASS_PTR)
#endif

/*---------------abstr template decl and def macros ------------------------*/

#define OSG_ABSTR_FIELD_CONTAINER_TMPL_DECL(OSG_CLASS_PTR)           \
    OSG_FC_TYPE_FUNCTIONS_DECL                                       \
    OSG_FC_SIZE_FUNCTIONS_DECL

#define OSG_ABSTR_FIELD_CONTAINER_INL_TMPL_DEF(OSG_CLASS,            \
                                               OSG_TMPL_PARAM,       \
                                               OSG_CLASS_PTR)        \
    OSG_FC_TYPE_FUNCTIONS_INL_TMPL_DEF  (OSG_TMPL_PARAM, OSG_CLASS)  \
    OSG_FC_SIZE_FUNCTIONS_INL_TMPL_DEF  (OSG_TMPL_PARAM, OSG_CLASS)


/*--------------experimental--------------*/
/* macros for standard field access functions */

#define OSG_SFIELD_ACCESS_FUNCTIONS_DECL( OSG_FIELDTYPE, OSG_TYPE,      \
        OSG_NAME )                                                      \
    inline OSG_FIELDTYPE * getSF##OSG_NAME( void );                         \
    inline OSG_TYPE get##OSG_NAME( void );                                  \
    inline OSG_TYPE get##OSG_NAME( void ) const;                        \
    inline void set##OSG_NAME( const OSG_TYPE value );


#define OSG_SFIELD_ACCESS_FUNCTIONS_INL( OSG_CLASS, OSG_FIELDTYPE,      \
        OSG_TYPE, OSG_NAME )                                            \
    inline OSG_FIELDTYPE * OSG_CLASS::getSF##OSG_NAME( void )           \
    {                                                                   \
        return &_##OSG_NAME;                                            \
    }                                                                   \
    inline OSG_TYPE OSG_CLASS::get##OSG_NAME(void)                      \
    {                                                                   \
        return _##OSG_NAME.getValue();                                  \
    }                                                                   \
    inline OSG_TYPE OSG_CLASS::get##OSG_NAME(void) const                \
    {                                                                   \
        return _##OSG_NAME.getValue();                                  \
    }                                                                   \
    inline void OSG_CLASS::set##OSG_NAME( const OSG_TYPE val)           \
    {                                                                   \
        _##OSG_NAME.setValue( val );                                    \
    }

OSG_END_NAMESPACE

#define OSGFIELDCONTAINER_HEADER_CVSID "@(#)$Id: $"

#endif /* _OSGFIELDCONTAINERIMPL_H_ */
