/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *                     Copyright 2000-2002 by OpenSG Forum                   *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

#ifndef _OSGNODEIMPL_H_
#define _OSGNODEIMPL_H_
#pragma once

#ifdef OSG_DOC_FILES_IN_MODULE
/*! \file OSGNodeImpl.h
    \ingroup GrpSystemFieldContainer
 */
#endif

#include <string>
#include <vector>
#include <set>

#include <OSGSystemDef.h>
#include <OSGBaseTypes.h>
#include <OSGMatrix.h>
#include <OSGFieldContainer.h>
#include <OSGSFSysTypes.h>
#include <OSGSFBaseTypes.h>

#include <OSGAttachmentContainer.h>
#include <OSGSFNodeCorePtr.h>
#include <OSGSFNodePtrImpl.h>
#include <OSGSFNodePtrDepImpl.h>
#include <OSGMFNodePtrImpl.h>
#include <OSGMFNodePtrDepImpl.h>

OSG_BEGIN_NAMESPACE

class NodeCore;
class BinaryDataHandler;
class FieldDescription;

/*! \ingroup GrpSystemFieldContainer
 */

class OSG_SYSTEMLIB_DLLMAPPING Node : public AttachmentContainer 
{
    /*=========================  PROTECTED  ===============================*/

  protected:

    typedef AttachmentContainer Inherited;

    /*==========================  PUBLIC  =================================*/

  public:

    enum 
    { 
        VolumeFieldId        = Inherited::NextFieldId, 
        TravMaskFieldId      = VolumeFieldId         + 1,
        ParentFieldId        = TravMaskFieldId       + 1,
        ChildrenFieldId      = ParentFieldId         + 1,
        CoreFieldId          = ChildrenFieldId       + 1,
        FlagsFieldId         = CoreFieldId           + 1,
        ExternalFlagsFieldId = FlagsFieldId + 1,
        IndexFieldId         = ExternalFlagsFieldId  + 1,
        NextFieldId          = IndexFieldId  + 1
    };


    // internal flags - "flags" field
    enum
    {
        NFSelected            = 1 << 0,
        NFLoaded              = 1 << 1,
        NFPersistent          = 1 << 2,
        NFInternal            = 1 << 3,
        NFAuxiliary           = 1 << 4,
        NFSingleSided         = 1 << 5,
        NFTempSelected        = 1 << 6,
        NFMaterialBall        = 1 << 7,
        NFSelectableAuxiliary = 1 << 8,
        NFManipulator       = 1 << 9,
        NFLightsource       = 1 << 10,
        NFCamera            = 1 << 11,
        NFTrash             = 1 << 12, // nodes in the trash should get this flag
        NFSelectionVisualization = 1 << 13,
        NFShadowCastingAuxiliary = 1 << 14,
        NFForceUpdate = 1 << 15,
        NFNotRenderedInDisplayCluster = 1 << 16,
        NFForceIsolateVisible = 1 << 17,
        NFForceIsolateInvisible = 1 << 18,
        NFForceIsolateChoiceAll = 1 << 19,
        NFInternalVisibleInRaytracing = 1 << 20,
        NFModuleGraphRoot = 1 << 21, // a root node a module graph in the material pool
        NFSceneRoot = 1 << 22, // Is the scenegraph root node
        NFShellWithNamedComponents = 1 << 23, // components that can be switched by variants
        NFIgnoreInRenderAPI = 1 << 24, // This node should be ignored in the RenderAPI
        NFManagedInRenderAPI = 1 << 25, // This node is handled by the RenderAPI and should be ignored by OpenGL
        NFOccludedAuxiliary = 1 << 26, // This auxiliary node is occluded by scene geometry
        NFRootVisualization = 1 << 27 // This is a root for a visualization (expected to not have a parent but still visible)
    };


    // external (saved) flags - "externalFlags" field
    enum
    {
        NFUnselectable        = 1 << 0,
        NFInternalLightsource = 1 << 1,
        NFInternalCamera      = 1 << 2,
        NFTrackingUser        = 1 << 3,
        NFLightVisibilityReference = 1 << 4,
        NFLocked = 1 << 5,
        NFTeleportTarget = 1 << 6
    };

    static const BitVector VolumeFieldMask;
    static const BitVector TravMaskFieldMask;
    static const BitVector ParentFieldMask;
    static const BitVector ChildrenFieldMask;
    static const BitVector CoreFieldMask;
    static const BitVector FlagsFieldMask;
    static const BitVector ExternalFlagsFieldMask;
    static const BitVector IndexFieldMask;

    typedef NodePtr Ptr;

    typedef BoxVolume     VolumeType;

    typedef MFNodePtr MFChildrenType;

    /*---------------------------------------------------------------------*/
    /*! \name        General Fieldcontainer Declaration                    */
    /*! \{                                                                 */

    OSG_FIELD_CONTAINER_DECL(NodePtr)

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                        Core                                  */
    /*! \{                                                                 */

    NodeCorePtr getCore(      void             );
    NodeCorePtr getCore(      void             ) const;

    void        setCore(const NodeCorePtr &core);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                       Parent                                 */
    /*! \{                                                                 */

    NodePtr getParent(void);
    void setParent(const NodePtr &parent);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                       Children                               */
    /*! \{                                                                 */

    SizeT   getNChildren  (void                     ) const;
    
    void    addChild      (const NodePtr &childP    );
    void    moveChildren   ( NodePtr& nodeP);

    void    insertChild   (      UInt32   childIndex, 
                           const NodePtr &childP    );

    void    insertChild   (const NodePtr &posChildP, 
                           const NodePtr &childP,    
                           bool after = false);

    void    replaceChild  (      UInt32   childIndex,    
                           const NodePtr &childP    );

    bool    replaceChildBy(const NodePtr &childP, 
                           const NodePtr &newChildP );

    Int32   findChild     (const NodePtr &childP    ) const;

    void    subChild      (const NodePtr &childP    );
    void    subChild      (      UInt32   childIndex);

    NodePtr getChild      (      UInt32   childIndex);

    void    clearChildren (void);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                  Active / TravMask                           */
    /*! \{                                                                 */

    bool   getActive  (void      ) const;
    void   setActive  (bool   val);

    bool   getActiveEvalFlags(void      ) const;
    bool   getVolumeActive(void      ) const;

    void   setTravMask(UInt32 val);
    UInt32 getTravMask(void      ) const;

    void   setFlags(UInt32 val);
    UInt32 getFlags(void      ) const;
    bool   hasFlag(UInt32 flag) const;

    void   setExternalFlags(UInt32 flags);
    UInt32 getExternalFlags(void      ) const;

    void   addExternalFlag(UInt32 flag);
    void   removeExternalFlag(UInt32 flag);
    bool   hasExternalFlag(UInt32 flag) const;

    void   setIndex(UInt32 val);
    UInt32 getIndex(void      ) const;

    void   setOcclusionMask(UInt8 val);
    UInt8  getOcclusionMask(void      ) const;

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                       Children                               */
    /*! \{                                                                 */

    NodePtr clone(void);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                     Access Fields                            */
    /*! \{                                                                 */

    const SFBoxVolume     *getSFVolume  (void) const;
          SFBoxVolume     *editSFVolume (void);

    const SFUInt32        *getSFTravMask(void) const;
          SFUInt32        *editSFTravMask(void);

    const SFUInt32        *getSFFlags(void) const;
          SFUInt32        *editSFFlags(void);

    const SFNodePtr       *getSFParent  (void) const;

    const SFNodeCorePtr   *getSFCore    (void) const;
          SFNodeCorePtr   *editSFCore    (void);

    const MFNodePtr       *getMFChildren(void) const;
          MFNodePtr       *editMFChildren(void);

    const SFNodePtr       *getSFAssociated(void) const;
          SFNodePtr       *editSFAssociated(void);

    const SFUInt32        *getSFExternalFlags(void) const;
          SFUInt32        *editSFExternalFlags(void);

    const SFUInt32        *getSFIndex(void) const;
          SFUInt32        *editSFIndex(void);

#ifndef OSG_2_PREP
    //SFDynamicVolume *getSFVolume  (void);
    SFUInt32        *getSFTravMask(void);
    SFUInt32        *getSFFlags   (void);
    SFNodePtr       *getSFParent  (void);
    SFNodeCorePtr   *getSFCore    (void);
    MFNodePtr       *getMFChildren(void);
    SFNodePtr       *getSFAssociated(void);
    SFUInt32        *getSFExternalFlags(void);
    SFUInt32        *getSFIndex(void);
#endif

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                    Transformation                            */
    /*! \{                                                                 */

    Matrix getToWorld(void          );
    
    void   getToWorld(Matrix &result);

    
    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                       Volume                                 */
    /*! \{                                                                 */
    

    

           BoxVolume     &getVolume       (bool update          );
           BoxVolume     &editVolume      (bool update          );
    const  BoxVolume     &getVolume       (void                 ) const;


           void           getWorldVolume  (BoxVolume     &result);
    
           void           updateVolume    (void                 );

           void           invalidateVolume(void                 );

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                        Changed                               */
    /*! \{                                                                 */

    virtual void changed(BitVector  whichField, 
                         UInt32     origin    );

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                   Binary Access                              */
    /*! \{                                                                 */

    virtual SizeT   getBinSize (const BitVector         &whichField);

    virtual void    copyToBin  (      BinaryDataHandler &pMem, 
                                const BitVector         &whichField);
    virtual void    copyFromBin(      BinaryDataHandler &pMem, 
                                const BitVector         &whichField);
    
    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                        Dump                                  */
    /*! \{                                                                 */

    virtual void dump(      UInt32    uiIndent = 0, 
                      const BitVector bvFlags  = 0) const;

    /*! \}                                                                 */

    bool getSelfLocked();
    bool getSelfUneditedAsset();
    bool getCoreIsLockedByAncestor();
    void invalidateSelfLockCache();

    /*=========================  PROTECTED  ===============================*/

  protected:

    friend class FieldContainer;
    friend class FieldContainerType;

    /*---------------------------------------------------------------------*/
    /*! \name                  Type information                            */
    /*! \{                                                                 */

    static       FieldDescription   *_desc[];
    static       FieldContainerType  _type;

    static const NodePtr              NullNode;

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                      Fields                                  */
    /*! \{                                                                 */


    SFBoxVolume     _sfVolume;

    
    SFUInt32        _sfTravMask;
    
    SFNodePtr       _sfParent;
    MFNodePtr       _mfChildren;

    SFNodeCorePtr   _sfCore;

    SFUInt32        _sfFlags;
    SFUInt32        _sfExternalFlags;
    SFUInt32        _sfIndex;

    UInt8           _occlusionMask;

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                   Constructors                               */
    /*! \{                                                                 */

    Node(void);
    Node(const Node &source);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                   Destructor                                 */
    /*! \{                                                                 */

    virtual ~Node (void);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                MT Construction                               */
    /*! \{                                                                 */

    SFNodePtr *editSFParent(void);

    void onCreate (const Node    *source = NULL);

#if defined(OSG_FIXED_MFIELDSYNC)
    virtual void onDestroyAspect(UInt32 uiId, UInt32 uiAspect);
#endif

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                     Sync                                     */
    /*! \{                                                                 */

#if !defined(OSG_FIXED_MFIELDSYNC)
    virtual void executeSync    (      FieldContainer &other,
                                 const BitVector      &whichField);

            void executeSyncImpl(      Node           *pOther,
                                 const BitVector      &whichField);
#else
    virtual void executeSync       (      FieldContainer &other,
                                    const BitVector      &whichField,
                                    const SyncInfo       &sInfo     );

            void executeSyncImpl   (      Node           *pOther,
                                    const BitVector      &whichField,
                                    const SyncInfo       &sInfo     );

    virtual void execBeginEdit     (const BitVector &whichField, 
                                          UInt32     uiAspect,
                                          UInt32     uiContainerSize);

            void execBeginEditImpl (const BitVector &whichField, 
                                          UInt32     uiAspect,
                                          UInt32     uiContainerSize);
#endif

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                     Pointer                                  */
    /*! \{                                                                 */

    NodePtr getPtr(void) const;

    /*! \}                                                                 */
    /*==========================  PRIVATE  ================================*/

  private:

    /* prohibit default function (move to 'public' if needed) */
    void operator =(const Node &source);
};


/*! \ingroup GrpSystemFieldContainerFuncs
 */

using CloneTreeNodeMapper = std::map<osg::UInt32, osg::UInt32>;

OSG_SYSTEMLIB_DLLMAPPING
NodePtr cloneTree(const NodePtr &pRootNode, bool cloneAttachments = false, CloneTreeNodeMapper* nodeMapper = nullptr);

OSG_SYSTEMLIB_DLLMAPPING
FieldContainerPtr deepClone(const FieldContainerPtr &src,
                            const std::vector<std::string> &share);

OSG_SYSTEMLIB_DLLMAPPING
FieldContainerPtr deepClone(const FieldContainerPtr &src,
                            const std::vector<UInt16> &shareGroupIds);

OSG_SYSTEMLIB_DLLMAPPING
FieldContainerPtr deepClone(const FieldContainerPtr &src,
                            const std::string &shareString = "");

OSG_SYSTEMLIB_DLLMAPPING
void deepCloneAttachments(const AttachmentContainerPtr &src,
                          AttachmentContainerPtr dst,
                          const std::vector<std::string> &share);

OSG_SYSTEMLIB_DLLMAPPING
void deepCloneAttachments(const AttachmentContainerPtr &src,
                          AttachmentContainerPtr dst,
                          const std::vector<UInt16> &shareGroupIds);

OSG_SYSTEMLIB_DLLMAPPING
void deepCloneAttachments(const NodePtr &src, NodePtr &dst,
                          const std::string &shareString = "");
                      
OSG_SYSTEMLIB_DLLMAPPING
NodePtr deepCloneTree(const NodePtr &src,
                      const std::vector<std::string> &share);

OSG_SYSTEMLIB_DLLMAPPING
NodePtr deepCloneTree(const NodePtr &src,
                      const std::vector<UInt16> &shareGroupIds);

OSG_SYSTEMLIB_DLLMAPPING
NodePtr deepCloneTree(const NodePtr &src,
                      const std::string &shareString = "");


OSG_SYSTEMLIB_DLLMAPPING
void addAttachments(const AttachmentContainerPtr &src,
                    AttachmentContainerPtr dst);

OSG_SYSTEMLIB_DLLMAPPING
void replaceAttachment(AttachmentContainerPtr parent, 
    osg::AttachmentMap::iterator iter, 
    osg::AttachmentPtr newAtt);


OSG_SYSTEMLIB_DLLMAPPING
NodePtr deepCloneTreeKeepShared(const NodePtr &src,
                                const std::vector<std::string> &share,
                                const std::set<osg::NodeCorePtr>& excludedNodeCores);

OSG_SYSTEMLIB_DLLMAPPING
NodePtr deepCloneTreeKeepShared(const NodePtr &src,
                                const std::string &shareString,
                                const std::set<osg::NodeCorePtr>& excludedNodeCores);

using FieldContainerTypes = std::vector<const FieldContainerType*>;

OSG_SYSTEMLIB_DLLMAPPING
FieldContainerTypes convertShareTypes(const std::vector<std::string> &shares);

OSG_SYSTEMLIB_DLLMAPPING
bool shouldShareContainer(const FieldContainerPtr &fc, const FieldContainerTypes &share, const FieldContainerPtr &parentFC = osg::NullFC, const FieldDescription* fdesc = nullptr);

OSG_END_NAMESPACE

#define OSGNODE_HEADER_CVSID "@(#)$Id: $"

#endif /* _OSGNODEIMPL_H_ */
