/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *                     Copyright 2000-2002 by OpenSG Forum                   *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

#ifndef _OSGNODEIMPL_INL_
#define _OSGNODEIMPL_INL_

#ifdef OSG_DOC_FILES_IN_MODULE
/*! \file OSGNodeImpl.inl
    \ingroup GrpSystemFieldContainer
 */
#endif

#include <stdlib.h>
#include <stdio.h>

#include "OSGConfig.h"

OSG_BEGIN_NAMESPACE

/*-------------------------------------------------------------------------*/
/*                                Get                                      */


FORCEINLINE
const BoxVolume &Node::getVolume(void) const
{
    return _sfVolume.getValue();
}

inline
BoxVolume &Node::getVolume(bool update)
{
    if(update == true)
        updateVolume();

    return _sfVolume.getValue();
}

inline
BoxVolume &Node::editVolume(bool update)
{
    if(update == true)
        updateVolume();

    return _sfVolume.getValue();
}

FORCEINLINE
NodePtr Node::getParent(void)
{
    return _sfParent.getValue();
}

FORCEINLINE
SizeT Node::getNChildren(void) const
{
    return _mfChildren.size();
}
 
FORCEINLINE
NodePtr Node::getChild(UInt32 childIndex)
{
    OSG_ASSERT((childIndex < _mfChildren.size()));

    return _mfChildren[childIndex];
}

FORCEINLINE
UInt32 Node::getTravMask(void) const
{
    return _sfTravMask.getValue();
}

FORCEINLINE
void  Node::setTravMask(UInt32 val)
{
    _sfTravMask.setValue(val);
}

FORCEINLINE
UInt32 Node::getFlags(void) const
{
    return _sfFlags.getValue();
}

FORCEINLINE
void  Node::setFlags(UInt32 val)
{
    _sfFlags.setValue(val);
}

FORCEINLINE
bool  Node::hasFlag(UInt32 flag) const
{
    return (getFlags() & flag) != 0;
}

FORCEINLINE
bool Node::getActive(void) const
{
    return getTravMask() == TypeTraits<UInt32>::getMax();
}

FORCEINLINE
void  Node::setActive(bool val)
{
    _sfTravMask.setValue(val ? TypeTraits<UInt32>::getMax()        :
                               TypeTraits<UInt32>::getZeroElement() );
}

FORCEINLINE
bool  Node::getActiveEvalFlags(void) const
{
    return (getActive() || (getFlags() & Node::NFForceIsolateVisible)) && !(getFlags() & Node::NFForceIsolateInvisible);
}

FORCEINLINE
bool  Node::getVolumeActive(void) const
{
    return (getActive() || hasFlag(Node::NFForceIsolateVisible));
}


FORCEINLINE
void Node::setOcclusionMask(UInt8 val)
{
    _occlusionMask = val;
}

FORCEINLINE
UInt8 Node::getOcclusionMask(void) const
{
    return _occlusionMask;
}

FORCEINLINE
    UInt32 Node::getExternalFlags(void) const
{
    return _sfExternalFlags.getValue();
}

FORCEINLINE
    void  Node::setExternalFlags(UInt32 flags)
{
    _sfExternalFlags.setValue(flags);
}

FORCEINLINE
    void  Node::addExternalFlag(UInt32 flag)
{
    setExternalFlags(getExternalFlags() | flag);
}

FORCEINLINE
    void  Node::removeExternalFlag(UInt32 flag)
{
    setExternalFlags(getExternalFlags() & ~flag);
}

FORCEINLINE
    bool  Node::hasExternalFlag(UInt32 flag) const
{
    return (getExternalFlags() & flag) != 0;
}

FORCEINLINE
void  Node::setIndex(UInt32 val)
{
    _sfIndex.setValue(val);
}

FORCEINLINE
UInt32 Node::getIndex(void) const
{
    return _sfIndex.getValue();
}

/*-------------------------------------------------------------------------*/
/*                          Access Fields                                  */


inline
const SFBoxVolume *Node::getSFVolume(void) const
{
    return &_sfVolume;
}

inline
SFBoxVolume *Node::editSFVolume(void)
{
    return &_sfVolume;
}


inline
const SFUInt32 *Node::getSFTravMask(void) const
{
    return &_sfTravMask;
}

inline
SFUInt32 *Node::editSFTravMask(void)
{
    return &_sfTravMask;
}

#ifndef OSG_2_PREP
inline
SFUInt32 *Node::getSFTravMask(void)
{
    return &_sfTravMask;
}
#endif

inline
const SFUInt32 *Node::getSFFlags(void) const
{
    return &_sfFlags;
}

inline
SFUInt32 *Node::editSFFlags(void)
{
    return &_sfFlags;
}

#ifndef OSG_2_PREP
inline
SFUInt32 *Node::getSFFlags(void)
{
    return &_sfFlags;
}
#endif

inline
const SFNodePtr *Node::getSFParent(void) const
{
    return &_sfParent;
}

inline
SFNodePtr *Node::editSFParent(void)
{
    return &_sfParent;
}

#ifndef OSG_2_PREP
inline
SFNodePtr *Node::getSFParent(void)
{
    return &_sfParent;
}
#endif

inline
const SFNodeCorePtr *Node::getSFCore(void) const
{
    return &_sfCore;
}

inline
SFNodeCorePtr *Node::editSFCore(void)
{
    return &_sfCore;
}

#ifndef OSG_2_PREP
inline
SFNodeCorePtr *Node::getSFCore(void)
{
    return &_sfCore;
}
#endif

inline
const MFNodePtr *Node::getMFChildren(void) const
{
    return &_mfChildren;
}

inline
MFNodePtr *Node::editMFChildren(void)
{
    return &_mfChildren;
}

#ifndef OSG_2_PREP
inline
MFNodePtr *Node::getMFChildren(void)
{
    return &_mfChildren;
}
#endif

inline
    const SFUInt32 *Node::getSFExternalFlags(void) const
{
    return &_sfExternalFlags;
}

inline
    SFUInt32 *Node::editSFExternalFlags(void)
{
    return &_sfExternalFlags;
}

#ifndef OSG_2_PREP
inline
    SFUInt32 *Node::getSFExternalFlags(void)
{
    return &_sfExternalFlags;
}
#endif

inline
    const SFUInt32 *Node::getSFIndex(void) const
{
    return &_sfIndex;
}

inline
    SFUInt32 *Node::editSFIndex(void)
{
    return &_sfIndex;
}

#ifndef OSG_2_PREP
inline
    SFUInt32 *Node::getSFIndex(void)
{
    return &_sfIndex;
}
#endif


/*-------------------------------------------------------------------------*/
/*                           Binary Interface                              */

inline
SizeT Node::getBinSize(const BitVector &whichField)
{
    SizeT returnValue = Inherited::getBinSize(whichField);

    if(FieldBits::NoField != (VolumeFieldMask & whichField))
    {
        returnValue += _sfVolume       .getBinSize();
    }

    if(FieldBits::NoField != (TravMaskFieldMask & whichField))
    {
        returnValue += _sfTravMask     .getBinSize();
    }

    if(FieldBits::NoField != (FlagsFieldMask & whichField))
    {
        returnValue += _sfFlags        .getBinSize();
    }

    if(FieldBits::NoField != (ParentFieldMask & whichField))
    {
        returnValue += _sfParent       .getBinSize();
    }

    if(FieldBits::NoField != (ChildrenFieldMask & whichField))
    {
        returnValue += _mfChildren     .getBinSize();
    }

    if(FieldBits::NoField != (CoreFieldMask & whichField))
    {
        returnValue += _sfCore         .getBinSize();
    }

    if(FieldBits::NoField != (ExternalFlagsFieldMask & whichField))
    {
        returnValue += _sfExternalFlags.getBinSize();
    }

    return returnValue;
}

inline
void Node::copyToBin(      BinaryDataHandler &pMem,
                     const BitVector         &whichField)
{
    Inherited::copyToBin(pMem, whichField);

    if(FieldBits::NoField != (VolumeFieldMask & whichField))
    {
        _sfVolume.copyToBin(pMem);
    }

    if(FieldBits::NoField != (TravMaskFieldMask & whichField))
    {
        _sfTravMask.copyToBin(pMem);
    }

    if(FieldBits::NoField != (FlagsFieldMask & whichField))
    {
        _sfFlags.copyToBin(pMem);
    }

    if(FieldBits::NoField != (ParentFieldMask & whichField))
    {
        _sfParent.copyToBin(pMem);
    }

    if(FieldBits::NoField != (ChildrenFieldMask & whichField))
    {
        _mfChildren.copyToBin(pMem);
    }

    if(FieldBits::NoField != (CoreFieldMask & whichField))
    {
        _sfCore.copyToBin(pMem);
    }

    if(FieldBits::NoField != (ExternalFlagsFieldMask & whichField))
    {
        _sfExternalFlags.copyToBin(pMem);
    }
}

inline
void Node::copyFromBin(      BinaryDataHandler &pMem,
                       const BitVector         &whichField)
{
    Inherited::copyFromBin(pMem, whichField);

    if(FieldBits::NoField != (VolumeFieldMask & whichField))
    {
        _sfVolume.copyFromBin(pMem);
    }

    if(FieldBits::NoField != (TravMaskFieldMask & whichField))
    {
        _sfTravMask.copyFromBin(pMem);
    }

    if(FieldBits::NoField != (FlagsFieldMask & whichField))
    {
        _sfFlags.copyFromBin(pMem);
    }

    if(FieldBits::NoField != (ParentFieldMask & whichField))
    {
        _sfParent.copyFromBin(pMem);
    }

    if(FieldBits::NoField != (ChildrenFieldMask & whichField))
    {
        _mfChildren.copyFromBin(pMem);
    }

    if(FieldBits::NoField != (CoreFieldMask & whichField))
    {
        _sfCore.copyFromBin(pMem);
    }

    if(FieldBits::NoField != (ExternalFlagsFieldMask & whichField))
    {
        _sfExternalFlags.copyFromBin(pMem);
    }
}


/*-------------------------------------------------------------------------*/
/*                           MT Construction                               */

inline
void Node::setParent(const NodePtr &parent)
{
    _sfParent.setValue(parent);
}

/*-------------------------------------------------------------------------*/
/*                                Sync                                     */

#if !defined(OSG_FIXED_MFIELDSYNC)
inline
void Node::executeSyncImpl(      Node      *pOther,
                           const BitVector &whichField)
{
    Inherited::executeSyncImpl(pOther, whichField);

    if (FieldBits::NoField != (VolumeFieldMask & whichField))
    {
        _sfVolume.syncWith(pOther->_sfVolume);
    }

    if (FieldBits::NoField != (TravMaskFieldMask & whichField))
    {
        _sfTravMask.syncWith(pOther->_sfTravMask);
    }

    if (FieldBits::NoField != (FlagsFieldMask & whichField))
    {
        _sfFlags.syncWith(pOther->_sfFlags);
    }

    if (FieldBits::NoField != (ParentFieldMask & whichField))
    {
        _sfParent.syncWith(pOther->_sfParent);
    }

    if (FieldBits::NoField != (ChildrenFieldMask & whichField))
    {
        _mfChildren.syncWith(pOther->_mfChildren);
    }

    if (FieldBits::NoField != (CoreFieldMask & whichField))
    {
        _sfCore.syncWith(pOther->_sfCore);
    }

    if (FieldBits::NoField != (ExternalFlagsFieldMask & whichField))
    {
        _sfExternalFlags.syncWith(pOther->_sfExternalFlags);
    }
}

inline
void Node::executeSync(      FieldContainer &other,
                       const BitVector      &whichField)
{
    this->executeSyncImpl(static_cast<Node *>(&other), whichField);
}
#else
inline
void Node::executeSyncImpl(      Node      *pOther,
                           const BitVector &whichField,
                           const SyncInfo  &sInfo     )
{
    Inherited::executeSyncImpl(pOther, whichField, sInfo);

    if (FieldBits::NoField != (VolumeFieldMask & whichField))
    {
        _sfVolume.syncWith(pOther->_sfVolume);
    }

    if (FieldBits::NoField != (TravMaskFieldMask & whichField))
    {
        _sfTravMask.syncWith(pOther->_sfTravMask);
    }

    if (FieldBits::NoField != (FlagsFieldMask & whichField))
    {
        _sfFlags.syncWith(pOther->_sfFlags);
    }

    if (FieldBits::NoField != (ParentFieldMask & whichField))
    {
        _sfParent.syncWith(pOther->_sfParent);
    }

    if (FieldBits::NoField != (ChildrenFieldMask & whichField))
    {
        _mfChildren.syncWith(pOther->_mfChildren, sInfo);
    }

    if (FieldBits::NoField != (CoreFieldMask & whichField))
    {
        _sfCore.syncWith(pOther->_sfCore);
    }

    if (FieldBits::NoField != (ExternalFlagsFieldMask & whichField))
    {
        _sfExternalFlags.syncWith(pOther->_sfExternalFlags);
    }
}

inline
void Node::executeSync(      FieldContainer &other,
                       const BitVector      &whichField,
                       const SyncInfo       &sInfo     )
{
    this->executeSyncImpl((Node *) &other, whichField, sInfo);
}

inline
void Node::execBeginEditImpl (const BitVector &whichField, 
                                    UInt32     uiAspect,
                                    UInt32     uiContainerSize)
{
    Inherited::execBeginEditImpl(whichField, uiAspect, uiContainerSize);

    if (FieldBits::NoField != (ChildrenFieldMask & whichField))
    {
        _mfChildren.beginEdit(uiAspect, uiContainerSize);
    }
}

inline
void Node::execBeginEdit(const BitVector &whichField, 
                               UInt32     uiAspect,
                               UInt32     uiContainerSize) 
{
    this->execBeginEditImpl(whichField, uiAspect, uiContainerSize);
}


#endif

OSG_END_NAMESPACE

#define OSGNODE_INLINE_CVSID "@(#)$Id: $"

#endif /* _OSGNODEIMPL_INL_ */
