/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright (C) 2000-2002 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

#ifdef OSG_DOC_FILES_IN_MODULE
/*! \file OSGChangeList.cpp
    \ingroup GrpSystemMultithreading
 */
#endif

#include <stdlib.h>
#include <stdio.h>

#include "OSGConfig.h"

#include <iostream>

#include "OSGChangeList.h"
#include "OSGThread.h"
#include "OSGThreadManager.h"
#include "OSGLog.h"
#include "OSGFieldContainerFactory.h"
#include "OSGSceneChanges.h"
#include <OSGFieldContainerPtr.h>

#include <oneapi/tbb/tick_count.h>

#define IGNORE_REFERENCE_COUNT_CHANGES

OSG_USING_NAMESPACE

bool ChangeList::_bReadWriteDefault = false;
UInt32 ChangeList::_max_changed_size = 0;

/*-------------------------------------------------------------------------*/
/*                            Constructors                                 */

ChangeList::ChangeList(void) :
     Inherited                (                   ),
    _bReadOnly                (!_bReadWriteDefault),
    _uiAspectId               (Thread::getAspect()),
    _listMode                 (Public             ),
    _vChangedFieldContainers  (),
    _vAddRefdFieldContainers  (),
    _vSubRefdFieldContainers  (),
    _vCreatedFieldContainers  (),
    _vDestroyedFieldContainers(),
    _extra_cl                 (NULL               )
{
//    Aspect::addList(this, _aspectId);
}

/*-------------------------------------------------------------------------*/
/*                             Destructor                                  */

ChangeList::~ChangeList(void)
{
}

/*-------------------------------------------------------------------------*/
/*                                Get                                      */

ChangeList::changed_size_type ChangeList::sizeChanged(void)
{    
    return _vChangedFieldContainers.size();
}

ChangeList::changed_const_iterator ChangeList::beginChanged(void) const
{
    return _vChangedFieldContainers.begin();
}

ChangeList::changed_const_iterator ChangeList::endChanged(void) const
{
    return _vChangedFieldContainers.end();
}

ChangeList::refd_size_type ChangeList::sizeAddRefd(void)
{
    return _vAddRefdFieldContainers.size();
}

ChangeList::refd_const_iterator ChangeList::beginAddRefd(void) const
{
    return _vAddRefdFieldContainers.begin();
}

ChangeList::refd_const_iterator ChangeList::endAddRefd(void) const
{
    return _vAddRefdFieldContainers.end();
}

ChangeList::refd_size_type ChangeList::sizeSubRefd(void)
{
    return _vSubRefdFieldContainers.size();
}

ChangeList::refd_const_iterator ChangeList::beginSubRefd(void) const
{
    return _vSubRefdFieldContainers.begin();
}

ChangeList::refd_const_iterator ChangeList::endSubRefd(void) const
{
    return _vSubRefdFieldContainers.end();
}

ChangeList::idrefd_const_iterator ChangeList::beginCreated(void) const
{
    return _vCreatedFieldContainers.begin();
}

ChangeList::idrefd_const_iterator ChangeList::endCreated(void) const
{
    return _vCreatedFieldContainers.end();
}

ChangeList::idrefd_size_type ChangeList::sizeCreated(void) const
{
    return _vCreatedFieldContainers.size();
}

ChangeList::idrefd_const_iterator ChangeList::beginDestroyed(void) const
{
    return _vDestroyedFieldContainers.begin();
}

ChangeList::idrefd_const_iterator ChangeList::endDestroyed(void) const
{
    return _vDestroyedFieldContainers.end();
}

ChangeList::idrefd_size_type ChangeList::sizeDestroyed(void) const
{
    return _vDestroyedFieldContainers.size();
}

SceneChanges *ChangeList::changes()
{
    return _changes.get();
}

/*-------------------------------------------------------------------------*/
/*                                Add                                      */

void ChangeList::addChanged(const FieldContainerPtr &pFieldContainer, 
                                  BitVector          bvWhichField)
{
    if(_extra_cl != NULL)
        _extra_cl->addChanged(pFieldContainer, bvWhichField);

    if(_bReadOnly == true)
        return;

    UInt32      uiContainerId(pFieldContainer.getFieldContainerId());

    ChangeEntry tmpEntry     (uiContainerId, bvWhichField);
#ifdef USE_CONCURRENT_VECTOR_TYPE
    _vChangedFieldContainers.push_back(tmpEntry);
#else
    if(_max_changed_size > 0)
    {
        if(_vChangedFieldContainers.size() > _max_changed_size)
            compactChanged();
    }

    try
    {
        _vChangedFieldContainers.push_back(tmpEntry);
    }
    
    // std::bad_alloc doesn't work.
    catch(...)
    {
        // on windows the maximum vector size is 16777216 well
        // vector.max_size() returns 268435455 ????
        // to save some memory we recreate a more compact changelist.
        std::vector<ChangeEntry>::size_type oldSize = _vChangedFieldContainers.size();
        SWARNING << "Compacting ChangeList ..." << std::endl;
        compactChanged();
        SWARNING << "Compacted ChangeList from " << oldSize
                 << " to " << _vChangedFieldContainers.size() << " entries." << std::endl;
        _vChangedFieldContainers.push_back(tmpEntry);
    }
#endif
}

void ChangeList::addAddRefd(const FieldContainerPtr &pFieldContainer)
{
#ifndef IGNORE_REFERENCE_COUNT_CHANGES
    if(_extra_cl != NULL)
        _extra_cl->addAddRefd(pFieldContainer);

    if(_bReadOnly == true)
        return;

    UInt32 uiContainerId = pFieldContainer.getFieldContainerId();
    _vAddRefdFieldContainers.push_back(uiContainerId);
#endif
}

void ChangeList::addSubRefd(const FieldContainerPtr &pFieldContainer)
{
#ifndef IGNORE_REFERENCE_COUNT_CHANGES
    if(_extra_cl != NULL)
        _extra_cl->addSubRefd(pFieldContainer);

#if !defined(OSG_FIXED_MFIELDSYNC)
    if(_bReadOnly == true)
        return;
#endif

    UInt32 uiContainerId = pFieldContainer.getFieldContainerId();
    _vSubRefdFieldContainers.push_back(uiContainerId);
#endif
}

void ChangeList::addCreated  (const UInt32 uiContainerId)
{
    if(_extra_cl != NULL)
        _extra_cl->addCreated(uiContainerId);

    if(_bReadOnly == true)
        return;

    _vCreatedFieldContainers.push_back(uiContainerId);
}

void ChangeList::addDestroyed(const UInt32 uiContainerId)
{
    if(_extra_cl != NULL)
        _extra_cl->addDestroyed(uiContainerId);

    if(_bReadOnly == true)
        return;

    _vDestroyedFieldContainers.push_back(uiContainerId);
}

 // direct access to the underlying arrays
const osg_cl_vector_type<ChangeList::ChangeEntry>& ChangeList::getChanged() const
{
    return _vChangedFieldContainers;
}

const osg_cl_vector_type<ChangeList::IdRefEntry>& ChangeList::getAddRefd() const
{
    return _vAddRefdFieldContainers;
}

const osg_cl_vector_type<ChangeList::IdRefEntry>& ChangeList::getSubRefd() const
{
    return _vSubRefdFieldContainers;
}    

const osg_cl_vector_type<ChangeList::IdRefEntry>& ChangeList::getCreated() const
{
    return _vCreatedFieldContainers;
}

const osg_cl_vector_type<ChangeList::IdRefEntry>& ChangeList::getDestroyed() const
{
    return _vDestroyedFieldContainers;
}



void ChangeList::collectChanges(SceneChanges *changes)
{
    _changes.reset(changes);
    changes->setChangeList(this);
}

/*-------------------------------------------------------------------------*/
/*                               Helper                                    */

void ChangeList::clearAll(void)
{
    _vChangedFieldContainers.clear();
    _vCreatedFieldContainers  .clear();
    _vDestroyedFieldContainers.clear();
    if (_changes)
        _changes->clear();

    constexpr size_t capacityLimit = 1024*1024; // limits to 16MB for changes
    
    if(_vChangedFieldContainers.capacity() > capacityLimit)
        _vChangedFieldContainers.shrink_to_fit();
    if(_vCreatedFieldContainers.capacity() > capacityLimit)
        _vCreatedFieldContainers.shrink_to_fit();
    if(_vDestroyedFieldContainers.capacity() > capacityLimit)
        _vDestroyedFieldContainers.shrink_to_fit();

#ifndef IGNORE_REFERENCE_COUNT_CHANGES
    _vAddRefdFieldContainers.clear();
    _vSubRefdFieldContainers.clear();

    if(_vAddRefdFieldContainers.capacity() > capacityLimit)
        _vAddRefdFieldContainers.shring_to_fit();
    if(_vSubRefdFieldContainers.capacity() > capacityLimit)
        _vSubRefdFieldContainers.shrink_to_fit();
#endif
}

void ChangeList::swap(ChangeList &clist)
{
    if(_extra_cl != nullptr && clist._extra_cl != nullptr)
    {
        _extra_cl->clearAll();
#ifdef USE_CONCURRENT_VECTOR_TYPE
        _extra_cl->_vChangedFieldContainers.assign(clist._vChangedFieldContainers.begin(), clist._vChangedFieldContainers.end());
        _extra_cl->_vCreatedFieldContainers.assign(clist._vCreatedFieldContainers.begin(), clist._vCreatedFieldContainers.end());
        _extra_cl->_vDestroyedFieldContainers.assign(clist._vDestroyedFieldContainers.begin(), clist._vDestroyedFieldContainers.end());
#ifndef IGNORE_REFERENCE_COUNT_CHANGES
        _extra_cl->_vAddRefdFieldContainers.assign(clist._vAddRefdFieldContainers.begin(), clist._vAddRefdFieldContainers.end());
        _extra_cl->_vSubRefdFieldContainers.assign(clist._vSubRefdFieldContainers.begin(), clist._vSubRefdFieldContainers.end());
#endif
#else
        _extra_cl->merge( clist);
#endif
    }
    _vChangedFieldContainers.swap(clist._vChangedFieldContainers);

    _vCreatedFieldContainers.swap(clist._vCreatedFieldContainers);
    _vDestroyedFieldContainers.swap(clist._vDestroyedFieldContainers);
#ifndef IGNORE_REFERENCE_COUNT_CHANGES
    _vAddRefdFieldContainers.swap(clist._vAddRefdFieldContainers);
    _vSubRefdFieldContainers.swap(clist._vSubRefdFieldContainers);
#endif
}

bool ChangeList::merge(const ChangeList &clist)
{
#ifdef USE_CONCURRENT_VECTOR_TYPE
    // if the changelist is the same there is no need to merge it
    if (this == &clist)
        return true;

    if (_extra_cl != NULL)
        _extra_cl->merge(clist);

    _vChangedFieldContainers.grow_by(clist._vChangedFieldContainers.begin(), clist._vChangedFieldContainers.end());
    _vCreatedFieldContainers.grow_by(clist._vCreatedFieldContainers.begin(), clist._vCreatedFieldContainers.end());
    _vDestroyedFieldContainers.grow_by(clist._vDestroyedFieldContainers.begin(), clist._vDestroyedFieldContainers.end());

#ifndef IGNORE_REFERENCE_COUNT_CHANGES
    _vAddRefdFieldContainers.grow_by(clist._vAddRefdFieldContainers.begin(), clist._vAddRefdFieldContainers.end());
    _vSubRefdFieldContainers.grow_by(clist._vSubRefdFieldContainers.begin(), clist._vSubRefdFieldContainers.end());
#endif

    return true;
#else
    if(_extra_cl != NULL)
        _extra_cl->merge( clist);

    bool returnValue = true;

    _vChangedFieldContainers.insert(_vChangedFieldContainers.end(),
                                    clist.beginChanged(), 
                                    clist.endChanged());
    
    
    _vAddRefdFieldContainers.insert(_vAddRefdFieldContainers.end(),
                                    clist.beginAddRefd(), 
                                    clist.endAddRefd());
    
    
    _vSubRefdFieldContainers.insert(_vSubRefdFieldContainers.end(),
                                    clist.beginSubRefd(), 
                                    clist.endSubRefd());

    _vCreatedFieldContainers.insert(_vCreatedFieldContainers.end(),
                                    clist.beginCreated(),
                                    clist.endCreated());

    _vDestroyedFieldContainers.insert(_vDestroyedFieldContainers.end(),
                                      clist.beginDestroyed(),
                                      clist.endDestroyed());

    return returnValue;
#endif
}

bool ChangeList::empty() const
{
    return _vChangedFieldContainers.empty() &&
#ifndef IGNORE_REFERENCE_COUNT_CHANGES
           _vAddRefdFieldContainers.empty() &&
           _vSubRefdFieldContainers.empty() &&
#endif
           _vCreatedFieldContainers.empty() &&
           _vDestroyedFieldContainers.empty();
}

void ChangeList::setAspect(UInt32 uiAspectId)
{
#ifdef USE_CONCURRENT_VECTOR_TYPE
#else
    if(_vChangedFieldContainers.size() != 0 ||
       _vAddRefdFieldContainers.size() != 0 ||
       _vSubRefdFieldContainers.size() != 0)
    {
        SWARNING << "Changing aspect on non empty changelist, all currrent "
                    << "entries will be lost" << std::endl;          
    }

    clearAll();

//    OSGAspect::moveList(this, _aspectId, aspectId);
#endif
    _uiAspectId = uiAspectId;
}

void ChangeList::setReadOnly(bool bReadOnly)
{
    _bReadOnly = bReadOnly;
}

/*! Define whether ChangeLists are created read only by default or not. 
Per default they are created read only, to not have a memory leak in applications that
don't use multiple threads and don't clear the ChangeList. Thus if you want to use the
ChangeLists, multiple threads and/or the cluster you have to call
ChangeList::setReadOnlyDefault(true).

This function should only be called before osgInit.
*/
void ChangeList::setReadWriteDefault(bool bReadWrite)
{
    if(GlobalSystemState != Startup)
        FWARNING(("setReadWriteDefault: called after startup!\n"));

    _bReadWriteDefault = bReadWrite;
}


void ChangeList::setMaxChangedSize(UInt32 size)
{
    _max_changed_size = size;
}

UInt32 ChangeList::getMaxChangedSize(void)
{
    return _max_changed_size;
}

void ChangeList::compactChanged(void)
{
#ifndef USE_CONCURRENT_VECTOR_TYPE
    std::map<UInt32, UInt32> clStore;

    // created
    for(ChangeList::idrefd_const_iterator i = beginCreated(); i != endCreated(); ++i)
    {
        std::map<UInt32, UInt32>::iterator ci = clStore.find(*i);
        if(ci == clStore.end())
            clStore.insert(std::pair<UInt32, UInt32>(*i, 0));
    }

    // addRef
    for(ChangeList::idrefd_const_iterator i = beginAddRefd(); i != endAddRefd(); ++i)
    {
        std::map<UInt32, UInt32>::iterator ci = clStore.find(*i);
        if(ci != clStore.end())
            (*ci).second++;
        //else
        //    FWARNING(("Called addRef on a not created fieldcontainer!\n"));
    }

    // subRef
    for(ChangeList::idrefd_const_iterator i = beginSubRefd(); i != endSubRefd(); ++i)
    {
        std::map<UInt32, UInt32>::iterator ci = clStore.find(*i);
        if(ci != clStore.end())
            (*ci).second--;
        //else
        //    FWARNING(("Called subRef on a not created fieldcontainer!\n"));
    }

    // destroyed
    for(ChangeList::idrefd_const_iterator i = beginDestroyed(); i != endDestroyed(); ++i)
    {
        std::map<UInt32, UInt32>::iterator ci = clStore.find(*i);
        if(ci != clStore.end())
            clStore.erase(ci);
    }

    clearAll();

    for(std::map<UInt32, UInt32>::iterator i = clStore.begin();i != clStore.end(); ++i)
    {
        UInt32 id = (*i).first;
        FieldContainerPtr fc = FieldContainerFactory::the()->getContainer(id);
        if(fc != NullFC)
        {
            addCreated(id);
            for(UInt32 j=0;j<(*i).second;++j)
                addAddRefd(fc);
            addChanged(fc, FieldBits::AllFields);
        }
    }
#endif
}

void ChangeList::setExtraChangeList(ChangeList *list)
{
    _extra_cl = list;
}

ChangeList *ChangeList::getExtraChangeList(void)
{
    return _extra_cl;
}

/*-------------------------------------------------------------------------*/
/*                               Apply                                     */

#ifndef OSG_DISABLE_DEPRECATED

void ChangeList::applyTo(UInt32 uiAspectId)
{
    UInt32 i;

    if(uiAspectId == _uiAspectId)
    {
        SWARNING << "Sync on the same aspect, ignored " << std::endl;
        return;
    }

    if(uiAspectId >= ThreadManager::getNumAspects())
    {
        SWARNING << "Invalid AspectId" << std::endl;
    }

    _bReadOnly = true;
    ChangeList* cl = Thread::getCurrentChangeList();
    if (cl != nullptr)
        cl->setReadOnly(true);

    FieldContainerPtr pTmp;

    for(i = 0; i < _vChangedFieldContainers.size(); i++)
    {
        pTmp = FieldContainerFactory::the()->getContainer(
            _vChangedFieldContainers[i].first);

        if(pTmp == NullFC)
            continue;

        pTmp.executeSync(_uiAspectId,
                          uiAspectId,
                         _vChangedFieldContainers[i].second);
    }
#ifndef IGNORE_REFERENCE_COUNT_CHANGES
    for(i = 0; i < _vAddRefdFieldContainers.size(); i++)
    {
        pTmp = FieldContainerFactory::the()->getContainer(
            _vAddRefdFieldContainers[i]);

        if(pTmp == NullFC)
            continue;

        addRefCP(pTmp);
    }

    for(i = 0; i < _vSubRefdFieldContainers.size(); i++)
    {
        pTmp = FieldContainerFactory::the()->getContainer(
            _vSubRefdFieldContainers[i]);

        if(pTmp == NullFC)
            continue;

        subRefCP(pTmp);
    }
#endif
    clearAll();

    _bReadOnly = false;
    if (cl != nullptr)
        cl->setReadOnly(false);

}

void ChangeList::applyToCurrent(void)
{
    applyTo(Thread::getAspect());
}

#endif

void ChangeList::apply(void)
{
    UInt32 i;

    if(osg::Thread::getCurrentChangeList() == this)
    {
        SWARNING << "try to apply current changelist : ignored" << std::endl;
    }

    _bReadOnly = true;

    ChangeList* cl = Thread::getCurrentChangeList();
    if (cl != nullptr)
        cl->setReadOnly(true);

    FieldContainerPtr pTmp;

    for(i = 0; i < _vChangedFieldContainers.size(); i++)
    {
        pTmp = FieldContainerFactory::the()->getContainer(
            _vChangedFieldContainers[i].first);

        if(pTmp == NullFC)
            continue;

        pTmp.executeSync(_uiAspectId,
                          Thread::getAspect(),
                         _vChangedFieldContainers[i].second);
    }
#ifndef IGNORE_REFERENCE_COUNT_CHANGES
#if !defined(OSG_FIXED_MFIELDSYNC)
    for(i = 0; i < _vAddRefdFieldContainers.size(); i++)
    {
        pTmp = FieldContainerFactory::the()->getContainer(
            _vAddRefdFieldContainers[i]);

        if(pTmp == NullFC)
            continue;

        addRefCP(pTmp);
    }

    for(i = 0; i < _vSubRefdFieldContainers.size(); i++)
    {
        pTmp = FieldContainerFactory::the()->getContainer(
            _vSubRefdFieldContainers[i]);

        if(pTmp == NullFC)
            continue;

        subRefCP(pTmp);
    }
#endif
#endif
    _bReadOnly = false;
    if(cl != nullptr)
        cl->setReadOnly(false);
}

#if defined(OSG_FIXED_MFIELDSYNC)
void ChangeList::applySubRefs(void)
{
    _bReadOnly = false;

    ChangeList* cl = Thread::getCurrentChangeList();
    if (cl != nullptr)
        cl->setReadOnly(false);

    FieldContainerPtr pTmp;

    std::vector<IdRefEntry>  tmpList;

    tmpList.swap(_vSubRefdFieldContainers);

    while(tmpList.size() != 0)
    {
        _vSubRefdFieldContainers.clear();

        for(UInt32 i = 0; i < tmpList.size(); i++)
        {
//            fprintf(stderr, "subref %d\n", tmpList[i]);

            pTmp = FieldContainerFactory::the()->getContainer(
                tmpList[i]);
            
            if(pTmp == NullFC)
                continue;
            
            pTmp.doSubRef();
        }
        
        tmpList.clear();
        tmpList.swap(_vSubRefdFieldContainers);
    }

    _bReadOnly = false;
    if( cl != nullptr)
        cl->setReadOnly(false);
}
#endif

void ChangeList::applyAndClear(void)
{
    apply   ();
    clearAll();
}

/*-------------------------------------------------------------------------*/
/*                              Dump                                       */

void ChangeList::dump(void)
{
    UInt32 i;

    fprintf(stderr, "CL: %u\n", _uiAspectId);
#if 0
    fprintf(stderr, "CLChanged:\n");

    for(i = 0; i < _vChangedFieldContainers.size(); i++)
    {
//        fprintf(stderr, "\t%d\n", 
//                _vChangedFieldContainers[i].first.getFieldContainerId());
        fprintf(stderr, "\t%u\n", 
                _vChangedFieldContainers[i].first);
    }
#endif

    fprintf(stderr, "CLAdd:\n");
    for(i = 0; i < _vAddRefdFieldContainers.size(); i++)
    {
//        fprintf(stderr, "\t%d\n", 
//                _vAddRefdFieldContainers[i].getFieldContainerId());
        fprintf(stderr, "\t%u\n", 
                _vAddRefdFieldContainers[i]);
    }

    fprintf(stderr, "CLSub:\n");
    for(i = 0; i < _vSubRefdFieldContainers.size(); i++)
    {
//        fprintf(stderr, "\t%d\n", 
//                _vSubRefdFieldContainers[i].getFieldContainerId());
        fprintf(stderr, "\t%u\n", 
                _vSubRefdFieldContainers[i]);
    }

    fprintf(stderr, "CLCreate:\n");
    for(i = 0; i < _vCreatedFieldContainers.size(); i++)
    {
        fprintf(stderr, "\t%u | ", _vCreatedFieldContainers[i]);

        FieldContainerPtr pTmp = FieldContainerFactory::the()->getContainer(
            _vCreatedFieldContainers[i]);
            
        if(pTmp != NullFC)
        {
            fprintf(stderr, "%s\n", pTmp->getType().getCName());
        }
        else
        {
            fprintf(stderr, "\n");
        }
    }
    
    fprintf(stderr, "CLDestroy:\n");
    for(i = 0; i < _vDestroyedFieldContainers.size(); i++)
    {
        fprintf(stderr, "\t%u\n", _vDestroyedFieldContainers[i]);
    }
}


/*-------------------------------------------------------------------------*/
/*                              cvs id's                                   */

#ifdef __sgi
#pragma set woff 1174
#endif

#ifdef OSG_LINUX_ICC
#pragma warning( disable : 177 )
#endif

namespace
{
    static Char8 cvsid_cpp[] = "@(#)$Id: $";
    static Char8 cvsid_hpp[] = OSGCHANGELIST_HEADER_CVSID;
}
