/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright (C) 2000-2002 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

#ifdef OSG_DOC_FILES_IN_MODULE
/*! \file OSGFieldContainer.cpp
    \ingroup GrpSystemFieldContainer
 */
#endif

#ifndef VR_RELEASE_BUILD
// #define OSG_FC_COMPARE_DEBUG_OUTPUT
#endif

#include <stdlib.h>
#include <stdio.h>

#include "OSGConfig.h"
#include "OSGFieldContainer.h"
#include "OSGFieldType.h"
#include "OSGField.h"
#include "OSGMField.h"
#include "OSGVector.h"
#include "OSGColor.h"
#include "OSGImage.h"
#include "OSGSFSysTypes.h"
#include "OSGSFBaseTypes.h"
#include "OSGSFVecTypes.h"
#include "OSGSFMathTypes.h"
#include "OSGMFBaseTypes.h"
#include "OSGMFSysTypes.h"
#include "OSGMFVecTypes.h"
#include "OSGSFFieldContainerPtr.h"
#include "OSGMFFieldContainerPtr.h"

OSG_USING_NAMESPACE

FieldContainerType FieldContainer::_type("FieldContainer");

/*-------------------------------------------------------------------------*/
/*                                Get                                      */

FieldContainerType &FieldContainer::getType(void)
{
    return _type;
}

const FieldContainerType &FieldContainer::getType(void) const
{
    return _type;
}

/*-------------------------------------------------------------------------*/
/*                            Binary Access                                */

SizeT FieldContainer::getBinSize(const BitVector &)
{
    return 0;
}

void FieldContainer::copyToBin(      BinaryDataHandler &,
                               const BitVector         &)
{
}

void FieldContainer::copyFromBin(      BinaryDataHandler &,
                                 const BitVector         &)
{

}

template <typename T>
static bool compareSField(Field *a, Field *b)
{
    T *fa = (T *)a;
    T *fb = (T *)b;

    if (fa == NULL && fb == NULL)
        return true;

    if (fa == NULL || fb == NULL)
    {
#ifdef OSG_FC_COMPARE_DEBUG_OUTPUT
        printf("FieldContainer compare: Single field %s, one pointer is null.\n",
            fa != NULL ? fa->getType().getCName() : fb->getType().getCName());
#endif
        return false;
    }

    if (fa->getValue() != fb->getValue())
    {
#ifdef OSG_FC_COMPARE_DEBUG_OUTPUT
        std::string aStr, bStr;
        a->getValueByStr(aStr);
        b->getValueByStr(bStr);
        printf("FieldContainer compare: Single field %s, values differ: %s, %s.\n", fa->getType().getCName(), aStr.c_str(), bStr.c_str());
#endif
        return false;
    }

    return true;
}

template <typename T>
static bool compareMField(Field *a, Field *b)
{
    MField<T> *mfa = dynamic_cast<MField<T> *>(a);
    MField<T> *mfb = dynamic_cast<MField<T> *>(b);

    if (mfa == NULL && mfb == NULL)
        return true;

    if (mfa == NULL || mfb == NULL)
    {
#ifdef OSG_FC_COMPARE_DEBUG_OUTPUT
        printf("FieldContainer compare: Multi field %s, one pointer is null.\n",
            mfa != NULL ? mfa->getType().getCName() : mfb->getType().getCName());
#endif
        return false;
    }

    if (mfa->size() != mfb->size())
    {
#ifdef OSG_FC_COMPARE_DEBUG_OUTPUT
        printf("FieldContainer compare: Multi field %s, sizes differ.\n", mfa->getType().getCName());
#endif
        return false;
    }

    for (UInt32 j = 0; j < mfa->size(); ++j)
    {
        if ((*mfa)[j] != (*mfb)[j])
        {
#ifdef OSG_FC_COMPARE_DEBUG_OUTPUT
            printf("FieldContainer compare: Multi field %s, entry %u differs.\n", mfa->getType().getCName(), j);
#endif
            return false;
        }
    }

    return true;
}

bool FieldContainer::isEqual(const osg::FieldContainerPtr &a,
    const osg::FieldContainerPtr &b)
{
    std::map<osg::FieldContainerPtr, std::set<osg::FieldContainerPtr> > equalFCs;
    IsEqualOptions options{};
    return isEqual(a, b, options, equalFCs);
}

/*!
* \brief Compares two field containers
* \param a field container a
* \param b field container b
* \param options options for the comparison
* \return true if equal.
*/
bool FieldContainer::isEqual(const osg::FieldContainerPtr &a,
    const osg::FieldContainerPtr &b,
    const FieldContainer::IsEqualOptions &options,
    std::map<osg::FieldContainerPtr, std::set<osg::FieldContainerPtr> > &equalFCs)
{
    // Compare the pointers.
    if (a == b)
        return true;

    if (a == NullFC || b == NullFC)
        return false;

    return a->equals(b, options, equalFCs);
}

bool FieldContainer::supportsAnimChannels() const
{
    return false;
}

void FieldContainer::getAnimChannels(ChannelData& data)
{
}

bool FieldContainer::equals(const FieldContainerPtr & b,
                            const IsEqualOptions &options, 
                            std::map<osg::FieldContainerPtr, std::set<osg::FieldContainerPtr> > &equalFCs) const
{
    // Code moved here from vrNodeUtils::isEqual 
    FieldContainerPtr a(this);
    
    // Compare the pointers.
    if (a == b)
        return true;

    if (a == NullFC || b == NullFC)
        return false;

    if (a->getType() != b->getType())
        return false;

    auto itA = equalFCs.find(a);
    if ((itA != equalFCs.end()) && (itA->second.find(b) != itA->second.end()))
    {
        return true;
    }
    auto itB = equalFCs.find(b);
    if ((itB != equalFCs.end()) && (itB->second.find(a) != itB->second.end()))
    {
        return true;
    }

#ifdef OSG_FC_COMPARE_DEBUG_OUTPUT
    printf("Comparing: %s\n", a->getType().getName().str());
#endif

    const FieldContainerType &type = a->getType();

    // ACHTUNG um die dynamischen Felder zu ignorieren wird Minimum genommen!
    UInt32 fcount = osgMin(type.getNumFieldDescs(), b->getType().getNumFieldDescs());

#ifdef OSG_FC_COMPARE_DEBUG_OUTPUT
    struct FailLogScopeGuard {
        std::string lastField;
        std::string val0;
        std::string val1;
        bool success {false};
        ~FailLogScopeGuard() {
            if (!success && !lastField.empty())
            {
                printf("FieldContainer::equals: Field '%s' failed: %s, %s\n",
                    lastField.c_str(), val0.c_str(), val1.c_str());
            }
        }
    };
    FailLogScopeGuard guard;
#endif

    for (UInt32 i = 1; i <= fcount; ++i)
    {
        const FieldDescription* fdesc = type.getFieldDescription(i);

        if (fdesc->isInternal())
            continue;

        // ignore attachments
        if (strcmp(fdesc->getCName(), "attachments") == 0)
            continue;

        // ignore parents
        if (strcmp(fdesc->getCName(), "parents") == 0)
            continue;

#if 1
        bool doSkipField = false;
        for(auto& skipField : options.skipFields)
        {
            if(std::strcmp(skipField.c_str(),  fdesc->getCName()) == 0)
            {
                doSkipField = true;
                break;
            }
        }
        if (doSkipField)
        {
            continue;
        }
#else
        // This is expensive because a temporary std::string object is created from "fdesc->getCName()"!
        if(options.skipFields.find(fdesc->getCName()) != options.skipFields.end())
            continue;
#endif

#ifdef OSG_FC_COMPARE_DEBUG_OUTPUT
        guard.lastField = fdesc->getCName();
        printf("Comparing field %s\n", fdesc->getCName());
#endif

        Field *a_field = a->getField(i);
        Field *b_field = b->getField(i);

        const FieldType &a_ftype = a_field->getType();
        const FieldType &b_ftype = b_field->getType();

        if (a_ftype != b_ftype)
            return false;              

        if(options.skipFieldIdCallback && options.skipFieldIdCallback(a, b, i))
        {
            continue;
        }

        if (strstr(a_ftype.getCName(), "Ptr") == NULL) // no "Ptr" type
        {
            // some speedup.
            // This is a HACK this will be changed in near future.
            if (a_field->getCardinality() == FieldType::MULTI_FIELD)
            {
                if(a_field->isEmpty() && b_field->isEmpty())
                    continue; // both are empty, can be skipped
                else if(a_field->getSize() != b_field->getSize())
                    return false; // multi field with different size

                // compare non-empty multi field with same size

                if (a_ftype == MFUInt8::getClassType())
                {
                    if (!compareMField<UInt8>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == MFInt8::getClassType())
                {
                    if (!compareMField<Int8>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == MFInt32::getClassType())
                {
                    if (!compareMField<Int32>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == MFUInt32::getClassType())
                {
                    if (!compareMField<UInt32>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == MFUInt16::getClassType())
                {
                    if (!compareMField<UInt16>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == MFInt16::getClassType())
                {
                    if (!compareMField<Int16>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == MFReal32::getClassType())
                {
                    if (!compareMField<Real32>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == MFVec2f::getClassType())
                {
                    if (!compareMField<Vec2f>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == MFVec3f::getClassType())
                {
                    if (!compareMField<Vec3f>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == MFVec4f::getClassType())
                {
                    if (!compareMField<Vec4f>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == MFPnt3f::getClassType())
                {
                    if (!compareMField<Pnt3f>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == MFPnt4f::getClassType())
                {
                    if (!compareMField<Pnt4f>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == MFColor3f::getClassType())
                {
                    if (!compareMField<Color3f>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == MFColor4f::getClassType())
                {
                    if (!compareMField<Color4f>(a_field, b_field))
                        return false;
                }
                else
                {
#ifndef VR_RELEASE_BUILD
                    FLOG(("Slow multi field string compare for '%s'!\n",
                        a_ftype.getCName()));
#endif
                    std::string av, bv;
                    a_field->getValueByStr(av);
                    b_field->getValueByStr(bv);
                    if (av != bv)
                        return false;
                }
            }
            else
            {

                if (a_ftype == SFBool::getClassType())
                {
                    if (!compareSField<SFBool>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == SFString::getClassType())
                {
                    if (!compareSField<SFString>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == SFGLenum::getClassType())
                {
                    if (!compareSField<SFGLenum>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == SFUInt8::getClassType())
                {
                    if (!compareSField<SFUInt8>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == SFUInt16::getClassType())
                {
                    if (!compareSField<SFUInt16>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == SFUInt32::getClassType())
                {
                    if (!compareSField<SFUInt32>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == SFUInt64::getClassType())
                {
                    if (!compareSField<SFUInt64>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == SFInt8::getClassType())
                {
                    if (!compareSField<SFInt8>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == SFInt16::getClassType())
                {
                    if (!compareSField<SFInt16>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == SFInt32::getClassType())
                {
                    if (!compareSField<SFInt32>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == SFInt64::getClassType())
                {
                    if (!compareSField<SFInt64>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == SFReal32::getClassType())
                {
                    if (!compareSField<SFReal32>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == SFReal64::getClassType())
                {
                    if (!compareSField<SFReal64>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == SFTime::getClassType())
                {
                    if (!compareSField<SFTime>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == SFVec2f::getClassType())
                {
                    if (!compareSField<SFVec2f>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == SFVec3f::getClassType())
                {
                    if (!compareSField<SFVec3f>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == SFVec4f::getClassType())
                {
                    if (!compareSField<SFVec4f>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == SFPnt2f::getClassType())
                {
                    if (!compareSField<SFPnt2f>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == SFPnt3f::getClassType())
                {
                    if (!compareSField<SFPnt3f>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == SFPnt4f::getClassType())
                {
                    if (!compareSField<SFPnt4f>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == SFColor3f::getClassType())
                {
                    if (!compareSField<SFColor3f>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == SFColor4f::getClassType())
                {
                    if (!compareSField<SFColor4f>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == SFPnt3f::getClassType())
                {
                    if (!compareSField<SFPnt3f>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == SFMatrix::getClassType())
                {
                    if (!compareSField<SFMatrix>(a_field, b_field))
                        return false;
                }
                else if (a_ftype == SFQuaternion::getClassType())
                {
                    if (!compareSField<SFQuaternion>(a_field, b_field))
                        return false;
                }
                else
                {
#ifndef VR_RELEASE_BUILD
                    FLOG(("Slow field string compare for '%s'!\n",
                        a_ftype.getCName()));
#endif
                    std::string av, bv;
                    a_field->getValueByStr(av);
                    b_field->getValueByStr(bv);
                    if (av != bv)
                        return false;
                }

                /*
                std::string av, bv;
                a_field->getValueByStr(av);
                b_field->getValueByStr(bv);
                if(av != bv)
                return false;
                */
            }
        }
        else
        {
            if (a_field->getCardinality() == FieldType::SINGLE_FIELD)
            {
                if(options.skipImageAttributes != 0 && a_ftype == SFImagePtr::getClassType())
                {
                    osg::ImagePtr imgA = ((SFImagePtr*)a_field)->getValue();
                    osg::ImagePtr imgB = ((SFImagePtr*)b_field)->getValue();
                    if( imgA != NullFC && imgA->hasAttributes(options.skipImageAttributes) && 
                        imgB != NullFC && imgB->hasAttributes(options.skipImageAttributes) &&
                        imgA != imgB)
                    {
                        equalFCs[imgA].insert(imgB);
                        equalFCs[imgB].insert(imgA);
                    }
                    else
                    {
                        if (!isEqual(imgA, imgB, options, equalFCs))
                        {
                            return false;
                        }
                        else
                        {
                            if (imgA != NullFC && imgB != NullFC && imgA != imgB)
                            {
                                equalFCs[imgA].insert(imgB);
                                equalFCs[imgB].insert(imgA);
                            }
                        }
                    }
                }
                else
                {
                    osg::FieldContainerPtr aFC = ((SFFieldContainerPtr *)a_field)->getValue();
                    osg::FieldContainerPtr bFC = ((SFFieldContainerPtr *)b_field)->getValue();
                    if (!isEqual(aFC, bFC, options, equalFCs))
                    {
                        return false;
                    }
                    else
                    {
                        if (aFC != NullFC && bFC != NullFC && aFC != bFC)
                        {
                            equalFCs[aFC].insert(bFC);
                            equalFCs[bFC].insert(aFC);
                        }
                    }
                }
            }
            else if (a_field->getCardinality() == FieldType::MULTI_FIELD)
            {
                if (((MFFieldContainerPtr*)a_field)->size() !=
                    ((MFFieldContainerPtr*)b_field)->size())
                    return false;

                for (UInt32 j = 0; j < ((MFFieldContainerPtr*)a_field)->size(); ++j)
                {
                    if(options.skipImageAttributes != 0 && a_ftype == MFImagePtr::getClassType())
                    {
                        osg::ImagePtr imgA = (*(((MFImagePtr *)a_field)))[j];
                        osg::ImagePtr imgB = (*(((MFImagePtr *)b_field)))[j];
                        if( imgA != NullFC && imgA->hasAttributes(options.skipImageAttributes) && 
                            imgB != NullFC && imgB->hasAttributes(options.skipImageAttributes) &&
                            imgA != imgB)
                        {
                            equalFCs[imgA].insert(imgB);
                            equalFCs[imgB].insert(imgA);
                        }
                        else
                        {
                            if (!isEqual(imgA, imgB, options, equalFCs))
                            {
                                return false;
                            }
                            else
                            {
                                if (imgA != NullFC && imgB != NullFC && imgA != imgB)
                                {
                                    equalFCs[imgA].insert(imgB);
                                    equalFCs[imgB].insert(imgA);
                                }
                            }
                        }
                    }
                    else
                    {
                        osg::FieldContainerPtr aFC = (*(((MFFieldContainerPtr *)a_field)))[j];
                        osg::FieldContainerPtr bFC = (*(((MFFieldContainerPtr *)b_field)))[j];
                        if (!isEqual(aFC, bFC, options, equalFCs))
                        {
                            return false;
                        }
                        else
                        {
                            if (aFC != NullFC && bFC != NullFC && aFC != bFC)
                            {
                                equalFCs[aFC].insert(bFC);
                                equalFCs[bFC].insert(aFC);
                            }
                        }
                    }
                }
            }
        }
    }
    if (a != NullFC && b != NullFC && a != b)
    {
        equalFCs[a].insert(b);
        equalFCs[b].insert(a);
    }

#ifdef OSG_FC_COMPARE_DEBUG_OUTPUT
    guard.success = true;
#endif

    return true;
}

void FieldContainer::initCallbacks()
{
    if(_callbacks == nullptr)
    {
        _callbacks = new std::vector < void* > ;
        _callbacks->push_back(new std::vector< std::weak_ptr<OnDestroyCallback> >);
        _callbacks->push_back(new std::vector< std::weak_ptr<ChangedCallback> >);
    }
}
void FieldContainer::deleteCallbacks()
{
    if (_callbacks != nullptr && _callbacks->size() >= 2)
    {
        std::vector< std::weak_ptr<OnDestroyCallback> >* onDestroyCallbacks = reinterpret_cast<std::vector< std::weak_ptr<OnDestroyCallback> >*>((*_callbacks)[0]);
        if(onDestroyCallbacks)
        {
            onDestroyCallbacks->clear();
            delete onDestroyCallbacks;
        }

        std::vector< std::weak_ptr<ChangedCallback> >* onChangeCallbacks = reinterpret_cast<std::vector< std::weak_ptr<ChangedCallback> >*>((*_callbacks)[1]);
        if(onChangeCallbacks)
        {
            onChangeCallbacks->clear();
            delete onChangeCallbacks;
        }

        delete _callbacks;
        _callbacks = nullptr;
    }
}

std::vector< std::weak_ptr<OnDestroyCallback> >* FieldContainer::getOnDestroyCallbacks() const
{
    if (_callbacks != nullptr && _callbacks->size() > 0)
        return reinterpret_cast<std::vector< std::weak_ptr<OnDestroyCallback> >*>((*_callbacks)[0]);

    return nullptr;
}

std::vector< std::weak_ptr<ChangedCallback> >* FieldContainer::getChangeCallbacks() const
{
    if (_callbacks != nullptr && _callbacks->size() > 1)
        return reinterpret_cast<std::vector< std::weak_ptr<ChangedCallback> >*>((*_callbacks)[1]);

    return nullptr;
}


void FieldContainer::registerOnDestroyCallback(std::shared_ptr<OnDestroyCallback> const& callback)
{
    initCallbacks();
    if (getOnDestroyCallbacks() == nullptr)
        return;

    std::vector< std::weak_ptr<OnDestroyCallback> >& onDestroyCallbacks = *getOnDestroyCallbacks();

    bool registered = std::find_if(onDestroyCallbacks.begin(), onDestroyCallbacks.end(),
        [&callback](const std::weak_ptr<OnDestroyCallback>& wp) { return wp.lock() == callback; }) != onDestroyCallbacks.end();

    if (!registered)
    {
        onDestroyCallbacks.push_back(callback);
    }
}

void FieldContainer::unregisterOnDestroyCallback(std::shared_ptr<OnDestroyCallback> const& callback)
{
    if (getOnDestroyCallbacks() == nullptr)
        return;

    std::vector< std::weak_ptr<OnDestroyCallback> >& onDestroyCallbacks = *getOnDestroyCallbacks();

    auto it = std::find_if(onDestroyCallbacks.begin(), onDestroyCallbacks.end(),
        [&callback](const std::weak_ptr<OnDestroyCallback>& wp) { return wp.lock() == callback; });

    if (it != onDestroyCallbacks.end())
    {
        onDestroyCallbacks.erase(it);
    }
}

void FieldContainer::unregisterOnDestroyCallbacks()
{
    if (getOnDestroyCallbacks() != nullptr)
        getOnDestroyCallbacks()->clear();
}

void FieldContainer::registerChangedCallback(std::shared_ptr<ChangedCallback> const& callback)
{
    initCallbacks();
    if (getChangeCallbacks() == nullptr)
        return;

    std::vector< std::weak_ptr<ChangedCallback> >& changedCallbacks = *getChangeCallbacks();

    bool registered = std::find_if(changedCallbacks.begin(), changedCallbacks.end(),
        [&callback](const std::weak_ptr<ChangedCallback>& wp) { return wp.lock() == callback; }) != changedCallbacks.end();

    if (!registered)
    {
        changedCallbacks.push_back(callback);
    }
}

void FieldContainer::unregisterChangedCallback(std::shared_ptr<ChangedCallback> const& callback)
{
    if (getChangeCallbacks() == nullptr)
        return;

    std::vector< std::weak_ptr<ChangedCallback> >& changedCallbacks = *getChangeCallbacks();

    auto it = std::find_if(changedCallbacks.begin(), changedCallbacks.end(),
        [&callback](const std::weak_ptr<ChangedCallback>& wp) { return wp.lock() == callback; });

    if (it != changedCallbacks.end())
    {
        changedCallbacks.erase(it);
    }
}

void FieldContainer::unregisterChangedCallbacks()
{
    if (getChangeCallbacks() != nullptr)
        getChangeCallbacks()->clear();
}


/*-------------------------------------------------------------------------*/
/*                               Changed                                   */

void FieldContainer::notifyChangeCallbacks(BitVector whichField, UInt32 origin)
{
    if (_callbacks != nullptr && getChangeCallbacks() != nullptr)
    {
        FieldContainerPtr tmp(this);
        if (tmp != NullFC)
        {
            UInt32 fcid = tmp.getFieldContainerId();
            const std::vector< std::weak_ptr<ChangedCallback> >& changedCallbacks = *getChangeCallbacks();
            for (size_t idx = 0; idx < changedCallbacks.size(); ++idx)
            {
                if (auto callback = changedCallbacks[idx].lock())
                {
                    (*callback)(fcid, whichField, origin);
                }
            }

        }
    }
}

void FieldContainer::changed(BitVector whichField,
                             UInt32    origin    )
{
    // fprintf(stderr, "FC Changed %d %d\n", whichField, fromSync);
    notifyChangeCallbacks(whichField, origin);
}

/*-------------------------------------------------------------------------*/
/*                             MT Destruction                              */

void FieldContainer::onDestroy(void)
{
    if (_callbacks != nullptr && getOnDestroyCallbacks() != nullptr)
    {
        FieldContainerPtr tmp(this);
        if (tmp != NullFC)
        {
            UInt32 fcid = tmp.getFieldContainerId();
            const std::vector< std::weak_ptr<OnDestroyCallback> >& onDestroyCallbacks = *getOnDestroyCallbacks();
            for (size_t idx = 0; idx < onDestroyCallbacks.size(); ++idx)
            {
                if (auto callback = onDestroyCallbacks[idx].lock())
                {
                    (*callback)(fcid);
                }
            }
        }
        deleteCallbacks();
    }

}

#if defined(OSG_FIXED_MFIELDSYNC)
void FieldContainer::onDestroyAspect(UInt32 uiId, UInt32 uiAspect)
{
}
#endif

/*-------------------------------------------------------------------------*/
/*                              cvs id's                                   */

#ifdef __sgi
#pragma set woff 1174
#endif

#ifdef OSG_LINUX_ICC
#pragma warning( disable : 177 )
#endif

namespace
{
    static Char8 cvsid_cpp[] = "@(#)$Id: $";
    static Char8 cvsid_hpp[] = OSGFIELDCONTAINER_HEADER_CVSID;
    static Char8 cvsid_inl[] = OSGFIELDCONTAINER_INLINE_CVSID;
}
