/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *               Copyright (C) 2000-2002 by the OpenSG Forum                 *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

#include <stdlib.h>
#include <stdio.h>

#include <OSGConfig.h>

#include <OSGSimpleAttachments.h>

#include "OSGName.h"

OSG_BEGIN_NAMESPACE

/***************************************************************************\
 *                            Description                                  *
\***************************************************************************/

/*! \class osg::Name
An attachment that stores a name string and the encoding of the name string.
*/

/***************************************************************************\
 *                           Class variables                               *
\***************************************************************************/

/***************************************************************************\
 *                           Class methods                                 *
\***************************************************************************/

void Name::initMethod (void)
{
}


/***************************************************************************\
 *                           Instance methods                              *
\***************************************************************************/

/*-------------------------------------------------------------------------*\
 -  private                                                                 -
\*-------------------------------------------------------------------------*/

/*----------------------- constructors & destructors ----------------------*/

Name::Name(void) :
    Inherited()
{
}

Name::Name(const Name &source) :
    Inherited(source)
{
}

Name::~Name(void)
{
}

/*----------------------------- class specific ----------------------------*/

void Name::changed(BitVector whichField, UInt32 origin)
{
    Inherited::changed(whichField, origin);
}

void Name::dump(      UInt32    , 
                         const BitVector ) const
{
    SLOG << "Dump Name NI" << std::endl;
}


/*-------------------------------------------------------------------------*/
/*                   Name Attachment Utility Functions                     */

/*!
  Return the name attached to the container, NULL if none attached or
  container is NULL.
 */

const Char8 *getName(AttachmentContainerPtr container)
{
    if(container == NullFC)
        return NULL;

    // Get attachment pointer
    AttachmentPtr att =
        container->findAttachment(Name::getClassType().getGroupId());

    if(att == NullFC)
        return NULL;

    // Cast to name pointer

    NamePtr name = NamePtr::dcast(att);

    if(name == NullFC)
        return NULL;

    return name->getName().c_str();
}

/*! Set the name attached to the container. If the container doesn't have a
    name attachement yet one is created.
 */

void setName(      AttachmentContainerPtr  container,
             const std::string            &namestring)
{
    /* by using setName(mat, name, true) you indicate, that you made sure the name is utf8 encoded. This is how it should be.
       You can use QString::toUtf8 and QString::fromUtf8 to achive this. */
/*
    printf("osg::setName: depricated latin1 function, use setName(node, name, true) with name in utf8\n");
#ifndef WIN32
    void *array[10];
    // get void*'s for all entries on the stack
    size_t size = backtrace(array, 10);
    // print out all the frames to stderr
    fprintf(stderr, "vrMaterials::printBT:\n");
    backtrace_symbols_fd(array, size, 2);
#endif
*/
    setName(container, namestring, false);
}

void setName(      AttachmentContainerPtr  container,
             const std::string            &namestring,
                   bool                    isUtf8)
{
    if(container == NullFC)
    {
        FFATAL(("setName: no container?!?\n"));
        return;
    }

    // Get attachment pointer

    NamePtr       name = NullFC;
    AttachmentPtr att  =
        container->findAttachment(Name::getClassType().getGroupId());

    if(att == NullFC)
    {
        name = Name::create();
        beginEditCP(container, AttachmentContainer::AttachmentsFieldMask);
        {
            container->addAttachment(name);
        }
        endEditCP(container, AttachmentContainer::AttachmentsFieldMask);
    }
    else
    {
        name = NamePtr::dcast(att);

        if(name == NullFC)
        {
            FFATAL(("setName: Name Attachment is not castable to Name?!?\n"));
            return;
        }
    }
    beginEditCP(name, Name::NameFieldMask);
    {
        name->setName(namestring);
    }
    endEditCP(name, Name::NameFieldMask);

    beginEditCP(name, Name::EncodingFieldMask);
    {
        if(isUtf8)
            name->setEncoding(Name::EncodingUtf8);
        else
            name->setEncoding(Name::EncodingLatin1);
    }
    endEditCP(name, Name::EncodingFieldMask);
}

/*!
  Set the name attached to the container. If the container doesn't have
  name attachement yet one is created. If the name is NULL, an attached
  name is removed.
 */

void setName(AttachmentContainerPtr container, const Char8 *name)
{
    if(name == NULL)
    {
        AttachmentPtr att =
            container->findAttachment(Name::getClassType().getGroupId());

        if(att != NullFC)
        {
            beginEditCP(container, AttachmentContainer::AttachmentsFieldMask);
            container->subAttachment(att);
            endEditCP(container, AttachmentContainer::AttachmentsFieldMask);
        }

        AttachmentPtr eatt =
            container->findAttachment(Encoding::getClassType().getGroupId());

        if(eatt != NullFC)
        {
            beginEditCP(container, AttachmentContainer::AttachmentsFieldMask);
            container->subAttachment(eatt);
            endEditCP(container, AttachmentContainer::AttachmentsFieldMask);
        }
    }
    else
    {
        setName(container, std::string(name));
    }
}

void   copyName(      AttachmentContainerPtr  srcContainer, 
    AttachmentContainerPtr  dstContainer)
{
    if(srcContainer == NullFC || dstContainer == NullFC)
        return;

    AttachmentPtr srcAtt  =
        srcContainer->findAttachment(Name::getClassType().getGroupId());

    if(srcAtt != NullFC)
    {
        // copy name from src to dst
        AttachmentPtr attCopy = AttachmentPtr::dcast(deepClone(srcAtt));
        if(attCopy != NullFC)
        {
            beginEditCP(dstContainer, AttachmentContainer::AttachmentsFieldMask);
            dstContainer->addAttachment(attCopy);
            endEditCP(dstContainer, AttachmentContainer::AttachmentsFieldMask);
        }
    }
    else
    {
        // src has no name, remove name from dst
        AttachmentPtr dstAtt  =
            dstContainer->findAttachment(Name::getClassType().getGroupId());
        if(dstAtt != NullFC)
        {
            beginEditCP(dstContainer, AttachmentContainer::AttachmentsFieldMask);
            dstContainer->subAttachment(dstAtt);
            endEditCP(dstContainer, AttachmentContainer::AttachmentsFieldMask);
        }
    }
}



/*------------------------------------------------------------------------*/
/*                              cvs id's                                  */

#ifdef OSG_SGI_CC
#pragma set woff 1174
#endif

#ifdef OSG_LINUX_ICC
#pragma warning( disable : 177 )
#endif

namespace
{
    static Char8 cvsid_cpp       [] = "@(#)$Id: FCTemplate_cpp.h,v 1.20 2006/03/16 17:01:53 dirk Exp $";
    static Char8 cvsid_hpp       [] = OSGNAMEBASE_HEADER_CVSID;
    static Char8 cvsid_inl       [] = OSGNAMEBASE_INLINE_CVSID;

    static Char8 cvsid_fields_hpp[] = OSGNAMEFIELDS_HEADER_CVSID;
}

#ifdef __sgi
#pragma reset woff 1174
#endif

OSG_END_NAMESPACE

