/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *                     Copyright 2000-2002 by OpenSG Forum                   *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

#ifdef OSG_DOC_FILES_IN_MODULE
/*! \file OSGNode.cpp
    \ingroup GrpSystemFieldContainer
 */
#endif

#define VR_SUPPORT_INDEX

#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>

#include <functional>
#include <algorithm>

#include "OSGConfig.h"
#include "OSGFieldContainerPtr.h"
#include "OSGFieldContainerType.h"
#include "OSGNode.h"
#include "OSGNodeCore.h"
#include "OSGNodeCoreFlags.h"
#include "OSGSceneFileHandler.h"
#include "OSGImage.h"
#include "OSGBinaryDataHandler.h"
#include "OSGSFFieldContainerPtr.h"
#include "OSGMFFieldContainerPtr.h"
#include "OSGName.h"
#include "OSGUnshareFunctions.h"

OSG_USING_NAMESPACE

const BitVector Node::VolumeFieldMask      =
        (TypeTraits<BitVector>::One << Node::VolumeFieldId     );
const BitVector Node::TravMaskFieldMask      =
        (TypeTraits<BitVector>::One << Node::TravMaskFieldId   );
const BitVector Node::ParentFieldMask      =
        (TypeTraits<BitVector>::One << Node::ParentFieldId     );
const BitVector Node::ChildrenFieldMask    =
        (TypeTraits<BitVector>::One << Node::ChildrenFieldId   );
const BitVector Node::CoreFieldMask        =
        (TypeTraits<BitVector>::One << Node::CoreFieldId       );
const BitVector Node::FlagsFieldMask        =
        (TypeTraits<BitVector>::One << Node::FlagsFieldId      );
const BitVector Node::ExternalFlagsFieldMask =
    (TypeTraits<BitVector>::One << Node::ExternalFlagsFieldId      );
const BitVector Node::IndexFieldMask =
    (TypeTraits<BitVector>::One << Node::IndexFieldId      );

FieldDescription *Node::_desc[] =
{

    new FieldDescription(SFBoxVolume::getClassType(),
                         "volume",
                         OSG_FC_FIELD_IDM_DESC(VolumeField),
                         FieldDescription::FDExternal,
                         reinterpret_cast<FieldAccessMethod>(
                             &Node::editSFVolume)),

    // Yes, this is wrong, it should be an UInt32, but changing
    // it now will break all old .osb files, and this info is
    // nearly never necessary to be loaded from files.
    new FieldDescription(SFBool::getClassType(),
                         "travMask",
                         OSG_FC_FIELD_IDM_DESC(TravMaskField),
                         FieldDescription::FDExternal,
                         reinterpret_cast<FieldAccessMethod>(
                             &Node::editSFTravMask)),

    new FieldDescription(SFNodePtr::getClassType(),
                         "parent",
                         OSG_FC_FIELD_IDM_DESC(ParentField),
                         FieldDescription::FDInternal,
                         reinterpret_cast<FieldAccessMethod>(
                             &Node::editSFParent)),

    new FieldDescription(MFNodePtr::getClassType(),
                         "children",
                         OSG_FC_FIELD_IDM_DESC(ChildrenField),
                         FieldDescription::FDExternal,
                         reinterpret_cast<FieldAccessMethod>(
                             &Node::editMFChildren)),

    new FieldDescription(SFNodeCorePtr::getClassType(),
                         "core",
                         OSG_FC_FIELD_IDM_DESC(CoreField),
                         FieldDescription::FDExternal,
                         reinterpret_cast<FieldAccessMethod>(&Node::editSFCore)),

    new FieldDescription(SFUInt32::getClassType(),
                         "flags",
                         OSG_FC_FIELD_IDM_DESC(FlagsField),
                         FieldDescription::FDInternal,
                         reinterpret_cast<FieldAccessMethod>(
                             &Node::editSFFlags)),

    new FieldDescription(SFUInt32::getClassType(),
                         "externalFlags",
                         OSG_FC_FIELD_IDM_DESC(ExternalFlagsField),
                         FieldDescription::FDExternal,
                         reinterpret_cast<FieldAccessMethod>(
                             &Node::editSFExternalFlags)),

    new FieldDescription(SFUInt32::getClassType(),
                         "index",
                         OSG_FC_FIELD_IDM_DESC(IndexField),
                         FieldDescription::FDInternal,
                         reinterpret_cast<FieldAccessMethod>(
                             &Node::editSFIndex)),
};

FieldContainerType Node::_type(
    "Node",
    "AttachmentContainer",
    0,
    reinterpret_cast<PrototypeCreateF>(&Node::createEmpty),
    0,
    _desc,
    sizeof(_desc));


const NodePtr Node::NullNode(NullFC);

OSG_FIELD_CONTAINER_DEF(Node, NodePtr)

/*-------------------------------------------------------------------------*/
/*                                Set                                      */

void Node::setCore(const NodeCorePtr &core)
{
    NodePtr thisP = getPtr();

    thisP.setParentFieldPos(CoreFieldId);

    addRefCP(core);

    if(_sfCore.getValue() != NullFC)
    {
        beginEditCP(_sfCore.getValue(), NodeCore::ParentsFieldMask);
        {
            _sfCore.getValue()->subParent(thisP);
        }
        endEditCP(_sfCore.getValue(), NodeCore::ParentsFieldMask);

        subRefCP(_sfCore.getValue());
    }

    _sfCore.setValue(core);

    if(_sfCore.getValue() != NullFC)
    {
        beginEditCP(_sfCore.getValue(), NodeCore::ParentsFieldMask);
        {
            _sfCore.getValue()->addParent(thisP);
        }
        endEditCP(_sfCore.getValue(), NodeCore::ParentsFieldMask);
    }

    // TODO Check if required (GV)
    invalidateVolume();
}

/*-------------------------------------------------------------------------*/
/*                             Children                                    */

void Node::addChild(const NodePtr &childP)
{
    if(childP != NullFC)
    {
        // do the ref early, to prevent destroys on getParent(a)->addChild(a)
        addRefCP(childP);

        // already somebody else's child?
        NodePtr parent = childP->getParent();
        if(parent != NullFC)
        {
            beginEditCP(parent, Node::ChildrenFieldMask);
            parent->subChild(childP);
            endEditCP(parent, Node::ChildrenFieldMask);
        }

        _mfChildren.push_back(childP);
#ifdef VR_SUPPORT_INDEX
        childP->setIndex(_mfChildren.size() - 1);
#endif
        beginEditCP(childP, Node::ParentFieldMask);
        {
            childP->setParent(getPtr());
        }
        endEditCP(childP, Node::ParentFieldMask);

        // TODO Check if required (GV)
#ifndef OSG_GV_BETA
        invalidateVolume();
#endif
    }
}

void Node::moveChildren( NodePtr& nodeP)
{
    if( nodeP == NullFC)
        return;

    const size_t nChildren = nodeP->getNChildren();
    // build a temprary vector for all the children
    std::vector<NodePtr> otherChildren;
    otherChildren.reserve(nChildren);
    for( size_t childIdx = 0; childIdx < nChildren; ++childIdx)
    {
        NodePtr childP = nodeP->getChild(childIdx);
        if(childP == NullFC)
            continue;
        addRefCP(childP);
        otherChildren.push_back(childP);
    }
    // remove the children from the old parent
    nodeP->clearChildren();

    // add them to the new parent
    for( size_t childIdx = 0; childIdx < otherChildren.size(); ++childIdx)
    {
        NodePtr childP = otherChildren[childIdx];
        _mfChildren.push_back(childP);
#ifdef VR_SUPPORT_INDEX
        childP->setIndex(_mfChildren.size() - 1);
#endif
        beginEditCP(childP, Node::ParentFieldMask);
        {
            childP->setParent(getPtr());
        }
        endEditCP(childP, Node::ParentFieldMask);
    }
}

void Node::insertChild(UInt32 childIndex, const NodePtr &childP)
{
    MFNodePtr::iterator childIt = _mfChildren.begin();

    if(childP != NullFC)
    {
        // do the ref early, to prevent destroys on getParent(a)->addChild(a)
        addRefCP(childP);

        // already somebody else's child?
        NodePtr parent = childP->getParent();
        if(parent != NullFC)
        {
            beginEditCP(parent, Node::ChildrenFieldMask);
            parent->subChild(childP);
            endEditCP(parent, Node::ChildrenFieldMask);
        }

        childIt += childIndex;

        _mfChildren.insert(childIt, childP);
#ifdef VR_SUPPORT_INDEX
        for(MFNodePtr::iterator it = _mfChildren.begin() + childIndex; it != _mfChildren.end(); ++it)
        {
            if((*it) != NullFC)
                (*it)->setIndex(it - _mfChildren.begin());
        }
#endif
        beginEditCP(childP, Node::ParentFieldMask);
        {
            childP->setParent(getPtr());
        }
        endEditCP(childP, Node::ParentFieldMask);
    }

    // TODO check if required (GV)
#ifndef OSG_GV_BETA
    invalidateVolume();
#endif
}

void Node::insertChild(const NodePtr &posChildP, const NodePtr &childP, bool after)
{
    if(posChildP == childP)
        return;

    if(childP == NullFC)
        return;

    MFNodePtr::iterator childIt = _mfChildren.find(posChildP);

    if(childIt == _mfChildren.end())
        return;

    // do the ref early, to prevent destroys on getParent(a)->addChild(a)
    addRefCP(childP);

    // already somebody else's child?
    NodePtr parent = childP->getParent();
    if(parent != NullFC)
    {
        beginEditCP(parent, Node::ChildrenFieldMask);
        parent->subChild(childP);
        endEditCP(parent, Node::ChildrenFieldMask);
    }

    // ok the subChild invalidates the childIt iterator!
    childIt = _mfChildren.find(posChildP);
    if(after)
        ++childIt;
    UInt32 childIndex = childIt - _mfChildren.begin();
    _mfChildren.insert(childIt, childP);
#ifdef VR_SUPPORT_INDEX
    for(MFNodePtr::iterator it = _mfChildren.find(childP); it != _mfChildren.end(); ++it)
    {
        if((*it) != NullFC)
            (*it)->setIndex(it - _mfChildren.begin());
    }
#endif
    beginEditCP(childP, Node::ParentFieldMask);
    {
        childP->setParent(getPtr());
    }
    endEditCP(childP, Node::ParentFieldMask);
}

void Node::replaceChild(UInt32 childIndex, const NodePtr &childP)
{
    if(childP != NullFC && childIndex < _mfChildren.size())
    {
        // do the ref early, to prevent destroys on getParent(a)->addChild(a)
        addRefCP(childP);
        // already somebody else's child?
        // moved it up could be a child of childIndex.
        NodePtr parent = childP->getParent();
        if(parent != NullFC)
        {
            beginEditCP(parent, Node::ChildrenFieldMask);
            parent->subChild(childP);
            endEditCP(parent, Node::ChildrenFieldMask);
        }

        // remove the current child
        beginEditCP(_mfChildren[childIndex], Node::ParentFieldMask);
        {
            _mfChildren[childIndex]->setParent(NullNode);
        }
        endEditCP(_mfChildren[childIndex], Node::ParentFieldMask);

        subRefCP(_mfChildren[childIndex]);

        // set the new child
        _mfChildren[childIndex] = childP;
#ifdef VR_SUPPORT_INDEX
        childP->setIndex(childIndex);
#endif
        beginEditCP(childP, Node::ParentFieldMask);
        {
            childP->setParent(getPtr());
        }
        endEditCP(childP, Node::ParentFieldMask);
    }

    // TODO check if required (GV)
#ifndef OSG_GV_BETA
    invalidateVolume();
#endif
}

//! return true on success, false on child not found

bool Node::replaceChildBy(const NodePtr &childP,
                          const NodePtr &newChildP)
{
    MFNodePtr::iterator childIt = _mfChildren.find(childP);

    if(newChildP != NullFC)
    {
        if(childIt != _mfChildren.end())
        {
            // do the ref early, to prevent destroys on
            // getParent(a)->addChild(a)

            addRefCP(newChildP);
            // already somebody else's child?
            // moved it up could be a child of childP.
            NodePtr parent = newChildP->getParent();
            if(parent != NullFC)
            {
                beginEditCP(parent, Node::ChildrenFieldMask);
                parent->subChild(newChildP);
                endEditCP(parent, Node::ChildrenFieldMask);
            }

            beginEditCP(childP, Node::ParentFieldMask);
            {
                childP->setParent(NullNode);
            }
            endEditCP(childP, Node::ParentFieldMask);

            subRefCP(childP);

            (*childIt) = newChildP;
#ifdef VR_SUPPORT_INDEX
            newChildP->setIndex(childIt - _mfChildren.begin());
#endif
            beginEditCP(newChildP, Node::ParentFieldMask);
            {
                newChildP->setParent(getPtr());
            }
            endEditCP(newChildP, Node::ParentFieldMask);

            // TODO check if required (GV)
#ifndef OSG_GV_BETA
            invalidateVolume();
#endif

            return true;
        }
    }

    return false;
}

Int32 Node::findChild(const NodePtr &childP) const
{
    UInt32 index;

    for(index = 0; index < _mfChildren.size(); index++)
    {
        if( _mfChildren[index] == childP)
            break;
    }

    if(index < _mfChildren.size())
        return index;
    else
        return -1;
}

void Node::subChild(const NodePtr &childP)
{
    if (childP == NullFC)
        return;
    auto index = childP->getIndex();
    MFNodePtr::iterator childIt;
    if (index < _mfChildren.size() && _mfChildren[index] == childP)
        childIt = _mfChildren.begin() + index;
    else
        childIt = _mfChildren.find(childP);
    if(childIt != _mfChildren.end())
    {
        beginEditCP(childP, Node::ParentFieldMask);
        {
            childP->setParent(NullNode);
        }
        endEditCP(childP, Node::ParentFieldMask);      
#ifdef VR_SUPPORT_INDEX
        size_t childIdx = childIt - _mfChildren.begin();
        _mfChildren.erase(childIt);
        subRefCP(childP);
        for( size_t idx = childIdx; idx < _mfChildren.size(); ++idx)
        {
            if(_mfChildren[idx] != NullFC)
                _mfChildren[idx]->setIndex(idx);
        }
#else
        subRefCP(childP);
        _mfChildren.erase(childIt);
#endif
    }
    else
    {
        SWARNING << "Node(" << this << ")::subChild: " << childP
                 << " is not one of my children!" << std::endl;
    }

    // TODO check if required (GV)
#ifndef OSG_GV_BETA
    invalidateVolume();
#endif
}

void Node::subChild(UInt32 childIndex)
{
    MFNodePtr::iterator childIt = _mfChildren.begin();

    childIt += childIndex;

    if(childIt != _mfChildren.end())
    {
        beginEditCP(*childIt, Node::ParentFieldMask);
        {
            (*childIt)->setParent(NullNode);
        }
        endEditCP(*childIt, Node::ParentFieldMask);

       const NodePtr childP = *childIt;
#ifdef VR_SUPPORT_INDEX
        size_t childIdx = childIt - _mfChildren.begin();
        _mfChildren.erase(childIt);
        subRefCP(childP);
        for( size_t idx = childIdx; idx < _mfChildren.size(); ++idx)
        {
            if(_mfChildren[idx] != NullFC)
                _mfChildren[idx]->setIndex(idx);
        }
#else 
        _mfChildren.erase(childIt);
        subRefCP(childP);
#endif
    }

    // TODO check if required (GV)
#ifndef OSG_GV_BETA
    invalidateVolume();
#endif
}

void Node::clearChildren(void)
{
    for(auto childIt = _mfChildren.begin();childIt != _mfChildren.end();++childIt)
    {
        beginEditCP(*childIt, Node::ParentFieldMask);
        {
            (*childIt)->setParent(NullNode);
        }
        endEditCP(*childIt, Node::ParentFieldMask);
        subRefCP(*childIt);
    }
    _mfChildren.clear();
}

/*-------------------------------------------------------------------------*/
/*                           Get Transformation                            */

Matrix Node::getToWorld(void)
{
    Matrix tmp;

    getToWorld(tmp);

    return tmp;
}

void Node::getToWorld(Matrix &result)
{
    NodePtr parent = getParent();
    if(parent != NullFC)
    {
        parent->getToWorld(result);
    }
    else
    {
        result.setIdentity();
    }
    NodeCorePtr core = getCore();
    if(core != NullFC)
        core->accumulateMatrix(result);
}

/*-------------------------------------------------------------------------*/
/*                           Volume                                        */
void Node::getWorldVolume(BoxVolume &result)
{
    Matrix m;
    NodePtr parent = getParent();
    if(parent != NullFC)
    {
        parent->getToWorld(m);
    }
    else
    {
        m.setIdentity();
    }

    updateVolume();

    result = getVolume();
    result.transform(m);
/*
Pnt3f low,high;
result.getBounds(low,high);
fprintf(stderr,"%p: node 0x%p gwv (%f %f %f  %f %f %f)\n",
            Thread::getCurrent(), this,
            low[0], low[1], low[2],
            high[0], high[1], high[2] );
*/
}

void Node::updateVolume(void)
{
    if(_sfVolume.getValue().isValid() == true ||
        !getVolumeActive())
    {
        return;             // still valid, nothing to do
    }
    // be careful to not change the real volume. If two threads
    // are updating the same aspect this will lead to chaos
    BoxVolume vol = _sfVolume.getValue();

//fprintf(stderr,"%p: node 0x%p update needed\n", Thread::getCurrent(), this);

    MFNodePtr::iterator it;

    vol.setEmpty();

    for(it = _mfChildren.begin(); it != _mfChildren.end(); ++it)
    {
        if(*it != NullFC && (*it)->getVolumeActive())
        {
            (*it)->updateVolume();
            vol.extendBy((*it)->getVolume());
        }
    }

    // test for null core. Shouldn't happen, but just in case...
    if(getCore() != NullFC)
        getCore()->adjustVolume(vol);

    NodePtr thisP = getPtr();

    beginEditCP(thisP, VolumeFieldMask);

    _sfVolume.setValue(vol);

    endEditCP(thisP, VolumeFieldMask);

}

void Node::invalidateVolume(void)
{
    Volume &vol = _sfVolume.getValue();
    if(vol.isValid() == true && vol.isStatic() == false)
    {
        NodePtr thisP = getPtr();

        beginEditCP(thisP, VolumeFieldMask);

        vol.setValid(false);

        endEditCP(thisP, VolumeFieldMask);

        if(getParent() != NullFC)
        {
            getParent()->invalidateVolume();
        }
    }
}


bool Node::getSelfLocked()
{
    NodeCorePtr core = getCore();
    bool isInternal = hasFlag(NFInternal | NFAuxiliary);
    if(core != NullFC && !isInternal)
    {
        return core->getSelfLocked();
    }
    else
    {
        // Fallback for nodes without core (should not happen) and internal nodes
        return queryLock(this);
    }
}

bool Node::getSelfUneditedAsset()
{
    NodeCorePtr core = getCore();
    bool isInternal = hasFlag(NFInternal | NFAuxiliary);
    if(core != NullFC && !isInternal)
    {
        return core->getSelfUneditedAsset();
    }
    else
    {
        // Fallback for nodes without core (should not happen) and internal nodes
        return queryIsUneditedAssetRoot(this);
    }
}

bool Node::getCoreIsLockedByAncestor()
{
    NodeCorePtr core = getCore();
    bool isInternal = hasFlag(NFInternal | NFAuxiliary);
    if(core != NullFC && !isInternal)
    {
        return core->getIsLockedByAncestor();
    }
    else
    {
        // Fallback for nodes without core (should not happen) and internal nodes
        NodePtr parent = getParent();
        if(parent != NullFC)
        {
             // Fallback for node without core (should not happen but in case it does)    
            return parent->getSelfLocked() || parent->getSelfUneditedAsset() || parent->getCoreIsLockedByAncestor();
        }
        return false;
    }
}

void Node::invalidateSelfLockCache()
{
    // Invalidate cached self lock states
    auto& core = _sfCore.getValue();
    if(core != NullFC)
    {
        core->editSelfLockFlag().store(0, std::memory_order_relaxed);
        core->editSelfAssetFlag().store(0, std::memory_order_relaxed);
    }
}


/*-------------------------------------------------------------------------*/
/*                              Changed                                    */

void Node::changed(BitVector  whichField,
                   UInt32     origin    )
{
    Inherited::changed(whichField, origin);

    if(whichField & CoreFieldMask)
    {
        invalidateVolume();
    }

    if(whichField & TravMaskFieldMask)
    {
        beginEditCP(getParent(), Node::VolumeFieldMask);
        if(getParent() != NullFC)
        {
            getParent()->invalidateVolume();
        }
        else
        {
            invalidateVolume();
        }
        endEditCP(getParent(), Node::VolumeFieldMask);
    }
    
    if(whichField & ChildrenFieldMask)
    {
        invalidateVolume();

        
    }

    // Invalidation of cached node states
    if(whichField & (AttachmentsFieldMask | CoreFieldMask | ParentFieldMask))
    {
        if(!osg::SceneFileHandler::the().isReading())
        {
            if(!hasFlag(NFInternal | NFAuxiliary)) // ignore changes on internal nodes
            {
                if(whichField & (AttachmentsFieldMask | CoreFieldMask))
                {
                    invalidateSelfLockCache();
                }
                invalidateAncestorLockCache();
            }
        }
    }
}



/*-------------------------------------------------------------------------*/
/*                                Dump                                     */

void Node::dump(      UInt32    uiIndent,
                const BitVector bvFlags) const
{
    UInt32 i;

    NodePtr thisP = getPtr();

    indentLog(uiIndent, PLOG);

    PLOG << "Node"
         << "("
         << getContainerId(thisP)
         << ") : "
         << _mfChildren.size()
         << " children | "
         << _attachmentMap.getValue().size()
         << " attachments | "
         << "Parent : " << std::hex;

    if(_sfParent.getValue() != NullFC)
        PLOG << "0x" << &(*(_sfParent.getValue())) << " | ";
    else
        PLOG << "NULL | ";

    PLOG << "0x" << this << std::dec << std::endl;

    indentLog(uiIndent, PLOG);

    PLOG << "[" << std::endl;

    if(_sfCore.getValue() != NullFC)
    {
        _sfCore.getValue()->dump(uiIndent + 4, bvFlags);
    }
    else
    {
        indentLog(uiIndent + 4, PLOG);
        PLOG << "Core : " << "NULL" << std::endl;
    }

    Inherited::dump(uiIndent, bvFlags);

    indentLog(uiIndent, PLOG);
    PLOG << "]" << std::endl;

    indentLog(uiIndent, PLOG);

    PLOG << "{" << std::endl;

    for(i = 0; i < _mfChildren.size(); i++)
    {
        _mfChildren[i]->dump(uiIndent + 4, bvFlags);
        PLOG << std::endl;
    }


    indentLog(uiIndent, PLOG);

    PLOG << "}" << std::endl;

/*
    for(i = 0; i < indent; i++)
    {
        cerr << " ";
    }
    cerr << "  Parent : ";


    for(i = 0; i < indent; i++)
    {
        cerr << " ";
    }

    for(i = 0; i < indent; i++)
        fprintf(stderr, " ");

    fprintf(stderr, "NAttachments : \n");

    map<UInt32, AttachmentPtr>::const_iterator fcI;

    fcI = _attachmentMap.getValue().begin();

    while(fcI != _attachmentMap.getValue().end())
    {
        (*fcI).second->dump(indent + 2);
        ++fcI;
    }

    for(i = 0; i < indent; i++)
    {
        cerr << " ";
    }

    cerr << "{" << std::endl;


    for(i = 0; i < indent; i++)
    {
        cerr << " ";
    }

    cerr << "}" << std::endl;
*/
}

/*-------------------------------------------------------------------------*/
/*                            Constructors                                 */

Node::Node(void) :
     Inherited    (),
    _sfVolume     (),
    _sfTravMask   (TypeTraits<UInt32>::getMax()),
    _sfParent     (),
    _mfChildren   (),
    _sfCore       (),
    _sfFlags      (0),
    _occlusionMask(0),
    _sfExternalFlags(0),
    _sfIndex(0)
{
}

Node::Node(const Node &source) :
     Inherited       (source),
    _sfVolume        (source._sfVolume),
    _sfTravMask      (source._sfTravMask),
    _sfParent        (),
    _mfChildren      (),
    _sfCore          (),
    _sfFlags         (source._sfFlags),
    _occlusionMask   (source._occlusionMask),
    _sfExternalFlags(source._sfExternalFlags),
    _sfIndex(source._sfIndex)
{
}

/*-------------------------------------------------------------------------*/
/*                             Destructor                                  */

Node::~Node(void)
{
    if(_sfCore.getValue() != NullFC)
    {
        NodePtr thisP = getPtr();

        beginEditCP(_sfCore.getValue(), NodeCore::ParentsFieldMask);
        {
            _sfCore.getValue()->subParent(thisP);
        }
        endEditCP(_sfCore.getValue(), NodeCore::ParentsFieldMask);

        subRefCP(_sfCore.getValue());
    }

    MFNodePtr::iterator       vChildIt    = _mfChildren.begin();
    MFNodePtr::const_iterator endChildren = _mfChildren.end  ();

    while(vChildIt != endChildren)
    {
        beginEditCP(*vChildIt, Node::ParentFieldMask);
        {
            (*vChildIt)->setParent(NullNode);
        }
        endEditCP(*vChildIt, Node::ParentFieldMask);

        subRefCP(*vChildIt);

        ++vChildIt;
    }    
}

#if defined(OSG_FIXED_MFIELDSYNC)
void Node::onDestroyAspect(UInt32 uiId, UInt32 uiAspect)
{
    _mfChildren.terminateShare(uiAspect, this->getContainerSize());
}
#endif

void osg::addAttachments(const AttachmentContainerPtr &src,
    AttachmentContainerPtr dst)
{
    if(src == NullFC || dst == NullFC)
        return;

    // add all attachments.
    SFAttachmentMap *amap = static_cast<SFAttachmentMap *>(src->getSFAttachments());
    AttachmentMap::const_iterator   mapIt = amap->getValue().begin();
    AttachmentMap::const_iterator   mapEnd = amap->getValue().end();

    beginEditCP(dst, AttachmentContainer::AttachmentsFieldMask);
    for(; mapIt != mapEnd; ++mapIt)
    {
        if(mapIt->second != NullFC)
            dst->addAttachment(AttachmentPtr::dcast(mapIt->second));
    }
    endEditCP(dst, AttachmentContainer::AttachmentsFieldMask);
}

NodePtr osg::cloneTree(const NodePtr &pRootNode, bool cloneAttachments, CloneTreeNodeMapper* nodeMapper)
{
    NodePtr returnValue = NullFC;

    if(pRootNode != NullFC)
    {
        NodePtr pChildClone = NullFC;

        returnValue = Node::create();

        if ( nodeMapper != nullptr )
            (*nodeMapper)[pRootNode.getFieldContainerId()] = returnValue.getFieldContainerId();

        if(cloneAttachments)
        {
            osg::addAttachments(pRootNode, returnValue);
        }
        else
        {
            // copy name
            osg::copyName(pRootNode, returnValue);
        }

        beginEditCP(returnValue, osg::FieldBits::AllFields);
        {
            returnValue->setTravMask(pRootNode->getTravMask());
            returnValue->setCore    (pRootNode->getCore());
            returnValue->setExternalFlags(pRootNode->getExternalFlags());

            for(UInt32 i = 0; i < pRootNode->getNChildren(); i++)
            {
                pChildClone = cloneTree(pRootNode->getChild(i), cloneAttachments, nodeMapper);
                returnValue->addChild(pChildClone);
            }
        }
        endEditCP(returnValue, osg::FieldBits::AllFields);
    }

    return returnValue;
}

// we need to do this for the light sources beacon field!
static void fixParents(const NodePtr &node)
{
    if(node == NullFC)
        return;

    NodeCorePtr core = NodeCorePtr::dcast(node->getCore());
    if(core != NullFC)
    {
        beginEditCP(node, osg::FieldBits::AllFields);
            node->setCore(core);
        endEditCP(node, osg::FieldBits::AllFields);
    }

    beginEditCP(node, osg::FieldBits::AllFields);
        std::vector<NodePtr> childs;
        for(UInt32 i=0;i<node->getNChildren();++i)
        {
            NodePtr child = node->getChild(i);
            addRefCP(child);
            childs.push_back(child);           
        }
        node->clearChildren();
        for(UInt32 i=0;i<childs.size();++i)
        {
            node->addChild(childs[i]);
            subRefCP(childs[i]);
        }
    endEditCP(node, osg::FieldBits::AllFields);

    for(UInt32 i=0;i<node->getNChildren();++i)
        fixParents(node->getChild(i));
}

// Stores all deep cloned fieldcontainers (source container -> cloned container id)
// This way already cloned containers will be reused in case a FC is used
// more than once in a structure.
// In some cases, e.g., when using the texture manager, objects that are
// set to an osg field are replaced or discarded, in this case created clones
// may be destroyed. Using a fieldContainerId as target of the map allows
// us to check if the created clone is still available.
using ClonedFieldContainers = std::map<FieldContainerPtr, osg::UInt32>;

bool osg::shouldShareContainer(const FieldContainerPtr &fc, const FieldContainerTypes &share, const FieldContainerPtr &parentFC, const FieldDescription* fdesc)
{
    if (fc == NullFC)
        return false;

    // never share non-sharable images
    if (fc->getType().isDerivedFrom(Image::getClassType()))
    {
        const osg::ImagePtr image = osg::ImagePtr::dcast(fc);
        if ((image == NullFC) || !image->isShareable())
            return false;
    }

    // Always share references by constraint objects
    static FieldContainerType *constraintObjectType = FieldContainerFactory::the()->findType("ConstraintObject");
    static FieldDescription* fieldDescription = constraintObjectType->findFieldDescription("reference");
    if ((constraintObjectType != nullptr) && (fdesc != nullptr) && (parentFC != osg::NullFC) && (parentFC->getType() == *constraintObjectType) && (fdesc == fieldDescription))
    {
        return true;
    }

    // Never share AnimNode, even when Node type is in share param
    static FieldContainerType *animNodeType = FieldContainerFactory::the()->findType("AnimNode");
    if ( (animNodeType != nullptr ) && (fc->getType() == *animNodeType))
       return false;

    // VRED-18845: Always share meta data sets
    static FieldContainerType *metadataSetType = FieldContainerFactory::the()->findType("MetadataSet");
    if ( (metadataSetType != nullptr ) && (fc->getType() == *metadataSetType))
        return true;

    for (auto fct : share)
        if ((fct != nullptr) && fc->getType().isDerivedFrom(*fct))
            return true;

    return false;
}



static FieldContainerPtr deepCloneR2(ClonedFieldContainers &cloned,
                                    const FieldContainerPtr &src, const FieldContainerTypes& share);

static FieldContainerPtr getCloneR2(FieldContainerPtr const &fc, const FieldContainerTypes& share, ClonedFieldContainers &cloned)
{
    const auto cit = cloned.find(fc);

    if (cit != cloned.end())
    {
        const osg::FieldContainerPtr alreadyClonedFC = FieldContainerFactory::the()->getContainer((*cit).second);
        if (alreadyClonedFC != NullFC)
            return alreadyClonedFC;

        // The created clone is no longer available.
        // It's likely that the "owning container" (=the container it has been set to) 
        // removed or replaced it. In this case we cleanup the map 
        // and create a new clone.
        cloned.erase(cit);
    }

    const FieldContainerPtr clonedFC = deepCloneR2(cloned, fc, share);
    // we need to do this for the light sources beacon field!
    if ((clonedFC != NullFC) && clonedFC->getType().isDerivedFrom(Node::getClassType()))
        fixParents(NodePtr::dcast(clonedFC));

    return clonedFC;
}

FieldContainerTypes osg::convertShareTypes(const std::vector<std::string> &shares)
{
    FieldContainerTypes sharedTypes;
    for (auto share : shares)
    {
        const auto* shareType = FieldContainerFactory::the()->findType(share.c_str());
        if ( shareType != nullptr)
            sharedTypes.push_back(shareType);
    }

    return sharedTypes;
}

// deep clone of a fieldcontainer.
// we should move this into OSGFieldContainer.cpp?
static FieldContainerPtr deepCloneR2(ClonedFieldContainers &cloned,
                                    const FieldContainerPtr &src, const FieldContainerTypes& share)
{
    if(src == NullFC)
        return NullFC;

    const FieldContainerType &type = src->getType();

    //FDEBUG(("deepClone: fieldcontainertype = %s\n", type.getCName()));
    
    FieldContainerPtr dst = FieldContainerFactory::the()->createFieldContainer(&type); // type.getName().str());
    cloned.insert(std::make_pair(src, dst.getFieldContainerId()));

    //UInt32 fcount = type.getNumFieldDescs();
    // ignore dynamic fields.
    UInt32 fcount = osgMin(type.getNumFieldDescs(), dst->getType().getNumFieldDescs());


    for(UInt32 i=1;i <= fcount;++i)
    {
        const FieldDescription* fdesc = type.getFieldDescription(i);

        if(fdesc->isInternal())
            continue;

        // ignore the parents field, would lead to a endless recursion
        // with the new image parents support!
        if (strcmp(fdesc->getCName(), "parents") == 0)
            continue;

        BitVector mask = fdesc->getFieldMask();

        Field *src_field = src->getField(i);
        

        const FieldType &ftype = src_field->getType();

        // attachements
        if(strcmp(fdesc->getCName(), "attachments") == 0)
        {
            SFAttachmentMap *amap = static_cast<SFAttachmentMap *>(src_field);

            AttachmentMap::const_iterator   mapIt = amap->getValue().begin();
            AttachmentMap::const_iterator   mapEnd = amap->getValue().end();

            beginEditCP(dst, mask);
            for(; mapIt != mapEnd; ++mapIt)
            {
                FieldContainerPtr fc = mapIt->second;
                if (fc == NullFC)
                    continue;

                const bool shareit = shouldShareContainer(fc, share, src, fdesc);
                if(!shareit)
                    fc = getCloneR2(fc, share, cloned);

                if(fc != NullFC)
                    AttachmentContainerPtr::dcast(dst)->addAttachment(
                        AttachmentPtr::dcast(fc));
            }
            endEditCP(dst, mask);
            continue;
        }

        // field
        Field *dst_field = dst->getField(i);
        if(strstr(ftype.getCName(), "Ptr") == NULL)
        {
            beginEditCP(dst, mask);
                dst_field->setAbstrValue(*src_field);
            endEditCP(dst, mask);
        }
        else // field with pointer
        {
            if(src_field->getCardinality() == FieldType::SINGLE_FIELD)
            {
                FieldContainerPtr fc = (static_cast<SFFieldContainerPtr *>(src_field)->getValue());
                if (fc == NullFC)
                    continue;
 
                const bool shareit = shouldShareContainer(fc, share, src, fdesc);
                if(!shareit)
                {
                    fc = getCloneR2(fc, share, cloned);
                }

                if(fc != NullFC)
                {
                    SFFieldContainerPtr *sfcp_field = static_cast<SFFieldContainerPtr *>(dst_field);
                    if(sfcp_field->getValue() != NullFC) // could be created in onCreate()
                        subRefCP(sfcp_field->getValue());
                    // increment reference counter!
                    addRefCP(fc);
                    beginEditCP(dst, mask);
                        sfcp_field->setValue(fc);
                    endEditCP(dst, mask);
                }
            }
            else if(src_field->getCardinality() == FieldType::MULTI_FIELD)
            {
                
                beginEditCP(dst, mask);
                for (UInt32 j = 0; j < (static_cast<MFFieldContainerPtr *>(src_field))->size(); ++j)
                {
                    FieldContainerPtr fc = (*((static_cast<MFFieldContainerPtr *>(src_field))))[j];
                    if (fc == NullFC)
                        continue;

                    const bool shareit = shouldShareContainer(fc, share, src, fdesc);
                    if (!shareit)
                        fc = getCloneR2(fc, share, cloned);

                    if (fc != NullFC)
                    {
                        // increment reference counter!
                        addRefCP(fc);
                        (static_cast<MFFieldContainerPtr *>(dst_field))->push_back(fc);
                    }
                }
                endEditCP(dst, mask);
            }
        }
    }
    dst->onDeepClone();
    return dst;
}

std::vector<std::string> getCommaSeparatedNames(const std::string& str)
{
    std::vector<std::string> result;

    // parse comma separated names.
    std::string::const_iterator nextComma;
    std::string::const_iterator curPos = str.begin();
    while(curPos < str.end())
    {
        nextComma = std::find(curPos, str.end(), ',');
        // strip leading spaces
        curPos = std::find_if(curPos, nextComma, [](unsigned char c){ return !std::isspace(c); });
        result.push_back(std::string(curPos, nextComma));
        if(nextComma != str.end())
            curPos = ++nextComma;
        else
            break;
    }

    return result;
}

// deep clone of a fieldcontainer.
// we should move this into OSGFieldContainer.cpp?
FieldContainerPtr osg::deepClone(const FieldContainerPtr &src,
                                 const std::vector<std::string> &share)
{
    ClonedFieldContainers cloned;
    return deepCloneR2(cloned, src, convertShareTypes(share));
}

FieldContainerPtr osg::deepClone(const FieldContainerPtr &src,
                                 const std::vector<UInt16> &shareGroupIds)
{
    std::vector<std::string> share;
    share.reserve(shareGroupIds.size());
    for(UInt32 i=0;i<shareGroupIds.size();++i)
    {
        const char *name = FieldContainerFactory::the()->findGroupName(shareGroupIds[i]);
        if(name != NULL)
            share.push_back(name);
    }
    return osg::deepClone(src, share);
}

// shareString is a comma separated FieldContainer type list
// e.g. "Material, Geometry"
FieldContainerPtr osg::deepClone(const FieldContainerPtr &src,
                                 const std::string &shareString)
{
    std::vector<std::string> share = getCommaSeparatedNames(shareString);
    return osg::deepClone(src, share);
}

// deep clone of attachements.
static void deepCloneAttachmentsR(ClonedFieldContainers &cloned,
                                  const AttachmentContainerPtr &src,
                                  AttachmentContainerPtr dst,
                                  const std::vector<std::string> &share)
{
    SFAttachmentMap *amap =
        static_cast<SFAttachmentMap *>(src->getSFAttachments());

    AttachmentMap::const_iterator   mapIt = amap->getValue().begin();
    AttachmentMap::const_iterator   mapEnd = amap->getValue().end();

    beginEditCP(dst, AttachmentContainer::AttachmentsFieldMask);
    const auto sharedTypes = convertShareTypes(share);
    for(; mapIt != mapEnd; ++mapIt)
    {
        FieldContainerPtr fc = mapIt->second;

        const bool shareit = shouldShareContainer(fc, sharedTypes);
        if(!shareit)
            fc = getCloneR2(fc, sharedTypes, cloned);

        if(fc != NullFC)
            dst->addAttachment(AttachmentPtr::dcast(fc));
    }
    endEditCP(dst, AttachmentContainer::AttachmentsFieldMask);
}

// deep clone of attachements.
void osg::deepCloneAttachments(const AttachmentContainerPtr &src,
                               AttachmentContainerPtr dst,
                               const std::vector<std::string> &share)
{
    ClonedFieldContainers cloned;
    deepCloneAttachmentsR(cloned, src, dst, share);
}

void osg::deepCloneAttachments(const AttachmentContainerPtr &src,
                               AttachmentContainerPtr dst,
                               const std::vector<UInt16> &shareGroupIds)
{
    std::vector<std::string> share;
    share.reserve(shareGroupIds.size());
    for(UInt32 i=0;i<shareGroupIds.size();++i)
    {
        const char *name = FieldContainerFactory::the()
                           ->findGroupName(shareGroupIds[i]);
        if(name != NULL)
            share.push_back(name);
    }
    osg::deepCloneAttachments(src, dst, share);
}

// shareString is a comma separated FieldContainer type list
// e.g. "Material, Geometry"
void osg::deepCloneAttachments(const NodePtr &src, NodePtr &dst,
                               const std::string &shareString)
{
    std::vector<std::string> share = getCommaSeparatedNames(shareString);
    osg::deepCloneAttachments(src, dst, share);
}

// deep clone of nodes.
static NodePtr deepCloneTreeR(const NodePtr &src,
                              const std::vector<std::string> &share)
{
    // FieldContainers that are shared within the src node should remain shared within the 
    // duplicated node. However, they should not be shared across the entire tree.
    ClonedFieldContainers cloned;

    NodePtr dst = NullFC;

    if(src != NullFC)
    {
        dst = Node::create();
        cloned.insert(std::make_pair(src, dst.getFieldContainerId()));
        deepCloneAttachmentsR(cloned, src, dst, share);

        beginEditCP(dst, osg::FieldBits::AllFields);
        {
            dst->setActive(src->getActive());
            dst->setTravMask(src->getTravMask());
            dst->setExternalFlags(src->getExternalFlags());
            dst->setCore(NodeCorePtr::dcast(deepCloneR2(cloned, src->getCore(), convertShareTypes(share))));

            for(UInt32 i = 0; i < src->getNChildren(); i++)
                dst->addChild(deepCloneTreeR(src->getChild(i), share));
        }
        endEditCP(dst, osg::FieldBits::AllFields);
    }

    return dst;
}

// deep clone of nodes.
NodePtr osg::deepCloneTree(const NodePtr &src,
                           const std::vector<std::string> &share)
{
    return deepCloneTreeR(src, share);
}

NodePtr osg::deepCloneTree(const NodePtr &src,
                           const std::vector<UInt16> &shareGroupIds)
{
    std::vector<std::string> share;
    share.reserve(shareGroupIds.size());
    for(UInt32 i=0;i<shareGroupIds.size();++i)
    {
        const char *name = FieldContainerFactory::the()->findGroupName(shareGroupIds[i]);
        if(name != NULL)
            share.push_back(name);
    }
    return osg::deepCloneTree(src, share);
}

// shareString is a comma separated FieldContainer type list
// e.g. "Material, Geometry"
NodePtr osg::deepCloneTree(const NodePtr &src,
                           const std::string &shareString)
{
    std::vector<std::string> share = getCommaSeparatedNames(shareString);
    return osg::deepCloneTree(src, share);
}


static bool hasExcludedParent(const NodePtr &src, const std::set<osg::NodeCorePtr>& excludedNodeCores)
{
    if(src == NullFC)
        return false;

    NodePtr parent = src->getParent();
    while(parent != NullFC)
    {
        NodeCorePtr parentCore = parent->getCore();
        if(excludedNodeCores.find(parentCore) != excludedNodeCores.end())
            return true; // parent with excluded core found

        parent = parent->getParent();
    }

    return false;
}

static bool isExcluded(const NodePtr &src, const NodeCorePtr &srcCore, const std::set<osg::NodeCorePtr>& excludedNodeCores)
{
    if(excludedNodeCores.empty())
        return false;
    bool isExcluded = excludedNodeCores.find(srcCore) != excludedNodeCores.end();
    return (isExcluded || hasExcludedParent(src, excludedNodeCores));
}

static NodePtr deepCloneTreeKeepSharedR(ClonedFieldContainers &cloned,
    std::map<NodeCorePtr, NodeCorePtr> &clonedCores,
    const NodePtr &src,
    const std::vector<std::string> &share,
    const std::set<osg::NodeCorePtr>& excludedNodeCores)
{
    NodePtr dst = NullFC;

    if(src != NullFC)
    {
        NodeCorePtr srcCore = src->getCore();

        dst = Node::create();
        cloned.insert(std::make_pair(src, dst.getFieldContainerId()));

        beginEditCP(dst, osg::FieldBits::AllFields);
        {
            dst->setActive(src->getActive());
            dst->setTravMask(src->getTravMask());
            dst->setExternalFlags(src->getExternalFlags());

            if(isExcluded(src, srcCore, excludedNodeCores))
            {
                dst->setCore(srcCore);
                osg::addAttachments(src, dst);
            }
            else
            {
                deepCloneAttachmentsR(cloned, src, dst, share);

                std::map<NodeCorePtr, NodeCorePtr>::const_iterator cit = clonedCores.find(srcCore);
                if(cit != clonedCores.end())
                {
                    dst->setCore(cit->second);
                }
                else
                {
                    osg::NodeCorePtr clonedCore = NodeCorePtr::dcast(deepCloneR2(cloned, srcCore, convertShareTypes(share)));
                    dst->setCore(clonedCore);
                    clonedCores.insert(std::make_pair(srcCore, clonedCore));
                }
            }

            for(UInt32 i = 0; i < src->getNChildren(); i++)
                dst->addChild(deepCloneTreeKeepSharedR(cloned, clonedCores, src->getChild(i), share, excludedNodeCores));
        }
        endEditCP(dst, osg::FieldBits::AllFields);
    }

    return dst;
}

NodePtr osg::deepCloneTreeKeepShared(const NodePtr &src,
    const std::vector<std::string> &share,
    const std::set<osg::NodeCorePtr>& excludedNodeCores)
{
    ClonedFieldContainers cloned;
    // stores all deep cloned cores and source core (source core -> cloned core)
    std::map<NodeCorePtr, NodeCorePtr> clonedCores;
    return deepCloneTreeKeepSharedR(cloned, clonedCores, src, share, excludedNodeCores);
}

/*! \brief Special behaviour for duplicating a node that has shared nodes
*          in its subtree.
*
*  See VRED-3177.
*  The duplicate's children will have shared nodes but will have 
*  no connection to the original (compare to Maya "Duplicate Input Graph")
*
*  \param shareString is a comma separated FieldContainer type list
*         e.g. "Material, Geometry"
*/
NodePtr osg::deepCloneTreeKeepShared(const NodePtr &src, 
                                     const std::string &shareString,
                                     const std::set<osg::NodeCorePtr>& excludedNodeCores)
{
    std::vector<std::string> share = getCommaSeparatedNames(shareString);
    return deepCloneTreeKeepShared(src, share, excludedNodeCores);
}


/*! \brief Replaces the attachment pointed to by iter by a new one
*/
void osg::replaceAttachment(AttachmentContainerPtr container, osg::AttachmentMap::iterator iter, osg::AttachmentPtr newAtt)
{
    osg::AttachmentPtr att = iter->second;
    if(att == NullFC || newAtt == NullFC)
        return;

    if(att->getGroupId() != newAtt->getGroupId())
    {
        SFATAL << "replaceAttachment: groupIds of attachments do not match: " << att->getGroupId() << " != " << newAtt->getGroupId() << std::endl;
        return;
    }
    
    addRefCP(newAtt);
    beginEditCP(newAtt, Attachment::ParentsFieldMask);
    {
        newAtt->addParent(container);
    }
    endEditCP(newAtt, Attachment::ParentsFieldMask);

    beginEditCP(att, Attachment::ParentsFieldMask);
    {
        att->subParent(container);
    }
    endEditCP(att, Attachment::ParentsFieldMask);
    subRefCP(att);

    iter->second = newAtt;
}

/*! \brief Replaces existing attachments of a container by duplicates.
*
*   \param cloned map of already cloned containers for reuse
*/
static void unshareAttachments(ClonedFieldContainers &cloned, const AttachmentContainerPtr &container, const FieldContainerTypes &sharedTypes)
{
    auto *amap =
        static_cast<SFAttachmentMap *>(container->getSFAttachments());
    auto mapIt = amap->getValue().begin();
    auto mapEnd = amap->getValue().end();

    beginEditCP(container, AttachmentContainer::AttachmentsFieldMask);
    for(; mapIt != mapEnd; ++mapIt)
    {
        AttachmentPtr att = mapIt->second;
        if(att == NullFC)
            continue;

        if(shouldShareContainer(att, sharedTypes))
            continue; // keep as is, this attachment type should not be unshared

        AttachmentPtr newAtt = AttachmentPtr::dcast(getCloneR2(att, sharedTypes, cloned));

        // Replace att by newAtt
        replaceAttachment(container, mapIt, newAtt);
    }
    endEditCP(container, AttachmentContainer::AttachmentsFieldMask);
}


/*! \brief Recreates the core and attachments of a node.
*          Note: Node cores are currently always cloned even when the core type is in "share", same as with deepCloneTree.
*/
void osg::unshareNode(const NodePtr& node, const std::vector<std::string>& share)
{
    if(node == NullFC)
        return;

    auto shareTypes = convertShareTypes(share);
    ClonedFieldContainers cloned;
    unshareAttachments(cloned, node, shareTypes);

    NodeCorePtr srcCore = node->getCore();
    if(srcCore != NullFC)
    {
        NodeCorePtr clonedCore = NodeCorePtr::dcast(deepCloneR2(cloned, srcCore, shareTypes));
        beginEditCP(node, Node::CoreFieldMask);
            node->setCore(clonedCore);
        endEditCP(node, Node::CoreFieldMask);
    }
}

/*! \brief Unshares node cores and attachments of the given nodes, not their children.
           Keeps sharing between the nodes of the passed list.
*/
void osg::unshareNodesKeepLocalShared(std::vector<osg::NodePtr>& nodes, const std::vector<std::string>& share)
{
    // map of already cloned cores (source core id -> cloned core) to reuse instead of cloning new ones
    // This is to keep clones as local clones within the list.
    std::map<UInt32, NodeCorePtr> clonedCores; 

    // map of already cloned containers for reuse
    // This is to keep shared attachments (e.g.) as local shares within the list.
    ClonedFieldContainers cloned;

    const auto shareTypes = convertShareTypes(share);
    for(const auto& node : nodes)
    {
        if(node == NullFC)
            continue;

        const NodeCorePtr srcCore = node->getCore();
        if(srcCore == NullFC)
            continue;

        unshareAttachments(cloned, node, shareTypes); // reuses already cloned attachments

        beginEditCP(node, Node::CoreFieldMask);
        const auto cit = clonedCores.find(srcCore.getFieldContainerId());
        if(cit != clonedCores.end())
        {
            // reuse an already cloned core
            node->setCore(cit->second);
        }
        else
        {
            osg::NodeCorePtr clonedCore = NodeCorePtr::dcast(deepCloneR2(cloned, srcCore, shareTypes));
            clonedCores.insert(std::make_pair(srcCore.getFieldContainerId(), clonedCore));
            node->setCore(clonedCore);
        }
        endEditCP(node, Node::CoreFieldMask);
    }
}





/*-------------------------------------------------------------------------*/
/*                              cvs id's                                   */

#ifdef __sgi
#pragma set woff 1174
#endif

#ifdef OSG_LINUX_ICC
#pragma warning( disable : 177 )
#endif

namespace
{
    static Char8 cvsid_cpp[] = "@(#)$Id: $";
    static Char8 cvsid_hpp[] = OSGNODE_HEADER_CVSID;
    static Char8 cvsid_inl[] = OSGNODE_INLINE_CVSID;
}
