/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *                     Copyright 2000-2002 by OpenSG Forum                   *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

#ifdef OSG_DOC_FILES_IN_MODULE
/*! \file OSGNodeCore.cpp
    \ingroup GrpSystemFieldContainer
 */
#endif

#include <stdlib.h>
#include <stdio.h>

#include "OSGConfig.h"

#include "OSGNode.h"
#include "OSGNodeCore.h"
#include "OSGBinaryDataHandler.h"
#include "OSGNodeCoreFlags.h"
#include "OSGGeometry.h"
#include "OSGName.h"
#include "OSGNodeFunctions.h"

OSG_USING_NAMESPACE

const BitVector
    NodeCore::ParentsFieldMask     = (1 << NodeCore::ParentsFieldId    );

constexpr UInt32 SkipEvalFlags = Node::NFAuxiliary | Node::NFInternal | Node::NFTrash;

FieldDescription *NodeCore::_desc[] =
{
    new FieldDescription(MFNodePtr::getClassType(),
                         "parents",
                         OSG_FC_FIELD_IDM_DESC(ParentsField),
                         FieldDescription::FDInternal,
                         reinterpret_cast<FieldAccessMethod>(
                             &NodeCore::getMFParents))
};

FieldContainerType NodeCore::_type("NodeCore",
                                   "AttachmentContainer",
                                   NULL,
                                   NULL,
                                   NULL,
                                   _desc,
                                   sizeof(_desc));


OSG_ABSTR_FIELD_CONTAINER_DEF(NodeCore, NodeCorePtr)

#if defined(OSG_FIXED_MFIELDSYNC)
void NodeCore::onDestroyAspect(UInt32 uiId, UInt32 uiAspect)
{
    _parents.terminateShare(uiAspect, this->getContainerSize());
}
#endif



static bool inGraphWithLockSupport(const osg::NodePtr& node)
{
    return isGraphNodeWithFlag(node, Node::NFSceneRoot | Node::NFModuleGraphRoot);
}

bool NodeCore::getSelfLocked() 
{
    // Relaxed ordering is sufficient as the final value will be correct 
    // even if two threads initialize the flag at the same time.

    UInt8 f = m_selfLockFlag.load(std::memory_order_relaxed);
    if(f == CF_CachedFlagDirty)
    {
        // Evaluate the state
        bool b = evalSelfLocked();

        // Update the flag
        f = (b ? (CF_CachedFlagTrue) : CF_CachedFlagFalse);
        m_selfLockFlag.store(f, std::memory_order_relaxed);
    }
    return (f == CF_CachedFlagTrue);
}

bool NodeCore::evalSelfLocked()
{
    bool hasLock = false;
    for(auto& instance : _parents)
    {
        // only check first valid parent, assuming clones with lock attachment all have the attachment
        if(instance && !instance->hasFlag(SkipEvalFlags))
        {
            hasLock = queryLock(instance);
            break;
        }
    }
    return hasLock;
}

bool NodeCore::getSelfUneditedAsset()
{
    if(getType().isDerivedFrom(osg::Geometry::getClassType()))
        return false; // geometry nodes cannot be assets, this early-out is for performance as it excludes many unnecessary checks

    // Relaxed ordering is sufficient as the final value will be correct 
    // even if two threads initialize the flag at the same time.

    UInt8 f = m_selfAssetFlag.load(std::memory_order_relaxed);
    if(f == CF_CachedFlagDirty)
    {
        // Evaluate the state
        bool b = evalSelfUneditedAsset();

        // Update the flag
        f = b ? CF_CachedFlagTrue : CF_CachedFlagFalse;
        m_selfAssetFlag.store(f, std::memory_order_relaxed);
    }
    return (f == CF_CachedFlagTrue);
}

bool NodeCore::evalSelfUneditedAsset()
{
    bool isUneditedAsset = false;
    for(auto& instance : _parents)
    {
        // only check first valid parent, assuming clones with asset attachment all have the asset attachment
        if(instance && !instance->hasFlag(SkipEvalFlags))
        {
            isUneditedAsset = queryIsUneditedAssetRoot(instance);
            break;
        }
    }
    return isUneditedAsset;
}


// We can do this without a mutex lock, because:
// the global timestamp will not change while this is evaluated, and
// if two threads evaluate this at the same time, the result will still be the same.
bool NodeCore::getIsLockedByAncestor()
{
    // Directly access the atomic member without memory ordering constraints
    UInt64 t = m_timeStamp.load(std::memory_order_relaxed);
    auto currT = g_ancestorLockTimeStamp.load(std::memory_order_relaxed);

    if(t != currT) // Timestamp is outdated
    {
        // Evaluate the state
        bool isLockedByAncestor = evalIsLockedByAncestor();

        // Update the flag
        m_lockedByAncestorFlag.store(isLockedByAncestor, std::memory_order_relaxed);

        // Update the timestamp
        m_timeStamp.store(currT, std::memory_order_relaxed);
    }

    return m_lockedByAncestorFlag.load(std::memory_order_relaxed);
}


bool NodeCore::evalIsLockedByAncestor()
{
    // check if an ancestor node of any instance is locking this node

    bool hasAncestorLock = false;
    for(auto& instance : _parents)
    {
        if(instance && !instance->hasFlag(SkipEvalFlags))
        {
             // the actual parent node
            auto parent = instance->getParent();
            if(parent && !parent->hasFlag(SkipEvalFlags))
            {
                // Ignore nodes outside scenegraph or module graphs, e.g. in copy buffer
                if(((parent->getSelfLocked() || parent->getSelfUneditedAsset()) && inGraphWithLockSupport(parent)) || parent->getCoreIsLockedByAncestor())
                {
                    hasAncestorLock = true;
                    break;
                }
            }
        }
    }
    return hasAncestorLock;
}


/*-------------------------------------------------------------------------*/
/*                                Dump                                     */

void NodeCore::dump(      UInt32    uiIndent,
                    const BitVector bvFlags) const
{
    UInt32 i;

    NodeCorePtr thisP = getPtr();

//    thisP.dump(0, FCDumpFlags::RefCount);

    indentLog(uiIndent, PLOG);

    PLOG << "Core"
         << "("
         << std::dec
         << getContainerId(thisP)
         << ") : " << getType().getName()
         << " "
         << _attachmentMap.getValue().size()
         << " attachments | "
         << this
         << std::endl;

    indentLog(uiIndent, PLOG);
    PLOG << "[" << std::endl;

    indentLog(uiIndent + 4, PLOG);
    PLOG << "Parents : " << std::endl;

    for(i = 0; i < _parents.size(); i++)
    {
        indentLog(uiIndent + 4, PLOG);
        PLOG << "           " << i << ") " << &(*(_parents[i])) << std::endl;
    }

    indentLog(uiIndent, PLOG);
    PLOG << "]" << std::endl;

    indentLog(uiIndent, PLOG);
    PLOG << "{" << std::endl;

    Inherited::dump(uiIndent, bvFlags);

    indentLog(uiIndent, PLOG);
    PLOG << "}" << std::endl;
}

/*-------------------------------------------------------------------------*/
/*                              cvs id's                                   */

#ifdef __sgi
#pragma set woff 1174
#endif

#ifdef OSG_LINUX_ICC
#pragma warning( disable : 177 )
#endif

namespace
{
    static Char8 cvsid_cpp[] = "@(#)$Id: $";
    static Char8 cvsid_hpp[] = OSGNODECORE_HEADER_CVSID;
    static Char8 cvsid_inl[] = OSGNODECORE_INLINE_CVSID;
}
