#pragma once

#include <OSGBaseTypes.h>
#include <OSGFieldContainerPtr.h>
#include <OSGRefPtr.h>
#include <OSGSystemDef.h>

OSG_BEGIN_NAMESPACE

/* Weak Pointer
 * The following operations are using the factory to check, if the pointer is valid
 * - bool operations -> if(ptr)
 * - comparision with NullFC -> if (ptr == NullFC)
 * - dynamic cast operations -> osg::dcast<Type>(ptr) or TypePtr::dcast(ptr)
 * - get -> x = ptr.get()
 * - implicit cast operations to FieldContainerPtr
 * The following operations are perfomed unchecked
 * - Pointer access ->, *, &
 */
template <class PtrType>
class WeakPtr
{
public:
    using StoredObjectType = typename PtrType::StoredObjectType;
    using FCPtrType = PtrType;

    WeakPtr() = default;

    WeakPtr(const FCPtrType &fcp)
        : _ptr(fcp)
        , _containerId(fcp.getFieldContainerId())
    {
    }
    WeakPtr(const WeakPtr<PtrType> &ptr)
        : _ptr(ptr._ptr)
        , _containerId(ptr._containerId)
    {
    }
    operator FCPtrType() const
    {
        return getFactoryContainer();
    }
    typename PtrType::StoredObjectType &operator*() const
    {
        return *(_ptr.operator->());
    }
    typename PtrType::StoredObjectType *operator->() const
    {
        return _ptr.operator->();
    }
    typename PtrType::StoredObjectType *getCPtr() const
    {
        return _ptr.getCPtr();
    }
    PtrType get() const
    {
        return getFactoryContainer();
    }

    WeakPtr<PtrType> &operator=(const FCPtrType &fcp)
    {
        setPtr(fcp);
        return *this;
    }
    WeakPtr<PtrType> &operator=(const WeakPtr<PtrType> &wcp)
    {
        _ptr = wcp._ptr;
        _containerId = wcp._containerId;
        return *this;
    }
    WeakPtr<PtrType> &operator=(const NullFieldContainerPtr &)
    {
        setPtr(NullFC);
        return *this;
    }

    bool operator<(const NullFieldContainerPtr &) const
    {
        return false;
    }
    bool operator==(const NullFieldContainerPtr &other) const
    {
        return getFactoryContainer() == NullFC;
    }
    bool operator!=(const NullFieldContainerPtr &other) const
    {
        return getFactoryContainer() != NullFC;
    }

    bool operator<(const FieldContainerPtr &other) const
    {
        return _ptr < other;
    }
    bool operator==(const FieldContainerPtr &other) const
    {
        return _ptr == other;
    }
    bool operator!=(const FieldContainerPtr &other) const
    {
        return !(_ptr == other);
    }

    bool operator<(const WeakPtr<PtrType> &other) const
    {
        return _ptr < other._ptr;
    }
    bool operator==(const WeakPtr<PtrType> &other) const
    {
        return _ptr == other._ptr;
    }
    bool operator!=(const WeakPtr<PtrType> &other) const
    {
        return !(_ptr == other._ptr);
    }
    explicit operator bool() const
    {
        return getFactoryContainer() != NullFC;
    }

    // used to allow Ptr::dcast
    UInt32 getContainerSize() const
    {
        return _ptr.getContainerSize();
    }
    // used to allow Ptr::dcast
    UInt16 getParentFieldPos() const
    {
        return _ptr.getParentFieldPos();
    }

private:
    void setPtr(const PtrType &ptr)
    {
        _ptr = ptr;
        _containerId = ptr.getFieldContainerId();
    }
    FCPtrType getFactoryContainer() const
    {
        return FCPtrType::dcast(FieldContainerFactory::the()->getContainer(_containerId));
    }

    PtrType _ptr{NullFC};
    UInt32 _containerId{0};
};

using FieldContainerWeakPtr = WeakPtr<FieldContainerPtr>;

OSG_END_NAMESPACE
