/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *                     Copyright 2000-2002 by OpenSG Forum                   *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

#include <algorithm>
#include <sstream>

#include "OSGOSGWriter.h"
#include <OSGAttachment.h>
#include <OSGName.h>
#include "OSGVRMLNodeDescs.h"

OSG_USING_NAMESPACE

#if 0
#if defined(OSG_WIN32_ICL)
#pragma warning (disable : 383)
#endif

#if defined(OSG_WIN32_ICL)
#pragma warning (default : 383)
#endif
#endif

const UInt32 OSGWriter::DefaultSFWidth = TypeTraits<UInt32>::getMax();
const UInt32 OSGWriter::DefaultMFWidth = 60;

/*-------------------------------------------------------------------------*/
/*                             Constructor                                 */

/*! Constructor. Set members to initial values.
 */
OSGWriter::FCInfoHelper::FCInfoHelper(void) :
    written      (false),
    hasName      (false),
    containerName(     )
{
}

/*! Constructor. Set members to initial values.
 */
OSGWriter::OSGWriter(std::ostream &stream, UInt32 indentStep, const std::string &options) :
    _visitedFCMap(                 ),
    _state       (0, DefaultSFWidth),
    _indent      (0, indentStep    ),
    _outStream   (stream           ),
    _exportAttachments(false),
    _exportCameras(false),
    _exportImages(false),
    _enableRounding(false)
{
    if(options.find("exportAttachments=true") != std::string::npos)
        _exportAttachments = true;
    if(options.find("exportAttachments=false") != std::string::npos)
        _exportAttachments = false;

    if(options.find("exportCameras=true") != std::string::npos)
        _exportCameras = true;
    if(options.find("exportCameras=false") != std::string::npos)
        _exportCameras = false;

    if(options.find("exportImages=true") != std::string::npos)
        _exportImages = true;
    if(options.find("exportImages=false") != std::string::npos)
        _exportImages = false;

    if(options.find("enableRounding=true") != std::string::npos)
        _enableRounding = true;
    if(options.find("enableRounding=false") != std::string::npos)
        _enableRounding = false;
}

/*-------------------------------------------------------------------------*/
/*                             Destructor                                  */

/*! Destructor. There are no dynamic members to destroy.
 */
OSGWriter::~OSGWriter(void)
{
}

/*-------------------------------------------------------------------------*/
/*                             Methods                                     */

/*! Write a single FieldContainer with all its "children", i.e. everything
 *  that can be reached via Ptr-Fields.
 */
void OSGWriter::write(FieldContainerPtr container)
{
    Real64 t = osg::getSystemTime();
    _visitedFCMap.clear();
    _indent.setIndent(0);

    _outStream << "#OSG V1.0 " << std::endl;

    visitContainer(container);
    writeContainer(container);
    printf("!!!! OSG write time %f\n", osg::getSystemTime() - t);
}


/*! Write all FieldContainers in containers with their "children",
 *  i.e. everything that can be reached via Ptr-Fields.
 */
void OSGWriter::write(std::vector<FieldContainerPtr> containers)
{
    _visitedFCMap.clear();
    _indent.setIndent(0);

    _outStream << "#OSG V1.0 " << std::endl;

    std::vector<FieldContainerPtr>::reverse_iterator iter;

    for(iter = containers.rbegin(); iter != containers.rend(); ++iter)
    {
        visitContainer( *iter );
    }

    for(iter = containers.rbegin(); iter != containers.rend(); ++iter)
    {
        writeContainer( *iter );
    }
}


/*! Set the name by which this FieldContainer is referenced. If available
 *  a NameAttachment is used, otherwise the name is constructed from
 *  the type name and the container id.
 */
void OSGWriter::FCInfoHelper::setName(const FieldContainerPtr pFC)
{
    const FieldContainerType& fcType = pFC->getType();
    AttachmentContainerPtr pAttCon;
    NamePtr                pNameAtt;

    if(fcType.isDerivedFrom(AttachmentContainer::getClassType()))
    {
        pAttCon = AttachmentContainerPtr::dcast(pFC);
        if(pAttCon != NullFC)
        {
            pNameAtt = NamePtr::dcast(pAttCon->findAttachment(
                Name::getClassType().getGroupId()));

            if(pNameAtt != NullFC)
            {
                containerName = pNameAtt->getName().c_str();
                return;
            }
        }
    }

    //no NameAttachment. Build name from Type and Id
    containerName = pFC->getTypeName() +
        TypeTraits<UInt32>::putToString(getContainerId(pFC));
}


void OSGWriter::visitContainer(const FieldContainerPtr pFC)
{

    if(pFC == NullFC)
    {
        return;
    }

    typedef std::pair<FCInfoHelperMap::iterator, bool> MapInsertInfo;

    std::string containerName;
    const FieldContainerType& fcType    = pFC->getType();
    UInt32              numFields = fcType.getNumFieldDescs();
    MapInsertInfo       insertInfo;

    // skip material pool!
    if(strcmp(fcType.getCName(), "MaterialPool") == 0)
    {
        return;
    }

    if(strcmp(fcType.getCName(), "UberCamera") == 0 && !_exportCameras)
    {
        return;
    }

    if(strcmp(fcType.getCName(), "Image") == 0 && !_exportImages)
    {
        return;
    }

    insertInfo = _visitedFCMap.insert(std::make_pair(pFC.getFieldContainerId(), FCInfoHelper()));

    if(insertInfo.second == true)
    {
        //the FC was NOT visited before
        for(UInt32 field=1; field<=numFields; field++)
        {
            const FieldDescription* fieldDesc =
                fcType.getFieldDescription(field);
            if(fieldDesc->isInternal())
            {
                continue;
            }
            visitField(pFC->getField(field));
        }
    }
    else
    {
        //the FC was in the map => FC is shared
        FCInfoHelperMap::iterator iter = _visitedFCMap.find(pFC.getFieldContainerId());
        if(iter == _visitedFCMap.end())
        {
            SWARNING << "OSGWriter::visitContainer(): FieldContainerPtr "
                     << "not found in map" << std::endl;
            return;
        }
        if(iter->second.hasName == false)
        {
            iter->second.setName(pFC);
            iter->second.hasName = true;
        }
    }
}

void OSGWriter::visitField(const Field* pF)
{
    if(pF == NULL)
    {
        return;
    }

    const FieldType& fType       = pF->getType();
//    const DataType & contentType = pF->getContentType();

    //handle SFAttachmentMap as special case here
    //if(fType.isDerivedFrom(SFAttachmentMap::getClassType()))
    if(strstr(fType.getCName(), "AttachmentMap") != NULL && _exportAttachments)
    {
        //visit the Attachment FCs

        const SFAttachmentMap *sfAttMap = static_cast<const SFAttachmentMap*>(pF);
              AttachmentMap    attMap   = sfAttMap->getValue();

        AttachmentMap::const_iterator iter = attMap.begin();
        AttachmentMap::const_iterator end  = attMap.end();

        for(; iter!=end; ++iter)
        {
            visitContainer(iter->second);
        }
    }
    //else if(contentType.isDerivedFrom(FieldContainerPtr::getClassType()))
    else if(strstr(fType.getCName(), "Ptr") != NULL)
    {
        //this Field points to FC

        //to access the content of a field one must know the cardinality
        if(pF->getCardinality() == FieldType::SINGLE_FIELD)
        {
            const SFFieldContainerPtr* sfFCPtr =
                static_cast<const SFFieldContainerPtr*>(pF);
            visitContainer(sfFCPtr->getValue());
        }
        else if(pF->getCardinality() == FieldType::MULTI_FIELD)
        {
            const MFFieldContainerPtr* mfFCPtr =
                static_cast<const MFFieldContainerPtr*>(pF);
            UInt32 mfSize = mfFCPtr->size();
            for(UInt32 i=0; i < mfSize; i++)
            {
                visitContainer((*(mfFCPtr))[i]);
            }
        }
    }
}

void OSGWriter::writeContainer(const FieldContainerPtr pFC)
{
    if(pFC == NullFC)
    {
        return;
    }

    const FieldContainerType& fcType    = pFC->getType();
    UInt32              numFields = fcType.getNumFieldDescs();

    FCInfoHelperMap::iterator iter = _visitedFCMap.find(pFC.getFieldContainerId());
    if(iter == _visitedFCMap.end())
    {
        //SWARNING << "OSGWriter::writeContainer(): FieldContainerPtr "
        //         << "not found in map" << std::endl;
        return;
    }

    if(!iter->second.written)
    {
        //FC is not written yet
        iter->second.written = true;
        if(iter->second.hasName)
        {
            _outStream << _indent                    << "DEF "
                       << iter->second.containerName << " "
                       << pFC->getTypeName()         << " {"
                       << std::endl;
        }
        else{
            _outStream << _indent <<  pFC->getTypeName() << " {"
                       << std::endl;
        }

        _indent++;

        for(UInt32 field=1; field<=numFields; field++)
        {
            const FieldDescription* fieldDesc =
                fcType.getFieldDescription(field);
            if(fieldDesc->isInternal())
            {
                continue;
            }
            writeField(pFC->getField(field), fieldDesc);
        }
        _indent--;
        _outStream << _indent << "}" << std::endl;
    }
    else
    {
        //FC is already written -> its shared -> write reference
        if(!iter->second.hasName)
        {
            SWARNING << "OSGWriter::writeContainer(): FieldContainer is "
                     << "shared, but not named"
                     << std::endl;
            return;
        }

        _outStream << _indent
                   << "USE "
                   << iter->second.containerName 
                   << std::endl;
    }

}

template <typename T>
bool OSGWriter::writeRoundedValue(const Field *field, const std::string &prefix, bool componentType)
{
    const T *mf = dynamic_cast<const T *>(field);

    if(mf == nullptr)
        return false;

    if(mf->empty())
        return true;

    std::string valStr;
    std::string outStr;
    _state.beginField(mf, outStr);
    for(SizeT i=0;i<mf->size();++i)
    {
        valStr.erase();
        std::stringstream ss;
        ss << std::fixed << std::setprecision(2) << mf->getValue(i);
        valStr = ss.str();
        if(componentType)
        {
            // for vector types replace all ',' with ' '
            std::replace(valStr.begin(), valStr.end(), ',', ' ');
        }
        _state.addValueStr(valStr, outStr);
    }
    _state.endField(mf, outStr);

    _outStream << prefix << outStr << std::endl;

    return true;
}

void OSGWriter::writeValue(const Field *field, const std::string &prefix)
{
    if(_enableRounding)
    {
        if(writeRoundedValue<MFReal32>(field, prefix))
            return;

        if(writeRoundedValue<MFPnt2f>(field, prefix, true))
            return;
        if(writeRoundedValue<MFPnt3f>(field, prefix, true))
            return;
        if(writeRoundedValue<MFPnt4f>(field, prefix, true))
            return;

        if(writeRoundedValue<MFVec2f>(field, prefix, true))
            return;
        if(writeRoundedValue<MFVec3f>(field, prefix, true))
            return;
        if(writeRoundedValue<MFVec4f>(field, prefix, true))
            return;
    }

    // string fallback
    std::string fieldValue;
    field->getValueByStr(fieldValue, _state);
    _outStream << prefix << fieldValue << std::endl;
}

void OSGWriter::writeField(const Field* pF, const FieldDescription* fieldDesc)
{

    if(pF == NULL)
    {
        return;
    }

    const FieldType& fType = pF->getType();
//    const DataType&  contentType = pF->getContentType();

    //handle SFAttachmentMap as special case here
    //if(fType.isDerivedFrom(SFAttachmentMap::getClassType()))
    if(strstr(fType.getCName(), "AttachmentMap") != NULL && _exportAttachments)
    {
        //write Attachments

        const SFAttachmentMap *sfAttMap = static_cast<const SFAttachmentMap*>(pF);
              AttachmentMap    attMap   = sfAttMap->getValue();

        AttachmentMap::const_iterator iter = attMap.begin();
        AttachmentMap::const_iterator end  = attMap.end();

        _outStream << _indent << fieldDesc->getName() << " [ ";
        _indent++;
        _state.setIndent(_indent.getIndent());
        
        //if the Attachment Map is empty write [] as its content
        if(iter==end)
        {
            _outStream << " ] " << std::endl;
            _indent--; 
        }
        else
        {
            _outStream << std::endl;
        
            for(; iter!=end; ++iter)
            {
                if(iter->second->getInternal().getValue() != true)
                {
                    writeContainer(iter->second);
                }
            }
            _indent--; 
            
            _outStream << _indent << " ] " << std::endl;
        }
    }
    //else if(contentType.isDerivedFrom(FieldContainerPtr::getClassType()))
    else if(strstr(fType.getCName(), "Ptr") != NULL)
    {
        //this Field points to FC

        _state.setIndent(_indent.getIndent());
        _outStream << _indent << fieldDesc->getName();

        //to access the content of a field via a Field*
        //one must know the cardinality
        if(pF->getCardinality() == FieldType::SINGLE_FIELD)
        {
            const SFFieldContainerPtr* sfFCPtr =
                static_cast<const SFFieldContainerPtr*>(pF);
            if(sfFCPtr->getValue() == NullFC)
            {
                _outStream << " NULL" << std::endl;
            }
            else
            {
                _outStream << std::endl;
                _indent++;
                writeContainer(sfFCPtr->getValue());
                _indent--;
            }
        }
        else if(pF->getCardinality() == FieldType::MULTI_FIELD)
        {
            _outStream << " [" << std::endl;
            _indent++;
            const MFFieldContainerPtr* mfFCPtr =
                static_cast<const MFFieldContainerPtr*>(pF);
            UInt32 mfSize = mfFCPtr->size();
            for(UInt32 i=0; i < mfSize; i++)
            {
                if((*(mfFCPtr))[i] == NullFC)
                {
                    _outStream << _indent << "NULL" << std::endl;
                }
                else
                {
                    writeContainer((*(mfFCPtr))[i]);
                }
            }
            _indent--;
            _outStream << _indent << "]" << std::endl;
        }
    }
    else
    {
        //this Field contains data -> write it out

        _state.setIndent(_indent.getIndent());
        _outStream << _indent << fieldDesc->getName();

        //std::string fieldValue;

        //to access the content of a field via a Field*
        //one must know the cardinality
        if(pF->getCardinality() == FieldType::SINGLE_FIELD)
        {
            _state.setIndent(0);
            _state.setWidth(DefaultSFWidth);
            writeValue(pF, " ");
            //pF->getValueByStr(fieldValue, _state);
            //_outStream << " " << fieldValue << std::endl;
        }
        else if(pF->getCardinality() == FieldType::MULTI_FIELD)
        {
            _outStream << " [" << std::endl;

            _indent++;
            _state.setIndent(_indent.getIndent());
            _state.setWidth(DefaultMFWidth);
            writeValue(pF);
            //pF->getValueByStr(fieldValue, _state);
            //_outStream << fieldValue << std::endl;
            _indent--;

            _outStream << _indent << "]" << std::endl;
        }
    }
}


/*-------------------------------------------------------------------------*/
/*                              cvs id's                                   */

#ifdef __sgi
#pragma set woff 1174
#endif

#ifdef OSG_LINUX_ICC
#pragma warning( disable : 177 )
#endif

namespace
{
    static Char8 cvsid_cpp[] = "@(#)$Id: $";
    static Char8 cvsid_hpp[] = OSGOSGWRITER_HEADER_CVSID;
}
