/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright (C) 2000-2002 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/
//-------------------------------
//  Includes
//-------------------------------

#include <string>
#include <iostream>
#include <array>
#include <chrono>

#include <OSGConfig.h>
#include <OSGGL.h>

#include <OSGLog.h>
#include <OSGImage.h>

#include "OSGImageCompressor.h"
#include "OSGDDSImageFileType.h"

OSG_USING_NAMESPACE


/*! \class osg::DDSImageFileType
    \ingroup GrpSystemImage

Image File Type to read/write and store/restore Image objects as
DDS data. Should work with binary and ascii dds/pbm/pgm/ppm data.

All the type specific code is included in the class. Does
not depend on external libs.

*/

class DDSReader
{
public:

    // header dwFlags
    using DDSD_FLAGS = enum DDSD_FLAGS
    {
        DDSD_CAPS			= 0x1,
        DDSD_HEIGHT			= 0x2,
        DDSD_WIDTH			= 0x4,
        DDSD_PITCH			= 0x8,
        DDSD_PIXELFORMAT	= 0x1000,
        DDSD_MIPMAPCOUNT	= 0x20000,
        DDSD_LINEARSIZE		= 0x80000,
        DDSD_DEPTH			= 0x800000
    };

    // header dwCaps
    using DDSC_FLAGS = enum DDSC_FLAGS
    {
        DDSCAPS_COMPLEX	= 0x8,
        DDSCAPS_TEXTURE	= 0x1000,
        DDSCAPS_MIPMAP	= 0x400000
    };

    // header dwCaps2
    using DDSC_FLAGS2 = enum DDSC_FLAGS2
    {
        DDSCAPS2_CUBEMAP			= 0x200,
        DDSCAPS2_CUBEMAP_POSITIVEX	= 0x400,
        DDSCAPS2_CUBEMAP_NEGATIVEX	= 0x800,
        DDSCAPS2_CUBEMAP_POSITIVEY	= 0x1000,
        DDSCAPS2_CUBEMAP_NEGATIVEY	= 0x2000,
        DDSCAPS2_CUBEMAP_POSITIVEZ	= 0x4000,
        DDSCAPS2_CUBEMAP_NEGATIVEZ	= 0x8000,
        DDSCAPS2_VOLUME				= 0x200000
    };

    // pixel format dwFlags
    using DDPF_FLAGS = enum DDPF_FLAGS
    {
        DDPF_ALPHAPIXELS 	= 0x1,
        DDPF_ALPHA 			= 0x2,
        DDPF_FOURCC 		= 0x4,
        DDPF_RGB 			= 0x40,
        DDPF_YUV 			= 0x200,
        DDPF_LUMINANCE 		= 0x20000
    };

    static inline const std::uint32_t FOURCC_DXT1 = 0x31545844;
    static inline const std::uint32_t FOURCC_DXT3 = 0x33545844;
    static inline const std::uint32_t FOURCC_DXT5 = 0x35545844;

    // DX10 dxgiFormat value
    using DXGI_FORMAT = enum DXGI_FORMAT
    { 
        DXGI_FORMAT_UNKNOWN                     = 0,
        DXGI_FORMAT_R32G32B32A32_TYPELESS       = 1,
        DXGI_FORMAT_R32G32B32A32_FLOAT          = 2,
        DXGI_FORMAT_R32G32B32A32_UINT           = 3,
        DXGI_FORMAT_R32G32B32A32_SINT           = 4,
        DXGI_FORMAT_R32G32B32_TYPELESS          = 5,
        DXGI_FORMAT_R32G32B32_FLOAT             = 6,
        DXGI_FORMAT_R32G32B32_UINT              = 7,
        DXGI_FORMAT_R32G32B32_SINT              = 8,
        DXGI_FORMAT_R16G16B16A16_TYPELESS       = 9,
        DXGI_FORMAT_R16G16B16A16_FLOAT          = 10,
        DXGI_FORMAT_R16G16B16A16_UNORM          = 11,
        DXGI_FORMAT_R16G16B16A16_UINT           = 12,
        DXGI_FORMAT_R16G16B16A16_SNORM          = 13,
        DXGI_FORMAT_R16G16B16A16_SINT           = 14,
        DXGI_FORMAT_R32G32_TYPELESS             = 15,
        DXGI_FORMAT_R32G32_FLOAT                = 16,
        DXGI_FORMAT_R32G32_UINT                 = 17,
        DXGI_FORMAT_R32G32_SINT                 = 18,
        DXGI_FORMAT_R32G8X24_TYPELESS           = 19,
        DXGI_FORMAT_D32_FLOAT_S8X24_UINT        = 20,
        DXGI_FORMAT_R32_FLOAT_X8X24_TYPELESS    = 21,
        DXGI_FORMAT_X32_TYPELESS_G8X24_UINT     = 22,
        DXGI_FORMAT_R10G10B10A2_TYPELESS        = 23,
        DXGI_FORMAT_R10G10B10A2_UNORM           = 24,
        DXGI_FORMAT_R10G10B10A2_UINT            = 25,
        DXGI_FORMAT_R11G11B10_FLOAT             = 26,
        DXGI_FORMAT_R8G8B8A8_TYPELESS           = 27,
        DXGI_FORMAT_R8G8B8A8_UNORM              = 28,
        DXGI_FORMAT_R8G8B8A8_UNORM_SRGB         = 29,
        DXGI_FORMAT_R8G8B8A8_UINT               = 30,
        DXGI_FORMAT_R8G8B8A8_SNORM              = 31,
        DXGI_FORMAT_R8G8B8A8_SINT               = 32,
        DXGI_FORMAT_R16G16_TYPELESS             = 33,
        DXGI_FORMAT_R16G16_FLOAT                = 34,
        DXGI_FORMAT_R16G16_UNORM                = 35,
        DXGI_FORMAT_R16G16_UINT                 = 36,
        DXGI_FORMAT_R16G16_SNORM                = 37,
        DXGI_FORMAT_R16G16_SINT                 = 38,
        DXGI_FORMAT_R32_TYPELESS                = 39,
        DXGI_FORMAT_D32_FLOAT                   = 40,
        DXGI_FORMAT_R32_FLOAT                   = 41,
        DXGI_FORMAT_R32_UINT                    = 42,
        DXGI_FORMAT_R32_SINT                    = 43,
        DXGI_FORMAT_R24G8_TYPELESS              = 44,
        DXGI_FORMAT_D24_UNORM_S8_UINT           = 45,
        DXGI_FORMAT_R24_UNORM_X8_TYPELESS       = 46,
        DXGI_FORMAT_X24_TYPELESS_G8_UINT        = 47,
        DXGI_FORMAT_R8G8_TYPELESS               = 48,
        DXGI_FORMAT_R8G8_UNORM                  = 49,
        DXGI_FORMAT_R8G8_UINT                   = 50,
        DXGI_FORMAT_R8G8_SNORM                  = 51,
        DXGI_FORMAT_R8G8_SINT                   = 52,
        DXGI_FORMAT_R16_TYPELESS                = 53,
        DXGI_FORMAT_R16_FLOAT                   = 54,
        DXGI_FORMAT_D16_UNORM                   = 55,
        DXGI_FORMAT_R16_UNORM                   = 56,
        DXGI_FORMAT_R16_UINT                    = 57,
        DXGI_FORMAT_R16_SNORM                   = 58,
        DXGI_FORMAT_R16_SINT                    = 59,
        DXGI_FORMAT_R8_TYPELESS                 = 60,
        DXGI_FORMAT_R8_UNORM                    = 61,
        DXGI_FORMAT_R8_UINT                     = 62,
        DXGI_FORMAT_R8_SNORM                    = 63,
        DXGI_FORMAT_R8_SINT                     = 64,
        DXGI_FORMAT_A8_UNORM                    = 65,
        DXGI_FORMAT_R1_UNORM                    = 66,
        DXGI_FORMAT_R9G9B9E5_SHAREDEXP          = 67,
        DXGI_FORMAT_R8G8_B8G8_UNORM             = 68,
        DXGI_FORMAT_G8R8_G8B8_UNORM             = 69,
        DXGI_FORMAT_BC1_TYPELESS                = 70,
        DXGI_FORMAT_BC1_UNORM                   = 71,
        DXGI_FORMAT_BC1_UNORM_SRGB              = 72,
        DXGI_FORMAT_BC2_TYPELESS                = 73,
        DXGI_FORMAT_BC2_UNORM                   = 74,
        DXGI_FORMAT_BC2_UNORM_SRGB              = 75,
        DXGI_FORMAT_BC3_TYPELESS                = 76,
        DXGI_FORMAT_BC3_UNORM                   = 77,
        DXGI_FORMAT_BC3_UNORM_SRGB              = 78,
        DXGI_FORMAT_BC4_TYPELESS                = 79,
        DXGI_FORMAT_BC4_UNORM                   = 80,
        DXGI_FORMAT_BC4_SNORM                   = 81,
        DXGI_FORMAT_BC5_TYPELESS                = 82,
        DXGI_FORMAT_BC5_UNORM                   = 83,
        DXGI_FORMAT_BC5_SNORM                   = 84,
        DXGI_FORMAT_B5G6R5_UNORM                = 85,
        DXGI_FORMAT_B5G5R5A1_UNORM              = 86,
        DXGI_FORMAT_B8G8R8A8_UNORM              = 87,
        DXGI_FORMAT_B8G8R8X8_UNORM              = 88,
        DXGI_FORMAT_R10G10B10_XR_BIAS_A2_UNORM  = 89,
        DXGI_FORMAT_B8G8R8A8_TYPELESS           = 90,
        DXGI_FORMAT_B8G8R8A8_UNORM_SRGB         = 91,
        DXGI_FORMAT_B8G8R8X8_TYPELESS           = 92,
        DXGI_FORMAT_B8G8R8X8_UNORM_SRGB         = 93,
        DXGI_FORMAT_BC6H_TYPELESS               = 94,
        DXGI_FORMAT_BC6H_UF16                   = 95,
        DXGI_FORMAT_BC6H_SF16                   = 96,
        DXGI_FORMAT_BC7_TYPELESS                = 97,
        DXGI_FORMAT_BC7_UNORM                   = 98,
        DXGI_FORMAT_BC7_UNORM_SRGB              = 99,
        DXGI_FORMAT_AYUV                        = 100,
        DXGI_FORMAT_Y410                        = 101,
        DXGI_FORMAT_Y416                        = 102,
        DXGI_FORMAT_NV12                        = 103,
        DXGI_FORMAT_P010                        = 104,
        DXGI_FORMAT_P016                        = 105,
        DXGI_FORMAT_420_OPAQUE                  = 106,
        DXGI_FORMAT_YUY2                        = 107,
        DXGI_FORMAT_Y210                        = 108,
        DXGI_FORMAT_Y216                        = 109,
        DXGI_FORMAT_NV11                        = 110,
        DXGI_FORMAT_AI44                        = 111,
        DXGI_FORMAT_IA44                        = 112,
        DXGI_FORMAT_P8                          = 113,
        DXGI_FORMAT_A8P8                        = 114,
        DXGI_FORMAT_B4G4R4A4_UNORM              = 115,
        DXGI_FORMAT_P208                        = 130,
        DXGI_FORMAT_V208                        = 131,
        DXGI_FORMAT_V408                        = 132,
        DXGI_FORMAT_FORCE_UINT                  = 0xffffffff
    };

    // DX10 resourceDimension value
    using D3D10_RESOURCE_DIMENSION = enum D3D10_RESOURCE_DIMENSION
    { 
        D3D10_RESOURCE_DIMENSION_UNKNOWN    = 0,
        D3D10_RESOURCE_DIMENSION_BUFFER     = 1,
        D3D10_RESOURCE_DIMENSION_TEXTURE1D  = 2,
        D3D10_RESOURCE_DIMENSION_TEXTURE2D  = 3,
        D3D10_RESOURCE_DIMENSION_TEXTURE3D  = 4
    };

    // DX10 miscFlag flags
    using D3D10_RESOURCE_MISC_FLAG = enum D3D10_RESOURCE_MISC_FLAG
    { 
        D3D10_RESOURCE_MISC_GENERATE_MIPS		= 0x1,
        D3D10_RESOURCE_MISC_SHARED				= 0x2,
        D3D10_RESOURCE_MISC_TEXTURECUBE			= 0x4,
        D3D10_RESOURCE_MISC_SHARED_KEYEDMUTEX	= 0x10,
        D3D10_RESOURCE_MISC_GDI_COMPATIBLE		= 0x20
    };

    // DX10 miscFlag flags
    using D3D11_RESOURCE_MISC_FLAG = enum D3D11_RESOURCE_MISC_FLAG
    { 
        D3D11_RESOURCE_MISC_GENERATE_MIPS					= 0x1,
        D3D11_RESOURCE_MISC_SHARED							= 0x2,
        D3D11_RESOURCE_MISC_TEXTURECUBE						= 0x4,
        D3D11_RESOURCE_MISC_DRAWINDIRECT_ARGS				= 0x10,
        D3D11_RESOURCE_MISC_BUFFER_ALLOW_RAW_VIEWS			= 0x20,
        D3D11_RESOURCE_MISC_BUFFER_STRUCTURED				= 0x40,
        D3D11_RESOURCE_MISC_RESOURCE_CLAMP					= 0x80,
        D3D11_RESOURCE_MISC_SHARED_KEYEDMUTEX				= 0x100,
        D3D11_RESOURCE_MISC_GDI_COMPATIBLE					= 0x200,
        D3D11_RESOURCE_MISC_SHARED_NTHANDLE					= 0x800,
        D3D11_RESOURCE_MISC_RESTRICTED_CONTENT				= 0x1000,
        D3D11_RESOURCE_MISC_RESTRICT_SHARED_RESOURCE		= 0x2000,
        D3D11_RESOURCE_MISC_RESTRICT_SHARED_RESOURCE_DRIVER	= 0x4000,
        D3D11_RESOURCE_MISC_GUARDED							= 0x8000,
        D3D11_RESOURCE_MISC_TILE_POOL						= 0x20000,
        D3D11_RESOURCE_MISC_TILED							= 0x40000,
        D3D11_RESOURCE_MISC_HW_PROTECTED					= 0x80000
    };

    // DX10 miscFlags2 flags
    using DX10_MISC_FLAG = enum DX10_MISC_FLAG
    {
        DDS_ALPHA_MODE_UNKNOWN			= 0x0,
        DDS_ALPHA_MODE_STRAIGHT			= 0x1,
        DDS_ALPHA_MODE_PREMULTIPLIED	= 0x2,
        DDS_ALPHA_MODE_OPAQUE			= 0x3,
        DDS_ALPHA_MODE_CUSTOM			= 0x4
    };

    using DDS_PIXELFORMAT = struct DDS_PIXELFORMAT
    {
        std::uint32_t size{};
        std::uint32_t flags{};
        std::uint32_t fourCC{};
        std::uint32_t RGBBitCount{};
        std::uint32_t RBitMask{};
        std::uint32_t GBitMask{};
        std::uint32_t BBitMask{};
        std::uint32_t ABitMask{};
    };

    using DDS_HEADER_DXT10 = struct DDS_HEADER_DXT10
    {
        DXGI_FORMAT					dxgiFormat{};
        D3D10_RESOURCE_DIMENSION	resourceDimension{};
        std::uint32_t			    miscFlag{};
        std::uint32_t				arraySize{};
        std::uint32_t				miscFlags2{};
    };

    using DDS_HEADER = struct DDSHeader
    {
        std::uint32_t	    size{};
        std::uint32_t		flags{};
        std::uint32_t		height{};
        std::uint32_t		width{};
        std::uint32_t		pitchOrLinearSize{};
        std::uint32_t		depth{};
        std::uint32_t		mipMapCount{};
        std::array<std::uint32_t, 11> reserved1{};
        DDS_PIXELFORMAT	    ddspf{};
        std::uint32_t		caps{};
        std::uint32_t		caps2{};
        std::uint32_t		caps3{};
        std::uint32_t		caps4{};
        std::uint32_t		reserved2{};
        DDS_HEADER_DXT10	d3d10ext{};
    };

    DDSReader(const ImagePtr &img) :
        _img(img),
        _dds{}
    {
    }

    ~DDSReader()
    {
    }

    bool read(std::istream &is)
    {
        auto start = std::chrono::high_resolution_clock::now();

        // Read the magic number
        std::array<char, 4> magic{};
        is.read(magic.data(), 4);
        if(memcmp(magic.data(), "DDS ", 4) != 0)
        {
            std::cerr << "DDSReader: Not a valid DDS stream!" << std::endl;
            return false;
        }

        // Read the header
        // read into struct up to dwReserved2; (EOF header)
        const std::size_t headerSize = offsetof(DDSHeader, reserved2) + sizeof(DDSHeader::reserved2);
        is.read(reinterpret_cast<char*>(&_dds), headerSize);

        if (_dds.size != headerSize)
        {
            std::cerr << "DDSReader: Invalid DDS header size: " << _dds.size << std::endl;
            return false;
        }

        const bool isDx10 = (_dds.ddspf.flags & DDPF_FOURCC) && (memcmp(&_dds.ddspf.fourCC, "DX10", 4) == 0 ? true : false);

        if (isDx10)
        {
            is.read(reinterpret_cast<char*>(&_dds.d3d10ext), sizeof(DDS_HEADER_DXT10));
        }

        const std::streamsize currentPos = is.tellg();
        is.seekg(0, std::ios::end);
        const std::size_t dwBufferSize = is.tellg() - currentPos;
        is.seekg(currentPos);

        std::uint32_t width{};
        std::uint32_t height{};
        std::uint32_t depth{1};
        std::uint32_t sides{1};
        std::uint32_t format{};
        std::uint32_t type{};
        std::size_t pitch{};
        std::int32_t mipMapCount{1};

        if(isDx10)
        {
            if (_dds.size != (sizeof(DDS_HEADER) - sizeof(DDS_HEADER_DXT10)) || _dds.ddspf.size != sizeof(DDS_PIXELFORMAT))
            {
                // We do not accept legacy DX9 'known variants' for modern "DX10" extension header files.
                return false;
            }

            sides = _dds.d3d10ext.arraySize;
            if(sides == 0)
                return false;

            std::tie(format, type) = convertDxgiToOsg(_dds.d3d10ext.dxgiFormat, is, dwBufferSize);

            if(format == Image::OSG_INVALID_PF)
            {
                std::cout << "DDSReader: Unsupported format (" << _dds.d3d10ext.dxgiFormat << ")!" << std::endl;
                return false;
            }

            switch (_dds.d3d10ext.resourceDimension)
            {
            case D3D10_RESOURCE_DIMENSION_TEXTURE1D:

                // D3DX writes 1D textures with a fixed Height of 1
                if ((_dds.flags & DDSD_HEIGHT) && _dds.height != 1)
                {
                    return false;
                }

                width = _dds.width;
                height = 1;
                depth = 1;
                break;

            case D3D10_RESOURCE_DIMENSION_TEXTURE2D:
                if (_dds.d3d10ext.miscFlag & D3D10_RESOURCE_MISC_TEXTURECUBE)
                {
                    sides *= 6;
                }

                width = _dds.width;
                height = _dds.height;
                depth = 1;
                break;

            case D3D10_RESOURCE_DIMENSION_TEXTURE3D:
                if (!(_dds.flags & DDSCAPS2_VOLUME))
                {
                    return false;
                }

                if (sides > 1)
                    return false; // not supported

                width = _dds.width;
                height = _dds.height;
                depth = _dds.depth;
                break;

            default:
                return false;
            }

            mipMapCount = _dds.mipMapCount;
        }
        else
        {
            if (_dds.flags & DDSD_HEIGHT)
                height = _dds.height;
            if (_dds.flags & DDSD_WIDTH)
                width = _dds.width;
            if (_dds.flags & DDSD_DEPTH)
                depth = _dds.depth;
            if (_dds.flags & DDSD_MIPMAPCOUNT)
                mipMapCount = _dds.mipMapCount;

            if(_dds.flags & DDSD_PITCH)
            {
                pitch = _dds.pitchOrLinearSize;
            }
            if(pitch != width * _dds.ddspf.RGBBitCount / 8)
            {
                std::cerr << "DDSReader: pitch of " << pitch << " is not supported!" << std::endl;
                return false;
            }

            if(_dds.ddspf.flags & DDPF_RGB)
            {
                std::tie(format, type) = convertRGBDxgiToOsg();
                if(format == Image::OSG_INVALID_PF)
                {
                    std::cout << "DDSReader: Unsupported format!" << std::endl;
                    return false;
                }
            }
            else if(_dds.ddspf.flags & DDPF_FOURCC)
            {
                std::tie(format, type) = convertCompressedDxgiToOsg(is, dwBufferSize);
                if(format == Image::OSG_INVALID_PF)
                {
                    std::cout << "DDSReader: Unsupported compressed format!" << std::endl;
                    return false;
                }
            }
            else
            {
                std::cerr << "DDSReader: Unsupported format!" << std::endl;
                return false;
            }
        }

        //bool compressed = this->isCompressed(_dds.d3d10ext.dxgiFormat);

        _img->set(format,
                  width,
                  height,
                  depth,
                  mipMapCount,
                  1, // frameCount
                  0.0, // Time
                  nullptr, // data
                  type,
                  true, // allocMem
                  sides); // sideCount


        const bool compressed = _img->isCompressed();
        _img->setIsCompressible(compressed ? true : false);

        if(_img->getRawSize() != dwBufferSize)
        {
            std::cerr << "DDSReader: Image size does not match buffer size!" << std::endl;
            return false;
        }

        is.read(reinterpret_cast<char*>(_img->getRawData(0, 0, 0)),  _img->getRawSize());
        if(is.gcount() != _img->getRawSize())
        {
            std::cerr << "DDSReader: read buffer failed, file has wrong size!" << std::endl;
            return false;
        }

#if 0
        std::cout << "isDx10: " << isDx10 << std::endl;
        std::cout << "format: " << format << std::endl;
        std::cout << "width: " << width << std::endl;
        std::cout << "height: " << height << std::endl;
        std::cout << "depth: " << depth << std::endl;
        std::cout << "pitch: " << pitch << std::endl;
        std::cout << "sides: " << sides << std::endl;
        std::cout << "mipMap Count: " << mipMapCount << std::endl;
        std::cout << "isCompressed: " << _img->isCompressed() << std::endl;
        std::cout << "bpp: " << _img->getBpp() << std::endl;
        std::cout << "components: " << static_cast<std::uint32_t>(_img->getComponents()) << std::endl;
#endif

        //return true;

        auto end = std::chrono::high_resolution_clock::now();
        std::cout << "DDSReader::read: reading dds image: " << std::chrono::duration_cast<std::chrono::milliseconds>(end - start).count() << " ms." << std::endl;

        // first we convert our osg image to nvtt surfaces
        ImageCompressor imgCompressor;
        if(!imgCompressor.setImage(_img))
        {
            std::cerr << "DDSReader::read: converting osg image to nvtt surface failed!" << std::endl;
            return false;
        }
        imgCompressor.convertCubemapLayoutFromDXToOGL();
        imgCompressor.getImage(_img);
        //_img->dump();
        return true;
    }

private:

    std::pair<std::int32_t, std::uint32_t> convertRGBDxgiToOsg()
    {
        std::uint32_t format{};
        std::uint32_t type{};

        const bool isBGR = (_dds.ddspf.RBitMask == 0x00FF0000 && _dds.ddspf.GBitMask == 0x0000FF00 && _dds.ddspf.BBitMask == 0x000000FF);
        switch (_dds.ddspf.RGBBitCount)
        {
        case 32:
            format = isBGR ? Image::OSG_BGRA_PF : Image::OSG_RGBA_PF;
            break;
        case 24:
            format = isBGR ? Image::OSG_BGR_PF : Image::OSG_RGB_PF;
            break;
        //case 8:
        //    format = Image::OSG_L_PF;
        //    break;
        }
        type = Image::OSG_UINT8_IMAGEDATA;

        return {format, type};
    }

    bool hasDXT1Alpha(std::istream &is, std::size_t dwBufferSize)
    {
        std::vector<char> buffer(dwBufferSize);
        // Store the current file position
        std::istream::pos_type currentPos = is.tellg();
        is.read(buffer.data(), buffer.size());
        // Restore the file position
        is.seekg(currentPos);

        // Check for alpha in the buffer
        for (std::size_t i = 0; i < buffer.size(); i += 8)
        {
            uint16_t color0 = *reinterpret_cast<uint16_t *>(&buffer[i]);
            uint16_t color1 = *reinterpret_cast<uint16_t *>(&buffer[i + 2]);
            uint32_t code = *reinterpret_cast<uint32_t *>(&buffer[i + 4]);

            // If color0 <= color1, the block contains alpha
            if (color0 <= color1)
            {
                for (int j = 0; j < 16; ++j)
                {
                    if (((code >> (2 * j)) & 0x03) == 3)
                    {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    std::pair<std::int32_t, std::uint32_t> convertCompressedDxgiToOsg(std::istream &is, std::size_t dwBufferSize)
    {
        std::uint32_t format{};
        std::uint32_t type{};

        switch (_dds.ddspf.fourCC)
        {
        case FOURCC_DXT1:
            if(hasDXT1Alpha(is, dwBufferSize))
                format = Image::OSG_RGBA_DXT1_PF;
            else
                format = Image::OSG_RGB_DXT1_PF;
            break;
        case FOURCC_DXT3:
            format = Image::OSG_RGBA_DXT3_PF;
            break;
        case FOURCC_DXT5:
            format = Image::OSG_RGBA_DXT5_PF;
            break;
        }
        type = Image::OSG_UINT8_IMAGEDATA;

        return {format, type};
    }

    std::pair<std::int32_t, std::uint32_t> convertDxgiToOsg(std::uint32_t dxgiFormat, std::istream &is, std::size_t dwBufferSize)
    {
        switch (dxgiFormat)
        {
            case DXGI_FORMAT_R8G8B8A8_UNORM:
                return {Image::OSG_RGBA_PF, Image::OSG_UINT8_IMAGEDATA};
            case DXGI_FORMAT_R8G8B8A8_UNORM_SRGB:
                return {Image::OSG_RGBA_PF, Image::OSG_UINT8_IMAGEDATA};
            case DXGI_FORMAT_B8G8R8A8_TYPELESS:
                return {Image::OSG_BGRA_PF, Image::OSG_UINT8_IMAGEDATA};
            case DXGI_FORMAT_B8G8R8A8_UNORM_SRGB:
                return {Image::OSG_BGRA_PF, Image::OSG_UINT8_IMAGEDATA};
            case DXGI_FORMAT_B8G8R8X8_TYPELESS:
                return {Image::OSG_BGRA_PF, Image::OSG_UINT8_IMAGEDATA};
            case DXGI_FORMAT_B8G8R8X8_UNORM_SRGB:
                return {Image::OSG_BGRA_PF, Image::OSG_UINT8_IMAGEDATA};
            case DXGI_FORMAT_R9G9B9E5_SHAREDEXP:
                return {Image::OSG_RGB9E5_PF, Image::OSG_FLOAT16_IMAGEDATA};
            case DXGI_FORMAT_BC1_UNORM:
            case DXGI_FORMAT_BC1_UNORM_SRGB:
                if(hasDXT1Alpha(is, dwBufferSize))
                    return {Image::OSG_RGBA_DXT1_PF, Image::OSG_UINT8_IMAGEDATA};
                else
                    return {Image::OSG_RGB_DXT1_PF, Image::OSG_UINT8_IMAGEDATA};
            case DXGI_FORMAT_BC2_UNORM:
                return {Image::OSG_RGBA_DXT3_PF, Image::OSG_UINT8_IMAGEDATA};
            case DXGI_FORMAT_BC2_UNORM_SRGB:
                return {Image::OSG_RGBA_DXT3_PF, Image::OSG_UINT8_IMAGEDATA};
            case DXGI_FORMAT_BC3_UNORM:
                return {Image::OSG_RGBA_DXT5_PF, Image::OSG_UINT8_IMAGEDATA};
            case DXGI_FORMAT_BC3_UNORM_SRGB:
                return {Image::OSG_RGBA_DXT5_PF, Image::OSG_UINT8_IMAGEDATA};
            case DXGI_FORMAT_BC5_UNORM:
                return {Image::OSG_RGBA_DXT5_PF, Image::OSG_UINT8_IMAGEDATA};
            case DXGI_FORMAT_R32G32B32_FLOAT:
                return {Image::OSG_RGB_PF, Image::OSG_FLOAT32_IMAGEDATA};
            case DXGI_FORMAT_R32G32B32A32_FLOAT:
                return {Image::OSG_RGBA_PF, Image::OSG_FLOAT32_IMAGEDATA};
            case DXGI_FORMAT_R16G16B16A16_FLOAT:
                return {Image::OSG_RGBA_PF, Image::OSG_FLOAT16_IMAGEDATA};
            case DXGI_FORMAT_BC6H_SF16:
                return {Image::OSG_RGB_BC6S_PF, Image::OSG_FLOAT16_IMAGEDATA};
            case DXGI_FORMAT_BC6H_UF16:
                return {Image::OSG_RGB_BC6U_PF, Image::OSG_FLOAT16_IMAGEDATA};
            case DXGI_FORMAT_BC7_UNORM:
                return {Image::OSG_RGBA_BC7_PF, Image::OSG_UINT8_IMAGEDATA};
            case DXGI_FORMAT_BC7_UNORM_SRGB:
                return {Image::OSG_RGBA_BC7_PF, Image::OSG_UINT8_IMAGEDATA};
            default:
                return {Image::OSG_INVALID_PF, Image::OSG_INVALID_IMAGEDATATYPE};
        }
        return {Image::OSG_INVALID_PF, Image::OSG_INVALID_IMAGEDATATYPE};
    }

    ImagePtr _img;
    DDSHeader _dds;

};

/*****************************
 *   Types
 *****************************/
// Static Class Varible implementations:
static const Char8 *suffixArray[] =
{
    "dds"
};

DDSImageFileType DDSImageFileType::_the("image/x-dds",
                                        suffixArray, sizeof(suffixArray),
                                        OSG_READ_SUPPORTED |
                                        OSG_WRITE_SUPPORTED);


/********************************
 *    Class methods
 *******************************/

//-------------------------------------------------------------------------
/*!
Class method to get the singleton Object
*/
DDSImageFileType& DDSImageFileType::the (void)
{
  return _the;
}

/*******************************
*public
*******************************/

//-------------------------------------------------------------------------
/*!
Tries to fill the image object with the data read from
the given input stream. Returns true on success.
*/
bool DDSImageFileType::read(ImagePtr &image, std::istream &is, const std::string &mimetype)
{
    DDSReader reader(image);
    return reader.read(is);
}

//-------------------------------------------------------------------------
/*!
Tries to determine the mime type of the data provided by an input stream
by searching for magic bytes. Returns the mime type or an empty string
when the function could not determine the mime type.
*/
std::string DDSImageFileType::determineMimetypeFromStream(std::istream &is)
{
    char filecode[4];
    is.read(filecode, 4);
    is.seekg(-4, std::ios::cur);
    return strncmp(filecode, "DDS ", 4) == 0 ?
        std::string(getMimeType()) : std::string();
}

//-------------------------------------------------------------------------
/*!
Constructor used for the singleton object
*/
DDSImageFileType::DDSImageFileType(const Char8 *mimeType,
                                   const Char8 *suffixArray[],
                                   UInt16 suffixByteCount,
                                   UInt32 flags) :
    ImageFileType(mimeType, suffixArray, suffixByteCount, flags),
    _flipImage(true),
    _flipCubeMap(false),
    _swapCubeMap(false)
{}

//-------------------------------------------------------------------------
/*!
Destructor
*/
DDSImageFileType::~DDSImageFileType()
{
}
