/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright (C) 2000-2002 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

//-------------------------------
//      Includes
//-------------------------------


#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <memory.h>

#include <OSGConfig.h>

#include <iostream>
#include <fstream>
#include <thread>
#include <mutex>
#include <string>
#include <set>

#include <errno.h>

#ifdef _WIN32
    #ifdef min
        #undef min
    #endif
    #ifdef max
        #undef max
    #endif
#endif

#ifdef OSG_WITH_IMF

#include <OpenEXR/OpenEXRConfig.h>
#define COMBINED_OPENEXR_VERSION ((10000*OPENEXR_VERSION_MAJOR) + \
                                  (100*OPENEXR_VERSION_MINOR) + \
                                  OPENEXR_VERSION_PATCH)
#if COMBINED_OPENEXR_VERSION >= 20599
 #include <Iex.h>
 #include <Imath/ImathInt64.h>
  #include <Imath/ImathBox.h>
  #include "ImfPixelType.h"
#else
 #include <OpenEXR/Iex.h>
 #include <OpenEXR/ImathInt64.h>
  #include <OpenEXR/ImathBox.h>
#endif


 #include <OpenEXR/ImfIO.h>
 #include <OpenEXR/ImfChannelList.h>
 #include <OpenEXR/ImfHeader.h>
 #include <OpenEXR/ImfVersion.h>
 #include <OpenEXR/ImfArray.h>
 #include <OpenEXR/ImfRgba.h>
 #include <OpenEXR/ImfRgbaFile.h>
 #include <OpenEXR/ImfInputFile.h>
 #include <OpenEXR/ImfOutputFile.h>
 #include <OpenEXR/ImfStandardAttributes.h>
 #include <OpenEXR/ImfLineOrder.h>
 #include <OpenEXR/ImfCompression.h>
#include <OpenEXR/ImfMultiPartInputFile.h>
#include <OpenEXR/ImfBoxAttribute.h>
#include <OpenEXR/ImfChannelListAttribute.h>
#include <OpenEXR/ImfChromaticitiesAttribute.h>
#include <OpenEXR/ImfCompressionAttribute.h>
#include <OpenEXR/ImfDoubleAttribute.h>
#include <OpenEXR/ImfEnvmapAttribute.h>
#include <OpenEXR/ImfFloatAttribute.h>
#include <OpenEXR/ImfIntAttribute.h>
#include <OpenEXR/ImfKeyCodeAttribute.h>
#include <OpenEXR/ImfLineOrderAttribute.h>
#include <OpenEXR/ImfMatrixAttribute.h>
#include <OpenEXR/ImfPreviewImageAttribute.h>
#include <OpenEXR/ImfRationalAttribute.h>
#include <OpenEXR/ImfStringAttribute.h>
#include <OpenEXR/ImfStringVectorAttribute.h>
#include <OpenEXR/ImfTileDescriptionAttribute.h>
#include <OpenEXR/ImfTimeCodeAttribute.h>
#include <OpenEXR/ImfVecAttribute.h>
#include <OpenEXR/ImfVersion.h>


#endif

#include <OSGLog.h>
#include <OSGImageFileHandler.h>
#include <OSGPathHandler.h>
#include <OSGFileSystem.h>
#include <OSGImageGenericAtt.h>
#include <OSGColorConversionFunctions.h>

#include "OSGEXRImageFileType.h"

#include <oneapi/tbb/parallel_for.h>

OSG_USING_NAMESPACE

// at the moment we only import the base layer from multilayer exr files since we have no use for the other
// layers. However, just undefine the next line and the import should already work. Note that envmaps are only imported from RGB and RGBA data
// and are not stored as such in any case.
#define IMPORT_SINGLE_LAYER_ONLY

/*! \class osg::EXRImageFileType
    \ingroup GrpSystemImage

Image File Type to read/write and store/restore Image objects as EXR data.
Does depend on external libs, namely OpenEXR (by ILM).

Needs to be build with OSG_WITH_IMF

 */

/*****************************
 *   Types
 *****************************/
// Static Class Varible implementations:

static const Char8 *suffixArray[] =
{
    "exr"
};

EXRImageFileType EXRImageFileType::_the( "image/x-exr",
                                         suffixArray, sizeof(suffixArray),
                                         OSG_READ_SUPPORTED |
                                         OSG_WRITE_SUPPORTED |
                                         OSG_LAYER_SUPPORTED | OSG_FLOATDATA_SUPPORTED);


class OSGImfThreadInit
{
public:
    static void init()
    {
        static OSGImfThreadInit me;
    }
private:
    OSGImfThreadInit()
    {
        // special hack for <= 2.2.0 with 2.3.0 there is no global lock anymore!
        // I added some multi threaded code to testOpenEXR to test this.
#if ILMBASE_VERSION_HEX <= 0x02020000
        std::lock_guard<std::mutex> lock(_mutex);

        // amz moved this from the constructor to here doing a lazy initialization!
        // The constructor is called via static initialization
        // this seems to work fine on windows 7 but not on windows 10 and also leads to big
        // problems (deadlock, not working, ...) with our vrThumbnailProvider.dll!
        // For OpenEXR 2.2.0 without this trick even writing n files with n threads in parallel won't
        // improve the writing speed looks like a global lock to me.
        Imf::setGlobalThreadCount(std::thread::hardware_concurrency());
#endif
    }

    static std::mutex _mutex;
};

std::mutex OSGImfThreadInit::_mutex;

/*****************************
 *    Classvariables
 *****************************/


/********************************
 *    Class methodes
 *******************************/

#ifdef OSG_WITH_IMF

//-------------------------------------------------------------------------
/** class StdIFStream -- an implementation of
    class IStream based on class std::istream
 */
class StdIStream : public Imf::IStream
{
  public:
    /** A constructor that uses a std::istream that has already
        been opened by the caller.  The StdIStream's destructor
        will not close the std::istream.
     */
    StdIStream(std::istream &is, const char fileName[]);
    virtual ~StdIStream();

    virtual bool       read(char c[], int n);
    virtual uint64_t   tellg();
    virtual void       seekg(uint64_t pos);
    virtual void       clear();

  private:
    bool checkError();

    std::istream *_is;
};

//-------------------------------------------------------------------------
/** class StdOStream -- an implementation of
    class OStream based on class std::ostream
 */
class StdOStream : public Imf::OStream
{
  public:
    /** A constructor that uses a std::ostream that has already
        been opened by the caller.  The StdOStream's destructor
        will not close the std::ostream.
     */
    StdOStream(std::ostream &os, const char fileName[]);
    virtual ~StdOStream();

    virtual void       write(const char c[], int n);
    virtual uint64_t   tellp();
    virtual void       seekp(uint64_t pos);

  private:
    void checkError();

    std::ostream *_os;
};

//-------------------------------------------------------------------------
StdIStream::StdIStream(std::istream &is, const char fileName[]) :
    Imf::IStream(fileName), _is(&is) { }

StdIStream::~StdIStream() { }

bool StdIStream::read(char c[], int n)
{
    if (!*_is)
       throw Iex::InputExc("Unexpected end of file.");

    errno = 0;
    _is->read(c, n);

    return checkError();
}

uint64_t StdIStream::tellg()
{
    return std::streamoff(_is->tellg());
}

void StdIStream::seekg(uint64_t pos)
{
    errno = 0;
    _is->seekg(pos);
    checkError();
}

void StdIStream::clear()
{
    _is->clear();
}

bool StdIStream::checkError()
{
    if (!*_is)
    {
        if (errno)
            Iex::throwErrnoExc();

        _is->clear(std::ios_base::goodbit);
        return false;
    }
    return true;
}

//-------------------------------------------------------------------------
StdOStream::StdOStream(std::ostream &os, const char fileName[]) :
    Imf::OStream(fileName), _os(&os) { }

StdOStream::~StdOStream() { }

void StdOStream::write(const char c[], int n)
{
    errno = 0;
    _os->write(c, n);
    checkError();
}

uint64_t StdOStream::tellp()
{
    return std::streamoff(_os->tellp());
}

void StdOStream::seekp(uint64_t pos)
{
    errno = 0;
    _os->seekp(pos);
    checkError();
}

void StdOStream::checkError()
{
    if (!*_os)
    {
        if (errno)
            Iex::throwErrnoExc();
        throw Iex::ErrnoExc ("Write failed.");
    }
}

#endif  //OSG_WITH_IMF

//-------------------------------------------------------------------------
/*!
Class method to get the singleton Object
*/
EXRImageFileType& EXRImageFileType::the (void)
{
    return _the;
}

osg::Matrix getToLinRec709Matrix(const Imf::Header &header)
{
    const auto* chromaticities = header.findTypedAttribute<Imf::ChromaticitiesAttribute>(Imf::ChromaticitiesAttribute::staticTypeName());
    if(chromaticities)
    {
        return getToLinRec709Matrix(chromaticities->value());
    }
    return osg::Matrix::identity();
}


/*******************************
*public
*******************************/

//-------------------------------------------------------------------------
/*!
    Tries to fill the image object with the data read from
    the given input stream. Returns true on success.
*/
bool EXRImageFileType::read(ImagePtr &image, std::istream &is, const std::string &mimetype)
{
#ifdef OSG_WITH_IMF
    if (!is.good())
        return false;

    const char *dummy = "";
    StdIStream file(is, dummy);

    uint64_t pos = file.tellg();

    bool check = isOpenExrFile(is);

    file.seekg(pos);

    if (!check)
    {
        FFATAL(( "Wrong format, no %s image given!\n",
                  mimetype.c_str() ));
        return false;
    }

    OSGImfThreadInit::init();

    // just read the header and get the channel count
    int channel_count = 0;
    size_t dataTypeSize = sizeof(Real16);
    Image::Type imgFormat = Image::OSG_FLOAT16_IMAGEDATA;
    Imf::PixelType pt = Imf::HALF; // default to half
    bool isCubeEnvMap = false;
    bool isLongLatEnvMap = false;
    pos = file.tellg();
    try
    {
        Imf::RgbaInputFile stream(file);
        const Imf::Header &header = stream.header();
        const Imf::ChannelList &channels = header.channels();
        const Imf::EnvmapAttribute *envmap = header.findTypedAttribute<Imf::EnvmapAttribute>("envmap");
        isCubeEnvMap = (envmap && envmap->value() == Imf::ENVMAP_CUBE) ? true : false;
        isLongLatEnvMap = (envmap && envmap->value() == Imf::ENVMAP_LATLONG) ? true : false;
        for(Imf::ChannelList::ConstIterator it=channels.begin();it!=channels.end();++it)
        {
            switch (it.channel().type)
            {
                case Imf::UINT:
                    pt = Imf::UINT;
                    imgFormat = Image::OSG_UINT32_IMAGEDATA;
                    dataTypeSize = sizeof(Int32);
                    break;
                case Imf::HALF:
                    pt = Imf::HALF;
                    dataTypeSize = sizeof(Real16);
                    imgFormat = Image::OSG_FLOAT16_IMAGEDATA;
                    break;
                case Imf::FLOAT:
                    pt = Imf::FLOAT;
                    dataTypeSize = sizeof(Real32);
                    imgFormat = Image::OSG_FLOAT32_IMAGEDATA;
                    break;
                default:
                    std::cout << "unknown format";
            }
            
            ++channel_count;
        }
    }
    catch(std::exception &e)
    {
        FFATAL(( "Error while trying to read OpenEXR Image from stream: %s\n",
                  e.what() ));
        return false;
    }
    file.seekg(pos);

    osg::Matrix colorMatrix = osg::Matrix::identity();
    // fast case for RGB and RGBA images
    if( (isCubeEnvMap && (channel_count == 4 || channel_count == 3)) || channel_count > 4)
    {
        try
        {
            int64_t width, height, numImg = 1;
            Imf::RgbaInputFile stream(file);
    
            Imath::Box2i dw = stream.dataWindow();
            Imf::Array2D<Imf::Rgba> pixels;
    
            const Imf::Header &header = stream.header();
            const Imf::LineOrder &order = header.lineOrder();
            const Imf::EnvmapAttribute *envmap = header.findTypedAttribute<Imf::EnvmapAttribute>("envmap");

            colorMatrix = getToLinRec709Matrix(header);

            
            width  = dw.max.x - dw.min.x + 1;
            height = dw.max.y - dw.min.y + 1;

            // read the custom header attributes
            for (Imf::Header::ConstIterator it = header.begin(); it != header.end(); ++it)
            {
                
                Imf::Attribute *copy = it.attribute().copy();
                Imf::StringAttribute *sa = dynamic_cast<Imf::StringAttribute *>(copy);
                if (sa != NULL)
                    image->setAttachmentField(it.name(), sa->value());
                delete copy;
            }

            pixels.resizeErase(height, width);
    
            if(envmap && envmap->value() == Imf::ENVMAP_CUBE)
            {
                numImg = 6;
                
    
                if (width != (height/numImg))
                {
                    FFATAL(( "Cubemaps must have squared size, but w=%d and h=%d!\n", width, height ));
                    numImg = 1;
                }
                else 
                {
                    height /= numImg;
                }
            }
    
            stream.setFrameBuffer(&pixels[0][0] - dw.min.x - dw.min.y * width, 1, width);
           
            stream.readPixels(dw.min.y, dw.max.y);

            switch (pt)
            {
                case Imf::UINT:
                {
                    FFATAL(( "32-bit unsigned integer currently not supported\n" ));
                    return false;
                }
                case Imf::HALF:
                {
                    if(channel_count >= 4)
                        image->set( Image::OSG_RGBA_PF, width, height, 1, 1, 1, 0, 0, Image::OSG_FLOAT16_IMAGEDATA, true, numImg );
                    else
                        image->set( Image::OSG_RGB_PF, width, height, 1, 1, 1, 0, 0, Image::OSG_FLOAT16_IMAGEDATA, true, numImg );

                    image->clear(0);
                    Real16 *data = reinterpret_cast<Real16*>(image->editData());
                    for (int64_t side = numImg-1; side >=0; side--)
                    {
                        size_t i, j, size = side * width * height * 4;
                        size_t yEnd = (side+1)*height;
                        for (size_t y = side*height; y < yEnd; ++y)
                        {
                            for (size_t x = 0; x < width; ++x)
                            {
                                if (numImg == 1 || side == 2 || side == 3)
                                {
                                    i = (2 * side + 1) * height - (y + 1); // new y
                                    j = x;
                                }
                                else
                                {
                                    i = y;
                                    j = width - x - 1; // new x
                                }
                                // need to clamp the values, otherwise they may get screwed up
                                *(data + size++) = Real16( std::min( HALF_MAX_VAL, static_cast<float>(pixels[i][j].r)));
                                *(data + size++) = Real16( std::min( HALF_MAX_VAL, static_cast<float>(pixels[i][j].g)));
                                *(data + size++) = Real16( std::min( HALF_MAX_VAL, static_cast<float>(pixels[i][j].b)));
                                if(channel_count >= 4)
                                    *(data + size++) = Real16( std::min( HALF_MAX_VAL, static_cast<float>(pixels[i][j].a)));
                            }
                        }
                    }
                    break;
                }
                case Imf::FLOAT:
                {
                    if(channel_count >= 4)
                        image->set( Image::OSG_RGBA_PF, width, height, 1, 1, 1, 0, 0, Image::OSG_FLOAT32_IMAGEDATA, true, numImg );
                    else
                        image->set( Image::OSG_RGB_PF, width, height, 1, 1, 1, 0, 0, Image::OSG_FLOAT32_IMAGEDATA, true, numImg );
                    image->clear(0);
                    Real32 *data = reinterpret_cast<Real32*>(image->editData());
                    for (int64_t side=numImg-1; side >=0; side--)
                    {
                        size_t i, j, size = side * width * height * 4;
                        size_t yEnd = (side+1)*height;
                        for (size_t y=side*height; y<yEnd; y++)
                        {
                            for (size_t x=0; x<width; x++)
                            {
                                if (numImg == 1 || side == 2 || side == 3)
                                {
                                    i = (2 * side + 1) * height - (y + 1); // new y
                                    j = x;
                                }
                                else
                                {
                                    i = y;
                                    j = width - x - 1; // new x
                                }
                                // this should not happen for 32 bit float but it does
                                if( isinf(pixels[i][j].r))
                                    *(data + size++) = Real32(HALF_MAX_VAL);
                                else
                                    *(data + size++) = Real32(pixels[i][j].r);

                                if( isinf(pixels[i][j].g))
                                    *(data + size++) = Real32(HALF_MAX_VAL);
                                else
                                    *(data + size++) = Real32(pixels[i][j].g);

                                if( isinf(pixels[i][j].b))
                                    *(data + size++) = Real32(HALF_MAX_VAL);
                                else
                                    *(data + size++) = Real32(pixels[i][j].b);

                                if(channel_count >= 4)
                                {
                                    if( isinf(pixels[i][j].a))
                                        *(data + size++) = 1.0f;
                                    else
                                        *(data + size++) = Real32(pixels[i][j].a);
                                }
                            }
                        }
                    }
                    break;
                }
                default:
                    return false;
            };
            convertImage(colorMatrix, image);
            return true;
        }
        catch(std::exception &e)
        {
            FFATAL(( "Error while trying to read OpenEXR Image from stream: %s\n",
                      e.what() ));
            return false;
        }
    }
    else
    {
        try
        {
            Imf::InputFile stream(file);
            const Imf::Header &header = stream.header();
            colorMatrix = getToLinRec709Matrix(header);
            Imath::Box2i dw = header.dataWindow();

            int width = dw.max.x - dw.min.x + 1;
            int height = dw.max.y - dw.min.y + 1;

            const Imf::ChannelList &channels = header.channels();
            // determine the number of channels for a single layer. Only EXR files with same number of channels in all layers are supported
            std::set<std::string> layerNames;
            channels.layers(layerNames);
            int numChannelPerLayer = 0;
            if (layerNames.size() > 0)
            {
                for (auto layerName = layerNames.begin(); layerName != layerNames.end(); ++layerName)
                {
                    Imf::ChannelList::ConstIterator layerBegin, layerEnd;
                    channels.channelsInLayer(*layerName, layerBegin, layerEnd);
                    int channelsInLayer = 0;

                    for (Imf::ChannelList::ConstIterator it = layerBegin; it != layerEnd; ++it)
                        channelsInLayer++;

                    if (numChannelPerLayer == 0)
                        numChannelPerLayer = channelsInLayer;
                    else
                    {
                        if (channelsInLayer != numChannelPerLayer)
                        {
                            FFATAL(("Error while trying to read OpenEXR Image from stream, different number of channels in layer are not supported!\n"));
                            return false;
                        }
                    }
                }
            }
            else
            {
                for (Imf::ChannelList::ConstIterator it = channels.begin(); it != channels.end(); ++it)
                    numChannelPerLayer++; 
            }

            if(numChannelPerLayer == 0)
                numChannelPerLayer = channel_count;

#ifdef IMPORT_SINGLE_LAYER_ONLY
            int num_img = 1;
#else
            int num_img = channel_count / numChannelPerLayer;
#endif
            switch (numChannelPerLayer)
            {
            case 1:
                image->set(Image::OSG_L_PF, width, height, 1, 1, 1, 0, 0, imgFormat, true, num_img);
                break;
            case 2:
                image->set(Image::OSG_LA_PF, width, height, 1, 1, 1, 0, 0, imgFormat, true, num_img);
                break;
            case 3:
                image->set(Image::OSG_RGB_PF, width, height, 1, 1, 1, 0, 0, imgFormat, true, num_img);
                break;
            case 4:
                image->set(Image::OSG_RGBA_PF, width, height, 1, 1, 1, 0, 0, imgFormat, true, num_img);
                break;
            default:
                image->set(Image::OSG_L_PF, width, height, 1, 1, 1, 0, 0, imgFormat, true, num_img);
            }
           
            /*if (imgFormat == Image::OSG_FLOAT16_IMAGEDATA)
                image->clearHalf();
            else if (imgFormat == Image::OSG_FLOAT32_IMAGEDATA)
                image->clearFloat();
            else*/
                image->clear(0);

            // now add custom attributes
            for(Imf::Header::ConstIterator it=header.begin();it!=header.end();++it)
            {
                Imf::Attribute *copy = it.attribute().copy();
                Imf::StringAttribute *sa = dynamic_cast<Imf::StringAttribute *>(copy);
                if(sa != NULL)
                {
                    image->setAttachmentField(it.name(), sa->value());
                }
    
                delete copy;
            }

            Imf::FrameBuffer frame_buffer;
            Imf::PixelType pixelType = (image->getDataType() == Image::OSG_FLOAT32_IMAGEDATA) ? Imf::FLOAT : Imf::HALF;
            size_t dataTypeSize = (image->getDataType() == Image::OSG_FLOAT32_IMAGEDATA) ? sizeof(Real32) : sizeof(Real16);
            // we need to do a vertical flip so we read single scan lines in.
            int current_scan_line = 0;
            size_t scanlineSize = dataTypeSize * numChannelPerLayer * width;
            for(int i=height-1;i>=0;--i)
            {
                int channelIdx = 0;
                for (Imf::ChannelList::ConstIterator it = channels.begin(); it != channels.end(); ++it)
                {
                    int side = channelIdx / numChannelPerLayer;
                    channelIdx++;
#ifdef IMPORT_SINGLE_LAYER_ONLY
                    if (side != 0)
                        break;
#endif
                    char *data = (reinterpret_cast<char *>(image->editData(0, 0, side))) + i * scanlineSize;
                    data -= current_scan_line * scanlineSize;
                    int channeloffset = getChannelIndexFromName(it.name());
                    frame_buffer.insert(it.name(), Imf::Slice(pt, data + (channeloffset * dataTypeSize), dataTypeSize * numChannelPerLayer, scanlineSize));
             
                }
                stream.setFrameBuffer(frame_buffer);
                stream.readPixels(current_scan_line, current_scan_line);
                ++current_scan_line;
            }
            // we need to make a safety check too make sure we don´t have any infinity values
            if(pt == Imf::HALF)
            {
                half* checkData = reinterpret_cast<half*>(image->getData());
                oneapi::tbb::parallel_for(oneapi::tbb::blocked_range<size_t>(0, width*height*numChannelPerLayer), [&](const oneapi::tbb::blocked_range<size_t> &range)
                {
                    for (auto idx = range.begin(); idx != range.end(); ++idx)
                    {
                        if( checkData[idx].isInfinity())
                            checkData[idx] = HALF_MAX_VAL;
                        if( checkData[idx].isNan())
                            checkData[idx] = 0.0f;
                    }
                });
            }
            convertImage(colorMatrix, image);
            return true;
        }
        catch(std::exception &e)
        {
            FFATAL(( "Error while trying to read OpenEXR Image from stream: %s\n",
                      e.what() ));
            return false;
        }
    }

    return false;

#else
    SWARNING << getMimeType()
             << " read is not compiled into the current binary "
             << std::endl;
    return false;
#endif
}

//-------------------------------------------------------------------------
/*!
    Tries to write the image object to the given output stream.
    Returns true on success.
*/
bool EXRImageFileType::write(const ImagePtr &image, std::ostream &os, const std::string &mimetype)
{
#ifdef OSG_WITH_IMF
    if (!os.good())
        return false;

    if(image->getDataType() != Image::OSG_FLOAT16_IMAGEDATA && image->getDataType() != Image::OSG_FLOAT32_IMAGEDATA)
    {
        FWARNING(("EXRImageFileType::write: Image has non float data type!\n"));
        return false;
    }

    OSGImfThreadInit::init();

    const unsigned int components = image->getComponents();

    try
    {
        int64_t width  = image->getWidth();
        int64_t height = image->getHeight();

        const char *dummy = "";
        StdOStream file(os, dummy);
        Imf::Header header(width, height);

        // now add custom attributes
        ImageGenericAttPtr att = ImageGenericAttPtr::dcast(
            image->findAttachment(ImageGenericAtt::getClassType().getGroupId()));

        if(att != NullFC)
        {
            FieldContainerType  &fcType = att->getType();
            int64_t count = att->getType().getNumFieldDescs();

            for(int64_t i=1;i<=count;++i)
            {
                FieldDescription *fDesc = fcType.getFieldDescription(i);
                Field *field = att->getField(i);
                if(fDesc != NULL && field != NULL)
                {
                    SFString *strField = dynamic_cast<SFString*>(field);
                    if(strField != NULL)
                    {
                        //printf("key: '%s' value: '%s'\n", fDesc->getCName(), strField->getValue().c_str());
                        
                        Imf::StringAttribute imfAttr(strField->getValue().c_str());
                        header.insert(fDesc->getCName(), imfAttr);
                    }
                }
            }
        }

        const std::string *comment = image->findAttachmentField("Comment");
        if(comment != NULL)
        {
            Imf::StringAttribute imfAttr(comment->c_str());
            header.insert("Comment", imfAttr);
        }
    
        
        
        Imf::Chromaticities bt709( Imath::V2f(0.6400, 0.3300), Imath::V2f (0.3000, 0.6000), Imath::V2f (0.1500, 0.0600), Imath::V2f (0.3127, 0.3290));
        Imf::ChromaticitiesAttribute fileChroma;
        if(image->getMFChromaticities()->size() == 4)
        {
            const osg::MFVec2f* imgChroma = image->getMFChromaticities();
            fileChroma = Imf::Chromaticities( Imath::V2f((*imgChroma)[0][0], (*imgChroma)[0][1]), Imath::V2f ((*imgChroma)[1][0], (*imgChroma)[1][1]), Imath::V2f ((*imgChroma)[2][0], (*imgChroma)[2][1]), Imath::V2f ((*imgChroma)[3][0], (*imgChroma)[3][1]));
        }
        else
        {
            fileChroma = bt709;
            
        }
        header.insert(Imf::ChromaticitiesAttribute::staticTypeName(), fileChroma);
        // we write each side as 4 channels out
        // side 0 RGBA
        // side 1 R1G1B1A1
        // ...
        std::vector<std::string> channelNames;
        if (_layers.size() == 0)
        {
            std::string componentNames[4] = { "R", "G", "B", "A" };
            if(image->getPixelFormat() == Image::OSG_L_PF)
                componentNames[0] = "Y";
            else if (image->getPixelFormat() == Image::OSG_LA_PF)
            {
                componentNames[0] = "Y";
                componentNames[1] = "A";
            }
            Imf::PixelType pixelType = (image->getDataType() == Image::OSG_FLOAT32_IMAGEDATA) ? Imf::FLOAT : Imf::HALF;
            for (size_t cIdx = 0; cIdx < components; ++cIdx)
            {
                channelNames.push_back(componentNames[cIdx]);
                header.channels().insert(componentNames[cIdx].c_str(), Imf::Channel(pixelType));
            }
            for (int64_t side = 1; side < image->getSideCount(); ++side)
            {
                const std::string sideName = std::to_string(side);
                for (size_t cIdx = 0; cIdx < components; ++cIdx)
                {
                    std::string componentName = sideName + "." + componentNames[cIdx];
                    channelNames.push_back(componentName);
                    header.channels().insert(componentName.c_str(), Imf::Channel(pixelType));
                }
            }
 
        }
        else
        {
            for (size_t layerIdx = 0; layerIdx < _layers.size(); ++layerIdx)
            {
                std::string currentLayerName = _layernames[layerIdx];
                ImagePtr currentImg = _layers[layerIdx];
                Imf::PixelType pixelType = (currentImg->getDataType() == Image::OSG_FLOAT32_IMAGEDATA) ? Imf::FLOAT : Imf::HALF;
                std::string componentNames[4] = { "R", "G", "B", "A" };
                if( currentImg->getPixelFormat() == Image::OSG_L_PF)
                    componentNames[0] = "Y";
                else if( currentImg->getPixelFormat() == Image::OSG_LA_PF)
                {
                    componentNames[0] = "Y";
                    componentNames[1] = "A";
                }
                if (currentLayerName == "beauty")
                {
                    for (size_t cIdx = 0; cIdx < components; ++cIdx)
                    {
                        channelNames.push_back(componentNames[cIdx]);
                        header.channels().insert(componentNames[cIdx].c_str(), Imf::Channel(pixelType));
                    }

                    for (int64_t side = 1; side < image->getSideCount(); ++side)
                    {
                        const std::string sideName = std::to_string(side);
                        for (size_t cIdx = 0; cIdx < components; ++cIdx)
                        {
                            std::string componentName = sideName + "." + componentNames[cIdx];
                            channelNames.push_back(componentName);
                            header.channels().insert(componentName.c_str(), Imf::Channel(pixelType));
                        }

                    }
                }
                else
                {
                    for (size_t cIdx = 0; cIdx < components; ++cIdx)
                    {
                        std::string componentName = currentLayerName + "." + componentNames[cIdx];
                        channelNames.push_back(componentName);
                        header.channels().insert(componentName.c_str(), Imf::Channel(pixelType));
                    }

                    for (int64_t side = 1; side < image->getSideCount(); ++side)
                    {
                        const std::string sideName = std::to_string(side);
                        for (size_t cIdx = 0; cIdx < components; ++cIdx)
                        {
                            std::string componentName = currentLayerName + sideName + "." + componentNames[cIdx];
                            channelNames.push_back(componentName);
                            header.channels().insert(componentName.c_str(), Imf::Channel(pixelType));
                        }
                    }
                }
            }
        }
        // enable ZIP-COMPRESSION when writing files
        header.compression() = Imf::ZIP_COMPRESSION;

        
        
        //header.setName (ss.str());
       // header.setType (Imf::SCANLINEIMAGE);
        //header.insert(TimeCodeAttribute::staticTypeName(), ta);
       

        Imf::OutputFile stream(file, header);
        Imf::FrameBuffer frame_buffer;

        const int64_t c = components;
    
        if (_layers.size() == 0)
        {
            Imf::PixelType pixelType = (image->getDataType() == Image::OSG_FLOAT32_IMAGEDATA) ? Imf::FLOAT : Imf::HALF;
            size_t dataTypeSize = (image->getDataType() == Image::OSG_FLOAT32_IMAGEDATA) ? sizeof(Real32) : sizeof(Real16);
            size_t dataTypeSizeAllChannels = dataTypeSize * c;
            // we need to do a vertical flip so we write single scan lines out.
            for (int i = height - 1; i >= 0; --i)
            {
                unsigned int nameIdx = 0;
                for (int64_t side = 0; side < image->getSideCount(); ++side)
                {
                    const char *data = (reinterpret_cast<const char *>(image->getData(0, 0, side))) + i * (dataTypeSizeAllChannels * width);
                    data -= stream.currentScanLine() * (dataTypeSizeAllChannels * width);
                    frame_buffer.insert(channelNames[nameIdx++].c_str(), Imf::Slice(pixelType, const_cast<char *>(data), dataTypeSizeAllChannels, dataTypeSizeAllChannels * width));
                    if (components >= 2)
                        frame_buffer.insert(channelNames[nameIdx++].c_str(), Imf::Slice(pixelType, const_cast<char *>(data)+1 * dataTypeSize, dataTypeSizeAllChannels, dataTypeSizeAllChannels * width));
                    if (components >= 3)
                        frame_buffer.insert(channelNames[nameIdx++].c_str(), Imf::Slice(pixelType, const_cast<char *>(data)+2 * dataTypeSize, dataTypeSizeAllChannels, dataTypeSizeAllChannels * width));
                    if (components >= 4)
                        frame_buffer.insert(channelNames[nameIdx++].c_str(), Imf::Slice(pixelType, const_cast<char *>(data)+3 * dataTypeSize, dataTypeSizeAllChannels, dataTypeSizeAllChannels * width));
                }
                stream.setFrameBuffer(frame_buffer);
                stream.writePixels(1);
            }
        }
        else
        {
            
            // we need to do a vertical flip so we write single scan lines out.
            for (int i = height - 1; i >= 0; --i)
            {
                unsigned int nameIdx = 0;
                for (size_t layerIdx = 0; layerIdx < _layers.size(); ++layerIdx)
                {
                    ImagePtr currentImg = _layers[layerIdx];
                    Imf::PixelType pixelType = (currentImg->getDataType() == Image::OSG_FLOAT32_IMAGEDATA) ? Imf::FLOAT : Imf::HALF;
                    size_t dataTypeSize = (currentImg->getDataType() == Image::OSG_FLOAT32_IMAGEDATA) ? sizeof(Real32) : sizeof(Real16);
                    size_t dataTypeSizeAllChannels = dataTypeSize * c;
                    for (int64_t side = 0; side < currentImg->getSideCount(); ++side)
                    {
                        const char *data = (reinterpret_cast<const char *>(currentImg->getData(0, 0, side))) + i * (dataTypeSizeAllChannels * width);
                        data -= stream.currentScanLine() * (dataTypeSizeAllChannels * width);
                        frame_buffer.insert(channelNames[nameIdx++].c_str(), Imf::Slice(pixelType, const_cast<char *>(data), dataTypeSizeAllChannels, dataTypeSizeAllChannels * width));
                        if (components >= 2)
                            frame_buffer.insert(channelNames[nameIdx++].c_str(), Imf::Slice(pixelType, const_cast<char *>(data)+1 * dataTypeSize, dataTypeSizeAllChannels, dataTypeSizeAllChannels * width));
                        if (components >= 3)
                            frame_buffer.insert(channelNames[nameIdx++].c_str(), Imf::Slice(pixelType, const_cast<char *>(data)+2 * dataTypeSize, dataTypeSizeAllChannels, dataTypeSizeAllChannels * width));
                        if (components >= 4)
                            frame_buffer.insert(channelNames[nameIdx++].c_str(), Imf::Slice(pixelType, const_cast<char *>(data)+3 * dataTypeSize, dataTypeSizeAllChannels, dataTypeSizeAllChannels * width));
                    }
                }
                stream.setFrameBuffer(frame_buffer);
                stream.writePixels(1);
            }
        }

        return true;
    }
    catch(std::exception &e)
    {
        FFATAL(( "Error while trying to write OpenEXR Image from stream: %s\n",
                  e.what() ));
        return false;
    }

#else
    SWARNING << getMimeType()
             << " write is not compiled into the current binary "
             << std::endl;
    return false;
#endif
}

bool EXRImageFileType::write(const ImagePtr &image, std::ostream &os, const std::string &mimetype, std::vector<ImagePtr> images, const std::string layernames[])
{
    for (size_t i = 0; i < images.size(); i++)
    {
        if (images[i] != NullFC)
        {
            _layers.push_back(images[i]);
            _layernames.push_back(layernames[i]);
        }
    }

    printf("EXRImageFileType::write MultiLayer EXR with %u layers\n", (unsigned int)_layers.size());

    bool result = write(_layers[0], os, mimetype);

    _layers.clear();
    _layernames.clear();

    return result;
}
//-------------------------------------------------------------------------
/*!
    Tries to restore the image data from the given memblock.
    Returns the amount of data read.
*/
UInt64 EXRImageFileType::restoreData(      ImagePtr &image,
                                     const UChar8   *buffer,
                                           Int32   OSG_CHECK_ARG(memSize) )
{
    image->setData(buffer);

    return image->getSize();
}

//-------------------------------------------------------------------------
/*!
    Tries to store the image data to the given memblock.
    Returns the amount of data written.
*/
UInt64 EXRImageFileType::storeData(const ImagePtr &image,
                                         UChar8   *buffer,
                                         Int32     OSG_CHECK_ARG(memSize))
{
    uint64_t dataSize = image->getSize();
    const UChar8 *src = image->getData();

    if ( dataSize && src && buffer )
        memcpy( buffer, src, dataSize);

    return dataSize;
}

//-------------------------------------------------------------------------
/*!
    Constructor used for the singleton object
*/
EXRImageFileType::EXRImageFileType ( const Char8 *mimeType,
                                     const Char8 *suffixArray[],
                                     UInt16 suffixByteCount,
                                     UInt32 flags  )
    : ImageFileType ( mimeType, suffixArray, suffixByteCount, flags )
{
    FINFO(( "EXRImageFileType: %s\n", mimeType));
}

//-------------------------------------------------------------------------
/*!
    Destructor
*/
EXRImageFileType::~EXRImageFileType(void)
{
}

//-------------------------------------------------------------------------
/*!
    Quick test for checking if file format is correct
*/
bool EXRImageFileType::isOpenExrFile(std::istream &is)
{
#ifdef OSG_WITH_IMF
    char bytes[4];

    is.read(bytes, sizeof(bytes));

    return !!is && Imf::isImfMagic(bytes);

#else
    return false;
#endif
}

const unsigned int EXRImageFileType::getChannelIndexFromName(const std::string& name) const
{
    const char lastChar = name.back();
    if (lastChar == 'A')
        return 3;
    if (lastChar == 'B')
        return 2;
    if (lastChar == 'G')
        return 1;
    //R and Y should return 0
    return 0;
}
