/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright (C) 2000-2002 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

//-------------------------------
//      Includes
//-------------------------------

#include <math.h>
#include <memory.h>
#include <stdio.h>
#include <stdlib.h>

#include <OSGConfig.h>
#include <OSGBaseFunctions.h>

#include <fstream>
#include <iostream>

#include <OSGFileSystem.h>
#include <OSGImageFileHandler.h>
#include <OSGLog.h>
#include <OSGPathHandler.h>

#include "OSGHDRImageFileType.h"

OSG_USING_NAMESPACE

#define MINELEN 8      // minimum scanline length for encoding
#define MAXELEN 0x7fff // maximum scanline length for encoding
#define MINRUN 4       // minimum run length
#define RED 0
#define GRN 1
#define BLU 2
#define EXP 3
#define COLXS 128

// copy source -> dest
#define copy_rgbe(c1, c2)                                                                          \
    (c2[RED] = c1[RED], c2[GRN] = c1[GRN], c2[BLU] = c1[BLU], c2[EXP] = c1[EXP])

#pragma intrinsic(pow)

/*! \class osg::HDRImageFileType
    \ingroup GrpSystemImage

Image File Type to read/write and store/restore Image objects as
HDR data.

All the type specific code is included in the class. Does
not depend on external libs.

 */

/*****************************
 *   Types
 *****************************/
// Static Class Varible implementations:

static const Char8 *suffixArray[] = {"hdr"};

HDRImageFileType HDRImageFileType::_the("image/x-hdr", suffixArray, sizeof(suffixArray),
                                        OSG_READ_SUPPORTED | OSG_WRITE_SUPPORTED);

/*****************************
 *    Classvariables
 *****************************/

/********************************
 *    Class methodes
 *******************************/

//-------------------------------------------------------------------------
/*!
Class method to get the singleton Object
*/
HDRImageFileType &HDRImageFileType::the(void)
{
    return _the;
}

/*******************************
 *public
 *******************************/

//-------------------------------------------------------------------------
/*!
Tries to fill the image object with the data read from
the given input stream. Returns true on success.
*/
bool HDRImageFileType::read(ImagePtr &image, std::istream &is, const std::string &mimetype)
{
    int width, height;

    if(!checkHDR(is, width, height))
    {
        FWARNING(("No valid RADIANCE picture format\n"));
        return false;
    }
    const auto swidth = validateValue({width});
    const auto sheight = validateValue({height});
#if 1
    image->set(Image::OSG_RGB_PF, swidth, sheight, 1, 1, 1, 0.0, 0, Image::OSG_FLOAT32_IMAGEDATA);
    if(!image->isValid())
        return false;

    image->clear();
    bool ok = false;
    bool use16BitFloat = false;
    Real32 *data = (reinterpret_cast<Real32 *>(image->editData()));
    ok = radiance2fp( is, data, swidth, sheight, use16BitFloat);

    if( use16BitFloat)
    {
        image->convertDataTypeTo(Image::OSG_FLOAT16_IMAGEDATA);
    }
    return ok;
#else
    bool use16BitFloat = (_options.find("use16BitFloat=true") != std::string::npos);

    

    image->set(Image::OSG_RGB_PF, swidth, sheight, 1, 1, 1, 0.0, 0,
               use16BitFloat ? Image::OSG_FLOAT16_IMAGEDATA : Image::OSG_FLOAT32_IMAGEDATA);

    if(!image->isValid())
        return false;

    image->clear();

    if(use16BitFloat)
    {
        Real16 *data = (reinterpret_cast<Real16 *>(image->editData()));
        return radiance2fp(is, data, swidth, sheight);
    }

    Real32 *data = (reinterpret_cast<Real32 *>(image->editData()));
    return radiance2fp(is, data, swidth, sheight);
#endif
}

//-------------------------------------------------------------------------
/*!
Tries to write the image object to the given output stream.
Returns true on success.
*/
bool HDRImageFileType::write(const ImagePtr &image, std::ostream &os, const std::string &mimetype)
{
    if((image->getDataType() != Image::OSG_FLOAT32_IMAGEDATA) &&
       (image->getDataType() != Image::OSG_FLOAT16_IMAGEDATA))
    {
        FWARNING(("HDRImageFileType::write: Image has non float data type!\n"));
        return false;
    }

    if(!os.good())
        return false;

    int width = image->getWidth();
    int height = image->getHeight();

    os << "#?RADIANCE" << std::endl;
    const std::string *comment = image->findAttachmentField("Comment");
    if(comment != NULL)
        os << "# " << *comment << std::endl;
    else
        os << "# Written with OpenSG" << std::endl;

    os << "FORMAT=32-bit_rle_rgbe" << std::endl;
    os << "EXPOSURE=" << 1.0f << std::endl << std::endl;
    os << "-Y " << height << " +X " << width << std::endl;

    RGBE *rgbe_scan = new RGBE[width];

    if(image->getDataType() == Image::OSG_FLOAT32_IMAGEDATA)
    {
        const Real32 *data = (reinterpret_cast<const Real32 *>(image->getData()));

        // upside down !!!
        for(Int64 y = height - 1; y >= 0; y--)
        {
            if(fwritecolrs(os, &data[y * Int64(width * 3)], rgbe_scan, width, height) < 0)
            {
                delete[] rgbe_scan;
                return false;
            }
        }
    }
    else // OSG_FLOAT16_IMAGEDATA
    {
        const Real16 *data = (reinterpret_cast<const Real16 *>(image->getData()));

        // upside down !!!
        for(Int64 y = height - 1; y >= 0; y--)
        {
            if(fwritecolrs(os, &data[y * Int64(width * 3)], rgbe_scan, width, height) < 0)
            {
                delete[] rgbe_scan;
                return false;
            }
        }
    }

    delete[] rgbe_scan;
    return true;
}

//-------------------------------------------------------------------------
/*!
Tries to fill the image object with the data read from
the given fileName. Returns true on success.
*/
bool HDRImageFileType::read(ImagePtr &image, const Char8 *fileName)
{
    FILE *file = osg::fopen(fileName, "rb");

    if(file == NULL && ImageFileHandler::the().getPathHandler())
    {
        // Try to find the file in the search path
        PathHandler *ph = ImageFileHandler::the().getPathHandler();
        file = osg::fopen(ph->findFile(fileName).c_str(), "rb");
    }

    if(file == NULL)
        return false;

    int width, height;
    if(!checkHDR(file, width, height))
    {
        fclose(file);
        return false;
    }
    const auto swidth = validateValue({width});
    const auto sheight = validateValue({height});
#if 1
    image->set(Image::OSG_RGB_PF, swidth, sheight, 1, 1, 1, 0.0, 0, Image::OSG_FLOAT32_IMAGEDATA);
    if(!image->isValid())
        return false;

    image->clear();
    bool ok = false;
    bool use16BitFloat = false;
    Real32 *data = (reinterpret_cast<Real32 *>(image->editData()));
    ok = radiance2fp(file, data, swidth, sheight, use16BitFloat);

    if( use16BitFloat)
    {
        image->convertDataTypeTo(Image::OSG_FLOAT16_IMAGEDATA);
    }
#else

    bool use16BitFloat = (_options.find("use16BitFloat=true") != std::string::npos);

    image->set(Image::OSG_RGB_PF, swidth, sheight, 1, 1, 1, 0.0, 0,
               use16BitFloat ? Image::OSG_FLOAT16_IMAGEDATA : Image::OSG_FLOAT32_IMAGEDATA);

    if(!image->isValid())
        return false;

    image->clear();

    bool ok = false;

    if(use16BitFloat)
    {
        Real16 *data = (reinterpret_cast<Real16 *>(image->editData()));
        ok = radiance2fp(file, data, swidth, sheight);
    }
    else
    {
        Real32 *data = (reinterpret_cast<Real32 *>(image->editData()));
        ok = radiance2fp(file, data, swidth, sheight);
    }
#endif
    fclose(file);

    return ok;
}

//-------------------------------------------------------------------------
/*!
Tries to write the image object to the given fileName.
Returns true on success.
*/
bool HDRImageFileType::write(const ImagePtr &image, const Char8 *fileName)
{
    if((image->getDataType() != Image::OSG_FLOAT32_IMAGEDATA) &&
       (image->getDataType() != Image::OSG_FLOAT16_IMAGEDATA))
    {
        FWARNING(("HDRImageFileType::write: Image has non float data type!\n"));
        return false;
    }

    FILE *file = osg::fopen(fileName, "wb");

    if(file == NULL)
        return false;

    int width = image->getWidth();
    int height = image->getHeight();

    fprintf(file, "#?RADIANCE");
    fputc(10, file);
    const std::string *comment = image->findAttachmentField("Comment");
    if(comment != NULL)
        fprintf(file, "# %s", comment->c_str());
    else
        fprintf(file, "# %s", "Written with OpenSG");
    // skip, if comment ends with \n
    if (comment == nullptr || comment->empty() || comment->back() != '\n')
        fputc(10, file);
    fprintf(file, "FORMAT=32-bit_rle_rgbe");
    fputc(10, file);
    fprintf(file, "EXPOSURE=%25.13f", 1.0);
    fputc(10, file);
    fputc(10, file);
    fprintf(file, "-Y %d +X %d", height, width);
    fputc(10, file);

    RGBE *rgbe_scan = new RGBE[width];

    if(image->getDataType() == Image::OSG_FLOAT32_IMAGEDATA)
    {
        const Real32 *data = reinterpret_cast<const Real32 *>(image->getData());

        // upside down !!!
        for(Int64 y = height - 1; y >= 0; y--)
        {
            if(fwritecolrs(file, &data[y * Int64(width * 3)], rgbe_scan, width, height) < 0)
            {
                fclose(file);
                delete[] rgbe_scan;
                return false;
            }
        }
    }
    else // 16 bit floating point data
    {
        const Real16 *data = reinterpret_cast<const Real16 *>(image->getData());

        // upside down !!!
        for(Int64 y = height - 1; y >= 0; y--)
        {
            if(fwritecolrs(file, &data[y * Int64(width * 3)], rgbe_scan, width, height) < 0)
            {
                fclose(file);
                delete[] rgbe_scan;
                return false;
            }
        }
    }
    fclose(file);
    delete[] rgbe_scan;
    return true;
}

//-------------------------------------------------------------------------
/*!
Tries to restore the image data from the given memblock.
Returns the amount of data read.
*/
UInt64 HDRImageFileType::restoreData(ImagePtr &image, const UChar8 *buffer,
                                     Int32 OSG_CHECK_ARG(memSize))
{
    image->setData(buffer);

    return image->getSize();
}

//-------------------------------------------------------------------------
/*!
Tries to store the image data to the given memblock.
Returns the amount of data written.
*/
UInt64 HDRImageFileType::storeData(const ImagePtr &image, UChar8 *buffer,
                                   Int32 OSG_CHECK_ARG(memSize))
{
    UInt32 dataSize = image->getSize();
    const UChar8 *src = image->getData();

    if(dataSize && src && buffer)
        memcpy(buffer, src, dataSize);

    return dataSize;
}

//-------------------------------------------------------------------------
/*!
Constructor used for the singleton object
*/
HDRImageFileType::HDRImageFileType(const Char8 *mimeType, const Char8 *suffixArray[],
                                   UInt16 suffixByteCount, UInt32 flags)
    : ImageFileType(mimeType, suffixArray, suffixByteCount, flags)
{
    //   FLOG(( "HDRImageFileType: %s %d\n", mimeType, flags ));
}

//-------------------------------------------------------------------------
/*!
Destructor
*/
HDRImageFileType::~HDRImageFileType(void)
{
}

// check header to see if this is really a HDR file
// if so get the resolution information
bool HDRImageFileType::checkHDR(FILE *file, int &width, int &height)
{
    char cs[256], st1[80], st2[80];
    bool resok = false;
    bool HDRok = false;
    while(!feof(file) && !resok)
    {
        fgets(cs, 255, file);
        if(strstr(cs, "32-bit_rle_rgbe"))
            HDRok = true;
        if(strcmp(cs, "\n") == 0)
        {
            // empty line found, next is resolution info, format: -Y N +X N
            // directly followed by data
            fgets(cs, 255, file);
            sscanf(cs, "%79s %d %79s %d", st1, &height, st2, &width);
            resok = true;
        }
    }
    return HDRok;
}

// check header and get resolution (streaming type)
bool HDRImageFileType::checkHDR(std::istream &is, int &width, int &height)
{
    char cs[256], st1[80], st2[80];
    bool resok = false;
    bool HDRok = false;
    int i = 0;

    while(!is.eof() && !resok)
    {
        is.getline(cs, sizeof(cs) - 1);

        if(strstr(cs, "32-bit_rle_rgbe"))
            HDRok = true;

        if(HDRok && (cs[0] == '\r' || cs[0] == '\n' || cs[0] == '\0'))
        {
            // empty line found, next is resolution info, format: -Y N +X N
            // directly followed by data
            is.getline(cs, sizeof(cs) - 1);

            i = sscanf(cs, "%79s %d %79s %d", st1, &height, st2, &width);
            if(i == 4)
                resok = true;
        }
    }

    return HDRok;
}

// convert radiance hdr to float image
bool HDRImageFileType::radiance2fp(FILE *file, Real32 *data, int width, int height, bool& is16BitFloat)
{
    // VRED-9604 No maximal values for HDR width and height are known
    if(width <= 0 || height <= 0)
        return false;

    is16BitFloat = true;
    int x, y, yx;
    RGBE *sline = new RGBE[width];

    for(y = height - 1; y >= 0; y--)
    {
        yx = y * width;
        if(!freadcolrs(file, sline, width))
            return false;

        Real32 *fcol = &data[yx * 3];
        for(x = 0; x < width; x++)
        {
            RGBE2Float(sline[x], fcol);
            is16BitFloat &= ((fcol[RED] < HALF_MAX_VAL) && (fcol[GRN] < HALF_MAX_VAL) && (fcol[GRN] < HALF_MAX_VAL)) ? true : false;
            fcol += 3;
        }
    }

    delete[] sline;
    return true;
}

// convert radiance hdr to float image (streaming type)
bool HDRImageFileType::radiance2fp(std::istream &is, Real32 *data, int width, int height, bool& is16BitFloat)
{
    RGBE *sline = new RGBE[width];
    if(!sline)
        return false;

    is16BitFloat = true;
    for(int y = height - 1; y >= 0; y--)
    {
        if(!freadcolrs(is, sline, width))
            return false;

        int yx = y * width;
        if(yx >= 0)
        {        
            Real32 *fcol = &data[yx * 3];
            for(int x = 0; x < width; x++)
            {
                RGBE2Float(sline[x], fcol);
                is16BitFloat &= ((fcol[RED] < HALF_MAX_VAL) && (fcol[GRN] < HALF_MAX_VAL) && (fcol[GRN] < HALF_MAX_VAL)) ? true : false;
                fcol += 3;
            }
        }
    }
    delete[] sline;

    return true;
}

// convert radiance hdr to float image
bool HDRImageFileType::radiance2fp(FILE *file, Real16 *data, int width, int height)
{
    RGBE *sline = new RGBE[width];
    for(int y = height - 1; y >= 0; y--)
    {
        if(!freadcolrs(file, sline, width))
            return false;

        int yx = y * width;
        if(yx >= 0)
        {
            Real16 *fcol = &data[yx * 3];
            for(int x = 0; x < width; x++)
            {
                RGBE2Half(sline[x], fcol);
                fcol += 3;
            }
        }
    }

    delete[] sline;
    return true;
}

// convert radiance hdr to float image (streaming type)
bool HDRImageFileType::radiance2fp(std::istream &is, Real16 *data, int width, int height)
{
    RGBE *sline = new RGBE[width];
    if(!sline)
        return false;

    for(int y = height - 1; y >= 0; y--)
    {
        if(!freadcolrs(is, sline, width))
            return false;

        int yx = y * width;
        if(yx >= 0)
        {
            Real16 *fcol = &data[yx * 3];
            for(int x = 0; x < width; x++)
            {
                RGBE2Half(sline[x], fcol);
                fcol += 3;
            }
        }
    }

    delete[] sline;
    return true;
}

// read and possibly RLE decode a rgbe scanline
bool HDRImageFileType::freadcolrs(FILE *file, RGBE *scan, int width)
{
    if(width <= 0)
        return false;

    int i, j, code, size;

    if((width < MINELEN) | (width > MAXELEN))
        return oldreadcolrs(file, scan, width);

    i = validateValue({getc(file)});
    if(i == EOF)
        return false;

    if(i != 2)
    {
        ungetc(i, file);
        return oldreadcolrs(file, scan, width);
    }

    const auto g = validateValue({static_cast<unsigned char>(getc(file))});
    const auto b = validateValue({static_cast<unsigned char>(getc(file))});

    scan[0][GRN] = g;
    scan[0][BLU] = b;

    i = validateValue({getc(file)});
    if(i == EOF)
        return false;

    size = (int(scan[0][BLU])) << 8;
    if((size | i) != width)
        return false;

    for(i = 0; i < 4; i++)
    {
        for(j = 0; j < width;)
        {
            code = validateValue({getc(file)});
            if(code == EOF)
                return false;
            if(code > 128)
            {
                code &= 127;
                auto val = validateValue({static_cast<unsigned char>(getc(file))});
                while(code--)
                    scan[j++][i] = val;
            }
            else
            {
                // much faster than calling getc in the while loop!
                unsigned char buffer[129];
                fread(&buffer[0], 1, code, file);
                unsigned char *b = &buffer[0];
                while(code--)
                    scan[j++][i] = validateValue({*b++});
                    //scan[j++][i] = *b++;

                // while (code--)
                //    scan[j++][i] = static_cast<unsigned char>(getc(file));
            }
        }
    }
    return feof(file) ? false : true;
}

// read and decode a rgbe scanline (streaming type)
bool HDRImageFileType::freadcolrs(std::istream &is, RGBE *scan, int width)
{
    if(width <= 0)
        return false;

    int i, j, code, size;

    if((width < MINELEN) | (width > MAXELEN))
        return oldreadcolrs(is, scan, width);

    //byte = static_cast<unsigned char>(is.get());
    auto byte = validateValue({static_cast<unsigned char>(is.get())});

    if(is.eof())
        return false;

    if(byte != 2)
    {
        is.putback(byte);
        return oldreadcolrs(is, scan, width);
    }

    //byte = static_cast<unsigned char>(is.get());
    byte = validateValue({static_cast<unsigned char>(is.get())});
    scan[0][GRN] = byte;

    //byte = static_cast<unsigned char>(is.get());
    byte = validateValue({static_cast<unsigned char>(is.get())});
    scan[0][BLU] = byte;

    size = (int(scan[0][BLU])) << 8;
    i = is.get();

    if((size | i) != width)
        return false;

    for(i = 0; i < 4; i++)
    {
        for(j = 0; j < width;)
        {
            if(is.eof())
                return false;

            code = is.get();

            if(code > 128)
            {
                code &= 127;
                //val = is.get();
                const auto val = validateValue({static_cast<unsigned char>(is.get())});
                while(code--)
                    scan[j++][i] = val;
            }
            else
            {
                while(code--)
                {
                    const auto val = validateValue({static_cast<unsigned char>(is.get())});
                    scan[j++][i] = val;
                }
            }
        }
    }

    return is.eof() ? false : true;
}

// old format
bool HDRImageFileType::oldreadcolrs(FILE *file, RGBE *scan, int width)
{
    if(width <= 0)
        return false;

    int i, rshift = 0, len = width;
    while(len > 0)
    {
        const auto r = validateValue({static_cast<unsigned char>(getc(file))});
        const auto g = validateValue({static_cast<unsigned char>(getc(file))});
        const auto b = validateValue({static_cast<unsigned char>(getc(file))});
        const auto e = validateValue({static_cast<unsigned char>(getc(file))});

        scan[0][RED] = r;
        scan[0][GRN] = g;
        scan[0][BLU] = b;
        scan[0][EXP] = e;
        if(feof(file) || ferror(file))
            return false;
        if(scan[0][RED] == 1 && scan[0][GRN] == 1 && scan[0][BLU] == 1)
        {
            for(i = scan[0][EXP] << rshift; i > 0; i--)
            {
                copy_rgbe(scan[-1], scan[0]);
                scan++;
                len--;
            }
            rshift += 8;
        }
        else
        {
            scan++;
            len--;
            rshift = 0;
        }
    }
    return true;
}

bool HDRImageFileType::oldreadcolrs(std::istream &is, RGBE *scan, int width)
{
    if(width <= 0)
        return false;

    unsigned char byte;
    int i, rshift = 0, len = width;
    while(len > 0)
    {
        scan[0][RED] = static_cast<unsigned char>(is.get());
        if(is.eof())
            return false;
        scan[0][GRN] = static_cast<unsigned char>(is.get());
        if(is.eof())
            return false;
        scan[0][BLU] = static_cast<unsigned char>(is.get());
        if(is.eof())
            return false;
        scan[0][EXP] = static_cast<unsigned char>(is.get());
        if(is.eof())
            return false;
        if(scan[0][RED] == 1 && scan[0][GRN] == 1 && scan[0][BLU] == 1)
        {
            for(i = scan[0][EXP] << rshift; i > 0; i--)
            {
                copy_rgbe(scan[-1], scan[0]);
                scan++;
                len--;
            }
            rshift += 8;
        }
        else
        {
            scan++;
            len--;
            rshift = 0;
        }
    }
    return true;
}

// rgbe -> float color
void HDRImageFileType::RGBE2Float(RGBE rgbe, Real32 *fcol)
{
    if(rgbe[EXP] == 0)
    {
        *(fcol + RED) = *(fcol + GRN) = *(fcol + BLU) = 0;
    }
    else
    {
        // Real32 f = ldexp(1., rgbe[EXP]-(COLXS+8));
        // much faster with special intrinsics pow.
        Real32 f = powf(2.f, rgbe[EXP] - (COLXS + 8));
        *(fcol + RED) = (rgbe[RED] + .5f) * f;
        *(fcol + GRN) = (rgbe[GRN] + .5f) * f;
        *(fcol + BLU) = (rgbe[BLU] + .5f) * f;
    }
}

// rgbe -> float color
void HDRImageFileType::RGBE2Half(RGBE rgbe, Real16 *fcol)
{
    if(rgbe[EXP] == 0)
    {
        *(fcol + RED) = *(fcol + GRN) = *(fcol + BLU) = 0;
    }
    else
    {
        // Real32 f = ldexp(1., rgbe[EXP]-(COLXS+8));
        // much faster with special intrinsics pow.
        Real32 f = powf(2.f, rgbe[EXP] - (COLXS + 8));
        *(fcol + RED) = Real16(std::min(HALF_MAX_VAL, (rgbe[RED] + .5f) * f));
        *(fcol + GRN) = Real16(std::min(HALF_MAX_VAL, (rgbe[GRN] + .5f) * f));
        *(fcol + BLU) = Real16(std::min(HALF_MAX_VAL, (rgbe[BLU] + .5f) * f));
    }
}

int HDRImageFileType::fwritecolrs(FILE *file, const Real32 *scan, RGBE *rgbe_scan, int width,
                                  int height)
{
    // convert scanline
    for(unsigned int i = 0; i < width; ++i)
    {
        float2RGBE(scan, rgbe_scan[i]);
        scan += 3;
    }
    // write the RGBE Data to a file
    return fwriteRGBE(file, rgbe_scan, width, height);
}

int HDRImageFileType::fwritecolrs(FILE *file, const Real16 *scan, RGBE *rgbe_scan, int width,
                                  int height)
{
    // convert scanline
    for(unsigned int i = 0; i < width; ++i)
    {
        half2RGBE(scan, rgbe_scan[i]);
        scan += 3;
    }
    // write the RGBE Data to a file
    return fwriteRGBE(file, rgbe_scan, width, height);
}

int HDRImageFileType::fwriteRGBE(FILE *file, RGBE *rgbe_scan, int width, int height)
{
    int i, j, beg, c2, cnt = 0;
    if((width < MINELEN) | (width > MAXELEN)) // OOBs, write out flat
        return (fwrite(reinterpret_cast<char *>(rgbe_scan), sizeof(RGBE), width, file) - width);

    // put magic header
    putc(2, file);
    putc(2, file);
    putc(static_cast<unsigned char>(width >> 8), file);
    putc(static_cast<unsigned char>(width & 255), file);

    // put components seperately
    for(i = 0; i < 4; i++)
    {
        for(j = 0; j < width; j += cnt)
        { // find next run
            for(beg = j; beg < width; beg += cnt)
            {
                for(cnt = 1; (cnt < 127) && ((beg + cnt) < width) &&
                             (rgbe_scan[beg + cnt][i] == rgbe_scan[beg][i]);
                    cnt++)
                    ;
                if(cnt >= MINRUN)
                    break; // long enough
            }
            if(((beg - j) > 1) && ((beg - j) < MINRUN))
            {
                c2 = j + 1;
                while(rgbe_scan[c2++][i] == rgbe_scan[j][i])
                    if(c2 == beg)
                    { // short run
                        putc(static_cast<unsigned char>(128 + beg - j), file);
                        putc(static_cast<unsigned char>(rgbe_scan[j][i]), file);
                        j = beg;
                        break;
                    }
            }
            while(j < beg)
            { // write out non-run
                if((c2 = beg - j) > 128)
                    c2 = 128;
                putc(static_cast<unsigned char>(c2), file);
                while(c2--)
                    putc(rgbe_scan[j++][i], file);
            }
            if(cnt >= MINRUN)
            { // write out run
                putc(static_cast<unsigned char>(128 + cnt), file);
                putc(rgbe_scan[beg][i], file);
            }
            else
            {
                cnt = 0;
            }
        }
    }
    return (ferror(file) ? -1 : 0);
}

int HDRImageFileType::fwritecolrs(std::ostream &os, const Real32 *scan, RGBE *rgbe_scan, int width,
                                  int height)
{
    // convert scanline
    for(unsigned int i = 0; i < width; i++)
    {
        float2RGBE(scan, rgbe_scan[i]);
        scan += 3;
    }

    return fwriteRGBE(os, rgbe_scan, width, height);
}

int HDRImageFileType::fwritecolrs(std::ostream &os, const Real16 *scan, RGBE *rgbe_scan, int width,
                                  int height)
{
    // convert scanline
    for(unsigned int i = 0; i < width; i++)
    {
        half2RGBE(scan, rgbe_scan[i]);
        scan += 3;
    }

    return fwriteRGBE(os, rgbe_scan, width, height);
}

int HDRImageFileType::fwriteRGBE(std::ostream &os, RGBE *rgbe_scan, int width, int height)
{
    int i, j, beg, c2, cnt = 0;

    if((width < MINELEN) | (width > MAXELEN))
    {
        // OOBs, write out flat
        os.write(reinterpret_cast<char *>(rgbe_scan), width);
        return 0;
    }

    // put magic header
    os << static_cast<unsigned char>(2);
    os << static_cast<unsigned char>(2);
    os << static_cast<unsigned char>(width >> 8);
    os << static_cast<unsigned char>(width & 255);

    // put components seperately
    for(i = 0; i < 4; i++)
    {
        for(j = 0; j < width; j += cnt)
        {
            // find next run
            for(beg = j; beg < width; beg += cnt)
            {
                for(cnt = 1; (cnt < 127) && ((beg + cnt) < width) &&
                             (rgbe_scan[beg + cnt][i] == rgbe_scan[beg][i]);
                    cnt++)
                    ;
                if(cnt >= MINRUN)
                    break;
                // long enough
            }
            if(((beg - j) > 1) && ((beg - j) < MINRUN))
            {
                c2 = j + 1;
                while(rgbe_scan[c2++][i] == rgbe_scan[j][i])
                {
                    if(c2 == beg)
                    {
                        // short run
                        os << static_cast<unsigned char>(128 + beg - j);
                        os << static_cast<unsigned char>(rgbe_scan[j][i]);
                        j = beg;
                        break;
                    }
                }
            }
            while(j < beg)
            {
                // write out non-run
                if((c2 = beg - j) > 128)
                    c2 = 128;
                os << static_cast<unsigned char>(c2);

                while(c2--)
                    os << rgbe_scan[j++][i];
            }
            if(cnt >= MINRUN)
            {
                // write out run
                os << static_cast<unsigned char>(128 + cnt);
                os << rgbe_scan[beg][i];
            }
            else
            {
                cnt = 0;
            }
        }
    }
    return (os.fail() ? -1 : 0);
}

// float color -> rgbe
void HDRImageFileType::float2RGBE(const Real32 *fcol, RGBE rgbe)
{
    Real32 d = (*(fcol + RED) > *(fcol + GRN)) ? *(fcol + RED) : *(fcol + GRN);

    if(*(fcol + BLU) > d)
        d = *(fcol + BLU);
    if(d <= 1e-32f)
    {
        rgbe[RED] = rgbe[GRN] = rgbe[BLU] = rgbe[EXP] = 0;
    }
    else
    {
        int e;
        d = frexp(d, &e) * 256.f / d;
        rgbe[RED] = static_cast<unsigned char>(*(fcol + RED) * d);
        rgbe[GRN] = static_cast<unsigned char>(*(fcol + GRN) * d);
        rgbe[BLU] = static_cast<unsigned char>(*(fcol + BLU) * d);
        rgbe[EXP] = static_cast<unsigned char>(e + COLXS);
    }
}

// half color -> rgbe
void HDRImageFileType::half2RGBE(const Real16 *fcol, RGBE rgbe)
{
    Real32 d = (*(fcol + RED) > *(fcol + GRN)) ? *(fcol + RED) : *(fcol + GRN);

    if(*(fcol + BLU) > d)
        d = *(fcol + BLU);
    if(d <= 1e-32f)
    {
        rgbe[RED] = rgbe[GRN] = rgbe[BLU] = rgbe[EXP] = 0;
    }
    else
    {
        int e;
        d = frexp(d, &e) * 256.f / d;
        rgbe[RED] = static_cast<unsigned char>(*(fcol + RED) * d);
        rgbe[GRN] = static_cast<unsigned char>(*(fcol + GRN) * d);
        rgbe[BLU] = static_cast<unsigned char>(*(fcol + BLU) * d);
        rgbe[EXP] = static_cast<unsigned char>(e + COLXS);
    }
}
