/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *               Copyright (C) 2000-2002 by the OpenSG Forum                 *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

#define OSG_COMPILEIMAGE

#include <stdlib.h>
#include <stdio.h>

#include <algorithm>
#include <limits>
#include <cctype>
#include <sstream>
#include <functional>
#include <chrono>
#include <filesystem>

#include <vrZipper.h>

#include <oneapi/tbb/tick_count.h>
#include <oneapi/tbb/parallel_for.h>
#include <oneapi/tbb/combinable.h>

#include <OSGConfig.h>
#include <OSGLog.h>
#include <OSGBinaryChecksumHandler.h>
#include <OSGImageGenericAtt.h>
#include <OSGFieldContainerFields.h>
#include <OSGFileSystem.h>
#include <OSGTextureChunk.h>
#include <OSGExternalLog.h>

#include "OSGImageFileHandler.h"
#include "OSGPathHandler.h"
#include "OSGSceneFileHandler.h"

#include "OSGImageCompressor.h"

#include "OSGImage.h"

#ifdef WIN32

#pragma intrinsic(_BitScanForward)
inline int bitscanForward(unsigned long& idx, unsigned long mask)
{
    return _BitScanForward(&idx, mask);
}
#else
unsigned long bitscanForward(unsigned long& idx, unsigned long mask)
{
    idx = __builtin_ctzl(mask);
    return mask;
}
#endif

//#define VR_USE_IMAGE_SEQUENCE_CACHE

OSG_USING_NAMESPACE

/*! \class osg::Image
1D/2D/3D Image with various pixel types data, can also optional hold mipMap and simple multi-frame data.
*/

/*------------------------------------------------------------------------*/
/*                              static member                             */

/*! Static dictionary to map pixelData values to the bytes per pixel (bpp) value.
    Internaly used in the createData() method.
*/
Int32 Image::_formatDic[][2] =
{
    { OSG_A_PF, 1 },
    { OSG_I_PF, 1 },
    { OSG_L_PF, 1 },
    { OSG_LA_PF, 2 },
    { OSG_RGB_PF, 3 },
    { OSG_RGBA_PF, 4 },
    { OSG_BGR_PF, 3 },
    { OSG_BGRA_PF, 4 },
    { OSG_RGB9E5_PF, 3 },
    { OSG_RGB_DXT1_PF, 3},
    { OSG_RGBA_DXT1_PF, 4},
    { OSG_RGBA_DXT3_PF, 4},
    { OSG_RGBA_DXT5_PF, 4},
    { OSG_RGB_BC7_PF, 3},
    { OSG_RGBA_BC7_PF, 4},
    { OSG_RGB_BC6U_PF, 3},
    { OSG_RGB_BC6S_PF, 3},
    { OSG_RGB_ASTC_PF, 3},
    { OSG_RGBA_ASTC_PF, 4},
    { OSG_ALPHA_INTEGER_PF, 1},
    { OSG_RGB_INTEGER_PF, 3},
    { OSG_RGBA_INTEGER_PF, 4},
    { OSG_BGR_INTEGER_PF, 3},
    { OSG_BGRA_INTEGER_PF, 4},
    { OSG_LUMINANCE_INTEGER_PF, 1},
    { OSG_LUMINANCE_ALPHA_INTEGER_PF, 2}
};

Int32 Image::_typeDic[][2] =
{
    { OSG_INVALID_IMAGEDATATYPE, 0 },
    { OSG_UINT8_IMAGEDATA, 1 },
    { OSG_UINT16_IMAGEDATA, 2 },
    { OSG_UINT32_IMAGEDATA, 4 },
    { OSG_FLOAT32_IMAGEDATA, 4 },
    { OSG_FLOAT16_IMAGEDATA, 2 },
    { OSG_INT16_IMAGEDATA, 2 },
    { OSG_INT32_IMAGEDATA, 4 }
};

/*----------------------------- class specific ----------------------------*/

void Image::initMethod(void)
{
}

/*! Inform parents, when image was changed
 */
void Image::changed(BitVector whichField, UInt32 origin)
{
    MFFieldContainerPtr::iterator parentsIt = _mfParents.begin();
    MFFieldContainerPtr::iterator parentsEnd = _mfParents.end();

    while (parentsIt != parentsEnd)
    {
        // for TextureChunks we want to be careful whether a reinit or a
        // refresh is triggered
        TextureChunkPtr texParent = TextureChunkPtr::dcast(*parentsIt);

        if ((texParent != NullFC) && ((whichField & ~(PixelFieldMask)) == 0))
        {
            texParent->imageContentChanged();
        }
        else
        {
            // call the generic change method
            if ((*parentsIt) != NullFC)
            {
                (*parentsIt)->changed(
                    TypeTraits<BitVector>::One << parentsIt->getParentFieldPos(),
                    ChangedOrigin::Child);
            }
        }
        ++parentsIt;
    }

    // Update internals
    Int32 mapSizeType = sizeof(_typeDic) / sizeof(UInt32[2]);
    UInt32 typeFormat = 0;
    Int32 i;
    for (i = 0; i < mapSizeType; i++)
    {
        if (_typeDic[i][0] == getDataType())
            typeFormat = _typeDic[i][1];
    }

    // component size, side size and frame size also need to be transfered properly by using beginEdit/endEdit.
    // By doing the if check, we avoid the infinite recursion due to beginEdit/endEdit pair inside a changed() call.
    if (whichField & (WidthFieldMask | HeightFieldMask | DepthFieldMask | DataTypeFieldMask | PixelFormatFieldMask | BppFieldMask | MipMapCountFieldMask | SideCountFieldMask))
    {
        ImagePtr temp(*this);
        beginEditCP(temp, ComponentSizeFieldMask | SideSizeFieldMask | FrameSizeFieldMask);
        setComponentSize(typeFormat);
        setSideSize(calcMipmapSumSize(getMipMapCount(), true));
        setFrameSize(getSideSize() * getSideCount());
        setRawSideSize(calcMipmapSumSize(getMipMapCount()));
        setRawFrameSize(getRawSideSize() * getSideCount());
        calcMipmapOffsets();
        endEditCP(temp, ComponentSizeFieldMask | SideSizeFieldMask | FrameSizeFieldMask);
    }

    if((whichField & ~(PixelFieldMask)) == 0)
    {
        std::scoped_lock lock(_uncompressedDataMutex);
        _uncompressedDataDirty = true;
    }

    Inherited::changed(whichField, origin);
}

/*----------------------------- output ------------------------------------*/

void Image::dump(UInt32,
    const BitVector) const
{
    const char  *pfStr = "UNDEF_PIXEL_FORMAT";
    const char  *typeStr = "INVALID_IMAGEDATA_TYPE";

    switch (getInternalPixelFormat())
    {
    case OSG_A_PF:
        pfStr = "ALPHA";
        break;
    case OSG_I_PF:
        pfStr = "INTENSITY";
        break;
    case OSG_L_PF:
        pfStr = "LUMINANCE";
        break;
    case OSG_LA_PF:
        pfStr = "LUMINANCE_ALPHA";
        break;
    case OSG_BGR_PF:
        pfStr = "BGR";
        break;
    case OSG_BGRA_PF:
        pfStr = "BGRA";
        break;
    case OSG_RGB_PF:
        pfStr = "RGB";
        break;
    case OSG_RGBA_PF:
        pfStr = "RGBA";
        break;
    case OSG_RGB9E5_PF:
        pfStr = "RGB9E5";
        break;
    case OSG_RGB_DXT1_PF:
        pfStr = "RGB_DXT1";
        break;
    case OSG_RGBA_DXT1_PF:
        pfStr = "RGBA_DXT1";
        break;
    case OSG_RGBA_DXT3_PF:
        pfStr = "RGBA_DXT3";
        break;
    case OSG_RGBA_DXT5_PF:
        pfStr = "RGBA_DXT5";
        break;
    case OSG_RGB_BC7_PF:
        pfStr = "RGB_BC7";
        break;
    case OSG_RGBA_BC7_PF:
        pfStr = "RGBA_BC7";
        break;
    case  OSG_RGB_BC6U_PF:
        pfStr = "RGB_BC6SU";
        break;
    case  OSG_RGB_BC6S_PF:
        pfStr = "RGB_BC6S";
        break;
    case OSG_RGB_ASTC_PF:
        pfStr = "RGB_ASTC";
        break;
    case OSG_RGBA_ASTC_PF:
        pfStr = "RGBA_ASTC";
        break;
    case OSG_ALPHA_INTEGER_PF:
        pfStr = "ALPHA_INTEGER";
        break;
    case OSG_RGB_INTEGER_PF:
        pfStr = "RGB_INTEGER";
        break;
    case OSG_RGBA_INTEGER_PF:
        pfStr = "RGBA_INTEGER";
        break;
    case OSG_BGR_INTEGER_PF:
        pfStr = "BGR_INTEGER";
        break;
    case OSG_BGRA_INTEGER_PF:
        pfStr = "BGRA_INTEGER";
        break;
    case OSG_LUMINANCE_INTEGER_PF:
        pfStr = "LUMINANCE_INTEGER";
        break;
    case OSG_LUMINANCE_ALPHA_INTEGER_PF:
        pfStr = "LUMINANCE_ALPHA_INTEGER";
        break;
    default:
        pfStr = "UNKNOWN_PIXEL_FORMAT";
        break;
    };

    switch (getDataType())
    {
    case OSG_UINT8_IMAGEDATA:
        typeStr = "IMAGEDATA_TYPE UCHAR8";
        break;
    case OSG_UINT16_IMAGEDATA:
        typeStr = "IMAGEDATA_TYPE UCHAR16";
        break;
    case OSG_UINT32_IMAGEDATA:
        typeStr = "IMAGEDATA_TYPE UCHAR32";
        break;
    case OSG_FLOAT16_IMAGEDATA:
        typeStr = "IMAGEDATA_TYPE FLOAT16";
        break;
    case OSG_FLOAT32_IMAGEDATA:
        typeStr = "IMAGEDATA_TYPE FLOAT32";
        break;
    case OSG_INT16_IMAGEDATA:
        typeStr = "IMAGEDATA_TYPE INT16";
        break;
    case OSG_INT32_IMAGEDATA:
        typeStr = "IMAGEDATA_TYPE INT32";
        break;

    default:
        typeStr = "UNKNOWN_IMAGEDATA_TYPE";
        break;
    };

    FLOG(("ImageDump: %s; %d/%d/%d; #mm: %d, #side: %d, #frame: %d, frameDelay %g, dataType %s, size: %zu, rawSize: %zu\n",
        pfStr, getWidth(),
        getHeight(), getDepth(),
        getMipMapCount(),
        getSideCount(),
        getFrameCount(), getFrameDelay(),
        typeStr,
        getSize(),
        getRawSize()
        ));
}

bool osg::Image::arePixelsEqual(const osg::ImagePtr &img1, const osg::ImagePtr &img2, const int mipmapLevel)
{
    if (img1 == NullFC || img2 == NullFC)
        return false;

    Int32 frameCount = img1->getFrameCount();
    Int32 sideCount = img1->getSideCount();
    if (frameCount != img2->getFrameCount())
        return false;
    if (sideCount != img2->getSideCount())
        return false;

    // Size of mipmap level 
    const unsigned long size1 = img1->calcMipmapLevelSize(mipmapLevel);
    const unsigned long size2 = img2->calcMipmapLevelSize(mipmapLevel);
    if (size1 != size2)
        return false;

    for (Int32 frame = 0; frame < frameCount; ++frame)
    {
        for (Int32 side = 0; side < sideCount; ++side)
        {
            const UInt8* data1 = img1->getRawData(mipmapLevel, frame, side);
            const UInt8* data2 = img2->getRawData(mipmapLevel, frame, side);
            if (data1 == NULL || data2 == NULL)
                return false;

            if (memcmp(data1, data2, size1) != 0)
            {
                return false;
            }
        }
    }

    return true;
}

bool osg::Image::isEqual(const ImagePtr &image1, const ImagePtr &image2, NameComparisonOptions nameOptions, const bool compareMipmaps)
{
    if (image1 == image2)
    {
        return true;
    }

    if (image1 == osg::NullFC)
    {
        return false;
    }

    bool alwaysConsiderWebEngineImagesDifferent = true;
    return image1->equals(image2, alwaysConsiderWebEngineImagesDifferent, nameOptions, compareMipmaps);
}

bool osg::Image::equals(const FieldContainerPtr & fc,
    const FieldContainer::IsEqualOptions &options,
    std::map<osg::FieldContainerPtr, std::set<osg::FieldContainerPtr> > &equalFCs) const
{
    // by default we consider images that are created by web engines different.
    // this way they wont be merged, and will always be handled separately, see VRED-7914, VRED-8191 
    bool alwaysConsiderWebEngineImagesDifferent = true;
    // By default, we do not compare the mipmaps since those are normally created dynamically.
    // One exception e.g. is the prefiltered environment image.
    bool compareMipmaps = false;
    return equals(osg::ImagePtr::dcast(fc), alwaysConsiderWebEngineImagesDifferent, OSG_NAMECMP_IGNORE, compareMipmaps);
}

static bool isNameEquivalent(const std::string &name, const std::string &otherName, osg::Image::NameComparisonOptions nameOptions)
{
    // check if names should be compared at all
    if (nameOptions != osg::Image::OSG_NAMECMP_IGNORE)
    {
        // parse the given names as file path
        const std::filesystem::path thisPath(name);
        const std::filesystem::path otherPath(otherName);

        // do full path comparison
        if (nameOptions == osg::Image::OSG_NAMECMP_FILEPATH)
        {
            return (thisPath == otherPath);
        }

        // do file name only comparison
        if (nameOptions == osg::Image::OSG_NAMECMP_FILENAME)
        {
            return (thisPath.filename() == otherPath.filename());
        }
    }

    // no comparison: treat all names as equal
    return true;
}

bool osg::Image::equals(const ImagePtr & otherImage, bool alwaysConsiderWebEngineImagesDifferent, NameComparisonOptions nameOptions, const bool compareMipmaps) const
{
    if (otherImage == osg::NullFC )
    {
        return false;
    }

    // explicitly NOT compared    
    // mip map count: Ignore the number of mip maps as they are created on demand (dynamically)
    // parents:       Parents are ignored, parents do not have to be equal to consider the image equal
    if ((otherImage->getWidth() != getWidth()) ||
        (otherImage->getHeight() != getHeight()) ||
        (otherImage->getDepth() != getDepth()) ||
        (otherImage->getInternalPixelFormat() != getInternalPixelFormat()) ||
        (otherImage->getDataType() != getDataType()) ||
        (otherImage->getSideCount() != getSideCount()) ||
        (otherImage->getAttributes() != getAttributes()) ||
        !isNameEquivalent(otherImage->getName(), getName(), nameOptions) ||
        (otherImage->getSize(compareMipmaps, true, true) != getSize(compareMipmaps, true, true)) ||  // ignore dynamically created mip maps here
        (otherImage->getImageSequence() != NullFC) || (getImageSequence() != NullFC)) // images with inline sequences can not be shared
    {
        return false;
    }

    // Prevent images that have been created by the web engine
    // from being considered equal. Even if they show the same content and have
    // the same properties, they are created by two different web engines, which 
    // means they are two independent images. 

    // The internal flag is currently only set by web engines. We have
    // to change this when web engine merging is implemented.
    if (alwaysConsiderWebEngineImagesDifferent && (otherImage->isInternal() || isInternal()) )
    {
        return false;
    }

    // now compare the pixels (compare only mipmap level 0)!
    if (!arePixelsEqual(ImagePtr(this), otherImage, 0))
    {
        return false;
    }
    else if (compareMipmaps)
    {
        for (UInt32 i = 1; i < getMipMapCount(); ++i)
        {
            if (!arePixelsEqual(ImagePtr(this), otherImage, i))
                return false;
        }
    }

    return true;
}


float Image::getAspect(const ImagePtr &image)
{
    // if there is no image, just return default value
    if (image == NullFC)
    {
        return 1.0f;
    }

    // return aspect ratio of image
    const float width = image->getWidth();
    const float height = image->getHeight();
    return (width / height);
}


// Return the number of components per pixel.

UInt8  Image::getComponents(void) const
{
    Int32 mapSizeFormat = sizeof(_formatDic) / sizeof(UInt32[2]);

    for (UInt16 i = 0; i < mapSizeFormat; i++)
    {
        if (_formatDic[i][0] == getInternalPixelFormat())
            return _formatDic[i][1];
    }
    FWARNING(("Image::getComponents: image %p has unknown pixel format 0x%x!",
        this, getInternalPixelFormat()));
    return 0;
}

/*------------------------------ set object data --------------------------*/
/*! method to set the image data. Use the doCopy parameter to specify, whether
    the method should copy or link the pixel data.
*/
bool Image::set(UInt32 pF,
    Int32 w, Int32 h,
    Int32 d, Int32 mmS, Int32 fS,
    Time fD, const UChar8 *da, Int32 t, bool allocMem,
    Int32 sS, float gamma)
{
    ImagePtr iPtr(this);
    
    iPtr->clearUncompressedDataCache();
    iPtr->setExternalData(NULL);

    beginEditCP(iPtr,
        PixelFormatFieldMask |
        WidthFieldMask |
        HeightFieldMask |
        DepthFieldMask |
        MipMapCountFieldMask |
        SideCountFieldMask |
        FrameCountFieldMask |
        FrameDelayFieldMask |
        DataTypeFieldMask |
        GammaFieldMask);

    setPixelFormat(pF);

    setWidth(osgMax(1, w));
    setHeight(osgMax(1, h));
    setDepth(osgMax(1, d));

    setMipMapCount(osgMax(1, mmS));
    setSideCount(osgMax(1, sS));
    setFrameCount(osgMax(1, fS));

    setFrameDelay(fD);

    setGamma(gamma);

    setDataType(t);

    const bool created = createData(da, allocMem);

    endEditCP(iPtr,
        PixelFormatFieldMask |
        WidthFieldMask |
        HeightFieldMask |
        DepthFieldMask |
        MipMapCountFieldMask |
        SideCountFieldMask |
        FrameCountFieldMask |
        FrameDelayFieldMask |
        DataTypeFieldMask |
        GammaFieldMask);

    calcMipmapOffsets();

    _contentChanged = true;
    m_imageInfoPresent = false;
    return created;
}

/*! method to set the image from another image object.
    Use the doCopy parameter to specify, whether
    the method should copy or link the pixel data.
*/
bool Image::set(ImagePtr image, bool useInternalPixelFormat)
{
    this->set(useInternalPixelFormat ? image->getInternalPixelFormat() : image->getPixelFormat(),
        image->getWidth(),
        image->getHeight(),
        image->getDepth(),
        image->getMipMapCount(),
        image->getFrameCount(),
        image->getFrameDelay(),
        useInternalPixelFormat ? image->getRawData() : image->getData(),
        image->getDataType(),
        true,
        image->getSideCount(),
        image->getGamma());

    /*
    ImagePtr iPtr(this);
    beginEditCP(iPtr, ImagesFieldMask);
    iPtr->getImages() = image->getImages();
    endEditCP(iPtr, ImagesFieldMask);
    */
    _contentChanged = true;
    m_imageInfoPresent = false;
    return true;
}

/*! method to set only the image pixel data, all parameter (e. pixelFormat
    width,height and depth) stay the same
*/
bool Image::setData(const UChar8 *da)
{
    if (da)
    {
        createData(da);
    }
    else
    {
        FWARNING(("Image::setData(Null) call\n"));
    }

    return (da ? true : false);
}

/*! validate pixel format, data type  and bpp.
    return true when something was repaired.
*/
bool Image::validate(void)
{
    bool repaired = false;

    Int32 i;
    Int32 mapSizeFormat = sizeof(_formatDic) / sizeof(UInt32[2]);
    Int32 mapSizeType = sizeof(_typeDic) / sizeof(UInt32[2]);

    // set bpp
    UInt32 pixelFormat = 0;
    UInt32 typeFormat = 0;
    for (i = 0; i < mapSizeFormat; i++)
    {
        if (_formatDic[i][0] == getInternalPixelFormat())
            pixelFormat = _formatDic[i][1];
    }
    for (i = 0; i < mapSizeType; i++)
    {
        if (_typeDic[i][0] == getDataType())
            typeFormat = _typeDic[i][1];
    }

    Int32 dimension = 0;
    if (getDepth() == 1)
    {
        if (getHeight() == 1)
            dimension = 1;
        else
            dimension = 2;
    }
    else
    {
        dimension = 3;
    }

    ImagePtr iPtr(this);

    if (getComponentSize() != typeFormat)
    {
        beginEditCP(iPtr, ComponentSizeFieldMask);
        setComponentSize(typeFormat);
        endEditCP(iPtr, ComponentSizeFieldMask);
        repaired = true;
    }

    if (getBpp() != pixelFormat * typeFormat)
    {
        beginEditCP(iPtr, BppFieldMask);
        setBpp(pixelFormat * typeFormat);
        endEditCP(iPtr, BppFieldMask);
        repaired = true;
    }

    if (dimension != getDimension())
    {
        beginEditCP(iPtr, DimensionFieldMask);
        setDimension(dimension);
        endEditCP(iPtr, DimensionFieldMask);
        repaired = true;
    }

    return repaired;
}

bool Image::setICCProfile(const UChar8* profile, UInt32 profileLength)
{
    if (profile == NULL)
    {
        FWARNING(("Image::setICCProfile(NULL) call\n"));
        return false;
    }

    ImagePtr iPtr(this);
    beginEditCP(iPtr, ICCProfileDataFieldMask);
    MFUInt8* pmf = iPtr->getMFICCProfileData();
    pmf->resize(profileLength);

    for (UInt32 i = 0; i < profileLength; ++i)
        (*pmf)[i] = profile[i];

    endEditCP(iPtr, ICCProfileDataFieldMask);
    return true;
}

void Image::removeICCProfile()
{
    ImagePtr iPtr(this);
    if(!iPtr->getICCProfileData().empty())
    {
        beginEditCP(iPtr, ICCProfileDataFieldMask);
        iPtr->editMFICCProfileData()->clear();
        endEditCP(iPtr, ICCProfileDataFieldMask);
    }
}

bool Image::setChromaticities( const osg::Vec2f &red, const osg::Vec2f &green,
                            const osg::Vec2f &blue, const osg::Vec2f &white)
{
    ImagePtr iPtr(this);
    beginEditCP(iPtr, ChromaticitiesFieldMask);
    MFVec2f* pmf = iPtr->getMFChromaticities();
    pmf->resize(4);
    (*pmf)[0] = red;
    (*pmf)[1] = green;
    (*pmf)[2] = blue;
    (*pmf)[3] = white;
    endEditCP(iPtr, ChromaticitiesFieldMask);

    return true;
}

void Image::removeChromaticities()
{
    ImagePtr iPtr(this);
    if(!iPtr->getChromaticities().empty())
    {
        beginEditCP(iPtr, ChromaticitiesFieldMask);
        iPtr->editMFChromaticities()->clear();
        endEditCP(iPtr, ChromaticitiesFieldMask);
    }
}

void Image::setShareable(bool s)
{
    _shareable = s;
}

bool Image::isShareable() const
{
    return _shareable;
}

bool Image::isCompressed() const
{
    // TODO add Image::OSG_RGB9E5_PF format?
    return (getInternalPixelFormat() == OSG_RGB_DXT1_PF ||
        getInternalPixelFormat() == OSG_RGBA_DXT1_PF ||
        getInternalPixelFormat() == OSG_RGBA_DXT3_PF ||
        getInternalPixelFormat() == OSG_RGBA_DXT5_PF ||
        getInternalPixelFormat() == OSG_RGB_BC7_PF ||
        getInternalPixelFormat() == OSG_RGBA_BC7_PF ||
        getInternalPixelFormat() == OSG_RGB_BC6U_PF ||
        getInternalPixelFormat() == OSG_RGB_BC6S_PF ||
        getInternalPixelFormat() == OSG_RGB_ASTC_PF ||
        getInternalPixelFormat() == OSG_RGBA_ASTC_PF);
}

static UInt32 convertCompressedToUncompressedPixelFormat(UInt32 format)
{
    switch (format)
    {
    case Image::OSG_RGB_DXT1_PF:
        return Image::OSG_RGB_PF;
    case Image::OSG_RGBA_DXT1_PF:
        return Image::OSG_RGBA_PF;
    case Image::OSG_RGBA_DXT3_PF:
        return Image::OSG_RGBA_PF;
    case Image::OSG_RGBA_DXT5_PF:
        return Image::OSG_RGBA_PF;
    case Image::OSG_RGB_BC7_PF:
        return Image::OSG_RGB_PF;
    case Image::OSG_RGBA_BC7_PF:
        return Image::OSG_RGBA_PF;
    case Image::OSG_RGB_BC6U_PF:
        return Image::OSG_RGB_PF;
    case Image::OSG_RGB_BC6S_PF:
        return Image::OSG_RGB_PF;
    case Image::OSG_RGB_ASTC_PF:
        return Image::OSG_RGB_PF;
    case Image::OSG_RGBA_ASTC_PF:
        return Image::OSG_RGBA_PF;
    default:
        return format;
    }
    return format;
}

UInt32 &Image::editPixelFormat()
{
    _uncompressedPixelFormat = convertCompressedToUncompressedPixelFormat(ImageBase::getPixelFormat());
    return _uncompressedPixelFormat;
}

const UInt32 &Image::getPixelFormat() const
{
    _uncompressedPixelFormat = convertCompressedToUncompressedPixelFormat(ImageBase::getPixelFormat());
    return _uncompressedPixelFormat;
}

UInt32 &Image::getPixelFormat()
{
    _uncompressedPixelFormat = convertCompressedToUncompressedPixelFormat(ImageBase::getPixelFormat());
    return _uncompressedPixelFormat;
}

UInt32 Image::getInternalPixelFormat() const
{
    return ImageBase::getPixelFormat();
}

void Image::setInternal(bool s)
{
    _internal = s;
}

bool Image::isInternal() const
{
    return _internal;
}

void Image::addAttribute(UInt32 attr)
{
    _sfAttributes.setValue(_sfAttributes.getValue() | attr);
}

void Image::subAttribute(UInt32 attr)
{
    _sfAttributes.setValue(_sfAttributes.getValue() & ~attr);
}

bool Image::hasAttribute(UInt32 attr) const
{
    return (_sfAttributes.getValue() & attr) == attr;
}

bool Image::hasAttributes(UInt32 attr) const
{
    unsigned long idx;
    while(bitscanForward( idx, attr))
    {
        UInt32 currentAttr = (1 << idx);
        attr &= ~currentAttr;
        if(!hasAttribute(currentAttr))
            return false;
    }
    return true;
}

bool Image::hasExternalFileReference() const
{
    // We have a kind of inverse definition here: We know which images don't have an external 
    // file reference, and assume all others have one.
    bool noReference = getName().empty() || hasAttribute(OSG_ATTR_GENERATED) || isInternal();
    return !noReference;
}

std::string Image::getExternalFilePath(std::string const& notExistingWarningMsg) const
{
    PathHandler *ph = ImageFileHandler::the().getPathHandler();
    const std::string path = getName();
    std::string fullPath = path;
    // convert to the full path
    if(ph != nullptr)
        fullPath = ph->findFile(path.c_str());

    if( File::tstAttr(fullPath.c_str(), AccessFlags::IsReadable))
    {
        return fullPath;
    }
    else
    {
        const std::string srcPath = getSourcePath();
        if(!srcPath.empty() && srcPath != path)
        {
            if(ph != nullptr)
                fullPath = ph->findFile(srcPath.c_str());

            if( File::tstAttr(fullPath.c_str(), AccessFlags::IsReadable))
            {
                return fullPath;
            }
            else
            {
                if(!notExistingWarningMsg.empty())
                    Log::warning(notExistingWarningMsg.c_str(), srcPath.c_str());
            }
        }
        else
        {
            if (!notExistingWarningMsg.empty())
                Log::warning(notExistingWarningMsg.c_str(), path.c_str());
        }
    }

    return std::string();
}



void osg::Image::syncImageSequenceFields(ImagePtr const& source)
{
    if ( source != NullFC )
    {
        ImagePtr iPtr(this);
        beginEditCP(iPtr, Image::ImageSequenceFieldMask | Image::NameFieldMask | Image::SourcePathFieldMask);
        setImageSequence(source->getImageSequence());
        setName(source->getName());
        setSourcePath(source->getSourcePath());
        endEditCP(iPtr, Image::ImageSequenceFieldMask | Image::NameFieldMask | Image::SourcePathFieldMask);

        /// currently not a field, just a non persistent member
        setShareable(source->isShareable());
    }
}

const UChar8* Image::getICCProfile(UInt32 &profileLength) const
{
    ImagePtr iPtr(this);

    MFUInt8* pmf = iPtr->getMFICCProfileData();
    if (pmf->size() != 0)
    {
        profileLength = pmf->size();
        return &((*pmf)[0]);
    }

    profileLength = 0;
    return NULL;
}


void Image::clearData(void)
{
    editMFPixel()->clear();
}

/*! method to update just a subregion of the image data
  all paramter (e. pixelFormat,width,height,depth) stay the same
*/
bool Image::setSubData(Int32 offX, Int32 offY, Int32 offZ,
    Int32 srcW, Int32 srcH, Int32 srcD,
    const UInt8 *src)
{
    UChar8 *dest = editData();
    UInt64 lineSize;

    FDEBUG(("Image::setSubData (%d %d %d) - (%d %d %d) - src %p\n",
        offX, offY, offZ, srcW, srcH, srcD, src));

    if (hasCompressedData())
    {
        FFATAL(("Invalid Image::setSubData for compressed image\n"));
        return false;
    }

    if (!src || !dest)
    {
        FFATAL(("Invalid data pointer in Image::setSubData\n"));
        return false;
    }

    // determine the area to actually copy
    UInt64 xMin = osgMax(0, offX);
    UInt64 yMin = osgMax(0, offY);
    UInt64 zMin = osgMax(0, offZ);

    UInt64 xMax = osgMin(getWidth(), offX + srcW);
    UInt64 yMax = osgMin(getHeight(), offY + srcH);
    UInt64 zMax = osgMin(getDepth(), offZ + srcD);

    // fill the destination buffer with the subdata
    UInt64 destIdx, srcIdx = 0;

    for (UInt64 z = zMin; z < zMax; z++)
    {
        for (UInt64 y = yMin; y < yMax; y++)
        {
            lineSize = (xMax - xMin) * getBpp();
            destIdx = ((z * getHeight() + y) * getWidth() + xMin) * getBpp();
            memcpy(&dest[destIdx], &src[srcIdx], size_t(lineSize));
            srcIdx += Int32((srcW - (xMax - xMin)) * getBpp() + lineSize);
        }
        srcIdx += (srcH - (yMax - yMin)) * srcW * getBpp();
    }

    return true;
}

// add our own external data buffer.
void Image::setExternalData(const UInt8 *data)
{
    _external_data = (UInt8 *)data;
}

const UInt8 *Image::getExternalData() const
{
    return _external_data;
}

/*! The Image is not just a 2D container. The class can hold 3D (volume)
    and movie data. If we have 3D/singleFrame or 2D/multiFrame data without
    mipmaps we can flip between this two formats by just swapping the
    getFrameCount() and getDepth() values.
*/
bool Image::flipDepthFrameData(void)
{
    bool retCode = false;
    Int32 value;

    if ((getMipMapCount() == 1) && ((getFrameCount() == 1) || (getDepth() == 1)))
    {
        value = getFrameCount();
        setFrameCount(getDepth());
        setDepth(value);
        retCode = true;
    }
    else
    {
        FWARNING(("Cant flipDepthFrameData(); invalid data layout\n"));
    }

    return retCode;
}

/*! Explicitly notfies parents about a change of the image contents. This is
    not strictly required because they are notified anyways, but can be used
    for optimization by specifying only the area that has actually changed.

    \note Currently only TextureChunks are notified.

    \warning Successive calls to this function will overwrite the previously
    set dirty area. If an application makes changes to multiple regions
    they have to accumulated by the user before calling this function.
 */
void
Image::imageContentChanged(
    Int32 minX, Int32 maxX, Int32 minY, Int32 maxY, Int32 minZ, Int32 maxZ)
{
    MFFieldContainerPtr::iterator parentsIt = _mfParents.begin();
    MFFieldContainerPtr::iterator parentsEnd = _mfParents.end();

    m_imageInfoPresent = false;

    for (; parentsIt != parentsEnd; ++parentsIt)
    {
        TextureChunkPtr texParent = TextureChunkPtr::dcast(*parentsIt);

        if (texParent != NullFC)
        {
            texParent->imageContentChanged(minX, maxX, minY, maxY, minZ, maxZ);
        }
    }
}

/*! This method is used by the parser to fill the image with
    string pixel data. It expects the data in VRML PixelTexture Format.
*/
bool Image::addValue(const char *value)
{
    static Image       *currentImage = 0;
    static UChar8      *currentData = 0;

    Int64        j;
    Int64        v;

    bool         isHead = strchr(value, ' ') ? true : false;

    if (hasCompressedData())
    {
        FFATAL(("Invalid Image::addValue for compressed image\n"));
        return false;
    }

    // make sure we only read one image at a time
    if (currentImage == this)
    {
        if (isHead)
        {
            FDEBUG(("Start new read cycle in image::addValue()\n"));
        }
    }
    else
    {
        if (!isHead)
        {
            FFATAL(("Additional image date for different image\n"));
        }
    }

    currentImage = this;

    if (isHead == true)
    {
        Int32        width;
        Int32        height;
        Int32        pixelDepth;
        PixelFormat  pf = osg::Image::OSG_INVALID_PF;

        // read the head
        sscanf(value, "%d %d %d", &width, &height, &pixelDepth);

        FDEBUG(("Image::addValue() set: w/h/bpp: %d/%d/%d\n",
            width, height, pixelDepth));

        switch (getDataType())
        {
        case OSG_UINT8_IMAGEDATA:
            switch (pixelDepth)
            {
            case 1:
                pf = osg::Image::OSG_L_PF;
                break;
            case 2:
                pf = osg::Image::OSG_LA_PF;
                break;
            case 3:
                pf = osg::Image::OSG_RGB_PF;
                break;
            case 4:
                pf = osg::Image::OSG_RGBA_PF;
                break;
            default:
                pf = osg::Image::OSG_INVALID_PF;
                FFATAL(("Invalid pixel depth: %d\n", pixelDepth));
                break;
            }
            break;
        case OSG_UINT16_IMAGEDATA:
            switch (pixelDepth)
            {
            case 2:
                pf = osg::Image::OSG_L_PF;
                break;
            case 4:
                pf = osg::Image::OSG_LA_PF;
                break;
            case 6:
                pf = osg::Image::OSG_RGB_PF;
                break;
            case 8:
                pf = osg::Image::OSG_RGBA_PF;
                break;
            default:
                pf = osg::Image::OSG_INVALID_PF;
                FFATAL(("Invalid pixel depth: %d\n", pixelDepth));
                break;
            }
            break;
        case OSG_UINT32_IMAGEDATA:
            switch (pixelDepth)
            {
            case 4:
                pf = osg::Image::OSG_L_PF;
                break;
            case 8:
                pf = osg::Image::OSG_LA_PF;
                break;
            case 12:
                pf = osg::Image::OSG_RGB_PF;
                break;
            case 16:
                pf = osg::Image::OSG_RGBA_PF;
                break;
            default:
                pf = osg::Image::OSG_INVALID_PF;
                FFATAL(("Invalid pixel depth: %d\n", pixelDepth));
                break;
            }
            break;
        case OSG_FLOAT32_IMAGEDATA:
            switch (pixelDepth)
            {
            case 4:
                pf = osg::Image::OSG_L_PF;
                break;
            case 8:
                pf = osg::Image::OSG_LA_PF;
                break;
            case 12:
                pf = osg::Image::OSG_RGB_PF;
                break;
            case 16:
                pf = osg::Image::OSG_RGBA_PF;
                break;
            default:
                pf = osg::Image::OSG_INVALID_PF;
                FFATAL(("Invalid pixel depth: %d\n", pixelDepth));
                break;
            }
            break;
        case OSG_FLOAT16_IMAGEDATA:
            switch (pixelDepth)
            {
            case 2:
                pf = osg::Image::OSG_L_PF;
                break;
            case 4:
                pf = osg::Image::OSG_LA_PF;
                break;
            case 6:
                pf = osg::Image::OSG_RGB_PF;
                break;
            case 8:
                pf = osg::Image::OSG_RGBA_PF;
                break;
            default:
                pf = osg::Image::OSG_INVALID_PF;
                FFATAL(("Invalid pixel depth: %d\n", pixelDepth));
                break;
            }
            break;
        case OSG_INT16_IMAGEDATA:
        case OSG_INT32_IMAGEDATA:
        {
            FFATAL((" 'addValue' NYI\n "));
        }
        break;
        default:
            setDataType(OSG_INVALID_IMAGEDATATYPE);
            FFATAL(("Invalid type of image data: %d\n", getDataType()));
        }

        if (pf != 0 && (width > 0) && (height > 0))
        {
            set(pf, width, height);

            currentData = editData();
        }
        else
        {
            currentData = NULL;
        }
    }
    else
    {
        if (currentData != NULL)
        {
            // add data
            // TODO; should we check the bounds, should be done by the parser

            v = strtoul(value, 0, strchr(value, 'x') ? 16 : 10);

            for (j = getBpp(); j--;)
            {
                *currentData++ = UChar8((v >> (8 * j)) & 255);
            }

        }
    }

    return currentData ? true : false;
}

/*! It is a simple method to reformat the image pixelFormat (not the size).
    So you can for example convert a RGBA to RGB or RGB to Grey image.
*/
bool Image::reformat(const Image::PixelFormat pixelFormat,
    ImagePtr destination)
{
    UChar8 *data = 0;
    const UChar8 *sourceData = 0;
    UInt64 srcI, destI, destSize = 0;
    UInt32 sum;
    Real64 sumReal;
    ImagePtr dest = destination;

    if (hasCompressedData())
    {
        FFATAL(("Invalid Image::reformat for compressed image\n"));
        return false;
    }

    FINFO(("Try to reformat image from pixelDepth %d to %d\n",
        getPixelFormat(), pixelFormat));

    // TODO !!! code all the cases !!!

    if (getSize() && pixelFormat &&
        (destination != NullFC || (pixelFormat != getPixelFormat())))
    {
        if (dest == NullFC)
        {
            dest = Image::create();
            addRefCP(dest);
        }

        dest->set(pixelFormat, getWidth(), getHeight(), getDepth(), getMipMapCount(),
            getFrameCount(), getFrameDelay(), NULL, getDataType(), true, getSideCount());
        sourceData = getData();
        data = dest->editData();
        destSize = dest->getSize();

        const UInt16 *sourceDataUC16 =
            reinterpret_cast<const UInt16*>(sourceData);
        const UInt32 *sourceDataUC32 =
            reinterpret_cast<const UInt32*>(sourceData);
        const Real32 *sourceDataF32 =
            reinterpret_cast<const Real32*>(sourceData);
        const Real16 *sourceDataH16 =
            reinterpret_cast<const Real16*>(sourceData);


        UInt16 *destDataUC16 = reinterpret_cast<UInt16*>(data);
        UInt32 *destDataUC32 = reinterpret_cast<UInt32*>(data);
        Real32 *destDataF32 = reinterpret_cast<Real32*>(data);
        Real16 *destDataH16 = reinterpret_cast<Real16*>(data);

        if (data)
        {
            switch (getPixelFormat())
            {
                //-----------------------------------------------------
            case OSG_A_PF:
                switch (pixelFormat) {
                case OSG_A_PF:
                case OSG_I_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;

                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;

                case OSG_L_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;

                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;

                case OSG_LA_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize;)
                        {
                            data[destI++] = sourceData[srcI];
                            data[destI++] = sourceData[srcI++];
                        }
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC16[destI++] = sourceDataUC16[srcI];
                            destDataUC16[destI++] = sourceDataUC16[srcI++];
                        }
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC32[destI++] = sourceDataUC32[srcI];
                            destDataUC32[destI++] = sourceDataUC32[srcI++];
                        }
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataF32[destI++] = sourceDataF32[srcI];
                            destDataF32[destI++] = sourceDataF32[srcI++];
                        }
                        break;

                    case OSG_FLOAT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataH16[destI++] = sourceDataH16[srcI];
                            destDataH16[destI++] = sourceDataH16[srcI++];
                        }
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;

                case OSG_RGB_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize;)
                        {
                            data[destI++] = sourceData[srcI];
                            data[destI++] = sourceData[srcI];
                            data[destI++] = sourceData[srcI++];
                        }
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC16[destI++] = sourceDataUC16[srcI];
                            destDataUC16[destI++] = sourceDataUC16[srcI];
                            destDataUC16[destI++] = sourceDataUC16[srcI++];
                        }
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC32[destI++] = sourceDataUC32[srcI];
                            destDataUC32[destI++] = sourceDataUC32[srcI];
                            destDataUC32[destI++] = sourceDataUC32[srcI++];
                        }
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataF32[destI++] = sourceDataF32[srcI];
                            destDataF32[destI++] = sourceDataF32[srcI];
                            destDataF32[destI++] = sourceDataF32[srcI++];
                        }
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataH16[destI++] = sourceDataH16[srcI];
                            destDataH16[destI++] = sourceDataH16[srcI];
                            destDataH16[destI++] = sourceDataH16[srcI++];
                        }
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;

                case OSG_RGBA_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize;)
                        {
                            data[destI++] = sourceData[srcI];
                            data[destI++] = sourceData[srcI];
                            data[destI++] = sourceData[srcI];
                            data[destI++] = sourceData[srcI++];
                        }
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC16[destI++] = sourceDataUC16[srcI];
                            destDataUC16[destI++] = sourceDataUC16[srcI];
                            destDataUC16[destI++] = sourceDataUC16[srcI];
                            destDataUC16[destI++] = sourceDataUC16[srcI++];
                        }
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC32[destI++] = sourceDataUC32[srcI];
                            destDataUC32[destI++] = sourceDataUC32[srcI];
                            destDataUC32[destI++] = sourceDataUC32[srcI];
                            destDataUC32[destI++] = sourceDataUC32[srcI++];
                        }
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataF32[destI++] = sourceDataF32[srcI];
                            destDataF32[destI++] = sourceDataF32[srcI];
                            destDataF32[destI++] = sourceDataF32[srcI];
                            destDataF32[destI++] = sourceDataF32[srcI++];
                        }
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataH16[destI++] = sourceDataH16[srcI];
                            destDataH16[destI++] = sourceDataH16[srcI];
                            destDataH16[destI++] = sourceDataH16[srcI];
                            destDataH16[destI++] = sourceDataH16[srcI++];
                        }
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;
                default:
                    FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                    break;
                }
                break;

                //-----------------------------------------------------
            case OSG_I_PF:
                switch (pixelFormat) {
                case OSG_A_PF:
                case OSG_I_PF:
                case OSG_L_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;

                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;

                case OSG_LA_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize;)
                        {
                            data[destI++] = sourceData[srcI];
                            data[destI++] = sourceData[srcI++];
                        }
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC16[destI++] = sourceDataUC16[srcI];
                            destDataUC16[destI++] = sourceDataUC16[srcI++];
                        }
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC32[destI++] = sourceDataUC32[srcI];
                            destDataUC32[destI++] = sourceDataUC32[srcI++];
                        }
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataF32[destI++] = sourceDataF32[srcI];
                            destDataF32[destI++] = sourceDataF32[srcI++];
                        }
                        break;

                    case OSG_FLOAT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataH16[destI++] = sourceDataH16[srcI];
                            destDataH16[destI++] = sourceDataH16[srcI++];
                        }
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;

                case OSG_RGB_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize;)
                        {
                            data[destI++] = sourceData[srcI];
                            data[destI++] = sourceData[srcI];
                            data[destI++] = sourceData[srcI++];
                        }
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC16[destI++] = sourceDataUC16[srcI];
                            destDataUC16[destI++] = sourceDataUC16[srcI];
                            destDataUC16[destI++] = sourceDataUC16[srcI++];
                        }
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC32[destI++] = sourceDataUC32[srcI];
                            destDataUC32[destI++] = sourceDataUC32[srcI];
                            destDataUC32[destI++] = sourceDataUC32[srcI++];
                        }
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataF32[destI++] = sourceDataF32[srcI];
                            destDataF32[destI++] = sourceDataF32[srcI];
                            destDataF32[destI++] = sourceDataF32[srcI++];
                        }
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataH16[destI++] = sourceDataH16[srcI];
                            destDataH16[destI++] = sourceDataH16[srcI];
                            destDataH16[destI++] = sourceDataH16[srcI++];
                        }
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;

                case OSG_RGBA_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize;)
                        {
                            data[destI++] = sourceData[srcI];
                            data[destI++] = sourceData[srcI];
                            data[destI++] = sourceData[srcI];
                            data[destI++] = sourceData[srcI++];
                        }
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC16[destI++] = sourceDataUC16[srcI];
                            destDataUC16[destI++] = sourceDataUC16[srcI];
                            destDataUC16[destI++] = sourceDataUC16[srcI];
                            destDataUC16[destI++] = sourceDataUC16[srcI++];
                        }
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC32[destI++] = sourceDataUC32[srcI];
                            destDataUC32[destI++] = sourceDataUC32[srcI];
                            destDataUC32[destI++] = sourceDataUC32[srcI];
                            destDataUC32[destI++] = sourceDataUC32[srcI++];
                        }
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataF32[destI++] = sourceDataF32[srcI];
                            destDataF32[destI++] = sourceDataF32[srcI];
                            destDataF32[destI++] = sourceDataF32[srcI];
                            destDataF32[destI++] = sourceDataF32[srcI++];
                        }
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataH16[destI++] = sourceDataH16[srcI];
                            destDataH16[destI++] = sourceDataH16[srcI];
                            destDataH16[destI++] = sourceDataH16[srcI];
                            destDataH16[destI++] = sourceDataH16[srcI++];
                        }
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;
                default:
                    FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                    break;
                }
                break;


                //-----------------------------------------------------
            case OSG_L_PF:
                switch (pixelFormat) {
                case OSG_A_PF:
                case OSG_I_PF:
                case OSG_L_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;

                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;

                case OSG_LA_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize;)
                        {
                            data[destI++] = sourceData[srcI];
                            data[destI++] = sourceData[srcI++];
                        }
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC16[destI++] = sourceDataUC16[srcI];
                            destDataUC16[destI++] = sourceDataUC16[srcI++];
                        }
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC32[destI++] = sourceDataUC32[srcI];
                            destDataUC32[destI++] = sourceDataUC32[srcI++];
                        }
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataF32[destI++] = sourceDataF32[srcI];
                            destDataF32[destI++] = sourceDataF32[srcI++];
                        }
                        break;

                    case OSG_FLOAT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataH16[destI++] = sourceDataH16[srcI];
                            destDataH16[destI++] = sourceDataH16[srcI++];
                        }
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;

                case OSG_RGB_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize;)
                        {
                            data[destI++] = sourceData[srcI];
                            data[destI++] = sourceData[srcI];
                            data[destI++] = sourceData[srcI++];
                        }
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC16[destI++] = sourceDataUC16[srcI];
                            destDataUC16[destI++] = sourceDataUC16[srcI];
                            destDataUC16[destI++] = sourceDataUC16[srcI++];
                        }
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC32[destI++] = sourceDataUC32[srcI];
                            destDataUC32[destI++] = sourceDataUC32[srcI];
                            destDataUC32[destI++] = sourceDataUC32[srcI++];
                        }
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataF32[destI++] = sourceDataF32[srcI];
                            destDataF32[destI++] = sourceDataF32[srcI];
                            destDataF32[destI++] = sourceDataF32[srcI++];
                        }
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataH16[destI++] = sourceDataH16[srcI];
                            destDataH16[destI++] = sourceDataH16[srcI];
                            destDataH16[destI++] = sourceDataH16[srcI++];
                        }
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;

                case OSG_RGBA_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize;)
                        {
                            data[destI++] = sourceData[srcI];
                            data[destI++] = sourceData[srcI];
                            data[destI++] = sourceData[srcI];
                            data[destI++] = sourceData[srcI++];
                        }
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC16[destI++] = sourceDataUC16[srcI];
                            destDataUC16[destI++] = sourceDataUC16[srcI];
                            destDataUC16[destI++] = sourceDataUC16[srcI];
                            destDataUC16[destI++] = sourceDataUC16[srcI++];
                        }
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC32[destI++] = sourceDataUC32[srcI];
                            destDataUC32[destI++] = sourceDataUC32[srcI];
                            destDataUC32[destI++] = sourceDataUC32[srcI];
                            destDataUC32[destI++] = sourceDataUC32[srcI++];
                        }
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataF32[destI++] = sourceDataF32[srcI];
                            destDataF32[destI++] = sourceDataF32[srcI];
                            destDataF32[destI++] = sourceDataF32[srcI];
                            destDataF32[destI++] = sourceDataF32[srcI++];
                        }
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataH16[destI++] = sourceDataH16[srcI];
                            destDataH16[destI++] = sourceDataH16[srcI];
                            destDataH16[destI++] = sourceDataH16[srcI];
                            destDataH16[destI++] = sourceDataH16[srcI++];
                        }
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;
                default:
                    FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                    break;
                }
                break;

                //-----------------------------------------------------
            case OSG_LA_PF:
                switch (pixelFormat) {
                case OSG_A_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize;)
                        {
                            srcI++;
                            data[destI++] = sourceData[srcI++];
                        }
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            srcI++;
                            destDataUC16[destI++] = sourceDataUC16[srcI++];
                        }
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            srcI++;
                            destDataUC32[destI++] = sourceDataUC32[srcI++];
                        }
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            srcI++;
                            destDataF32[destI++] = sourceDataF32[srcI++];
                        }
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            srcI++;
                            destDataH16[destI++] = sourceDataH16[srcI++];
                        }
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;
                case OSG_I_PF:
                case OSG_L_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize;)
                        {
                            data[destI++] = sourceData[srcI++];
                            srcI++;
                        }
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC16[destI++] = sourceDataUC16[srcI++];
                            srcI++;
                        }
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC32[destI++] = sourceDataUC32[srcI++];
                            srcI++;
                        }
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataF32[destI++] = sourceDataF32[srcI++];
                            srcI++;
                        }
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataH16[destI++] = sourceDataH16[srcI++];
                            srcI++;
                        }
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;
                case OSG_LA_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;

                case OSG_RGB_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize;)
                        {
                            data[destI++] = sourceData[srcI];
                            data[destI++] = sourceData[srcI];
                            data[destI++] = sourceData[srcI++];
                            srcI++;
                        }
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC16[destI++] = sourceDataUC16[srcI];
                            destDataUC16[destI++] = sourceDataUC16[srcI];
                            destDataUC16[destI++] = sourceDataUC16[srcI++];
                            srcI++;
                        }
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC32[destI++] = sourceDataUC32[srcI];
                            destDataUC32[destI++] = sourceDataUC32[srcI];
                            destDataUC32[destI++] = sourceDataUC32[srcI++];
                            srcI++;
                        }
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataF32[destI++] = sourceDataF32[srcI];
                            destDataF32[destI++] = sourceDataF32[srcI];
                            destDataF32[destI++] = sourceDataF32[srcI++];
                            srcI++;
                        }
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataH16[destI++] = sourceDataH16[srcI];
                            destDataH16[destI++] = sourceDataH16[srcI];
                            destDataH16[destI++] = sourceDataH16[srcI++];
                            srcI++;
                        }
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;

                case OSG_RGBA_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize;)
                        {
                            data[destI++] = sourceData[srcI];
                            data[destI++] = sourceData[srcI];
                            data[destI++] = sourceData[srcI++];
                            data[destI++] = sourceData[srcI++];
                        }
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC16[destI++] = sourceDataUC16[srcI];
                            destDataUC16[destI++] = sourceDataUC16[srcI];
                            destDataUC16[destI++] = sourceDataUC16[srcI++];
                            destDataUC16[destI++] = sourceDataUC16[srcI++];
                        }
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC32[destI++] = sourceDataUC32[srcI];
                            destDataUC32[destI++] = sourceDataUC32[srcI];
                            destDataUC32[destI++] = sourceDataUC32[srcI++];
                            destDataUC32[destI++] = sourceDataUC32[srcI++];
                        }
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataF32[destI++] = sourceDataF32[srcI];
                            destDataF32[destI++] = sourceDataF32[srcI];
                            destDataF32[destI++] = sourceDataF32[srcI++];
                            destDataF32[destI++] = sourceDataF32[srcI++];
                        }
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataH16[destI++] = sourceDataH16[srcI];
                            destDataH16[destI++] = sourceDataH16[srcI];
                            destDataH16[destI++] = sourceDataH16[srcI++];
                            destDataH16[destI++] = sourceDataH16[srcI++];
                        }
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;
                default:
                    FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                    break;
                }
                break;

                //-----------------------------------------------------
            case OSG_RGB_PF:
                switch (pixelFormat) {
                case OSG_A_PF:
                case OSG_I_PF:
                case OSG_L_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize;)
                        {
                            sum = 0;
                            sum += sourceData[srcI++];
                            sum += sourceData[srcI++];
                            sum += sourceData[srcI++];
                            data[destI++] = sum / 3;
                        }
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sum = 0;
                            sum += sourceDataUC16[srcI++];
                            sum += sourceDataUC16[srcI++];
                            sum += sourceDataUC16[srcI++];
                            destDataUC16[destI++] = sum / 3;
                        }
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sum = 0;
                            sum += sourceDataUC32[srcI++];
                            sum += sourceDataUC32[srcI++];
                            sum += sourceDataUC32[srcI++];
                            destDataUC32[destI++] = sum / 3;
                        }
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sumReal = 0;
                            sumReal += sourceDataF32[srcI++];
                            sumReal += sourceDataF32[srcI++];
                            sumReal += sourceDataF32[srcI++];
                            destDataF32[destI++] = sumReal / 3.0;
                        }
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sumReal = 0;
                            sumReal += sourceDataH16[srcI++];
                            sumReal += sourceDataH16[srcI++];
                            sumReal += sourceDataH16[srcI++];
                            destDataH16[destI++] = sumReal / 3.0;
                        }
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;
                case OSG_LA_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize;)
                        {
                            sum = 0;
                            sum += sourceData[srcI++];
                            sum += sourceData[srcI++];
                            sum += sourceData[srcI++];
                            data[destI++] = sum / 3;
                            data[destI++] = sum / 3;
                        }
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sum = 0;
                            sum += sourceDataUC16[srcI++];
                            sum += sourceDataUC16[srcI++];
                            sum += sourceDataUC16[srcI++];
                            destDataUC16[destI++] = sum / 3;
                            destDataUC16[destI++] = sum / 3;
                        }
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sum = 0;
                            sum += sourceDataUC32[srcI++];
                            sum += sourceDataUC32[srcI++];
                            sum += sourceDataUC32[srcI++];
                            destDataUC32[destI++] = sum / 3;
                            destDataUC32[destI++] = sum / 3;
                        }
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sumReal = 0;
                            sumReal += sourceDataF32[srcI++];
                            sumReal += sourceDataF32[srcI++];
                            sumReal += sourceDataF32[srcI++];
                            destDataF32[destI++] = sumReal / 3.0;
                            destDataF32[destI++] = sumReal / 3.0;
                        }
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sumReal = 0;
                            sumReal += sourceDataH16[srcI++];
                            sumReal += sourceDataH16[srcI++];
                            sumReal += sourceDataH16[srcI++];
                            destDataH16[destI++] = sumReal / 3.0;
                            destDataH16[destI++] = sumReal / 3.0;
                        }
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;
                case OSG_RGB_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;

                case OSG_RGBA_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize;)
                        {
                            sum = 0;
                            sum += data[destI++] = sourceData[srcI++];
                            sum += data[destI++] = sourceData[srcI++];
                            sum += data[destI++] = sourceData[srcI++];
                            data[destI++] = sum / 3;
                        }
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sum = 0;
                            sum += destDataUC16[destI++] = sourceDataUC16[srcI++];
                            sum += destDataUC16[destI++] = sourceDataUC16[srcI++];
                            sum += destDataUC16[destI++] = sourceDataUC16[srcI++];
                            destDataUC16[destI++] = sum / 3;
                        }
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sum = 0;
                            sum += destDataUC32[destI++] = sourceDataUC32[srcI++];
                            sum += destDataUC32[destI++] = sourceDataUC32[srcI++];
                            sum += destDataUC32[destI++] = sourceDataUC32[srcI++];
                            destDataUC32[destI++] = sum / 3;
                        }
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sumReal = 0;
                            sumReal += destDataF32[destI++] = sourceDataF32[srcI++];
                            sumReal += destDataF32[destI++] = sourceDataF32[srcI++];
                            sumReal += destDataF32[destI++] = sourceDataF32[srcI++];
                            destDataF32[destI++] = sumReal / 3.0;
                        }
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sumReal = 0;
                            sumReal += destDataH16[destI++] = sourceDataH16[srcI++];
                            sumReal += destDataH16[destI++] = sourceDataH16[srcI++];
                            sumReal += destDataH16[destI++] = sourceDataH16[srcI++];
                            destDataH16[destI++] = sumReal / 3.0;
                        }
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;
                default:
                    FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                    break;
                }
                break;
                //-----------------------------------------------------
            case OSG_RGBA_PF:
                switch (pixelFormat) {
                case OSG_A_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize;)
                        {
                            srcI += 3;
                            data[destI++] = sourceData[srcI++];;
                        }
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            srcI += 3;
                            destDataUC16[destI++] = sourceDataUC16[srcI++];;
                        }
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            srcI += 3;
                            destDataUC32[destI++] = sourceDataUC32[srcI++];;
                        }
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            srcI += 3;
                            destDataF32[destI++] = sourceDataF32[srcI++];
                        }
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            srcI += 3;
                            destDataH16[destI++] = sourceDataH16[srcI++];
                        }
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;
                case OSG_I_PF:
                case OSG_L_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize;)
                        {
                            sum = 0;
                            sum += sourceData[srcI++];
                            sum += sourceData[srcI++];
                            sum += sourceData[srcI++];
                            data[destI++] = sum / 3;
                            srcI++;
                        }
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sum = 0;
                            sum += sourceDataUC16[srcI++];
                            sum += sourceDataUC16[srcI++];
                            sum += sourceDataUC16[srcI++];
                            destDataUC16[destI++] = sum / 3;
                            srcI++;
                        }
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sum = 0;
                            sum += sourceDataUC32[srcI++];
                            sum += sourceDataUC32[srcI++];
                            sum += sourceDataUC32[srcI++];
                            destDataUC32[destI++] = sum / 3;
                            srcI++;
                        }
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sumReal = 0;
                            sumReal += sourceDataF32[srcI++];
                            sumReal += sourceDataF32[srcI++];
                            sumReal += sourceDataF32[srcI++];
                            destDataF32[destI++] = sumReal / 3.0;
                            srcI++;
                        }
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sumReal = 0;
                            sumReal += sourceDataH16[srcI++];
                            sumReal += sourceDataH16[srcI++];
                            sumReal += sourceDataH16[srcI++];
                            destDataH16[destI++] = sumReal / 3.0;
                            srcI++;
                        }
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;
                case OSG_LA_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize;)
                        {
                            sum = 0;
                            sum += sourceData[srcI++];
                            sum += sourceData[srcI++];
                            sum += sourceData[srcI++];
                            data[destI++] = sum / 3;
                            data[destI++] = sourceData[srcI++];;
                        }
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sum = 0;
                            sum += sourceDataUC16[srcI++];
                            sum += sourceDataUC16[srcI++];
                            sum += sourceDataUC16[srcI++];
                            destDataUC16[destI++] = sum / 3;
                            destDataUC16[destI++] = sourceDataUC16[srcI++];;
                        }
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sum = 0;
                            sum += sourceDataUC32[srcI++];
                            sum += sourceDataUC32[srcI++];
                            sum += sourceDataUC32[srcI++];
                            destDataUC32[destI++] = sum / 3;
                            destDataUC32[destI++] = sourceDataUC32[srcI++];;
                        }
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sumReal = 0;
                            sumReal += sourceDataF32[srcI++];
                            sumReal += sourceDataF32[srcI++];
                            sumReal += sourceDataF32[srcI++];
                            destDataF32[destI++] = sumReal / 3.0;
                            destDataF32[destI++] = sourceDataF32[srcI++];
                        }
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sumReal = 0;
                            sumReal += sourceDataH16[srcI++];
                            sumReal += sourceDataH16[srcI++];
                            sumReal += sourceDataH16[srcI++];
                            destDataH16[destI++] = sumReal / 3.0;
                            destDataH16[destI++] = sourceDataH16[srcI++];
                        }
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;
                case OSG_RGB_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize;)
                        {
                            data[destI++] = sourceData[srcI++];
                            data[destI++] = sourceData[srcI++];
                            data[destI++] = sourceData[srcI++];
                            srcI++;
                        }
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC16[destI++] = sourceDataUC16[srcI++];
                            destDataUC16[destI++] = sourceDataUC16[srcI++];
                            destDataUC16[destI++] = sourceDataUC16[srcI++];
                            srcI++;
                        }
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC32[destI++] = sourceDataUC32[srcI++];
                            destDataUC32[destI++] = sourceDataUC32[srcI++];
                            destDataUC32[destI++] = sourceDataUC32[srcI++];
                            srcI++;
                        }
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataF32[destI++] = sourceDataF32[srcI++];
                            destDataF32[destI++] = sourceDataF32[srcI++];
                            destDataF32[destI++] = sourceDataF32[srcI++];
                            srcI++;
                        }
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataH16[destI++] = sourceDataH16[srcI++];
                            destDataH16[destI++] = sourceDataH16[srcI++];
                            destDataH16[destI++] = sourceDataH16[srcI++];
                            srcI++;
                        }
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;
                case OSG_RGBA_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        memcpy(data, getData(), destSize);
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;

                default:
                    break;
                }
                break;
                //-----------------------------------------------------
            case OSG_BGRA_PF:
                switch (pixelFormat) {
                case OSG_A_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize;)
                        {
                            srcI += 3;
                            data[destI++] = sourceData[srcI++];;
                        }
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            srcI += 3;
                            destDataUC16[destI++] = sourceDataUC16[srcI++];;
                        }
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            srcI += 3;
                            destDataUC32[destI++] = sourceDataUC32[srcI++];;
                        }
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            srcI += 3;
                            destDataF32[destI++] = sourceDataF32[srcI++];
                        }
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            srcI += 3;
                            destDataH16[destI++] = sourceDataH16[srcI++];
                        }
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;
                case OSG_I_PF:
                case OSG_L_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize;)
                        {
                            sum = 0;
                            sum += sourceData[srcI++];
                            sum += sourceData[srcI++];
                            sum += sourceData[srcI++];
                            data[destI++] = sum / 3;
                            srcI++;
                        }
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sum = 0;
                            sum += sourceDataUC16[srcI++];
                            sum += sourceDataUC16[srcI++];
                            sum += sourceDataUC16[srcI++];
                            destDataUC16[destI++] = sum / 3;
                            srcI++;
                        }
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sum = 0;
                            sum += sourceDataUC32[srcI++];
                            sum += sourceDataUC32[srcI++];
                            sum += sourceDataUC32[srcI++];
                            destDataUC32[destI++] = sum / 3;
                            srcI++;
                        }
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sumReal = 0;
                            sumReal += sourceDataF32[srcI++];
                            sumReal += sourceDataF32[srcI++];
                            sumReal += sourceDataF32[srcI++];
                            destDataF32[destI++] = sumReal / 3.0;
                            srcI++;
                        }
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sumReal = 0;
                            sumReal += sourceDataH16[srcI++];
                            sumReal += sourceDataH16[srcI++];
                            sumReal += sourceDataH16[srcI++];
                            destDataH16[destI++] = sumReal / 3.0;
                            srcI++;
                        }
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;
                case OSG_LA_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize;)
                        {
                            sum = 0;
                            sum += sourceData[srcI++];
                            sum += sourceData[srcI++];
                            sum += sourceData[srcI++];
                            data[destI++] = sum / 3;
                            data[destI++] = sourceData[srcI++];;
                        }
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sum = 0;
                            sum += sourceDataUC16[srcI++];
                            sum += sourceDataUC16[srcI++];
                            sum += sourceDataUC16[srcI++];
                            destDataUC16[destI++] = sum / 3;
                            destDataUC16[destI++] = sourceDataUC16[srcI++];;
                        }
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sum = 0;
                            sum += sourceDataUC32[srcI++];
                            sum += sourceDataUC32[srcI++];
                            sum += sourceDataUC32[srcI++];
                            destDataUC32[destI++] = sum / 3;
                            destDataUC32[destI++] = sourceDataUC32[srcI++];;
                        }
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sumReal = 0;
                            sumReal += sourceDataF32[srcI++];
                            sumReal += sourceDataF32[srcI++];
                            sumReal += sourceDataF32[srcI++];
                            destDataF32[destI++] = sumReal / 3.0;
                            destDataF32[destI++] = sourceDataF32[srcI++];
                        }
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sumReal = 0;
                            sumReal += sourceDataH16[srcI++];
                            sumReal += sourceDataH16[srcI++];
                            sumReal += sourceDataH16[srcI++];
                            destDataH16[destI++] = sumReal / 3.0;
                            destDataH16[destI++] = sourceDataH16[srcI++];
                        }
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;
                case OSG_RGB_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize;)
                        {
                            data[destI + 2] = sourceData[srcI++];
                            data[destI + 1] = sourceData[srcI++];
                            data[destI + 0] = sourceData[srcI++];
                            destI += 3;
                            srcI++;
                        }
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC16[destI + 2] = sourceDataUC16[srcI++];
                            destDataUC16[destI + 1] = sourceDataUC16[srcI++];
                            destDataUC16[destI + 0] = sourceDataUC16[srcI++];
                            destI += 3;
                            srcI++;
                        }
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC32[destI + 2] = sourceDataUC32[srcI++];
                            destDataUC32[destI + 1] = sourceDataUC32[srcI++];
                            destDataUC32[destI + 0] = sourceDataUC32[srcI++];
                            destI += 3;
                            srcI++;
                        }
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataF32[destI + 2] = sourceDataF32[srcI++];
                            destDataF32[destI + 1] = sourceDataF32[srcI++];
                            destDataF32[destI + 0] = sourceDataF32[srcI++];
                            destI += 3;
                            srcI++;
                        }
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataH16[destI + 2] = sourceDataH16[srcI++];
                            destDataH16[destI + 1] = sourceDataH16[srcI++];
                            destDataH16[destI + 0] = sourceDataH16[srcI++];
                            destI += 3;
                            srcI++;
                        }
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;
                case OSG_RGBA_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize;)
                        {
                            data[destI + 2] = sourceData[srcI++];
                            data[destI + 1] = sourceData[srcI++];
                            data[destI + 0] = sourceData[srcI++];
                            data[destI + 3] = sourceData[srcI++];
                            destI += 4;
                        }
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC16[destI + 2] = sourceDataUC16[srcI++];
                            destDataUC16[destI + 1] = sourceDataUC16[srcI++];
                            destDataUC16[destI + 0] = sourceDataUC16[srcI++];
                            destDataUC16[destI + 3] = sourceDataUC16[srcI++];
                            destI += 4;
                        }
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC32[destI + 2] = sourceDataUC32[srcI++];
                            destDataUC32[destI + 1] = sourceDataUC32[srcI++];
                            destDataUC32[destI + 0] = sourceDataUC32[srcI++];
                            destDataUC32[destI + 3] = sourceDataUC32[srcI++];
                            destI += 4;
                        }
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataF32[destI + 2] = sourceDataF32[srcI++];
                            destDataF32[destI + 1] = sourceDataF32[srcI++];
                            destDataF32[destI + 0] = sourceDataF32[srcI++];
                            destDataF32[destI + 3] = sourceDataF32[srcI++];
                            destI += 4;
                        }
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataH16[destI + 2] = sourceDataH16[srcI++];
                            destDataH16[destI + 1] = sourceDataH16[srcI++];
                            destDataH16[destI + 0] = sourceDataH16[srcI++];
                            destDataH16[destI + 3] = sourceDataH16[srcI++];
                            destI += 4;
                        }
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;

                default:
                    break;
                }
                break;
            case OSG_BGR_PF:
                switch (pixelFormat) {
                case OSG_A_PF:
                case OSG_I_PF:
                case OSG_L_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize;)
                        {
                            sum = 0;
                            sum += sourceData[srcI++];
                            sum += sourceData[srcI++];
                            sum += sourceData[srcI++];
                            data[destI++] = sum / 3;                            
                        }
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sum = 0;
                            sum += sourceDataUC16[srcI++];
                            sum += sourceDataUC16[srcI++];
                            sum += sourceDataUC16[srcI++];
                            destDataUC16[destI++] = sum / 3;                            
                        }
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sum = 0;
                            sum += sourceDataUC32[srcI++];
                            sum += sourceDataUC32[srcI++];
                            sum += sourceDataUC32[srcI++];
                            destDataUC32[destI++] = sum / 3;                            
                        }
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sumReal = 0;
                            sumReal += sourceDataF32[srcI++];
                            sumReal += sourceDataF32[srcI++];
                            sumReal += sourceDataF32[srcI++];
                            destDataF32[destI++] = sumReal / 3.0;                            
                        }
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sumReal = 0;
                            sumReal += sourceDataH16[srcI++];
                            sumReal += sourceDataH16[srcI++];
                            sumReal += sourceDataH16[srcI++];
                            destDataH16[destI++] = sumReal / 3.0;                            
                        }
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;
                case OSG_LA_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize;)
                        {
                            sum = 0;
                            sum += sourceData[srcI++];
                            sum += sourceData[srcI++];
                            sum += sourceData[srcI++];
                            const auto average = sum /3;
                            data[destI++] = average;
                            data[destI++] = average;
                        }
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sum = 0;
                            sum += sourceDataUC16[srcI++];
                            sum += sourceDataUC16[srcI++];
                            sum += sourceDataUC16[srcI++];
                            const auto average = sum / 3;
                            destDataUC16[destI++] = average;
                            destDataUC16[destI++] = average;
                        }
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sum = 0;
                            sum += sourceDataUC32[srcI++];
                            sum += sourceDataUC32[srcI++];
                            sum += sourceDataUC32[srcI++];
                            const auto average = sum / 3;
                            destDataUC32[destI++] = average;
                            destDataUC32[destI++] = average;
                        }
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sumReal = 0;
                            sumReal += sourceDataF32[srcI++];
                            sumReal += sourceDataF32[srcI++];
                            sumReal += sourceDataF32[srcI++];
                            const auto average = sumReal / 3.0;
                            destDataF32[destI++] = average;
                            destDataF32[destI++] = average;
                        }
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            sumReal = 0;
                            sumReal += sourceDataH16[srcI++];
                            sumReal += sourceDataH16[srcI++];
                            sumReal += sourceDataH16[srcI++];
                            const auto average = sumReal / 3.0;
                            destDataH16[destI++] = average;
                            destDataH16[destI++] = average;
                        }
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;
                case OSG_RGB_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize;)
                        {
                            data[destI + 2] = sourceData[srcI++];
                            data[destI + 1] = sourceData[srcI++];
                            data[destI + 0] = sourceData[srcI++];
                            destI += 3;                            
                        }
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC16[destI + 2] = sourceDataUC16[srcI++];
                            destDataUC16[destI + 1] = sourceDataUC16[srcI++];
                            destDataUC16[destI + 0] = sourceDataUC16[srcI++];
                            destI += 3;                            
                        }
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataUC32[destI + 2] = sourceDataUC32[srcI++];
                            destDataUC32[destI + 1] = sourceDataUC32[srcI++];
                            destDataUC32[destI + 0] = sourceDataUC32[srcI++];
                            destI += 3;                            
                        }
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataF32[destI + 2] = sourceDataF32[srcI++];
                            destDataF32[destI + 1] = sourceDataF32[srcI++];
                            destDataF32[destI + 0] = sourceDataF32[srcI++];
                            destI += 3;                            
                        }
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            destDataH16[destI + 2] = sourceDataH16[srcI++];
                            destDataH16[destI + 1] = sourceDataH16[srcI++];
                            destDataH16[destI + 0] = sourceDataH16[srcI++];
                            destI += 3;                            
                        }
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;
                case OSG_RGBA_PF:
                    switch (getDataType())
                    {
                    case OSG_UINT8_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize;)
                        {
                            auto sum = 0;
                            sum += data[destI + 2] = sourceData[srcI++];
                            sum += data[destI + 1] = sourceData[srcI++];
                            sum += data[destI + 0] = sourceData[srcI++];
                            data[destI + 3] = sum / 3;
                            destI += 4;
                        }
                        break;
                    case OSG_UINT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            auto sum = 0;
                            sum += destDataUC16[destI + 2] = sourceDataUC16[srcI++];
                            sum += destDataUC16[destI + 1] = sourceDataUC16[srcI++];
                            sum += destDataUC16[destI + 0] = sourceDataUC16[srcI++];
                            destDataUC16[destI + 3] = sum / 3;
                            destI += 4;
                        }
                        break;
                    case OSG_UINT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            auto sum = 0;
                            sum += destDataUC32[destI + 2] = sourceDataUC32[srcI++];
                            sum += destDataUC32[destI + 1] = sourceDataUC32[srcI++];
                            sum += destDataUC32[destI + 0] = sourceDataUC32[srcI++];
                            destDataUC32[destI + 3] = sum / 3;
                            destI += 4;
                        }
                        break;
                    case OSG_FLOAT32_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            auto sumReal = 0.0;
                            sumReal += destDataF32[destI + 2] = sourceDataF32[srcI++];
                            sumReal += destDataF32[destI + 1] = sourceDataF32[srcI++];
                            sumReal += destDataF32[destI + 0] = sourceDataF32[srcI++];
                            destDataF32[destI + 3] = sumReal / 3.0;
                            destI += 4;
                        }
                        break;
                    case OSG_FLOAT16_IMAGEDATA:
                        for (srcI = destI = 0; destI < destSize / getComponentSize();)
                        {
                            auto sumReal = 0.0;
                            sumReal += destDataH16[destI + 2] = sourceDataH16[srcI++];
                            sumReal += destDataH16[destI + 1] = sourceDataH16[srcI++];
                            sumReal += destDataH16[destI + 0] = sourceDataH16[srcI++];
                            destDataH16[destI + 3] = sumReal / 3.0;
                            destI += 4;
                        }
                        break;
                    default:
                        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
                        break;
                    }
                    break;

                default:
                    break;
                }
                break;
            case OSG_ALPHA_INTEGER_PF:
            case OSG_RGB_INTEGER_PF:
            case OSG_RGBA_INTEGER_PF:
            case OSG_BGR_INTEGER_PF:
            case OSG_BGRA_INTEGER_PF:
            case OSG_LUMINANCE_INTEGER_PF:
            case OSG_LUMINANCE_ALPHA_INTEGER_PF:
            {
                FFATAL((" 'reformat' NYI\n "));
            }
            break;
            default:
                FWARNING(("Unvalid pixeldepth (%d) in reformat() !\n",
                    pixelFormat));
            }
        }
        if (data)
        {
            // rip the data from the local destImage if necessary
            if (destination == NullFC)
            {
                this->set(dest);
                subRefCP(dest);
            }
        }
    }

    return (data ? true : false);
}

void Image::swapDataEndian(void)
{
    UChar8 *data = editData();

    SizeT size = getSize() / getComponentSize();
    UInt16 *dataUC16 = reinterpret_cast<UInt16*>(data);
    UInt32 *dataUC32 = reinterpret_cast<UInt32*>(data);
    Real32 *dataF32 = reinterpret_cast<Real32*>(data);


    ImagePtr iPtr(this);
    beginEditCP(iPtr, PixelFieldMask);

    switch (getDataType())
    {
    case OSG_UINT8_IMAGEDATA:
        // do nothing
        break;

    case OSG_UINT16_IMAGEDATA:
        for (SizeT i = 0; i < size; ++i)
        {
            UInt16 p = dataUC16[i];
            dataUC16[i] = (((p >> 8)) | (p << 8));
        }
        break;

    case OSG_UINT32_IMAGEDATA:
        for (SizeT i = 0; i < size; ++i)
        {
            UInt32 p = dataUC32[i];
            dataUC32[i] = (((p & 0x000000FF) << 24) | ((p & 0x0000FF00) << 8) |
                ((p & 0x00FF0000) >> 8) | ((p & 0xFF000000) >> 24));
        }
        break;

    case OSG_FLOAT32_IMAGEDATA:
        for (SizeT i = 0; i < size; ++i)
        {
            Real32 p = dataF32[i];
            UInt8 *b = reinterpret_cast<UInt8 *>(&p);
            std::swap(b[0], b[3]);
            std::swap(b[1], b[2]);
            dataF32[i] = p;
        }
        break;

    case OSG_INT16_IMAGEDATA:
    case OSG_INT32_IMAGEDATA:
    {
        FFATAL((" 'swapDataEndian' NYI\n "));
    }
    break;

    default:
        FWARNING(("invalid source data type \n"));
        break;
    }

    endEditCP(iPtr, PixelFieldMask);
}

/*! It is a simple method to convert the image dataType. Does not change the pixelFormat.
    So you can for example convert a image consisting of UChar8 data to Float data.
*/
bool Image::convertDataTypeTo(Int32 destDataType)
{
    if (hasCompressedData())
    {
        FFATAL(("Invalid Image::convertDataTypeTo for compressed image\n"));
        return false;
    }
    if (destDataType == getDataType())
    {
        //FWARNING (( "source image and destination image have same data types: no conversion possible"));
        return true;
    }
    if (destDataType == OSG_INVALID_IMAGEDATATYPE)
    {
        return false;
    }

    FINFO(("Try to convert image from dataType %d to %d\n",
        getDataType(), destDataType));

    ImagePtr dest;
    dest = Image::create();
    addRefCP(dest);

    dest->set(getPixelFormat(),
        getWidth(), getHeight(), getDepth(),
        getMipMapCount(),
        getFrameCount(), 0.0,
        0, destDataType,
        true,
        getSideCount());

    const UChar8 *sourceData = getData();
    UChar8 *destData = dest->editData();

    SizeT sourceSize = getSize() / getComponentSize();
    SizeT destSize = dest->getSize() / dest->getComponentSize();

    const UInt16 *sourceDataUC16 =
        reinterpret_cast<const UInt16*>(sourceData);
    const UInt32 *sourceDataUC32 =
        reinterpret_cast<const UInt32*>(sourceData);
    const Real32 *sourceDataF32 =
        reinterpret_cast<const Real32*>(sourceData);
    const Real16 *sourceDataH16 =
        reinterpret_cast<const Real16*>(sourceData);

    UInt16 *destDataUC16 = reinterpret_cast<UInt16*>(destData);
    UInt32 *destDataUC32 = reinterpret_cast<UInt32*>(destData);
    Real32 *destDataF32 = reinterpret_cast<Real32*>(destData);
    Real16 *destDataH16 = reinterpret_cast<Real16*>(destData);

    switch (getDataType())
    {
    case OSG_UINT8_IMAGEDATA:
        switch (destDataType)
        {
        case OSG_UINT16_IMAGEDATA:
            oneapi::tbb::parallel_for(SizeT(0), sourceSize, SizeT(1) , [&](SizeT i)
            {
                destDataUC16[i] = UInt16(sourceData[i] << 8);
            });
            break;
        case OSG_UINT32_IMAGEDATA:
            oneapi::tbb::parallel_for(SizeT(0), sourceSize, SizeT(1) , [&](SizeT i)
            {
                destDataUC32[i] = UInt32(sourceData[i] << 24);
            });
            break;
        case OSG_FLOAT32_IMAGEDATA:
            oneapi::tbb::parallel_for(SizeT(0), sourceSize, SizeT(1) , [&](SizeT i)
            {
                destDataF32[i] = Real32(sourceData[i] / 255.0);
            });
            break;
        case OSG_FLOAT16_IMAGEDATA:
            oneapi::tbb::parallel_for(SizeT(0), sourceSize, SizeT(1) , [&](SizeT i)
            {
                destDataH16[i] = Real16(sourceData[i] / 255.0);
            });
            break;
        default:
            FWARNING(("invalid destination data type \n"));
            break;
        }

        break;
    case OSG_UINT16_IMAGEDATA:
        switch (destDataType)
        {
        case OSG_UINT8_IMAGEDATA:
        {
#ifdef MINMAX_RESCALE
            UInt16 nMin = UInt16(65535);
            UInt16 nMax = UInt16(0);
            oneapi::tbb::parallel_for(SizeT(0), sourceSize, SizeT(1) , [&](SizeT i)
            {
                if (sourceDataUC16[i] > nMax)
                    nMax = sourceDataUC16[i];
                if (sourceDataUC16[i] < nMin)
                    nMin = sourceDataUC16[i];
            });

            Real32 fRange = Real32(nMax - nMin);
            if (fRange <= 0.0)
            {
                oneapi::tbb::parallel_for(SizeT(0), sourceSize, SizeT(1) , [&](SizeT i)
                {
                    destData[i] = 0;
                });
            }
            else
            {
                oneapi::tbb::parallel_for(SizeT(0), sourceSize, SizeT(1) , [&](SizeT i)
                {
                    destData[i] = UInt8(255.0 * (Real32(sourceDataUC16[i] - nMin)) / fRange);
                });
            }
        }
#else
            oneapi::tbb::parallel_for(SizeT(0), sourceSize, SizeT(1) , [&](SizeT i)
            {
                destData[i] = UInt8(255.0 * (Real32(sourceDataUC16[i])) / 65535.0);
            });
#endif
        }
        break;
        case OSG_UINT32_IMAGEDATA:
            oneapi::tbb::parallel_for(SizeT(0), sourceSize, SizeT(1) , [&](SizeT i)
            {
                destDataUC32[i] = UInt32(sourceDataUC16[i] << 16);
            });
            break;
        case OSG_FLOAT32_IMAGEDATA:
            oneapi::tbb::parallel_for(SizeT(0), sourceSize, SizeT(1) , [&](SizeT i)
            {
                destDataF32[i] = Real32(sourceDataUC16[i] / 65535.0);
            });
            break;
        case OSG_FLOAT16_IMAGEDATA:
            oneapi::tbb::parallel_for(SizeT(0), sourceSize, SizeT(1) , [&](SizeT i)
            {
                destDataH16[i] = Real16(sourceDataUC16[i] / 65535.0);
            });
            break;
        default:
            FWARNING(("invalid destination data type \n"));
            break;
        }
        break;

    case OSG_UINT32_IMAGEDATA:
        switch (destDataType)
        {
        case OSG_UINT8_IMAGEDATA:
        {
#ifdef MINMAX_RESCALE
            UInt32 nMin = UInt32(4294967295ul);
            UInt32 nMax = UInt32(0);
            oneapi::tbb::parallel_for(SizeT(0), sourceSize, SizeT(1) , [&](SizeT i)
            {
                if (sourceDataUC32[i] > nMax)
                    nMax = sourceDataUC32[i];
                if (sourceDataUC32[i] < nMin)
                    nMin = sourceDataUC32[i];
            });

            Real32 fRange = Real32(nMax - nMin);
            if (fRange <= 0.0)
            {
                oneapi::tbb::parallel_for(SizeT(0), sourceSize, SizeT(1) , [&](SizeT i)
                {
                    destData[i] = 0;
                });
            }
            else
            {
                oneapi::tbb::parallel_for(SizeT(0), sourceSize, SizeT(1) , [&](SizeT i)
                {
                    destData[i] = UInt8(255.0 * (Real32(sourceDataUC32[i] - nMin)) / fRange);
                });
            }
#else
            oneapi::tbb::parallel_for(SizeT(0), sourceSize, SizeT(1) , [&](SizeT i)
            {
                destData[i] = UInt8(255.0 * (sourceDataUC32[i] / 4294967295.0));
            });
#endif
        }
        break;
        case OSG_UINT16_IMAGEDATA:
        {
#ifdef MINMAX_RESCALE
            UInt32 nMin = UInt32(4294967295ul);
            UInt32 nMax = UInt32(0);
            oneapi::tbb::parallel_for(SizeT(0), sourceSize, SizeT(1) , [&](SizeT i)
            {
                if (sourceDataUC32[i] > nMax)
                    nMax = sourceDataUC32[i];
                if (sourceDataUC32[i] < nMin)
                    nMin = sourceDataUC32[i];
            });

            Real32 fRange = Real32(nMax - nMin);
            if (fRange <= 0.0)
            {
                oneapi::tbb::parallel_for(SizeT(0), sourceSize, SizeT(1) , [&](SizeT i)
                {
                    destDataUC16[i] = 0;
                });
            }
            else
            {
                oneapi::tbb::parallel_for(SizeT(0), sourceSize, SizeT(1) , [&](SizeT i)
                {
                    destDataUC16[i] = UInt16(65535.0 * (Real32(sourceDataUC32[i] - nMin)) / fRange);
                });
            }
#else
            oneapi::tbb::parallel_for(SizeT(0), sourceSize, SizeT(1) , [&](SizeT i)
            {
                destDataUC16[i] = UInt16(65535.0 * (sourceDataUC32[i] / 4294967295.0));
            });
#endif
        }
        break;
        case OSG_FLOAT32_IMAGEDATA:
            oneapi::tbb::parallel_for(SizeT(0), sourceSize, SizeT(1) , [&](SizeT i)
            {
                destDataF32[i] = Real32(sourceDataUC32[i] / 4294967295.0);
            });
            break;
        case OSG_FLOAT16_IMAGEDATA:
            oneapi::tbb::parallel_for(SizeT(0), sourceSize, SizeT(1) , [&](SizeT i)
            {
                destDataH16[i] = Real16(sourceDataUC32[i] / 4294967295.0);
            });
            break;
        default:
            FWARNING(("invalid destination data type \n"));
            break;
        }
        break;

    case OSG_FLOAT32_IMAGEDATA:
        switch (destDataType)
        {
        case OSG_UINT8_IMAGEDATA:
            oneapi::tbb::parallel_for(SizeT(0), sourceSize, SizeT(1) , [&](SizeT i)
            {
                destData[i] = UInt8(sourceDataF32[i] * 255.0);
            });
            break;
        case OSG_UINT16_IMAGEDATA:
            oneapi::tbb::parallel_for(SizeT(0), sourceSize, SizeT(1) , [&](SizeT i)
            {
                destDataUC16[i] = UInt16(sourceDataF32[i] * 65535.0);
            });
            break;
        case OSG_UINT32_IMAGEDATA:
            oneapi::tbb::parallel_for(SizeT(0), sourceSize, SizeT(1) , [&](SizeT i)
            {
                destDataUC32[i] = UInt32(sourceDataF32[i] * 4294967295.0);
            });
            break;
        case OSG_FLOAT16_IMAGEDATA:
            oneapi::tbb::parallel_for(SizeT(0), sourceSize, SizeT(1) , [&](SizeT i)
            {
                destDataH16[i] = Real16( std::min( HALF_MAX_VAL, sourceDataF32[i])); // half-constructor
            });
            break;
        default:
            FWARNING(("invalid destination data type \n"));
            break;
        }
        break;
    case OSG_FLOAT16_IMAGEDATA:
        switch (destDataType)
        {
        case OSG_UINT8_IMAGEDATA:
            oneapi::tbb::parallel_for(SizeT(0), sourceSize, SizeT(1) , [&](SizeT i)
            {
                destData[i] = UInt8(sourceDataH16[i] * 255.0);
            });
            break;
        case OSG_UINT16_IMAGEDATA:
            oneapi::tbb::parallel_for(SizeT(0), sourceSize, SizeT(1) , [&](SizeT i)
            {
                destDataUC16[i] = UInt16(sourceDataH16[i] * 65535.0);
            });
            break;
        case OSG_UINT32_IMAGEDATA:
            oneapi::tbb::parallel_for(SizeT(0), sourceSize, SizeT(1) , [&](SizeT i)
            {
                destDataUC32[i] = UInt32(sourceDataH16[i] * 4294967295.0);
            });
            break;
        case OSG_FLOAT32_IMAGEDATA:
            oneapi::tbb::parallel_for(SizeT(0), sourceSize, SizeT(1) , [&](SizeT i)
            {
                destDataF32[i] = Real16(sourceDataH16[i]); // half-constructor
            });
            break;
        default:
            FWARNING(("invalid destination data type \n"));
            break;
        }
        break;
    case OSG_INT16_IMAGEDATA:
    case OSG_INT32_IMAGEDATA:
    {
        FFATAL((" 'convertDataTypeTo' NYI\n "));
    }
    break;

    default:
        FWARNING(("invalid source data type \n"));
        break;
    }
    if (dest->getData() != NULL)
    {
        this->set(dest);
        subRefCP(dest);
    }
    return (getData() != nullptr) ? true : false;
}

/*! It just fills the hole image data with the given pixel value. It is
    mainly used to initialize the image data.
*/
void Image::clear(UChar8 pixelValue)
{
    if (getData() != NULL)
        memset(editData(), pixelValue, getSize());
}

void Image::clearFloat(Real32 pixelValue)
{
    SizeT   n = getSize() / getComponentSize();
    Real32 *d = reinterpret_cast<Real32*>(editData());

    if (n && d)
        while (n--)
            *d++ = pixelValue;
}

void Image::clearHalf(Real16 pixelValue)
{
    SizeT   n = getSize() / getComponentSize();
    Real16 *d = reinterpret_cast<Real16*>(editData());

    if (n && d)
        while (n--)
            *d++ = pixelValue;
}

/*-------------------------------------------------------------------------*/
/*                             attachment handling                         */

/*! returns true if the image has any attachments
 */
bool Image::hasAttachment(void) const
{
    Image *img = const_cast<Image*>(this);
    ImageGenericAttPtr att = ImageGenericAttPtr::dcast(
        img->Inherited::findAttachment(
            ImageGenericAtt::getClassType().getGroupId()));
    if (att != NullFC && att->getType().getNumFieldDescs() > 1)
        return true;
    else
        return false;
}

/*! returns the number of attachments
 */
UInt32 Image::attachmentCount(void) const
{
    Image *img = const_cast<Image*>(this);
    ImageGenericAttPtr att = ImageGenericAttPtr::dcast(
        img->Inherited::findAttachment(
            ImageGenericAtt::getClassType().getGroupId()));
    if (att != NullFC)
        return att->getType().getNumFieldDescs() - 1;
    else
        return 0;
}

/*! set a single string attachment for the given key/data pair
 */
void Image::setAttachmentField(const std::string &key,
    const std::string &data)
{
    ImageGenericAttPtr att = ImageGenericAttPtr::dcast(
        findAttachment(
            ImageGenericAtt::getClassType().getGroupId()));
    if (att == NullFC)
    {
        att = ImageGenericAtt::create();
        addAttachment(att);
    }
    Field *field = att->getField(key.c_str());
    if (field == NULL)
    {
        FieldDescription desc(
            SFString::getClassType(),
            key.c_str(),
            0, 0,
            true,
            reinterpret_cast<FieldIndexAccessMethod>(&ImageGenericAtt::getDynamicField));

        UInt32 fieldId = att->addField(desc);
        field = att->getField(fieldId);
    }
    SFString *strField = dynamic_cast<SFString*>(field);
    if (strField)
        strField->setValue(data);
}

/*! returns the string attachment for the given key or Null
 */
const std::string * Image::findAttachmentField(const std::string &key) const
{
    Image *img = const_cast<Image*>(this);
    ImageGenericAttPtr att = ImageGenericAttPtr::dcast(
        img->findAttachment(
            ImageGenericAtt::getClassType().getGroupId()));
    if (att != NullFC)
    {
        Field *field = att->getField(key.c_str());
        if (field != NULL)
        {
            SFString *strField = dynamic_cast<SFString*>(field);
            if (strField != NULL)
                return &strField->getValue();
        }
    }
    return NULL;
}

/*! returns all attachments
 */
void Image::getAttachmentFields(std::vector<std::string> &keys, std::vector<std::string> &values) const
{
    Image *img = const_cast<Image*>(this);
    ImageGenericAttPtr att = ImageGenericAttPtr::dcast(img->findAttachment(ImageGenericAtt::getClassType().getGroupId()));
    if (att == NullFC)
        return;

    for (UInt32 i = 1; i <= att->getType().getNumFieldDescs(); ++i)
    {
        FieldDescription    *fdesc = att->getType().getFieldDescription(i);
        std::string fname = fdesc->getCName();

        Field *field = att->getField(i);
        if (field != NULL)
        {
            SFString *strField = dynamic_cast<SFString*>(field);
            if (strField != NULL)
            {
                keys.push_back(fname);
                values.push_back(strField->getValue());
            }
        }
    }
}

/*! Method to scale the image. It just does a very simple but fast
    'nearest' scale. Should handle mipmap and frame data correct.
    The method can operate on the object or stores the result in
    the optional destination Image.
*/
bool Image::scale(Int32 width, Int32 height, Int32 depth,
    ImagePtr destination)
{
    ImagePtr destImage;
    UInt32  sw, sh, sd, dw, dh, dd;
    Int32   frame, side, mipmap;
    const UChar8  *src;
    UChar8  *dest;
    Int32   oldWidth = getWidth();
    Int32   oldHeight = getHeight();
    Int32   oldDepth = getDepth();

    if ((oldWidth == width) &&
        (oldHeight == height) &&
        (oldDepth == depth))
    {
        if (destination != osg::NullFC)
            *destination = *this;

        return true;
    }

    if (hasCompressedData())
    {
        FFATAL(("Invalid Image::scale for compressed image\n"));
        return false;
    }

    if (destination != NullFC)
        destImage = destination;
    else
        destImage = ImagePtr(this);

    // get pixel
    // !!!!!!!!        WARNING WARNING        !!!!!!!!!!!
    // !!!!!!!! Obscure copy of the old Image !!!!!!!!!!!
    const MFUInt8 srcPixel = (*getMFPixel());
    // set image data
    destImage->set(static_cast<PixelFormat>(getPixelFormat()),
        width, height, depth, getMipMapCount(),
        getFrameCount(), getFrameDelay(), 0, getDataType(),
        true,
        getSideCount(),
        getGamma());

    beginEditCP(destImage, PixelFieldMask);

    // copy every mipmap in every side in every frame
    for (frame = 0; frame < getFrameCount(); frame++)
    {
        for (side = 0; side < getSideCount(); side++)
        {
            for (mipmap = 0; mipmap < getMipMapCount(); mipmap++)
            {
                // get the memory pointer
                src = (&srcPixel[0]) +
                    (side  * getSideSize()) +
                    (frame * getFrameSize());
                if (mipmap)
                    src += calcMipmapSumSize(mipmap,
                        oldWidth, oldHeight, oldDepth);
                dest = destImage->editData(mipmap, frame, side);

                // calc the mipmap size
                sw = oldWidth >> mipmap;
                sh = oldHeight >> mipmap;
                sd = oldDepth >> mipmap;
                destImage->calcMipmapGeometry(mipmap, dw, dh, dd);

                // copy and scale the data
                scaleData(src, sw, sh, sd, dest, dw, dh, dd);
            }
        }
    }

    endEditCP(destImage, PixelFieldMask);

    return true;
}

/*! Scale the image to the next power of 2 dimensions
    The method can operate on the object or stores the result in
    the optional destination Image.
*/
bool Image::scaleNextPower2(ImagePtr destination)
{
    return scale(osgnextpower2(getWidth()),
        osgnextpower2(getHeight()),
        osgnextpower2(getDepth()),
        destination);
}

/*! Crop the image to the given bounding box.
    The method can operate on the object or stores the result in
    the optional destination Image.
*/
bool Image::subImage(Int32 offX, Int32 offY, Int32 offZ,
    Int32 destW, Int32 destH, Int32 destD,
    ImagePtr destination)
{
    ImagePtr destImage = destination;
    bool     retCode = true;

    if (hasCompressedData())
    {
        FFATAL(("Invalid Image::subImage for compressed image\n"));
        return false;
    }

    if (destination == NullFC)
    {
        destImage = Image::create();
        addRefCP(destImage);
    }

    destImage->set(static_cast<PixelFormat>(getPixelFormat()),
        destW, destH, destD, 1, 1, 0.0, 0, getDataType());

    const UChar8  *src = getData();
    UChar8 *dest = destImage->editData();

    FDEBUG(("Image::subImage (%d %d %d) - (%d %d %d) - destPtr %p\n",
        offX, offY, offZ, destW, destH, destD, dest));

    // ensure destination data is zero
    memset(dest, 0, destImage->getSize());

    // determine the area to actually copy
    UInt64 xMin = offX;
    UInt64 yMin = offY;
    UInt64 zMin = offZ;

    UInt64 xMax = osgMin(getWidth(), offX + destW);
    UInt64 yMax = osgMin(getHeight(), offY + destH);
    UInt64 zMax = osgMin(getDepth(), offZ + destD);

    // fill the destination buffer with the subdata
    UInt64 destIdx = 0;

    for (UInt64 z = zMin; z < zMax; z++)
    {
        for (UInt64 y = yMin; y < yMax; y++)
        {
            for (UInt64 x = xMin; x < xMax; x++)
            {
                for (UInt64 i = 0; i < getBpp(); i++)
                {
                    dest[destIdx] = src[((z * getHeight() + y) *
                        getWidth() + x) * getBpp() + i];
                    destIdx++;
                }
            }
            destIdx += (destW - (xMax - xMin)) * getBpp();
        }
        destIdx += (destH - (yMax - yMin)) * destW * getBpp();
    }

    // rip the data from the local destImage if necessary
    if (destination == NullFC)
    {
        this->set(destImage);
        subRefCP(destImage);
    }

    return retCode;
}

/*! Crop a slice.
    The method can operate on the object or stores the result in
    the optional destination Image.
*/
bool Image::slice(Int32 offX, Int32 offY, Int32 offZ,
    ImagePtr destination)
{
    ImagePtr destImage = destination;
    bool     retCode = true;
    UInt32   counter = 0;

    if (hasCompressedData())
    {
        FFATAL(("Invalid Image::slice for compressed image\n"));
        return false;
    }

    if (destination == NullFC)
    {
        destImage = Image::create();
        addRefCP(destImage);
    }

    FDEBUG(("Image::slice (%d %d %d)\n",
        offX, offY, offZ));

    if (offX >= 0) counter++;
    if (offY >= 0) counter++;
    if (offZ >= 0) counter++;

    if (counter != 1) {
        FWARNING(("Image::slice - more/less than one non negative value\n"));
        return false;
    }

    if (offZ >= 0) {
        // XY slice
        retCode = subImage(0, 0, offZ, getWidth(), getHeight(), 1,
            destImage);
    }

    if (offY >= 0) {
        // XZ slice
        destImage->set(static_cast<PixelFormat>(getPixelFormat()),
            getWidth(), getDepth(), 1, 1, 1, 0.0, 0, getDataType());

        const UChar8  *src = getData();
        UChar8  *dest = destImage->editData();

        // ensure destination data is zero
        memset(dest, 0, destImage->getSize());

        for (UInt64 z = 0; z < getDepth(); z++)
        {
            for (UInt64 x = 0; x < getWidth(); x++)
            {
                for (UInt64 i = 0; i < getBpp(); i++)
                {
                    dest[(z * getWidth() + x) * getBpp() + i] = src[((z * getHeight() + offY) *
                        getWidth() + x) *
                        getBpp() + i];
                }
            }
        }
    }

    if (offX >= 0) {
        // YZ slice
        destImage->set(static_cast<PixelFormat>(getPixelFormat()),
            getWidth(), getDepth(), 1, 1, 1, 0.0, 0, getDataType());

        const UChar8  *src = getData();
        UChar8  *dest = destImage->editData();

        // ensure destination data is zero
        memset(dest, 0, destImage->getSize());

        for (UInt64 z = 0; z < getDepth(); z++)
        {
            for (UInt64 y = 0; y < getHeight(); y++)
            {
                for (UInt64 i = 0; i < getBpp(); i++)
                {
                    dest[(z * getHeight() + y) * getBpp() + i] = src[((z * getHeight() + y) *
                        getWidth() + offX) *
                        getBpp() + i];
                }
            }
        }
    }

    // rip the data from the local destImage if necessary
    if (destination == NullFC)
    {
        this->set(destImage);
        subRefCP(destImage);
    }

    return retCode;
}

/*! Create mipmaps data, level defines the number of level
    The method can operate on the object or stores the result in
    the optional destination Image.
*/
bool Image::createMipmap(Int32 level, ImagePtr destination)
{
    struct Offset
    {
        Int32   d;
        Int32   h;
        Int32   w;
    };

    Offset  offset[][8] =
    {
        {   // 000
            { 0, 0, 0 }, { 0, 0, 0 }, { 0, 0, 0 }, { 0, 0, 0 },
            { 0, 0, 0 }, { 0, 0, 0 }, { 0, 0, 0 }, { 0, 0, 0 } },
        {   // 100
            { 0, 0, 0 }, { 1, 0, 0 }, { 0, 0, 0 }, { 0, 0, 0 },
            { 0, 0, 0 }, { 0, 0, 0 }, { 0, 0, 0 }, { 0, 0, 0 } },
        {   // 010
            { 0, 0, 0 }, { 0, 1, 0 }, { 0, 0, 0 }, { 0, 0, 0 },
            { 0, 0, 0 }, { 0, 0, 0 }, { 0, 0, 0 }, { 0, 0, 0 } },
        {   // 110
            { 0, 0, 0 }, { 0, 1, 0 }, { 1, 0, 0 }, { 1, 1, 0 },
            { 0, 0, 0 }, { 0, 0, 0 }, { 0, 0, 0 }, { 0, 0, 0 } },
        {   // 001
            { 0, 0, 0 }, { 0, 0, 1 }, { 0, 0, 0 }, { 0, 0, 0 },
            { 0, 0, 0 }, { 0, 0, 0 }, { 0, 0, 0 }, { 0, 0, 0 } },
        {   // 101
            { 0, 0, 0 }, { 1, 0, 0 }, { 0, 0, 1 }, { 1, 0, 1 },
            { 0, 0, 0 }, { 0, 0, 0 }, { 0, 0, 0 }, { 0, 0, 0 } },
        {   // 011
            { 0, 0, 0 }, { 0, 0, 1 }, { 0, 1, 0 }, { 0, 1, 1 },
            { 0, 0, 0 }, { 0, 0, 0 }, { 0, 0, 0 }, { 0, 0, 0 } },
        {   // 111
            { 0, 0, 0 }, { 1, 0, 0 }, { 0, 1, 0 }, { 1, 1, 0 },
            { 0, 0, 1 }, { 1, 0, 1 }, { 0, 1, 1 }, { 1, 1, 1 } }
    };
    Int32   offsetSize[] = { 0, 2, 2, 4, 2, 4, 4, 8 };
    ImagePtr destImage = destination;
    Int32   w = getWidth(), h = getHeight(), d = getDepth();
    Int64   wm, hm, dm, wi, hi, di;
    const UChar8 *src;
    UChar8 *dest;

    const UInt16 *sourceDataUC16;
    const UInt32 *sourceDataUC32;
    const Real32 *sourceDataF32;
    const Real16 *sourceDataH16;

    UInt16 *destDataUC16;
    UInt32 *destDataUC32;
    Real32 *destDataF32;
    Real16 *destDataH16;

    if (hasCompressedData())
    {
        FFATAL(("Invalid Image::createMipmap for compressed image\n"));
        return false;
    }

    if (destImage == NullFC)
    {
        destImage = Image::create();
        addRefCP(destImage);
    }

    Real32 valueFloat;
    Int32   value, i, elem, dim, side, frame, size, mipmap;
    Int32   channel, lineSize, sliceSize;

    // calc the level count
    if (level < 0)
    {
        level = calcMipmapLevelCount();
    }

    // create destination image
    destImage->set(getPixelFormat(),
        getWidth(), getHeight(), getDepth(),
        level, getFrameCount(),
        getFrameDelay(), 0, getDataType(),
        true,
        getSideCount());

    // copy the data;
    switch (getDataType())
    {
    case OSG_UINT8_IMAGEDATA:
        for (frame = 0; frame < getFrameCount(); frame++)
        {
            for (side = 0; side < getSideCount(); side++)
            {
                src = this->getData(0, frame, side);
                dest = destImage->editData(0, frame, side);
                size = getWidth() * getHeight() * getDepth() * getBpp();
                memcpy(dest, src, size);
                src = dest;
                dest = dest + size;
                w = getWidth();
                h = getHeight();
                d = getDepth();

                for (mipmap = 1; mipmap < level; mipmap++)
                {
                    lineSize = w * getBpp();
                    sliceSize = w * h * getBpp();
                    wm = (w == 1) ? w : (w >> 1);
                    hm = (h == 1) ? h : (h >> 1);
                    dm = (d == 1) ? d : (d >> 1);

                    dim = (d > dm) * 1 + (h > hm) * 2 + (w > wm) * 4;
                    elem = offsetSize[dim];

                    for (di = 0; di < dm; di++)
                    {
                        for (hi = 0; hi < hm; hi++)
                        {
                            for (wi = 0; wi < wm; wi++)
                            {
                                for (channel = 0; channel < getBpp(); channel++)
                                {
                                    value = 0;
                                    for (i = 0; i < elem; i++)
                                    {
                                        value += src[
                                            ((wi * 2) + offset[dim][i].w) * getBpp() +
                                                ((hi * 2) + offset[dim][i].h) * lineSize +
                                                ((di * 2) + offset[dim][i].d) * sliceSize +
                                                channel];
                                    }
                                    *dest++ = Int8(value / elem);
                                }
                            }
                        }
                    }

                    src += sliceSize;
                    w = wm;
                    h = hm;
                    d = dm;
                }
            }
        }
        break;

    case OSG_UINT16_IMAGEDATA:
        for (frame = 0; frame < getFrameCount(); frame++)
        {
            for (side = 0; side < getSideCount(); side++)
            {
                src = this->getData(0, frame, side);
                // sourceDataUC16 = (UInt16*) this->getData(0, frame);
                dest = destImage->editData(0, frame, side);
                // destDataUC16 = (UInt16*) destImage->getData(0, frame);

                size = getWidth() * getHeight() * getDepth() * getBpp();
                // UInt32 sizeUC16 = getWidth() * getHeight() * getDepth() * getBpp()/getComponentSize();
                // for (int i = 0; i < sizeUC16; i++) destDataUC16[i] = sourceDataUC16[i];

                memcpy(dest, src, size);

                src = dest;
                dest = dest + size;
                // sourceDataUC16 = destDataUC16;
                // destDataUC16 = sourceDataUC16 + sizeUC16;

                w = getWidth();
                h = getHeight();
                d = getDepth();

                sourceDataUC16 = reinterpret_cast<const UInt16*>(src);
                destDataUC16 = reinterpret_cast<UInt16*>(dest);

                for (mipmap = 1; mipmap < level; mipmap++)
                {
                    lineSize = w * getBpp() / getComponentSize();
                    sliceSize = w * h * getBpp() / getComponentSize();
                    wm = (w == 1) ? w : (w >> 1);
                    hm = (h == 1) ? h : (h >> 1);
                    dm = (d == 1) ? d : (d >> 1);

                    dim = (d > dm) * 1 + (h > hm) * 2 + (w > wm) * 4;
                    elem = offsetSize[dim];

                    for (di = 0; di < dm; di++)
                    {
                        for (hi = 0; hi < hm; hi++)
                        {
                            for (wi = 0; wi < wm; wi++)
                            {
                                for (channel = 0; channel < (getBpp() / getComponentSize()); channel++)
                                {
                                    value = 0;
                                    for (i = 0; i < elem; i++)
                                    {
                                        value += sourceDataUC16[
                                            ((wi * 2) + offset[dim][i].w) * (getBpp() / getComponentSize()) +
                                                ((hi * 2) + offset[dim][i].h) * lineSize +
                                                ((di * 2) + offset[dim][i].d) * sliceSize +
                                                channel];
                                    }
                                    *destDataUC16++ = UInt16(value / elem);
                                }
                            }
                        }
                    }

                    sourceDataUC16 += sliceSize;
                    w = wm;
                    h = hm;
                    d = dm;
                }
            }
        }
        break;

    case OSG_UINT32_IMAGEDATA:
        for (frame = 0; frame < getFrameCount(); frame++)
        {
            for (side = 0; side < getSideCount(); side++)
            {
                src = this->getData(0, frame, side);
                // sourceDataUC32 = (UInt32*) this->getData(0, frame);
                dest = destImage->editData(0, frame, side);
                // destDataUC32 = (UInt32*) destImage->getData(0, frame);

                size = getWidth() * getHeight() * getDepth() * getBpp();
                // UInt32 sizeUC32 = getWidth() * getHeight() * getDepth() * getBpp()/getComponentSize();

                memcpy(dest, src, size);

                src = dest;
                dest = dest + size;
                // sourceDataUC32 = destDataUC32;
                // destDataUC32 = sourceDataUC32 + sizeUC32;

                w = getWidth();
                h = getHeight();
                d = getDepth();

                sourceDataUC32 = reinterpret_cast<const UInt32*>(src);
                destDataUC32 = reinterpret_cast<UInt32*>(dest);

                for (mipmap = 1; mipmap < level; mipmap++)
                {
                    lineSize = w * (getBpp() / getComponentSize());
                    sliceSize = w * h * (getBpp() / getComponentSize());
                    wm = (w == 1) ? w : (w >> 1);
                    hm = (h == 1) ? h : (h >> 1);
                    dm = (d == 1) ? d : (d >> 1);

                    dim = (d > dm) * 1 + (h > hm) * 2 + (w > wm) * 4;
                    elem = offsetSize[dim];

                    for (di = 0; di < dm; di++)
                    {
                        for (hi = 0; hi < hm; hi++)
                        {
                            for (wi = 0; wi < wm; wi++)
                            {
                                for (channel = 0; channel < (getBpp() / getComponentSize()); channel++)
                                {
                                    value = 0;
                                    for (i = 0; i < elem; i++)
                                    {
                                        value += (sourceDataUC32[
                                            ((wi * 2) + offset[dim][i].w) * (getBpp() / getComponentSize()) +
                                                ((hi * 2) + offset[dim][i].h) * lineSize +
                                                ((di * 2) + offset[dim][i].d) * sliceSize +
                                                channel] / elem);
                                    }
                                    *destDataUC32++ = UInt32(value);
                                }
                            }
                        }
                    }

                    sourceDataUC32 += sliceSize;
                    w = wm;
                    h = hm;
                    d = dm;
                }
            }
        }
        break;

    case OSG_FLOAT32_IMAGEDATA:
        for (frame = 0; frame < getFrameCount(); frame++)
        {
            for (side = 0; side < getSideCount(); side++)
            {
                src = this->getData(0, frame, side);
                dest = destImage->editData(0, frame, side);
                size = getWidth() * getHeight() * getDepth() * getBpp();
                memcpy(dest, src, size);
                src = dest;
                dest = dest + size;
                w = getWidth();
                h = getHeight();
                d = getDepth();

                sourceDataF32 = reinterpret_cast<const Real32*>(src);
                destDataF32 = reinterpret_cast<Real32*>(dest);

                for (mipmap = 1; mipmap < level; mipmap++)
                {
                    lineSize = w * (getBpp() / getComponentSize());
                    sliceSize = w * h * (getBpp() / getComponentSize());
                    wm = (w == 1) ? w : (w >> 1);
                    hm = (h == 1) ? h : (h >> 1);
                    dm = (d == 1) ? d : (d >> 1);

                    dim = (d > dm) * 1 + (h > hm) * 2 + (w > wm) * 4;
                    elem = offsetSize[dim];

                    for (di = 0; di < dm; di++)
                    {
                        for (hi = 0; hi < hm; hi++)
                        {
                            for (wi = 0; wi < wm; wi++)
                            {
                                for (channel = 0; channel < (getBpp() / getComponentSize()); channel++)
                                {
                                    valueFloat = 0;
                                    for (i = 0; i < elem; i++)
                                    {
                                        valueFloat += sourceDataF32[
                                            ((wi * 2) + offset[dim][i].w) * (getBpp() / getComponentSize()) +
                                                ((hi * 2) + offset[dim][i].h) * lineSize +
                                                ((di * 2) + offset[dim][i].d) * sliceSize +
                                                channel];
                                    }
                                    *destDataF32++ = Real32(valueFloat / elem);
                                }
                            }
                        }
                    }

                    sourceDataF32 += sliceSize;
                    w = wm;
                    h = hm;
                    d = dm;
                }
            }
        }
        break;
    case OSG_FLOAT16_IMAGEDATA:
        for (frame = 0; frame < getFrameCount(); frame++)
        {
            for (side = 0; side < getSideCount(); side++)
            {
                src = this->getData(0, frame, side);
                dest = destImage->editData(0, frame, side);
                size = getWidth() * getHeight() * getDepth() * getBpp();
                memcpy(dest, src, size);
                src = dest;
                dest = dest + size;
                w = getWidth();
                h = getHeight();
                d = getDepth();

                sourceDataH16 = reinterpret_cast<const Real16*>(src);
                destDataH16 = reinterpret_cast<Real16*>(dest);

                for (mipmap = 1; mipmap < level; mipmap++)
                {
                    lineSize = w * (getBpp() / getComponentSize());
                    sliceSize = w * h * (getBpp() / getComponentSize());
                    wm = (w == 1) ? w : (w >> 1);
                    hm = (h == 1) ? h : (h >> 1);
                    dm = (d == 1) ? d : (d >> 1);

                    dim = (d > dm) * 1 + (h > hm) * 2 + (w > wm) * 4;
                    elem = offsetSize[dim];

                    for (di = 0; di < dm; di++)
                    {
                        for (hi = 0; hi < hm; hi++)
                        {
                            for (wi = 0; wi < wm; wi++)
                            {
                                for (channel = 0; channel < (getBpp() / getComponentSize()); channel++)
                                {
                                    valueFloat = 0;
                                    for (i = 0; i < elem; i++)
                                    {
                                        valueFloat += sourceDataH16[
                                            ((wi * 2) + offset[dim][i].w) * (getBpp() / getComponentSize()) +
                                                ((hi * 2) + offset[dim][i].h) * lineSize +
                                                ((di * 2) + offset[dim][i].d) * sliceSize +
                                                channel];
                                    }
                                    *destDataH16++ = Real16(valueFloat / elem);
                                }
                            }
                        }
                    }
                    sourceDataH16 += sliceSize;
                    w = wm;
                    h = hm;
                    d = dm;
                }
            }
        }
        break;

    case OSG_INT16_IMAGEDATA:
    case OSG_INT32_IMAGEDATA:
    {
        FFATAL((" 'createMipmap' NYI\n "));
    }
    break;

    default:
        FWARNING(("Invalid IMAGE_DATA_TYPE\n"));
        break;
    }

    // rip the data from the local destImage if necessary
    if (destination == NullFC)
    {
        this->set(destImage);
        subRefCP(destImage);
    }

    return true;
}

bool Image::removeMipmap(void)
{
    if (getMipMapCount() == 1) // no mipmaps nothing to do.
        return true;

    // create destination image
    ImagePtr destImage = Image::create();
    destImage->set(getPixelFormat(),
        getWidth(), getHeight(), getDepth(),
        1, getFrameCount(),
        getFrameDelay(), 0, getDataType(),
        true,
        getSideCount(),
        getGamma());

    if (!destImage->isValid())
    {
        subRefCP(destImage);
        return false;
    }

    // copy the data;
    for (Int32 frame = 0; frame < getFrameCount(); frame++)
    {
        for (Int32 side = 0; side < getSideCount(); side++)
        {
            const UChar8 *src = this->getData(0, frame, side);
            UChar8 *dest = destImage->editData(0, frame, side);
            SizeT size = (SizeT)getWidth() * getHeight() * getDepth() * getBpp();
            memcpy(dest, src, size);
        }
    }

    this->set(destImage);
    subRefCP(destImage);

    return true;
}


/*! Write the image to the a file. The mimetype will be set automatically
    from the fileName suffix. Returns true on success.
*/
bool Image::write(const Char8 *fileName)
{
    return ImageFileHandler::the().write(ImagePtr(this), fileName);
}

/*! Read the image data from a file. Returns true on success.
 */
bool Image::read(const Char8 *fileName)
{
    ImagePtr iPtr(this);

    return ImageFileHandler::the().read(iPtr, fileName);
}


bool osg::Image::hasSequenceRange() const
{
    int first = 0;
    int last = 0;
    int count = 0;
    getSequenceRange(first, last, count);

    if (count > 1)
        return true;

    return false;    
}

void Image::parseFilenameDigits(const std::string &filename, int &digit_pos, int& number_length)
{
    digit_pos = -1;
    number_length = 0;
    for(int i = filename.length() - 1 ; i >= 0; --i)
    {
        // we only want to look for a digit in the basename
        if(filename[i] == '/' || filename[i] == '\\')
            break;

        if(std::isdigit(filename[i]))
        {
            digit_pos = i;
            ++number_length;
        }
        else
        {
            if(digit_pos != -1)
                break;
        }
    }
}

bool Image::addImageNumber(std::string &filename, UInt32 imageNumber)
{
    if(filename.empty())
        return false;

    //printf("Image::addImageNumber1: '%s' %d\n", filename.c_str(), imageNumber);
    // extract image number as string to get also the number of leading zeros.
    // we search from the end of the string so the chance is quite high that
    // we find the right number if there are multiple numbers in the file path.
    int digit_pos;
    int number_length;
    parseFilenameDigits(filename, digit_pos, number_length);

    if(digit_pos == -1)
        return false;

    // currently we support only filenames with leading zeros image numbers.
    const int leading_zeros = number_length;

    std::ostringstream ss;
    ss << std::setw(leading_zeros) << std::setfill('0') << imageNumber;
    filename.replace(digit_pos, number_length, ss.str());
    //printf("Image::addImageNumber2: lz %d '%s'\n", leading_zeros, filename.c_str());
    return true;
}

int Image::getImageSequencePosition(const std::string &filename)
{
    if(filename.empty())
        return 0;

    int digit_pos;
    int number_length;
    parseFilenameDigits(filename, digit_pos, number_length);

    // inform caller that we were not able to find a number at all.
    if(digit_pos == -1)
        return -1;

    return std::stoi(filename.substr(digit_pos, number_length));
}

void Image::getSequenceRange(int &first, int &last, int &count) const
{
    ImagePtr iPtr(this);

    count = 0;
    first = -1;
    last = 0;

    std::function<bool (int)> exists;
    ImageSequencePtr sequence = iPtr->getImageSequence();
    if( sequence == NullFC || sequence->getImages().getSize() == 0)
    {
        exists = [iPtr](int i) -> bool
        {
            std::string filename = iPtr->getSourcePath();
            if(filename.empty())
            {
                PathHandler *ph = ImageFileHandler::the().getPathHandler();
                if(ph != nullptr)
                    filename = ph->findFile(iPtr->getName().c_str());
            }
            if(!addImageNumber(filename, i))
                return false;
            return File::tstAttr(filename.c_str(), AccessFlags::IsReadable);
        };
    }
    else
    {
        // create a vector with the filenames for our fast binary_search
        std::vector<std::string> filenames;

        vrZipper zipper;
        if(zipper.open(sequence->getImages().getValues().data(), sequence->getImages().getValues().size(), vrZipper::OpenMode::Read))
        {
            zipper.extract([&filenames](const std::filesystem::path& path, const std::vector<unsigned char>& data)
            {
                const auto u8Filename = path.filename().u8string();
                const std::string filename = std::string(u8Filename.begin(), u8Filename.end());
                filenames.emplace_back(filename);
                return true;
            }, true);
            zipper.close();
            // to be sure it is really sorted.
            std::sort(filenames.begin(), filenames.end());
        }
        else
        {
            printf("Image::getSequenceRange: failed to open zip file!\n");
        }

        exists = [filenames](int i) -> bool
        {
            if(filenames.empty())
                return false;
            std::string filename = filenames[0];
            if(!addImageNumber(filename, i))
                return false;
            // filenames is already sorted so we can use the fast binary_search.
            return std::binary_search(filenames.begin(), filenames.end(), filename);
        };
    }

    int i = 0;
    while(true)
    {
        if(exists(i))
        {
            if(first == -1)
                first = i;
            last = i;
            ++count;
        }
        else
        {
            if(first != -1)
            {
                break;
            }
            else
            {
                // should start with 0 or 1.
                if(i >= 1)
                    break;
            }

        }
        ++i;
    }
    if(count == 0)
        first = 0;

    //printf("Image::getSequenceRange: %d %d %d\n", first, last, count);
}

#ifdef VR_USE_IMAGE_SEQUENCE_CACHE
static std::vector<std::vector<UInt8>> _cache;
#endif

/*! Read the image via image number for texture animation via image sequence.
 */
bool Image::readSequence(UInt32 imageNumber, bool firstImage)
{
    ImagePtr iPtr(this);
    ImageSequencePtr sequence = iPtr->getImageSequence();
    if( sequence == osg::NullFC || sequence->getImages().getSize() == 0)
    {
#ifdef VR_USE_IMAGE_SEQUENCE_CACHE
        if(imageNumber < _cache.size())
        {
            iPtr->restore(&_cache[imageNumber][0], _cache[imageNumber].size());
            //beginEditCP(iPtr, Image::PixelFieldMask);
            //    memcpy(&iPtr->getPixel()[0], &_cache[imageNumber][0], _cache[imageNumber].size());
            //endEditCP(iPtr, Image::PixelFieldMask);
            return true;
        }
#endif

        std::string filename = iPtr->getName();
        addImageNumber(filename, imageNumber);
        bool r = ImageFileHandler::the().read(iPtr, filename.c_str());

#ifdef VR_USE_IMAGE_SEQUENCE_CACHE
        if(imageNumber >= _cache.size())
            _cache.resize(imageNumber + 1);
        _cache[imageNumber].resize(iPtr->getSize(false, false, false));
        UInt64 size = iPtr->store(/*nullptr*/ "image/jpeg", &_cache[imageNumber][0], _cache[imageNumber].size());
        _cache[imageNumber].resize(size);
        _cache[imageNumber].shrink_to_fit();
        //_cache[imageNumber].resize(iPtr->getSize(false, false, false));
        //memcpy(&_cache[imageNumber][0], &iPtr->getPixel()[0], _cache[imageNumber].size());
#endif
        return r;
    }
    else
    {
        // extract first filename
        std::string filename;

        // First, trying to extract the filename
        if (vrZipper zipper; zipper.open(sequence->getImages().getValues().data(), sequence->getImages().getValues().size(), vrZipper::OpenMode::Read))
        {
            // return value does not matter here, we got a filename or not.
            zipper.extract([&filename](const std::filesystem::path &path, const std::vector<unsigned char> &data)
                                {
                                    const auto u8Filename = path.filename().u8string();
                                    filename = std::string(u8Filename.begin(), u8Filename.end());
                                    // we found what we needed, so we can stop the extraction.
                                    return false;
                                },
                                true);
        }
        else
        {
            printf("Image::readSequence: failed to open zip file!\n");
        }

        // could not extract the filename from the zip file.
        if (filename.empty())
        {
            return false;
        }

        if (!firstImage)
            addImageNumber(filename, imageNumber);


        if (vrZipper zipper; zipper.open(sequence->getImages().getValues().data(), sequence->getImages().getValues().size(), vrZipper::OpenMode::Read))
        {

            bool result = false;
            zipper.extract([&iPtr, &filename, &result](const std::filesystem::path &path, const std::vector<unsigned char> &data)
                           {

                               std::stringstream ss(std::string(data.begin(), data.end()), std::ios::in | std::ios::out | std::ios::binary);
                               result = ImageFileHandler::the().read(iPtr, ss, ImageFileHandler::the().determineMimetypeFromName(filename), false);
                               if (!result)
                                   printf("Image::readSequence: read failed!\n");
                               // As we gave a filter, it does not really matter here if we return true or false
                               // If we return false, extract will directly return true
                               // If we return true, extract will not directly return, but returns true, because there is a filter
                               return result;
                           },
                           false, std::filesystem::u8path(filename));
            return result;
        }
        else
        {
            printf("Image::readSequence: failed to open zip file!\n");
        }
    }

    return false;
}

std::vector<std::filesystem::path> Image::getFilenames(int last)
{
    ImagePtr iPtr(this);
    std::vector<std::filesystem::path> filenames;

    for(int i=0;i<=last;++i)
    {
        std::string filename = iPtr->getSourcePath();
        if(filename.empty())
        {
            PathHandler *ph = ImageFileHandler::the().getPathHandler();
            if(ph != nullptr)
                filename = ph->findFile(iPtr->getName().c_str());
        }
        if(!addImageNumber(filename, i))
            return filenames;
        const auto filePath = std::filesystem::u8path(filename);
        if(std::filesystem::exists(filePath))
        {
            filenames.emplace_back(filePath);
        }
    }
    return filenames;
}

bool Image::setInlineSequence(bool s)
{
    ImagePtr iPtr(this);

    if(s)
    {
        ImageSequencePtr sequence = iPtr->getImageSequence();
        if( sequence != NullFC && sequence->getImages().getSize() > 0)
            return true; // already inlined
        if(sequence == NullFC)
        {
            sequence = ImageSequence::create();
        }
        int first = 0;
        int last = 0;
        int count = 0;
        getSequenceRange(first, last, count);
        if(count <= 1)
            return false;

        //double t = getSystemTime();
        auto filenames = getFilenames(last);

        // calculate the size of the zip file.
        auto estimatedZipSize = vrZipper::compressBound(filenames);

        // we write the zip file directly into the sequence images vector.
        auto &zip_vec = sequence->getImages().getValues();

        beginEditCP(sequence, ImageSequence::ImagesFieldMask);
            zip_vec.resize(estimatedZipSize);

            vrZipper zipper;
            zipper.setCompressionLevel(0);
            if(zipper.open(&zip_vec[0], zip_vec.size(), vrZipper::OpenMode::Write))
            {
                for(const auto &filename : filenames)
                {
                    if(!zipper.add(filename))
                        return false;
                }
                const auto size = zipper.close();
                zip_vec.resize(size);
            }
            else
            {
                printf("Image::setInlineSequence: failed to open zip file!\n");
            }

        endEditCP(sequence, ImageSequence::ImagesFieldMask);

        beginEditCP(iPtr, ImageSequenceFieldMask);
        iPtr->setImageSequence(sequence);
        endEditCP(iPtr, ImageSequenceFieldMask);

        return true;
    }
    else
    {
        ImageSequencePtr sequence = iPtr->getImageSequence();
        if( sequence == NullFC || sequence->getImages().getSize() == 0)
            return true; // already inlined
        std::string filename = iPtr->getSourcePath();
        if(filename.empty())
        {
            PathHandler *ph = ImageFileHandler::the().getPathHandler();
            if(ph != nullptr)
                filename = ph->findFile(iPtr->getName().c_str());
        }

        if(File::tstAttr(filename.c_str(), AccessFlags::IsReadable))
        {
            beginEditCP(iPtr, ImageSequenceFieldMask);
                iPtr->setImageSequence(NullFC);
            endEditCP(iPtr, ImageSequenceFieldMask);
            return true;
        }
        else
        {
            return false;
        }
    }

    return false;
}

bool Image::hasSequenceOnDisk()
{
    ImagePtr iPtr(this);

    ImageSequencePtr sequence = iPtr->getImageSequence();
    if(sequence == NullFC)
        return false;

    std::string filename = iPtr->getSourcePath();
    if(filename.empty())
    {
        PathHandler *ph = ImageFileHandler::the().getPathHandler();
        if(ph != nullptr)
            filename = ph->findFile(iPtr->getName().c_str());
    }

    return File::tstAttr(filename.c_str(), AccessFlags::IsReadable);
}

bool Image::getInlineSequence()
{
    ImagePtr iPtr(this);
    ImageSequencePtr sequence = iPtr->getImageSequence();
    if(sequence != NullFC && sequence->getImages().getSize() > 0)
        return true;

    return false;
}

/*! Store the image to the given mem block as 'mimeType'.
    mimeType can be 0, in which case the method will store the
    object as uncompressed mtd data.
    Returns the number of bytes used.
*/
UInt64 Image::store(const Char8 *mimeType, UChar8 *mem, Int32 memSize)
{
    ImagePtr iPtr(this);
    return ImageFileHandler::the().store(iPtr,
        mimeType, mem, memSize);
}

/*! Restore the image from the given mem block. Returns the
    number of bytes used.
*/
UInt64 Image::restore(const UChar8 *mem, Int32 memSize)
{
    ImagePtr iPtr(this);
    return ImageFileHandler::the().restore(iPtr, mem, memSize);;
}

/*-------------------------------------------------------------------------*/
/*                               Constructor / Destructor                  */

/*! Default Constructor. Creates a invalid Image of the size 0x0x0
 */
Image::Image(void) :
    Inherited(),
    _mipmapOffset(),
    _external_data(NULL),
    m_imageInfoPresent(false),
    m_imageValueRange(osg::Vec2f(0.0f, 0.0f)),
    _shareable(true),
    _internal(false),
    _neverInline(false),
    _contentChanged(true),
    _uncompressedDataDirty(true),
    _uncompressedData(),
    _uncompressedDataMutex(),
    _uncompressedPixelFormat(OSG_INVALID_PF)
{
    setPixelFormat(OSG_INVALID_PF);
    return;
}

/*! Copy Constructor. Creates a copy of the given image
 */
Image::Image(const Image &obj) :
    Inherited(obj),
    _mipmapOffset(obj._mipmapOffset),
    _external_data(NULL),
    m_imageInfoPresent(false),
    m_imageValueRange(osg::Vec2f(0.0f, 0.0f)),
    _shareable(obj._shareable),
    _internal(obj._internal),
    _neverInline(obj._neverInline),
    _contentChanged(true),
    _uncompressedDataDirty(true),
    _uncompressedData(),
    _uncompressedDataMutex(),
    _uncompressedPixelFormat(OSG_INVALID_PF)
{
}

/*! Destructor.
 */
Image::~Image(void)
{
}

/**
*	\brief called on deletion of an instance
*/
void Image::onDestroy()
{
    if(_sfCDF.getValue() != NullFC)
        subRefCP(_sfCDF.getValue());
    
    if(_sfImageSequence.getValue() != NullFC)
        subRefCP(_sfImageSequence.getValue());
    Inherited::onDestroy();
}

#if defined(OSG_FIXED_MFIELDSYNC)
void Image::onDestroyAspect(UInt32 uiId, UInt32 uiAspect)
{
    Inherited::onDestroyAspect(uiId, uiAspect);

    AttachmentMap::iterator attIt = _attachmentMap.getValue().begin();
    AttachmentMap::iterator attEnd = _attachmentMap.getValue().end();

    AttachmentContainerPtr thisP = getPtr();

    while (attIt != attEnd)
    {
        beginEditCP((*attIt).second, Attachment::ParentsFieldMask);
        {
            (*attIt).second->subParent(thisP);
        }
        endEditCP((*attIt).second, Attachment::ParentsFieldMask);

        subRefCP((*attIt).second);

        ++attIt;
    }

    _attachmentMap.getValue().clear();
}
#endif

/*! Method to check, whether the object data defines a alpha channel or not
 */
bool Image::hasAlphaChannel(void)
{
    return
        getForceAlphaChannel()
        || getInternalPixelFormat() == OSG_RGBA_PF
        || getInternalPixelFormat() == OSG_BGRA_PF
        || getInternalPixelFormat() == OSG_RGBA_DXT1_PF
        || getInternalPixelFormat() == OSG_RGBA_DXT3_PF
        || getInternalPixelFormat() == OSG_RGBA_DXT5_PF
        || getInternalPixelFormat() == OSG_RGBA_BC7_PF
        || getInternalPixelFormat() == OSG_A_PF
        || getInternalPixelFormat() == OSG_I_PF
        || getInternalPixelFormat() == OSG_LA_PF
        || getInternalPixelFormat() == OSG_ALPHA_INTEGER_PF
        || getInternalPixelFormat() == OSG_RGBA_INTEGER_PF
        || getInternalPixelFormat() == OSG_BGRA_INTEGER_PF
        || getInternalPixelFormat() == OSG_LUMINANCE_ALPHA_INTEGER_PF;
}

/*! Method to check, whether the alpha channel is just fully transparent/
    fully opaque
 */
bool Image::isAlphaBinary(void)
{
    return
        getForceAlphaBinary()
        || getInternalPixelFormat() == OSG_RGBA_DXT1_PF;
}

/*! Method to check, whether the data is compressed
 */
bool Image::hasCompressedData(void)
{
    return
        (getForceCompressedData() ||
        (getInternalPixelFormat() == OSG_RGB_DXT1_PF) ||
        (getInternalPixelFormat() == OSG_RGBA_DXT1_PF) ||
        (getInternalPixelFormat() == OSG_RGBA_DXT3_PF) ||
        (getInternalPixelFormat() == OSG_RGBA_DXT5_PF) ||
        (getInternalPixelFormat() == OSG_RGB_BC7_PF) ||
        (getInternalPixelFormat() == OSG_RGBA_BC7_PF) ||
        (getInternalPixelFormat() == OSG_RGB_BC6U_PF) ||
        (getInternalPixelFormat() == OSG_RGB_BC6S_PF) ||
        (getInternalPixelFormat() == OSG_RGB_ASTC_PF) ||
        (getInternalPixelFormat() == OSG_RGBA_ASTC_PF));


}

/*! Method to check, whether the object data defines a color channel or not
 */
bool Image::hasColorChannel(void)
{
    return !(getInternalPixelFormat() == OSG_A_PF ||
        getInternalPixelFormat() == OSG_I_PF ||
        getInternalPixelFormat() == OSG_L_PF ||
        getInternalPixelFormat() == OSG_LA_PF ||
        getInternalPixelFormat() == OSG_ALPHA_INTEGER_PF ||
        getInternalPixelFormat() == OSG_LUMINANCE_ALPHA_INTEGER_PF)
        || getForceColorChannel();
}

/*! Method returns the right frame data for the given time.
 */

#ifndef OSG_2_PREP
UInt8 *Image::getDataByTime(Time   time, UInt32)
#else
const UInt8 *Image::getDataByTime(Time   time, UInt32) const
#endif
{
    UInt32 frameNum = calcFrameNum(time, true);

    return getData(0, frameNum);
}

/*! Check all the alpha values to see if they're 0 or 1, return true if they
are, false if no alpha or intermediate values. No Alpha channel is considered
0.
 */
bool Image::calcIsAlphaBinary(void)
{
    if (!hasAlphaChannel() || getPixelFormat() == OSG_RGBA_DXT1_PF)
        return true;

    if (getInternalPixelFormat() == OSG_RGB9E5_PF || getInternalPixelFormat() == OSG_RGBA_DXT3_PF || getInternalPixelFormat() == OSG_RGBA_DXT5_PF ||
        getInternalPixelFormat() == OSG_RGB_BC7_PF || getInternalPixelFormat() == OSG_RGB_BC6U_PF || getInternalPixelFormat() == OSG_RGB_BC6S_PF ||
        getInternalPixelFormat() == OSG_RGB_ASTC_PF)
    {
        FWARNING(("Image::calcIsAlphaBinary: not implemenetd for DXT3 "
            "and DXT5 yet, assuming false.\n"));
        return false;
    }

    UInt32 npix = getWidth() * getHeight() * getDepth() * getFrameCount();
    UInt8 pixelsize = getBpp();

    const UInt8 *data = getData();

    switch (getInternalPixelFormat())
    {
    case OSG_LA_PF:     data += getComponentSize(); break;
    case OSG_BGRA_PF:
    case OSG_RGBA_PF:   data += getComponentSize() * 3; break;
    default:
        FWARNING(("Image::calcIsAlphaBinary: found unknown "
            "image format %x, assumning false.\n",
            getInternalPixelFormat()));
        return false;
    }

    switch (getDataType())
    {
    case OSG_UINT8_IMAGEDATA:
        for (; npix > 0; --npix, data += pixelsize)
        {
            if (*data != 0 && *data != 0xffU)
                break;
        }
        break;
    case OSG_UINT16_IMAGEDATA:
        for (; npix > 0; --npix, data += pixelsize)
        {
            const UInt16 *d = reinterpret_cast<const UInt16*>(data);

            if (*d != 0 && *d != 0xffffU)
                break;
        }
        break;
    case OSG_UINT32_IMAGEDATA:
        for (; npix > 0; --npix, data += pixelsize)
        {
            const UInt32 *d = reinterpret_cast<const UInt32*>(data);
            if (*d != 0 && *d != 0xffffffffU)
                break;
        }
        break;
    case OSG_FLOAT16_IMAGEDATA:
        for (; npix > 0; --npix, data += pixelsize)
        {
            const Real16 *d = reinterpret_cast<const Real16*>(data);
            if (*d != 0 && *d != 1)
                break;
        }
        break;
    case OSG_FLOAT32_IMAGEDATA:
        for (; npix > 0; --npix, data += pixelsize)
        {
            const Real32 *d = reinterpret_cast<const Real32*>(data);
            if (*d != 0 && *d != 1)
                break;
        }
        break;
    case OSG_INT16_IMAGEDATA:
    case OSG_INT32_IMAGEDATA:
    {
        FFATAL((" 'calcIsAlphaBinary' NYI\n "));
    }
    break;
    default:
        FWARNING(("Image::calcIsAlphaBinary: found unknown "
            "data type %d, assumning false.\n",
            getDataType()));
        return false;
    }

    return npix == 0;
}

/*! Method which returns the frame number for the given time
 */
UInt32 Image::calcFrameNum(Time time, bool OSG_CHECK_ARG(loop)) const
{
    UInt64 frameNum = ((getFrameDelay() > 0) && (getFrameCount() > 0)) ?
        (UInt64(time / getFrameDelay()) % getFrameCount()) : 0;

    return ((frameNum > 0) ? UInt32(frameNum) : 0);
}

/*! Internal used method to calculate the next mipmap geo for the given level
 */
void Image::calcMipmapGeometry(UInt32 mipmapNum,
    UInt32 &width,
    UInt32 &height,
    UInt32 &depth) const
{
    width = getWidth() ? osgMax(getWidth() >> mipmapNum, 1) : 0;
    height = getHeight() ? osgMax(getHeight() >> mipmapNum, 1) : 0;
    depth = getDepth() ? osgMax(getDepth() >> mipmapNum, 1) : 0;
}

#ifdef __sgi
#pragma set woff 1209
#endif

/*! Internal used method to calculate the number of mipmaps levels
 */
UInt32 Image::calcMipmapLevelCount(void) const
{
    UInt32  w = getWidth(), h = getHeight(), d = getDepth();
    UInt32 level;

    for (level = 1; true; level++)
    {
        if ((w == 1) && (h == 1) && (d == 1))
            break;
        else
        {
            w = (w >>= 1) ? w : 1;
            h = (h >>= 1) ? h : 1;
            d = (d >>= 1) ? d : 1;
        }
    }
    return level;
}

UInt32 Image::calcCRC(bool fast)
{
    UInt32 crc = 0;
    if(!isValid())
        return crc;

    //auto t0 = std::chrono::steady_clock::now();

    std::size_t size = 0;
    if(fast)
    {
        // ok use at least a mipmap level with 16 x 16 pixels.
        int level = getMipMapCount() - 5;
        if (level < 0)
            level = 0;

        UInt8 *data = getRawData(level);
        size = calcMipmapLevelSize(level);

        if (data != nullptr)
        {
            crc = BinaryChecksumHandler::calcCrc32c(data, size);
        }
        else
        {
            size = getMFPixel()->size();
            crc = BinaryChecksumHandler::calcCrc32c(&(*getMFPixel())[0], size);
        }
    }
    else
    {
        size = getMFPixel()->size();
        crc = BinaryChecksumHandler::calcCrc32c(&(*getMFPixel())[0], size);
    }

    //auto t1 = std::chrono::steady_clock::now();
    //std::chrono::duration<double, std::milli> diff{t1 - t0};
    //std::cout << "**** Creating image checksum " << crc << " of size " << size << " (fast mode " << (fast ? "on" : "off") << ")" << " bytes took " << diff.count() << " milliseconds." << std::endl;
    return crc;
}

#ifdef __sgi
#pragma reset woff 1209
#endif

/*-------------------------------------------------------------------------*/
/*                            Calculate Mipmap Size                        */


/*! Method to calculate the mem sum of a mipmap level in byte
 */
SizeT Image::calcMipmapLevelSize(UInt32 mipmapNum,
    UInt32 w, UInt32 h, UInt32 d, bool forceUncompressed) const
{
    UInt64 sum = 0;

    if(forceUncompressed)
    {
        sum = (UInt64)(w ? w : 1) * (h ? h : 1) * getBpp();
    }
    else
    {
        switch (getInternalPixelFormat())
        {
        case OSG_RGB9E5_PF:
            sum = (UInt64)(w ? w : 1) * (h ? h : 1) * 4;
            break;
        case OSG_RGB_DXT1_PF:
        case OSG_RGBA_DXT1_PF:
            sum = (UInt64)(((w ? w : 1) + 3) / 4) * (((h ? h : 1) + 3) / 4) * 8;
            break;
        case OSG_RGBA_DXT3_PF:
        case OSG_RGBA_DXT5_PF:
            sum = (UInt64)(((w ? w : 1) + 3) / 4) * (((h ? h : 1) + 3) / 4) * 16;
            break;
        case OSG_RGB_BC7_PF:
        case OSG_RGBA_BC7_PF:
            sum = (UInt64)(((w ? w : 1) + 3) / 4) * (((h ? h : 1) + 3) / 4) * 16;
            break;
        case OSG_RGB_BC6U_PF:
        case OSG_RGB_BC6S_PF:
            sum = (UInt64)(((w ? w : 1) + 3) / 4) * (((h ? h : 1) + 3) / 4) * 16;
            break;
        case OSG_RGB_ASTC_PF:
        case OSG_RGBA_ASTC_PF:
            sum = (UInt64)(((w ? w : 1) + 3) / 4) * (((h ? h : 1) + 3) / 4) * 16;
            break;

        default:
            sum = (UInt64)(w ? w : 1) * (h ? h : 1) * getBpp();
            break;
        }
    }

    sum *= (d ? d : 1);

    //FINFO(("OSGImage::calcMipmapLevelSize: mmsize: %lu\n", sum));

    return (SizeT)(((sizeof(SizeT) == 4) && (sum > (UInt64)TypeTraits<UInt32>::getMax())) ? 0 : sum);
}

/*! MethMethod to calculate the mem of one mipmap level in byte
            for the current Image paramter
 */
SizeT Image::calcMipmapLevelSize(UInt32 mipmapNum, bool forceUncompressed) const
{
    UInt32 w, h, d;
    calcMipmapGeometry(mipmapNum, w, h, d);
    return calcMipmapLevelSize(mipmapNum, w, h, d, forceUncompressed);
}

/*! Method to calculate the mem sum of all mipmap levels in byte
 */
SizeT Image::calcMipmapSumSize(UInt32 mipmapNum,
    UInt32 w, UInt32 h, UInt32 d, bool forceUncompressed) const
{
    UInt64 sum = 0;

    if (w && h && d)
    {
        while (mipmapNum--)
        {
            sum += calcMipmapLevelSize(mipmapNum, w, h, d, forceUncompressed);

            w >>= 1;
            h >>= 1;
            d >>= 1;
        }
    }

    return (SizeT)(((sizeof(SizeT) == 4) && (sum > (UInt64)TypeTraits<UInt32>::getMax())) ? 0 : sum);
}

/*! Method to calculate the mem sum of all mipmap levels in byte
    for the current Image paramter
*/
SizeT Image::calcMipmapSumSize(UInt32 mipmapNum, bool forceUncompressed) const
{
    return calcMipmapSumSize(mipmapNum, getWidth(), getHeight(), getDepth(), forceUncompressed);
}

void Image::calcMipmapOffsets(void)
{
    UInt32 mipMapCount = getMipMapCount();

    if (mipMapCount == 0)
        mipMapCount = 1;

    _mipmapOffset.resize(mipMapCount);

    /*
    for(UInt32 i=0;i<mipMapCount;++i)
        _mipmapOffset[i] = calcMipmapSumSize[i];
    */

    SizeT sum = 0;

    UInt32 w = getWidth();
    UInt32 h = getHeight();
    UInt32 d = getDepth();

    _mipmapOffset[0] = 0;
    for (UInt32 i = 1; i < mipMapCount; ++i)
    {
        sum += calcMipmapLevelSize(i, w, h, d);
        _mipmapOffset[i] = sum;

        w >>= 1;
        h >>= 1;
        d >>= 1;
    }
}

/*-------------------------------------------------------------------------*/
/*                            Image data                                   */

/*! Internal method to set the data and update related properties.
 */
bool Image::createData(const UInt8 *data, bool allocMem)
{
    Int32 i;
    Int32 mapSizeFormat = sizeof(_formatDic) / sizeof(UInt32[2]);
    Int32 mapSizeType = sizeof(_typeDic) / sizeof(UInt32[2]);

    SizeT byteCount = 0;
    ImagePtr iPtr(this);

    beginEditCP(iPtr,
        BppFieldMask |
        DimensionFieldMask |
        FrameSizeFieldMask |
        ComponentSizeFieldMask |
        SideSizeFieldMask |
        PixelFieldMask);

    // set bpp
    UInt32 pixelFormat = 0;
    UInt32 typeFormat = 0;
    for (i = 0; i < mapSizeFormat; i++)
    {
        if (_formatDic[i][0] == getInternalPixelFormat())
            pixelFormat = _formatDic[i][1];
    }
    for (i = 0; i < mapSizeType; i++)
    {
        if (_typeDic[i][0] == getDataType())
            typeFormat = _typeDic[i][1];
    }

    setComponentSize(typeFormat);
    setBpp(pixelFormat * typeFormat);

    // set dimension
    setDimension(0);
    if (getDepth() == 1)
    {
        if (getHeight() == 1)
        {
            setDimension(1);
        }
        else
        {
            setDimension(2);
        }
    }
    else
        setDimension(3);

    setSideSize(calcMipmapSumSize(getMipMapCount(), true));
    setFrameSize(getSideSize() * getSideCount());

    setRawSideSize(calcMipmapSumSize(getMipMapCount()));
    setRawFrameSize(getRawSideSize() * getSideCount());

    // copy the data
    if (allocMem && (byteCount = getRawSize()))
    {
        if (getMFPixel()->getSize() != byteCount)
        {
            try
            {
                //Real64 t = osg::getSystemTime();
                editMFPixel()->fast_resize(byteCount);
                //printf("image resize %f\n", osg::getSystemTime() - t);
            }
            catch (...)
            {
                FFATAL(("Image::createData : Couldn't allocate %u bytes!\n", byteCount));
                return false;
            }
        }
        if (data)
        {
            std::copy_n(data, byteCount, editMFPixel()->begin());
        }
    }
    else
    {
        editMFPixel()->clear();
        editMFPixel()->shrink_to_fit();
    }

    endEditCP(iPtr,
        BppFieldMask |
        DimensionFieldMask |
        FrameSizeFieldMask |
        ComponentSizeFieldMask |
        SideSizeFieldMask |
        PixelFieldMask);

    return (getRawData(0, 0, 0) != NULL);
}

/*! Internal method to scale image data blocks
 */
bool Image::scaleData(const UInt8 *srcData, Int32 srcW, Int32 srcH, Int32 srcD,
    UInt8 *destData, Int32 destW, Int32 destH, Int32 destD)
{
    Real32  sx = Real32(srcW) / Real32(destW);
    Real32  sy = Real32(srcH) / Real32(destH);
    Real32  sz = Real32(srcD) / Real32(destD);
    SizeT   srcSize = srcW * srcH * srcD * getBpp();

    //  Int32 destDize = destW * destH * destD;
    Int32   x, y, z, p;
    const UInt8  *slice, *line, *pixel;

    if (destW == srcW && destH == srcH && destD == srcD)
    {
        // same size, just copy
        memcpy(destData, srcData, srcSize);
    }
    else
    {       // different size, to 'nearest' copy
        for (z = 0; z < destD; z++)
        {
            slice = srcData + int(sz * z) * getBpp() * srcW * srcH;
            for (y = 0; y < destH; y++)
            {
                line = slice + int(sy * y) * getBpp() * srcW;
                for (x = 0; x < destW; x++)
                {
                    pixel = line + int(sx * x) * getBpp();
                    p = getBpp();
                    while (p--)
                        *destData++ = *pixel++;
                }
            }
        }
    }

    return true;
}

/*! Assign operator. Does a copy of the given Image object.
 */
Image &Image::operator=(const Image &image)
{
    this->set(static_cast<PixelFormat>(image.getInternalPixelFormat()), image.getWidth(),
        image.getHeight(), image.getDepth(),
        image.getMipMapCount(), image.getFrameCount(),
        image.getFrameDelay(),
        image.getData(),
        image.getDataType(),
        true, image.getSideCount());

    return *this;
}

/*! Less operator; compares the data sizes of the two images
*/
bool Image::operator<(const Image &image)
{
    return (getSize() < image.getSize()) ? true : false;
}

/*! Method to compare the object to another Image instance;
    Checks first all parameter and afterwards the Image data;
*/
bool Image::operator==(const Image &image)
{
    unsigned long   i, s = getSize();

    if ((getWidth() == image.getWidth()) &&
        (getHeight() == image.getHeight()) &&
        (getDepth() == image.getDepth()) &&
        (getMipMapCount() == image.getMipMapCount()) &&
        (getFrameCount() == image.getFrameCount()) &&
        (getFrameDelay() == image.getFrameDelay()) &&
        (getInternalPixelFormat() == image.getInternalPixelFormat()) &&
        (getDataType() == image.getDataType()) &&
        (getSideCount() == image.getSideCount()))
    {
        for (i = 0; i < s; ++i)
        {
            if (image.getRawData()[i] != getRawData()[i])
                return false;
        }
        return true;
    }
    return false;
}

/*! Method to compare the object to another Image instance;
    Checks first all parameter and afterwards the Image data;
*/
bool Image::operator!=(const Image &image)
{
    return !(*this == image);
}

osg::Vec2f Image::getImageValueRange( const float epsilon)
{
    if (m_imageInfoPresent)
        return osg::Vec2f(m_imageValueRange[0] - epsilon, m_imageValueRange[1] + epsilon);

    m_imageValueRange = osg::Vec2f(FLT_MAX, -FLT_MAX);

    oneapi::tbb::combinable<float> minValue(FLT_MAX);
    oneapi::tbb::combinable<float> maxValue(-FLT_MAX);

    if (getDataType() == Image::OSG_UINT16_IMAGEDATA)
    {
        size_t valueCount = getBpp() * getWidth() * getHeight() * getDepth() / 2;
        osg::UInt16* data16 = reinterpret_cast<osg::UInt16*>(getData(0));
        oneapi::tbb::parallel_for(oneapi::tbb::blocked_range<size_t>(0, valueCount), [&](const oneapi::tbb::blocked_range<size_t>& r)
        {
            float& minimum = minValue.local();
            float& maximum = maxValue.local();
            for (size_t idx = r.begin(); idx < r.end(); idx++)
            {
                minimum = std::min( minimum, data16[idx] / 65535.0f);
                maximum = std::max( maximum, data16[idx] / 65535.0f);
            }
        });
    }
    else if (getDataType() == Image::OSG_INT16_IMAGEDATA)
    {
        size_t valueCount = getBpp() * getWidth() * getHeight() * getDepth() / 2;
        osg::Int16* data16i = reinterpret_cast<osg::Int16*>(getData(0));
         oneapi::tbb::parallel_for(oneapi::tbb::blocked_range<size_t>(0, valueCount), [&](const oneapi::tbb::blocked_range<size_t>& r)
        {
            float& minimum = minValue.local();
            float& maximum = maxValue.local();
            for (size_t idx = r.begin(); idx < r.end(); idx++)
            {
                minimum = std::min( minimum, data16i[idx] / 32767.0f);
                maximum = std::max( maximum, data16i[idx] / 32767.0f);
             }
        });
    }
    else if (getDataType() == Image::OSG_INT32_IMAGEDATA)
    {
        size_t valueCount = getBpp() * getWidth() * getHeight() * getDepth() / 4;
        osg::Int32* data32i= reinterpret_cast<osg::Int32*>(getData(0));
         oneapi::tbb::parallel_for(oneapi::tbb::blocked_range<size_t>(0, valueCount), [&](const oneapi::tbb::blocked_range<size_t>& r)
        {
            float& minimum = minValue.local();
            float& maximum = maxValue.local();
            for (size_t idx = r.begin(); idx < r.end(); idx++)
            {
                minimum = std::min( minimum, static_cast<float>(static_cast<double>(data32i[idx]) / INT_MAX));
                maximum = std::max( maximum, static_cast<float>(static_cast<double>(data32i[idx]) / INT_MAX));
            }
        });
    }
    else if (getDataType() == Image::OSG_UINT32_IMAGEDATA)
    {
        size_t valueCount = getBpp() * getWidth() * getHeight() * getDepth() / 4;
        osg::UInt32* data32 = reinterpret_cast<osg::UInt32*>(getData(0));
        oneapi::tbb::parallel_for(oneapi::tbb::blocked_range<size_t>(0, valueCount), [&](const oneapi::tbb::blocked_range<size_t>& r)
        {
            float& minimum = minValue.local();
            float& maximum = maxValue.local();
            for (size_t idx = r.begin(); idx < r.end(); idx++)
            {
                minimum = std::min( minimum, static_cast<float>(static_cast<double>(data32[idx]) / UINT_MAX));
                maximum = std::max( maximum, static_cast<float>(static_cast<double>(data32[idx]) / UINT_MAX));
            }
        });
    }
    else if (getDataType() == Image::OSG_FLOAT16_IMAGEDATA)
    {
        size_t valueCount = getBpp() * getWidth() * getHeight() * getDepth() / 2;
        osg::Real16* data16f = reinterpret_cast<osg::Real16 *>(getData(0));
         oneapi::tbb::parallel_for(oneapi::tbb::blocked_range<size_t>(0, valueCount), [&](const oneapi::tbb::blocked_range<size_t>& r)
        {
            float& minimum = minValue.local();
            float& maximum = maxValue.local();
            for (size_t idx = r.begin(); idx < r.end(); idx++)
            {
                minimum = std::min( minimum, static_cast<float>(data16f[idx]));
                maximum = std::max( maximum, static_cast<float>(data16f[idx]));
             }
        });

    }
    else if(getDataType() == Image::OSG_FLOAT32_IMAGEDATA)
    {
        size_t valueCount = getBpp() * getWidth() * getHeight() * getDepth() / 4;
        osg::Real32* data32f = reinterpret_cast<osg::Real32 *>(getData(0));
         oneapi::tbb::parallel_for(oneapi::tbb::blocked_range<size_t>(0, valueCount), [&](const oneapi::tbb::blocked_range<size_t>& r)
        {
            float& minimum = minValue.local();
            float& maximum = maxValue.local();
            for (size_t idx = r.begin(); idx < r.end(); idx++)
            {
                minimum = std::min( minimum, data32f[idx]);
                maximum = std::max( maximum, data32f[idx]);
            }
        });
    }
    else
    {
        size_t valueCount = getBpp() * getWidth() * getHeight() * getDepth();
        osg::UInt8* data8 = reinterpret_cast<osg::UInt8*>(getData(0));
         oneapi::tbb::parallel_for(oneapi::tbb::blocked_range<size_t>(0, valueCount), [&](const oneapi::tbb::blocked_range<size_t>& r)
        {
            float& minimum = minValue.local();
            float& maximum = maxValue.local();
            for (size_t idx = r.begin(); idx < r.end(); idx++)
            {
                minimum = std::min( minimum, data8[idx] / 255.0f);
                maximum = std::max( maximum, data8[idx] / 255.0f);
            }
        });
    }
    
    m_imageValueRange[0] = minValue.combine([](const float& a, const float& b)
    {
        return std::min(a,b);
    });
    m_imageValueRange[1] = maxValue.combine([](const float& a, const float& b)
    {
        return std::max(a,b);
    });

    if (m_imageValueRange[0] >= m_imageValueRange[1])
    {
        m_imageValueRange[0] = 0.0f; 
        m_imageValueRange[1] = 1.0f;
    }
    m_imageInfoPresent = true;
    return osg::Vec2f(m_imageValueRange[0] - epsilon, m_imageValueRange[1] + epsilon);
}

bool Image::hasContentChanged()
{
    if(_contentChanged)
    {
        _contentChanged = false;
        return true;
    }
    return false;
}

UInt8 *Image::getRawData ( UInt32 mipmapNum, UInt32 frameNum, UInt32 sideNum)
{
    if(_external_data != nullptr)
    {
        UInt8 *data = _external_data + (getRawSideSize() * sideNum) + (getRawFrameSize() * frameNum);

        if (mipmapNum)
        {
            data += calcMipmapSumSize(mipmapNum);
        }
        return data;
    }

    if(getMFPixel()->empty())
        return nullptr;

    UInt8 *data = (&(*getMFPixel())[0]) + 
      (getRawSideSize()  * sideNum) +
      (getRawFrameSize() * frameNum);
    
    if (mipmapNum)
    {
        data += calcMipmapSumSize(mipmapNum);
    }
  
    return data;
}

const UInt8 *Image::getRawData( UInt32 mipmapNum, 
                                    UInt32 frameNum,
                                    UInt32 sideNum) const
{
    return const_cast<Image *>(this)->getRawData(mipmapNum, frameNum, sideNum);
}

/*! returns a data pointer for a single frame/mipmap chunk
 */
UInt8 *Image::editData(UInt32 mipmapNum, 
                              UInt32 frameNum,
                              UInt32 sideNum)
{
    return getData(mipmapNum, frameNum, sideNum);
}

#ifndef OSG_2_PREP
/*! returns a data pointer for a single frame/mipmap chunk
 */
UInt8 *Image::getData( UInt32 mipmapNum, 
                              UInt32 frameNum,
                              UInt32 sideNum)
{
    if(isCompressed())
    {
        std::scoped_lock lock(_uncompressedDataMutex);
        if(_uncompressedDataDirty)
        {
            ImagePtr tmp(*this);
            ImageCompressor ic;
            if(ic.setImage(tmp))
            {
                ic.getData(_uncompressedData, getPixelFormat(), getDataType());
            }
            else
            {
                FFATAL(("Image::getData: ImageCompressor failed"));
                _uncompressedData.resize(getWidth() * getHeight() * getDepth() * getBpp());
            }
            _uncompressedDataDirty = false;
            //FINFO(("Image::getData: uncompressed image."));
            //std::cout << "*** Image::getData: uncompressed image: " << getName() << std::endl;
        }
        if( (mipmapNum == 0) & (frameNum == 0) & (sideNum == 0))
        {
            return _uncompressedData.data();
        }
        auto sideSize = calcMipmapSumSize(getMipMapCount(), true);
        UInt8 *data = _uncompressedData.data() + (sideSize  * sideNum) + (getFrameSize() * frameNum);
        if (mipmapNum)
        {
            data += calcMipmapSumSize(mipmapNum, true);
        }
        return data;
    }

    return getRawData(mipmapNum, frameNum, sideNum);
}
#endif

/*! returns a data pointer for a single frame/mipmap chunk
 */
const UInt8 *Image::getData( UInt32 mipmapNum, 
                                    UInt32 frameNum,
                                    UInt32 sideNum) const
{
    return const_cast<Image *>(this)->getData(mipmapNum, frameNum, sideNum);
}

void Image::clearUncompressedDataCache()
{
    std::scoped_lock lock(_uncompressedDataMutex);
    _uncompressedDataDirty = true;
    _uncompressedData.clear();
    _uncompressedData.shrink_to_fit();
}

/*------------------------------------------------------------------------*/
/*                              cvs id's                                  */

#ifdef OSG_SGI_CC
#pragma set woff 1174
#endif

#ifdef OSG_LINUX_ICC
#pragma warning( disable : 177 )
#endif

namespace
{
    static Char8 cvsid_cpp[] = "@(#)$Id: $";
    static Char8 cvsid_hpp[] = OSGIMAGEBASE_HEADER_CVSID;
    static Char8 cvsid_inl[] = OSGIMAGEBASE_INLINE_CVSID;

    static Char8 cvsid_fields_hpp[] = OSGIMAGEFIELDS_HEADER_CVSID;
}

#ifdef __sgi
#pragma reset woff 1174
#endif
