/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *               Copyright (C) 2000-2002 by the OpenSG Forum                 *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

#ifndef _OSGIMAGE_H_
#define _OSGIMAGE_H_
#pragma once

#include <OSGConfig.h>
#include <OSGLog.h>
#include <OSGImageBase.h>
#include <OSGReal16.h>
#include <OSGVector.h>
#include <OSGBaseTypeTraits.h>
#include <memory>
#include <filesystem>
#include <mutex>

#define HALF_MAX_VAL 65504.0f

OSG_BEGIN_NAMESPACE

/*! \brief Image class. See \ref PageSystemImage
           for a description.
*/

class OSG_SYSTEMLIB_DLLMAPPING Image : public ImageBase
{
  private:

    typedef ImageBase Inherited;

    /*==========================  PUBLIC  =================================*/
  public:

    enum PixelFormat {   OSG_INVALID_PF = 0,

                         OSG_A_PF       = OSG_GL_ALPHA,
                         OSG_I_PF       = OSG_GL_INTENSITY,

                         OSG_L_PF       = OSG_GL_LUMINANCE,
                         OSG_LA_PF      = OSG_GL_LUMINANCE_ALPHA,

/*** BGR ***/
#if defined(OSG_GL_BGR)
                         OSG_BGR_PF     = OSG_GL_BGR,
#elif defined(OSG_GL_BGR_EXT)
                         OSG_BGR_PF     = OSG_GL_BGR_EXT,
#else
                         OSG_BGR_PF     = 0,
#endif

/*** BGRA ***/
#if defined(OSG_GL_BGRA)
                         OSG_BGRA_PF    = OSG_GL_BGRA,
#elif defined(OSG_GL_BGRA_EXT)
                         OSG_BGRA_PF    = OSG_GL_BGRA_EXT,
#else
                         OSG_BGRA_PF    = 0,
#endif

                         OSG_RGB9E5_PF  = OSG_GL_RGB9_E5,

/*** RGB_DXT1 ***/
#if defined(OSG_GL_COMPRESSED_RGB_S3TC_DXT1_EXT)
                         OSG_RGB_DXT1_PF   = OSG_GL_COMPRESSED_RGB_S3TC_DXT1_EXT,
#else
                         OSG_RGB_DXT1_PF   = 0,
#endif

/*** RGBA_DXT1 ***/
#if defined(OSG_GL_COMPRESSED_RGBA_S3TC_DXT1_EXT)
                         OSG_RGBA_DXT1_PF  = OSG_GL_COMPRESSED_RGBA_S3TC_DXT1_EXT,
#else
                         OSG_RGBA_DXT1_PF   = 0,
#endif

/*** RGBA_DXT3 ***/
#if defined(OSG_GL_COMPRESSED_RGBA_S3TC_DXT3_EXT)
                         OSG_RGBA_DXT3_PF  = OSG_GL_COMPRESSED_RGBA_S3TC_DXT3_EXT,
#else
                         OSG_RGBA_DXT3_PF   = 0,
#endif

/*** RGBA_DXT5 ***/
#if defined(OSG_GL_COMPRESSED_RGBA_S3TC_DXT5_EXT)
                         OSG_RGBA_DXT5_PF  = OSG_GL_COMPRESSED_RGBA_S3TC_DXT5_EXT,
#else
                         OSG_RGBA_DXT5_PF   = 0,
#endif

                        OSG_RGB_BC7_PF = OSG_GL_COMPRESSED_RGB_BPTC_UNORM,
                        OSG_RGBA_BC7_PF = OSG_GL_COMPRESSED_RGBA_BPTC_UNORM,
                        OSG_RGB_BC6U_PF = OSG_GL_COMPRESSED_RGB_BPTC_UNSIGNED_FLOAT,
                        OSG_RGB_BC6S_PF = OSG_GL_COMPRESSED_RGB_BPTC_SIGNED_FLOAT,
                        OSG_RGB_ASTC_PF = OSG_GL_COMPRESSED_RGB_ASTC_4x4,
                        OSG_RGBA_ASTC_PF = OSG_GL_COMPRESSED_RGBA_ASTC_4x4,
                        
                         OSG_RGB_PF     = OSG_GL_RGB,
                         OSG_RGBA_PF    = OSG_GL_RGBA,

                         OSG_ALPHA_INTEGER_PF = OSG_GL_ALPHA_INTEGER_EXT,
                         OSG_RGB_INTEGER_PF = OSG_GL_RGB_INTEGER_EXT,
                         OSG_RGBA_INTEGER_PF = OSG_GL_RGBA_INTEGER_EXT,
                         OSG_BGR_INTEGER_PF = OSG_GL_BGR_INTEGER_EXT,
                         OSG_BGRA_INTEGER_PF = OSG_GL_BGRA_INTEGER_EXT,
                         OSG_LUMINANCE_INTEGER_PF = OSG_GL_LUMINANCE_INTEGER_EXT,
                         OSG_LUMINANCE_ALPHA_INTEGER_PF = OSG_GL_LUMINANCE_ALPHA_INTEGER_EXT
    };

    enum Type {
                         OSG_INVALID_IMAGEDATATYPE  = OSG_GL_NONE,
                         OSG_UINT8_IMAGEDATA        = OSG_GL_UNSIGNED_BYTE,
                         OSG_UINT16_IMAGEDATA       = OSG_GL_UNSIGNED_SHORT,
                         OSG_UINT32_IMAGEDATA       = OSG_GL_UNSIGNED_INT,
                         OSG_FLOAT16_IMAGEDATA      = OSG_GL_HALF_FLOAT_NV,
                         OSG_FLOAT32_IMAGEDATA      = OSG_GL_FLOAT,
                         OSG_INT16_IMAGEDATA         = OSG_GL_SHORT,
                         OSG_INT32_IMAGEDATA         = OSG_GL_INT
    };

    enum ResUnit {
                         OSG_RESUNIT_INVALID       = 0,
                         OSG_RESUNIT_NONE          = 1,
                         OSG_RESUNIT_INCH          = 2
    };

    // never change the order or the values or remove values, adding new is allowed!
    enum Attributes {    OSG_ATTR_NONE             = 0,
                         OSG_ATTR_BUMPMAP          = 1 << 0,
                         OSG_ATTR_NORMALMAP        = 1 << 1,
                         OSG_ATTR_DISPLACEMENTMAP  = 1 << 2,
                         OSG_ATTR_DYNAMICMAP       = 1 << 3,
                         OSG_ATTR_GENERATED        = 1 << 4,
                         OSG_ATTR_NUMERICAL        = 1 << 5, // small texture created from Substance numerical output
                         OSG_ATTR_SUBSTANCE        = 1 << 6, // texture created by substance
                         OSG_ATTR_EMPTY            = 1 << 7 // empty texture
    };

    enum NameComparisonOptions {
        OSG_NAMECMP_IGNORE = 0, // all names are treated as equal
        OSG_NAMECMP_FILENAME = 1, // names are parsed as a full file path and the filename part is compared
        OSG_NAMECMP_FILEPATH = 2 // names are parsed as a full file path and compared
    };

    /*---------------------------------------------------------------------*/
    /*! \name                      Sync                                    */
    /*! \{                                                                 */

    virtual void changed(BitVector  whichField,
                         UInt32     origin    );

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                     Output                                   */
    /*! \{                                                                 */

    virtual void dump(      UInt32     uiIndent = 0,
                      const BitVector  bvFlags  = 0) const;

    /*! \}                                                                 */

    /*---------------------------------------------------------------------*/
    /*! \name                comparison                                    */
    /*! \{                                                                 */
    // overwriting field container interface 
    bool equals(const FieldContainerPtr & fc,
        const IsEqualOptions &options,
        std::map<osg::FieldContainerPtr, std::set<osg::FieldContainerPtr> > &equalFCs) const override;
    
    virtual bool equals(const ImagePtr & image, bool alwaysConsiderWebEngineImagesDifferent, NameComparisonOptions nameOptions, const bool compareMipmaps) const;
    static bool arePixelsEqual(const osg::ImagePtr &img1, const osg::ImagePtr &img2, const int mipmapLevel);
    static bool isEqual(const ImagePtr & image1, const ImagePtr & image2, NameComparisonOptions nameOptions = OSG_NAMECMP_IGNORE, const bool compareMipmaps = false);
    static float getAspect(const ImagePtr &image);

    bool isShareable() const;
    void setShareable(bool shareable);
    bool isCompressed() const;

    UInt32 &editPixelFormat();
    const UInt32 &getPixelFormat() const;
    UInt32 &getPixelFormat();

    UInt32 getInternalPixelFormat() const;

    /*! \}                                                                 */


    /*---------------------------------------------------------------------*/
    /*! \name                   Set Object Data                            */
    /*! \{                                                                 */

    bool set                (      UInt32      pixelFormat,
                                   Int32       width,
                                   Int32       height = 1,
                                   Int32       depth = 1,
                                   Int32       mipmapCount = 1,
                                   Int32       frameCount = 1,
                                   Time        frameDelay = 0.0,
                                   const UInt8       *data = 0,
                                   Int32 type = OSG_UINT8_IMAGEDATA,
                                   bool        allocMem = true,
                                   Int32       sideCount = 1,
                                   float       gamma = -1.0f);

    bool set                (      ImagePtr   image, bool useInternalPixelFormat = true);
    bool setData            (const UInt8     *data = 0         );
    void clearData          (void                              );
    bool setSubData ( Int32 offX, Int32 offY, Int32 offZ,
                      Int32 srcW, Int32 srcH, Int32 srcD,
                      const UInt8 *data );
    void setExternalData(const UInt8 *data);
    const UInt8 *getExternalData() const;
    bool flipDepthFrameData (void                              );

    void imageContentChanged(Int32 minX = -1, Int32 maxX = -1,
                             Int32 minY = -1, Int32 maxY = -1,
                             Int32 minZ = -1, Int32 maxZ = -1 );

    osg::Vec2f getImageValueRange(const float epsilon = 1.e-5f);

    bool validate(void);

    bool setICCProfile( const UChar8* profile, UInt32 profileLength );
    void removeICCProfile();

    bool setChromaticities( const osg::Vec2f &red, const osg::Vec2f &green,
                            const osg::Vec2f &blue, const osg::Vec2f &white);
    void removeChromaticities();

    void setInternal(bool s);
    bool isInternal() const;

    void addAttribute(UInt32 attr);
    void subAttribute(UInt32 attr);
    bool hasAttribute(UInt32 attr) const;
    bool hasAttributes(UInt32 attr) const;

    bool hasExternalFileReference() const;

    /*
    @return the full file path to a referenced texture file if readable, empty string otherwise.
    @param notExistingWarningMsg optional message text with one string parameter %s (the image path) 
                                 that is output in case the path does not exist (is not readable)
    */
    std::string getExternalFilePath(std::string const& notExistingWarningMsg = "") const;

    void syncImageSequenceFields( ImagePtr const& source);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                     Add Value                                */
    /*! \{                                                                 */

    bool addValue (const char *value);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                     Reformate                                */
    /*! \{                                                                 */

    bool reformat ( const PixelFormat pixelFormat,
                    ImagePtr destination = NullFC);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                    Convert dataType                          */
    /*! \{                                                                 */

    void swapDataEndian(void);
    bool convertDataTypeTo ( Int32 destDataType = OSG_UINT8_IMAGEDATA );

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                       Scale                                  */
    /*! \{                                                                 */

    bool scale          ( Int32 width, Int32 height = 1,
                          Int32 depth = 1,
                          ImagePtr destination = NullFC );
    bool scaleNextPower2( ImagePtr destination = NullFC );

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                      SubImage                                */
    /*! \{                                                                 */

    bool subImage ( Int32 offX, Int32 offY, Int32 offZ,
                    Int32 destW, Int32 destH, Int32 destD,
                    ImagePtr destination = NullFC);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                      Slice                                   */
    /*! \{                                                                 */

    bool slice ( Int32 offX = -1, Int32 offY = -1, Int32 offZ = -1,
                 ImagePtr destination = NullFC);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                      Mipmap                                  */
    /*! \{                                                                 */

    bool createMipmap ( Int32 level = -1, ImagePtr destination = NullFC);

    bool removeMipmap (void);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                   Read/Write                                 */
    /*! \{                                                                 */

    bool write (const Char8 *fileName);
    bool read  (const Char8 *fileName);

    bool hasSequenceRange() const;
    void getSequenceRange(int &first, int &last, int &count) const;
    bool readSequence(UInt32 imageNumber, bool firstImage = false);
    bool setInlineSequence(bool s);
    bool getInlineSequence();
    bool hasSequenceOnDisk();

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                 Storage/Restore                              */
    /*! \{                                                                 */

    UInt64 store   (const Char8 *mimeType, UInt8* mem, Int32 memSize = -1);
    UInt64 restore (const UInt8* mem, Int32 memSize = -1);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name               Comparison/Assign                              */
    /*! \{                                                                 */

    Image &operator =  (const Image &image);
    bool   operator <  (const Image &image);
    bool   operator == (const Image &image);
    bool   operator != (const Image &image);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                    Get  Methods                              */
    /*! \{                                                                 */

    inline bool   isValid           (void) const;
           bool   hasAlphaChannel   (void);
           bool   isAlphaBinary     (void);
           bool   hasColorChannel   (void);
           bool   hasCompressedData (void);
           UInt8  getComponents     (void) const;

           bool   calcIsAlphaBinary (void);

    const UChar8* getICCProfile		(UInt32 &profileLength) const;

    void                setCDF( const FieldContainerPtr &value);
     void               setImageSequence( const ImageSequencePtr &value);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                      Size                                    */
    /*! \{                                                                 */

    inline SizeT getSize ( bool withMipmap = true,
                           bool withFrames = true,
                           bool withSides  = true ) const;

    inline SizeT getRawSize ( bool withMipmap = true,
                           bool withFrames = true,
                           bool withSides  = true ) const;

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                   Get Methods                                */
    /*! \{                                                                 */

    const UInt8 *getData ( UInt32 mipmapNum = 0,
                           UInt32 frameNum = 0,
                           UInt32 sideNum = 0) const;
          UInt8 *editData( UInt32 mipmapNum = 0,
                           UInt32 frameNum = 0,
                           UInt32 sideNum = 0);
#ifndef OSG_2_PREP
          UInt8 *getData ( UInt32 mipmapNum = 0,
                           UInt32 frameNum = 0,
                           UInt32 sideNum = 0);
#endif

    UInt8 *getRawData ( UInt32 mipmapNum = 0,
                           UInt32 frameNum = 0,
                           UInt32 sideNum = 0);

    const UInt8 *getRawData ( UInt32 mipmapNum = 0,
                           UInt32 frameNum = 0,
                           UInt32 sideNum = 0) const;

    void clearUncompressedDataCache();

#ifndef OSG_2_PREP
          UInt8 *getDataByTime(Time time, UInt32 mipmapNum = 1);
#else
    const UInt8 *getDataByTime(Time time, UInt32 mipmapNum = 1) const;
#endif

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                      Calculate                               */
    /*! \{                                                                 */

    void   calcMipmapGeometry   ( UInt32 mipmapNum,
                                  UInt32 &width,
                                  UInt32 &height,
                                  UInt32 &depth       ) const;

    UInt32 calcMipmapLevelCount ( void                       ) const;
    UInt32 calcFrameNum         ( Time time, bool loop = true) const;

    UInt32 calcCRC(bool fast = true);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                   Clear Image                                */
    /*! \{                                                                 */

    virtual void clear (UChar8 pixelValue = 0);
    virtual void clearFloat (Real32 pixelValue = 0.0);
    virtual void clearHalf (Real16 pixelValue = Real16( 0.0 ));

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name               attachment handling                            */
    /*! \{                                                                 */

    bool   hasAttachment   (void) const;
    UInt32 attachmentCount (void) const;
    void   setAttachmentField ( const std::string &key,
                                const std::string &data);
    const std::string * findAttachmentField ( const std::string &key) const;
    void                getAttachmentFields(std::vector<std::string> &keys, std::vector<std::string> &values) const;

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name               Calculate Mipmap Size                          */
    /*! \{                                                                 */

    SizeT calcMipmapLevelSize( UInt32 mipmapNum,
                               UInt32 w, UInt32 h, UInt32 d, bool forceUncompressed = false) const;
    SizeT calcMipmapLevelSize( UInt32 mipmapNum, bool forceUncompressed = false) const;

    SizeT calcMipmapSumSize  ( UInt32 mipmapNum,
                               UInt32 w, UInt32 h, UInt32 d, bool forceUncompressed = false) const;
    SizeT calcMipmapSumSize  ( UInt32 mipmapNum, bool forceUncompressed = false) const;

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                Set / Get Parents                             */
    /*! \{                                                                 */

    inline void addParent(const FieldContainerPtr &parent);
    inline void subParent(const FieldContainerPtr &parent);

    bool hasContentChanged();

    static void parseFilenameDigits(const std::string &filename, int &digit_pos, int& number_length);
    static bool addImageNumber(std::string &filename, UInt32 imageNumber);
    static int getImageSequencePosition(const std::string &filename);

    /*! \}                                                                 */

    inline bool isNeverInline() const;
    inline void setNeverInline(bool state); 

    /*=========================  PROTECTED  ===============================*/
  protected:

    /*---------------------------------------------------------------------*/
    /*! \name                  static element                              */
    /*! \{                                                                 */

    static Int32 _formatDic[][2];
    static Int32 _typeDic[][2];

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                  Constructors                                */
    /*! \{                                                                 */

    Image(void);
    Image(const Image &source);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                   Destructors                                */
    /*! \{                                                                 */

    virtual ~Image(void);

    virtual void onDestroy(void);
    /*! \}                                                                 */

#if defined(OSG_FIXED_MFIELDSYNC)
    virtual void onDestroyAspect(UInt32 uiId, UInt32 uiAspect);
#endif

    /*==========================  PRIVATE  ================================*/
  private:
    /*---------------------------------------------------------------------*/
    /*! \name                   Image Data                                 */
    /*! \{                                                                 */

    bool createData ( const UInt8 *data, bool allocMem = true );
    bool scaleData  ( const UInt8* srcData,
                      Int32 srcW, Int32 srcH, Int32 srcD,
                      UInt8* destData,
                      Int32 destW, Int32 destH, Int32 destD );

    void calcMipmapOffsets(void);

    std::vector<std::filesystem::path> getFilenames(int last);

    /*! \}                                                                 */

    friend class FieldContainer;
    friend class ImageBase;

    std::vector<SizeT> _mipmapOffset;

    UInt8 *_external_data;

    bool        m_imageInfoPresent;
    osg::Vec2f  m_imageValueRange;

    bool        _shareable;
    bool        _internal;
    bool        _contentChanged;
    bool        _neverInline;

    mutable bool _uncompressedDataDirty;
    mutable std::vector<UInt8> _uncompressedData;
    mutable std::recursive_mutex _uncompressedDataMutex;
    mutable UInt32 _uncompressedPixelFormat;

    static void initMethod(void);
};

typedef Image *ImageP;

OSG_END_NAMESPACE

#include <OSGImageBase.inl>
#include <OSGImage.inl>

#define OSGIMAGE_HEADER_CVSID "@(#)$Id: $"

#endif /* _OSGIMAGE_H_ */
