/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright (C) 2000-2002 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

//-------------------------------
// Includes
//-------------------------------
#include <stdlib.h>
#include <stdio.h>
#include <limits.h>

#include <sstream>

#include "OSGConfig.h"
#include "OSGFileSystem.h"

#ifdef   OSG_SGI_LIB
#include <limits>
#endif

#include "OSGJPGImageFileType.h"

#include <OSGLog.h>

#include <iostream>

#ifdef OSG_WITH_JPG
extern "C" {

#ifdef WIN32
#define __WIN32__
#endif

#ifdef __APPLE__
// CONFLICT WITH TBB ON macOS
// tbb defines TRUE and FALSE as 1 and 0 but jmorecfg.h defines it as an enum
#undef TRUE
#undef FALSE
#endif

#include <setjmp.h>
#include <jpeglib.h>
}
#endif

#ifndef OSG_DO_DOC
#    ifdef OSG_WITH_JPG
#        define OSG_JPG_ARG(ARG) ARG
#    else
#        define OSG_JPG_ARG(ARG)
#    endif
#else
#    define OSG_JPG_ARG(ARG) ARG
#endif

/*
 * Since an ICC profile can be larger than the maximum size of a JPEG marker
 * (64K), we need provisions to split it into multiple markers.  The format
 * defined by the ICC specifies one or more APP2 markers containing the
 * following data:
 *	Identifying string	ASCII "ICC_PROFILE\0"  (12 bytes)
 *	Marker sequence number	1 for first APP2, 2 for next, etc (1 byte)
 *	Number of markers	Total number of APP2's used (1 byte)
 *      Profile data		(remainder of APP2 data)
 * Decoders should use the marker sequence numbers to reassemble the profile,
 * rather than assuming that the APP2 markers appear in the correct sequence.
 */

#define ICC_MARKER  (JPEG_APP0 + 2)	/* JPEG marker code for ICC */
#define ICC_OVERHEAD_LEN  14		/* size of non-profile data in APP2 */
#define MAX_BYTES_IN_MARKER  65533	/* maximum data len of a JPEG marker */
#define MAX_DATA_BYTES_IN_MARKER  (MAX_BYTES_IN_MARKER - ICC_OVERHEAD_LEN)
#define MAX_SEQ_NO  255				/* sufficient since marker numbers are bytes */

OSG_USING_NAMESPACE

/*! \class osg::JPGImageFileType
    \ingroup GrpSystemImage

Image File Type to read/write and store/restore Image objects as
JPEG data.

To be able to load JPEG images you need the IJG JPEG library,
version 6 or later (check the Prerequisites page on www.opensg.org).
The lib comes with all Linux distributions.

You have to --enable-jpg in the configure line to enable
the singleton object.

*/

#ifdef OSG_WITH_JPG

static const unsigned long BUFFERSIZE = 4096;

typedef struct SourceManager
{
    struct jpeg_source_mgr pub;
    std::istream *is;
    char *buffer;
    SourceManager(j_decompress_ptr cinfo, std::istream &is);
} SourceManager;

static void istream_init_source(j_decompress_ptr cinfo) {} // no action necessary

static boolean istream_fill_input_buffer(j_decompress_ptr cinfo)
{
    SourceManager *sourceManager = reinterpret_cast<SourceManager*>(cinfo->src);

    sourceManager->is->read(sourceManager->buffer, BUFFERSIZE);

    cinfo->src->next_input_byte =
        reinterpret_cast<const JOCTET*>(sourceManager->buffer);

    if (sourceManager->is->gcount() == 0)
    {
        /* Insert a fake EOI marker */
        sourceManager->buffer[0] = JOCTET(0xFF);
        sourceManager->buffer[1] = JOCTET(JPEG_EOI);
        cinfo->src->bytes_in_buffer = 2;
    }
    else
        cinfo->src->bytes_in_buffer = sourceManager->is->gcount();

    return TRUE;
}

static void istream_skip_input_data(j_decompress_ptr cinfo, long num_bytes)
{
    if (static_cast<unsigned long>(num_bytes) <= cinfo->src->bytes_in_buffer)
    {
        cinfo->src->bytes_in_buffer -= num_bytes;
        cinfo->src->next_input_byte += num_bytes;
    }
    else
    {
        num_bytes -= cinfo->src->bytes_in_buffer;
        SourceManager *sourceManager = reinterpret_cast<SourceManager*>(cinfo->src);
        sourceManager->is->ignore(num_bytes);
        cinfo->src->bytes_in_buffer = 0;
        cinfo->src->next_input_byte = 0;
    }
}

static void istream_term_source(j_decompress_ptr cinfo) {} // no action necessary

SourceManager::SourceManager(j_decompress_ptr cinfo, std::istream &is)
{
    pub.init_source = istream_init_source;
    pub.fill_input_buffer = istream_fill_input_buffer;
    pub.skip_input_data = istream_skip_input_data;
    pub.resync_to_restart = jpeg_resync_to_restart; /* use default method */
    pub.term_source = istream_term_source;
    pub.bytes_in_buffer = 0; /* forces fill_input_buffer on first read */
    pub.next_input_byte = 0; /* until buffer loaded */
    this->is = &is;
    buffer = static_cast<char*>((*cinfo->mem->alloc_small)(reinterpret_cast<j_common_ptr>(cinfo), JPOOL_IMAGE, BUFFERSIZE));
}

typedef struct DestinationManager
{
    struct jpeg_destination_mgr pub;
    std::ostream *os;
    char *buffer;
    DestinationManager(j_compress_ptr cinfo, std::ostream &os);
} DestinationManager;

static void ostream_init_destination(j_compress_ptr cinfo) {} // no action necessary

static boolean ostream_empty_output_buffer(j_compress_ptr cinfo)
{
    DestinationManager *destinationManager = reinterpret_cast<DestinationManager*>(cinfo->dest);

    destinationManager->os->write(destinationManager->buffer, BUFFERSIZE);

    destinationManager->pub.next_output_byte = reinterpret_cast<JOCTET*>(destinationManager->buffer);
    destinationManager->pub.free_in_buffer = BUFFERSIZE;

    return destinationManager->os->good() != false ? TRUE : FALSE;
}

static void ostream_term_destination(j_compress_ptr cinfo)
{
    DestinationManager *destinationManager = reinterpret_cast<DestinationManager*>(cinfo->dest);

    if(destinationManager->pub.free_in_buffer > 0)
    {
        destinationManager->os->write(destinationManager->buffer, BUFFERSIZE - destinationManager->pub.free_in_buffer);
    }
}

DestinationManager::DestinationManager(j_compress_ptr cinfo, std::ostream &os)
{
    pub.init_destination = ostream_init_destination;
    pub.empty_output_buffer = ostream_empty_output_buffer;
    pub.term_destination = ostream_term_destination;
    this->os = &os;
    buffer = static_cast<char*>((*cinfo->mem->alloc_small)(reinterpret_cast<j_common_ptr>(cinfo), JPOOL_IMAGE, BUFFERSIZE));
    pub.free_in_buffer = BUFFERSIZE;
    pub.next_output_byte = reinterpret_cast<JOCTET *>(buffer);
}

struct osg_jpeg_error_mgr
{
    struct jpeg_error_mgr pub;
    jmp_buf setjmp_buffer;
};

static void osg_jpeg_error_exit(j_common_ptr cinfo)
{
    /* Always display the message */
    (*cinfo->err->output_message) (cinfo);

    /* Let the memory manager delete any temp files before we die */
    jpeg_destroy(cinfo);

    /* Return control to the setjmp point */
    struct osg_jpeg_error_mgr *osgerr = reinterpret_cast<struct osg_jpeg_error_mgr *>(cinfo->err);
    longjmp(osgerr->setjmp_buffer, 1);
}

static void osg_jpeg_output_message(j_common_ptr cinfo)
{
    char buffer[JMSG_LENGTH_MAX];

    /* Create the message */
    (*cinfo->err->format_message) (cinfo, buffer);

    /* Send it to the OSG logger, adding a newline */
    FFATAL (("JPG: %s\n", buffer));
}

struct jpeg_mem
{
    struct jpeg_destination_mgr dest;
    struct jpeg_source_mgr      src;
    UChar8                      *buffer;
    UInt32                      memSize;
    UInt32                      dataSize;
} jpeg_mem;

/* */
static void jpeg_mem_init_source(j_decompress_ptr OSG_CHECK_ARG(cinfo))
{
    jpeg_mem.src.next_input_byte = static_cast<JOCTET *>(jpeg_mem.buffer);
    jpeg_mem.src.bytes_in_buffer = size_t(jpeg_mem.dataSize);
}

/* */
static boolean jpeg_mem_fill_input_buffer(j_decompress_ptr OSG_CHECK_ARG(cinfo))
{
    SFATAL << "Missing data. Given data block too small." << std::endl;
    return FALSE;
}

/* */
static void jpeg_mem_skip_input_data(j_decompress_ptr OSG_CHECK_ARG(cinfo    ),
                                     long                           num_bytes)
{
    jpeg_mem.src.next_input_byte += num_bytes;
    jpeg_mem.src.bytes_in_buffer -= num_bytes;
}

/* */
static boolean jpeg_mem_resync_to_restart(j_decompress_ptr OSG_CHECK_ARG(cinfo  ),
                                   int              OSG_CHECK_ARG(desired))
{
    return FALSE;
}

/* */
static void jpeg_mem_term_source(j_decompress_ptr OSG_CHECK_ARG(cinfo))
{
}

/* */
static void jpeg_mem_init_destination(j_compress_ptr OSG_CHECK_ARG(cinfo))
{
    jpeg_mem.dest.next_output_byte = static_cast<JOCTET *>(jpeg_mem.buffer);
    jpeg_mem.dest.free_in_buffer = size_t(jpeg_mem.memSize);
}

/* */
static boolean jpeg_mem_empty_output_buffer(j_compress_ptr OSG_CHECK_ARG(cinfo))
{
    SFATAL << "Not enough space left in buffer." << std::endl;
    return FALSE;
}

/* */
static void jpeg_mem_term_destination(j_compress_ptr OSG_CHECK_ARG(cinfo))
{
    jpeg_mem.dataSize = (static_cast<UChar8 *>(jpeg_mem.dest.next_output_byte)) - (static_cast<UChar8 *>(jpeg_mem.buffer));
}

/* */
static void jpeg_memory_dest(struct jpeg_compress_struct *cinfo,
                             UChar8               *buffer,
                             UInt32                memSize)
{
    jpeg_mem.buffer=buffer;
    jpeg_mem.memSize=memSize;
    jpeg_mem.dest.init_destination    = jpeg_mem_init_destination;
    jpeg_mem.dest.empty_output_buffer = jpeg_mem_empty_output_buffer;
    jpeg_mem.dest.term_destination    = jpeg_mem_term_destination;
    cinfo->dest=&jpeg_mem.dest;
}

/* */
static void jpeg_memory_src(struct jpeg_decompress_struct *cinfo,
                     const  UChar8                 *buffer,
                            UInt32                  dataSize)
{
    jpeg_mem.buffer = const_cast < UChar8 * > (buffer);
    jpeg_mem.dataSize = dataSize;
    jpeg_mem.src.init_source       = jpeg_mem_init_source;
    jpeg_mem.src.fill_input_buffer = jpeg_mem_fill_input_buffer;
    jpeg_mem.src.skip_input_data   = jpeg_mem_skip_input_data;
    jpeg_mem.src.resync_to_restart = jpeg_mem_resync_to_restart;
    jpeg_mem.src.term_source       = jpeg_mem_term_source;
    cinfo->src = &jpeg_mem.src;
}


void write_icc_profile (j_compress_ptr cinfo,
                        const JOCTET *icc_data_ptr,
                        UInt32 icc_data_len)
{
    UInt32 num_markers;		/* total number of markers we'll write */
    Int32 cur_marker = 1;		/* per spec, counting starts at 1 */
    UInt32 length;			/* number of bytes to write in this marker */

    /* Calculate the number of markers we'll need, rounding up of course */
    num_markers = icc_data_len / MAX_DATA_BYTES_IN_MARKER;
    if (num_markers * MAX_DATA_BYTES_IN_MARKER != icc_data_len)
        num_markers++;

    while (icc_data_len > 0)
    {
        /* length of profile to put in this marker */
        length = icc_data_len;
        if (length > MAX_DATA_BYTES_IN_MARKER)
            length = MAX_DATA_BYTES_IN_MARKER;

        icc_data_len -= length;

        /* Write the JPEG marker header (APP2 code and marker length) */
        jpeg_write_m_header(cinfo, ICC_MARKER, (UInt32)(length + ICC_OVERHEAD_LEN));

        /* Write the marker identifying string "ICC_PROFILE" (null-terminated).
         * We code it in this less-than-transparent way so that the code works
         * even if the local character set is not ASCII.
         */
        jpeg_write_m_byte(cinfo, 0x49);
        jpeg_write_m_byte(cinfo, 0x43);
        jpeg_write_m_byte(cinfo, 0x43);
        jpeg_write_m_byte(cinfo, 0x5F);
        jpeg_write_m_byte(cinfo, 0x50);
        jpeg_write_m_byte(cinfo, 0x52);
        jpeg_write_m_byte(cinfo, 0x4F);
        jpeg_write_m_byte(cinfo, 0x46);
        jpeg_write_m_byte(cinfo, 0x49);
        jpeg_write_m_byte(cinfo, 0x4C);
        jpeg_write_m_byte(cinfo, 0x45);
        jpeg_write_m_byte(cinfo, 0x0);

        /* Add the sequencing info */
        jpeg_write_m_byte(cinfo, cur_marker);
        jpeg_write_m_byte(cinfo, (int) num_markers);

        /* Add the profile data */
        while (length--)
        {
            jpeg_write_m_byte(cinfo, *icc_data_ptr);
            icc_data_ptr++;
        }

        cur_marker++;
    }
}

static boolean marker_is_icc (jpeg_saved_marker_ptr marker)
{
  return
    (marker->marker == ICC_MARKER &&
    marker->data_length >= ICC_OVERHEAD_LEN &&
    /* verify the identifying string */
    GETJOCTET(marker->data[0]) == 0x49 &&
    GETJOCTET(marker->data[1]) == 0x43 &&
    GETJOCTET(marker->data[2]) == 0x43 &&
    GETJOCTET(marker->data[3]) == 0x5F &&
    GETJOCTET(marker->data[4]) == 0x50 &&
    GETJOCTET(marker->data[5]) == 0x52 &&
    GETJOCTET(marker->data[6]) == 0x4F &&
    GETJOCTET(marker->data[7]) == 0x46 &&
    GETJOCTET(marker->data[8]) == 0x49 &&
    GETJOCTET(marker->data[9]) == 0x4C &&
    GETJOCTET(marker->data[10]) == 0x45 &&
    GETJOCTET(marker->data[11]) == 0x0) ? TRUE : FALSE;
}

boolean read_icc_profile (j_decompress_ptr cinfo, JOCTET **icc_data_ptr, UInt32 *icc_data_len)
{
    jpeg_saved_marker_ptr marker;
    Int32 num_markers = 0;
    Int32 seq_no;
    JOCTET *icc_data;
    UInt32 total_length;
    Char8 marker_present[MAX_SEQ_NO+1];		/* 1 if marker found */
    UInt32 data_length[MAX_SEQ_NO+1];		/* size of profile data in marker */
    UInt32 data_offset[MAX_SEQ_NO+1];		/* offset for data in marker */

    *icc_data_ptr = NULL;		/* avoid confusion if FALSE return */
    *icc_data_len = 0;

    /* This first pass over the saved markers discovers whether there are
    * any ICC markers and verifies the consistency of the marker numbering.
    */

    for (seq_no = 1; seq_no <= MAX_SEQ_NO; seq_no++)
        marker_present[seq_no] = 0;

    for (marker = cinfo->marker_list; marker != NULL; marker = marker->next)
    {
        if (marker_is_icc(marker))
        {
            if (num_markers == 0)
                num_markers = GETJOCTET(marker->data[13]);
            else if (num_markers != GETJOCTET(marker->data[13]))
                return FALSE;		/* inconsistent num_markers fields */

            seq_no = GETJOCTET(marker->data[12]);

            if (seq_no <= 0 || seq_no > num_markers)
                return FALSE;		/* bogus sequence number */
            if (marker_present[seq_no])
                return FALSE;		/* duplicate sequence numbers */

            marker_present[seq_no] = 1;
            data_length[seq_no] = marker->data_length - ICC_OVERHEAD_LEN;
        }
    }

    if (num_markers == 0)
        return FALSE;

    /* Check for missing markers, count total space needed,
    * compute offset of each marker's part of the data.
    */

    total_length = 0;
    for (seq_no = 1; seq_no <= num_markers; seq_no++)
    {
        if (marker_present[seq_no] == 0)
            return FALSE;		/* missing sequence number */

        data_offset[seq_no] = total_length;
        total_length += data_length[seq_no];
    }

    if (total_length <= 0)
        return FALSE;		/* found only empty markers? */

    /* Allocate space for assembled data */
    icc_data = (JOCTET *) malloc(total_length * sizeof(JOCTET));
    if (icc_data == NULL)
        return FALSE;		/* oops, out of memory */

    /* and fill it in */
    for (marker = cinfo->marker_list; marker != NULL; marker = marker->next)
    {
        if (marker_is_icc(marker))
        {
            JOCTET FAR *src_ptr;
            JOCTET *dst_ptr;
            unsigned int length;
            seq_no = GETJOCTET(marker->data[12]);
            dst_ptr = icc_data + data_offset[seq_no];
            src_ptr = marker->data + ICC_OVERHEAD_LEN;
            length = data_length[seq_no];

            while (length--)
                *dst_ptr++ = *src_ptr++;
        }
    }

    *icc_data_ptr = icc_data;
    *icc_data_len = total_length;

    return TRUE;
}

// EXIF + XMP meta data

typedef unsigned short uint16_t;
typedef unsigned char uint8_t;
typedef unsigned int uint32_t;

#define INTEL_BYTE_ORDER 0x4949
#define EXIF_OFFSET 0x8769 // offset to Exif SubIFD
#define APP1 JPEG_APP0+1

// Read a 16-bit word
static
    uint16_t read16(uint8_t* arr, int pos,  int swapBytes)
{
    uint8_t b1 = arr[pos];
    uint8_t b2 = arr[pos+1];

    return (swapBytes) ?  ((b2 << 8) | b1) : ((b1 << 8) | b2);
}


// Read a 32-bit word
static
    uint32_t read32(uint8_t* arr, int pos,  int swapBytes)
{

    if(!swapBytes) {

        return (arr[pos]   << 24) |
            (arr[pos+1] << 16) |
            (arr[pos+2] << 8) |
            arr[pos+3];
    }

    return arr[pos] |
        (arr[pos+1] << 8) |
        (arr[pos+2] << 16) |
        (arr[pos+3] << 24);
}

static
    char* readASCII(uint8_t* arr, int pos, int count)
{
    char* dest = new char[count+1];
    memset(dest, 0, count+1);
    memcpy(dest, (char*) (arr+pos), count);
    return dest;
}

// Tag Data Types:
// 1 = Byte
// 2 = ASCII, 8 bit byte containing 7 bit Ascii code
// 3 = SHORT, 16-bit (2 byte) unsigned int
// 4 = LONG, 32 bit (4 byte) unsigned int
// 5 = RATIONAL, two LONGs (8 byte), first numerator, second denominator
// 7 = UNDEFINED, 8 bit byte with any value depending on the field definition
// 9 = SLONG, signed 32 bit integer
// 10 = RATIONAL, 2 SLONGS, first numerator, second denominator

// Currently no support for SLONG (9) and SRATIONAL (10)
static
    int read_tag(uint8_t* arr, int pos,  int swapBytes, void* dest)
{
    uint32_t format = read16(arr, pos + 2, swapBytes); // bytes 2-3
    uint32_t components = read32(arr, pos + 4, swapBytes); // bytes 4-7
    uint32_t offset; // Points to the value

    // if the value fits in 4 bytes, the value itself is recorded
    // otherwise it is an offset to the position where the value is recorded
    static int numBytes[11] = {0, 1, 1, 2, 4, 8, 0, 1, 0, 4, 8};
    if ((numBytes[format] * components) <= 4)
        offset = pos + 8;
    else
        offset =  read32(arr, pos + 8, swapBytes); // value offset

    //std::cout << "read_tag format count offset: " << format << " " << components << " " << offset << " |pos " << (pos+8) << std::endl;

    switch (format)
    {
    case 5: // Rational
        {
            int32_t num = read32(arr, offset, swapBytes);
            int32_t den = read32(arr, offset + 4, swapBytes);
            if (den <= 0)
                return 0;

            *(double*) dest = (double)num / den;
        }
        break;
    case 2: // ascii
        *(char**) dest = readASCII(arr, offset, components);
        break;
    case 3: // uint 16
        *(int*) dest = read16(arr, offset, swapBytes);
        break;
    case 4: // uint 32
        *(uint32_t*) dest = read32(arr, offset, swapBytes);
        break;
    default:  return 0;
    }

    return 1;
}

const char exifId[] = "Exif\0\0"; // APP1
const char jfifId[] = "JFIF\0";
const char xmpId[]  = "http://ns.adobe.com/xap/1.0/\0"; // APP1, XMP namespace URI, used as unique ID

// Handler for EXIF data
static
    boolean handle_jpg_baseline_and_exif(struct jpeg_decompress_struct* cinfo, ImagePtr image/*std::string& exifString*/)
{
    boolean ret = FALSE;
    jpeg_saved_marker_ptr ptr;
    uint32_t ifd_ofs;
    int pos = 0, swapBytes = 0;
    uint32_t i, numEntries;

    for (ptr = cinfo ->marker_list; ptr; ptr = ptr ->next) {

        if ((ptr ->marker == APP1) && ptr ->data_length > 6) {
            JOCTET FAR* data = ptr -> data;

            //The length value is 2 (the length field itself) plus the length of the namespace field, plus the length of the data in bytes.
            //std::cout << "data_length " << ptr ->data_length << std::endl;

            if (memcmp(data, exifId, 6) == 0)
            {
                data += 6; // Skip EXIF marker

                // 8 byte TIFF header
                // first two determine byte order
                pos = 0;
                if (read16(data, pos, 0) == INTEL_BYTE_ORDER) {
                    swapBytes = 1;
                }

                pos += 2;

                // next two bytes are always 0x002A (TIFF version)
                pos += 2;

                // offset to Image File Directory (includes the previous 8 bytes)
                ifd_ofs = read32(data, pos, swapBytes);

                numEntries = read16(data, ifd_ofs, swapBytes);

                uint32_t subIFDOffset = 0;
                bool foundSubIDfOffset = false;


                char* valueS;
                double valueD = 0.0;
                uint32_t valueI = 0;

                std::stringstream s;

                for (i=0; i < numEntries; i++) {

                    uint32_t entryOffset = ifd_ofs + 2 + (12 * i);
                    uint32_t tag = read16(data, entryOffset, swapBytes); // each tag is a unique 2-byte number

                    switch (tag) {
                    case EXIF_OFFSET:
                        foundSubIDfOffset = read_tag(data, entryOffset, swapBytes, &subIFDOffset) != 0;
                        ret = foundSubIDfOffset ? TRUE : FALSE;
                        break;
                    case 0x011a:
                        {
                            if (read_tag(data, entryOffset, swapBytes, &valueD))
                                s << "@X_RESOLUTION" << valueD << "\n";
                        }
                        break;
                    case 0x011b:
                        {
                            if (read_tag(data, entryOffset, swapBytes, &valueD))
                                s << "@Y_RESOLUTION" << valueD << "\n";
                        }
                        break;
                    case 0x0128:
                        {
                            if (read_tag(data, entryOffset, swapBytes, &valueI))
                                s << "@UNIT_RESOLUTION" << valueI << "\n";
                        }
                        break;
                    case 0x010f: // make
                        {
                            if(read_tag(data, entryOffset, swapBytes, &valueS))
                                s << "@MAKE" << valueS << "\n";
                        }
                        break;
                    case 0x0110: // model
                        {
                            if(read_tag(data, entryOffset, swapBytes, &valueS))
                                s << "@MODEL" << valueS << "\n";
                        }
                        break;
                    case 0x0131: // software
                        {
                            if(read_tag(data, entryOffset, swapBytes, &valueS))
                                s << "@SOFTWARE" << valueS << "\n";
                        }
                        break;
                    case 0x013b: // artist
                        {
                            if(read_tag(data, entryOffset, swapBytes, &valueS))
                                s << "@ARTIST" << valueS << "\n";
                        }
                        break;
                    case 0x010e: // image description
                        {
                            if(read_tag(data, entryOffset, swapBytes, &valueS))
                                s << "@IMAGEDESCRIPTION" << valueS << "\n";
                        }
                        break;
                    case 0x8298: // copyright
                        {
                            if(read_tag(data, entryOffset, swapBytes, &valueS))
                                s << "@COPYRIGHT" << valueS << "\n";
                        }
                        break;
                    case 0x0102: // bits per sample
                        {
                            if(read_tag(data, entryOffset, swapBytes, &valueI))
                                s << "@BITS_PER_SAMPLE" << valueI << "\n";
                        }
                        break;
                    case 0x0115: // sample per pixel
                        {
                            if(read_tag(data, entryOffset, swapBytes, &valueI))
                                s << "@SAMPLES_PER_PIXEL" << valueI << "\n";
                        }
                        break;

                    default:;
                    }
                }
                // Attach data to field
                if (s.rdbuf()->in_avail() != 0)
                    image->setAttachmentField("BaselineData", s.str());

                if(foundSubIDfOffset)
                {
                    //s = std::stringstream();
                    s.str("");
                    s.clear();
                    std::string exifRead;

                    ifd_ofs = subIFDOffset;
                    numEntries = read16(data, ifd_ofs, swapBytes);
                    for (i=0; i < numEntries; i++) {
                        uint32_t entryOffset = ifd_ofs + 2 + (12 * i);
                        uint32_t tag = read16(data, entryOffset, swapBytes); // each tag is a unique 2-byte number

                        // Tags are from
                        // https://sno.phy.queensu.ca/~phil/exiftool/TagNames/EXIF.html
                        // http://www.exiv2.org/tags.html

                        switch (tag) {
                        case 0x8827:  // ISO, EXIF 2.2
                        case 0x8833:  // ISO speed, EXIF 2.3
                            {
                                if (read_tag(data, entryOffset, swapBytes, &valueI))
                                    s << "@ISO_SPEED" << valueI << "\n";
                            }
                            break;
                        case 0xa001: // color space, mostly used 0x1 = sRGB
                            {
                                if (read_tag(data, entryOffset, swapBytes, &valueI))
                                    s << "@COLOR_SPACE" << valueI << "\n";
                            }
                        break;
                        case 0x829a: // exposure time
                            {
                                valueS = nullptr;
                                if (read_tag(data, entryOffset, swapBytes, &valueD))
                                    s << "@EXPOSURE_TIME" << valueD << "\n";
                            }
                            break;
                        case 0x829d: // f number
                            {
                                if (read_tag(data, entryOffset, swapBytes, &valueD))
                                    s << "@F_NUMBER" << valueD << "\n";
                            }
                            break;
                        case 0x9202: // aperture value
                            {
                                if (read_tag(data, entryOffset, swapBytes, &valueD))
                                    s << "@APERTURE" << valueD << "\n";
                            }
                            break;
                        case 0x9203: // brightness value
                            {
                                if (read_tag(data, entryOffset, swapBytes, &valueD))
                                    s << "@BRIGHTNESS" << valueD << "\n";
                            }
                            break;
                        case 0x9204: // exposure bias
                            {
                                if (read_tag(data, entryOffset, swapBytes, &valueD))
                                    s << "@EXPOSURE_BIAS" << valueD << "\n";
                            }
                            break;
                        case 0x920a: // focal length
                            {
                                if (read_tag(data, entryOffset, swapBytes, &valueD))
                                    s << "@FOCAL_LENGTH" << valueD << "\n";
                            }
                            break;
                        case 0xa405: // focal length in 35 mm film
                            {
                                if (read_tag(data, entryOffset, swapBytes, &valueI))
                                    s << "@FOCAL_LENGTH_IN_35MM_FILM" << valueI << "\n";
                            }
                            break;
                        case 0x9201: // shutter speed value
                            {
                                if (read_tag(data, entryOffset, swapBytes, &valueD))
                                    s << "@SHUTTER_SPEED" << valueD << "\n";
                            }
                            break;
                        case 0x9206: // subject distance
                            {
                                if (read_tag(data, entryOffset, swapBytes, &valueD))
                                {
                                    s << "@SUBJECT_DISTANCE";
                                    if (valueD == UINT_MAX)
                                    {
                                        s << "Infinity";
                                    }
                                    else if (valueD <= 0)
                                    {
                                        s << "Distance unknown";
                                    }
                                    else
                                    {
                                        s << valueD;
                                    }
                                    s << "\n";
                                }
                            }
                            break;
                        case 0xa20e: // FocalPlaneXResolution
                            {
                                if (read_tag(data, entryOffset, swapBytes, &valueD))
                                    s << "@FOCALPLANE_X_RESOLUTION" << valueD << "\n";
                            }
                            break;
                        case 0xa20f: // FocalPlaneYResolution
                            {
                                if (read_tag(data, entryOffset, swapBytes, &valueD))
                                    s << "@FOCALPLANE_Y_RESOLUTION" << valueD << "\n";
                            }
                            break;
                        case 0xa210: // FocalPlaneResolutionUnit
                            {
                                if (read_tag(data, entryOffset, swapBytes, &valueI))
                                {
                                    s << "@FOCALPLANE_RESOLUTION_UNIT";
                                    switch (valueI)
                                    {
                                        case 1: // No absolute unit of measurement
                                            s << "No absolute unit of measurement";
                                            break;
                                        case 2: // Inch
                                            s << "Inch";
                                            break;
                                        case 3: // Centimeter
                                            s << "Centimeter";
                                            break;
                                        default:
                                            s << "Inch";
                                            break;
                                    }
                                    s << "\n";
                                }
                            }
                            break;
                        }
                    }
                    // Attach data to field
                    if (s.rdbuf()->in_avail() != 0)
                        image->setAttachmentField("EXIFData", s.str());
                }
            }
        }
    }
    return ret;
}

// Handler for Exif and XMP data
static
    boolean Handle_XMP_EXIF(struct jpeg_decompress_struct* cinfo, std::string& xmpString)
{
    jpeg_saved_marker_ptr ptr;
    for (ptr = cinfo ->marker_list; ptr; ptr = ptr ->next)
    {
        if (ptr ->marker == APP1 ) {
            JOCTET FAR* data = ptr -> data;

            // data_length:
            // length of the namespace field,
            // plus the length of the data in bytes.

            // XMP
            if(ptr->data_length > 29 && memcmp(data, xmpId, 29) == 0)
            {
                data += 29; // Skip XMP marker
                //std::cout << "found " << xmpId << std::endl;
                UInt32 xmp_data_len = ptr ->data_length - 29;
                xmpString.append((const char*)data, xmp_data_len);
            }
            // Exif
            if (memcmp(data, exifId, 6) == 0)
            {
                data += 6; // Skip EXIF marker
                //std::cout << "found " << exifId << std::endl;

                // todo
                // ...
            }
        }
    }
    return FALSE;
}


boolean write_xmp(j_compress_ptr cinfo,
    const JOCTET *xmp_data_ptr,
    UInt32 xmp_data_len)
{
    if(xmp_data_len == 0 || xmp_data_ptr == NULL) return FALSE;

    const int xmpIdSize = 29;
    JOCTET *xmp = (JOCTET*)malloc((xmp_data_len + xmpIdSize) * sizeof(JOCTET));
    if(xmp == NULL) return FALSE;
    memcpy(xmp, xmpId, xmpIdSize);

    for(unsigned int i = 0; i < xmp_data_len; i += 65504L) {
        unsigned length = std::min<long>((long)(xmp_data_len - i), 65504L);

        memcpy(xmp + xmpIdSize, xmp_data_ptr + i, length);
        jpeg_write_marker(cinfo, APP1, xmp, (length + xmpIdSize));
    }

    free(xmp);

    return TRUE;
}


#endif

/*****************************
 *   Types
 *****************************/
/*****************************
 *  Classvariables
 *****************************/
// Static Class Varible implementations:
static const Char8                  *suffixArray[] = { "jpg", "jpeg", "jpe", "jfif" };

JPGImageFileType JPGImageFileType:: _the("image/jpeg",
                                         suffixArray, sizeof(suffixArray),
                                         OSG_READ_SUPPORTED |
                                         OSG_WRITE_SUPPORTED );

/********************************
 *  Class methodes
 *******************************/


//-------------------------------------------------------------------------
/*!
Class method to get the singleton Object
*/
JPGImageFileType& JPGImageFileType::the (void)
{
  return _the;
}

/*******************************
*public
*******************************/

void JPGImageFileType::setQuality(UInt32 cl)
{
    if(cl > 100)
        cl = 100;

    _quality = cl;
}

UInt32 JPGImageFileType::getQuality(void)
{
    return _quality;
}

#ifdef OSG_DEBUG_OLD_C_CASTS
#ifdef jpeg_create_compress
#undef jpeg_create_compress
#endif
#ifdef jpeg_create_decompress
#undef jpeg_create_decompress
#endif

#define jpeg_create_compress(cinfo) \
    jpeg_CreateCompress((cinfo), JPEG_LIB_VERSION, \
                        size_t(sizeof(struct jpeg_compress_struct)))
#define jpeg_create_decompress(cinfo) \
    jpeg_CreateDecompress((cinfo), JPEG_LIB_VERSION, \
                          size_t(sizeof(struct jpeg_decompress_struct)))
#endif

//-------------------------------------------------------------------------
/*!
Tries to fill the image object with the data read from
the given input stream. Returns true on success.
*/
bool JPGImageFileType::read(ImagePtr &OSG_JPG_ARG(image), std::istream &OSG_JPG_ARG(is),
                            const std::string &OSG_JPG_ARG(mimetype))
{
#ifdef OSG_WITH_JPG

    struct osg_jpeg_error_mgr jerr;
    struct jpeg_decompress_struct cinfo;

    cinfo.err = jpeg_std_error(&jerr.pub);
    if (setjmp(jerr.setjmp_buffer))
        return false;
    cinfo.err->error_exit = osg_jpeg_error_exit;
    cinfo.err->output_message = osg_jpeg_output_message;

    jpeg_create_decompress(&cinfo);

#if 1
    // prepare for read icc profile
    /* Tell the library to keep any APP2 data it may find */
    jpeg_save_markers(&cinfo, ICC_MARKER, 0xFFFF);
    jpeg_save_markers(&cinfo, JPEG_APP0+1, 0xFFFF); /* EXIF */ // Exif uses APP1(0xFFE1) Marker
#endif

    SourceManager *sourceManager =
        new ((*cinfo.mem->alloc_small)(reinterpret_cast<j_common_ptr>(&cinfo), JPOOL_IMAGE, sizeof(SourceManager)))
        SourceManager(&cinfo, is);
    cinfo.src = reinterpret_cast<jpeg_source_mgr*>(sourceManager);

    jpeg_read_header(&cinfo, TRUE);

    handle_jpg_baseline_and_exif(&cinfo, image);

    const std::string xmpkey = "XMPData";
    const std::string* xmp = image->findAttachmentField(xmpkey);
    if(xmp != NULL) // existence of attachment field XMPData indicates that Exif/XMP data should be examined
    {
        std::string xmpRead;
        Handle_XMP_EXIF(&cinfo, xmpRead);
        image->setAttachmentField(xmpkey, xmpRead); // update
    }

#if 1
    // read icc profile
    UInt32 profileLength = 0;
    UChar8* profileBuffer = NULL;
    read_icc_profile(&cinfo, &profileBuffer, &profileLength);
    if( profileBuffer != NULL && profileLength != 0 )
        image->setICCProfile(profileBuffer, profileLength);
#endif

    jpeg_start_decompress(&cinfo);

    Image::PixelFormat pixelFormat;
    switch (cinfo.output_components)
    {
    case 1:
        pixelFormat = Image::OSG_L_PF;
        break;
    case 3:
        pixelFormat = Image::OSG_RGB_PF;
        break;
    default:
        pixelFormat = Image::OSG_INVALID_PF;
        break;
    };

    bool retCode;
    if (image->set(pixelFormat, cinfo.output_width, cinfo.output_height) == true)
    {
        Real32 res_x = Real32(cinfo.X_density);
        Real32 res_y = Real32(cinfo.Y_density);
        UInt16 res_unit = UInt16(cinfo.density_unit);
        if(res_unit == 2) // centimeter
        {
            // convert dpcm to dpi.
            res_x *= 2.54f;
            res_y *= 2.54f;
            res_unit = Image::OSG_RESUNIT_INCH;
        }
        image->setResX(res_x);
        image->setResY(res_y);
        image->setResUnit(res_unit);

        unsigned char *destData = image->editData() + image->getSize();
        int row_stride = cinfo.output_width * cinfo.output_components;
        while (cinfo.output_scanline < cinfo.output_height)
        {
            destData -= row_stride;
            jpeg_read_scanlines(&cinfo, &destData, 1);
        }
        retCode = true;
    }
    else
        retCode = false;

    jpeg_finish_decompress(&cinfo);
    jpeg_destroy_decompress(&cinfo);

    return retCode;

#else

    SWARNING <<
        getMimeType() <<
        " read is not compiled into the current binary " <<
        std::endl;
    return false;

#endif
}

//-------------------------------------------------------------------------
/*!
Tries to write the image object to the given output stream.
Returns true on success.
*/
bool JPGImageFileType::write(const ImagePtr &OSG_JPG_ARG(image), std::ostream &OSG_JPG_ARG(os),
                             const std::string &OSG_JPG_ARG(mimetype))
{
#ifdef OSG_WITH_JPG

    if (image->getDepth() != 1)
    {
        SWARNING <<
            getMimeType() <<
            " JPEG write only works for 2D images " <<
            std::endl;
        return false;
    }

    ImagePtr img = image;
    if(image->getBpp() != 1 && image->getBpp() != 3)
    {
        // create temporary image with bpp == 3
        img = ImagePtr::dcast(osg::deepClone(image));
        img->reformat(osg::Image::OSG_RGB_PF);
    }

    struct osg_jpeg_error_mgr jerr;
    struct jpeg_compress_struct cinfo;

    cinfo.err = jpeg_std_error(&jerr.pub);
    if (setjmp(jerr.setjmp_buffer))
        return false;
    cinfo.err->error_exit = osg_jpeg_error_exit;
    cinfo.err->output_message = osg_jpeg_output_message;

    jpeg_create_compress(&cinfo);

    DestinationManager *destinationManager =
        new ((*cinfo.mem->alloc_small)(reinterpret_cast<j_common_ptr>(&cinfo), JPOOL_IMAGE, sizeof(DestinationManager)))
        DestinationManager(&cinfo, os);
    cinfo.dest = reinterpret_cast<jpeg_destination_mgr*>(destinationManager);

    cinfo.image_width = img->getWidth();
    cinfo.image_height = img->getHeight();
    cinfo.input_components = img->getBpp();
    cinfo.in_color_space = (img->getBpp() == 1) ? JCS_GRAYSCALE : JCS_RGB;

    jpeg_set_defaults(&cinfo);
    jpeg_set_quality(&cinfo, _quality, TRUE);

    cinfo.density_unit = 1;  // dpi
    cinfo.X_density = UInt16(img->getResX() < 0.0f ?
                             img->getResX() - 0.5f :
                             img->getResX() + 0.5f);
    cinfo.Y_density = UInt16(img->getResY() < 0.0f ?
                             img->getResY() - 0.5f :
                             img->getResY() + 0.5f);

    jpeg_start_compress(&cinfo, TRUE);

    const std::string *comment = img->findAttachmentField("Comment");
    if(comment != NULL)
        jpeg_write_marker(&cinfo, JPEG_COM, (JOCTET *) comment->c_str(), comment->length());

    const std::string *xmp = img->findAttachmentField("XMPData");
    if(xmp != NULL)
        write_xmp(&cinfo, (JOCTET *) xmp->c_str(), xmp->length());

#if 1
    // embed icc profile
    UInt32 profileLength = 0;
    const UChar8* profile = img->getICCProfile(profileLength);
    if( profile != NULL && profileLength != 0 )
        write_icc_profile(&cinfo, profile, profileLength);
#endif

    bool ok = true;
    const unsigned char *srcData = img->getData() + (Int64(img->getHeight()) * Int64(img->getWidth()) * Int64(img->getBpp()));
    int row_stride = cinfo.image_width * cinfo.input_components;
    while (cinfo.next_scanline < cinfo.image_height)
    {
        srcData -= row_stride;
        if(jpeg_write_scanlines(&cinfo, const_cast<unsigned char **>(&srcData), 1) != 1)
        {
            ok = false;
            break;
        }
    }

    jpeg_finish_compress(&cinfo);
    jpeg_destroy_compress(&cinfo);

    // destroy temporary image
    if(img != image)
        subRefCP(img);

    return ok;

#else

    SWARNING <<
        getMimeType() <<
        " write is not compiled into the current binary " <<
        std::endl;
    return false;

#endif
}

//-------------------------------------------------------------------------
/*!
Tries to determine the mime type of the data provided by an input stream
by searching for magic bytes. Returns the mime type or an empty string
when the function could not determine the mime type.
*/
std::string JPGImageFileType::determineMimetypeFromStream(std::istream &is)
{
    char filecode[2];
    is.read(filecode, 2);
    is.seekg(-2, std::ios::cur);
    return strncmp(filecode, "\xff\xd8", 2) == 0 ?
        std::string(getMimeType()) : std::string();
}

//-------------------------------------------------------------------------

bool JPGImageFileType::validateHeader( const Char8 *fileName, bool &implemented )
{
    implemented = true;

    if(fileName == NULL)
        return false;

    FILE *file = osg::fopen(fileName, "rb");
    if(file == NULL)
        return false;

    UInt16 magic = 0;
    fread(static_cast<void *>(&magic), sizeof(magic), 1, file);
    fclose(file);

#if BYTE_ORDER == LITTLE_ENDIAN
    if(magic == 0xd8ff) // the magic header is big endian need to swap it.
#else
    if(magic == 0xffd8)
#endif
    {
        return true;
    }

    return false;
}


/* */
UInt64 JPGImageFileType::restoreData(      ImagePtr &OSG_JPG_ARG(image  ),
                                     const UChar8   *OSG_JPG_ARG(buffer ),
                                           Int32     OSG_JPG_ARG(memSize))
{
#ifdef OSG_WITH_JPG
    UInt64    retCode = 0;
    struct local_error_mgr
    {
        struct jpeg_error_mgr   pub;
        jmp_buf                 setjmp_buffer;
    };

    unsigned char                   *destData;
    Image::PixelFormat              pixelFormat = osg::Image::OSG_INVALID_PF;

    unsigned long                    imageSize;
    typedef struct local_error_mgr  *local_error_ptr;
    struct local_error_mgr          jerr;
    struct jpeg_decompress_struct   cinfo;
    JSAMPARRAY                      imagebuffer;

    int                             row_stride;

    cinfo.err = jpeg_std_error(&jerr.pub);
    if(setjmp(jerr.setjmp_buffer))
    {
        jpeg_destroy_decompress(&cinfo);
        return 0;
    }

    jpeg_create_decompress(&cinfo);
    jpeg_memory_src(&cinfo, buffer, memSize);
    jpeg_read_header(&cinfo, TRUE);
    jpeg_start_decompress(&cinfo);

    switch(cinfo.output_components)
    {
    case 1:
        pixelFormat = Image::OSG_L_PF;
        break;
    case 2:
        pixelFormat = Image::OSG_LA_PF;
        break;
    case 3:
        pixelFormat = Image::OSG_RGB_PF;
        break;
    case 4:
        pixelFormat = Image::OSG_RGBA_PF;
        break;
    };

    if(image->set(pixelFormat, cinfo.output_width, cinfo.output_height))
    {
        imageSize = image->getSize();
        destData = image->editData() + imageSize;
        row_stride = cinfo.output_width * cinfo.output_components;
        imagebuffer = (*cinfo.mem->alloc_sarray) (reinterpret_cast<j_common_ptr>(&cinfo), JPOOL_IMAGE, row_stride, 1);
        while(cinfo.output_scanline < cinfo.output_height)
        {
            destData -= row_stride;
            jpeg_read_scanlines(&cinfo, imagebuffer, 1);
            memcpy(destData, *imagebuffer, row_stride);
        }

        retCode = imageSize;
    }
    else
        retCode = 0;

    jpeg_finish_decompress(&cinfo);
    jpeg_destroy_decompress(&cinfo);

    return retCode;

#else
    SWARNING <<
        getMimeType() <<
        " read is not compiled into the current binary " <<
        std::endl;
    return 0;
#endif
}

//-------------------------------------------------------------------------
/*!
Tries to restore the image data from the given memblock.
Returns the amount of data read.
*/
UInt64 JPGImageFileType::storeData(const ImagePtr &OSG_JPG_ARG(image  ),
                                   UChar8         *OSG_JPG_ARG(buffer ),
                                   Int32           OSG_JPG_ARG(memSize))
{
#ifdef OSG_WITH_JPG
    if((image->getBpp() != 1 && image->getBpp() != 3)
       || image->getDepth() != 1)
    {
        SWARNING <<
            getMimeType() <<
            " JPEG storeData only works for 2D 1 or 3 bpp images " <<
            std::endl;
        return 0;
    }

    struct local_error_mgr
    {
        struct jpeg_error_mgr   pub;
        jmp_buf                 setjmp_buffer;
    };

    typedef struct local_error_mgr  *local_error_ptr;

    struct local_error_mgr          jerr;
    struct jpeg_compress_struct     cinfo;
    JSAMPARRAY                      imagebuffer;
    const UChar8                   *data;

    cinfo.err = jpeg_std_error(&jerr.pub);
    if(setjmp(jerr.setjmp_buffer))
    {
        jpeg_destroy_compress(&cinfo);
        return 0;
    }

    jpeg_create_compress(&cinfo);
    jpeg_memory_dest(&cinfo, buffer, memSize);

    cinfo.image_width = image->getWidth();
    cinfo.image_height = image->getHeight();
    cinfo.input_components = image->getBpp();
    cinfo.in_color_space = (image->getBpp() == 1) ? JCS_GRAYSCALE : JCS_RGB;

    jpeg_set_defaults(&cinfo);
    jpeg_set_quality(&cinfo, _quality, TRUE);
    jpeg_start_compress(&cinfo, TRUE);

    imagebuffer = const_cast<UChar8 **>(&data);
    while(cinfo.next_scanline < cinfo.image_height)
    {
        data = image->getData() +
            (image->getHeight() - 1 - cinfo.next_scanline) *
            image->getWidth() *
            image->getBpp();
        jpeg_write_scanlines(&cinfo, imagebuffer, 1);
    }

    jpeg_finish_compress(&cinfo);
    jpeg_destroy_compress(&cinfo);

    return jpeg_mem.dataSize;

#else
    SWARNING <<
        getMimeType() <<
        " write is not compiled into the current binary " <<
        std::endl;
    return 0;
#endif
}

//-------------------------------------------------------------------------
/*!
Constructor used for the singleton object
*/
JPGImageFileType::JPGImageFileType( const Char8 *mimeType,
                                    const Char8 *suffixArray[],
                                    UInt16 suffixByteCount,
                                    UInt32 flags) :
    ImageFileType(mimeType, suffixArray, suffixByteCount, flags),
    _quality(90)
{}

//-------------------------------------------------------------------------
/*!
Destructor
*/
JPGImageFileType::~JPGImageFileType(void) {}


