/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright (C) 2000-2002 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

//-------------------------------
// Includes
//-------------------------------

#ifdef __hpux // prevent int32 clash (model.h/tiff.h)
#define _INT32
#endif

#include <stdlib.h>
#include <stdio.h>
#include <limits.h>

#include <sstream>

#include "OSGConfig.h"
#include "OSGFileSystem.h"

#include "OSGTIFImageFileType.h"

#ifdef OSG_WITH_TIF
#include <tiff.h>
#include <tiffio.h>
#include <tiffvers.h>

#if defined(TIFFLIB_VERSION)
// Those symbols are in 4.6 but not in 4.0
#if  TIFFLIB_VERSION < 20230908
/*
 * TIFF/EP tags equivalent to EXIF tags
 *     Note that TIFF-EP and EXIF use nearly the same metadata tag set, but TIFF-EP stores the tags in IFD 0,
 *     while EXIF store the tags in a separate IFD. Either location is allowed by DNG, but the EXIF location is preferred.
 */
#define TIFFTAG_EP_EXPOSURETIME 33434             /* Exposure time */
#define TIFFTAG_EP_FNUMBER 33437                  /* F number */
#define TIFFTAG_EP_EXPOSUREPROGRAM 34850          /* Exposure program */
#define TIFFTAG_EP_SPECTRALSENSITIVITY 34852      /* Spectral sensitivity */
#define TIFFTAG_EP_ISOSPEEDRATINGS 34855          /* ISO speed rating */
#define TIFFTAG_EP_OECF 34856                     /* Optoelectric conversion factor */
#define TIFFTAG_EP_DATETIMEORIGINAL 36867         /* Date and time of original data generation */
#define TIFFTAG_EP_COMPRESSEDBITSPERPIXEL 37122   /* Image compression mode */
#define TIFFTAG_EP_SHUTTERSPEEDVALUE 37377        /* Shutter speed */
#define TIFFTAG_EP_APERTUREVALUE 37378            /* Aperture */
#define TIFFTAG_EP_BRIGHTNESSVALUE 37379          /* Brightness */
#define TIFFTAG_EP_EXPOSUREBIASVALUE 37380        /* Exposure bias */
#define TIFFTAG_EP_MAXAPERTUREVALUE 37381         /* Maximum lens aperture */
#define TIFFTAG_EP_SUBJECTDISTANCE 37382          /* Subject distance */
#define TIFFTAG_EP_METERINGMODE 37383             /* Metering mode */
#define TIFFTAG_EP_LIGHTSOURCE 37384              /* Light source */
#define TIFFTAG_EP_FLASH 37385                    /* Flash */
#define TIFFTAG_EP_FOCALLENGTH 37386              /* Lens focal length */
#define TIFFTAG_EP_SUBJECTLOCATION 37396          /* Subject location (area) */

#define EXIFTAG_ISOSPEED 34867                  /* ISO speed value */
#endif
#endif

#endif
#include <OSGLog.h>

#include <iostream>

#ifndef OSG_DO_DOC
#    ifdef OSG_WITH_TIF
#        define OSG_TIF_ARG(ARG) ARG
#    else
#        define OSG_TIF_ARG(ARG)
#    endif
#else
#    define OSG_TIF_ARG(ARG) ARG
#endif


OSG_USING_NAMESPACE


/*! \class osg::TIFImageFileType
    \ingroup GrpSystemImage

Image File Type to read/write and store/restore Image objects as
TIF (tif/tiff suffix) data.

To be able to load TIFF images you need the IJG TIFF library,
(check the Prerequisites page on www.opensg.org).
The lib comes with all Linux distributions.

You have to --enable-tif in the configure line to enable
the singleton object.

*/

#ifdef OSG_WITH_TIF

static tsize_t isReadProc(thandle_t fd, tdata_t buf, tsize_t size)
{
    std::istream *is = reinterpret_cast<std::istream*>(fd);
    is->read(static_cast<char*>(buf), size);
    return is->gcount();
}

static tsize_t osReadProc(thandle_t fd, tdata_t buf, tsize_t size)
{
    return 0;
}

static tsize_t isWriteProc(thandle_t fd, tdata_t buf, tsize_t size)
{
    return 0;
}

static tsize_t osWriteProc(thandle_t fd, tdata_t buf, tsize_t size)
{
    std::ostream *os = reinterpret_cast<std::ostream*>(fd);
    os->write(static_cast<char*>(buf), size);
    return os->good() ? size : 0;
}

static toff_t isSeekProc(thandle_t fd, toff_t off, int i)
{
    std::istream *is = reinterpret_cast<std::istream*>(fd);
    switch (i)
    {
    case SEEK_SET:
        is->seekg(off, std::ios::beg);
        break;
    case SEEK_CUR:
        is->seekg(off, std::ios::cur);
        break;
    case SEEK_END:
        is->seekg(off, std::ios::end);
        break;
    }
    return is->tellg();
}

static toff_t osSeekProc(thandle_t fd, toff_t off, int i)
{
    std::ostream *os = reinterpret_cast<std::ostream*>(fd);
    switch (i)
    {
    case SEEK_SET:
        os->seekp(off, std::ios::beg);
        break;
    case SEEK_CUR:
        os->seekp(off, std::ios::cur);
        break;
    case SEEK_END:
        os->seekp(off, std::ios::end);
        break;
    }
    return os->tellp();
}

static int closeProc(thandle_t fd)
{
    return 0; // no action necessary
}

static toff_t isSizeProc(thandle_t fd)
{
    std::istream *is = reinterpret_cast<std::istream*>(fd);
    std::ios::pos_type pos = is->tellg();
    is->seekg(0, std::ios::end);
    std::ios::pos_type size = is->tellg();
    is->seekg(pos, std::ios::beg);
    return size;
}

static toff_t osSizeProc(thandle_t fd)
{
    std::ostream *os = reinterpret_cast<std::ostream*>(fd);
    std::ios::pos_type pos = os->tellp();
    os->seekp(0, std::ios::end);
    std::ios::pos_type size = os->tellp();
    os->seekp(pos, std::ios::beg);
    return size;
}

static int mapFileProc(thandle_t fd, tdata_t *buf, toff_t *size)
{
    return 0;
}

static void unmapFileProc(thandle_t fd, tdata_t buf, toff_t size) {}

static void warningHandler (const char *module, const char *fmt, va_list ap)
{
    Char8   buffer[4096];

#ifdef OSG_HAS_VSNPRINTF
    vsnprintf(buffer, sizeof(buffer) - 1, fmt, ap);
#else
    vsprintf(buffer, fmt, ap);
#endif

    FWARNING (("TiffLib: %s;%s\n", module ? module : "Mod", buffer));
}

static void errorHandler (const char *module, const char *fmt, va_list ap)
{
    Char8   buffer[4096];

#ifdef OSG_HAS_VSNPRINTF
    vsnprintf(buffer, sizeof(buffer) - 1, fmt, ap);
#else
    vsprintf(buffer, fmt, ap);
#endif

    FFATAL (("TiffLib: %s;%s\n", module ? module : "Mod", buffer));
}

void handle_tif_baseline(TIFF *in, ImagePtr image)
{
    char* valueChar;
    uint32 valueI = 0;

    // Base data
    // Used http://www.awaresystems.be/imaging/tiff/tifftags/baseline.html for tag reference.

    std::stringstream s;

    if (TIFFGetField(in, TIFFTAG_MAKE, &valueChar))
        s << "@MAKE" << valueChar << "\n";

    if (TIFFGetField(in, TIFFTAG_MODEL, &valueChar))
        s << "@MODEL" << valueChar << "\n";

    if (TIFFGetField(in, TIFFTAG_SOFTWARE, &valueChar))
        s << "@SOFTWARE" << valueChar << "\n";

    if (TIFFGetField(in, TIFFTAG_ARTIST, &valueChar))
        s << "@ARTIST" << valueChar << "\n";

    if (TIFFGetField(in, TIFFTAG_IMAGEDESCRIPTION, &valueChar))
        s << "@IMAGEDESCRIPTION" << valueChar << "\n";

    if (TIFFGetField(in, TIFFTAG_COPYRIGHT, &valueChar))
        s << "@COPYRIGHT" << valueChar << "\n";

    if (TIFFGetField(in, TIFFTAG_BITSPERSAMPLE, &valueI))
        s << "@BITS_PER_SAMPLE" << valueI << "\n";

    if (TIFFGetField(in, TIFFTAG_SAMPLESPERPIXEL, &valueI))
        s << "@SAMPLES_PER_PIXEL" << valueI << "\n";

    // Attach data to field
    if (s.rdbuf()->in_avail() != 0)
        image->setAttachmentField("BaselineData", s.str());
}

void handle_tif_exif(TIFF *in, ImagePtr image)
{
    char* valueChar;
    float valueF = 0.0;
    uint32 valueI = 0;
    std::stringstream s;

    // TIFFTAG_*
    if (TIFFGetField(in, TIFFTAG_RESOLUTIONUNIT, &valueI))
        s << "@UNIT_RESOLUTION" << valueI << "\n";

    if (TIFFGetField(in, TIFFTAG_XRESOLUTION, &valueF))
        s << "@X_RESOLUTION" << valueF << "\n";

    if (TIFFGetField(in, TIFFTAG_YRESOLUTION, &valueF))
        s << "@Y_RESOLUTION" << valueF << "\n";
    // TIFFTAG_* end

    // these are also stored in IFD 0 so trying to read these from EXIF IFD will lead to a crash!
    // TIFFTAG_EP_*
    if (TIFFGetField(in, TIFFTAG_EP_ISOSPEEDRATINGS, &valueI))
        s << "@ISO_SPEED" << valueI << "\n";

    if (TIFFGetField(in, TIFFTAG_EP_EXPOSURETIME, &valueF))
        s << "@EXPOSURE_TIME" << valueF << "\n";

    if (TIFFGetField(in, TIFFTAG_EP_FNUMBER, &valueF))
        s << "@F_NUMBER" << valueF << "\n";

    if (TIFFGetField(in, TIFFTAG_EP_FOCALLENGTH, &valueF))
        s << "@FOCAL_LENGTH" << valueF << "\n";

    if (TIFFGetField(in, TIFFTAG_EP_SHUTTERSPEEDVALUE, &valueF))
        s << "@SHUTTER_SPEED" << valueF << "\n";

    if (TIFFGetField(in, TIFFTAG_EP_APERTUREVALUE, &valueF))
        s << "@APERTURE" << valueF << "\n";

    if (TIFFGetField(in, TIFFTAG_EP_BRIGHTNESSVALUE, &valueF))
        s << "@BRIGHTNESS" << valueF << "\n";

    if (TIFFGetField(in, TIFFTAG_EP_EXPOSUREBIASVALUE, &valueF))
        s << "@EXPOSURE_BIAS" << valueF << "\n";

    if (TIFFGetField(in, TIFFTAG_EP_SUBJECTDISTANCE, &valueF))
    {
        s << "@SUBJECT_DISTANCE";
        if (valueF == UINT_MAX)
        {
            s << "Infinity";
        }
        else if (valueF <= 0)
        {
            s << "Distance unknown";
        }
        else
        {
            s << valueF;
        }
        s << "\n";
    }
    // get TIFFTAG_EP_* end

    // EXIFTAG_*
    tdir_t curdir = TIFFCurrentDirectory(in);
    uint32_t exifOffset{};
    if (TIFFGetField(in, TIFFTAG_EXIFIFD, &exifOffset))
    {
        if (TIFFReadEXIFDirectory(in, exifOffset))
        {
            if (TIFFGetField(in, EXIFTAG_ISOSPEED, &valueI))
                s << "@ISO_SPEED" << valueI << "\n";

            if (TIFFGetField(in, EXIFTAG_COLORSPACE, &valueI))
                s << "@COLOR_SPACE" << valueI << "\n";

            if (TIFFGetField(in, EXIFTAG_FOCALLENGTHIN35MMFILM, &valueI))
                s << "@FOCAL_LENGTH_IN_35MM_FILM" << valueI << "\n";

            if (TIFFGetField(in, EXIFTAG_FOCALPLANEXRESOLUTION, &valueF))
                s << "@FOCALPLANE_X_RESOLUTION" << valueF << "\n";

            if (TIFFGetField(in, EXIFTAG_FOCALPLANEYRESOLUTION, &valueF))
                s << "@FOCALPLANE_Y_RESOLUTION" << valueF << "\n";

            if (TIFFGetField(in, EXIFTAG_FOCALPLANERESOLUTIONUNIT, &valueI))
            {
                s << "@FOCALPLANE_RESOLUTION_UNIT";
                switch (valueI)
                {
                case 1: // No absolute unit of measurement
                    s << "No absolute unit of measurement";
                    break;
                case 2: // Inch
                    s << "Inch";
                    break;
                case 3: // Centimeter
                    s << "Centimeter";
                    break;
                default:
                    s << "Inch";
                    break;
                }
                s << "\n";
            }
            TIFFSetDirectory(in, curdir);
        }
    }
    // EXIFTAG_* end
 
    // Attach data to field
    if (s.rdbuf()->in_avail() != 0)
    {
        image->setAttachmentField("EXIFData", s.str());
    }
}


#endif // OSG_WITH_TIF

// Static Class Varible implementations:
static const Char8 *suffixArray[] = {
  "tif", "tiff"
};

TIFImageFileType TIFImageFileType:: _the("image/tiff",
    suffixArray, sizeof(suffixArray),
    OSG_READ_SUPPORTED |
    OSG_WRITE_SUPPORTED);

/* enum VecBase::VectorSizeE
 * brief
*/

/* var VecBase::VectorSizeE VecBase::_iSize
 *
*/

/* const char *VecBase::getClassName(void)
 *  brief Classname
*/

/* var ValueTypeT VecBase:: _value[Size];
 * brief value store
*/

/*****************************
 *   Types
 *****************************/

 /*****************************
  *  Classvariables
  *****************************/

  /********************************
   *  Class methodes
   *******************************/

//-------------------------------------------------------------------------
/*!
Class method to get the singleton Object
*/
TIFImageFileType& TIFImageFileType::the(void)
{
    return _the;
}


/*******************************
*public
*******************************/

//-------------------------------------------------------------------------
/*!
Tries to fill the image object with the data read from
the given input stream. Returns true on success.
*/
bool TIFImageFileType::read(ImagePtr &OSG_TIF_ARG(image), std::istream &OSG_TIF_ARG(is),
    const std::string &OSG_TIF_ARG(mimetype))
{
#ifdef OSG_WITH_TIF

    TIFF *in = TIFFClientOpen("dummy", "rm",
        reinterpret_cast<thandle_t>(&is),
        isReadProc, isWriteProc, isSeekProc, closeProc,
        isSizeProc, mapFileProc, unmapFileProc);
    if (in == 0)
        return false;

    uint32 w, h;
    TIFFGetField(in, TIFFTAG_IMAGEWIDTH, &w);
    TIFFGetField(in, TIFFTAG_IMAGELENGTH, &h);

    float res_x, res_y;
    uint16 res_unit;
    TIFFGetField(in, TIFFTAG_XRESOLUTION, &res_x);
    TIFFGetField(in, TIFFTAG_YRESOLUTION, &res_y);
    TIFFGetField(in, TIFFTAG_RESOLUTIONUNIT, &res_unit);

    if (res_unit == RESUNIT_CENTIMETER)
    {
        // convert it to dpi.
        res_x *= 2.54f;
        res_y *= 2.54f;
        res_unit = Image::OSG_RESUNIT_INCH;
    }

    // try to read icc profile
    UInt32 profileLength = 0;
    UChar8* profileBuffer = NULL;
    TIFFGetField(in, TIFFTAG_ICCPROFILE, &profileLength, &profileBuffer);

    // buffer data retrieved by TIFFGetField should not be accessed after TIFFClose
    // setting the icc profile in the image
    if( profileBuffer != NULL && profileLength != 0 )
        image->setICCProfile(profileBuffer, profileLength);

    // existance of attachment field XMPData indicates that Exif/XMP data should be read
    UInt32 xmpLength = 0;
    UChar8* xmpBuffer = NULL;
    const std::string xmpkey = "XMPData";
    const std::string* xmp = image->findAttachmentField(xmpkey);
    if(xmp != NULL)
    {
        if(TIFFGetField(in, TIFFTAG_XMLPACKET, &xmpLength, &xmpBuffer) && xmpBuffer != NULL && xmpLength != 0)
        {
            image->setAttachmentField(xmpkey, std::string((char*)xmpBuffer, xmpLength));
        }
    }

    handle_tif_baseline(in, image);
    handle_tif_exif(in, image);

    uint16 bpp;
    TIFFGetFieldDefaulted(in, TIFFTAG_SAMPLESPERPIXEL, &bpp);
    uint16 bps;
    uint16 spp;
    uint16 sampleFormat;
    TIFFGetField(in, TIFFTAG_BITSPERSAMPLE, &bps);
    TIFFGetField(in, TIFFTAG_SAMPLESPERPIXEL, &spp);
    TIFFGetField(in, TIFFTAG_SAMPLEFORMAT, &sampleFormat);

    if (bpp == 4)
    {   // accept unspecified extra samples as associated alpha
        uint16 *sampleinfo;
        uint16 extrasamples;
        TIFFGetFieldDefaulted(in, TIFFTAG_EXTRASAMPLES, &extrasamples,
            &sampleinfo);
        if (sampleinfo && sampleinfo[0] == EXTRASAMPLE_UNSPECIFIED)
        {
            uint16 si = EXTRASAMPLE_ASSOCALPHA;
            TIFFSetField(in, TIFFTAG_EXTRASAMPLES, 1, &si);
        }
    }

    Image::PixelFormat type = Image::OSG_INVALID_PF;
    switch (bpp)
    {
    case 1:
        type = Image::OSG_L_PF;
        break;
    case 2:
        type = Image::OSG_LA_PF;
        break;
    case 3:
        type = Image::OSG_RGB_PF;
        break;
    case 4:
        type = Image::OSG_RGBA_PF;
        break;
    }
    // ok, the library is dumb and does not know that it can actually read 32 bit floating point images
#if 0
    char errorMessage[1024];
    if (TIFFRGBAImageOK(in, errorMessage) == 0)
    {
        SWARNING << "Tiff reader failed: " << errorMessage << std::endl;
        TIFFClose(in);
        return false;
    }
#endif

    int dataType = Image::OSG_UINT8_IMAGEDATA;

    if (sampleFormat == SAMPLEFORMAT_IEEEFP)
    {
        if (bps == 24)
        {
            SWARNING << "Tiff reader does not support 24bit floating point images" << std::endl;
            TIFFClose(in);
            return false;
        }
        dataType = (bps == 16) ? Image::OSG_FLOAT16_IMAGEDATA : Image::OSG_FLOAT32_IMAGEDATA;
    }
    else
    {
        dataType = (bps == 8) ? Image::OSG_UINT8_IMAGEDATA : (bps == 16) ? Image::OSG_UINT16_IMAGEDATA : Image::OSG_UINT32_IMAGEDATA;
    }
    beginEditCP(image, osg::FieldBits::AllFields);
    image->set(type, w, h, 1, 1, 1, 0.0, 0, dataType, true, 1);
    image->setResX(res_x);
    image->setResY(res_y);
    image->setResUnit(res_unit);

#if 0 // reading scanlines only works in special cases, therefore we prefer reading strips instead
    size_t scanLineSize = TIFFScanlineSize(in);
    UChar8 *dst = image->editData();
    UChar8* buf = dst;
    buf += h * scanLineSize; // we need to write upside do since TIFFReadScanline will fail if accessed in non-sequential order
    bool succeeded = true;
    for (int32 row = 0; row < h; row++)
    {
        buf -= scanLineSize;
        int result = TIFFReadScanline(in, reinterpret_cast<tdata_t>(buf), row, 0);
        if (result == -1)
        {
            succeeded = false;
            SWARNING << "Reading scanline from tiff failed, try reading strips" << std::endl;
            break;
        }
    }
#endif
    UChar8 *dst = image->editData();
   
    if(TIFFIsTiled(in))
    {
        uint32 tileWidth, tileLength; 
        TIFFGetField(in, TIFFTAG_TILEWIDTH, &tileWidth);
        TIFFGetField(in, TIFFTAG_TILELENGTH, &tileLength);
        UChar8 *buf = dst;
        tdata_t tileBuffer = _TIFFmalloc(TIFFTileSize(in));
        osg::UInt32 bytesPerPixel = image->getBpp();
        for(size_t y = 0; y < h; y+=tileLength)
        {
            for(size_t x = 0; x < w; x+=tileWidth)
            {
                int result = TIFFReadTile(in, tileBuffer, x,y, 0,0);
                if (result == -1)
                {
                    SWARNING << "Reading tile from tiff failed" << std::endl;
                    _TIFFfree(tileBuffer);
                    TIFFClose(in);
                    endEditCP(image, osg::FieldBits::AllFields);
                    return false;
                }
                for( size_t tileY = 0; tileY < tileLength; ++tileY)
                {
                    for( size_t tileX = 0; tileX < tileWidth; ++tileX)
                    {
                        size_t dstX = x + tileX;
                        size_t dstY = y + tileY;
                        if(dstY >= h || dstX >= w)
                            continue;

                        for( size_t ixBpp = 0; ixBpp < bytesPerPixel; ++ixBpp)
                        {
                            buf[bytesPerPixel * (dstY * w + dstX) + ixBpp] = ((UChar8*)tileBuffer)[bytesPerPixel * (tileY * tileWidth + tileX) + ixBpp];
                        }
                    }
                }
            }
        }
        _TIFFfree(tileBuffer);
    }
    else
    {
        size_t numberOfStrips = TIFFNumberOfStrips(in);
        size_t stripSize = TIFFStripSize(in);
         UChar8* buf = dst;
        for (int32 strip = 0; strip < numberOfStrips; strip++)
        {
            int result = TIFFReadEncodedStrip(in, strip, reinterpret_cast<tdata_t>(buf), (tsize_t)-1);
            buf += stripSize;
            if (result == -1)
            {
                SWARNING << "Reading strips from tiff failed" << std::endl;
                TIFFClose(in);
                endEditCP(image, osg::FieldBits::AllFields);
                return false;
            }
        }
    }
    TIFFClose(in);

    // tiff is flipped in y, so we need to flip the image
    osg::UInt32 bytesPerPixel = image->getBpp();
    for (UInt32 ixHeight = 0; ixHeight < (h / 2); ++ixHeight)
    {
        osg::SizeT beginIndex = (SizeT)ixHeight * w * bytesPerPixel;
        osg::SizeT endIndex = (SizeT)(h - 1 - ixHeight) * w * bytesPerPixel;
        for (UInt32 ixWidth = 0; ixWidth < w; ++ixWidth)
        {
            // swap data
            for (UInt32 ixBpp = 0; ixBpp < bytesPerPixel; ++ixBpp)
            {
                osg::UInt8 tmp = dst[beginIndex];
                dst[beginIndex++] = dst[endIndex];
                dst[endIndex++] = tmp;
            }
        }
    }
    endEditCP(image, osg::FieldBits::AllFields);

#if 0
    else
    {
        UInt32 numPixels = w * h;
        uint32 *buffer = new uint32[numPixels];
        if (TIFFReadRGBAImage(in, w, h, buffer, 1) == 0)
        {
            std::cout << "Reading failed" << std::endl;
            delete[] buffer;
            TIFFClose(in);
            endEditCP(image, osg::FieldBits::AllFields);
            return false;
        }

        TIFFClose(in);

        UChar8 *dst = image->editData();
        uint32 *src = buffer;
        switch (bpp)
        {
        case 4:
            for (UInt32 i = numPixels; i > 0; --i)
            {
                *dst++ = TIFFGetR(*src);
                *dst++ = TIFFGetG(*src);
                *dst++ = TIFFGetB(*src);
                *dst++ = TIFFGetA(*src++);
            }
            break;
        case 3:
            for (UInt32 i = numPixels; i > 0; --i)
            {
                *dst++ = TIFFGetR(*src);
                *dst++ = TIFFGetG(*src);
                *dst++ = TIFFGetB(*src++);
            }
            break;
        case 2:
            for (UInt32 i = numPixels; i > 0; --i)
            {
                *dst++ = TIFFGetG(*src);
                *dst++ = TIFFGetA(*src++);
            }
            break;
        case 1:
            for (UInt32 i = numPixels; i > 0; --i)
                *dst++ = TIFFGetG(*src++);
            break;
        }
        endEditCP(image, osg::FieldBits::AllFields);
        delete[] buffer;
    }
#endif
    return true;

#else

    SWARNING <<
        getMimeType() <<
        " read is not compiled into the current binary " <<
        std::endl;
    return false;

#endif // OSG_WITH_TIF
}

//-------------------------------------------------------------------------
/*!
Tries to write the image object to the given output stream.
Returns true on success.
*/
bool TIFImageFileType::write(const ImagePtr &OSG_TIF_ARG(image), std::ostream &OSG_TIF_ARG(os),
    const std::string &OSG_TIF_ARG(mimetype))
{
    bool                retCode = true;

#ifdef OSG_WITH_TIF

    if (image->getDimension() < 1 || image->getDimension() > 2)
    {
        FWARNING(("TIFImageFileType::write: invalid dimension %d!\n",
            image->getDimension()));
        return false;
    }

    TIFF         *out = TIFFClientOpen("dummy", "wm",
        reinterpret_cast<thandle_t>(&os),
        osReadProc, osWriteProc, osSeekProc, closeProc,
        osSizeProc, mapFileProc, unmapFileProc);
    Int64         lineSize = image->getWidth() * image->getBpp();
    int           photometric = 0, samplesPerPixel = 0;
    const UChar8 *data;
    int           row;

    // TODO: implement all cases correct
    switch (image->getBpp())
    {
    case 1:
        samplesPerPixel = 1;
        photometric = PHOTOMETRIC_MINISBLACK;
        break;
    case 2:
        samplesPerPixel = 2;
        photometric = PHOTOMETRIC_MINISBLACK;
        break;
    case 3:
        samplesPerPixel = 3;
        photometric = PHOTOMETRIC_RGB;
        break;
    case 4:
        samplesPerPixel = 4;
        photometric = PHOTOMETRIC_RGB;
        break;
    case 12: // RGB 32bit float
        samplesPerPixel = 3;
        photometric = PHOTOMETRIC_RGB;
        break;
    case 16: // RGBA 32bit float
        samplesPerPixel = 4;
        photometric = PHOTOMETRIC_RGB;
        break;
    }

    int bps = 8;
    int fmt = SAMPLEFORMAT_UINT;
    switch(image->getDataType())
    {
    case Image::OSG_UINT8_IMAGEDATA:
        bps = 8;
        fmt = SAMPLEFORMAT_UINT;
        break;
    case Image::OSG_FLOAT32_IMAGEDATA:
        bps = 32;
        fmt = SAMPLEFORMAT_IEEEFP;
        break;
    default:
        FWARNING(("TIFImageFileType::write: image data type not supported!\n"));
        return false;
        break;
    }

    if(out)
    {
        const std::string *comment = image->findAttachmentField("Comment");
        if(comment != NULL)
            TIFFSetField(out, TIFFTAG_IMAGEDESCRIPTION, comment->c_str());

        TIFFSetField(out, TIFFTAG_IMAGEWIDTH, image->getWidth());
        TIFFSetField(out, TIFFTAG_IMAGELENGTH, image->getHeight());
        TIFFSetField(out, TIFFTAG_XRESOLUTION, image->getResX());
        TIFFSetField(out, TIFFTAG_YRESOLUTION, image->getResY());
        TIFFSetField(out, TIFFTAG_RESOLUTIONUNIT, image->getResUnit());
        TIFFSetField(out, TIFFTAG_ORIENTATION, ORIENTATION_TOPLEFT);
        TIFFSetField(out, TIFFTAG_SAMPLESPERPIXEL, samplesPerPixel);
        TIFFSetField(out, TIFFTAG_BITSPERSAMPLE, bps);
        TIFFSetField(out, TIFFTAG_SAMPLEFORMAT, fmt);
        TIFFSetField(out, TIFFTAG_PLANARCONFIG, PLANARCONFIG_CONTIG);
        TIFFSetField(out, TIFFTAG_PHOTOMETRIC, photometric);

        if(_options.find("compressionType=LZW") != std::string::npos)
            TIFFSetField(out, TIFFTAG_COMPRESSION, COMPRESSION_LZW);
        else
            TIFFSetField(out, TIFFTAG_COMPRESSION, COMPRESSION_NONE);

        UInt32 profileLength = 0;
        const UChar8* profile = image->getICCProfile(profileLength);
        if( profile != NULL && profileLength != 0 )
        {
            printf("got PROFILE with length %i\n", profileLength);
            TIFFSetField(out, TIFFTAG_ICCPROFILE, profileLength, profile);
        }

        const std::string *xmp = image->findAttachmentField("XMPData");
        if(xmp != NULL)
        {
            UInt32 xmpLength = xmp->length() ;
            if(xmpLength != 0)
            {
                TIFFSetField(out, TIFFTAG_XMLPACKET, xmpLength, xmp->c_str());
            }
        }

        TIFFSetField(out, TIFFTAG_ROWSPERSTRIP, TIFFDefaultStripSize(out, 0));

        for(row = 0; row < image->getHeight(); row++)
        {
            data = image->getData() + (Int64(image->getHeight() - row - 1) * lineSize);
            if(TIFFWriteScanline(out,
                static_cast<tdata_t>(
                    const_cast<UChar8 *>(data)),
                row,
                0) < 0)
            {
                retCode = false;
                break;
            }
        }

        TIFFClose(out);
    }

#else
    SWARNING <<
        getMimeType() <<
        " write is not compiled into the current binary " <<
        std::endl;
#endif
    return retCode;
}

//-------------------------------------------------------------------------
/*!
Tries to determine the mime type of the data provided by an input stream
by searching for magic bytes. Returns the mime type or an empty string
when the function could not determine the mime type.
*/
std::string TIFImageFileType::determineMimetypeFromStream(std::istream &is)
{
    char filecode[4];
    is.read(filecode, 4);
    is.seekg(-4, std::ios::cur);
    if (strncmp(filecode, "MM\x00\x2a", 4) == 0)
        return std::string(getMimeType());
    if (strncmp(filecode, "II\x2a\x00", 4) == 0)
        return std::string(getMimeType());
    return std::string();
}

//-------------------------------------------------------------------------

bool TIFImageFileType::validateHeader( const Char8 *fileName, bool &implemented)
{
    implemented = true;

    if(fileName == NULL)
        return false;

    FILE *file = osg::fopen(fileName, "rb");

    if(file == NULL)
        return false;

    std::string magic;
    magic.resize(2);
    fread(static_cast<void *>(&magic[0]), 2, 1, file);
    fclose(file);

    if(magic == "MM" || magic == "II")
    {
        return true;
    }

    return false;
}

//-------------------------------------------------------------------------
/*!
Constructor used for the singleton object
*/
TIFImageFileType::TIFImageFileType(const Char8 *mimeType,
    const Char8 *suffixArray[],
    UInt16 suffixByteCount,
    UInt32 flags) :
    ImageFileType(mimeType, suffixArray, suffixByteCount, flags)
{
#ifdef OSG_WITH_TIF
    TIFFSetWarningHandler(&warningHandler);
    TIFFSetErrorHandler(&errorHandler);
#endif
}

//-------------------------------------------------------------------------
/*!
Destructor
*/
TIFImageFileType::~TIFImageFileType(void) {}
