/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright (C) 2000-2002 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/
//-------------------------------
// Includes
//-------------------------------
#include <stdlib.h>
#include <stdio.h>
#include <filesystem>

#include "OSGConfig.h"

#include "OSGZIPImageFileType.h"
#include <OSGLog.h>

#include <iostream>
#include <sstream>

#include <vrZipper.h>

#include "OSGImageFileHandler.h"

OSG_BEGIN_NAMESPACE

// Static Class Variable implementations:
static const Char8 *suffixArray[] = {
    "zip"
};

ZIPImageFileType ZIPImageFileType::_the("image/zip",
                                        suffixArray, sizeof(suffixArray),
                                        OSG_READ_SUPPORTED |
                                        OSG_WRITE_SUPPORTED);

//-------------------------------------------------------------------------
/*!
Class method to get the singleton Object
*/
ZIPImageFileType& ZIPImageFileType::the (void)
{
    return _the;
}

//-------------------------------------------------------------------------
/*!
Tries to fill the image object with the data read from
the given input stream. Returns true on success.
*/
bool ZIPImageFileType::read(ImagePtr &image, std::istream &is,
                            const std::string &mimetype)
{
    if(image == NullFC)
        return false;

    try
    {
        ImageSequencePtr sequence = image->getImageSequence();
        if(sequence == NullFC)
        {
            sequence = ImageSequence::create();
            beginEditCP(image, Image::ImageSequenceFieldMask);
            image->setImageSequence(sequence);
            endEditCP(image, Image::ImageSequenceFieldMask);
        }
        sequence->getImages().getValues().assign(std::istreambuf_iterator<char>(is), {});
        if(sequence->getImages().getSize() == 0)
            return false;
        image->readSequence(0, true);
    }
    catch(...)
    {
        printf("ZIPImageFileType::read: FAILED!\n");
        return false;
    }


    return true;
}

//-------------------------------------------------------------------------
/*!
    Tries to write the image object to the given output stream.
    Returns true on success.
*/
bool ZIPImageFileType::write(const ImagePtr &image, std::ostream &os, const std::string &mimetype)
{
    if(image == NullFC)
        return false;

    if(!image->getInlineSequence())
        return false;

    ImageSequencePtr sequence = image->getImageSequence();
    if( sequence == NullFC)
        return false;

    os.write((const char *) &sequence->getImages()[0], sequence->getImages().size());

    return true;
}

bool ZIPImageFileType::writeImages(ImagePtr &image, const char *path)
{
    if(image == NullFC)
        return false;

    if(!image->getInlineSequence())
        return false;

    if(path == nullptr)
        return false;

    ImageSequencePtr sequence = image->getImageSequence();
    if(sequence == NullFC)
        return false;

    const std::string unzipPath = path;
    vrZipper zipper;
    if(!zipper.open(sequence->getImages().getValues().data(), sequence->getImages().getValues().size(), vrZipper::OpenMode::Read))
        return false;
    const bool r = zipper.extract(std::filesystem::u8path(unzipPath));
    zipper.close();

    return r;
}

//-------------------------------------------------------------------------
/*!
Tries to determine the mime type of the data provided by an input stream
by searching for magic bytes. Returns the mime type or an empty string
when the function could not determine the mime type.
*/
std::string ZIPImageFileType::determineMimetypeFromStream(std::istream &is)
{
    char filecode[2];
    is.read(filecode, 2);
    is.seekg(-2, std::ios::cur);
    return strncmp(filecode, "PK", 2) == 0 ?
        std::string(getMimeType()) : std::string();
}

//-------------------------------------------------------------------------
/*!
Constructor used for the singleton object
*/
ZIPImageFileType::ZIPImageFileType(const Char8 *mimeType,
                                   const Char8 *suffixArray[],
                                   UInt16 suffixByteCount,
                                   UInt32 flags) :
    ImageFileType(mimeType,suffixArray, suffixByteCount, flags)
{}

//-------------------------------------------------------------------------
/*!
Destructor
*/
ZIPImageFileType::~ZIPImageFileType(void) {}


OSG_END_NAMESPACE
