#ifndef OSG_CHECKSUM_CALCULATOR_H__
#define OSG_CHECKSUM_CALCULATOR_H__

#pragma once

#include <math.h>
#include <set>
#include <unordered_map>

#include "OSGConfig.h"
#include <OSGBinaryChecksumHandler.h>
#include <OSGImage.h>
#include <OSGImageAttachment.h>
#include <OSGSFVecTypes.h>
#include <OSGSFMathTypes.h>
#include "OSGMFVecTypes.h"

OSG_BEGIN_NAMESPACE

static Real32 roundReal32(Real32 v)
{
    return roundf(v * 1.0e5f) * 1.0e-5f;
}

template<typename SFieldType, int Size>
static void copyRoundedSF(BinaryChecksumHandler &bch, const Field* fieldPtr)
{
    const auto* sf = static_cast<const SFieldType*>(fieldPtr);
    const auto& v = sf->getValue();
    for(int i = 0; i < Size; i++)
    {
        bch.putValue(roundReal32(v[i]));
    }
}

template<typename MFieldType, int ElementSize>
static void copyRoundedMF(BinaryChecksumHandler &bch, const Field* fieldPtr)
{
    const auto* mf = static_cast<const MFieldType*>(fieldPtr);
    const auto& v = mf->getValues();
    bch.putValueSize(mf->size());
    for (UInt32 j = 0; j < mf->size(); ++j)
    {
        const auto& e = v[j];
        for(int i = 0; i < ElementSize; i++)
        {
            bch.putValue(roundReal32(e[i]));
        }
    }
}

template<>
void copyRoundedSF<SFMatrix, 16>(BinaryChecksumHandler &bch, const Field* fieldPtr)
{
    const auto* sf = static_cast<const SFMatrix*>(fieldPtr);
    const auto* v = sf->getValue().getValues();
    for(int i = 0; i < 16; i++)
    {
        bch.putValue(roundReal32(v[i]));
    }
}

static bool copyRoundedSF(BinaryChecksumHandler &bch, const FieldType& fType, const Field* fieldPtr)
{
    if(fType == SFVec2f::getClassType())
    {
        copyRoundedSF<SFVec2f, 2>(bch, fieldPtr);
        return true;
    }
    else if(fType == SFVec3f::getClassType())
    {
        copyRoundedSF<SFVec3f, 3>(bch, fieldPtr);
        return true;
    }
    else if(fType == SFVec4f::getClassType())
    {
        copyRoundedSF<SFVec4f, 4>(bch, fieldPtr);
        return true;
    }
    else if(fType == SFColor3f::getClassType())
    {
        copyRoundedSF<SFColor3f, 3>(bch, fieldPtr);
        return true;
    }
    else if(fType == SFColor4f::getClassType())
    {
        copyRoundedSF<SFColor4f, 4>(bch, fieldPtr);
        return true;
    }
    else if(fType == SFQuaternion::getClassType())
    {
        copyRoundedSF<SFQuaternion, 4>(bch, fieldPtr);
        return true;
    }
    else if(fType == SFMatrix::getClassType())
    {
        copyRoundedSF<SFMatrix, 16>(bch, fieldPtr);
        return true;
    }
    return false;
}

static bool copyRoundedMF(BinaryChecksumHandler &bch, const FieldType& fType, const Field* fieldPtr)
{
    if(fType == MFVec2f::getClassType())
    {
        copyRoundedMF<MFVec2f, 2>(bch, fieldPtr);
        return true;
    }
    else if(fType == MFVec3f::getClassType())
    {
        copyRoundedMF<MFVec3f, 3>(bch, fieldPtr);
        return true;
    }
    else if(fType == MFVec4f::getClassType())
    {
        copyRoundedMF<MFVec4f, 4>(bch, fieldPtr);
        return true;
    }
    else if(fType == MFColor3f::getClassType())
    {
        copyRoundedMF<MFColor3f, 3>(bch, fieldPtr);
        return true;
    }
    else if(fType == MFColor4f::getClassType())
    {
        copyRoundedMF<MFColor4f, 4>(bch, fieldPtr);
        return true;
    }
    return false;
}

static bool copyRounded(BinaryChecksumHandler &bch, const FieldType& fType, const Field* fieldPtr)
{
    // Round field values that have an epsilon in their operator ==

    if(fieldPtr->getCardinality() == FieldType::SINGLE_FIELD)
    {
        return copyRoundedSF(bch, fType, fieldPtr);
    }
    else
    {
        return copyRoundedMF(bch, fType, fieldPtr);
    }
    return false;
}


// std::less<> provides operator<(std::string, const char*) so that find or count can 
// be used without creation of temporary std::string object.
typedef std::set<std::string, std::less<>> StringSet;
static const StringSet _crc_exclude_types = {"ImageAttachment", "AnimAttachment", "Name", "TagAttachment", "TemperatureAttachment", "RaytracingLightAttachment", "UNPRAttachment"};

static void calcChecksumR(BinaryChecksumHandler &bch,
                          const FieldContainerPtr &fc,
                          const StringSet &excludeTypes,
                          const StringSet &excludeFields = {},
                          const std::unordered_map<std::string, StringSet> &excludeFieldsPerType = {}, // exclude fields by concrete type and field names e.g. {"Transform3D", {"translation","rotation"}}
                          const std::function<bool(BinaryChecksumHandler &bch, const FieldContainerPtr &)>& callback = nullptr,
                          const bool roundValues = false,
                          const bool forceMipmapLevel0 = false,
                          const std::function<bool(const FieldContainerPtr&, unsigned int)>& skipFieldIdCallback = nullptr)
{
    if (fc == NullFC)
        return;

    if (callback != nullptr && callback(bch, fc))
        return;

    FieldContainerType  &fcType = fc->getType();
    if (excludeTypes.count(fcType.getCName()) > 0)
    {
        //printf("excluding '%s'\n", fcType.getCName());
        return;
    }

    ImagePtr img = ImagePtr::dcast(fc);
    if (img != NullFC && img->isValid())
    {
        // ok use at least a mipmap level with 16 x 16 pixels.
        int level = forceMipmapLevel0 ? 0 : (img->getMipMapCount() - 5);
        if (level < 0)
            level = 0;

        UInt8 *data = img->getRawData(level, 0, 0);
        //printf("image '%s' crc count %d size %u\n", img->getName().c_str(), img->getMipMapCount(), (unsigned int) img->calcMipmapLevelSize(level));
        if (data != NULL)
            bch.write(data, img->calcMipmapLevelSize(level));
        return;
    }

    // debug
    bool print_fields = false;

    const auto excludeFieldsIt = excludeFieldsPerType.find(fcType.getCName());

    if (print_fields)
        printf("---------------------\n");
    //go through all fields
    for (UInt32 i = 1; i <= fcType.getNumFieldDescs(); ++i)
    {
        FieldDescription    *fDesc = fcType.getFieldDescription(i);
        Field               *fieldPtr = fc->getField(i);
        const FieldType     &fType = fieldPtr->getType();
        BitVector           mask = fDesc->getFieldMask();

        // we alwasy render the previews in realistic mode so we can ignore the render mode field
        // in the UberMaterial. The field is internal so we are done.
        if (!fDesc->isInternal())
        {
            // ignore node volume
            if (fcType == Node::getClassType() &&
                fDesc->getFieldMask() == Node::VolumeFieldMask)
            {
                continue;
            }

            // ignore parents field.
            if (!strcmp(fDesc->getCName(), "parents"))
            {
                continue;
            }

            // We can selectively exclude certain fields for specific fieldcontainer types...
            if (excludeFieldsIt != excludeFieldsPerType.end() && excludeFieldsIt->second.count(fDesc->getCName()) > 0)
            {
                //printf("-- excluding field '%s' of container %s\n", fDesc->getCName(), fcType.getCName());
                continue;
            }

            // exclude field by name only
            if (excludeFields.count(fDesc->getCName()) > 0)
            {
                //printf("-- excluding field '%s' of container %s\n", fDesc->getCName(), fcType.getCName());
                continue;
            }

            if(skipFieldIdCallback != nullptr && skipFieldIdCallback(fc, i))
            {
                continue;
            }

            if (strstr(fType.getCName(), "Ptr") != NULL)
            {
                if (fieldPtr->getCardinality() == FieldType::SINGLE_FIELD)
                {
                    calcChecksumR(bch, static_cast<SFFieldContainerPtr *>(fieldPtr)->getValue(), excludeTypes, excludeFields, excludeFieldsPerType, callback, roundValues, forceMipmapLevel0, skipFieldIdCallback);
                }
                else if (fieldPtr->getCardinality() == FieldType::MULTI_FIELD)
                {
                    MFFieldContainerPtr *mfield = static_cast<MFFieldContainerPtr *>(fieldPtr);
                    if (!mfield->empty())
                    {
                        SizeT noe = mfield->size();
                        for (SizeT i = 0; i < noe; ++i)
                            calcChecksumR(bch, (*(mfield))[i], excludeTypes, excludeFields, excludeFieldsPerType, callback, roundValues, forceMipmapLevel0, skipFieldIdCallback);
                    }
                }
            }
            else if (!strcmp(fDesc->getCName(), "attachments"))
            {
                SFAttachmentMap *amap = static_cast<SFAttachmentMap *>(fieldPtr);
                if (!amap->getValue().empty())
                {
                    AttachmentMap::const_iterator mapIt = amap->getValue().begin();
                    AttachmentMap::const_iterator mapEnd = amap->getValue().end();

                    for (; mapIt != mapEnd; ++mapIt)
                        calcChecksumR(bch, mapIt->second, excludeTypes, excludeFields, excludeFieldsPerType, callback, roundValues, forceMipmapLevel0, skipFieldIdCallback);
                }
            }
            else
            {
                if (print_fields)
                {
                    std::string val;
                    fieldPtr->getValueByStr(val);
                    printf("fieldName: '%s' '%s'\n", fDesc->getCName(), val.c_str());
                }

                if(roundValues)
                {
                    if(copyRounded(bch, fType, fieldPtr))
                        continue;
                }
                fc->copyToBin(bch, mask);
            }

        }
    }
    if (print_fields)
        printf("---------------------\n");
}


//-----------------------------------------------------------------------------------------------//
//  Implementation of combineChecksum is from Boost 1.71, file boost/container_hash/hash.hpp,
//  boost::hash_detail::hash_combine_impl 

// Copyright 2005-2014 Daniel James.
// Distributed under the Boost Software License, Version 1.0. 
// (See http://www.boost.org/LICENSE_1_0.txt)

//  Based on Peter Dimov's proposal
//  http://www.open-std.org/JTC1/SC22/WG21/docs/papers/2005/n1756.pdf
//  issue 6.18.

/* Combine multiple checksums into a new checksum.
   Different order of checksums gives different results.
     unsigned int seed = 0;
     combineChecksum(seed, checksum1);
     combineChecksum(seed, checksum2);
*/
static void combineChecksum(unsigned int& seed, unsigned int checksum)
{
    seed ^= checksum + 0x9e3779b9 + (seed << 6) + (seed >> 2);
}

//-----------------------------------------------------------------------------------------------//


OSG_END_NAMESPACE

#endif
