/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright (C) 2000-2002 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

#include <stdlib.h>
#include <stdio.h>

#include "OSGConfig.h"

#include <OSGGL.h>
#include "OSGMaterial.h"

#include <OSGBinaryChecksumHandler.h>
#include <OSGImage.h>
#include <OSGImageAttachment.h>

#include <OSGChecksumCalculator.h>
#include <OSGName.h>
#include <OSGHitPoint.h>

OSG_USING_NAMESPACE


/***************************************************************************\
 *                            Description                                  *
\***************************************************************************/

/*! \class osg::Material
    \ingroup GrpSystemMaterial

The material base class.

\ext The Material has two interfaces to return a State that represents it.
osg::Material::makeState() creates a new osg::State and returns it. This is ok
for rare use, but for every frame this is going to be too expensive. For these
cases osg::Material::rebuildState() and osg::Material::getState() are used, which
modify and return an internal copy of the State. 

The other method to implement is osg::Material::isTransparent(), to identify
transparent materials to be rendered after the opaque ones.
\endext
*/

/*! \fn osg::Material::makeState()

    Creates an instance of a osg::State that is used to represent the material.
    This is used by the osg::DrawAction.  
*/

/*! \fn osg::Material::rebuildState()

    Update the internal osg::State. 
*/

/*! \fn osg::Material::getState()

    Access the internal osg::State, used by the osg::RenderAction.
*/

/*! \fn osg::Material::isTransparent()

    Check if the Material is transparent and thus has to be rendered afte the
    opaque ones.
*/

/***************************************************************************\
 *                           Class variables                               *
\***************************************************************************/

OSG_BEGIN_NAMESPACE
OSG_SYSTEMLIB_DLLMAPPING MaterialPtr NullMaterial;
OSG_END_NAMESPACE

const Int32 Material::NoStateSorting = 0x7fffffff;
const Int32 Material::TransparencyAutoDetection = 0;
const Int32 Material::TransparencyForceTransparent = 1;
const Int32 Material::TransparencyForceOpaque = 2;

UInt32      Material::_defaultShadowId = 0;
UInt32      Material::_defaultStudioId = 0;
UInt32      Material::_defaultEnvironmentId = 0;

/***************************************************************************\
 *                           Class methods                                 *
\***************************************************************************/

/*-------------------------------------------------------------------------*\
 -  private                                                                -
\*-------------------------------------------------------------------------*/

void Material::initMethod (void)
{
}

/***************************************************************************\
 *                           Instance methods                              *
\***************************************************************************/

/*-------------------------------------------------------------------------*\
 -  public                                                                 -
\*-------------------------------------------------------------------------*/


/*------------- constructors & destructors --------------------------------*/

Material::Material(void) :
    Inherited(),
    m_checksumDirty(true),
    m_checksum(0),
    _pState(NullFC)
{   
}

Material::Material(const Material &source) :
    Inherited(source),
    m_checksumDirty(true),
    m_checksum(0),
    _pState(NullFC)
{
// Doing this kills using prototypes with preset states. 
// What's the point of this anyway? Sharing _pState between Materials never
// makes sense
//    setRefdCP(_pState, source._pState); 
}

Material::~Material(void)
{
    subRefCP(_pState);
    for( StatePtr state : m_shadowStates)
            subRefCP(state);
    m_shadowStates.clear();
}

#if defined(OSG_FIXED_MFIELDSYNC)
void Material::onDestroyAspect(UInt32 uiId, UInt32 uiAspect)
{
    Inherited::onDestroyAspect(uiId, uiAspect);

    subRefCP(_pState);
    _pState = NullFC;

    for( StatePtr state : m_shadowStates)
            subRefCP(state);
    m_shadowStates.clear();
}
#endif

StatePtr Material::getState(UInt32 OSG_CHECK_ARG(index))
{
    return _pState;
}

StatePtr Material::getShadowState(UInt32 index)
{
    return (index < m_shadowStates.size()) ? m_shadowStates[index] : NullFC;
}

bool Material::isMultiPass(void) const
{
    return false;
}

UInt32 Material::getNPasses(void) const
{
    return 1;
}

UInt32 Material::getNShadowPasses() const
{
    return 1;
}


void Material::changed(BitVector whichField, UInt32 origin)
{
    Inherited::changed(whichField, origin);
    m_checksumDirty = true;
    rebuildState();
}

Int32 Material::getRealSortKey(void) const
{
    return _sfSortKey.getValue();
}


bool Material::hasMaterial(const MaterialPtr &, SearchMaterialOption) const
{
    return false;
}

bool Material::supportsAnimChannels() const
{
    return true;
}

void Material::getAnimChannels(Material::ChannelData& data)  
{
    
}

void Material::setupStudioEnvironment(void)
{
    SWARNING << "setupStudioEnvironment: wrong material type!" << std::endl;
}

bool Material::initializeAsDefaultShadow(void)
{
    SWARNING << "initializeAsDefaultShadow: wrong material type!" << std::endl;
    return false;
}

bool Material::initializeAsDefaultStudio(void)
{
    SWARNING << "initializeAsDefaultStudio: wrong material type!" << std::endl;
    return false;
}

bool Material::initializeAsDefaultEnvironment(void)
{
    SWARNING << "initializeAsDefaultEnvironment: wrong material type!" << std::endl;
    return false;
}

MaterialPtr Material::getDefaultShadow(void)
{
    if( _defaultShadowId == 0 )
        return NullFC;

    MaterialPtr defaultShadow = MaterialPtr::dcast(FieldContainerFactory::the()->getContainer(_defaultShadowId));
    if( !defaultShadow ) // default shadow has somehow been deleted
        _defaultShadowId = 0;

    return defaultShadow;
}

MaterialPtr Material::getDefaultStudio(void)
{
    if( _defaultStudioId == 0 )
        return NullFC;

    MaterialPtr defaultStudio = MaterialPtr::dcast(FieldContainerFactory::the()->getContainer(_defaultStudioId));
    if( !defaultStudio ) // default shadow has somehow been deleted
        _defaultStudioId = 0;

    return defaultStudio;
}

MaterialPtr Material::getDefaultEnvironment(void)
{
    if( _defaultEnvironmentId == 0 )
        return NullFC;

    MaterialPtr defaultEnv = MaterialPtr::dcast(FieldContainerFactory::the()->getContainer(_defaultEnvironmentId));
    if( !defaultEnv ) // default env has somehow been deleted
        _defaultEnvironmentId = 0;

    return defaultEnv;
}

void Material::clearDefaultStudio()
{   
    _defaultStudioId = 0;
}

void Material::clearDefaultEnvironment()
{
    _defaultEnvironmentId = 0;
}


MaterialPtr Material::getFirstNonTransparentMaterial(const HitPoint &p) const
{
    MaterialPtr thisPtr(*this);
    return thisPtr;
}


Vec3f Material::getTransparency(const HitPoint &p) const
{
    return Vec3f(0.0f, 0.0f, 0.0f); // opaque
}

bool Material::isFullyTransparent() const
{
    return false;
}

bool Material::isCutoutActiveForState(UInt32 index) const
{
    return isCutoutActive();
}

bool Material::isCutoutActive() const
{
    return false;
}

/*------------------------------- dump ----------------------------------*/

void Material::dump(      UInt32    ,
                    const BitVector ) const
{
    SLOG << "Material::dump called: NIY!" << std::endl;
//   Inherited::dump(uiIndent, bvFlags);
}

/*-------------------------- comparison -----------------------------------*/

bool Material::operator < (const Material &other) const
{
    return this < &other;
}

bool Material::operator == (const Material& other) const
{
    return _pState == other._pState;
}

bool Material::operator != (const Material &other) const
{
    return ! (*this == other);
}

void Material::setChecksumDirty()
{
    m_checksumDirty = true;
}

unsigned int Material::calcChecksum()
{
    if (m_checksumDirty)
    {
        MaterialPtr mat(*this);
        if (mat == NullFC)
            return 0;

        BinaryChecksumHandler bch;
        calcChecksumR(bch, mat, _crc_exclude_types);
        m_checksum = bch.getChecksum();
        m_checksumDirty = false;
    }
    return m_checksum;
}

bool Material::isMultiState(void) const
{
    return false;
}

UInt32 Material::getNStates(void) const
{
    return 0;
}

std::string Material::getStateName(UInt32 /*stateIndex*/) const
{
    return std::string();
}

std::vector<std::string> Material::getStateNames() const
{
    return std::vector<std::string>();
}

void Material::applyState(UInt32 /*stateIndex*/) 
{
}

bool Material::getCurrentState(UInt32 & index) const
{
    return false;
}

const char* Material::getName(osg::MaterialPtr const& mat)
{
    const char *name;

    if (mat == NullFC)
    {
        name = "<noname>";
        return name;
    }
   
    NamePtr namea = NamePtr::dcast(mat->findAttachment(Name::getClassType()));
    if (namea != NullFC)
    {
        name = namea->getName().c_str();
    }
    else
    {
        name = mat->getType().getName().str();
    }
    return name;
}

bool Material::getCurrentStateName(std::string &name) const
{
    if ( !isMultiState() )
        return false;

    UInt32 index;
    if (!getCurrentState(index))
        return false;

    const std::vector<std::string> names = getStateNames();
    if (index >= names.size())
        return false;

    name = names[index];
    return true;
}

/*------------------------------------------------------------------------*/
/*                              cvs id's                                  */

#ifdef OSG_SGI_CC
#pragma set woff 1174
#endif

#ifdef OSG_LINUX_ICC
#pragma warning( disable : 177 )
#endif

namespace
{
    static Char8 cvsid_cpp       [] = "@(#)$Id: FCTemplate_cpp.h,v 1.13 2002/06/01 10:37:25 vossg Exp $";
    static Char8 cvsid_hpp       [] = OSGMATERIAL_HEADER_CVSID;
    static Char8 cvsid_inl       [] = OSGMATERIAL_INLINE_CVSID;

    static Char8 cvsid_fields_hpp[] = OSGMATERIALFIELDS_HEADER_CVSID;
}

#ifdef __sgi
#pragma reset woff 1174
#endif

