/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *               Copyright (C) 2000-2002 by the OpenSG Forum                 *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

#include <stdlib.h>
#include <stdio.h>

#include <OSGConfig.h>

#include "OSGChecksumCalculator.h"
#include "OSGMultiPassMaterial.h"

OSG_USING_NAMESPACE

/***************************************************************************\
 *                            Description                                  *
\***************************************************************************/

/*! \class osg::MultiPassMaterial

*/

/***************************************************************************\
 *                           Class variables                               *
\***************************************************************************/

/***************************************************************************\
 *                           Class methods                                 *
\***************************************************************************/

void MultiPassMaterial::initMethod (void)
{
}


/***************************************************************************\
 *                           Instance methods                              *
\***************************************************************************/

/*-------------------------------------------------------------------------*\
 -  private                                                                 -
\*-------------------------------------------------------------------------*/

/*----------------------- constructors & destructors ----------------------*/

MultiPassMaterial::MultiPassMaterial(void) :
    Inherited()
{
}

MultiPassMaterial::MultiPassMaterial(const MultiPassMaterial &source) :
    Inherited(source)
{
}

MultiPassMaterial::~MultiPassMaterial(void)
{
}

/*----------------------------- class specific ----------------------------*/

void MultiPassMaterial::changed(BitVector whichField, UInt32 origin)
{
   
    Inherited::changed(whichField, origin);
}



/**
 * @brief MultiPassMaterial::insertMaterial
 * @param index
 * @param mat
 */
void MultiPassMaterial::insertMaterial(UInt32 index, MaterialPtr mat)
{
    if( !mat )
        return;

    if( index >= _mfMaterials.size() )
    {
        addMaterial(mat);
        return;
    }

    _mfMaterials.resize(_mfMaterials.size() + 1);

    for( UInt32 i = _mfMaterials.size() - 1; i > index; --i )
        _mfMaterials[i] = _mfMaterials[i - 1];

    _mfMaterials[index] = mat;
    addRefCP(mat);
}

/**
 * @brief MultiPassMaterial::replaceMaterial
 * @param index
 * @param mat
 */
void MultiPassMaterial::replaceMaterial(UInt32 index, MaterialPtr mat)
{
    if( !mat )
        return;

    if( index >= _mfMaterials.size() )
    {
        SWARNING << "MultiPassMaterial::replaceMaterial(" << this << ") index out of range "
                 << index << std::endl;
        return;
    }

    subRefCP(_mfMaterials[index]);
    _mfMaterials[index] = mat;
    addRefCP(mat);
}



/*! Create a osg::State that represents this Material and return it.
*/

StatePtr MultiPassMaterial::makeState(void)
{
    return NullFC;
}

/*! Rebuild the internal State. Just collects the chunks in the State.
*/

void MultiPassMaterial::rebuildState(void)
{
    for(UInt32 i = 0; i < _mfMaterials.size(); ++i)
    {
        if(_mfMaterials[i] != NullFC)
            _mfMaterials[i]->rebuildState();
    }
}

// getNPasses and getState must use the same logic
StatePtr MultiPassMaterial::getState(UInt32 index)
{
    if(index >= getNPasses())
    {
        SWARNING << "MultiPassMaterial::getState: index out of range!" << std::endl;
        return NullFC;
    }

    UInt32 totalpasses = 0;
    for(UInt32 i = 0; i < _mfMaterials.size(); ++i)
    {
        if(_mfMaterials[i] != NullFC)
        {
            UInt32 npasses = _mfMaterials[i]->getNPasses();
            totalpasses += npasses;
            if(index < totalpasses)
                return _mfMaterials[i]->getState(index - (totalpasses - npasses));
        }
    }

    return NullFC;
}

// getNPasses and getState must use the same logic
UInt32 MultiPassMaterial::getNPasses(void) const
{
    UInt32 npasses = 0;
    for(UInt32 i = 0; i < _mfMaterials.size(); ++i)
    {
        if(_mfMaterials[i] != NullFC)
            npasses += _mfMaterials[i]->getNPasses();
    }
    return npasses;
}

//getNShadowPasses and getShadowState must use the same logic
StatePtr MultiPassMaterial::getShadowState(UInt32 index)
{
    if(index >= getNPasses())
    {
        SWARNING << "MultiPassMaterial::getState: index out of range!" << std::endl;
        return NullFC;
    }

    UInt32 totalpasses = 0;
    for(auto & mat : _mfMaterials)
    {
        if(mat != NullFC)
        {
            const bool matIsTransparent = mat->isTransparent();
            if(!matIsTransparent || mat->isCutoutActive())
            {
                UInt32 npasses = mat->getNPasses();
                totalpasses += npasses;
                if(index < totalpasses)
                    return mat->getShadowState(index - (totalpasses - npasses));

                if(!matIsTransparent)
                    break;
            }
        }
    }
    return NullFC;
}

//getNShadowPasses and getShadowState must use the same logic
UInt32 MultiPassMaterial::getNShadowPasses(void) const
{
    UInt32 npasses = 0;
    for(const auto& mat : _mfMaterials)
    {
        if(mat != NullFC )
        {
            const bool matIsTransparent = mat->isTransparent();
            if(!matIsTransparent || mat->isCutoutActive())
            {
                npasses += mat->getNShadowPasses();
                // need to render only until first solid layer
                if(!matIsTransparent)
                    break;
            }
        }
    }
    return npasses;
}


bool MultiPassMaterial::isMultiPass(void) const
{
    return true;
}


/*! Check if the Material, i.e. all of its materials are transparent
*/
bool MultiPassMaterial::isTransparent(void) const
{
    Int32 tm = getTransparencyMode();
    if(tm != Material::TransparencyAutoDetection)
    {
        return (tm == Material::TransparencyForceTransparent);
    }

    // if any material is solid, the multipass is solid
    for(const auto& mat : _mfMaterials)
    {
        if(mat != NullFC && !mat->isTransparent())
            return false; // found solid
    }
    // all are transparent or we have no sub materials
    return true;
}

bool MultiPassMaterial::isCutoutActiveForState(UInt32 index) const
{
    if(index >= getNPasses())
    {
        SWARNING << "MultiPassMaterial::isCutoutActiveForState: index out of range!" << std::endl;
        return false;
    }

    UInt32 totalpasses = 0;
    for(UInt32 i = 0; i < _mfMaterials.size(); ++i)
    {
        if(_mfMaterials[i] != NullFC)
        {
            UInt32 npasses = _mfMaterials[i]->getNPasses();
            totalpasses += npasses;
            if(index < totalpasses)
                return _mfMaterials[i]->isCutoutActiveForState(index - (totalpasses - npasses));
        }
    }
    return false;
}

bool MultiPassMaterial::isCutoutActive() const
{
    Int32 tm = getTransparencyMode();
    if(tm != Material::TransparencyAutoDetection)
    {
        return false;
    }

    bool cutoutActive = false;
    // cutout is active for the multipass if all layers are transparent and any layer has active cutout
    for(const auto& mat : _mfMaterials)
    {
        if(mat != NullFC && !mat->isTransparent())
            return false; // found solid
        if(mat != NullFC && mat->isCutoutActive())
            cutoutActive = true;
    }
    return cutoutActive;
}

MaterialPtr MultiPassMaterial::getBaseMaterial() const
{
    MFMaterialPtr::const_iterator it = _mfMaterials.begin();
    MFMaterialPtr::const_iterator matsEnd = _mfMaterials.end();

    if (it != matsEnd)
    {
        return (*it);
    }
    return NullFC;
}

void MultiPassMaterial::dump(      UInt32    , 
                         const BitVector ) const
{
    SLOG << "Dump MultiPassMaterial NI" << std::endl;
}


MaterialPtr MultiPassMaterial::getFirstNonTransparentMaterial(const HitPoint &p) const
{
    // get hit layer

    const int numPasses = _mfMaterials.size();
    if(numPasses > 0)
    {
        // search for opaque layer at given hit point; 
        for( int i = numPasses - 1; i >= 0; --i)
        { 
            MaterialPtr mat = _mfMaterials[i];

            Vec3f t = mat->getTransparency(p);
            static const float g_lum[3] = { 0.212671f, 0.715160f, 0.072169f };
            float transparency = t[0] * g_lum[0] + t[1] * g_lum[1] + t[2] * g_lum[2];
            if(transparency < 0.9999f) // todo: define threshold; transparency == 1.0: fully transparent
            {
                return mat->getFirstNonTransparentMaterial(p);
            }
        }
        return _mfMaterials[0]; // base layer
    }

    return NullFC;
}


Vec3f MultiPassMaterial::getTransparency(const HitPoint &p) const
{
    const int numPasses = _mfMaterials.size();
    if(numPasses == 0) 
        return Vec3f(0.0f, 0.0f, 0.0f); // opaque

    Vec3f transparency = Vec3f( 1.0f, 1.0f, 1.0f); // fully transparent

    for( int i = numPasses - 1; i >= 0; --i)
    { 
        Vec3f t_i = _mfMaterials[i]->getTransparency( p);
        transparency[0] *= t_i[0];
        transparency[1] *= t_i[1];
        transparency[2] *= t_i[2];
    }
    return transparency;
}

bool MultiPassMaterial::isFullyTransparent() const
{
    const int numPasses = _mfMaterials.size();
    if(numPasses == 0)
        return false; 

    for(int i = numPasses - 1; i >= 0; --i)
    {
        if(!_mfMaterials[i]->isFullyTransparent())
            return false;
    }
    return true;

}

unsigned int MultiPassMaterial::calcChecksum()
{
    m_checksumDirty = false;
    BinaryChecksumHandler bch;
    copyToBin(bch, PreviewSceneFieldMask);
    m_checksum = bch.getChecksum();
    for (auto &material : _mfMaterials)
    {
        if(material != NullFC)
            combineChecksum(m_checksum, material->calcChecksum());
    }
    return m_checksum;
}

/*------------------------------------------------------------------------*/
/*                              cvs id's                                  */

#ifdef OSG_SGI_CC
#pragma set woff 1174
#endif

#ifdef OSG_LINUX_ICC
#pragma warning( disable : 177 )
#endif

namespace
{
    static Char8 cvsid_cpp       [] = "@(#)$Id: OSGMultiPassMaterial.cpp,v 1.7 2006/09/27 10:20:11 a-m-z Exp $";
    static Char8 cvsid_hpp       [] = OSGMULTIPASSMATERIALBASE_HEADER_CVSID;
    static Char8 cvsid_inl       [] = OSGMULTIPASSMATERIALBASE_INLINE_CVSID;

    static Char8 cvsid_fields_hpp[] = OSGMULTIPASSMATERIALFIELDS_HEADER_CVSID;
}

#ifdef __sgi
#pragma reset woff 1174
#endif

