/*---------------------------------------------------------------------------*\
*                                OpenSG                                     *
*                                                                           *
*                                                                           *
*             Copyright (C) 2000-2002 by the OpenSG Forum                   *
*                                                                           *
*                            www.opensg.org                                 *
*                                                                           *
*   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
*                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS forA PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License formore details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
 \*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 \*---------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

#include <stdlib.h>
#include <stdio.h>

#include "OSGConfig.h"

#include <OSGGL.h>

#include <OSGAction.h>
#include <OSGDrawAction.h>
#include <OSGGeometry.h>

#include <OSGStateChunk.h>
#include <OSGState.h>
#include <OSGMaterialChunk.h>

#include "OSGSimpleMaterial.h"

OSG_USING_NAMESPACE


/***************************************************************************\
 *                            Description                                  *
 \***************************************************************************/

 /*! \class osg::SimpleMaterial
     \ingroup GrpSystemMaterial

     The simple material class. See \ref PageSystemMaterialSimpleMaterial for a
     description.

     This material wraps the standard calls to glMaterial() in
     osg::SimpleMaterial::_sfAmbient, osg::SimpleMaterial::_sfDiffuse,
     osg::SimpleMaterial::_sfEmission, osg::SimpleMaterial::_sfSpecular,
     osg::SimpleMaterial::_sfShininess. In addition it supports transparency
     (osg::SimpleMaterial::_sfTransparency), can switch lighting
     (osg::SimpleMaterial::_sfLit) and the color material
     (osg::SimpleMaterial::_sfColorMaterial).

     */

     /***************************************************************************\
      *                           Class methods                                 *
      \***************************************************************************/

      /*-------------------------------------------------------------------------*\
       -  protected                                                              -
       \*-------------------------------------------------------------------------*/

       /* Create the chunks needed by this Material, one for the material properties,
          one for the optional transparency blending.
          */

void SimpleMaterial::prepareLocalChunks(void)
{
    if (_shlChunk == NullFC)
    {
        _shlChunk = SHLChunk::create();
        std::string vp = SHLChunk::getShaderHeaderString(osg::SHLChunk::kVertexProgram, true, false) +
            "layout (location = 0) in vec3 inPosition; \n"
            "layout (location = 1) in vec3 inNormal; \n"
            "layout (location = 11) in vec4 inColor; \n"
            "uniform bool useVertexColors; \n"
            "out VertexData  \n"
            "{             \n"
            "    vec4 wP;  \n"
            "    vec3 wN; \n"
            "    vec4 color; \n"
            "} vs_out;    \n"
            "void main()  \n"
            "{             \n"
            "    if(useVertexColors) \n"
            "       vs_out.color = inColor; \n"
            "    else \n"
            "       vs_out.color = vec4(0.0, 0.0, 0.0, 1.0); \n"
            "    vec4 wP = object.worldMatrix * vec4(inPosition.xyz, 1.0); \n"
            "	 vs_out.wN =  mat3(object.transInvWorldMatrix) * inNormal; \n"
            "	 vs_out.wP = wP; \n"
            "   gl_Position = camera.viewProjectionMatrix[0] * wP; \n"
            "#ifdef GLAD_GL_NV_stereo_view_rendering \n"
            "   gl_SecondaryPositionNV = camera.viewProjectionMatrix[1] * wP; \n"
            "   gl_Layer = 0; \n"
            "#endif \n"
			"#ifdef USE_LENS_MATCHED_SHADING \n"
			"	gl_ViewportMask[0] = 15; \n"
			"#ifdef GLAD_GL_NV_stereo_view_rendering\n"
			"	gl_SecondaryViewportMaskNV[0] = 15;\n"
			"#endif \n"
			"#endif \n"
            "}\n";

        std::string fp = SHLChunk::getShaderHeaderString(osg::SHLChunk::kFragmentProgram, true, false) +
            "uniform vec3 diffuseColor; \n"
            "uniform vec3 specularColor; \n"
            "uniform vec3 emissiveColor; \n"
            "uniform float shininess;  \n"
            "uniform float alpha; \n"
            "uniform bool isLit; \n"
            "uniform bool ignoreDisplayLuminance; \n"
            "in VertexData  \n"
            "{\n"
            "    vec4 wP; \n"
            "    vec3 wN; \n"
            "    vec4 color; \n"
            "} fs_in;\n"
            "layout (location = 0) out vec4 colorBufferRGBA;\n"
            "layout (location = 1) out vec2 motionVectorRG;\n"
            "void main()  \n"
            "{  \n"
            "   if( isLit)\n"
            "   { \n"
            "       vec3 eP = (camera.viewMatrix[eye] * fs_in.wP).xyz; \n"
            "       vec3 eN = mat3(camera.viewMatrix[eye]) *  fs_in.wN; \n"
            "	    vec3 N = normalize(eN); \n"
            "	    vec3 V = -normalize(eP);  \n"
            "	    N = faceforward( N, V, -N); \n"
            "	    float NdotV =  abs( dot(N, V)); \n"
            "	    float spec = pow( NdotV, shininess);\n"
            "	    vec3 diffuse = diffuseColor * NdotV;  \n"
            "	    vec3 specular = specularColor * spec;  \n"
            "	    colorBufferRGBA = vec4( fs_in.color.rgb + emissiveColor + diffuse + specular, alpha); \n"
            "   }\n"
            "   else \n"
            "   { \n"
            "       float displayLuminance = ignoreDisplayLuminance ? 1.0 : viewport.displayLuminance; \n"
            "       colorBufferRGBA = vec4( displayLuminance * (fs_in.color.rgb + diffuseColor + emissiveColor), alpha); \n"
            "       motionVectorRG = vec2(0.0);\n"
            "   } \n"
            "} \n";

        beginEditCP(_shlChunk, osg::FieldBits::AllFields);
        _shlChunk->setVertexProgram(vp);
        _shlChunk->setGeometryProgram("");
        _shlChunk->setFragmentProgram(fp);
        _shlChunk->clearUniformParameters();
        endEditCP(_shlChunk, osg::FieldBits::AllFields);

        addRefCP(_shlChunk);
    }

    if (_blendChunk == NullFC)
    {
        _blendChunk = BlendChunk::create();

        beginEditCP(_blendChunk, osg::FieldBits::AllFields);
        {
            _blendChunk->setSrcFactor(GL_SRC_ALPHA);
            _blendChunk->setDestFactor(GL_ONE_MINUS_SRC_ALPHA);
            _blendChunk->setAlphaSrcFactor(GL_ONE);
            _blendChunk->setAlphaDestFactor(GL_ONE_MINUS_SRC_ALPHA);
        }
        endEditCP(_blendChunk, osg::FieldBits::AllFields);

        addRefCP(_blendChunk);
    }
    if(_depthTestChunk == NullFC)
    {
        _depthTestChunk = DepthTestChunk::create();
        beginEditCP(_depthTestChunk, osg::FieldBits::AllFields);
        {
            _depthTestChunk->setDepthFunc(GL_ALWAYS);
        }
        endEditCP(_depthTestChunk, osg::FieldBits::AllFields);
        addRefCP(_depthTestChunk);
    }
}

/*-------------------------------------------------------------------------*\
 -  private                                                                -
 \*-------------------------------------------------------------------------*/

void SimpleMaterial::initMethod(void)
{
}

/***************************************************************************\
 *                           Instance methods                              *
 \***************************************************************************/

/*-------------------------------------------------------------------------*\
 -  private                                                                 -
 \*-------------------------------------------------------------------------*/

/*------------- constructors & destructors --------------------------------*/

SimpleMaterial::SimpleMaterial(void) :
Inherited(),
_shlChunk(NullFC),
_blendChunk(NullFC),
_depthTestChunk(NullFC)
{
}

SimpleMaterial::SimpleMaterial(const SimpleMaterial &source) :
Inherited(source),
_shlChunk(NullFC),
_blendChunk(NullFC),
_depthTestChunk(NullFC)
{
}

SimpleMaterial::~SimpleMaterial(void)
{
    subRefCP(_shlChunk);
    subRefCP(_blendChunk);
    subRefCP(_depthTestChunk);
}

void SimpleMaterial::changed(BitVector whichField, UInt32 origin)
{
    Inherited::changed(whichField, origin);
}

/*-------------------------- your_category---------------------------------*/

StatePtr SimpleMaterial::makeState(void)
{
    StatePtr state = State::create();

    Color3f v3;
    Color4f v4;
    float alpha = 1.f - getTransparency();

    prepareLocalChunks();

    beginEditCP(_shlChunk, SHLChunk::ParametersFieldMask);
    {
        _shlChunk->setUniformParameter("diffuseColor", Vec3f(getDiffuse()[0], getDiffuse()[1], getDiffuse()[2]));
        _shlChunk->setUniformParameter("specularColor", Vec3f(getSpecular()[0], getSpecular()[1], getSpecular()[2]));
        _shlChunk->setUniformParameter("emissiveColor", Vec3f(getEmission()[0], getEmission()[1], getEmission()[2]));
        _shlChunk->setUniformParameter("shininess", getShininess());
        _shlChunk->setUniformParameter("alpha", alpha);
        _shlChunk->setUniformParameter("isLit", getLit());
        _shlChunk->setUniformParameter("ignoreDisplayLuminance", true); //getIgnoreDisplayLuminance());
        _shlChunk->setUniformParameter("useVertexColors", getUseVertexColors());            
    }
    endEditCP(_shlChunk, SHLChunk::ParametersFieldMask);
    state->addChunk(_shlChunk);
    
    if (isTransparent())
    {
        state->addChunk(_blendChunk);
    }
     
    if (getAlwaysOnTop())
    {
        state->addChunk(_depthTestChunk);
    }
    addChunks(state); // XXX DR This is a hack. Should call Inherited

    return state;
}

void SimpleMaterial::rebuildState(void)
{
    Color3f v3;
    Color4f v4;
    Real32  alpha = 1.f - getTransparency();

    if (_pState != NullFC)
    {
        _pState->clearChunks();
    }
    else
    {
        _pState = State::create();

        addRefCP(_pState);
    }

    prepareLocalChunks();

        beginEditCP(_shlChunk, SHLChunk::ParametersFieldMask);
        {
            _shlChunk->setUniformParameter("diffuseColor", Vec3f(getDiffuse()[0], getDiffuse()[1], getDiffuse()[2]));
            _shlChunk->setUniformParameter("specularColor", Vec3f(getSpecular()[0], getSpecular()[1], getSpecular()[2]));
            _shlChunk->setUniformParameter("emissiveColor", Vec3f(getEmission()[0], getEmission()[1], getEmission()[2]));
            _shlChunk->setUniformParameter("shininess", getShininess());
            _shlChunk->setUniformParameter("alpha", alpha);
            _shlChunk->setUniformParameter("isLit", getLit());
            _shlChunk->setUniformParameter("ignoreDisplayLuminance", true); //getIgnoreDisplayLuminance());
            _shlChunk->setUniformParameter("useVertexColors", getUseVertexColors());
        }
        endEditCP(_shlChunk, SHLChunk::ParametersFieldMask);
        _pState->addChunk(_shlChunk);
    

    if (isTransparent())
    {
        _pState->addChunk(_blendChunk);
    }
    if (getAlwaysOnTop())
    {
        _pState->addChunk(_depthTestChunk);
    }

    addChunks(_pState); // XXX DR This is a hack. Should call Inherited
}

bool SimpleMaterial::isTransparent(void) const
{
    return ((getTransparency() > Eps) || (Inherited::isTransparent()));
}

/*------------------------------- dump ----------------------------------*/

void SimpleMaterial::dump(UInt32    uiIndent,
    const BitVector OSG_CHECK_ARG(bvFlags)) const
{

    SimpleMaterialPtr thisP(*this);

    thisP.dump(uiIndent, FCDumpFlags::RefCount);

    indentLog(uiIndent, PLOG);
    PLOG << "SimpleMaterial at " << this << std::endl;

    indentLog(uiIndent, PLOG);
    PLOG << "\tambient: " << getAmbient() << std::endl;

    indentLog(uiIndent, PLOG);
    PLOG << "\tdiffuse: " << getDiffuse() << std::endl;

    indentLog(uiIndent, PLOG);
    PLOG << "\tspecular: " << getSpecular() << std::endl;

    indentLog(uiIndent, PLOG);
    PLOG << "\tshininess: " << getShininess() << std::endl;

    indentLog(uiIndent, PLOG);
    PLOG << "\temission: " << getEmission() << std::endl;

    indentLog(uiIndent, PLOG);
    PLOG << "\ttransparency: " << getTransparency() << std::endl;

    indentLog(uiIndent, PLOG);
    PLOG << "\tlit: " << getLit() << std::endl;

    indentLog(uiIndent, PLOG);
    PLOG << "\tChunks: " << std::endl;

    for (MFStateChunkPtr::const_iterator i = _mfChunks.begin();
        i != _mfChunks.end(); i++)
    {
        indentLog(uiIndent, PLOG);
        PLOG << "\t" << *i << std::endl;
    }

    indentLog(uiIndent, PLOG);
    PLOG << "SimpleMaterial end " << this << std::endl;
}

/*------------------------------------------------------------------------*/
/*                              cvs id's                                  */

#ifdef OSG_SGI_CC
#pragma set woff 1174
#endif

#ifdef OSG_LINUX_ICC
#pragma warning(disable : 177)
#endif

namespace
{
    static Char8 cvsid_cpp[] = "@(#)$Id:$";
    static Char8 cvsid_hpp[] = OSGSIMPLEMATERIAL_HEADER_CVSID;
    static Char8 cvsid_inl[] = OSGSIMPLEMATERIAL_INLINE_CVSID;

    static Char8 cvsid_fields_hpp[] = OSGSIMPLEMATERIALFIELDS_HEADER_CVSID;
}

#ifdef __sgi
#pragma reset woff 1174
#endif

