/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *               Copyright (C) 2000-2002 by the OpenSG Forum                 *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

#include <stdlib.h>
#include <stdio.h>

#include <OSGConfig.h>

#include <OSGSimpleGeometry.h>
#include <OSGBinaryChecksumHandler.h>
#include <OSGChecksumCalculator.h>
#include "OSGSwitchMaterial.h"

OSG_USING_NAMESPACE

/***************************************************************************\
 *                            Description                                  *
\***************************************************************************/

/*! \class osg::SwitchMaterial

*/

/***************************************************************************\
 *                           Class variables                               *
\***************************************************************************/


/***************************************************************************\
 *                           Class methods                                 *
\***************************************************************************/

void SwitchMaterial::initMethod (void)
{
}


/***************************************************************************\
 *                           Instance methods                              *
\***************************************************************************/

/*-------------------------------------------------------------------------*\
 -  private                                                                 -
\*-------------------------------------------------------------------------*/

/*----------------------- constructors & destructors ----------------------*/

SwitchMaterial::SwitchMaterial(void) :
    Inherited()
{
}

SwitchMaterial::SwitchMaterial(const SwitchMaterial &source) :
    Inherited(source)
{
}

SwitchMaterial::~SwitchMaterial(void)
{
    // ACHTUNG _pState wird der State vom jeweiligen aktiven Material zugewiesen.
    // wenn nun die Materialien zerstoert werden dann wird im Material::~Material
    // ein subRefCP(_pState) aufgerufen. Dieser subRefCP wuerde dann nochmal mit dem
    // gleichen ungueltigen pointer fuer das SwitchMaterial aufgerufen werden und
    // das kracht nat�rlich, deswegen wird er jetzt einfach auf NullFC gesetzt!
    _pState = NullFC;    
}

/*----------------------------- class specific ----------------------------*/

void SwitchMaterial::changed(BitVector whichField, UInt32 origin)
{
    Inherited::changed(whichField, origin);
}

MaterialPtr SwitchMaterial::getMaterial(UInt32 index)
{
    if(index >= _mfMaterials.size())
    {
        FWARNING(("SwitchMaterial::getMaterial : index %u out of range\n", index));
        return NullFC;
    }
    return _mfMaterials[index];
}

MaterialPtr SwitchMaterial::getCurrentMaterial() const
{
    UInt32 choice = getChoice();
    if(choice >= _mfMaterials.size())
    {
        FWARNING(("SwitchMaterial::getCurrentMaterial : current choice %u out of range\n", choice));
        return NullFC;
    }
    return _mfMaterials[choice];
}

void SwitchMaterial::getAnimChannels(osg::Material::ChannelData& data)
{
    SwitchMaterialPtr thisPtr(*this);
    
    data.push_back( Material::ChannelAttributes(thisPtr.getFieldContainerId(), "choice") );
    
    Inherited::getAnimChannels(data);
}


/*! Create a osg::State that represents this Material and return it.
*/

StatePtr SwitchMaterial::makeState(void)
{
    UInt32 choice = getChoice();
    if(choice >= _mfMaterials.size())
    {
        if(!_mfMaterials.empty())
            SWARNING << "SwitchMaterial::makeState: choice index (" << choice << ") out of range!" << std::endl;
        return osg::getDefaultMaterial()->makeState();
    }

    if(_mfMaterials[choice] != NullFC)
    {
        SwitchMaterialPtr tmpPtr(*this);
        if(_mfMaterials[choice] != tmpPtr)
        {
            return _mfMaterials[choice]->makeState();
        }
        else
        {
            SFATAL << "SwitchMaterial::makeState: material recursion!" << std::endl;
        }
    }

    return osg::getDefaultMaterial()->makeState();
}

/*! Rebuild the internal State. Just collects the chunks in the State.
*/

void SwitchMaterial::rebuildState(void)
{
    UInt32 choice = getChoice();
    if(choice >= _mfMaterials.size())
    {
        if(!_mfMaterials.empty())
            SWARNING << "SwitchMaterial::rebuildState: choice index (" << choice << ") out of range!" << std::endl;
        osg::getDefaultMaterial()->rebuildState();
        return;
    }

    if(_mfMaterials[choice] != NullFC)
    {
        SwitchMaterialPtr tmpPtr(*this);
        if(_mfMaterials[choice] != tmpPtr)
        {
            _mfMaterials[choice]->rebuildState();
        }
        else
        {
            SFATAL << "SwitchMaterial::rebuildState: material recursion!" << std::endl;
        }
    }

    osg::getDefaultMaterial()->rebuildState();
}

StatePtr SwitchMaterial::getState(UInt32 index)
{
    UInt32 choice = getChoice();
    if(choice >= _mfMaterials.size())
    {
        if(!_mfMaterials.empty())
            SWARNING << "SwitchMaterial::getState: choice index out of range!" << std::endl;
        return NullFC;
    }

    if(_mfMaterials[choice] != NullFC)
    {
        SwitchMaterialPtr tmpPtr(*this);

        if(_mfMaterials[choice] != tmpPtr)
        {
            if(_mfMaterials[choice]->getState(index) == NullFC)
                rebuildState();
        }
        else
        {
            SFATAL << "SwitchMaterial::getState: material recursion!" << std::endl;
        }

        if(_mfMaterials[choice] != tmpPtr)
            return _mfMaterials[choice]->getState(index);
        else
            SFATAL << "SwitchMaterial::getState: material recursion!" << std::endl;
    }

    return NullFC;
}

StatePtr SwitchMaterial::getShadowState(UInt32 index)
{
    UInt32 choice = getChoice();
    if(choice >= _mfMaterials.size())
    {
        if(!_mfMaterials.empty())
            SWARNING << "SwitchMaterial::getState: choice index out of range!" << std::endl;
        return NullFC;
    }

    if(_mfMaterials[choice] != NullFC)
    {
        SwitchMaterialPtr tmpPtr(*this);

        if(_mfMaterials[choice] != tmpPtr)
        {
            if(_mfMaterials[choice]->getShadowState(index) == NullFC)
                rebuildState();
        }
        else
        {
            SFATAL << "SwitchMaterial::getState: material recursion!" << std::endl;
        }

        if(_mfMaterials[choice] != tmpPtr)
            return _mfMaterials[choice]->getShadowState(index);
        else
            SFATAL << "SwitchMaterial::getState: material recursion!" << std::endl;
    }

    return NullFC;
}

bool SwitchMaterial::isMultiPass(void) const
{
    UInt32 choice = getChoice();
    if(choice >= _mfMaterials.size())
    {
        if(!_mfMaterials.empty())
            SWARNING << "SwitchMaterial::isMultiPass: choice index out of range!" << std::endl;
        return false;
    }

    if(_mfMaterials[choice] != NullFC)
    {
        SwitchMaterialPtr tmpPtr(*this);
        if(_mfMaterials[choice] != tmpPtr)
            return _mfMaterials[choice]->isMultiPass();
        else
            SFATAL << "SwitchMaterial::isMultiPass: material recursion!" << std::endl;
    }

    return false;
}

UInt32 SwitchMaterial::getNPasses(void) const
{
    UInt32 choice = getChoice();
    if(choice >= _mfMaterials.size())
    {
        if(!_mfMaterials.empty())
            SWARNING << "SwitchMaterial::getNPasses: choice index out of range!" << std::endl;
        return 1;
    }

    if(_mfMaterials[choice] != NullFC)
    {
        SwitchMaterialPtr tmpPtr(*this);
        if(_mfMaterials[choice] != tmpPtr)
            return _mfMaterials[choice]->getNPasses();
        else
            SFATAL << "SwitchMaterial::getNPasses: material recursion!" << std::endl;
    }

    return 1;
}

UInt32 SwitchMaterial::getNShadowPasses(void) const
{
    UInt32 choice = getChoice();
    if(choice >= _mfMaterials.size())
    {
        if(!_mfMaterials.empty())
            SWARNING << "SwitchMaterial::getNShadowPasses: choice index out of range!" << std::endl;
        return 1;
    }

    if(_mfMaterials[choice] != NullFC)
    {
        SwitchMaterialPtr tmpPtr(*this);
        if(_mfMaterials[choice] != tmpPtr)
            return _mfMaterials[choice]->getNShadowPasses();
        else
            SFATAL << "SwitchMaterial::getNShadowPasses: material recursion!" << std::endl;
    }

    return 1;
}

/*! Check if the active Material is transparent
*/
bool SwitchMaterial::isTransparent(void) const
{
    UInt32 choice = getChoice();
    if(choice >= _mfMaterials.size())
    {
        if(!_mfMaterials.empty())
            SWARNING << "SwitchMaterial::isTransparent: choice index out of range!" << std::endl;
        return false;
    }

    if(_mfMaterials[choice] != NullFC)
    {
        SwitchMaterialPtr tmpPtr(*this);
        if(_mfMaterials[choice] != tmpPtr)
            return _mfMaterials[choice]->isTransparent();
        else
            SFATAL << "SwitchMaterial::isTransparent: material recursion!" << std::endl;
    }

    return false;
}

bool SwitchMaterial::isCutoutActiveForState(UInt32 index) const
{
    UInt32 choice = getChoice();
    if(choice >= _mfMaterials.size())
    {
        if(!_mfMaterials.empty())
            SWARNING << "SwitchMaterial::isCutoutActiveForState: choice index out of range!" << std::endl;
        return 1;
    }

    if(_mfMaterials[choice] != NullFC)
    {
        SwitchMaterialPtr tmpPtr(*this);
        if(_mfMaterials[choice] != tmpPtr)
            return _mfMaterials[choice]->isCutoutActiveForState(index);
        else
            SFATAL << "SwitchMaterial::isCutoutActiveForState: material recursion!" << std::endl;
    }

    return false;
}

bool SwitchMaterial::isCutoutActive() const
{
    UInt32 choice = getChoice();
    if(choice >= _mfMaterials.size())
    {
        if(!_mfMaterials.empty())
            SWARNING << "SwitchMaterial::isCutoutActive: choice index out of range!" << std::endl;
        return 1;
    }

    if(_mfMaterials[choice] != NullFC)
    {
        SwitchMaterialPtr tmpPtr(*this);
        if(_mfMaterials[choice] != tmpPtr)
            return _mfMaterials[choice]->isCutoutActive();
        else
            SFATAL << "SwitchMaterial::isCutoutActive: material recursion!" << std::endl;
    }

    return false;
}


Int32 SwitchMaterial::getRealSortKey(void) const
{
    UInt32 choice = getChoice();
    if(choice >= _mfMaterials.size())
    {
        if(!_mfMaterials.empty())
            SWARNING << "SwitchMaterial::getRealSortKey: choice index out of range!" << std::endl;
        return Inherited::getRealSortKey();
    }

    if(_mfMaterials[choice] != NullFC)
    {
        SwitchMaterialPtr tmpPtr(*this);
        if(_mfMaterials[choice] != tmpPtr)
            return _mfMaterials[choice]->getRealSortKey();
        else
            SFATAL << "SwitchMaterial::getRealSortKey: material recursion!" << std::endl;
    }

    return Inherited::getRealSortKey();
}

void SwitchMaterial::dump(      UInt32    , 
                         const BitVector ) const
{
    SLOG << "Dump SwitchMaterial NI" << std::endl;
}

MaterialPtr SwitchMaterial::getFirstNonTransparentMaterial(const HitPoint &p) const
{
    UInt32 choice = getChoice();
    if(choice >= _mfMaterials.size())
    {
        if(!_mfMaterials.empty())
            SWARNING << "SwitchMaterial::getFirstNonTransparentMaterial: choice index out of range!" << std::endl;
        return Material::getFirstNonTransparentMaterial(p);
    }

    if(_mfMaterials[choice] != NullFC)
    {
        SwitchMaterialPtr tmpPtr(*this);
        if(_mfMaterials[choice] != tmpPtr)
            return _mfMaterials[choice]->getFirstNonTransparentMaterial(p);
        else
            SFATAL << "SwitchMaterial::getFirstNonTransparentMaterial: material recursion!" << std::endl;
    }

    return Material::getFirstNonTransparentMaterial(p);
}


Vec3f SwitchMaterial::getTransparency(const HitPoint &p) const
{
    UInt32 choice = getChoice();
    if(choice >= _mfMaterials.size())
    {
        if(!_mfMaterials.empty())
            SWARNING << "SwitchMaterial::getTransparency: choice index out of range!" << std::endl;
        return Material::getTransparency(p);
    }

    if(_mfMaterials[choice] != NullFC)
    {
        SwitchMaterialPtr tmpPtr(*this);
        if(_mfMaterials[choice] != tmpPtr)
            return _mfMaterials[choice]->getTransparency(p);
        else
            SFATAL << "SwitchMaterial::getTransparency: material recursion!" << std::endl;
    }

    return Material::getTransparency(p);
}

bool SwitchMaterial::isFullyTransparent() const
{
    UInt32 choice = getChoice();
    if(choice >= _mfMaterials.size())
    {
        if(!_mfMaterials.empty())
            SWARNING << "SwitchMaterial::isFullyTransparent: choice index out of range!" << std::endl;
        return false;
    }

    if(_mfMaterials[choice] != NullFC)
    {
        SwitchMaterialPtr tmpPtr(*this);
        if(_mfMaterials[choice] != tmpPtr)
            return _mfMaterials[choice]->isFullyTransparent();
        else
            SFATAL << "SwitchMaterial::isFullyTransparent: material recursion!" << std::endl;
    }

    return false;

}

unsigned int SwitchMaterial::calcChecksum()
{
    BinaryChecksumHandler bch;
    copyToBin(bch, ChoiceFieldMask|PreviewSceneFieldMask);
    m_checksum = bch.getChecksum();
    UInt32 choice = getChoice();
    if (choice < _mfMaterials.size() && _mfMaterials[choice] != NullFC)
    {
        combineChecksum(m_checksum, _mfMaterials[choice]->calcChecksum());
    }
    m_checksumDirty = false;
    return m_checksum;
}

bool SwitchMaterial::isMultiState(void) const
{
    return true;
}

UInt32 SwitchMaterial::getNStates(void) const
{
    return _mfMaterials.size();
}

std::string SwitchMaterial::getStateName(UInt32 stateIndex) const
{
    if (stateIndex < _mfMaterials.size())
        return Material::getName(_mfMaterials[stateIndex]);

    return std::string();
}

std::vector<std::string> SwitchMaterial::getStateNames() const
{
    std::vector<std::string> stateNames;

    stateNames.reserve(getNStates());
    for (const auto & mat : _mfMaterials)
        stateNames.push_back(Material::getName(mat));

    return stateNames;
}

void SwitchMaterial::applyState(UInt32 stateIndex)
{
    if (stateIndex < getMaterials().getSize())
    {
        SwitchMaterialPtr temp(*this);
        beginEditCP(temp, osg::SwitchMaterial::ChoiceFieldMask);
        setChoice(stateIndex);
        endEditCP(temp, osg::SwitchMaterial::ChoiceFieldMask);
    }
}

bool osg::SwitchMaterial::getCurrentState(UInt32 & index) const
{
    if ( getChoice() >= _mfMaterials.size() )
        return false;

    index = getChoice();
    return true;
}

/*------------------------------------------------------------------------*/
/*                              cvs id's                                  */

#ifdef OSG_SGI_CC
#pragma set woff 1174
#endif

#ifdef OSG_LINUX_ICC
#pragma warning( disable : 177 )
#endif

namespace
{
    static Char8 cvsid_cpp       [] = "@(#)$Id: OSGSwitchMaterial.cpp,v 1.6 2008/12/02 11:37:11 a-m-z Exp $";
    static Char8 cvsid_hpp       [] = OSGSWITCHMATERIALBASE_HEADER_CVSID;
    static Char8 cvsid_inl       [] = OSGSWITCHMATERIALBASE_INLINE_CVSID;

    static Char8 cvsid_fields_hpp[] = OSGSWITCHMATERIALFIELDS_HEADER_CVSID;
}

#ifdef __sgi
#pragma reset woff 1174
#endif

