/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright(C) 2000-2002 by the OpenSG Forum                    *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
 //  Includes
//---------------------------------------------------------------------------

#include <stdlib.h>
#include <stdio.h>
#include <iostream>

#include "OSGConfig.h"
#include <OSGGL.h>

#include <OSGAction.h>
#include <OSGDrawAction.h>
#include <OSGRenderAction.h>
#include <OSGIntersectAction.h>
#include <OSGRenderAction.h>
#include <OSGMaterial.h>
#include <OSGChunkMaterial.h>
#include <OSGSimpleMaterial.h>
#include <OSGSimpleTexturedMaterial.h>
#include "OSGDrawable.h"
#include "OSGGeometry.h"
#include "OSGGeoFunctions.h"


#include <OSGTransform3D.h>

#include "OSGPrimitiveIterator.h"
#include "OSGTriangleIterator.h"
#include "OSGFaceIterator.h"
#include "OSGLineIterator.h"
#include "OSGEdgeIterator.h"

#include "OSGGeoPropPtrs.h"
OSG_USING_NAMESPACE


/***************************************************************************\
 *                            Description                                  *
\***************************************************************************/

/*! \class osg::Geometry
    \ingroup GrpSystemNodeCoresDrawablesGeometry

The base class for all Geometry node types, see \ref PageSystemGeometry for a
description.

*/

/***************************************************************************\
 *                           Class variables                               *
\***************************************************************************/

const UInt16 Geometry::MapPosition       = 1;
const UInt16 Geometry::MapNormal         = Geometry::MapPosition << 1;
const UInt16 Geometry::MapColor          = Geometry::MapNormal << 1;
const UInt16 Geometry::MapSecondaryColor = Geometry::MapColor << 1;
const UInt16 Geometry::MapTexCoords      = Geometry::MapSecondaryColor << 1;
const UInt16 Geometry::MapTexCoords1     = Geometry::MapTexCoords << 1;
const UInt16 Geometry::MapTexCoords2     = Geometry::MapTexCoords1 << 1;
const UInt16 Geometry::MapTexCoords3     = Geometry::MapTexCoords2 << 1;
const UInt16 Geometry::MapTexCoords4     = Geometry::MapTexCoords3 << 1;
const UInt16 Geometry::MapTexCoords5     = Geometry::MapTexCoords4 << 1;
const UInt16 Geometry::MapTexCoords6     = Geometry::MapTexCoords5 << 1;
const UInt16 Geometry::MapTexCoords7     = Geometry::MapTexCoords6 << 1;
const UInt16 Geometry::MapEmpty          = Geometry::MapTexCoords7 << 1;

/***************************************************************************\
 *                           Class methods                                 *
\***************************************************************************/

/*-------------------------------------------------------------------------*\
 -  public                                                                 -
\*-------------------------------------------------------------------------*/

/*! A little helper function to map the OpenGL primitive type to a name.
*/
const char *Geometry::mapType(UInt8 type)
{
    switch(type)
    {
    case OSG_GL_POINTS:         return "Points";
    case OSG_GL_LINES:          return "Lines";
    case OSG_GL_LINE_LOOP:      return "LineLoop";
    case OSG_GL_LINE_STRIP:     return "LineStrip";
    case OSG_GL_TRIANGLES:      return "Triangles";
    case OSG_GL_TRIANGLE_STRIP: return "TriangleStrip";
    case OSG_GL_TRIANGLE_FAN:   return "TriangleFan";
    case OSG_GL_QUADS:          return "Quads";
    case OSG_GL_QUAD_STRIP:     return "QuadStrip";
    case OSG_GL_POLYGON:        return "Polygon";
    }

    return "Unknown Primitive";
}

/*-------------------------------------------------------------------------*\
 -  private                                                                -
\*-------------------------------------------------------------------------*/

void Geometry::initMethod(void)
{
    IntersectAction::registerEnterDefault(getClassType(),
                                     std::bind(reinterpret_cast<Action::Callback>(&Geometry::intersect),
                                               std::placeholders::_1, std::placeholders::_2));
}


/***************************************************************************\
 *                           Instance methods                              *
\***************************************************************************/

/*-------------------------------------------------------------------------*\
 -  public                                                                 -
\*-------------------------------------------------------------------------*/


/*------------- constructors & destructors --------------------------------*/

Geometry::Geometry(void) :
    Inherited(),
    m_vbo(nullptr),
    _volumeCache()
{
}

Geometry::Geometry(const Geometry &source) :
    Inherited(source),
    m_vbo(source.m_vbo),
    _volumeCache(source._volumeCache)
{
}

/*! The destructor automatically dereferences the used properties. If this
    Geometry was the last user that will automatically destroy them.
*/
Geometry::~Geometry(void)
{
    auto &mfIndices = getTempIndices();
    for (UInt32 i = 0; i < mfIndices.getSize(); ++i)
    {
        if (NullFC != mfIndices[i])
            subRefCP(mfIndices[i]);
        mfIndices[i] = NullFC;
    }
}

DynamicVolume Geometry::getGeometryVolume()
{
    updateVolumeCache();

    return _volumeCache;
}

void Geometry::onCreate(const Geometry *source)
{
    // if we're in startup this is the prototype, which shouldn't have an id
    if(GlobalSystemState == Startup)
        return;

    GeometryPtr tmpPtr(*this);
    Inherited::onCreate(source);
}

void Geometry::onDestroy(void)
{
    Inherited::onDestroy();

    GeometryPtr thisP = getPtr();

    if(_sfTypes.getValue() != NullFC)
    {
        beginEditCP(_sfTypes.getValue(), Attachment::ParentsFieldMask);
        {
            _sfTypes.getValue()->subParent(thisP);
        }
        endEditCP(_sfTypes.getValue(), Attachment::ParentsFieldMask);

        subRefCP(_sfTypes.getValue());
    }

    if(_sfLengths.getValue() != NullFC)
    {
        beginEditCP(_sfLengths.getValue(), Attachment::ParentsFieldMask);
        {
            _sfLengths.getValue()->subParent(thisP);
        }
        endEditCP(_sfLengths.getValue(), Attachment::ParentsFieldMask);

        subRefCP(_sfLengths.getValue());
    }

    if(_sfPositions.getValue() != NullFC)
    {
        beginEditCP(_sfPositions.getValue(), Attachment::ParentsFieldMask, ChangedOrigin::External | ChangedOrigin::SubParent);
        {
            _sfPositions.getValue()->subParent(thisP);
        }
        endEditCP(_sfPositions.getValue(), Attachment::ParentsFieldMask, ChangedOrigin::External | ChangedOrigin::SubParent);

        subRefCP(_sfPositions.getValue());
    }

    if(_sfNormals.getValue() != NullFC)
    {
        beginEditCP(_sfNormals.getValue(), Attachment::ParentsFieldMask);
        {
            _sfNormals.getValue()->subParent(thisP);
        }
        endEditCP(_sfNormals.getValue(), Attachment::ParentsFieldMask);

        subRefCP(_sfNormals.getValue());
    }

    if(_sfColors.getValue() != NullFC)
    {
        beginEditCP(_sfColors.getValue(), Attachment::ParentsFieldMask);
        {
            _sfColors.getValue()->subParent(thisP);
        }
        endEditCP(_sfColors.getValue(), Attachment::ParentsFieldMask);

        subRefCP(_sfColors.getValue());
    }

    if(_sfSecondaryColors.getValue() != NullFC)
    {
        beginEditCP(_sfSecondaryColors.getValue(),
                        Attachment::ParentsFieldMask);
        {
            _sfSecondaryColors.getValue()->subParent(thisP);
        }
        endEditCP(_sfSecondaryColors.getValue(),
                        Attachment::ParentsFieldMask);

        subRefCP(_sfSecondaryColors.getValue());
    }

    if(_sfTexCoords.getValue() != NullFC)
    {
        beginEditCP(_sfTexCoords.getValue(), Attachment::ParentsFieldMask);
        {
            _sfTexCoords.getValue()->subParent(thisP);
        }
        endEditCP(_sfTexCoords.getValue(), Attachment::ParentsFieldMask);

        subRefCP(_sfTexCoords.getValue());
    }

    if(_sfTexCoords1.getValue() != NullFC)
    {
        beginEditCP(_sfTexCoords1.getValue(), Attachment::ParentsFieldMask);
        {
            _sfTexCoords1.getValue()->subParent(thisP);
        }
        endEditCP(_sfTexCoords1.getValue(), Attachment::ParentsFieldMask);

        subRefCP(_sfTexCoords1.getValue());
    }

    if(_sfTexCoords2.getValue() != NullFC)
    {
        beginEditCP(_sfTexCoords2.getValue(), Attachment::ParentsFieldMask);
        {
            _sfTexCoords2.getValue()->subParent(thisP);
        }
        endEditCP(_sfTexCoords2.getValue(), Attachment::ParentsFieldMask);

        subRefCP(_sfTexCoords2.getValue());
    }

    if(_sfTexCoords3.getValue() != NullFC)
    {
        beginEditCP(_sfTexCoords3.getValue(), Attachment::ParentsFieldMask);
        {
            _sfTexCoords3.getValue()->subParent(thisP);
        }
        endEditCP(_sfTexCoords3.getValue(), Attachment::ParentsFieldMask);

        subRefCP(_sfTexCoords3.getValue());
    }

    if(_sfTexCoords4.getValue() != NullFC)
    {
        beginEditCP(_sfTexCoords4.getValue(), Attachment::ParentsFieldMask);
        {
            _sfTexCoords4.getValue()->subParent(thisP);
        }
        endEditCP(_sfTexCoords4.getValue(), Attachment::ParentsFieldMask);

        subRefCP(_sfTexCoords4.getValue());
    }

    if(_sfTexCoords5.getValue() != NullFC)
    {
        beginEditCP(_sfTexCoords5.getValue(), Attachment::ParentsFieldMask);
        {
            _sfTexCoords5.getValue()->subParent(thisP);
        }
        endEditCP(_sfTexCoords5.getValue(), Attachment::ParentsFieldMask);

        subRefCP(_sfTexCoords5.getValue());
    }

    if(_sfTexCoords6.getValue() != NullFC)
    {
        beginEditCP(_sfTexCoords6.getValue(), Attachment::ParentsFieldMask);
        {
            _sfTexCoords6.getValue()->subParent(thisP);
        }
        endEditCP(_sfTexCoords6.getValue(), Attachment::ParentsFieldMask);

        subRefCP(_sfTexCoords6.getValue());
    }

    if(_sfTexCoords7.getValue() != NullFC)
    {
        beginEditCP(_sfTexCoords7.getValue(), Attachment::ParentsFieldMask);
        {
            _sfTexCoords7.getValue()->subParent(thisP);
        }
        endEditCP(_sfTexCoords7.getValue(), Attachment::ParentsFieldMask);

        subRefCP(_sfTexCoords7.getValue());
    }

    if(_sfIndices.getValue() != NullFC)
    {
        beginEditCP(_sfIndices.getValue(), Attachment::ParentsFieldMask);
        {
            _sfIndices.getValue()->subParent(thisP);
        }
        endEditCP(_sfIndices.getValue(), Attachment::ParentsFieldMask);

        subRefCP(_sfIndices.getValue());
    }

    subRefCP(_sfMaterial.getValue());
}

void Geometry::makeSingleIndexed()
{
    MFUInt16 indexMapping = getIndexMapping();
    if (indexMapping.size() > 1)
    {
        GeometryPtr geo(this);
        if(geo != osg::NullFC)
        {
            // separate the properties
            osg::separateProperties(geo);
            osg::createSingleIndex(geo);
        }
    }
}
/*------------------------------ access -----------------------------------*/

void Geometry::updateVolumeCache()
{
    if (!_volumeCache.isEmpty())
        return;

    bool useExternalVolume = false;

    if (m_getVolume)
    {
        m_getVolume(this, _volumeCache);
        useExternalVolume = !_volumeCache.isEmpty();
    }
    
    if (!useExternalVolume)
    {
        // calculate volume.
        GeoPositionsPtr pos = getPositions();

        _volumeCache.setValid();

        if (pos == NullFC || pos->getSize() == 0)
        {
            // get the first parent, assuming that all children are identical
            if ( getParents().size() > 0 && (getGeometryType() & GEOMETRY_TYPE_SHELL))
            {
                NodePtr parentNode = getParents()[0];
                if (parentNode != NullFC)
                {
                    _volumeCache.setEmpty(true);
                    for (size_t idx = 0; idx < parentNode->getNChildren(); ++idx)
                    {
                        NodePtr child = parentNode->getChild(idx);
                        if (child != NullFC && child->getCore() != NullFC)
                        {
                            GeometryPtr childGeo = GeometryPtr::dcast(child->getCore());
                            if (childGeo != NullFC)
                            {
                                DynamicVolume vol;
                                vol.setEmpty(true);
                                childGeo->adjustVolume(vol);
                                _volumeCache.extendBy(vol);
                                Pnt3f min, max;
                                _volumeCache.getBounds(min, max);
                            }
                        }
                    }
                }
            }

            return;                  // Node has no points, no volume
        }

        _volumeCache.setEmpty(true); // reset the volume cache

        GeoPositions4fPtr positions4f = GeoPositions4fPtr::dcast(getPositions());
        GeoPositions3fPtr positions3f = GeoPositions3fPtr::dcast(getPositions());
        Pnt3f minExtend = Pnt3f(FLT_MAX, FLT_MAX, FLT_MAX);
        Pnt3f maxExtend = Pnt3f(-FLT_MAX, -FLT_MAX, -FLT_MAX);
        if (positions4f != NullFC)
        {
            Pnt4f* vertices = reinterpret_cast<Pnt4f*>(&(positions4f->getField()[0]));
            const size_t numVertices = positions4f->getSize();
            for (size_t idx = 0; idx < numVertices; ++idx)
            {
                minExtend[0] = std::min(minExtend[0], vertices[idx][0]);
                minExtend[1] = std::min(minExtend[1], vertices[idx][1]);
                minExtend[2] = std::min(minExtend[2], vertices[idx][2]);
                maxExtend[0] = std::max(maxExtend[0], vertices[idx][0]);
                maxExtend[1] = std::max(maxExtend[1], vertices[idx][1]);
                maxExtend[2] = std::max(maxExtend[2], vertices[idx][2]);
            }
        }
        else if (positions3f != NullFC)
        {
            Pnt3f* vertices = reinterpret_cast<Pnt3f*>(&(positions3f->getField()[0]));
            const size_t numVertices = positions3f->getSize();
            for (size_t idx = 0; idx < numVertices; ++idx)
            {
                minExtend[0] = std::min(minExtend[0], vertices[idx][0]);
                minExtend[1] = std::min(minExtend[1], vertices[idx][1]);
                minExtend[2] = std::min(minExtend[2], vertices[idx][2]);
                maxExtend[0] = std::max(maxExtend[0], vertices[idx][0]);
                maxExtend[1] = std::max(maxExtend[1], vertices[idx][1]);
                maxExtend[2] = std::max(maxExtend[2], vertices[idx][2]);
            }
        }
        _volumeCache.extendBy(minExtend);
        _volumeCache.extendBy(maxExtend);
    }
}

void Geometry::adjustVolume(Volume & volume)
{
    updateVolumeCache();

    Transform3DPtr geoTrans = _sfTransform.getValue();
    if (!_volumeCache.isEmpty())
    {
        // use cached volume.
        volume.setValid();
        volume.extendBy(_volumeCache);
        if (geoTrans != NullFC)
            volume.transform(geoTrans->getMatrix());
        return;
    }
}

void Geometry::invalidateVolumeCache()
{
    if(m_invalidateVolume)
    {
        m_invalidateVolume(this);
    }
    _volumeCache.setEmpty();
}

/*---------------------------- pointer ------------------------------------*/

GeometryPtr Geometry::getPtr(void) const
{
    return GeometryPtr(*this);
}




/*------------------------------- dump ----------------------------------*/

void Geometry::dump(      UInt32    uiIndent,
                    const BitVector bvFlags) const
{
    UInt32 i;

    GeometryPtr thisP = getPtr();

    indentLog(uiIndent, PLOG);

    PLOG << "GeoCore"
         << "("
         << getContainerId(thisP)
         << ") : "
         << getType().getName()
         << " "
         << _attachmentMap.getValue().size()
         << " attachments | "
         << this
         << std::endl;

    indentLog(uiIndent, PLOG);
    PLOG << "[" << std::endl;

    indentLog(uiIndent + 4, PLOG);
    PLOG << "Parents : " << std::endl;

    for(i = 0; i < _parents.size(); i++)
    {
        indentLog(uiIndent + 4, PLOG);
        PLOG << "           " << i << ") " << &(*(_parents[i])) << std::endl;
    }

    indentLog(uiIndent, PLOG);
    PLOG << "]" << std::endl;

    indentLog(uiIndent, PLOG);
    PLOG << "{" << std::endl;

    uiIndent += 4;

    if(getPositions() != NullFC)
    {
        getPositions()->dump(uiIndent, bvFlags);
        for(i = 0; i < getPositions()->getSize(); i++)
        {
            Pnt3f p = getPositions()->getValue(i);
            printf("p %d: %f %f %f\n", i, p[0], p[1], p[2]);
        }
    }

    if(getIndices() != NullFC)
    {
        getIndices()->dump(uiIndent, bvFlags);
        Int32 indexMapSize = getMFIndexMapping()->getSize();
        Int32 indexCount = getIndices()->getSize() / indexMapSize;
        for(i = 0; i < indexCount; i++)
        {
            for(Int32 j = 0; j < indexMapSize; j++)
            {
                Int32 index = getIndices()->getValue(i * indexMapSize + j);
                printf("i %d/%d: %d\n", i, j, index);
            }
        }
    }

    if(getMaterial() != NullFC)
    {
        getMaterial()->dump(uiIndent, bvFlags);
    }

    if(getTypes() != NullFC)
    {
        getTypes()->dump(uiIndent, bvFlags);
        for(i = 0; i < getTypes()->getSize(); i++)
        {
            Int32 t = getTypes()->getValue(i);
            printf("t %d: %d\n", i, t);
        }
    }

    if(getLengths() != NullFC)
    {
        getLengths()->dump(uiIndent, bvFlags);
        for(i = 0; i < getLengths()->getSize(); i++)
        {
            UInt32 l = getLengths()->getValue(i);
            printf("l %d: %u\n", i, l);
        }
    }

    if(getNormals() != NullFC)
    {
        getNormals()->dump(uiIndent, bvFlags);
        for(i = 0; i < getNormals()->getSize(); i++)
        {
            Vec3f n = getNormals()->getValue(i);
            printf("n %d: %f %f %f\n", i, n[0], n[1], n[2]);
        }
    }

    if(getColors() != NullFC)
    {
        getColors()->dump(uiIndent, bvFlags);
    }

    if(getTexCoords() != NullFC)
    {
        getTexCoords()->dump(uiIndent, bvFlags);
    }

    if(getTexCoords1() != NullFC)
    {
        getTexCoords1()->dump(uiIndent, bvFlags);
    }

    if(getTexCoords2() != NullFC)
    {
        getTexCoords2()->dump(uiIndent, bvFlags);
    }

    if(getTexCoords3() != NullFC)
    {
        getTexCoords3()->dump(uiIndent, bvFlags);
    }

    if(getTexCoords4() != NullFC)
    {
        getTexCoords4()->dump(uiIndent, bvFlags);
    }

    if(getTexCoords5() != NullFC)
    {
        getTexCoords5()->dump(uiIndent, bvFlags);
    }

    if(getTexCoords6() != NullFC)
    {
        getTexCoords6()->dump(uiIndent, bvFlags);
    }

    if(getTexCoords7() != NullFC)
    {
        getTexCoords7()->dump(uiIndent, bvFlags);
    }

    uiIndent -= 4;

    AttachmentContainer::dump(uiIndent, bvFlags);

    indentLog(uiIndent, PLOG);
    PLOG << "}" << std::endl;
}

#ifndef OSG_SUPPORT_NO_GEO_INTERFACE

/*! Find the property pointer for the given mapID, i.e.
osg::Geometry::MapPosition and relatives.
*/
GeoPropertyArrayInterface *Geometry::getProperty(Int32 mapID)
{
    GeoPropertyArrayInterface *pP = 0;

    switch(mapID)
    {
        case 0:
            pP = 0;
            break;
        case MapPosition:
            pP =(getPositions()       == NullFC) ? 0 : &(*getPositions());
            break;
        case MapNormal:
            pP =(getNormals()         == NullFC) ? 0 : &(*getNormals());
            break;
        case MapColor:
            pP =(getColors()          == NullFC) ? 0 : &(*getColors());
            break;
        case MapSecondaryColor:
            pP =(getSecondaryColors() == NullFC) ? 0 : &(*getSecondaryColors());
            break;
        case MapTexCoords:
            pP =(getTexCoords()       == NullFC) ? 0 : &(*getTexCoords());
            break;
        case MapTexCoords1:
            pP =(getTexCoords1()      == NullFC) ? 0 : &(*getTexCoords1());
            break;
        case MapTexCoords2:
            pP =(getTexCoords2()      == NullFC) ? 0 : &(*getTexCoords2());
            break;
        case MapTexCoords3:
            pP =(getTexCoords3()      == NullFC) ? 0 : &(*getTexCoords3());
            break;
        case MapTexCoords4:
            pP =(getTexCoords4()      == NullFC) ? 0 : &(*getTexCoords4());
            break;
        case MapTexCoords5:
            pP =(getTexCoords5()      == NullFC) ? 0 : &(*getTexCoords5());
            break;
        case MapTexCoords6:
            pP =(getTexCoords6()      == NullFC) ? 0 : &(*getTexCoords6());
            break;
        case MapTexCoords7:
            pP =(getTexCoords7()      == NullFC) ? 0 : &(*getTexCoords7());
            break;
        default:
            FFATAL(("Invalid mapID(%d) in Geometry::getProperty()\n",
                    mapID));
          break;
    }

    return pP;
}

#endif

/*! Calc the indices into the index field for the given attributes. This is the
    index of the given attribute's index in a multi-index block.

    Returns -1 for non- or single-indexed geometries, or if the given attribute
    is not used.
*/
Int16  Geometry::calcMappingIndex(UInt16 attrib) const
{
    UInt16 nmappings = getMFIndexMapping()->size();
    Int16 i;

    for(i = nmappings - 1; i >= 0; i--)
    {
        if(getIndexMapping(i) & attrib )
            break;
    }

    return i;
}



/*! Check if the geometry can be merged into this one, return true if yes
They need to have the same material and the same mappings or the same set of
attributes.
*/
bool Geometry::isMergeable( const GeometryPtr other )
{
    if(other == NullFC)
        return false;

    // check the transforms!
    if(getTransform() != NullFC && other->getTransform() != NullFC)
    {
        if(getTransform()->getMatrix() != other->getTransform()->getMatrix())
            return false;
    }
    else if(getTransform() != NullFC)
    {
        if(!getTransform()->isIdentity())
            return false;
    }
    else if(other->getTransform() != NullFC)
    {
        if(!other->getTransform()->isIdentity())
            return false;
    }

    // to make shure shells are only merged with shells and so on
    if( getGeometryType() != other->getGeometryType())
        return false;

    if (MergeIndex(other)!=-1 || getPositions()==NullFC)
        return true;
    else
        return false;
}

/*! Merge the geometry into this one, return true if successful.
*/
bool Geometry::merge( const GeometryPtr other )
{
    if (other == NullFC)
    {
        FDEBUG(("Geometry::merge: other = NullFC!!!\n"));
        return false;
    }

    // check the transforms!
    if(getTransform() != NullFC && other->getTransform() != NullFC)
    {
        if(getTransform()->getMatrix() != other->getTransform()->getMatrix())
            return false;
    }
    else if(getTransform() != NullFC)
    {
        if(!getTransform()->isIdentity())
            return false;
    }
    else if(other->getTransform() != NullFC)
    {
        if(!other->getTransform()->isIdentity())
            return false;
    }

    //first check whethet the current geometry is empty
    //if empty just add everything from the other geometry
    if (getPositions()==NullFC)
    {
        if (other->getPositions()!=NullFC)
            setPositions(other->getPositions()->clone());

        if (other->getTypes()!=NullFC)
            setTypes(other->getTypes()->clone());

        if (other->getLengths()!=NullFC)
            setLengths(other->getLengths()->clone());

        if (other->getNormals()!=NullFC)
            setNormals(other->getNormals()->clone());

        if (other->getColors()!=NullFC)
            setColors(other->getColors()->clone());

        if (other->getSecondaryColors()!=NullFC)
            setSecondaryColors(other->getSecondaryColors()->clone());

        if (other->getTexCoords()!=NullFC)
            setTexCoords(other->getTexCoords()->clone());

        if (other->getTexCoords1()!=NullFC)
            setTexCoords1(other->getTexCoords1()->clone());

        if (other->getTexCoords2()!=NullFC)
            setTexCoords2(other->getTexCoords2()->clone());

        if (other->getTexCoords3()!=NullFC)
            setTexCoords3(other->getTexCoords3()->clone());

        if (other->getTexCoords4()!=NullFC)
            setTexCoords4(other->getTexCoords4()->clone());

        if (other->getTexCoords5()!=NullFC)
            setTexCoords5(other->getTexCoords5()->clone());

        if (other->getTexCoords6()!=NullFC)
            setTexCoords6(other->getTexCoords6()->clone());

        if (other->getTexCoords7()!=NullFC)
            setTexCoords7(other->getTexCoords7()->clone());

        if(other->getIndices()!=NullFC)
            setIndices(other->getIndices()->clone());

        if(other->getMFIndexMapping()!=NULL)
            editMFIndexMapping()->setValues(*(other->getMFIndexMapping()));

        setMaterial(other->getMaterial());


        return true;
    }


    //if not empty continue trying a normal merge
    Int16 mergetype = MergeIndex( other );

    if(mergetype >= 0 && mergetype <= 6)
    {
        // in this case we create 32 bit indices to have enough room for both geometries.
        if(GeoIndicesUI16Ptr::dcast(getIndices()) != NullFC)
        {
            GeoIndicesPtr indices = getIndices();
            UInt32 indices_size = indices->getSize();
            GeoIndicesUI32Ptr indicesUI32 = GeoIndicesUI32::create();
            MFUInt32 &dst = indicesUI32->editField();
            dst.reserve(indices_size);
            beginEditCP(indicesUI32, osg::FieldBits::AllFields);
                for (UInt32 i = 0; i < indices_size; ++i)
                    dst.push_back(indices->getValue(i));
            endEditCP(indicesUI32, osg::FieldBits::AllFields);

            beginEditCP(static_cast<GeometryPtr>(this), Geometry::IndicesFieldMask);
                setIndices(indicesUI32);
            endEditCP(static_cast<GeometryPtr>(this), Geometry::IndicesFieldMask);
        }
    }

    switch ( mergetype )
    {
    case 0: merge0( other ); break;
    case 1: merge1( other ); break;
    case 2: merge2( other ); break;
    case 3: merge3( other ); break;
    case 4: merge4( other ); break;
    case 5: merge5( other ); break;
    case 6: merge6( other ); break;
    default: return false;
    }
    return true;
}


Action::ResultE Geometry::intersect(Action * action)
{
    IntersectAction     * ia  = dynamic_cast<IntersectAction*>(action);

    const BoxVolume      &vol = ia->getActNode()->editVolume(true);

    if(vol.isValid() && !vol.intersect(ia->getLine()))
    {
        return Action::Skip; //bv missed -> can not hit children
    }

    Transform3DPtr geoTrans = _sfTransform.getValue();
    if( geoTrans != NullFC )
    {
        Matrix m = geoTrans->getMatrix();

        m.invert();

        Pnt3f pos;
        Vec3f dir;

#ifndef OSG_2_PREP
        m.multFullMatrixPnt(ia->getLine().getPosition (), pos);
        m.multMatrixVec    (ia->getLine().getDirection(), dir);
#else
        m.multFull(ia->getLine().getPosition (), pos);
        m.mult    (ia->getLine().getDirection(), dir);
#endif

        ia->setLine(Line(pos, dir), ia->getMaxDist());
        ia->scale(dir.length());
    }

    TriangleIterator it  = this->beginTriangles();
    TriangleIterator end = this->endTriangles  ();
    Real32 t;
    Vec3f norm;

    for(; it != end; ++it)
    {
        if(ia->getLine().intersect(it.getPosition(0),
                                     it.getPosition(1),
                                     it.getPosition(2), t, &norm))
        {
            ia->setHit(t, ia->getActNode(), it.getIndex(), norm);
        }
    }

    if( geoTrans != NullFC )
    {
        Matrix m = geoTrans->getMatrix();

        Pnt3f pos;
        Vec3f dir;

#ifndef OSG_2_PREP
        m.multFullMatrixPnt(ia->getLine().getPosition (), pos);
        m.multMatrixVec    (ia->getLine().getDirection(), dir);
#else
        m.multFull(ia->getLine().getPosition (), pos);
        m.mult    (ia->getLine().getDirection(), dir);
#endif

        ia->setLine(Line(pos, dir), ia->getMaxDist());
        ia->scale(dir.length());
    }

    return Action::Continue;
}

/*! React to field changes, take care of incrementing/decrementing the
    reference count of the changed properties.
*/
void Geometry::changed(BitVector whichField,
                       UInt32    origin   )
{
    if(whichField & TypesFieldMask)
    {

    }

    if(whichField & LengthsFieldMask)
    {

    }
    if( whichField & GeometryTypeFieldMask)
    {
        invalidateVolumeCache();
    }
    if(whichField & PositionsFieldMask)
    {
        // added this to detect subParent() changes from Geometry::destroy() which would lead
        // to a invalid getParent() in invalidateVolume() for shared geometries or geo properties!
        if (!(origin & ChangedOrigin::SubParent))
        {
            invalidateVolumeCache();
            invalidateVolume();
        }
    }

    if(whichField & NormalsFieldMask)
    {

    }

    if(whichField & ColorsFieldMask)
    {

    }

    if(whichField & SecondaryColorsFieldMask)
    {

    }

    if(whichField & TexCoordsFieldMask)
    {

    }

    if(whichField & TexCoords1FieldMask)
    {

    }

    if(whichField & TexCoords2FieldMask)
    {

    }

    if(whichField & TexCoords3FieldMask)
    {

    }

    if(whichField & TexCoords4FieldMask)
    {

    }

    if(whichField & TexCoords5FieldMask)
    {

    }

    if(whichField & TexCoords6FieldMask)
    {

    }

    if(whichField & TexCoords7FieldMask)
    {

    }

    if(whichField & IndicesFieldMask)
    {

    }

    if(whichField & MaterialFieldMask)
    {

    }

    if(whichField & (IndicesFieldMask | LengthsFieldMask |
                     TypesFieldMask   | IndexMappingFieldMask)
      )
    {
    }

    Inherited::changed(whichField, origin);

    const BitVector shellUpdateMask = ~(NodeCore::ParentsFieldMask | TempIndicesFieldMask | AttachmentsFieldMask | MaterialFieldMask);
    if(whichField & shellUpdateMask)
    {
        requestShellUpdate();
    }

    // If the indices has changed which implies the topology may be different now, clear all the stored
    // temporary indices buffers.
    if (whichField & IndicesFieldMask)
    {
        GeometryPtr thisP = getPtr();
        GeoTmpIndexEditInterface::setNullforAll(thisP);
    }
}

void Geometry::requestShellUpdate()
{
    
    if( getGeometryType() & Geometry::GEOMETRY_TYPE_SHELL)
        return;

    // to handle changes for component geometries, we need to trigger a change on the parent geometry to get a rebuild
    if( getParents().size() > 0)
    {
        const size_t numParents = getParents().size();
        for( size_t idx = 0; idx < numParents; ++idx)
        {
            NodePtr geoNode = getParents()[idx];
            if( geoNode != osg::NullFC)
            {
                NodePtr parentNode = geoNode->getParent();
                if( parentNode != NullFC)
                {
                    GeometryPtr parentGeo = GeometryPtr::dcast(parentNode->getCore());
                    if( parentGeo != osg::NullFC && (parentGeo->getGeometryType() & Geometry::GEOMETRY_TYPE_SHELL))
                    {
                        // we need to trigger an update on the shell to rebuild the VBO
                        beginEditCP( parentGeo, Geometry::GeometryTypeFieldMask);
                        endEditCP( parentGeo, Geometry::GeometryTypeFieldMask);
                    }
                }
            }
        }
    }
}

/*--------------------------- Triangle Iterator --------------------------------*/

/*! Return a TriangleIterator poiting to the beginning of the Geometry.
*/
TriangleIterator Geometry::beginTriangles(void) const
{
    TriangleIterator it(this->getPtr());

    it.setToBegin();

    return it;
}

/*! Return a TriangleIterator poiting to the end of the Geometry.
*/
TriangleIterator Geometry::endTriangles(void) const
{
    TriangleIterator it(this->getPtr());

    it.setToEnd();

    return it;
}

/*-------------------------- Primitive Iterator --------------------------------*/

/*! Return a PrimitiveIterator poiting to the beginning of the Geometry.
*/
PrimitiveIterator Geometry::beginPrimitives(void) const
{
    PrimitiveIterator it(this->getPtr());

    it.setToBegin();

    return it;
}

/*! Return a PrimitiveIterator poiting to the end of the Geometry.
*/
PrimitiveIterator Geometry::endPrimitives(void) const
{
    PrimitiveIterator it(this->getPtr());

    it.setToEnd();

    return it;
}

/*---------------------------- Face Iterator ---------------------------------*/

/*! Return a FaceIterator poiting to the beginning of the Geometry.
*/
FaceIterator Geometry::beginFaces(void) const
{
    FaceIterator it(this->getPtr());

    it.setToBegin();

    return it;
}

/*! Return a FaceIterator poiting to the end of the Geometry.
*/
FaceIterator Geometry::endFaces(void) const
{
    FaceIterator it(this->getPtr());

    it.setToEnd();

    return it;
}

/*---------------------------- Line Iterator ---------------------------------*/

/*! Return a LineIterator poiting to the beginning of the Geometry.
*/
LineIterator Geometry::beginLines(void) const
{
    LineIterator it(this->getPtr());

    it.setToBegin();

    return it;
}

/*! Return a LineIterator poiting to the end of the Geometry.
*/
LineIterator Geometry::endLines(void) const
{
    LineIterator it(this->getPtr());

    it.setToEnd();

    return it;
}

/*---------------------------- Edge Iterator ---------------------------------*/

/*! Return a EdgeIterator poiting to the beginning of the Geometry.
*/
EdgeIterator Geometry::beginEdges(void) const
{
    EdgeIterator it(this->getPtr());

    it.setToBegin();

    return it;
}

/*! Return a EdgeIterator poiting to the end of the Geometry.
*/
EdgeIterator Geometry::endEdges(void) const
{
    EdgeIterator it(this->getPtr());

    it.setToEnd();

    return it;
}


/*! Clone the geometry, i.e. create a new Geometry that uses all the same
    properties the given one uses.
*/

GeometryPtr Geometry::clone(void)
{
    GeometryPtr geo = Geometry::create();
    cloneTo(geo);
    return geo;
}

/*! Clone the geometry properties into a target geometry.
*/
void Geometry::cloneTo(GeometryPtr target)
{
    if (target == NullFC)
        return;

    // Create copies of the attributes and set them on the target
    beginEditCP(target, osg::FieldBits::AllFields);
    {
        if(getTypes() != NullFC)
        {
            target->setTypes(getTypes()->clone());
        }

        if(getLengths() != NullFC)
        {
            target->setLengths(getLengths()->clone());
        }

        if(getPositions() != NullFC)
        {
            target->setPositions(getPositions()->clone());
        }

        if(getNormals() != NullFC)
        {
            target->setNormals(getNormals()->clone());
        }

        if(getColors() != NullFC)
        {
            target->setColors(getColors()->clone());
        }

        if(getSecondaryColors() != NullFC)
        {
            target->setSecondaryColors(getSecondaryColors()->clone());
        }

        if(getTexCoords() != NullFC)
        {
            target->setTexCoords(getTexCoords()->clone());
        }

        if(getTexCoords1() != NullFC)
        {
            target->setTexCoords1(getTexCoords1()->clone());
        }

        if(getTexCoords2() != NullFC)
        {
            target->setTexCoords2(getTexCoords2()->clone());
        }

        if(getTexCoords3() != NullFC)
        {
            target->setTexCoords3(getTexCoords3()->clone());
        }

        if(getTexCoords4() != NullFC)
        {
            target->setTexCoords4(getTexCoords4()->clone());
        }

        if(getTexCoords5() != NullFC)
        {
            target->setTexCoords5(getTexCoords5()->clone());
        }

        if(getTexCoords6() != NullFC)
        {
            target->setTexCoords6(getTexCoords6()->clone());
        }

        if(getTexCoords7() != NullFC)
        {
            target->setTexCoords7(getTexCoords7()->clone());
        }

        if(getIndices() != NullFC)
        {
            target->setIndices(getIndices()->clone());
        }

        if(getMFIndexMapping() != NULL)
        {
            target->editMFIndexMapping()->setValues(*getMFIndexMapping());
        }

        target->setMaterial(getMaterial());
    }
    endEditCP(target, osg::FieldBits::AllFields);
}

bool Geometry::CompareMaterials(MaterialPtr m1, MaterialPtr m2)
{
    if(m1 == m2)
        return true;

    ChunkMaterialPtr cm1=ChunkMaterialPtr::dcast(m1);
    ChunkMaterialPtr cm2=ChunkMaterialPtr::dcast(m2);

    if (cm1==NullFC || cm2==NullFC) return false;

    const MFStateChunkPtr &chunks1 = (*cm1->getMFChunks());
    const MFStateChunkPtr &chunks2 = (*cm2->getMFChunks());

    if (chunks1.size()!=chunks2.size()) return false;

    MFStateChunkPtr::const_iterator matIt  = chunks1.begin();
    MFStateChunkPtr::const_iterator matEnd = chunks1.end ();

    MFStateChunkPtr::const_iterator i;

    while (matIt!=matEnd)
    {
        i = chunks2.find(*matIt);

        if (i == chunks2.end()) return false;

        ++matIt;
    }

    if(cm1->getState() == NullFC)
        cm1->rebuildState();
    if(cm2->getState() == NullFC)
        cm2->rebuildState();

    const MFStateChunkPtr &statechunks1 = (*cm1->getState()->getMFChunks());
    const MFStateChunkPtr &statechunks2 = (*cm2->getState()->getMFChunks());

    if (statechunks1.size()!=statechunks2.size()) return false;

    matIt  = statechunks1.begin();
    matEnd = statechunks1.end ();

    while (matIt!=matEnd)
    {
        i = statechunks2.find(*matIt);

        if (i == statechunks2.end()) return false;

        ++matIt;
    }

    return true;
}

/*! Checks if geometry can be merged into this one.
return -1 if merging is not possible.
0 - geometries are identical ( use merge0 )
1 - ...
2 - ...
3 - ...
...

They need to have the same material and the same set of
attributes.
*/
Int16 Geometry::MergeIndex( const GeometryPtr other )
{
    if ( getMaterial() != other->getMaterial() ) return -1;
    //if (!CompareMaterials(getMaterial(),other->getMaterial())) return -1;


    //compare the existing attributes
    if ( ( (        getNormals()            != NullFC ) ^
           ( other->getNormals()            != NullFC )
        ) ||
        ( (        getColors()              != NullFC ) ^
          ( other->getColors()              != NullFC )
        ) ||
        ( (        getSecondaryColors()     != NullFC ) ^
          ( other->getSecondaryColors()     != NullFC )
        ) ||
        ( (        getTexCoords()           != NullFC ) ^
          ( other->getTexCoords()           != NullFC )
        ) ||
        ( (        getTexCoords1()          != NullFC ) ^
          ( other->getTexCoords1()          != NullFC )
        ) ||
        ( (        getTexCoords2()          != NullFC ) ^
          ( other->getTexCoords2()          != NullFC )
        ) ||
        ( (        getTexCoords3()          != NullFC ) ^
          ( other->getTexCoords3()          != NullFC )
        ) ||
        ( (        getTexCoords4()          != NullFC ) ^
          ( other->getTexCoords4()          != NullFC )
        ) ||
        ( (        getTexCoords5()          != NullFC ) ^
          ( other->getTexCoords5()          != NullFC )
        ) ||
        ( (        getTexCoords6()          != NullFC ) ^
          ( other->getTexCoords6()          != NullFC )
        ) ||
        ( (        getTexCoords7()          != NullFC ) ^
          ( other->getTexCoords7()          != NullFC )
        ) )
        return -1;

    /*at this point the geometries should have
    the same material and attributes
    so they should be mergeable
    remains to check indexing.
    */
    UInt16 nmap  = getMFIndexMapping()->size();
    UInt16 onmap = other->getMFIndexMapping()->size();

    if ( nmap == onmap ) return 0;
    else
    {
        //non-indexed in single-indexed
        if ( nmap == 1 && onmap == 0 ) return 1;
        //single-indexed in non-indexed
        if ( nmap == 0 && onmap == 1 ) return 2;

        //non-indexed in multi-indexed
        if ( nmap > 1 && onmap == 0 ) return 3;
        //multi-indexed in non-indexed
        if ( nmap == 0 && onmap > 1 ) return 4;

        //single-indexed in multi-indexed
        if ( nmap > 1 && onmap == 1 ) return 5;
        //multi-indexed in single-indexed
        if ( nmap == 1 && onmap > 1 ) return 6;
    }

    //hmm...another case?
    return -1;
}

#define copyAttrib(Name, TypeT, Type) \
    { \
    Geo ## TypeT ## Ptr prop1 = Geo ## TypeT ## Ptr::dcast(get ## Type ()); \
    Geo ## TypeT ## Ptr prop2 = Geo ## TypeT ## Ptr::dcast(other->get ## Type ()); \
    if(prop1 != NullFC && prop2 != NullFC) \
    { \
        beginEditCP(prop1, osg::FieldBits::AllFields); \
        Name##Base = prop1->getSize(); \
        prop1->resize(Name##Base + prop2->getSize()); \
        for(i = 0; i < prop2->getSize(); ++i) \
            prop1->editField()[Name##Base + i] = prop2->getField()[i]; \
        endEditCP(prop1, osg::FieldBits::AllFields); \
    } \
    }

#ifndef OSG_NO_INT8_PNT

    #define copyAllAttrib \
    { \
        copyAttrib(pos, Positions2f, Positions) \
        copyAttrib(pos, Positions3f, Positions) \
        copyAttrib(pos, Positions4f, Positions) \
        copyAttrib(pos, Positions2d, Positions) \
        copyAttrib(pos, Positions3d, Positions) \
        copyAttrib(pos, Positions4d, Positions) \
        copyAttrib(pos, Positions2s, Positions) \
        copyAttrib(pos, Positions3s, Positions) \
        copyAttrib(pos, Positions4s, Positions) \
        copyAttrib(type, PTypesUI8, Types) \
        copyAttrib(length, PLengthsUI8, Lengths) \
        copyAttrib(length, PLengthsUI16, Lengths) \
        copyAttrib(length, PLengthsUI32, Lengths) \
        copyAttrib(normal, Normals3f, Normals) \
        copyAttrib(normal, Normals3s, Normals) \
        copyAttrib(normal, Normals3b, Normals) \
        copyAttrib(color, Colors3f, Colors) \
        copyAttrib(color, Colors4f, Colors) \
        copyAttrib(color, Colors3ub, Colors) \
        copyAttrib(color, Colors4ub, Colors) \
        copyAttrib(seccolor, Colors3f, SecondaryColors) \
        copyAttrib(seccolor, Colors4f, SecondaryColors) \
        copyAttrib(seccolor, Colors3ub, SecondaryColors) \
        copyAttrib(seccolor, Colors4ub, SecondaryColors) \
        copyAttrib(texcoord, TexCoords1f, TexCoords) \
        copyAttrib(texcoord, TexCoords2f, TexCoords) \
        copyAttrib(texcoord, TexCoords3f, TexCoords) \
        copyAttrib(texcoord, TexCoords4f, TexCoords) \
        copyAttrib(texcoord1, TexCoords1f, TexCoords1) \
        copyAttrib(texcoord1, TexCoords2f, TexCoords1) \
        copyAttrib(texcoord1, TexCoords3f, TexCoords1) \
        copyAttrib(texcoord1, TexCoords4f, TexCoords1) \
        copyAttrib(texcoord2, TexCoords1f, TexCoords2) \
        copyAttrib(texcoord2, TexCoords2f, TexCoords2) \
        copyAttrib(texcoord2, TexCoords3f, TexCoords2) \
        copyAttrib(texcoord2, TexCoords4f, TexCoords2) \
        copyAttrib(texcoord3, TexCoords1f, TexCoords3) \
        copyAttrib(texcoord3, TexCoords2f, TexCoords3) \
        copyAttrib(texcoord3, TexCoords3f, TexCoords3) \
        copyAttrib(texcoord3, TexCoords4f, TexCoords3) \
        copyAttrib(texcoord4, TexCoords1f, TexCoords4) \
        copyAttrib(texcoord4, TexCoords2f, TexCoords4) \
        copyAttrib(texcoord4, TexCoords3f, TexCoords4) \
        copyAttrib(texcoord4, TexCoords4f, TexCoords4) \
        copyAttrib(texcoord5, TexCoords1f, TexCoords5) \
        copyAttrib(texcoord5, TexCoords2f, TexCoords5) \
        copyAttrib(texcoord5, TexCoords3f, TexCoords5) \
        copyAttrib(texcoord5, TexCoords4f, TexCoords5) \
        copyAttrib(texcoord6, TexCoords1f, TexCoords6) \
        copyAttrib(texcoord6, TexCoords2f, TexCoords6) \
        copyAttrib(texcoord6, TexCoords3f, TexCoords6) \
        copyAttrib(texcoord6, TexCoords4f, TexCoords6) \
        copyAttrib(texcoord7, TexCoords1f, TexCoords7) \
        copyAttrib(texcoord7, TexCoords2f, TexCoords7) \
        copyAttrib(texcoord7, TexCoords3f, TexCoords7) \
        copyAttrib(texcoord7, TexCoords4f, TexCoords7) \
        GeometryPtr copyAllAttribTempPtr(*this); \
        beginEditCP(copyAllAttribTempPtr, osg::FieldBits::AllFields); \
        endEditCP(copyAllAttribTempPtr, osg::FieldBits::AllFields); \
    }

#else

    #define copyAllAttrib \
    { \
        copyAttrib(pos, Positions2f, Positions) \
        copyAttrib(pos, Positions3f, Positions) \
        copyAttrib(pos, Positions4f, Positions) \
        copyAttrib(pos, Positions2d, Positions) \
        copyAttrib(pos, Positions3d, Positions) \
        copyAttrib(pos, Positions4d, Positions) \
        copyAttrib(pos, Positions2s, Positions) \
        copyAttrib(pos, Positions3s, Positions) \
        copyAttrib(pos, Positions4s, Positions) \
        copyAttrib(type, PTypesUI8, Types) \
        copyAttrib(length, PLengthsUI8, Lengths) \
        copyAttrib(length, PLengthsUI16, Lengths) \
        copyAttrib(length, PLengthsUI32, Lengths) \
        copyAttrib(normal, Normals3f, Normals) \
        copyAttrib(normal, Normals3s, Normals) \
        copyAttrib(color, Colors3f, Colors) \
        copyAttrib(color, Colors4f, Colors) \
        copyAttrib(color, Colors3ub, Colors) \
        copyAttrib(color, Colors4ub, Colors) \
        copyAttrib(seccolor, Colors3f, SecondaryColors) \
        copyAttrib(seccolor, Colors4f, SecondaryColors) \
        copyAttrib(seccolor, Colors3ub, SecondaryColors) \
        copyAttrib(seccolor, Colors4ub, SecondaryColors) \
        copyAttrib(texcoord, TexCoords1f, TexCoords) \
        copyAttrib(texcoord, TexCoords2f, TexCoords) \
        copyAttrib(texcoord, TexCoords3f, TexCoords) \
        copyAttrib(texcoord, TexCoords4f, TexCoords) \
        copyAttrib(texcoord1, TexCoords1f, TexCoords1) \
        copyAttrib(texcoord1, TexCoords2f, TexCoords1) \
        copyAttrib(texcoord1, TexCoords3f, TexCoords1) \
        copyAttrib(texcoord1, TexCoords4f, TexCoords1) \
        copyAttrib(texcoord2, TexCoords1f, TexCoords2) \
        copyAttrib(texcoord2, TexCoords2f, TexCoords2) \
        copyAttrib(texcoord2, TexCoords3f, TexCoords2) \
        copyAttrib(texcoord2, TexCoords4f, TexCoords2) \
        copyAttrib(texcoord3, TexCoords1f, TexCoords3) \
        copyAttrib(texcoord3, TexCoords2f, TexCoords3) \
        copyAttrib(texcoord3, TexCoords3f, TexCoords3) \
        copyAttrib(texcoord3, TexCoords4f, TexCoords3) \
        copyAttrib(texcoord4, TexCoords1f, TexCoords4) \
        copyAttrib(texcoord4, TexCoords2f, TexCoords4) \
        copyAttrib(texcoord4, TexCoords3f, TexCoords4) \
        copyAttrib(texcoord4, TexCoords4f, TexCoords4) \
        copyAttrib(texcoord5, TexCoords1f, TexCoords5) \
        copyAttrib(texcoord5, TexCoords2f, TexCoords5) \
        copyAttrib(texcoord5, TexCoords3f, TexCoords5) \
        copyAttrib(texcoord5, TexCoords4f, TexCoords5) \
        copyAttrib(texcoord6, TexCoords1f, TexCoords6) \
        copyAttrib(texcoord6, TexCoords2f, TexCoords6) \
        copyAttrib(texcoord6, TexCoords3f, TexCoords6) \
        copyAttrib(texcoord6, TexCoords4f, TexCoords6) \
        copyAttrib(texcoord7, TexCoords1f, TexCoords7) \
        copyAttrib(texcoord7, TexCoords2f, TexCoords7) \
        copyAttrib(texcoord7, TexCoords3f, TexCoords7) \
        copyAttrib(texcoord7, TexCoords4f, TexCoords7) \
        GeometryPtr copyAllAttribTempPtr(*this); \
        beginEditCP(copyAllAttribTempPtr, osg::FieldBits::AllFields); \
        endEditCP(copyAllAttribTempPtr, osg::FieldBits::AllFields); \
    }
#endif

//merge two identical geometries
void Geometry::merge0( const GeometryPtr other )
{
    UInt32 posBase = 0,typeBase = 0,lengthBase = 0,normalBase = 0,colorBase = 0,
           seccolorBase = 0,texcoordBase = 0,texcoord1Base = 0,texcoord2Base = 0,
           texcoord3Base = 0, texcoord4Base = 0, texcoord5Base = 0, texcoord6Base = 0,
           texcoord7Base = 0;

    // append the data
    UInt32 i;

    copyAllAttrib;

    if ( getIndices() != NullFC )
    {
        // indices

        GeoIndicesPtr ind  =        getIndices();
        GeoIndicesPtr oind = other->getIndices();

        beginEditCP( ind , osg::FieldBits::AllFields);

        UInt32 indBase = ind->getSize();
        ind->resize( indBase + oind->getSize() );

        // single index?
        if ( getMFIndexMapping()->size() < 2 )
        {
            for ( i = 0; i < oind->getSize(); i++ )
                ind->setValue( oind->getValue(i) + posBase, indBase + i );
        }
        else // multi-index
        {
            UInt32 * offsets = new UInt32 [ getMFIndexMapping()->size() ];
            Int16 mind;
            UInt16 nmap = getMFIndexMapping()->size();
            UInt16 j;

            if ( ( mind = calcMappingIndex( MapPosition ) ) >= 0 )
                offsets[ mind ] = posBase;

            if ( ( mind = calcMappingIndex( MapNormal ) ) >= 0 )
                offsets[ mind ] = normalBase;

            if ( ( mind = calcMappingIndex( MapColor ) ) >= 0 )
                offsets[ mind ] = colorBase;

            if ( ( mind = calcMappingIndex( MapSecondaryColor ) ) >= 0 )
                offsets[ mind ] = seccolorBase;

            if ( ( mind = calcMappingIndex( MapTexCoords ) ) >= 0 )
                offsets[ mind ] = texcoordBase;

            if ( ( mind = calcMappingIndex( MapTexCoords1 ) ) >= 0 )
                offsets[ mind ] = texcoord1Base;

            if ( ( mind = calcMappingIndex( MapTexCoords2 ) ) >= 0 )
                offsets[ mind ] = texcoord2Base;

            if ( ( mind = calcMappingIndex( MapTexCoords3 ) ) >= 0 )
                offsets[ mind ] = texcoord3Base;

            if ( ( mind = calcMappingIndex( MapTexCoords4 ) ) >= 0 )
                offsets[ mind ] = texcoord4Base;

            if ( ( mind = calcMappingIndex( MapTexCoords5 ) ) >= 0 )
                offsets[ mind ] = texcoord5Base;

            if ( ( mind = calcMappingIndex( MapTexCoords6 ) ) >= 0 )
                offsets[ mind ] = texcoord6Base;

            if ( ( mind = calcMappingIndex( MapTexCoords7 ) ) >= 0 )
                offsets[ mind ] = texcoord7Base;

            // bump every index by its offset
            for ( i = 0, j = 0; i < oind->getSize();
                i++, j = ( j + 1 ) % nmap )
            {
                ind->setValue(oind->getValue(i) + offsets[j],
                    indBase + i );
            }

            delete [] offsets;
        }

        endEditCP( ind , osg::FieldBits::AllFields);
    }
}


//merges non-indexed geometry in single-indexed one
void Geometry::merge1( const GeometryPtr other )
{
    UInt32 posBase = 0, typeBase = 0, lengthBase = 0, normalBase = 0,
        colorBase = 0, seccolorBase = 0, texcoordBase = 0, texcoord1Base = 0,
        texcoord2Base = 0, texcoord3Base = 0, texcoord4Base = 0,
        texcoord5Base = 0, texcoord6Base = 0, texcoord7Base = 0;

    UInt32 i;

    copyAllAttrib;

    GeoIndicesPtr ind = getIndices();

    UInt32 indBase = ind->getSize();
    ind->resize(indBase + other->getPositions()->getSize());

    beginEditCP(ind, osg::FieldBits::AllFields);

    for (i = 0; i < other->getPositions()->getSize(); i++ )
        ind->setValue( i + posBase, indBase + i );

    endEditCP( ind , osg::FieldBits::AllFields);
}

//merges single-indexed geometry in non-indexed one
//the non-indexed geometry is converted to single-indexed
void Geometry::merge2( const GeometryPtr other )
{
    UInt32 posBase = 0, typeBase = 0, lengthBase = 0, normalBase = 0,
        colorBase = 0, seccolorBase = 0, texcoordBase = 0, texcoord1Base = 0,
        texcoord2Base = 0, texcoord3Base = 0, texcoord4Base = 0,
        texcoord5Base = 0, texcoord6Base = 0, texcoord7Base = 0;

    UInt32 i;

    copyAllAttrib;

    GeoIndicesUI32Ptr indices = GeoIndicesUI32::create();
    beginEditCP(indices, GeoIndicesUI32::GeoPropDataFieldMask);

    indices->resize(posBase + other->getIndices()->getSize());

    for (i = 0; i < posBase; i++)
        indices->setValue(i, i);

    for (i = 0; i < other->getIndices()->getSize(); i++)
        indices->setValue(posBase + other->getIndices()->getValue(i), posBase + i);

    endEditCP(indices, GeoIndicesUI32::GeoPropDataFieldMask);

    beginEditCP(static_cast<GeometryPtr>(this), Geometry::IndicesFieldMask);

    setIndices(indices);

    endEditCP(static_cast<GeometryPtr>(this), Geometry::IndicesFieldMask);
}

//merges non-indexed geometry in multi-indexed one
void Geometry::merge3( const GeometryPtr other )
{
    UInt32 posBase = 0,typeBase = 0,lengthBase = 0,normalBase = 0,
        colorBase = 0, seccolorBase = 0, texcoordBase = 0, texcoord1Base = 0,
        texcoord2Base = 0, texcoord3Base = 0, texcoord4Base = 0,
        texcoord5Base = 0, texcoord6Base = 0, texcoord7Base = 0;

    UInt32 i;

    copyAllAttrib;

    // indices
    GeoIndicesPtr ind = getIndices();
    UInt16 nmap = getMFIndexMapping()->size();
    Int16 mind;
    UInt32 indBase = ind->getSize();
    ind->resize( indBase + other->getPositions()->getSize()*nmap );

    beginEditCP( ind , osg::FieldBits::AllFields);

    for (i = 0; i < other->getPositions()->getSize(); i++)
    {
        if ( ( mind = calcMappingIndex( MapPosition ) ) >= 0 )
            ind->setValue(posBase + i, indBase + i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapNormal ) ) >= 0 )
            ind->setValue(normalBase + i, indBase + i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapColor ) ) >= 0 )
            ind->setValue(colorBase + i, indBase + i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapSecondaryColor ) ) >= 0 )
            ind->setValue(seccolorBase + i, indBase + i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords ) ) >= 0 )
            ind->setValue(texcoordBase + i, indBase + i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords1 ) ) >= 0 )
            ind->setValue(texcoord1Base + i, indBase + i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords2 ) ) >= 0 )
            ind->setValue(texcoord2Base + i, indBase + i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords3 ) ) >= 0 )
            ind->setValue(texcoord3Base + i, indBase + i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords4 ) ) >= 0 )
            ind->setValue(texcoord4Base + i, indBase + i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords5 ) ) >= 0 )
            ind->setValue(texcoord5Base + i, indBase + i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords6 ) ) >= 0 )
            ind->setValue(texcoord6Base + i, indBase + i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords7 ) ) >= 0 )
            ind->setValue(texcoord7Base + i, indBase + i*nmap + mind);
    }

    endEditCP( ind , osg::FieldBits::AllFields);
}

//merges multi-indexed geometry in non-indexed one
//the non-indexed geometry is converted to multi-indexed
void Geometry::merge4( const GeometryPtr other )
{
    UInt32 posBase = 0,typeBase = 0,lengthBase = 0,normalBase = 0,colorBase = 0,
           seccolorBase = 0,texcoordBase = 0,texcoord1Base = 0,texcoord2Base = 0,
           texcoord3Base = 0, texcoord4Base = 0, texcoord5Base = 0, texcoord6Base = 0,
           texcoord7Base = 0;

    UInt32 i;

    copyAllAttrib;

    GeoIndicesPtr oind = other->getIndices();
    UInt16 nmap = other->getMFIndexMapping()->size();
    Int16 mind;
    GeoIndicesUI32Ptr indices = GeoIndicesUI32::create();
    beginEditCP(indices, GeoIndicesUI32::GeoPropDataFieldMask);

    indices->resize( posBase*nmap + oind->getSize() );

    for (i = 0; i < posBase; i++)
    {
        if ( ( mind = calcMappingIndex( MapPosition ) ) >= 0 )
            indices->setValue(i, i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapNormal ) ) >= 0 )
            indices->setValue(i, i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapColor ) ) >= 0 )
            indices->setValue(i, i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapSecondaryColor ) ) >= 0 )
            indices->setValue(i, i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords ) ) >= 0 )
            indices->setValue(i, i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords1 ) ) >= 0 )
            indices->setValue(i, i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords2 ) ) >= 0 )
            indices->setValue(i, i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords3 ) ) >= 0 )
            indices->setValue(i, i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords4 ) ) >= 0 )
            indices->setValue(i, i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords5 ) ) >= 0 )
            indices->setValue(i, i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords6 ) ) >= 0 )
            indices->setValue(i, i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords7 ) ) >= 0 )
            indices->setValue(i, i*nmap + mind);
    }

    for (i = 0; i < other->getIndices()->getSize(); i++)
        indices->setValue(posBase*nmap + other->getIndices()->getValue(i), posBase*nmap + i);

    endEditCP(indices, GeoIndicesUI32::GeoPropDataFieldMask);

    beginEditCP(static_cast<GeometryPtr>(this), Geometry::IndicesFieldMask);

    setIndices(indices);

    endEditCP(static_cast<GeometryPtr>(this), Geometry::IndicesFieldMask);
}

//merges single-indexed geometry in multi-indexed one
void Geometry::merge5( const GeometryPtr other )
{
    UInt32 posBase = 0, typeBase = 0, lengthBase = 0, normalBase = 0,
        colorBase = 0, seccolorBase = 0, texcoordBase = 0, texcoord1Base = 0,
        texcoord2Base = 0, texcoord3Base = 0, texcoord4Base = 0,
        texcoord5Base = 0, texcoord6Base = 0, texcoord7Base = 0;

    UInt32 i;

    copyAllAttrib;

    // indices
    GeoIndicesPtr ind = getIndices();
    GeoIndicesPtr oind = other->getIndices();
    UInt16 nmap = getMFIndexMapping()->size();
    Int16 mind;
    UInt32 indBase = ind->getSize();
    ind->resize( indBase + oind->getSize()*nmap );

    beginEditCP( ind , osg::FieldBits::AllFields);

    for (i = 0; i < oind->getSize(); i++)
    {
        if ( ( mind = calcMappingIndex( MapPosition ) ) >= 0 )
            ind->setValue(posBase + oind->getValue(i), indBase + i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapNormal ) ) >= 0 )
            ind->setValue(normalBase + oind->getValue(i), indBase + i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapColor ) ) >= 0 )
            ind->setValue(colorBase + oind->getValue(i), indBase + i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapSecondaryColor ) ) >= 0 )
            ind->setValue(seccolorBase + oind->getValue(i), indBase + i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords ) ) >= 0 )
            ind->setValue(texcoordBase + oind->getValue(i), indBase + i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords1 ) ) >= 0 )
            ind->setValue(texcoord1Base + oind->getValue(i), indBase + i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords2 ) ) >= 0 )
            ind->setValue(texcoord2Base + oind->getValue(i), indBase + i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords3 ) ) >= 0 )
            ind->setValue(texcoord3Base + oind->getValue(i), indBase + i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords4 ) ) >= 0 )
            ind->setValue(texcoord4Base + oind->getValue(i), indBase + i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords5 ) ) >= 0 )
            ind->setValue(texcoord5Base + oind->getValue(i), indBase + i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords6 ) ) >= 0 )
            ind->setValue(texcoord6Base + oind->getValue(i), indBase + i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords7 ) ) >= 0 )
            ind->setValue(texcoord7Base + oind->getValue(i), indBase + i*nmap + mind);
    }

    endEditCP( ind , osg::FieldBits::AllFields);
}

//merges multi-indexed geometry in single-indexed one
//the single-indexed geometry is converted to multi-indexed
void Geometry::merge6( const GeometryPtr other )
{
    UInt32 posBase = 0, typeBase = 0, lengthBase = 0, normalBase = 0,
        colorBase = 0, seccolorBase = 0, texcoordBase = 0, texcoord1Base = 0,
        texcoord2Base = 0, texcoord3Base = 0, texcoord4Base = 0,
        texcoord5Base = 0, texcoord6Base = 0, texcoord7Base = 0;

    UInt32 i;

    copyAllAttrib;

    // indices
    GeoIndicesPtr ind = getIndices();
    GeoIndicesPtr indclone = getIndices()->clone();
    GeoIndicesPtr oind = other->getIndices();
    UInt16 nmap = other->getMFIndexMapping()->size();
    Int16 mind;
    UInt32 indBase = ind->getSize();
    ind->resize( indBase*nmap + oind->getSize() );

    beginEditCP( ind , osg::FieldBits::AllFields);

    for (i = 0; i < indclone->getSize(); i++)
    {
        if ( ( mind = calcMappingIndex( MapPosition ) ) >= 0 )
            ind->setValue(indclone->getValue(i), i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapNormal ) ) >= 0 )
            ind->setValue(indclone->getValue(i), i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapColor ) ) >= 0 )
            ind->setValue(indclone->getValue(i), i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapSecondaryColor ) ) >= 0 )
            ind->setValue(indclone->getValue(i), i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords ) ) >= 0 )
            ind->setValue(indclone->getValue(i), i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords1 ) ) >= 0 )
            ind->setValue(indclone->getValue(i), i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords2 ) ) >= 0 )
            ind->setValue(indclone->getValue(i), i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords3 ) ) >= 0 )
            ind->setValue(indclone->getValue(i), i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords4 ) ) >= 0 )
            ind->setValue(indclone->getValue(i), i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords5 ) ) >= 0 )
            ind->setValue(indclone->getValue(i), i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords6 ) ) >= 0 )
            ind->setValue(indclone->getValue(i), i*nmap + mind);

        if ( ( mind = calcMappingIndex( MapTexCoords7 ) ) >= 0 )
            ind->setValue(indclone->getValue(i), i*nmap + mind);
    }

    for (i = 0; i < oind->getSize(); i++)
        ind->setValue(oind->getValue(i), indBase*nmap + i);

    endEditCP( ind , osg::FieldBits::AllFields);
}

bool Geometry::isPlanar() const
{
    // simple check if all normals are facing into the same direction
    GeoNormalsPtr normals = getNormals();
    if (NullFC == normals)
        return false;

    if (normals->getSize() < 2)
        return true;

    const Real32 eps = 0.00001f;
    Vec3f n0 = normals->getValue(0);

    for (UInt32 i = 1; i < normals->getSize(); i++)
    {
        Vec3f n = normals->getValue(i);
        Real32 dp = n.dot(n0);
        // if both normals point into the same direction,
        // their dot product should be equal to 1.
        // because of possible floating point precision
        // problem, we add an epsilon to the comparison.
        if (fabs(dp - 1.0f) > eps)
            return false;
    }

    return true;
}


GeoTmpIndexEditInterface::GeoTmpIndexEditInterface(GeometryPtr const &geo, UInt32 label)
    : m_geometry(geo)
    , m_mfIdx(0)
    , m_pendingEdit(false)
{
    if (NullFC != m_geometry)
    {
        bool callTouch = false;
        // first try to initialize the first entry which is the info vector if it does not exist.
        if (getMFIndicesRef().getSize() == 0)
        {
            GeoIndicesPtr i = GeoIndicesUI32::create();
            beginEditCP(i, osg::FieldBits::AllFields);
            addRefCP(i);
            i->addValue(0); // source of stamp
            endEditCP(i, osg::FieldBits::AllFields);
            beginEditCP(m_geometry, Geometry::TempIndicesFieldMask);
            getMFIndicesRef().push_back(i);
            endEditCP(m_geometry, Geometry::TempIndicesFieldMask);
            callTouch = true;
        }
        auto &infoVec = getMFIndicesRef()[0];
        // try to find the entry for this label
        UInt32 entryIdx = 1;
        for (; entryIdx < infoVec->getSize(); entryIdx += 3)
        {
            if (infoVec->getValue(entryIdx) == label)
                break;
        }
        if (entryIdx >= infoVec->getSize())
        {
            entryIdx = infoVec->getSize();
            beginEditCP(infoVec, osg::FieldBits::AllFields);
            infoVec->addValue(label); // first comes label
            infoVec->addValue(GL_TRIANGLES); // second comes type whose default is GL_TRIANGLES
            infoVec->addValue(0); // this is the stamp value, wait to call touch() to set the value to it.
            endEditCP(infoVec, osg::FieldBits::AllFields);
            beginEditCP(m_geometry, Geometry::TempIndicesFieldMask);
            getMFIndicesRef().push_back(NullFC);
            endEditCP(m_geometry, Geometry::TempIndicesFieldMask);
            callTouch = true;
        }
        m_mfIdx = (entryIdx - 1) / 3 + 1;
        if (callTouch) 
            editAndTouch();
    }
}

GeoTmpIndexEditInterface::~GeoTmpIndexEditInterface()
{
    endPendingEdit();
}

UInt32 GeoTmpIndexEditInterface::getSize() const
{
    const auto indices = getIndicesRef();
    if ( indices == osg::NullFC )
        return 0;

    return indices->getSize();
}

UInt32* GeoTmpIndexEditInterface::getData() const
{
    const auto indices = getIndicesRef();
    if (indices == osg::NullFC)
        return nullptr;

    return static_cast<UInt32*>(static_cast<void*>(indices->getData()));
}

UInt32 GeoTmpIndexEditInterface::getValue(const UInt32 index) const
{
    const auto indices = getIndicesRef();
    if (indices == osg::NullFC)
        return 0;

    return indices->getValue(index);
}

void GeoTmpIndexEditInterface::setValue(const UInt32 val, const UInt32 index)
{
    editAndTouch();

    const auto indices = getIndicesRef();
    if (indices == osg::NullFC)
        return;

    indices->setValue(val, index);
}

void GeoTmpIndexEditInterface::addValue(const UInt32 val)
{
    editAndTouch();

    const auto indices = getIndicesRef();
    if (indices == osg::NullFC)
        return;

    indices->addValue(val);
}

void osg::GeoTmpIndexEditInterface::set(std::vector<UInt32> const& data, const UInt32 type)
{
    if (data.empty())
    {
        setNull();
    }
    else
    {
        if (isNull())
            make();
        if (getType() != type)
            setType(type);
        clear();
        for (UInt32 v : data)
            addValue(v);
    }
    commit();
}

// bool GeoTmpIndexEditInterface::insertValue(const UInt32 val, const UInt32 index)
// {
//     editAndTouch();
//     return getIndicesRef()->insertValue(val, index);
// }

void GeoTmpIndexEditInterface::clear()
{
    editAndTouch();

    const auto indices = getIndicesRef();
    if (indices == osg::NullFC)
        return;

    indices->clear();
}

 void GeoTmpIndexEditInterface::resize(size_t newsize)
 {
     editAndTouch();

     const auto indices = getIndicesRef();
     if (indices == osg::NullFC)
         return;

     indices->resize(newsize);
 }

void GeoTmpIndexEditInterface::push_back(const UInt32 val)
{
    editAndTouch();

    const auto indices = getIndicesRef();
    if (indices == osg::NullFC)
        return;

    indices->push_back(val);
}

// void GeoTmpIndexEditInterface::shrink()
// {
//     editAndTouch();
//     getIndicesRef()->shrink();
// }

GeoIndicesPtr GeoTmpIndexEditInterface::copyData() const
{
    const auto indices = getIndicesRef();
    if (indices == osg::NullFC)
        return osg::NullFC;

    return indices->clone();
}

bool GeoTmpIndexEditInterface::isNull() const
{
    return NullFC == m_geometry || 0 == m_mfIdx || NullFC == getIndicesRef();
}

void GeoTmpIndexEditInterface::setNull()
{
    if (isNull())
        return;

    endPendingEdit();

    beginEditCP(m_geometry, Geometry::TempIndicesFieldMask);
    beginEditCP(getMFIndicesRef()[0], osg::FieldBits::AllFields);
    auto &indicesRef = getIndicesRef();
    subRefCP(indicesRef);
    indicesRef = NullFC;
    touch();
    endEditCP(getMFIndicesRef()[0], osg::FieldBits::AllFields);
    endEditCP(m_geometry, Geometry::TempIndicesFieldMask);
}

void GeoTmpIndexEditInterface::make()
{
    if (NullFC == m_geometry || 0 == m_mfIdx || NullFC != getIndicesRef())
        return;

    endPendingEdit();

    beginEditCP(m_geometry, Geometry::TempIndicesFieldMask);
    beginEditCP(getMFIndicesRef()[0], osg::FieldBits::AllFields);
    GeoIndicesPtr i = GeoIndicesUI32::create();
    addRefCP(i);
    getIndicesRef() = i;
    touch();
    endEditCP(getMFIndicesRef()[0], osg::FieldBits::AllFields);
    endEditCP(m_geometry, Geometry::TempIndicesFieldMask);
}

UInt32 GeoTmpIndexEditInterface::getLabel() const
{
    return getMFIndicesRef()[0]->getValue((m_mfIdx - 1) * 3 + 1);
}

void GeoTmpIndexEditInterface::setType(const UInt32 type)
{
    editAndTouch();
    getMFIndicesRef()[0]->setValue(type, (m_mfIdx - 1) * 3 + 2); //*(m_infoPtr + 1) = type;
}

UInt32 GeoTmpIndexEditInterface::getType() const
{
    return getMFIndicesRef()[0]->getValue((m_mfIdx - 1) * 3 + 2); //*(m_infoPtr + 1);
}

UInt32 GeoTmpIndexEditInterface::getStamp() const
{
    return getMFIndicesRef()[0]->getValue((m_mfIdx - 1) * 3 + 3);//*(m_infoPtr + 2);
}

void GeoTmpIndexEditInterface::commit()
{
    endPendingEdit();
}

void GeoTmpIndexEditInterface::setNullforAll(GeometryPtr &geo)
{
    if (NullFC == geo)
        return;
    auto &indices = geo->getTempIndices();
    if (indices.getSize() <= 1)
        return;
    auto &infoVec = indices[0];
    if (infoVec->getSize() <= 1)
        return;
    for (UInt32 i = 1; i < infoVec->getSize(); i += 3)
    {
        GeoTmpIndexEditInterface edtObj(geo, infoVec->getValue(i));
        edtObj.setNull();
    }
}

void GeoTmpIndexEditInterface::touch()
{
    const UInt32 stampVal = getMFIndicesRef()[0]->getValue(0) + 1;
    getMFIndicesRef()[0]->setValue(stampVal, 0);
    getMFIndicesRef()[0]->setValue(stampVal, (m_mfIdx - 1) * 3 + 3); //*(m_infoPtr + 2) = ++(*m_stSrcPtr);
}

void GeoTmpIndexEditInterface::editAndTouch()
{
    if (!m_pendingEdit) {
        beginEditCP(m_geometry, Geometry::TempIndicesFieldMask);
        beginEditCP(getMFIndicesRef()[0], osg::FieldBits::AllFields);
        beginEditCP(getIndicesRef(), osg::FieldBits::AllFields);
        m_pendingEdit = true;
    }
    touch();
}

void GeoTmpIndexEditInterface::endPendingEdit()
{
    if (m_pendingEdit)
    {
        endEditCP(getIndicesRef(), osg::FieldBits::AllFields);
        endEditCP(getMFIndicesRef()[0], osg::FieldBits::AllFields);
        endEditCP(m_geometry, Geometry::TempIndicesFieldMask);
        m_pendingEdit = false;
    }
}

GeoIndicesPtr& GeoTmpIndexEditInterface::getIndicesRef()
{
    return m_geometry->getTempIndices(m_mfIdx);
}

GeoIndicesPtr const& GeoTmpIndexEditInterface::getIndicesRef() const
{
    return m_geometry->getTempIndices(m_mfIdx);
}

MFGeoIndicesPtr& GeoTmpIndexEditInterface::getMFIndicesRef()
{
    return m_geometry->getTempIndices();
}

MFGeoIndicesPtr const& GeoTmpIndexEditInterface::getMFIndicesRef() const
{
    return m_geometry->getTempIndices();
}



//#undef copyAttrib
//#undef copyAllAttrib


/*------------------------------------------------------------------------*/
/*                              cvs id's                                  */

#ifdef OSG_SGI_CC
#pragma set woff 1174
#endif

#ifdef OSG_LINUX_ICC
#pragma warning(disable : 177)
#endif

namespace
{
    static Char8 cvsid_cpp       [] = "@(#)$Id: $";
    static Char8 cvsid_hpp       [] = OSGGEOMETRY_HEADER_CVSID;
    static Char8 cvsid_inl       [] = OSGGEOMETRY_INLINE_CVSID;
}

