/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright (C) 2000-2002 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/


#ifndef _OSGGEOMETRY_H_
#define _OSGGEOMETRY_H_
#pragma once

#include <functional>

#include <OSGConfig.h>
#include <OSGAction.h>
#include <OSGWindow.h>

#include <OSGGeometryBase.h>
#include <OSGGeoPropPtrs.h>
#include <OSGMaterial.h>
#include <OSGDynamicVolume.h>

class vrVertexBufferObject;

OSG_BEGIN_NAMESPACE

class DrawActionBase;

class TriangleIterator;
class PrimitiveIterator;
class FaceIterator;
class LineIterator;
class EdgeIterator;

typedef std::function<void(Geometry* geo, DynamicVolume& volume)> GetVolumeCacheFunction;
typedef std::function<void(Geometry* geo)> InvalidateVolumeCacheFunction;

/*! The Geometry class, see \ref PageSystemGeometry for a description.
*/
class OSG_SYSTEMLIB_DLLMAPPING Geometry : public GeometryBase
{
    /*==========================  PUBLIC  =================================*/
  public:
      enum {    GEOMETRY_TYPE_DEFAULT = 0,
                GEOMETRY_TYPE_SHELL = 1 << 1,
                GEOMETRY_IS_B_SIDE = 1 << 2
      };

      enum {    GEOMETRY_TEXTURE_SLOT0 = 0,
                GEOMETRY_TEXTURE_SLOT1,
                GEOMETRY_TEXTURE_SLOT2,
                GEOMETRY_TEXTURE_SLOT3,
                GEOMETRY_TEXTURE_SLOT4,
                GEOMETRY_TEXTURE_SLOT5,
                GEOMETRY_TEXTURE_SLOT6,
                GEOMETRY_TEXTURE_SLOT7,
                GEOMETRY_LAST_TEXTURE_SLOT = GEOMETRY_TEXTURE_SLOT7
      };
    /*---------------------------------------------------------------------*/
    /*! \name                IndexMapping Constants                        */
    /*! \{                                                                 */

    static const UInt16 MapPosition;
    static const UInt16 MapNormal;
    static const UInt16 MapColor;
    static const UInt16 MapSecondaryColor;
    static const UInt16 MapTexCoords;
    static const UInt16 MapTexCoords1;
    static const UInt16 MapTexCoords2;
    static const UInt16 MapTexCoords3;
    static const UInt16 MapTexCoords4;
    static const UInt16 MapTexCoords5;
    static const UInt16 MapTexCoords6;
    static const UInt16 MapTexCoords7;
    static const UInt16 MapEmpty;

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                    Class Get                                 */
    /*! \{                                                                 */

    static const char *getClassname(void) { return "Geometry"; };

    DynamicVolume getGeometryVolume();

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                       Class Specific                         */
    /*! \{                                                                 */

    static const char *mapType          (UInt8 type);

    Int16              calcMappingIndex (UInt16 attrib) const;

    virtual void       changed          (BitVector whichField,
                                         UInt32    origin    );

    GeometryPtr        getPtr           (void) const;

    GeometryPtr        clone            (void);
    void               cloneTo          (GeometryPtr target);

    virtual void       dump             (      UInt32    uiIndent = 0,
                                         const BitVector bvFlags = 0) const;

    bool               isMergeable      (const GeometryPtr other);

    bool               merge            (const GeometryPtr other);

    void               requestShellUpdate();

    void                makeSingleIndexed();

    void                    setVertexBufferObject(vrVertexBufferObject* vbo);
    vrVertexBufferObject*   getVertexBufferObject() const;

    void setVolumeCacheFunctions(GetVolumeCacheFunction getVolumeFunction, 
                                 InvalidateVolumeCacheFunction invalidateVolumeFunction);
    void invalidateVolumeCache();

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                    Abstract                                  */
    /*! \{                                                                 */

#ifndef OSG_SUPPORT_NO_GEO_INTERFACE
    virtual GeoPropertyArrayInterface *getProperty(Int32 mapID);
#endif


    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                    Field Set                                 */
    /*! \{                                                                 */

    void setTypes          (const GeoPTypesPtr    &value);
    void setLengths        (const GeoPLengthsPtr  &value);
    void setPositions      (const GeoPositionsPtr &value);
    void setNormals        (const GeoNormalsPtr   &value);
    void setColors         (const GeoColorsPtr    &value);
    void setSecondaryColors(const GeoColorsPtr    &value);

    void setTexSlot        (const GeoTexCoordsPtr &value, osg::UInt32 slot);
    GeoTexCoordsPtr getTexSlot(osg::UInt32 slot);

    static UInt16 getIndexMappingConstant(osg::UInt32 slot);
    static UInt16 getTexSlotFieldId(osg::UInt32 slot);
    static BitVector getTexSlotFieldMask(osg::UInt32 slot);

    void setTexCoords      (const GeoTexCoordsPtr &value);
    void setTexCoords1     (const GeoTexCoordsPtr &value);
    void setTexCoords2     (const GeoTexCoordsPtr &value);
    void setTexCoords3     (const GeoTexCoordsPtr &value);
    void setTexCoords4     (const GeoTexCoordsPtr &value);
    void setTexCoords5     (const GeoTexCoordsPtr &value);
    void setTexCoords6     (const GeoTexCoordsPtr &value);
    void setTexCoords7     (const GeoTexCoordsPtr &value);
    void setIndices        (const GeoIndicesPtr   &value);
    void setMaterial       (const MaterialPtr     &value);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                       Iterator Access                        */
    /*! \{                                                                 */

    TriangleIterator   beginTriangles   (void) const;
    TriangleIterator   endTriangles     (void) const;

    PrimitiveIterator  beginPrimitives  (void) const;
    PrimitiveIterator  endPrimitives    (void) const;

    FaceIterator       beginFaces       (void) const;
    FaceIterator       endFaces         (void) const;

    LineIterator       beginLines       (void) const;
    LineIterator       endLines         (void) const;

    EdgeIterator       beginEdges       (void) const;
    EdgeIterator       endEdges         (void) const;

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                       Draw                                   */
    /*! \{                                                                 */

    // should these be public?
    virtual Action::ResultE intersect      (Action * action );


    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                      NodeCore Specific                       */
    /*! \{                                                                 */

    // this only calculates a bbox - therefore it is useful and does no harm
    void            adjustVolume     (Volume & volume);

    bool            isShell() const;
    bool            isBSide() const;
    bool            isPlanar() const;
    void            setToBSide(bool bSide);

    /*! \}                                                                 */
    /*=========================  PROTECTED  ===============================*/
  protected:

    /*---------------------------------------------------------------------*/
    /*! \name                   Constructors                               */
    /*! \{                                                                 */

    Geometry(void);
    Geometry(const Geometry &source);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                   Destructors                                */
    /*! \{                                                                 */

    virtual ~Geometry(void);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                   Class Specific                             */
    /*! \{                                                                 */


    void onCreate(const Geometry *source = NULL);
    void onDestroy(void);

    bool CompareMaterials(MaterialPtr m1, MaterialPtr m2);
    Int16 MergeIndex( const GeometryPtr other );

    void merge0( const GeometryPtr other );
    void merge1( const GeometryPtr other );
    void merge2( const GeometryPtr other );
    void merge3( const GeometryPtr other );
    void merge4( const GeometryPtr other );
    void merge5( const GeometryPtr other );
    void merge6( const GeometryPtr other );

#ifndef OSG_2_PREP
    DynamicVolume   _volumeCache;
#else
    BoxVolume       _volumeCache;
#endif
    /*! \}                                                                 */
    /*==========================  PRIVATE  ================================*/
  private:
    SFGeoTexCoordsPtr & getTexSlotField(osg::UInt32 slot);

    typedef GeometryBase Inherited;

    friend class FieldContainer;
    friend class GeometryBase;

    static void initMethod( void );

    void operator =(const Geometry &source);
    void updateVolumeCache();

    vrVertexBufferObject*   m_vbo;

    // functions to get and invalidate an externally calculated volume cache for the geometry
    GetVolumeCacheFunction m_getVolume;
    InvalidateVolumeCacheFunction m_invalidateVolume;

};

/*!
 * \class  GeoTmpIndexEditInterface
 * \brief  Helper class that facilitates operations for temporary indices for each label.
 * \author Jincheng Li @ Autodesk 
 * \date   7/29/2015
 */
class OSG_SYSTEMLIB_DLLMAPPING GeoTmpIndexEditInterface
{
public:
    GeoTmpIndexEditInterface(GeometryPtr const &geo, UInt32 label);
    ~GeoTmpIndexEditInterface();

    // interfaces for accessing GeoIndicesPtr
    UInt32                  getSize() const;
    UInt32*                 getData() const;
    UInt32                  getValue(const UInt32 index) const;
    void                    setValue(const UInt32 val, const UInt32 index);
    void                    addValue(const UInt32 val);
    void                    set(std::vector<UInt32> const& data, const UInt32 type);
    //bool                    insertValue(const UInt32 val, const UInt32 index);
    void                    clear();
    void                    resize(size_t newsize);
    void                    push_back(const UInt32 val);
    //void                    shrink();

    // other interfaces 
    GeoIndicesPtr           copyData() const;
    bool                    isNull() const;
    void                    setNull();
    void                    make();
    UInt32                  getLabel() const;
    void                    setType(const UInt32 type);
    UInt32                  getType() const;
    UInt32                  getStamp() const;
    void                    commit();

    static void             setNullforAll(GeometryPtr &geo);

private:
    // non-copy constructible/assignable
    GeoTmpIndexEditInterface(GeoTmpIndexEditInterface const &);
    GeoTmpIndexEditInterface& operator=(GeoTmpIndexEditInterface const &);

    void                    touch();
    void                    editAndTouch();
    void                    endPendingEdit();
    GeoIndicesPtr&          getIndicesRef();
    GeoIndicesPtr const&    getIndicesRef() const;
    MFGeoIndicesPtr&        getMFIndicesRef();
    MFGeoIndicesPtr const&  getMFIndicesRef() const;

private:
    GeometryPtr             m_geometry;
    UInt32                  m_mfIdx;
    bool                    m_pendingEdit;
};

typedef Geometry *GeometryP;

OSG_END_NAMESPACE

#include <OSGGeometryBase.inl>
#include <OSGGeometry.inl>

#define OSGGEOMETRY_HEADER_CVSID "@(#)$Id: $"

#endif /* _OSGGEOMETRY_H_ */
