/*---------------------------------------------------------------------------*\
 *                           OpenSG NURBS Library                            *
 *                                                                           *
 *                                                                           *
 * Copyright (C) 2001-2006 by the University of Bonn, Computer Graphics Group*
 *                                                                           *
 *                         http://cg.cs.uni-bonn.de/                         *
 *                                                                           *
 * contact: edhellon@cs.uni-bonn.de, guthe@cs.uni-bonn.de, rk@cs.uni-bonn.de *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------


#include <stdlib.h>
#include <stdio.h>


#include <OSGConfig.h>

#include <OSGAction.h>
#include <OSGDrawAction.h>     
#include <OSGRenderAction.h>   
#include <OSGIntersectAction.h>
#include <OSGRenderAction.h>
#include <OSGSimpleMaterial.h>

#include <OSGGeometry.h>   
#include <OSGGeoProperty.h>
#include <OSGTransform3D.h>

#include <OSGGeoFunctions.h>


#include <OSGGL.h>
#include <OSGGLU.h>
#include <OSGGLEXT.h>

#include "OSGSurface.h"

/*
#ifdef OSG_USE_ATI
#  include <GL/glx.h>
#  include <GL/glext.h>
#  include <GL/glATI.h>
#endif
*/

#include "OSGPrimitiveIterator.h"

#include <OSGBSplineTrimmedSurface.h>
#include <OSGNurbsPatchSurface.h>
#include <OSGSimplePolygon.h>

#include <OSGpredicates.h> //for exactinit()

OSG_USING_NAMESPACE

#ifdef __sgi
#pragma set woff 1174
#endif

namespace
{
    static char cvsid_cpp[] = "@(#)$Id: OSGSurface.cpp,v 1.13 2006-11-17 17:59:30 edhellon Exp $";
    static char cvsid_hpp[] = OSGSURFACE_HEADER_CVSID;
    static char cvsid_inl[] = OSGSURFACE_INLINE_CVSID;
}

#ifdef __sgi
#pragma reset woff 1174
#endif

/***************************************************************************\
 *                               Types                                     *
\***************************************************************************/

/***************************************************************************\
 *                           Class variables                               *
\***************************************************************************/

const osg::BitVector  Surface::CurveFieldMask = 
    (Surface::NumCurvesFieldMask |
     Surface::KnotLengthsFieldMask |
     Surface::DimensionsFieldMask |
     Surface::CurveControlPointsFieldMask |
     Surface::KnotsFieldMask |
     Surface::CurvesPerLoopFieldMask );

const osg::BitVector  Surface::SurfaceFieldMask = 
    (Surface::DimUFieldMask |
     Surface::DimVFieldMask |
     Surface::KnotsUFieldMask |
     Surface::KnotsVFieldMask |
     Surface::ControlPointsFieldMask);



/*! \class osg::Surface
    \ingroup Geometries
This is the OpenSG NURBS surface node. It supports an arbitrary number of trimming curves. The interface lets you define both rational and nonrational surfaces, and you can also vary rational and nonrational trimming curves.

You can edit the surface properties directly via the usual OpenSG methods.
You are required to use beginEditCP()/endEditCP() pairs, of course.
You may use the convenience fieldmask SurfaceFieldMask.

The error field means the maximum Euclidean error the tessellation will differ
from the analytical surface in 3D space. Generally the lesser it is,
the finer the tessellation is. For most surfaces 0.1 - 0.5 are generally
good choices.
*/

/*----------------------- constructors & destructors ----------------------*/

//! Constructor

Surface::Surface(void) :
    Inherited()
{

}

//! Copy Constructor

Surface::Surface(const Surface &source) :
    Inherited(source)
{

}

//! Destructor

Surface::~Surface(void)
{
}


/*----------------------------- class specific ----------------------------*/

//! initialize the static features of the class, e.g. action callbacks

void Surface::initMethod (void)
{

    IntersectAction::registerEnterDefault(getClassType(),
                                     std::bind(reinterpret_cast<Action::Callback>(&Geometry::intersect),
                                               std::placeholders::_1, std::placeholders::_2));
    // init robust predicates
    ::exactinit();
    // turn appearance preserving tessellation on
    CErrorQuadTree::m_sbNormalApproximation = true;

}

//! react to field changes

void Surface::changed(BitVector whichField, UInt32 origin)
{
    if(whichField & ControlPointsFieldMask)
    {

    }
  
    // you should call the parent's changed() function in case
    // it needs to update inherited fields.
    Inherited::changed( whichField, origin ); 

    
//    std::cerr<<"Surface::changed:   texcoords ptr: " << getTexCoords() << std::endl;
//    std::cerr<<"Surface::changed:   texcoords1 ptr: " << getTexCoords1() << std::endl;
    
}




// protected


/* given an set of arbitrary bounding volumes
 * find the vector to the point on the volume that is closest to the view pos
*/
void Surface::FindClosestPoint( Vec3f& rDist, const Pnt3f viewPos ) const
{
    const UInt32    parent_cnt = getParents( ).size( );
    UInt32          parent_idx;
    Real64          min_dist = 1e300;
    Vec3f           temp_dist;
    Real64          curr_dist;

    for( parent_idx = 0; parent_idx < parent_cnt; ++parent_idx )
    {
                Pnt3f   bb_min, bb_max;

                getParents( )[ 0 ]->getVolume( ).getBounds( bb_min, bb_max );

//              std::cerr << bb_min << " " << bb_max << " " << viewPos << std::endl;

                if( viewPos[0] < bb_min[0] )
                {
                    temp_dist[0] = viewPos[0] - bb_min[0];
                }
                else if( viewPos[0] > bb_max[0] )
                {
                    temp_dist[0] = viewPos[0] - bb_max[0];
                }
                else
                {
                    temp_dist[0] = 0.0;
                }

                if( viewPos[1] < bb_min[1] )
                {
                    temp_dist[1] = viewPos[1] - bb_min[1];
                }
                else if( viewPos[1] > bb_max[1] )
                {
                    temp_dist[1] = viewPos[1] - bb_max[1];
                }
                else
                {
                    temp_dist[1] = 0.0;
                }

                if( viewPos[2] < bb_min[2] )
                {
                    temp_dist[2] = viewPos[2] - bb_min[2];
                }
                else if( viewPos[2] > bb_max[2] )
                {
                    temp_dist[2] = viewPos[2] - bb_max[2];
                }
                else
                {
                    temp_dist[2] = 0.0;
                }

        curr_dist = temp_dist.squareLength( );
        if( curr_dist < min_dist )
        {
            min_dist = curr_dist;
            rDist = temp_dist;
        }
    }
}


void Surface::FindClosestPointExact( Vec3f& rDist, const Pnt3f viewPos ) const
{
//  const GeoPositions3fPtr cpcl_points = GeoPositions3fPtr::dcast( cpcl_geo->getPositions( ) );
    const GeoPositions3fPtr cpcl_points = GeoPositions3fPtr::dcast( getPositions() );
    const unsigned int      cui_idx_cnt = cpcl_points->getSize( );
    Pnt3f                   cl_pos;
    unsigned int            ui_idx;
    Vec3f                   cl_temp;
    double                  d_min = DBL_MAX;
    double                  d_temp;

    for( ui_idx = 0; ui_idx < cui_idx_cnt; ++ui_idx )
    {
        cl_pos = cpcl_points->getValue( ui_idx );
        cl_temp = cl_pos - viewPos;
        d_temp = cl_temp.squareLength( );
        if( d_temp <= d_min )
        {
            d_min = d_temp;
            rDist = cl_temp;
        }
    }
}



void Surface::calcIndexMapping(void)
{
    editMFIndexMapping()->clear();

    UInt16 im = 0;
    if(getPositions() != NullFC)
        im |= Geometry::MapPosition;
    if(getNormals() != NullFC)
        im |= Geometry::MapNormal;
    if(getTexCoords() != NullFC)
        im |= Geometry::MapTexCoords;
    if(getTexCoords1() != NullFC)
        im |= Geometry::MapTexCoords1;

    if(im != 0)
        editMFIndexMapping()->push_back(im);
}



//! Add a (polynomial) trimming curve to the surface.
/*!
 *
 *  This function allows you to add a new trimming curve to
 *  the Surface.
 *
 *  \param dim the dimension of the curve
 *  \param knots the knotvector of the curve
 *  \param controlpoints the (rational) control points for the curve
 *  \param newloop false: this curve continues the current curveloop <BR>
 *                 true: starts a new loop
 *
 *  If it's the first curve added to the surface, it always starts
 *  a new loop, regardless of the value of \param newloop.
 *
 *  You are expected to call this function between corresponding
 *  beginEditCP()/endEditCP() pairs with the mask CurveFieldMask.
 *  You are responsible that when calling endEditCP() the 
 *  trimming curves are in a consistend state (they form closed
 *  loops and the like).
 *
 */     
void Surface::addCurve( UInt32 dim,
               std::vector<Real64>& knots,
               std::vector<Pnt2f>& controlpoints,
               bool newloop)
{
    UInt32 cpsize = controlpoints.size();
    std::vector<Pnt3f> ratcontrolpoints;

    ratcontrolpoints.reserve( cpsize );
    for (UInt32 i = 0; i < cpsize; ++i )
    {
        ratcontrolpoints.push_back( Pnt3f(controlpoints[i][0], 
                                          controlpoints[i][1],
                                          1.0f) );
    }    
    addCurve(dim, knots, ratcontrolpoints, newloop);
}

//! Add a (rational) trimming curve to the surface.
/*!
 *
 *  This function allows you to add a new trimming curve to
 *  the Surface.
 *
 *  \param dim the dimension of the curve
 *  \param knots the knotvector of the curve
 *  \param controlpoints the (nonrational) control points for the curve
 *  \param newloop false: this curve continues the current curveloop <BR>
 *                 true: starts a new loop
 *
 *  If it's the first curve added to the surface, it always starts
 *  a new loop, regardless of the value of \param newloop.
 *
 *  You are expected to call this function between corresponding
 *  beginEditCP()/endEditCP() pairs with the mask CurveFieldMask.
 *  You are responsible that when calling endEditCP() the 
 *  trimming curves are in a consistend state (they form closed
 *  loops and the like).
 *
 */
void Surface::addCurve( UInt32 dim,
               std::vector<Real64>& knots,
               std::vector<Pnt3f>& controlpoints,
               bool newloop)
{
    UInt32 cpsize = controlpoints.size();
    UInt32 knotsize = knots.size();
  
    if ( dim + cpsize + 1 != knotsize ) 
    {
//        SWARNING <<"Surface::addCurve: inconsistent curve attributes.." << endLog;
        return;
    }
//    SLOG <<"addCurve NI " << std::endl;    

    if ( _sfNumCurves.getValue() == 0 )
    {
        //this is the first curve -> it starts a new loop
//        SLOG <<"addcurve first curve"<<endLog;
        newloop = true;
    }
    _mfDimensions.push_back( dim );
    if ( newloop ) 
    {
        _mfCurvesPerLoop.push_back( 1 ); //1 curve in a new loop
//        SLOG <<"addcurve newloop"<<endLog;
    }
    else
    {
//        SLOG <<"addcurve NOT newloop"<<endLog;
        UInt32 cplsize = _mfCurvesPerLoop.size();
//        _mfCurvesPerLoop.setValue( _mfCurvesPerLoop.getValue( cplsize - 1 ) + 1, 
//                                   cplsize - 1 );
        _mfCurvesPerLoop[ cplsize - 1 ] = _mfCurvesPerLoop[ cplsize - 1 ] + 1;
    }
    _mfKnotLengths.push_back( knotsize );
    UInt32 i;
    for ( i = 0; i < knotsize; ++i )
    {
        _mfKnots.push_back( Real32(knots[ i ]) );
    }
    for ( i = 0; i < cpsize; ++i )
    {
        _mfCurveControlPoints.push_back( controlpoints[ i ] );
    }  
    _sfNumCurves.setValue( _sfNumCurves.getValue() + 1 );
}

    //! Remove all trimming curves from the surface.
    /*!
     * 
     *  This function removes all trimming curves from the surface.
     *
     *  You are expected to call this function between corresponding
     *  beginEditCP()/endEditCP() pairs with the mask CurveFieldMask.
     *
     */
void Surface::removeCurves( void )
{
    if ( _sfNumCurves.getValue() == 0 ) //we have no curves
    {
        return;
    }

    _sfNumCurves.setValue(0);
    _mfKnotLengths.clear();
    _mfDimensions.clear();
    _mfCurveControlPoints.clear();
    _mfKnots.clear();
    _mfCurvesPerLoop.clear();
}


//! Clone the Surface.
SurfacePtr Surface::clone( void )
{
    SurfacePtr surf = Surface::create();
    
    // clone the geometry properties
    cloneTo(surf);

    //  create copies of the attributes
    beginEditCP(surf, osg::FieldBits::AllFields);
    {
        if (getMFKnotsU() != NULL)
        {
            surf->editMFKnotsU()->setValues(*getMFKnotsU());
        }
        if (getMFKnotsV() != NULL)
        {
            surf->editMFKnotsV()->setValues(*getMFKnotsV());
        }
        if (getMFKnotLengths() != NULL)
        {
            surf->editMFKnotLengths()->setValues(*getMFKnotLengths());
        }
        if (getMFDimensions() != NULL)
        {
            surf->editMFDimensions()->setValues(*getMFDimensions());
        }
        if (getMFCurveControlPoints() != NULL)
        {
            surf->editMFCurveControlPoints()->setValues(*getMFCurveControlPoints());
        }
        if (getMFKnots() != NULL)
        {
            surf->editMFKnots()->setValues(*getMFKnots());
        }
        if (getMFCurvesPerLoop() != NULL)
        {
            surf->editMFCurvesPerLoop()->setValues(*getMFCurvesPerLoop());
        }
        surf->setMaterial  (getMaterial  ());
        surf->setDimU(getDimU());
        surf->setDimV(getDimV());
        surf->setNumCurves(getNumCurves());
        surf->setControlPoints(getControlPoints());

    }
    endEditCP(surf, osg::FieldBits::AllFields);

    return surf;
}


//! Read from a .tso file. Temporary function
/*!
 *  Read surface (and trimming) information from
 *  a file in .tso format. This is mainly used
 *  for debugging the code (with complicated models), 
 *  it's not meant to be a general user-loader function!
 *
 *  This function may or may not be removed/changed/etc. later, 
 *  you have been warned...
 *
 *  Does not perform too many checks on the input data...
 *
 *  You are expected to call this function between corresponding
 *  beginEditCP()/endEditCP() pairs with the mask
 *  CurveFieldMask|SurfaceFieldMask.
 *
 *  \param infile istream to read surface data from.
 */
//#pragma optimize("",off)
void Surface::readfromtso( std::istream &infile, bool useTextures )
{
    
}
//#pragma optimize("",on)


void Surface::writetotso( std::ostream &outfile )
{
}

void Surface::setWireframe(const GeometryPtr &value)
{
    setRefdCP(_sfWireframe.getValue(), value);
}

//! Write tessellated geometry to an .obj file. Absolutely temporary function
/*!
 *
 *  \param outfile istream to read surface data from.
 */
UInt32 Surface::writetoobj( std::ostream &outfile, UInt32 offset )
{
    unsigned int uicnt;
//     std::cerr << "current offset:" << g_current_obj_offset << std::endl;
    outfile << "g obj " << std::endl;
    GeoPositionsPtr pcl_points = getPositions();
    GeoNormalsPtr   pcl_norms = getNormals();
    GeoIndicesPtr   pcl_indices = getIndices();
    GeoTexCoordsPtr pcl_texcoords = getTexCoords();
    std::cerr << " indices size: " << pcl_indices->size() << std::endl;
    std::cerr << " points size: " << pcl_points->size() << std::endl;
    UInt32 ui_faces = pcl_indices->size() / 3;
    UInt32 uivertsize = pcl_points->size( );
    outfile << "# vertices " << uivertsize << std::endl;

    //write out vertices
    for ( uicnt = 0; uicnt < uivertsize; uicnt++) 
    {
        Pnt3f ppp = pcl_points->getValue( uicnt );
        
        outfile << "v " << ppp.x() << " " << 
                           ppp.y() << " " <<
                           ppp.z() << std::endl;
                               
    }
    //write out normals
    for ( uicnt = 0; uicnt < uivertsize; uicnt++) 
    {
        Pnt3f ppp = pcl_norms->getValue( uicnt );
            
        outfile << "vn " << ppp.x() << " " << 
                            ppp.y() << " " <<
                            ppp.z() << std::endl;
                               
    }
    //write out texture coordinates
    for ( uicnt = 0; uicnt < uivertsize; uicnt++) 
    {
        Pnt2f ppp = pcl_texcoords->getValue( uicnt );
            
        outfile << "vt " << ppp.x() << " " << 
                            ppp.y() << " " << std::endl;                          
    }
    
    outfile << "# faces " << ui_faces << std::endl;
    //write out triangles
    for ( unsigned int uitricnt = 0; uitricnt < ui_faces; uitricnt++)
    {
        UInt32 ind1 = pcl_indices->getValue( uitricnt * 3 );
        UInt32 ind2 = pcl_indices->getValue( uitricnt * 3 + 1 );
        UInt32 ind3 = pcl_indices->getValue( uitricnt * 3 + 2 );
        UInt32 i1 = ind1 + offset + 1;
        UInt32 i2 = ind2 + offset + 1;
        UInt32 i3 = ind3 + offset + 1;
            
        outfile << "f ";
        outfile << i1 << "/" << i1 << "/" << i1 << " " <<
                   i2 << "/" << i2 << "/" << i2 << " " <<
                   i3 << "/" << i3 << "/" << i3 << std::endl;
    }

    return offset + uivertsize;
}

//! output the instance for debug purposes

void Surface::dump(      UInt32    , 
                         const BitVector ) const
{
    SLOG << "Dump Surface NI" << endLog;
}

void Surface::onCreate(const Surface *source)
{
    // if we're in startup this is the prototype, which shouldn't have an id
    if(GlobalSystemState == Startup)
        return;

    Inherited::onCreate(source);
}

Transform3DPtr Surface::getTransformCore()
{
    return getTransform();
}

void Surface::onDestroy(void)
{
    Inherited::onDestroy();

    SurfacePtr thisP(*this);

    if(_sfControlPoints.getValue() != NullFC)
    {
        beginEditCP(_sfControlPoints.getValue(), Attachment::ParentsFieldMask);
        {
            _sfControlPoints.getValue()->subParent(thisP);
        }
        endEditCP(_sfControlPoints.getValue(), Attachment::ParentsFieldMask);

        subRefCP(_sfControlPoints.getValue());
    }
}

// Calculate the volume based on bounding box of the control points.
// If any of the control points have 0 weight, also take into
// account the tessellated geometry, otherwise the volume will be 
// too small. 
// FIXME: a better solution might be to do de Casteljau steps
// FIXME: until there are no control points w/ 0 weights and
// FIXME: redo the calculation...
void Surface::adjustVolume(Volume & volume)
{
    
    // check for tessellated data, if present use that for bounding box calculations since it creates a tighter bounding box
    if(getPositions() != NullFC  && getPositions()->getSize() > 0)
    {
        Geometry::adjustVolume(volume);
        return;
    }
#if 0
    Transform3DPtr geoTrans = _sfTransform.getValue();
    GeoPositionsPtr pos = getControlPoints();
    bool has_zeroweights = false;
    
    volume.setValid();
    volume.setEmpty();

    GeoPositions3fPtr pPos = GeoPositions3fPtr::dcast(
        _sfControlPoints.getValue());
    GeoPositions4fPtr pRatPos = GeoPositions4fPtr::dcast(
        _sfControlPoints.getValue());

    if(pos == NullFC)
    {
        if( geoTrans != NullFC )
            volume.transform(geoTrans->getMatrix());
        return;                  // Node has no points, no volume
    }

    if( pPos != NullFC )
    {
        for(UInt32 i = 0; i < pPos->size(); ++i)
        {
            volume.extendBy(pPos->getValue(i));
        }
    }
    else if( pRatPos != NullFC )
    {
        for(UInt32 i = 0; i < pRatPos->size(); ++i)
        {
            Pnt3f   pnt;

            if( osgabs( pRatPos->getFieldPtr()->getValue(i)[3] ) > DCTP_EPS )
            {
                pnt[0] = pRatPos->getFieldPtr()->getValue(i)[0] / pRatPos->getFieldPtr()->getValue(i)[3];
                pnt[1] = pRatPos->getFieldPtr()->getValue(i)[1] / pRatPos->getFieldPtr()->getValue(i)[3];
                pnt[2] = pRatPos->getFieldPtr()->getValue(i)[2] / pRatPos->getFieldPtr()->getValue(i)[3];
                volume.extendBy(pnt);
            }
            else
            {
                has_zeroweights = true;
            }
        }
    }
    if (has_zeroweights)
    {
        GeoPositionsPtr points = getPositions();
        if (points != NullFC)
        {
            for(UInt32 i = 0; i < points->size(); ++i)
            {
                volume.extendBy(points->getValue(i));
            }
        }
        // FIXME: a warning should be printed here since the calculated
        // FIXME: volume will not be exact if there was no tessellated 
        // FIXME: geometry, but in that case there's nothing to render 
        // FIXME: anyway so we skip the warning (which would be  
        // FIXME: unnecessarily (and annoyingly) printed at startup 
        // FIXME: e.g. when the SSM::showall() method is called).
        // FIXME: Better suggestions are welcome.
    }
    
    if( geoTrans != NullFC )
        volume.transform(geoTrans->getMatrix());
#else // we store the bent normals and the occlusion factor in texCoord7
    GeoPositions4fPtr positions4f = GeoPositions4fPtr::dcast(getControlPoints());
    GeoPositions3fPtr positions3f = GeoPositions3fPtr::dcast(getControlPoints());
    Pnt3f minExtend = Pnt3f(FLT_MAX, FLT_MAX, FLT_MAX);
    Pnt3f maxExtend = Pnt3f(-FLT_MAX, -FLT_MAX, -FLT_MAX);
    if (positions4f != NullFC && positions4f->getSize() > 0)
    {
        Pnt4f* vertices = reinterpret_cast<Pnt4f*>(&(positions4f->getField()[0]));
        const size_t numVertices = positions4f->getSize();
        for (size_t idx = 0; idx < numVertices; ++idx)
        {
            float homogenous = 1.0f;
            if(vertices[idx][3] > DCTP_EPS)
                homogenous = 1.0f / vertices[idx][3];

            minExtend[0] = std::min(minExtend[0], vertices[idx][0] * homogenous);
            minExtend[1] = std::min(minExtend[1], vertices[idx][1] * homogenous);
            minExtend[2] = std::min(minExtend[2], vertices[idx][2] * homogenous);
            maxExtend[0] = std::max(maxExtend[0], vertices[idx][0] * homogenous);
            maxExtend[1] = std::max(maxExtend[1], vertices[idx][1] * homogenous);
            maxExtend[2] = std::max(maxExtend[2], vertices[idx][2] * homogenous);
        }
    }
    else if (positions3f != NullFC && positions3f->getSize() > 0)
    {
        Pnt3f* vertices = reinterpret_cast<Pnt3f*>(&(positions3f->getField()[0]));
        const size_t numVertices = positions3f->getSize();
        for (size_t idx = 0; idx < numVertices; ++idx)
        {
            minExtend[0] = std::min(minExtend[0], vertices[idx][0]);
            minExtend[1] = std::min(minExtend[1], vertices[idx][1]);
            minExtend[2] = std::min(minExtend[2], vertices[idx][2]);
            maxExtend[0] = std::max(maxExtend[0], vertices[idx][0]);
            maxExtend[1] = std::max(maxExtend[1], vertices[idx][1]);
            maxExtend[2] = std::max(maxExtend[2], vertices[idx][2]);
        }
    }
    _volumeCache.extendBy(minExtend);
    _volumeCache.extendBy(maxExtend);

    volume.setValid();
    volume.setEmpty();
    volume.extendBy(_volumeCache);
    Transform3DPtr geoTrans = _sfTransform.getValue();
    if (geoTrans != NullFC)
        volume.transform(geoTrans->getMatrix());
#endif
}



