/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright (C) 2000-2002 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

#include <stdlib.h>
#include <stdio.h>

#include "OSGConfig.h"

#include <OSGGL.h>

#include <OSGDrawAction.h>
#include <OSGRenderAction.h>
#include <OSGTransform3D.h>
#include <OSGMatrix.h>
#include <OSGVector.h>

#include "OSGClipPlane.h"


OSG_USING_NAMESPACE

/*! \class osg::ClipPlane
    \ingroup GrpSystemNodeCoresClipPlanes
    
*/

/*-------------------------------------------------------------------------*/
/*                             Chunk                                       */

SClipPlaneChunkPtr ClipPlane::getChunk(void)
{
    return _pChunk;
}

void ClipPlane::makeChunk(void)
{
    if(_pChunk == NullFC)
    {
        _pChunk = SClipPlaneChunk::create();
    }

    _pChunk->setEquation(getEquation());
}

/*-------------------------------------------------------------------------*/
/*                             Sync                                     */

void ClipPlane::changed(BitVector whichField, UInt32 origin)
{
    if(whichField & TransformFieldMask)
    {

        ClipPlanePtr tmpPtr(*this);
        Transform3DPtr pTrans = _sfTransform.getValue();
        if( pTrans != NullFC)
            pTrans->setAssociatedCore(tmpPtr); // set the core reference
 //       invalidateVolume();
    }
    Inherited::changed(whichField, origin);
}


/*-------------------------------------------------------------------------*/
/*                                Dump                                     */

void ClipPlane::dump(      UInt32    uiIndent, 
                     const BitVector bvFlags) const
{
   Inherited::dump(uiIndent, bvFlags);
}

/*-------------------------------------------------------------------------*/
/*                            Constructors                                 */

ClipPlane::ClipPlane(void) :
     Inherited(),
    _pChunk   (NullFC)
{
}

ClipPlane::ClipPlane(const ClipPlane &source) :
     Inherited(source),
    _pChunk   (source._pChunk)
{
}

/*-------------------------------------------------------------------------*/
/*                             Destructor                                  */

ClipPlane::~ClipPlane(void)
{
    if(_pChunk != NullFC)
        subRefCP(_pChunk);
}


void ClipPlane::onCreate(const ClipPlane *source)
{
    // if we're in startup this is the prototype, which shouldn't have an id
    if(GlobalSystemState == Startup)
        return;

    ClipPlanePtr tmpPtr(*this);

    // ok calling tmpPtr->setTransform() leads to crash while loading a vrml file,
    // the vrml loader calls shallowCopy()
    beginEditCP(tmpPtr, ClipPlane::TransformFieldMask);
        //tmpPtr->setTransform(Transform3D::create());
        Transform3DPtr t3d = Transform3D::create();
        t3d->setAssociatedCore(tmpPtr); // set the core reference
        addRefCP(t3d);
        _sfTransform.setValue(t3d);
    endEditCP(tmpPtr, ClipPlane::TransformFieldMask);
}

void ClipPlane::onDestroy(void)
{
    if( getTransform() != NullFC)
        getTransform()->setAssociatedCore(NullFC);
    subRefCP(_sfTransform.getValue());
    Inherited::onDestroy();
}


/*-------------------------------------------------------------------------*/
/*                               Init                                      */

//! initialize the static features of the class, e.g. action callbacks

void ClipPlane::initMethod(void)
{

}

osg::Vec4f ClipPlane::getClippingEquation() const
{
    if( getBeacon() != osg::NullFC)
    {
        osg::Matrix objectToWorld = getBeacon()->getToWorld();
        osg::Pnt3f P = objectToWorld * osg::Pnt3f(0.0f, 0.0f, 0.0f);

        osg::Matrix normalMatrix( objectToWorld);
        normalMatrix.invert();
        normalMatrix.transpose();
        
        osg::Vec3f N = normalMatrix * osg::Vec3f(0.0f, 0.0f, 1.0f);
        return osg::Vec4f( N.x(), N.y(), N.z(), -(N.dot(P)));
    }
    else if( getTransform() != osg::NullFC)
    {
        osg::Matrix identity;
        if( getTransform()->isIdentity())
        {
            return getEquation();
        }
        else
        {
            osg::Matrix objectToWorld = getTransform()->getMatrix();
            osg::Pnt3f P = objectToWorld * osg::Pnt3f(0.0f, 0.0f, 0.0f);
            osg::Matrix normalMatrix( objectToWorld);
            normalMatrix.invert();
            normalMatrix.transpose();
            osg::Vec3f N = normalMatrix * osg::Vec3f(getEquation()[0], getEquation()[1], getEquation()[2]);
            return osg::Vec4f(N.x(), N.y(), N.z(), -(N.dot(P)));
        }
    }
    return getEquation();
}


/*-------------------------------------------------------------------------*/
/*                              cvs id's                                   */

#ifdef __sgi
#pragma set woff 1174
#endif

#ifdef OSG_LINUX_ICC
#pragma warning( disable : 177 )
#endif

namespace
{
    static Char8 cvsid_cpp[] = "@(#)$Id: OSGClipPlane.cpp,v 1.1 2007/04/26 15:22:01 a-m-z Exp $";
    static Char8 cvsid_hpp[] = OSGCLIPPLANE_HEADER_CVSID;
    static Char8 cvsid_inl[] = OSGCLIPPLANE_INLINE_CVSID;
}

