/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *               Copyright (C) 2000-2002 by the OpenSG Forum                 *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

#include <stdlib.h>
#include <stdio.h>

#include <OSGConfig.h>

#include "OSGDatabase.h"

OSG_BEGIN_NAMESPACE

/***************************************************************************\
 *                            Description                                  *
\***************************************************************************/

/*! \class osg::Database

*/

/***************************************************************************\
 *                           Class variables                               *
\***************************************************************************/

/***************************************************************************\
 *                           Class methods                                 *
\***************************************************************************/

void Database::initMethod (void)
{
}


/***************************************************************************\
 *                           Instance methods                              *
\***************************************************************************/

/*-------------------------------------------------------------------------*\
 -  private                                                                 -
\*-------------------------------------------------------------------------*/

/*----------------------- constructors & destructors ----------------------*/

Database::Database(void) :
    Inherited()
{
}

Database::Database(const Database &source) :
    Inherited(source)
{
}

Database::~Database(void)
{
   
}

void Database::onDestroy(void)
{
    // if we're in shutdown this is the prototype ...
    if(osg::GlobalSystemState == osg::Shutdown)
        return;

     // make sure attributes are released (ref count removed)
    clearAllEntries();
}

/*----------------------------- class specific ----------------------------*/

void Database::changed(BitVector whichField, UInt32 origin)
{
    Inherited::changed(whichField, origin);
}

void Database::dump(      UInt32    , 
                         const BitVector ) const
{
    SLOG << "Dump Database NI" << std::endl;
}

void Database::initLookupTable()
{
    if(m_map.size() != _mfEntries.size())
    {
        m_map.clear();
        for (auto & entry : _mfEntries)
        {
            m_map.insert({ entry->getUuid(), entry});
        }
    }
}

osg::DatabaseEntryPtr osg::Database::getEntry(const std::string& uuid)
{
    initLookupTable();
    const auto& entry = m_map.find(uuid);
    if( entry != m_map.end())
    {
        return entry->second;
    }
    return osg::NullFC;
}


bool osg::Database::addEntry(const DatabaseEntryPtr & entry)
{
    if ( entry == osg::NullFC || entry->getUuid().empty())
        return false;

    if( getEntry(entry->getUuid()) != osg::NullFC)
    {
        return true;
    }
    addRefCP(entry);
    DatabasePtr temp(*this);
    beginEditCP(temp, Database::EntriesFieldMask);
        _mfEntries.push_back(entry);
    endEditCP(temp, Database::EntriesFieldMask);
    m_map.insert({ entry->getUuid(), entry});
    return true;
}

bool osg::Database::deleteEntry(const DatabaseEntryPtr & entry)
{
    if ( entry == osg::NullFC || entry->getUuid().empty())
        return false;

    const std::string& uuid = entry->getUuid();
    // remove the entry from the lookup table
    const auto& mapEntry = m_map.find(uuid);
    if( mapEntry != m_map.end())
    {
        m_map.erase(mapEntry);
    }
    // now remove the entry from the database
    DatabasePtr temp(*this);
    beginEditCP(temp, Database::EntriesFieldMask);
    for (auto it = _mfEntries.begin(); it != _mfEntries.end(); ++it)
    {
        if((*it)->getUuid() == uuid)
        {
            subRefCP(*it);
            _mfEntries.erase(it);
            break;
        }
    }
    endEditCP(temp, Database::EntriesFieldMask);
    return true;
}


void osg::Database::clearAllEntries()
{
    DatabasePtr temp(*this);
    m_map.clear();
    beginEditCP(temp, Database::EntriesFieldMask);
    for ( auto & entry : _mfEntries )
        subRefCP(entry);

    _mfEntries.clear();
    endEditCP(temp, Database::EntriesFieldMask);
}

bool osg::Database::hasEntry(const std::string& uuid)
{
    initLookupTable();
    return m_map.count(uuid) != 0;
}

bool osg::Database::mergeDatabase(const DatabasePtr& other)
{
    for (auto it = other->getMFEntries()->begin(); it != other->getMFEntries()->end(); ++it)
    {
        addEntry(*it);
    }
    return true;
}

OSG_END_NAMESPACE

