/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright (C) 2000-2002 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

#include <stdlib.h>
#include <stdio.h>

#include <OSGConfig.h>

#include "OSGNodePtr.h"
#include "OSGMaterialPool.h"
#include "OSGMaterialGroup.h"
#include "OSGMaterialDrawable.h"
#include "OSGName.h"
#include <OSGCast.h>

namespace {
    constexpr char _root_name[] = "Materialpool Root";
    constexpr char _matgroup_root_name[] = "Materials";
    constexpr char _light_root_name[] = "LightMaterialPoolRoot";
    constexpr char _camera_root_name[] = "CameraMaterialPoolRoot";
    constexpr char _constraint_root_name[] = "ConstraintMaterialPoolRoot";
    constexpr char _annotation_root_name[] = "AnnotationMaterialPoolRoot";
    constexpr char _sceneplate_root_name[] = "SceneplateMaterialPoolRoot";
    constexpr char _physics_root_name[] = "PhysicsMaterialPoolRoot";
    constexpr char _overlay_root_name[] = "OverlayMaterialPoolRoot";
    constexpr char _lightsets_root_name[] = "LightSetsMaterialPoolRoot";    
}

OSG_USING_NAMESPACE

/*! \class osg::MaterialPool
*/

/*-------------------------------------------------------------------------*/
/*                                Sync                                     */

static void addRootFlag(const osg::NodePtr& root)
{
    if(root != osg::NullFC)
    {
        beginEditCP(root, Node::FlagsFieldMask);
        root->setFlags(root->getFlags() | Node::NFModuleGraphRoot);
        endEditCP(root, Node::FlagsFieldMask);
    }
    
}

void MaterialPool::changed(BitVector whichField, UInt32 origin)
{
    if(whichField & MaterialsFieldMask)
    {
        sync();
    }

    // Ensure all Root nodes have the NFModuleGraphRoot flag set
    if(whichField & RootFieldMask)
    {
        addRootFlag(getRoot());
    }
    if(whichField & CameraRootFieldMask)
    {
        addRootFlag(getCameraRoot());
    }
    if(whichField & LightRootFieldMask)
    {
        addRootFlag(getLightRoot());
    }
    if(whichField & LightSetsRootFieldMask)
    {
        addRootFlag(getLightSetsRoot());
    }
    if(whichField & OverlayRootFieldMask)
    {
        addRootFlag(getOverlayRoot());
    }
    if(whichField & SceneplateRootFieldMask)
    {
        addRootFlag(getSceneplateRoot());
    }
    if(whichField & PhysicsRootFieldMask)
    {
        addRootFlag(getPhysicsRoot());
    }
    if(whichField & ConstraintsRootFieldMask)
    {
        addRootFlag(getConstraintsRoot());
    }
    if(whichField & AnnotationRootFieldMask)
    {
        addRootFlag(getAnnotationRoot());
    }

    Inherited::changed(whichField, origin);
}

/*-------------------------------------------------------------------------*/
/*                                Dump                                     */

void MaterialPool::dump(      UInt32    uiIndent,
                         const BitVector bvFlags) const
{
    Inherited::dump(uiIndent, bvFlags);
}

/*-------------------------------------------------------------------------*/
/*                            Constructors                                 */

MaterialPool::MaterialPool(void) :
    Inherited(),
    _mats()
{
}

MaterialPool::MaterialPool(const MaterialPool &source) :
    Inherited(source),
    _mats(source._mats)
{
}

/*-------------------------------------------------------------------------*/
/*                             Destructor                                  */

MaterialPool::~MaterialPool(void)
{
    subRefCP(_sfRoot.getValue());
    subRefCP(_sfLightRoot.getValue());
    subRefCP(_sfLightSetsRoot.getValue());
    subRefCP(_sfOverlayRoot.getValue());
    subRefCP(_sfSceneplateRoot.getValue());
    subRefCP(_sfConstraintsRoot.getValue());
    subRefCP(_sfCameraRoot.getValue());
    subRefCP(_sfAnnotationRoot.getValue());
    subRefCP(_sfResourceDatabase.getValue());
    subRefCP(_sfPhysicsRoot.getValue());
}

void MaterialPool::onDestroy(void)
{
    // if we're in shutdown this is the prototype ...
    if(osg::GlobalSystemState == osg::Shutdown)
        return;

    MFMaterialPtr::iterator       matIt        = _mfMaterials.begin();
    MFMaterialPtr::const_iterator endMaterials = _mfMaterials.end  ();

    FINFO(("MaterialPool::onDestroy : subrefing %u materials\n", _mfMaterials.size()));
    while(matIt != endMaterials)
    {
        subRefCP(*matIt);
        ++matIt;
    }

    for (auto &metadataSet : _mfMetadataSets)
        subRefCP(metadataSet);
    _mfMetadataSets.clear();

    Inherited::onDestroy();
}

/*-------------------------------------------------------------------------*/
/*                                Init                                     */

void MaterialPool::initMethod(void)
{
}


void MaterialPool::sync(void)
{
    // should be safe as the materials interface is protected.
    if(_mfMaterials.size() == _mats.size())
    {
        FINFO(("MaterialPool::sync : %u materials already synced, nothing to do.\n", _mfMaterials.size()));
        return;
    }

    MFMaterialPtr::iterator       matIt        = _mfMaterials.begin();
    MFMaterialPtr::const_iterator endMaterials = _mfMaterials.end  ();

	_mats.clear();
    while(matIt != endMaterials)
    {
        // this is already done in the osb loader!
        //addRefCP(*matIt);
        _mats.insert(*matIt);
        ++matIt;
    }
    FINFO(("MaterialPool::sync : synced %u materials\n", _mfMaterials.size()));
}

/*-------------------------------------------------------------------------*/
/*                             Access                                      */


bool MaterialPool::isEmpty() const
{
    // there are materials in the pool
    if (getCount() > 0)
        return false;

    const NodePtr& lightRoot = getLightRoot();
    if (lightRoot != osg::NullFC && (lightRoot->getNChildren() > 0))
        return false;

    const NodePtr& overlayRoot = getOverlayRoot();
    if (overlayRoot != osg::NullFC && (overlayRoot->getNChildren() > 0))
        return false;

    const NodePtr& sceneplateRoot = getSceneplateRoot();
    if (sceneplateRoot != osg::NullFC && (sceneplateRoot->getNChildren() > 0))
        return false;

    const NodePtr& physicsRoot = getPhysicsRoot();
    if (physicsRoot != osg::NullFC && (physicsRoot->getNChildren() > 0))
        return false;

    const NodePtr& lightSetRoot = getLightSetsRoot();
    if (lightSetRoot != osg::NullFC && (lightSetRoot->getNChildren() > 0))
        return false;

    const NodePtr& constraintRoot = getConstraintsRoot();
    if (constraintRoot != osg::NullFC && (constraintRoot->getNChildren() > 0))
        return false;

    const NodePtr& cameraRoot = getCameraRoot();
    if (cameraRoot != osg::NullFC && (cameraRoot->getNChildren() > 0))
        return false;

    const NodePtr& annotationRoot = getAnnotationRoot();
    if (annotationRoot != osg::NullFC && (annotationRoot->getNChildren() > 0))
        return false;

    if (_mfMetadataSets.size() > 0)
        return false;

    return true;
}


bool MaterialPool::add(const MaterialPtr &mat)
{
    return add(mat, NullFC, true, 0);
}

bool MaterialPool::add(const MaterialPtr &mat, const NodePtr &parent, bool create_node, const UInt32 pos)
{
    if(mat == NullFC)
        return false;

    if(_mats.count(mat) > 0)
        return false;

    addRefCP(mat);
    _mats.insert(mat);
    _mfMaterials.push_back(mat);

    NodePtr parent_group = parent;

    if(parent_group == NullFC)
    {
        initMaterialRoot();
        parent_group = getRoot()->getChild(0);
    }

    if(create_node)
    {
        MaterialGroupPtr mgc = MaterialGroup::create();
        beginEditCP(mgc, osg::FieldBits::AllFields);
            mgc->setMaterial(mat);
        endEditCP(mgc, osg::FieldBits::AllFields);
        NodePtr mg = Node::create();
        beginEditCP(mg, osg::FieldBits::AllFields);
            mg->setCore(mgc);
        endEditCP(mg, osg::FieldBits::AllFields);
        beginEditCP(parent_group, Node::ChildrenFieldMask);
            parent_group->insertChild( (pos <= parent_group->getNChildren()) ? pos : 0 , mg);
        endEditCP(parent_group, Node::ChildrenFieldMask);
    }

    return true;
}

void MaterialPool::add(const NodePtr &root)
{
    if(root == NullFC)
        return;

    traverse(root, std::bind(&MaterialPool::addMaterialCB, this, std::placeholders::_1));

    removeInvalidMaterialGroups();
}

void MaterialPool::add(const MaterialPoolPtr &mp)
{
    if(mp == NullFC)
        return;

    if(mp == getPtr())
        return;

    initMaterialRoot();
   
    bool create_node = false;
    if(mp->getRoot() != NullFC && mp->getRoot()->getNChildren() > 0 && mp->getRoot()->getChild(0)->getNChildren() > 0)
    {
        beginEditCP(getRoot(), osg::FieldBits::AllFields);
            for(UInt32 i=0;i<mp->getRoot()->getChild(0)->getNChildren();++i)
                getRoot()->getChild(0)->addChild(osg::deepCloneTree(mp->getRoot()->getChild(0)->getChild(i), "Material, Metadata"));
        endEditCP(getRoot(), osg::FieldBits::AllFields);
    }
    else
    {
        // there was no root nor children so we add all materials to the root node later.
        create_node = true;
    }

    addLights(mp);

    addLightSets(mp);
    
    addOverlays(mp);

    addSceneplates(mp);

    addPhysics(mp);
    
    addConstraints(mp);

    addCameras(mp);

    addAnnotations(mp);

    addResourceDatabase(mp);
    addMetadata(mp);
    
    MFMaterialPtr::iterator       matIt        = mp->_mfMaterials.begin();
    MFMaterialPtr::const_iterator endMaterials = mp->_mfMaterials.end  ();

    while(matIt != endMaterials)
    {
        add(*matIt, NullFC, create_node, 0);
        ++matIt;
    }

    removeInvalidMaterialGroups();
}

void MaterialPool::addLights(const MaterialPoolPtr &mp)
{
    if(mp == NullFC || mp == getPtr())
        return;

    initLightRoot();

    if(mp->getLightRoot() != NullFC && mp->getLightRoot()->getNChildren() > 0)
    {
        beginEditCP(getLightRoot(), osg::FieldBits::AllFields);

        while(mp->getLightRoot()->getNChildren())
        {
            //getLightRoot()->addChild(osg::deepCloneTree(mp->getLightRoot()->getChild(i), "Light"));
            // we can not copy these since this would break the connection of the nodes in the associated field
            getLightRoot()->addChild(mp->getLightRoot()->getChild(0));
        }
        endEditCP(getLightRoot(), osg::FieldBits::AllFields);
    }
}

void MaterialPool::addLightSets(const MaterialPoolPtr &mp)
{
    if(mp == NullFC || mp == getPtr())
        return;

    initLightSetsRoot();

    if (mp->getLightSetsRoot() != NullFC && mp->getLightSetsRoot()->getNChildren() > 0)
    {
        beginEditCP(getLightSetsRoot(), osg::FieldBits::AllFields);

        while (mp->getLightSetsRoot()->getNChildren())
        {
            //getLightRoot()->addChild(osg::deepCloneTree(mp->getLightRoot()->getChild(i), "Light"));
            // we can not copy these since this would break the connection of the nodes in the associated field
            getLightSetsRoot()->addChild(mp->getLightSetsRoot()->getChild(0));
        }
        endEditCP(getLightSetsRoot(), osg::FieldBits::AllFields);
    }
}

void MaterialPool::addCameras(const MaterialPoolPtr &mp)
{
    if(mp == NullFC || mp == getPtr())
        return;

    initCameraRoot();

    if(mp->getCameraRoot() != NullFC && mp->getCameraRoot()->getNChildren() > 0)
    {
        beginEditCP(getCameraRoot(), Node::ChildrenFieldMask);

        while(mp->getCameraRoot()->getNChildren())
            getCameraRoot()->addChild(mp->getCameraRoot()->getChild(0));

        endEditCP(getCameraRoot(), Node::ChildrenFieldMask);
    }
}

void MaterialPool::addConstraints(const MaterialPoolPtr &mp)
{
    if(mp == NullFC || mp == getPtr())
        return;

    initConstraintsRoot();

    if(mp->getConstraintsRoot() != NullFC && mp->getConstraintsRoot()->getNChildren() > 0)
    {
        beginEditCP(getConstraintsRoot(), Node::ChildrenFieldMask);

        while(mp->getConstraintsRoot()->getNChildren())
            getConstraintsRoot()->addChild(mp->getConstraintsRoot()->getChild(0));

        endEditCP(getConstraintsRoot(), Node::ChildrenFieldMask);
    }
}

void MaterialPool::addOverlays(const MaterialPoolPtr &mp)
{
    if(mp == NullFC || mp == getPtr())
        return;

    initSceneplateRoot();

    if(mp->getOverlayRoot() != NullFC && mp->getOverlayRoot()->getNChildren() > 0)
    {
        beginEditCP(getSceneplateRoot(), osg::FieldBits::AllFields);
        while(mp->getOverlayRoot()->getNChildren())
        {
            getSceneplateRoot()->addChild(mp->getOverlayRoot()->getChild(0));
        }
        endEditCP(getSceneplateRoot(), osg::FieldBits::AllFields);
    }
}

std::vector<NodePtr> MaterialPool::addSceneplates(const MaterialPoolPtr &mp)
{
    std::vector<NodePtr> addedNodes;

    if (mp == NullFC || mp == getPtr())
        return addedNodes;

    initSceneplateRoot();

    if (mp->getSceneplateRoot() != NullFC && mp->getSceneplateRoot()->getNChildren() > 0)
    {
        beginEditCP(getSceneplateRoot(), osg::FieldBits::AllFields);
        while (mp->getSceneplateRoot()->getNChildren())
        {
            osg::NodePtr child = mp->getSceneplateRoot()->getChild(0);
            addedNodes.push_back(child);
            getSceneplateRoot()->addChild(child);
        }
        endEditCP(getSceneplateRoot(), osg::FieldBits::AllFields);
    }

    return addedNodes;
}

void MaterialPool::addPhysics(const MaterialPoolPtr &mp)
{
    if (mp == NullFC || mp == getPtr())
        return;

    initPhysicsRoot();

    if (mp->getPhysicsRoot() != NullFC && mp->getPhysicsRoot()->getNChildren() > 0)
    {
        beginEditCP(getPhysicsRoot(), osg::FieldBits::AllFields);
        while (mp->getPhysicsRoot()->getNChildren())
        {
            getPhysicsRoot()->addChild(mp->getPhysicsRoot()->getChild(0));
        }
        endEditCP(getPhysicsRoot(), osg::FieldBits::AllFields);
    }
}

void MaterialPool::addAnnotations(const MaterialPoolPtr &mp)
{
    if (mp == NullFC || mp == getPtr())
        return;

    initAnnotationRoot();

    if (mp->getAnnotationRoot() != NullFC && mp->getAnnotationRoot()->getNChildren() > 0)
    {
        beginEditCP(getAnnotationRoot(), Node::ChildrenFieldMask);

        while (mp->getAnnotationRoot()->getNChildren())
            getAnnotationRoot()->addChild(mp->getAnnotationRoot()->getChild(0));

        endEditCP(getAnnotationRoot(), Node::ChildrenFieldMask);
    }
}

void MaterialPool::addResourceDatabase(const MaterialPoolPtr &mp)
{
    if (mp == NullFC || mp == getPtr())
        return;

    initResourceDatabase();

    auto thisDb = getResourceDatabase();
    auto otherDb = mp->getResourceDatabase();
    if (otherDb != NullFC && thisDb != NullFC)
    {
        thisDb->initLookupTable();
        beginEditCP(thisDb, Database::EntriesFieldMask);
        for (auto it = otherDb->editMFEntries()->begin(); it != otherDb->editMFEntries()->end(); ++it)
        {
            auto newEntry = (*it);
            // check if the entry already exists
            thisDb->addEntry(newEntry);
        }
        endEditCP(thisDb, Database::EntriesFieldMask);
        otherDb->clearAllEntries(); // clear the other database
    }
}

void MaterialPool::addMetadata(const MaterialPoolPtr &src)
{
    if (src == NullFC || src == getPtr())
        return;

    for (auto &metadataSet : src->_mfMetadataSets)
    {
        addRefCP(metadataSet);
        _mfMetadataSets.push_back(metadataSet);
    }
}

bool MaterialPool::addMetadataSet(const AttachmentContainerPtr &value)
{
    if (!containsMetadataSet(value))
    {
        addRefCP(value);
        _mfMetadataSets.push_back(value);
        return true;
    }
    return false;
}

bool MaterialPool::removeMetadataSet(const AttachmentContainerPtr &value)
{
    if (auto it = _mfMetadataSets.find(value); it != _mfMetadataSets.end())
    {
        _mfMetadataSets.erase(it);
        subRefCP(value);
        return true;
    }
    return false;
}

bool MaterialPool::containsMetadataSet(const AttachmentContainerPtr &value)
{
    return (_mfMetadataSets.find(value) != _mfMetadataSets.end());
}

void MaterialPool::clearMetadataSets()
{
    for (auto &metadataSet : _mfMetadataSets)
        subRefCP(metadataSet);
    _mfMetadataSets.clear();
}

Int32 MaterialPool::find(const MaterialPtr &mat) const
{
    if(mat == NullFC)
        return -1;

    if(_mats.count(mat) == 0)
        return -1;

    UInt32 index;

    for(index = 0; index < _mfMaterials.size(); ++index)
    {
        if(_mfMaterials[index] == mat)
            return index;
    }
    return -1;
}

NodePtr MaterialPool::find(const NodePtr &node, const MaterialPtr &mat) const
{
    if(node == NullFC || mat == NullFC)
        return NullFC;

    NodePtr r = NullFC;
    MaterialGroupPtr mgc = MaterialGroupPtr::dcast(node->getCore());

    if(mgc != NullFC && mgc->getMaterial() == mat)
        return node;

    for(UInt32 i=0;i<node->getNChildren();++i)
    {
        r = find(node->getChild(i), mat);
        if(r != NullFC)
            break;
    }

    return r;
}

void MaterialPool::find(std::vector<NodePtr> &nodes, const NodePtr &node, const MaterialPtr &mat) const
{
    if(node == NullFC || mat == NullFC)
        return;

    MaterialGroupPtr mgc = MaterialGroupPtr::dcast(node->getCore());

    if(mgc != NullFC && mgc->getMaterial() == mat)
        nodes.push_back(node);

    for(UInt32 i=0;i<node->getNChildren();++i)
    {
        find(nodes, node->getChild(i), mat);
    }
}

void MaterialPool::replace(const MaterialPtr &old_mat, const MaterialPtr &new_mat)
{
    if(old_mat == NullFC || new_mat == NullFC || old_mat == new_mat)
        return;

    if(_mats.count(old_mat) == 0)
    {
        SWARNING << "MaterialPool(" << this << ")::replace: " << old_mat
                 << " is not one of my materials!" << std::endl;
        return;
    }

    MFMaterialPtr::iterator matIt = _mfMaterials.find(old_mat);

    if(matIt != _mfMaterials.end())
    {
        bool new_mat_referenced = (_mats.count(new_mat) > 0);
        if(new_mat_referenced)
        {
            // the new mat is already in the pool so we increase the ref count to keep it alive.
            // and remove it from the pool.
            addRefCP(new_mat);
            sub(new_mat); // might invalidate matIt as new_mat is erased from _mfMaterials
            matIt = _mfMaterials.find(old_mat); // get valid iterator 
            if(matIt == _mfMaterials.end())
                return;
        }

        std::vector<NodePtr> nodes;
        find(nodes, getRoot(), old_mat);

        for(size_t i = 0; i < nodes.size(); i++)
        {
            NodePtr node = nodes[i];
            if(node != NullFC)
            {
                MaterialGroupPtr mg = MaterialGroupPtr::dcast(node->getCore());
                if(mg != NullFC)
                {
                    beginEditCP(mg, osg::FieldBits::AllFields);
                    mg->setMaterial(new_mat);
                    endEditCP(mg, osg::FieldBits::AllFields);
                }
            }
        }
        _mats.erase(old_mat);
        subRefCP(old_mat);

        *matIt = new_mat;
        if(!new_mat_referenced)
            addRefCP(new_mat);
        _mats.insert(new_mat);
    }
    else
    {
        SWARNING << "MaterialPool(" << this << ")::replace: " << old_mat
                 << " is not one of my materials!" << std::endl;
    }
}

void MaterialPool::sub(const MaterialPtr &mat)
{
    if(mat == NullFC)
        return;

    if(_mats.count(mat) == 0)
    {
        SWARNING << "MaterialPool(" << this << ")::sub: " << mat
                 << " is not one of my materials!" << std::endl;
        return;
    }

    MFMaterialPtr::iterator matIt = _mfMaterials.find(mat);

    if(matIt != _mfMaterials.end())
    {
        std::vector<NodePtr> nodes;
        find(nodes, getRoot(), mat);

        for(size_t i = 0; i < nodes.size(); i++)
        {
            NodePtr node = nodes[i];
            if(node != NullFC)
            {
                NodePtr parent = node->getParent();
                if(parent != NullFC)
                {
                    beginEditCP(parent, Node::ChildrenFieldMask);
                        parent->subChild(node);
                    endEditCP(parent, Node::ChildrenFieldMask);
                }
                else
                {
                    SWARNING << "MaterialPool(" << this << ")::sub: " << mat
                             << " node parent is NULL!" << std::endl;
                }
            }
        }
        _mats.erase(mat);
        subRefCP(mat);

        _mfMaterials.erase(matIt);
    }
    else
    {
        SWARNING << "MaterialPool(" << this << ")::sub: " << mat
                 << " is not one of my materials!" << std::endl;
    }
}

void MaterialPool::sub(UInt32 index)
{
    MFMaterialPtr::iterator matIt = _mfMaterials.begin();

    matIt += index;

    if(matIt != _mfMaterials.end())
    {
        subRefCP(*matIt);
        _mats.erase(*matIt);
        _mfMaterials.erase(matIt);
    }
}

void MaterialPool::get(std::vector<MaterialPtr> &mats)
{
    mats.clear();
    mats.reserve(_mfMaterials.getSize());
    MFMaterialPtr::iterator       matIt        = _mfMaterials.begin();
    MFMaterialPtr::const_iterator endMaterials = _mfMaterials.end  ();

    while(matIt != endMaterials)
    {
        mats.push_back(*matIt);
        ++matIt;
    }
}

const std::set<MaterialPtr> &MaterialPool::get(void)
{
    return _mats;
}

void MaterialPool::clear(void)
{
    MFMaterialPtr::iterator       matIt        = _mfMaterials.begin();
    MFMaterialPtr::const_iterator endMaterials = _mfMaterials.end  ();

    while(matIt != endMaterials)
    {
        subRefCP(*matIt);
        ++matIt;
    }
    _mfMaterials.clear();
    _mats.clear();
    subRefCP(getRoot());
    setRoot(NullFC);

    subRefCP(getLightRoot());
    setLightRoot(NullFC);

    subRefCP(getLightSetsRoot());
    setLightSetsRoot(NullFC);
    
    subRefCP(getOverlayRoot());
    setOverlayRoot(NullFC);

    subRefCP(getSceneplateRoot());
    setSceneplateRoot(NullFC);

    subRefCP(getPhysicsRoot());
    setPhysicsRoot(NullFC);

    subRefCP(getConstraintsRoot());
    setConstraintsRoot(NullFC);

    subRefCP(getCameraRoot());
    setCameraRoot(NullFC);

    subRefCP(getAnnotationRoot());
    setAnnotationRoot(NullFC);

    subRefCP(getResourceDatabase());
    setResourceDatabase(NullFC);
    
    clearMetadataSets();
    
}

Action::ResultE MaterialPool::addMaterialCB(const NodePtr &node)
{
    if(node == NullFC)
        return Action::Continue;

    FieldContainerPtr core = node->getCore();

    if(core == NullFC)
        return Action::Continue;

    MaterialGroupPtr mg = MaterialGroupPtr::dcast(core);
    if(mg != NullFC)
    {
        // ignore the materials form our light sources geometry!
        FieldContainerType *value_pair_type = FieldContainerFactory::the()->findType("ValuePair");
        if(value_pair_type != NULL)
        {
            AttachmentPtr vp = node->findAttachment(*value_pair_type);
            if(vp != NullFC)
            {
                Field *field = vp->getField("key");
                if(field != NULL)
                {
                     MFString *keys = dynamic_cast<MFString *>(field);
                     if(keys != NULL)
                     {
                         if(!keys->empty())
                         {
                             if(keys->getValue(0) == "LightBeaconGroup")
                             {
                                 // we skip all children!
                                 return Action::Skip;
                             }
                         }
                     }
                }
            }
        }
        add(mg->getMaterial(), NullFC, false, 0);
        return Action::Continue;
    }

    MaterialPoolPtr mp = MaterialPoolPtr::dcast(core);
    if(mp != NullFC)
    {
        add(mp);
        return Action::Continue;
    }

    MaterialDrawablePtr md = MaterialDrawablePtr::dcast(core);
    if(md != NullFC)
    {
        // add materials referenced in an attached raytracing geo attachment
        static FieldContainerType* raytracing_geo_attachment_type = FieldContainerFactory::the()->findType("RaytracingGeoAttachment");
        if(raytracing_geo_attachment_type == nullptr)
            raytracing_geo_attachment_type = FieldContainerFactory::the()->findType("RaytracingGeoAttachment");

        if(raytracing_geo_attachment_type != nullptr)
        {
            AttachmentPtr rtGeoAttachment = node->findAttachment(*raytracing_geo_attachment_type);
            if(rtGeoAttachment != NullFC)
            {
                Field *field = rtGeoAttachment->getField("overrideMaterial");
                if(field != NULL)
                {
                    MaterialPtr overrideMat = MaterialPtr::dcast((static_cast<SFFieldContainerPtr *>(field))->getValue());
                    if(overrideMat != NullFC)
                        add(overrideMat, NullFC, false, 0);
                }
            }
        }
        add(md->getMaterial(), NullFC, false, 0);
        return Action::Continue;
    }

    const char *typename2 = core->getType().getName().str();
    if(!strcmp(typename2, "DVRVolume"))
    {
        Field *field = core->getField("renderMaterial");
        if(field != NULL)
        {
            add(MaterialPtr::dcast((static_cast<SFFieldContainerPtr *>(field))->getValue()), NullFC, false, 0);
        }
    }
    else if (!strcmp(typename2, "UberVolume"))
    {
        if(auto *field = core->getField("material"); field != nullptr)
        {
            add(MaterialPtr::dcast((static_cast<SFFieldContainerPtr *>(field))->getValue()), NullFC, false, 0);
        }
    }

    return Action::Continue;
}

void MaterialPool::initMaterialRoot()
{
    // 2-level root structure:
    
    // - Root ("Materialpool Root")
    //    - MaterialGroupRoot ("Materials")

    // Init Root.
    if(getRoot() == NullFC)
    {
        NodePtr root = Node::create();
        osg::setName(root, _root_name, true);
        beginEditCP(root, osg::FieldBits::AllFields);
        root->setCore(Group::create());
        endEditCP(root, osg::FieldBits::AllFields);

        MaterialPoolPtr tmp(*this);
        beginEditCP(tmp, RootFieldMask);
        tmp->setRoot(root);
        endEditCP(tmp, RootFieldMask);
    }

    // Init materials group root.
    if(getMaterialGroupRoot() == NullFC && getRoot() != NullFC)
    {
        NodePtr child0 = Node::create();
        osg::setName(child0, _matgroup_root_name, true);
        beginEditCP(child0, osg::FieldBits::AllFields);
        child0->setCore(Group::create());
        endEditCP(child0, osg::FieldBits::AllFields);
        beginEditCP(getRoot(), osg::FieldBits::AllFields);
        getRoot()->insertChild(0, child0);
        endEditCP(getRoot(), osg::FieldBits::AllFields);
    }
}

osg::NodePtr MaterialPool::getMaterialGroupRoot()
{   
    if(getRoot() != NullFC && getRoot()->getNChildren() > 0)
    {
        return getRoot()->getChild(0);
    }
    return NullFC;
}

void MaterialPool::initLightRoot()
{
    if(getLightRoot() == NullFC)
    {
        NodePtr root = Node::create();
        osg::setName(root, _light_root_name, true);
        beginEditCP(root, Node::CoreFieldMask | Node::FlagsFieldMask);
        root->setCore(Group::create());
        root->setFlags(Node::NFModuleGraphRoot);
        endEditCP(root, Node::CoreFieldMask | Node::FlagsFieldMask);

        MaterialPoolPtr tmp(*this);
        beginEditCP(tmp, LightRootFieldMask);
        tmp->setLightRoot(root);
        endEditCP(tmp, LightRootFieldMask);
    }
}

void MaterialPool::initLightSetsRoot()
{
    if(getLightSetsRoot() == NullFC)
    {
        NodePtr lightSetsRoot = Node::create();
        osg::setName(lightSetsRoot, _lightsets_root_name, true);

        beginEditCP(lightSetsRoot, Node::CoreFieldMask | Node::FlagsFieldMask);
        lightSetsRoot->setCore(Group::create());
        lightSetsRoot->setFlags(Node::NFModuleGraphRoot);
        endEditCP(lightSetsRoot, Node::CoreFieldMask | Node::FlagsFieldMask);

        MaterialPoolPtr tmp(*this);
        beginEditCP(tmp, MaterialPool::LightSetsRootFieldMask);
        tmp->setLightSetsRoot(lightSetsRoot);
        endEditCP(tmp, MaterialPool::LightSetsRootFieldMask);
    }
}

void MaterialPool::initOverlayRoot()
{
    if(getOverlayRoot() == NullFC)
    {
        NodePtr root = Node::create();
        osg::setName(root, _overlay_root_name, true);
        beginEditCP(root, Node::CoreFieldMask | Node::FlagsFieldMask);
        root->setCore(Group::create());
        root->setFlags(Node::NFModuleGraphRoot);
        endEditCP(root, Node::CoreFieldMask | Node::FlagsFieldMask);

        MaterialPoolPtr tmp(*this);
        beginEditCP(tmp, OverlayRootFieldMask);
        tmp->setOverlayRoot(root);
        endEditCP(tmp, OverlayRootFieldMask);
    }
}

void MaterialPool::initSceneplateRoot()
{
    if (getSceneplateRoot() == NullFC)
    {
        NodePtr root = Node::create();
        osg::setName(root, _sceneplate_root_name, true);
        beginEditCP(root, Node::CoreFieldMask | Node::FlagsFieldMask);
        root->setCore(Group::create());
        root->setFlags(Node::NFModuleGraphRoot);
        endEditCP(root, Node::CoreFieldMask | Node::FlagsFieldMask);

        MaterialPoolPtr tmp(*this);
        beginEditCP(tmp, SceneplateRootFieldMask);
        tmp->setSceneplateRoot(root);
        endEditCP(tmp, SceneplateRootFieldMask);
    }
}

void MaterialPool::initPhysicsRoot()
{
    if (getPhysicsRoot() == NullFC)
    {
        NodePtr root = Node::create();
        osg::setName(root, _physics_root_name, true);
        beginEditCP(root, Node::CoreFieldMask | Node::FlagsFieldMask);
        root->setCore(osg::dcast<osg::NodeCore>(osg::FieldContainerFactory::the()->createFieldContainer("PhysicsGroup")));
        root->setFlags(Node::NFModuleGraphRoot);
        endEditCP(root, Node::CoreFieldMask | Node::FlagsFieldMask);

        MaterialPoolPtr tmp(*this);
        beginEditCP(tmp, PhysicsRootFieldMask);
        tmp->setPhysicsRoot(root);
        endEditCP(tmp, PhysicsRootFieldMask);
    }
}

void MaterialPool::initConstraintsRoot()
{
    if( getConstraintsRoot() == NullFC )
    {
        NodePtr root = Node::create();
        osg::setName(root, _constraint_root_name, true);
        beginEditCP(root, Node::CoreFieldMask | Node::FlagsFieldMask);
            root->setCore(Group::create());
            root->setFlags(Node::NFModuleGraphRoot);
        endEditCP(root, Node::CoreFieldMask | Node::FlagsFieldMask);

        MaterialPoolPtr tmp(*this);
        beginEditCP(tmp, ConstraintsRootFieldMask);
            tmp->setConstraintsRoot(root);
        endEditCP(tmp, ConstraintsRootFieldMask);
    }
}

void MaterialPool::initCameraRoot()
{
    if( getCameraRoot() == NullFC )
    {
        NodePtr root = Node::create();
        osg::setName(root, _camera_root_name, true);
        beginEditCP(root, Node::CoreFieldMask | Node::FlagsFieldMask);
            root->setCore(Group::create());
            root->setFlags(Node::NFModuleGraphRoot);
        endEditCP(root, Node::CoreFieldMask | Node::FlagsFieldMask);

        MaterialPoolPtr tmp(*this);
        beginEditCP(tmp, CameraRootFieldMask);
            tmp->setCameraRoot(root);
        endEditCP(tmp, CameraRootFieldMask);
    }
}

void MaterialPool::initAnnotationRoot()
{
    if (getAnnotationRoot() == NullFC)
    {
        NodePtr root = Node::create();
        osg::setName(root, _annotation_root_name, true);
        beginEditCP(root, Node::CoreFieldMask | Node::FlagsFieldMask);
        root->setCore(Group::create());
        root->setFlags(Node::NFModuleGraphRoot);
        endEditCP(root, Node::CoreFieldMask | Node::FlagsFieldMask);

        MaterialPoolPtr tmp(*this);
        beginEditCP(tmp, AnnotationRootFieldMask);
        tmp->setAnnotationRoot(root);
        endEditCP(tmp, AnnotationRootFieldMask);
    }

}

void MaterialPool::initResourceDatabase()
{
    if (getResourceDatabase() == NullFC)
    {
        MaterialPoolPtr tmp(*this);
        beginEditCP(tmp, ResourceDatabaseFieldMask);
        tmp->setResourceDatabase(Database::create());
        endEditCP(tmp, ResourceDatabaseFieldMask);
    }
}

void MaterialPool::removeInvalidMaterialGroups()
{
    if(getRoot() == NullFC)
        return;

    // custom stack for traversal
    std::vector<NodePtr> localStack;
    localStack.reserve(32);
    localStack.push_back(getRoot());
    
    // temporary list of nodes to delete
    std::vector<NodePtr> deleteNodes;
    deleteNodes.reserve(32);
    while(!localStack.empty())
    {
        NodePtr node = localStack.back();
        localStack.pop_back();
        if(node == NullFC)
            continue;

        for(UInt32 i=0;i<node->getNChildren();++i)
        {
            NodePtr child = node->getChild(i);
            if(child == NullFC)
                continue;

            MaterialGroupPtr mg = MaterialGroupPtr::dcast(child->getCore());
            if(mg != NullFC)
            {
                MaterialPtr mat = mg->getMaterial();
                if(mat == NullFC)
                {
                    deleteNodes.push_back(child);
                }
            }
            else
            {
                if(child->getNChildren() > 0)
                {
                    localStack.push_back(child);
                }
            }
        }
        for(auto &child : deleteNodes)
        {
            beginEditCP(node, Node::ChildrenFieldMask);
                node->subChild(child);
            endEditCP(node, Node::ChildrenFieldMask);
        }
        deleteNodes.clear();
    }
}
