/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *               Copyright (C) 2000-2002 by the OpenSG Forum                 *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

#include <stdlib.h>
#include <stdio.h>
#include <sstream>

#include <OSGConfig.h>
#include <OSGDrawAction.h>
#include <OSGRenderAction.h>
#include <OSGIntersectAction.h>
#include <OSGStringAttributeMap.h>

#include "OSGTransform3D.h"

#include <OSGTransformLimits.h>
#include <OSGCast.h>

#include <OSGSceneFileHandler.h>

//#define USE_MAYA_TRANSFORM

OSG_BEGIN_NAMESPACE

/***************************************************************************\
 *                            Description                                  *
\***************************************************************************/

/*! \class osg::Transform3D
Holds values for translation, rotation, scale, shearing and pivots. Constructs the finals transformation matrix out of the values.
*/

/***************************************************************************\
 *                           Class variables                               *
\***************************************************************************/

/***************************************************************************\
 *                           Class methods                                 *
\***************************************************************************/

void Transform3D::initMethod (void)
{

    IntersectAction::registerEnterDefault(getClassType(),
                                     std::bind(reinterpret_cast<Action::Callback>(&Transform3D::intersectEnter),
                                               std::placeholders::_1, std::placeholders::_2));
    IntersectAction::registerLeaveDefault(getClassType(),
                                     std::bind(reinterpret_cast<Action::Callback>(&Transform3D::intersectLeave),
                                               std::placeholders::_1, std::placeholders::_2));
}

 BitVector Transform3D::getCombinedTransformAttributesMask()
 {
     return (Transform3D::MatrixFieldMask | Transform3D::IsIdentityFieldMask | Transform3D::TranslationFieldMask |
         Transform3D::EulerRotationFieldMask | Transform3D::ScaleFieldMask | Transform3D::RotationFieldMask |
         Transform3D::ShearFieldMask | Transform3D::ScalePivotFieldMask | Transform3D::ScalePivotTranslationFieldMask |
         Transform3D::RotatePivotFieldMask | Transform3D::RotatePivotTranslationFieldMask | Transform3D::RotationOrientationFieldMask |
         Transform3D::RotationModeFieldMask | Transform3D::VersionFieldMask |
         Transform3D::EulerRotationOrientationFieldMask | Transform3D::EulerRotationOrderFieldMask | Transform3D::FlagsFieldMask);
 }


/***************************************************************************\
 *                           Instance methods                              *
\***************************************************************************/

/*-------------------------------------------------------------------------*\
 -  private                                                                 -
\*-------------------------------------------------------------------------*/

/*----------------------- constructors & destructors ----------------------*/

Transform3D::Transform3D(void) :
    Inherited()
{
    _sfScale.getValue().setValues(1.f, 1.f, 1.f);
    _sfEulerRotationOrder.setValue("xyz");
}

Transform3D::Transform3D(const Transform3D &source) :
    Inherited(source)
{
}

Transform3D::~Transform3D(void)
{
    clearRefCP(_sfTransformLimits.getValue());
}

/*----------------------------- class specific ----------------------------*/

void Transform3D::changed(BitVector whichField, UInt32 origin)
{
    auto limits = _sfTransformLimits.getValue();

    const bool limitsChanged = (whichField & Transform3D::TransformLimitsFieldMask);

    if (limits != osg::NullFC)
    {
        if (limitsChanged || (whichField & Transform3D::TranslationFieldMask))
        {
            osg::Vec3f translation = getTranslation();
            if( limits->limitTranslation(translation))
            {
                osg::Transform3DPtr t3d(*this);
                if( t3d != NullFC )
                {
                    beginEditCP(t3d, Transform3D::TranslationFieldMask);
                    t3d->setTranslation(translation);
                    endEditCP(t3d, Transform3D::TranslationFieldMask);
                }
            }
        }
        // euler and quaternin must always be changed together
        if (limitsChanged || (whichField & Transform3D::EulerRotationFieldMask))
        {
            osg::Vec3f eulerRotation = getEulerRotation();
            if( limits->limitEulerRotation(eulerRotation))
            {
                osg::Transform3DPtr t3d(*this);
                if( t3d != NullFC )
                {
                    beginEditCP(t3d, Transform3D::EulerRotationFieldMask|Transform3D::RotationFieldMask);
                    t3d->setEulerRotation(eulerRotation);
                    t3d->updateQuatByEuler();
                    endEditCP(t3d, Transform3D::EulerRotationFieldMask|Transform3D::RotationFieldMask);
                }
            }
        }
        if (limitsChanged || (whichField & Transform3D::ScaleFieldMask))
        {
            osg::Vec3f scaling = getScale();
            if( limits->limitScaling(scaling))
            {
                osg::Transform3DPtr t3d(*this);
                if( t3d != NullFC )
                {
                    beginEditCP(t3d, Transform3D::ScaleFieldMask);
                    t3d->setScale(scaling);
                    endEditCP(t3d, Transform3D::ScaleFieldMask);
                }
            }
        }
    }

    // to avoid unnecessary changes and volume invalidations we need to only change the
    // matrix if it needs an update. If the matrix field is already set we assume
    // it is the correct one already
    static BitVector TransformAttributesMask = (~(Transform3D::MatrixFieldMask|Transform3D::IsIdentityFieldMask)) & Transform3D::getCombinedTransformAttributesMask();
    if( limitsChanged || (whichField == FieldBits::AllFields) || ((whichField & TransformAttributesMask) && !(whichField & Inherited::MatrixFieldMask)))
    {
        Matrix final = computeFinalMatrix();
        _sfMatrix.setValue(final);
        whichField |= MatrixFieldMask; // need to add the field so the volume is correctly invalidated
    }
    Inherited::changed(whichField, origin);
}

void Transform3D::addFlag(UInt64 flag)
{
    _sfFlags.setValue(_sfFlags.getValue() | flag);
}

void Transform3D::subFlag(UInt64 flag)
{
    _sfFlags.setValue(_sfFlags.getValue() & ~flag);
}

bool Transform3D::hasFlag(UInt64 flag) const
{
    return (_sfFlags.getValue() & flag) != 0;
}

void Transform3D::onCreate(const FieldContainer *source)
{
    Inherited::onCreate(source);
    
    // if we're in startup this is the prototype, which shouldn't have an id
    if (GlobalSystemState == Startup)
        return;

    Transform3DPtr temp(this);
    if (temp != NullFC)
    {
        // Note. Setting the current version on creating a new object and
        // not doing it on loading the object
        if(!SceneFileHandler::the().isReading())
            temp->setVersion(CURRENT_TRANSFORM3D_VERSION);
    }
}

Matrix Transform3D::computeFinalMatrix()
{
    return computeFinalMatrix(
                getTranslation(),
                getScale(),
                getRotation(),
                getRotationOrientation(),
                getRotatePivot(),
                getRotatePivotTranslation(),
                getScalePivot(),
                getScalePivotTranslation(),
                getShear(),
                _sfRotationMode.getValue());
}

Matrix Transform3D::computeFinalMatrix(
        const Vec3f &translation,
        const Vec3f &scale,
        const Quaternion &rotation,
        const Quaternion &rotationOrientation,
        const Pnt3f &rotatePivot,
        const Vec3f &rotatePivotTranslation,
        const Pnt3f &scalePivot,
        const Vec3f &scalePivotTranslation,
        const Vec3f &shear,
        const UInt32 &rotationMode)
{
    Matrix final;
    final.setIdentity();

    Matrix r, ro, s, t, sh, sp, spInv, spt, rp, rpInv, rpt;
    t.setTranslate(translation);
    s.setScale(scale);
    rotation.getValue(r);
    rotationOrientation.getValue(ro);
   
    rp.setTranslate(rotatePivot);
    rpInv.setTranslate( osg::Pnt3f(-rotatePivot[0], -rotatePivot[1], -rotatePivot[2]));
    rpt.setTranslate(rotatePivotTranslation);

    sp.setTranslate(scalePivot);
    spInv.setTranslate( osg::Pnt3f(-scalePivot[0], -scalePivot[1], -scalePivot[2]));
    spt.setTranslate(scalePivotTranslation);
   
    sh.setValue(1.0, shear.x(), shear.y(), 0.0,
                 0.0, 1.0, shear.z(), 0.0,
                 0.0, 0.0, 1.0, 0.0,
                 0.0, 0.0, 0.0, 1.0);

    final.mult(t);
    final.mult(rpt);
    final.mult(rp);
    final.mult(r);
    // Maya compatibility mode
    if (rotationMode == kMayaRotationMode)
        final.mult(ro);

    final.mult(rpInv);
    final.mult(spt);
    final.mult(sp);
    final.mult(sh);
    final.mult(s);
    final.mult(spInv);

    return final;
}

Matrix Transform3D::getPivotMatrix() const
{
    Matrix r, ro, t, rp, rpInv, rpt;
    t.setTranslate(getTranslation());
    getRotation().getValue(r);
    getRotationOrientation().getValue(ro);
    rp.setTranslate(getRotatePivot());
    rp.inverse(rpInv);
    rpt.setTranslate(getRotatePivotTranslation());

    Matrix m = t;
    m.mult(rpt);
    m.mult(rp);
    m.mult(r);
    m.mult(ro);
    m.mult(rpInv);

    return m;
}

void Transform3D::updateVersion()
{
    if (getVersion() >= CURRENT_TRANSFORM3D_VERSION)
        return;
    
    Transform3DPtr temp(*this);
    beginEditCP(temp, Transform3D::VersionFieldMask);
    temp->setVersion(CURRENT_TRANSFORM3D_VERSION);
    endEditCP(temp, Transform3D::VersionFieldMask);
}

void Transform3D::reset()
{
    _sfTranslation.setValue(Vec3f());
    _sfRotation.setValue(Quaternion());
    _sfScale.setValue(Vec3f(1.0, 1.0, 1.0));
    _sfRotatePivot.setValue(Pnt3f());
    _sfScalePivot.setValue(Pnt3f());
    _sfRotationOrientation.setValue(Quaternion());
    updateRotationDirectionVector();
    _sfShear.setValue(Vec3f());
    _sfRotatePivotTranslation.setValue(Vec3f());
    _sfScalePivotTranslation.setValue(Vec3f());
    _sfEulerRotation.setValue(Vec3f());
    _sfEulerRotationOrientation.setValue(Vec3f());
    _sfEulerRotationOrder.setValue("xyz");

    // ensure the matrix is identity
    Matrix m;
    m.setIdentity();
    _sfMatrix.setValue(m);
    _sfIsIdentity.setValue(true);
}

void Transform3D::updateQuatByEuler(void)
{
    Quaternion q = convertEulerToQuat(_sfEulerRotation.getValue(), true);
    _sfRotation.setValue(q);
}

void Transform3D::updateQuatOrientByEuler(void)
{
    Quaternion q = convertEulerToQuat(_sfEulerRotationOrientation.getValue(), false);
    _sfRotationOrientation.setValue(q);
    updateRotationDirectionVector();
}

Quaternion Transform3D::convertEulerToQuat(const Vec3f& euler, bool addOrientation)
{
    Quaternion q;
    q.setValueAsEuler(euler, _sfEulerRotationOrder.getValue(), addOrientation && (_sfRotationMode.getValue() != kMayaRotationMode) ? _sfRotationOrientation.getValue() : Quaternion(), _sfRotationMode.getValue() == Transform3D::kLegacyRotationMode);
    return q;
}

void Transform3D::updateEulerByQuat(void)
{
    Vec3f euler = convertQuatToEuler(_sfRotation.getValue(), true);
    _sfEulerRotation.setValue(euler);
}

void Transform3D::updateEulerByQuatOrient(void)
{
    Vec3f euler = convertQuatToEuler(_sfRotationOrientation.getValue(), false);
    _sfEulerRotationOrientation.setValue(euler);
}

Vec3f Transform3D::convertQuatToEuler(const Quaternion& value, bool addOrientation)
{
    Vec3f euler;
    value.getValueAsEuler(euler, _sfEulerRotationOrder.getValue(), addOrientation && (_sfRotationMode.getValue() != kMayaRotationMode) ? _sfRotationOrientation.getValue() : Quaternion(), _sfRotationMode.getValue() == kLegacyRotationMode);
    return euler;
}

static bool setRotationDirectionValuesInAttachment(const Transform3DPtr &trans, const Vec3f &dir, const Int32 &axis, bool createAttachmentIfNotExisting = false)
{
    if (trans == NullFC)
        return false;

    StringAttributeMapPtr sam = StringAttributeMapPtr::dcast(trans->findAttachment(StringAttributeMap::getClassType()));
    if (sam == NullFC && createAttachmentIfNotExisting)
    {
        sam = StringAttributeMap::create();
        beginEditCP(trans, Transform3D::AttachmentsFieldMask);
            trans->addAttachment(sam);
        endEditCP(trans, Transform3D::AttachmentsFieldMask);
    }

    if (sam != NullFC)
    {
        std::stringstream ss(std::stringstream::in | std::stringstream::out);
        ss << dir[0] << " " << dir[1] << " " << dir[2] << " " << axis;

        beginEditCP(sam, StringAttributeMap::KeysFieldMask);
            sam->setAttribute("rotationDirection", ss.str());
        endEditCP(sam, StringAttributeMap::KeysFieldMask);

        return true;
    }

    return false;
}


static Vec3f calculateRotationDirection(const Quaternion &rot, UInt32 axis, Real32 length)
{
    if (length <= 0.0f)
        length = 1.0f;

    Vec3f dir(length, 0.0f, 0.0f); // axis should be 0
    if (axis == 1)
        dir = Vec3f(0.0f, length, 0.0f);
    else if (axis == 2)
        dir = Vec3f(0.0f, 0.0f, length);

    Matrix rotMat;
    rotMat.setRotate(rot);
    rotMat.multMatrixVec(dir, dir);

    return dir;
}

static bool getRotationDirectionValuesFromAttachment(const Transform3DPtr &trans, Vec3f &dir, Int32 &axis)
{
    if (trans == NullFC)
        return false;

    StringAttributeMapPtr sam = StringAttributeMapPtr::dcast(trans->findAttachment(StringAttributeMap::getClassType()));
    if (sam != NullFC)
    {
        std::string value;
        if (sam->hasAttribute("rotationDirection") && sam->getAttribute("rotationDirection", value))
        {
            std::stringstream ss(std::stringstream::in | std::stringstream::out);
            ss << value;
            ss >> dir[0] >> dir[1] >> dir[2] >> axis;

            return true;
        }
    }

    return false;
}

bool Transform3D::getRotationDirectionValues(Vec3f &dir, Int32 &axis)
{
    return getRotationDirectionValuesFromAttachment(Transform3DPtr(*this), dir, axis);
}

void Transform3D::resetRotationDirection()
{
    StringAttributeMapPtr attachment = StringAttributeMapPtr::dcast(findAttachment(StringAttributeMap::getClassType()));
    if (attachment != NullFC)
    {
        beginEditCP(attachment, StringAttributeMap::KeysFieldMask | osg::StringAttributeMap::ValuesFieldMask);
        attachment->removeAttribute("rotationDirection");
        endEditCP(attachment, StringAttributeMap::KeysFieldMask | osg::StringAttributeMap::ValuesFieldMask);

        if (attachment->getKeys().getSize() == 0)
        {
            Transform3DPtr temp(*this);
            beginEditCP(temp, AttachmentContainer::AttachmentsFieldMask);
            subAttachment(attachment);
            endEditCP(temp, AttachmentContainer::AttachmentsFieldMask);
        }
    }
}
void Transform3D::setRotationDirection(const Vec3f &dir, Int32 axis)
{
    setRotationDirectionValuesInAttachment(Transform3DPtr(*this), dir, axis, true);
}

Vec3f Transform3D::getRotationDirection(Int32 axis)
{
    Vec3f dir(1.0f, 0.0f, 0.0f);

    Int32 currentAxis = 0;
    if (getRotationDirectionValuesFromAttachment(Transform3DPtr(*this), dir, currentAxis))
    {
        if (currentAxis != axis)
            dir = calculateRotationDirection(getRotationOrientation(), axis, dir.length());
    }
    else // fallback: no attachment, compute values on the fly
    {
        dir = calculateRotationDirection(getRotationOrientation(), axis, 1.0f);
    }

    return dir;
}

Vec3f Transform3D::updateRotationDirection(UInt32 axis, Real32 length)
{
    if (length <= 0.0f)
        length = 1.0f;

    Vec3f dir = calculateRotationDirection(getRotationOrientation(), axis, length);

    setRotationDirectionValuesInAttachment(Transform3DPtr(*this), dir, axis);

    return dir;
}

void Transform3D::updateRotationDirectionVector()
{
    Vec3f dir;
    Int32 axis;

    if (getRotationDirectionValuesFromAttachment(Transform3DPtr(*this), dir, axis))
        updateRotationDirection(axis, dir.length());
}

bool Transform3D::equals( const Transform3DPtr& other, const float tolerance) const
{
    if( other == NullFC)
        return false;

    bool equal = true;
    equal &= getTranslation().equals(other->getTranslation(), tolerance);
    equal &= getScale().equals(other->getScale(), tolerance);
    equal &= getRotation().equals(other->getRotation(), tolerance);
    equal &= getRotationOrientation().equals(other->getRotationOrientation(), tolerance);
    equal &= getRotatePivot().equals(other->getRotatePivot(), tolerance);
    equal &= getRotatePivotTranslation().equals(other->getRotatePivotTranslation(), tolerance);
    equal &= getScalePivot().equals(other->getScalePivot(), tolerance);
    equal &= getScalePivotTranslation().equals(other->getScalePivotTranslation(), tolerance);
    equal &= getShear().equals(other->getShear(), tolerance);
    equal &= (getRotationMode() == other->getRotationMode()) ? true : false;
    return equal;
}

void Transform3D::dump(      UInt32    ,
                         const BitVector ) const
{
    SLOG << "Dump Transform3D NI" << std::endl;
}


void Transform3D::syncWith(const Transform3DPtr& pOther, const BitVector& whichField)
{
    executeSyncImpl(pOther.getCPtr(), whichField);
}

/*------------------------------------------------------------------------*/
/*                              cvs id's                                  */

#ifdef OSG_SGI_CC
#pragma set woff 1174
#endif

#ifdef OSG_LINUX_ICC
#pragma warning( disable : 177 )
#endif

namespace
{
    static Char8 cvsid_cpp       [] = "@(#)$Id: FCTemplate_cpp.h,v 1.20 2006/03/16 17:01:53 dirk Exp $";
    static Char8 cvsid_hpp       [] = OSGTRANSFORM3DBASE_HEADER_CVSID;
    static Char8 cvsid_inl       [] = OSGTRANSFORM3DBASE_INLINE_CVSID;

    static Char8 cvsid_fields_hpp[] = OSGTRANSFORM3DFIELDS_HEADER_CVSID;
}

#ifdef __sgi
#pragma reset woff 1174
#endif

OSG_END_NAMESPACE

