/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *               Copyright (C) 2000-2002 by the OpenSG Forum                 *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

#include <stdlib.h>
#include <stdio.h>

#include <OSGConfig.h>
#include <OSGMatrix.h>
#include <OSGQuaternion.h>

#include "OSGTransformLimits.h"

OSG_BEGIN_NAMESPACE

/***************************************************************************\
 *                            Description                                  *
\***************************************************************************/

/*! \class osg::TransformLimits
Class for storing object transformation limits
*/

/***************************************************************************\
 *                           Class variables                               *
\***************************************************************************/

TransformLimits::LimitsChangedCallback TransformLimits::s_limitsChangedCallback = nullptr;

/***************************************************************************\
 *                           Class methods                                 *
\***************************************************************************/

void TransformLimits::initMethod(void)
{
}

void TransformLimits::setLimitsChangedCallback(LimitsChangedCallback callback)
{
    s_limitsChangedCallback = callback;
}

/***************************************************************************\
 *                           Instance methods                              *
\***************************************************************************/

/*-------------------------------------------------------------------------*\
 -  private                                                                 -
\*-------------------------------------------------------------------------*/

/*----------------------- constructors & destructors ----------------------*/

TransformLimits::TransformLimits(void)
    : Inherited()
{
}

TransformLimits::TransformLimits(const TransformLimits &source)
    : Inherited(source)
{
}

TransformLimits::~TransformLimits(void)
{
    if (s_limitsChangedCallback != nullptr)
        s_limitsChangedCallback();
}

/*----------------------------- class specific ----------------------------*/

void TransformLimits::changed(BitVector whichField, UInt32 origin)
{
    Inherited::changed(whichField, origin);
    
    // Call the registered callback if one exists
    if (s_limitsChangedCallback != nullptr)
        s_limitsChangedCallback();
}

void TransformLimits::dump(UInt32,
                                      const BitVector) const
{
    SLOG << "Dump TransformLimits NI" << std::endl;
}

/***************************************************************************\
 *                            Limit Transform                              *
\***************************************************************************/

#if 0
void TransformLimits::limitNodeTransformation(osg::NodePtr const &node)
{
    if (hasRotationLimits())
    {
        if (_sfUseWorldSpaceRotationLimits.getValue())
        {
            NodePtr parentNode = node->getParent();
            if (parentNode == NullFC)
                return;

            osg::Matrix parentToWorldInv = parentNode->getToWorld();
            parentToWorldInv.invert();
            Vec3f pt, ps;
            Quaternion pr, psr;
            parentToWorldInv.getTransform(pt, pr, ps, psr);

            const osg::Matrix toWorld = node->getToWorld();
            Vec3f t, s;
            Quaternion r, sr;
            toWorld.getTransform(t, r, s, sr);

            parentToWorldInv.multMatrixPnt(s, s);

            NodeUtils::changeTransform3D(node, [this, &r, &pr](const osg::NodePtr &, const osg::Transform3DPtr &t3d)
                                         {
                                             osg::Vec3f eulerRotation;
                                             r.getValueAsEuler(eulerRotation, t3d->getEulerRotationOrder());
                                             limitEulerRotation(eulerRotation);

                                             r.setValueAsEuler(eulerRotation, t3d->getEulerRotationOrder());

                                             r.multLeft(pr);

                                             beginEditCP(t3d, Transform3D::EulerRotationFieldMask);
                                             t3d->setRotationAsQuaternion(r);
                                             // t3d->setRotationAsEuler(eulerRotation);
                                             endEditCP(t3d, Transform3D::EulerRotationFieldMask);
                                         });
        }
        else // object
        {
            NodeUtils::changeTransform3D(node, [this](const osg::NodePtr &, const osg::Transform3DPtr &t3d)
                                         {
                                             osg::Vec3f eulerRotation = t3d->getEulerRotation();
                                             limitEulerRotation(eulerRotation);

                                             beginEditCP(t3d, Transform3D::EulerRotationFieldMask);
                                             t3d->setRotationAsEuler(eulerRotation);
                                             endEditCP(t3d, Transform3D::EulerRotationFieldMask);
                                         });
        }
    }

    if (hasScalingLimits())
    {
        if (_sfUseWorldSpaceScalingLimits.getValue())
        {
            NodePtr parentNode = node->getParent();
            if (parentNode == NullFC)
                return;

            osg::Matrix parentToWorldInv = parentNode->getToWorld();
            parentToWorldInv.invert();

            const osg::Matrix toWorld = node->getToWorld();
            Vec3f t, s;
            Quaternion r, sr;
            toWorld.getTransform(t, r, s, sr);
            limitScaling(s);

            parentToWorldInv.multMatrixPnt(s, s);

            NodeUtils::changeTransform3D(node, [&s](const osg::NodePtr &, const osg::Transform3DPtr &t3d)
                                         {
                                             beginEditCP(t3d, Transform3D::ScaleFieldMask);
                                             t3d->setScale(s);
                                             endEditCP(t3d, Transform3D::ScaleFieldMask);
                                         });
        }
        else // object
        {
            NodeUtils::changeTransform3D(node, [this](const osg::NodePtr &, const osg::Transform3DPtr &t3d)
                                         {
                                             osg::Vec3f s = t3d->getScale();
                                             // fprintf(stderr, "scale before: %f %f %f\n", s[0], s[1], s[2]);
                                             limitScaling(s);
                                             // fprintf(stderr, "scale after: %f %f %f\n", s[0], s[1], s[2]);
                                             beginEditCP(t3d, Transform3D::ScaleFieldMask);
                                             t3d->setScale(s);
                                             endEditCP(t3d, Transform3D::ScaleFieldMask);
                                         });
        }
    }

    if (hasTranslationLimits())
    {
        if (_sfUseWorldSpaceTranslationLimits.getValue())
        {
            NodePtr parentNode = node->getParent();
            if (parentNode == NullFC)
                return;

            osg::Matrix parentToWorldInv = parentNode->getToWorld();
            parentToWorldInv.invert();

            const osg::Matrix toWorld = node->getToWorld();
            Vec3f t, s;
            Quaternion r, sr;
            toWorld.getTransform(t, r, s, sr);
            limitTranslation(t);
            // fprintf(stderr, "limited trans to world %f %f %f\n", t[0], t[1], t[2]);

            parentToWorldInv.multMatrixPnt(t, t);
            // fprintf(stderr, "limited trans to local %f %f %f\n", t[0], t[1], t[2]);

            NodeUtils::changeTransform3D(node, [&t](const osg::NodePtr &, const osg::Transform3DPtr &t3d)
                                         {
                                             beginEditCP(t3d, Transform3D::TranslationFieldMask);
                                             t3d->setTranslation(t);
                                             endEditCP(t3d, Transform3D::TranslationFieldMask);
                                         });
        }
        else // object
        {
            NodeUtils::changeTransform3D(node, [this](const osg::NodePtr &, const osg::Transform3DPtr &t3d)
                                         {
                                             osg::Vec3f translation = t3d->getTranslation();
                                             limitTranslation(translation);

                                             beginEditCP(t3d, Transform3D::TranslationFieldMask);
                                             t3d->setTranslation(translation);
                                             endEditCP(t3d, Transform3D::TranslationFieldMask);
                                         });
        }
    }
}
#endif

/***************************************************************************\
 *                           Translation                                   *
\***************************************************************************/

bool TransformLimits::hasTranslationLimits() const
{
    return _sfUseTranslationLimits.getValue() &&
           (_sfUseMaxXTranslationLimit.getValue() || _sfUseMinXTranslationLimit.getValue() ||
            _sfUseMaxYTranslationLimit.getValue() || _sfUseMinYTranslationLimit.getValue() ||
            _sfUseMaxZTranslationLimit.getValue() || _sfUseMinZTranslationLimit.getValue());
}

bool TransformLimits::limitTranslation(osg::Matrix &transform) const
{
    if (hasTranslationLimits())
    {
        Vec3f t, s;
        Quaternion r, sr;
        transform.getTransform(t, r, s, sr);
        if( limitTranslation(t))
        {
            transform.setTranslate(t);
            return true;
        }
    }
    return false;
}

bool TransformLimits::limitTranslation(osg::Vec3f &translation) const
{
    if (hasTranslationLimits())
    {
        const osg::Vec3f orgTranslation = translation;
        translation[0] = clamp(translation[0], _sfUseMinXTranslationLimit.getValue(), _sfMinXTranslationLimit.getValue(), _sfUseMaxXTranslationLimit.getValue(), _sfMaxXTranslationLimit.getValue());
        translation[1] = clamp(translation[1], _sfUseMinYTranslationLimit.getValue(), _sfMinYTranslationLimit.getValue(), _sfUseMaxYTranslationLimit.getValue(), _sfMaxYTranslationLimit.getValue());
        translation[2] = clamp(translation[2], _sfUseMinZTranslationLimit.getValue(), _sfMinZTranslationLimit.getValue(), _sfUseMaxZTranslationLimit.getValue(), _sfMaxZTranslationLimit.getValue());
        return (orgTranslation != translation);
    }
    return false;
}

bool TransformLimits::limitScaling(osg::Vec3f &scaling) const
{
    if (hasScalingLimits())
    {
        const osg::Vec3f orgScaling = scaling;
        scaling[0] = clamp(scaling[0], _sfUseMinXScalingLimit.getValue(), _sfMinXScalingLimit.getValue(), _sfUseMaxXScalingLimit.getValue(), _sfMaxXScalingLimit.getValue());
        scaling[1] = clamp(scaling[1], _sfUseMinYScalingLimit.getValue(), _sfMinYScalingLimit.getValue(), _sfUseMaxYScalingLimit.getValue(), _sfMaxYScalingLimit.getValue());
        scaling[2] = clamp(scaling[2], _sfUseMinZScalingLimit.getValue(), _sfMinZScalingLimit.getValue(), _sfUseMaxZScalingLimit.getValue(), _sfMaxZScalingLimit.getValue());
        return (orgScaling != scaling);
    }
    return false;
}

bool TransformLimits::hasRotationLimits() const
{
    return _sfUseEulerRotationLimits.getValue() &&
           (_sfUseMaxXEulerRotationLimit.getValue() || _sfUseMinXEulerRotationLimit.getValue() ||
            _sfUseMaxYEulerRotationLimit.getValue() || _sfUseMinYEulerRotationLimit.getValue() ||
            _sfUseMaxZEulerRotationLimit.getValue() || _sfUseMinZEulerRotationLimit.getValue());
}

bool TransformLimits::hasScalingLimits() const
{
    return _sfUseScalingLimits.getValue() &&
           (_sfUseMaxXScalingLimit.getValue() || _sfUseMinXScalingLimit.getValue() ||
            _sfUseMaxYScalingLimit.getValue() || _sfUseMinYScalingLimit.getValue() ||
            _sfUseMaxZScalingLimit.getValue() || _sfUseMinZScalingLimit.getValue());
}

float TransformLimits::limitRotation(float value, bool useMin, float min, bool useMax, float max) const
{
    if (!useMin && !useMax)
        return value;

    // adjust the values to be in the range [0, 360]
    min = std::fmodf(min, 360.f);
    max = std::fmodf(max, 360.f);
    value = std::fmodf(value + 360.f, 360.f);

    // calculate relative angle between min and max
    const auto range = (min > max) ? (360 - min + max) : (max - min);

    // inverted range [min; 360] + [0; max]
    if (min > max)
    {
        // adjust the value to be in the range [min, 360]
        if (value > max)
        {
            value = std::clamp(value, min, 360.f);
        }
        else // adjust the value to be in the range [0, max]
        {
            value = std::clamp(value, 0.f, max);
        }
    }
    else // normal range [min; max]
    {
        value = std::clamp(value, min, min + range);
    }

    return value;
}

bool TransformLimits::limitEulerRotation(osg::Vec3f &euler) const
{
    if (hasRotationLimits())
    {
        const osg::Vec3f orgEuler = euler;
        euler[0] = clamp(euler[0], _sfUseMinXEulerRotationLimit.getValue(), _sfMinXEulerRotationLimit.getValue(), _sfUseMaxXEulerRotationLimit.getValue(), _sfMaxXEulerRotationLimit.getValue());
        euler[1] = clamp(euler[1], _sfUseMinYEulerRotationLimit.getValue(), _sfMinYEulerRotationLimit.getValue(), _sfUseMaxYEulerRotationLimit.getValue(), _sfMaxYEulerRotationLimit.getValue());
        euler[2] = clamp(euler[2], _sfUseMinZEulerRotationLimit.getValue(), _sfMinZEulerRotationLimit.getValue(), _sfUseMaxZEulerRotationLimit.getValue(), _sfMaxZEulerRotationLimit.getValue());
        return (orgEuler != euler);
    }
    return false;
}

float TransformLimits::clamp(float value, bool useMin, float min, bool useMax, float max) const
{
    if (useMax)
        value = std::min(value, max);

    if (useMin)
        value = std::max(value, min);

    return value;
}

/*------------------------------------------------------------------------*/
/*                              cvs id's                                  */

#ifdef OSG_SGI_CC
#pragma set woff 1174
#endif

#ifdef OSG_LINUX_ICC
#pragma warning(disable : 177)
#endif

namespace
{
static Char8 cvsid_cpp[] = "@(#)$Id: FCTemplate_cpp.h,v 1.20 2006/03/16 17:01:53 dirk Exp $";
static Char8 cvsid_hpp[] = OSGTRANSFORMLIMITSBASE_HEADER_CVSID;
static Char8 cvsid_inl[] = OSGTRANSFORMLIMITSBASE_INLINE_CVSID;

static Char8 cvsid_fields_hpp[] = OSGTRANSFORMLIMITSFIELDS_HEADER_CVSID;
} // namespace

#ifdef __sgi
#pragma reset woff 1174
#endif

OSG_END_NAMESPACE
