/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *               Copyright (C) 2000-2002 by the OpenSG Forum                 *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
 \*---------------------------------------------------------------------------*/
 /*---------------------------------------------------------------------------*\
  *                                License                                    *
  *                                                                           *
  * This library is free software; you can redistribute it and/or modify it   *
  * under the terms of the GNU Library General Public License as published    *
  * by the Free Software Foundation, version 2.                               *
  *                                                                           *
  * This library is distributed in the hope that it will be useful, but       *
  * WITHOUT ANY WARRANTY; without even the implied warranty of                *
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
  * Library General Public License for more details.                          *
  *                                                                           *
  * You should have received a copy of the GNU Library General Public         *
  * License along with this library; if not, write to the Free Software       *
  * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
  *                                                                           *
  \*---------------------------------------------------------------------------*/
  /*---------------------------------------------------------------------------*\
   *                                Changes                                    *
   *                                                                           *
   *                                                                           *
   *                                                                           *
   *                                                                           *
   *                                                                           *
   *                                                                           *
   \*---------------------------------------------------------------------------*/

   //---------------------------------------------------------------------------
   //  Includes
   //---------------------------------------------------------------------------

#include <stdlib.h>
#include <stdio.h>

#include <sstream>

#include <OSGConfig.h>

#include <OSGGL.h>
#include <OSGGLU.h>
#include <OSGGLEXT.h>
#include <OSGRemoteAspect.h>
#include <OSGCamera.h>
#include <OSGViewport.h>
#include <OSGStereoCameraDecorator.h>
#include <OSGRenderAction.h>
#include <OSGCRC32.h>
#include <OSGFileSystem.h>
#include <OSGCrypt.h>

#include <OSGShaderParameter.h>
#include <OSGShaderParameterBool.h>
#include <OSGShaderParameterInt.h>
#include <OSGShaderParameterUInt.h>
#include <OSGShaderParameterReal.h>
#include <OSGShaderParameterVec2f.h>
#include <OSGShaderParameterVec3f.h>
#include <OSGShaderParameterVec4f.h>
#include <OSGShaderParameterMatrix.h>
#include <OSGShaderParameterHandle.h>
#include <OSGShaderParameterSubroutine.h>

#include <OSGShaderParameterMInt.h>
#include <OSGShaderParameterMUInt.h>
#include <OSGShaderParameterMReal.h>
#include <OSGShaderParameterMVec2f.h>
#include <OSGShaderParameterMVec3f.h>
#include <OSGShaderParameterMVec4f.h>
#include <OSGShaderParameterMMatrix.h>
#include <OSGShaderParameterMHandle.h>

#include "OSGSHLChunk.h"
#include "OSGGLTextureUnits.h"
#include <oneapi/tbb/tick_count.h>
#ifndef BINARY_SHADER_VERSION
#define BINARY_SHADER_VERSION "V1"
#endif

#ifndef VR_RELEASE_BUILD
#define DEBUG_COMPILE_ERROR
#endif

#define USE_BINARY true
// disable warnings about missing parameters since there is no chance for us to detect this.
// The shader compiler might optimize the shader based on the uniform variables and completely omit variables that are 
// not used even though they are part of the source code. 
#define OUTPUT_PARAMETER_WARNING 0

#define VOLUME_SCATTER_AVAILABLE 1

OSG_USING_NAMESPACE

/*! \class osg::SHLChunk

*/

/***************************************************************************\
 *                           Class variables                               *
 \***************************************************************************/

    StateChunkClass SHLChunk::_class("SHL");

Int32 SHLChunk::_clusterId = -1;

SHLChunk::parametercbfp SHLChunk::_userParametersCallback = NULL;
SHLChunk::programcbfp SHLChunk::_userCreateProgramCallback = NULL;
SHLChunk::programcbfp SHLChunk::_userDeleteProgramCallback = NULL;

Int32   SHLChunk::s_uniformBlockSize = 16384; // default to the required minimum
bool    SHLChunk::s_hasStereoViewRenderingSupport = false;
bool	SHLChunk::s_hasLensMatchedShadingSupport = false;
bool	SHLChunk::s_useBindlessShadowTextures = false;
bool	SHLChunk::s_useFog = false;
bool	SHLChunk::s_useVolumeScattering = true;

bool    SHLChunk::s_isAMD = false;
bool    SHLChunk::s_isIntel = true; // default to intel gpu as the lowest common base
bool    SHLChunk::s_isNvidia = false;

osg::UInt32         SHLChunk::s_currentProgram = 0;
osg::Int32          SHLChunk::s_currentGeometryInfoUniform = -1;
// prototypes


//#define USE_BINDLESS_LIGHTTEXTURE


/***************************************************************************\
 *                           Class methods                                 *
 \***************************************************************************/

void SHLChunk::initMethod(void)
{
}


/***************************************************************************\
 *                           Instance methods                              *
 \***************************************************************************/

 /*-------------------------------------------------------------------------*\
  -  private                                                                 -
  \*-------------------------------------------------------------------------*/

  /*----------------------- constructors & destructors ----------------------*/

SHLChunk::SHLChunk(void) :
    Inherited(),
    _osgParametersCallbacks(),
    _oldParameterSize(0),
    m_vertexHash(0),
    m_geometryHash(0),
    m_fragmentHash(0),
    _uniforms(),
    m_program(0),
    m_programChanged(true),
    m_programParameterChanged(true),
    m_stereoViewRendering(false),
    m_lensMatchedShadingRendering(false),
    m_singleViewRenderingOnly(false),
    m_useBindlessShadowTextures(false),
    m_useFog(true),
    _userParameterCallbacks()
{
}

SHLChunk::SHLChunk(const SHLChunk &source) :
    Inherited(source),
    _osgParametersCallbacks(source._osgParametersCallbacks),
    _oldParameterSize(source._oldParameterSize),
    m_vertexHash(0),
    m_geometryHash(0),
    m_fragmentHash(0),
    m_program(0),
    m_programChanged(true),
    m_programParameterChanged(true),
    m_stereoViewRendering(source.m_stereoViewRendering),
    m_lensMatchedShadingRendering(source.m_lensMatchedShadingRendering),
    m_singleViewRenderingOnly(source.m_singleViewRenderingOnly),
    m_useBindlessShadowTextures(source.m_useBindlessShadowTextures),
    m_useFog(source.m_useFog),
    _userParameterCallbacks(source._userParameterCallbacks)
{
}

SHLChunk::~SHLChunk(void)
{

}

void SHLChunk::onCreate(const SHLChunk *source)
{
    Inherited::onCreate(source);

    // ignore prototypes.
    if (GlobalSystemState == Startup)
        return;

}

void SHLChunk::onDestroy(void)
{
    Inherited::onDestroy();

    if (m_program != 0)
    {
        if (osgMakeCurrent())
        {
            if (_userDeleteProgramCallback != NULL)
                _userDeleteProgramCallback(m_program);

            glDeleteProgram(m_program);
            m_program = 0;
            m_programChanged = true;
            m_programParameterChanged = true;
        }
    }
}

const StateChunkClass *SHLChunk::getClass(void) const
{
    return &_class;
}

/*----------------------------- class specific ----------------------------*/

void SHLChunk::parseUniformVariables(void)
{
    _uniforms.clear();
    std::string token;
    std::istringstream is(getVertexProgram() + getFragmentProgram() + getGeometryProgram());
    while (is.good())
    {
        is >> token;
        if (token == "uniform")
        {
            // skip type
            is >> token;
            std::string uniform;
            is >> uniform;
            // remove ;
            if (!uniform.empty())
            {
                if (uniform.substr(uniform.length() - 1) == ";")
                    uniform = uniform.substr(0, uniform.length() - 1);
            }
            _uniforms.insert(uniform);
        }
    }
}

bool SHLChunk::hasUniform(const std::string &name)
{
    return (_uniforms.count(name) > 0);
}

void SHLChunk::changed(BitVector whichField, UInt32 origin)
{

    if ((whichField & VertexProgramFieldMask) ||
        (whichField & FragmentProgramFieldMask) ||
        (whichField & GeometryProgramFieldMask))
    {
        parseUniformVariables();
        m_programChanged = true;
        m_programParameterChanged = true;
    }

    if ((whichField & ParametersFieldMask) ||
        (whichField & ProgramParameterNamesFieldMask) ||
        (whichField & ProgramParameterValuesFieldMask))
    {
        m_programParameterChanged = true;
    }

    Inherited::changed(whichField, origin);
}

void SHLChunk::dump(UInt32,
    const BitVector) const
{
    SLOG << "Dump SHLChunk NI" << std::endl;
}

static void crypt(UInt8 *binary, Int32 size)
{
    static std::vector<UInt8> key;
    if (key.empty())
    {
        key.reserve(16);
        key.push_back('6');
        key.push_back('g');
        key.push_back('a');
        key.push_back('5');
        key.push_back('d');
        key.push_back('8');
        key.push_back('b');
        key.push_back('1');
        key.push_back('e');
        key.push_back('6');
        key.push_back('f');
        key.push_back('f');
        key.push_back('9');
        key.push_back('b');
        key.push_back('a');
        key.push_back('0');
    }

    Crypt c(key);
    for (UInt32 i = 0; i < size; ++i)
        binary[i] = c.code(binary[i]);
}

bool SHLChunk::recompileRequired() const
{
    if( ( m_useBindlessShadowTextures != getUseBindlessShadowTextures()) ||
        ( m_stereoViewRendering != getStereoViewRenderingEnabled()) ||
        ( m_lensMatchedShadingRendering != getLensMatchedShadingEnabled()) ||
        ( m_useFog != getFogEnabled()))
    {
        return true;
    }
    return false;
}

void SHLChunk::updateProgram()
{
    if (!GLAD_GL_VERSION_4_0)
        return;

    if( m_program != 0 && !m_programChanged && !recompileRequired())
        return;

    auto t0 = oneapi::tbb::tick_count::now();

    // build the full shader string
    std::string vertexProg = getShaderVersionString() + getShaderDefinesString(kVertexProgram, m_singleViewRenderingOnly) + getVertexProgram();
    std::string geoProg = getShaderVersionString() + getShaderDefinesString(kGeometryProgram, m_singleViewRenderingOnly) + getGeometryProgram();
    std::string fragmentProg = getShaderVersionString() + getShaderDefinesString(kFragmentProgram, m_singleViewRenderingOnly) + getFragmentProgram();

    CRC32 crc;
    UInt32 vertex_hash = 0;
    UInt32 vShader = 0;
    OSGGLint has_vertex = 0;
    // reload programs
    if (!getVertexProgram().empty())
    {
        crc.fullCRC((const UInt8 *)vertexProg.c_str(), vertexProg.size(), &vertex_hash);
    }
    UInt32 fragment_hash = 0;
    UInt32 fShader = 0;
    OSGGLint has_fragment = 0;
    if (!getFragmentProgram().empty())
    {
        crc.fullCRC((const UInt8 *)fragmentProg.c_str(), fragmentProg.size(), &fragment_hash);
    }
    UInt32 geometry_hash = 0;
    UInt32 gShader = 0;
    OSGGLint has_geometry = 0;
    // reload programs
    if (!getGeometryProgram().empty())
    {
        crc.fullCRC((const UInt8 *)geoProg.c_str(), geoProg.size(), &geometry_hash);
    }


    if (m_program != 0)
    {
        if (m_vertexHash == vertex_hash &&
            m_fragmentHash == fragment_hash &&
            m_geometryHash == geometry_hash)
        {
            m_programChanged = false;
            return; // do nothing, everything is fine
        }

        glUseProgram(0); // remove binding
        if (_userDeleteProgramCallback != NULL)
            _userDeleteProgramCallback(m_program);

        glDeleteProgram(m_program);
        m_program = 0;
        m_programChanged = true;
    }

    m_program = glCreateProgram();

    m_vertexHash = vertex_hash;
    m_fragmentHash = fragment_hash;
    m_geometryHash = geometry_hash;

    // create hash filename.
    std::string hash_name;
    if (vertex_hash != 0)
    {
        char hash[20];
        sprintf(hash, "%X", vertex_hash);
        hash_name += hash;
    }
    if (fragment_hash != 0)
    {
        char hash[20];
        sprintf(hash, "%X", fragment_hash);
        hash_name += hash;
    }
    if (geometry_hash != 0)
    {
        char hash[20];
        sprintf(hash, "%X", geometry_hash);
        hash_name += hash;
    }
    bool loaded_binary = false;
    std::string filename;
    UInt8 *binary = NULL;
    Int32 binary_length = 0;
    Int32 binary_format = 0;

    if (USE_BINARY && GLAD_GL_VERSION_4_1) // core since 4.1
    {

        // add gl version + vendor string to get a unique name for different driver versions!
        static std::string glinfo;
        if (glinfo.empty())
        {
            if (glGetString(GL_VERSION) != NULL)
                glinfo += (const char *)glGetString(GL_VENDOR);
            if (glGetString(GL_VERSION) != NULL)
                glinfo += (const char *)glGetString(GL_VERSION);
            glinfo.erase(std::remove(glinfo.begin(), glinfo.end(), ' '), glinfo.end());
        }

        filename = Directory::getTmpPath("OSGGLSLCACHE");
        filename += OSG_DIRSEP_STR;
        filename += hash_name;
        filename += glinfo;
        filename += BINARY_SHADER_VERSION;

        FILE *f = osg::fopen(filename.c_str(), "rb");

        if (f != NULL)
        {
            fseek(f, 0, SEEK_END);
            binary_length = (Int32)ftell(f);
            binary_length -= sizeof(binary_format); // sub sizeof binary format header.
            binary = new UInt8[binary_length];
            fseek(f, 0, SEEK_SET);
            fread(&binary_format, sizeof(binary_format), 1, f);
            fread(binary, binary_length, 1, f);
            // decrypt shader.
            crypt(binary, binary_length);
            fclose(f);
        }
    }

    if (binary != NULL)
    {

        OSGGLenum binaryFormat = (OSGGLenum)binary_format;
        glProgramBinary(m_program, binaryFormat, binary, binary_length);

        delete[] binary;

        OSGGLint success;
        //glGetProgramiv(progId, GL_LINK_STATUS, &success);
        glGetProgramiv(m_program, GL_LINK_STATUS, &success);
        if (success)
        {
            if (_userCreateProgramCallback != NULL)
                _userCreateProgramCallback(m_program);
            loaded_binary = true;
        }
        else
        {
            char *debug = NULL;
            OSGGLint debugLength;
            glGetProgramiv(m_program, GL_INFO_LOG_LENGTH, &debugLength);
            if (debugLength > 0)
            {
                debug = new char[debugLength];
                debug[0] = 0;
                glGetProgramInfoLog(m_program, debugLength, &debugLength, debug);
                if (debug != NULL && debug[0] != 0)
                {
                    FWARNING(("SHLChunk: binary link status: %s\n", debug));
                }
                //else
                //{
                //    FWARNING(("SHLChunk: linked binary shader\n"));
                //}
            }
            delete[] debug;
            // remove binary shader file.
            osg::remove(filename.c_str());
        }
    }
    bool compileError = false;
    if (!loaded_binary)
    {
        // reload programs
        if (!getVertexProgram().empty() && !vertexProg.empty())
        {
            OSGGLenum shader_type = GL_VERTEX_SHADER;
            vShader = glCreateShader(shader_type);
            const char *source = vertexProg.c_str();
            glShaderSource(vShader, 1, &source, nullptr);
            
            glCompileShader(vShader);
            glGetShaderiv(vShader, GL_COMPILE_STATUS, &has_vertex);

            if (has_vertex == 0)
            {
                compileError |= true;
                char *debug = NULL;
                OSGGLint debugLength;
                glGetShaderiv(vShader, GL_INFO_LOG_LENGTH, &debugLength);

                if (debugLength > 0)
                {
                    debug = new char[debugLength];
                    debug[0] = 0;
                    glGetShaderInfoLog(vShader, debugLength, &debugLength, debug);
                }

                if (debug != NULL && debug[0] != 0)
                {
                    FFATAL(("Couldn't compile vertex program!\n%s\n", debug));
                }
                else
                {
                    FFATAL(("Couldn't compile vertex program!\n"));
                }
#ifdef DEBUG_COMPILE_ERROR
                std::ofstream outF("VertexProgramError.glsl", std::ios::out|std::ios::binary);
                outF << vertexProg << std::endl;
                outF.close();
#endif
                delete[] debug;
                glDeleteShader(vShader);
            }
        }
        if (!getFragmentProgram().empty() && !fragmentProg.empty())
        {
            OSGGLenum shader_type = GL_FRAGMENT_SHADER;
            fShader = glCreateShader(shader_type);
            const char *source = fragmentProg.c_str();
            glShaderSource(fShader, 1, &source, nullptr);

            glCompileShader(fShader);
            glGetShaderiv(fShader, GL_COMPILE_STATUS, &has_fragment);

            if (has_fragment == 0)
            {
                compileError |= true;
                char *debug = NULL;
                OSGGLint debugLength;
                glGetShaderiv(fShader, GL_INFO_LOG_LENGTH, &debugLength);

                if (debugLength > 0)
                {
                    debug = new char[debugLength];
                    debug[0] = 0;
                    glGetShaderInfoLog(fShader, debugLength, &debugLength, debug);
                }

                if (debug != NULL && debug[0] != 0)
                {
                    FFATAL(("Couldn't compile fragment program!\n%s\n", debug));
                }
                else
                {
                    FFATAL(("Couldn't compile fragment program!\n"));
                }
#ifdef DEBUG_COMPILE_ERROR
                std::ofstream outF("FragmentProgramError.glsl", std::ios::out|std::ios::binary);
                outF << fragmentProg << "\n" << debug << std::endl;
                outF.close();
#endif
                delete[] debug;
                glDeleteShader(fShader);
            }
        }
        if (!getGeometryProgram().empty() && !geoProg.empty())
        {
            if (GLAD_GL_VERSION_4_0) // core since opengl 4.1
            {
                OSGGLenum shader_type = GL_GEOMETRY_SHADER;

                gShader = glCreateShader(shader_type);
                const char *source = geoProg.c_str();
                glShaderSource(gShader, 1, &source, nullptr);

                int success = 0;
                glCompileShader(gShader);
                glGetShaderiv(gShader, GL_COMPILE_STATUS, &has_geometry);

                if (has_geometry == 0)
                {
                    compileError |= true;
                    char *debug = NULL;
                    OSGGLint debugLength;
                    glGetShaderiv(gShader, GL_INFO_LOG_LENGTH, &debugLength);

                    if (debugLength > 0)
                    {
                        debug = new char[debugLength];
                        debug[0] = 0;
                        glGetShaderInfoLog(gShader, debugLength, &debugLength, debug);
                    }

                    if (debug != NULL && debug[0] != 0)
                    {
                        FFATAL(("Couldn't compile geometry program!\n%s\n", debug));
                    }
                    else
                    {
                        FFATAL(("Couldn't compile geometry program!\n"));
                    }
#ifdef DEBUG_COMPILE_ERROR
                    std::ofstream outF("GeometryProgramError.glsl", std::ios::out|std::ios::binary);
                    outF << geoProg << std::endl;
                    outF.close();
#endif
                    delete[] debug;
                    glDeleteShader(gShader);
                }
            }
            else
            {
                FWARNING(("GL_EXT_geometry_shader4 extension not supported!\n"));
            }
        }

        if (has_vertex || has_fragment || has_geometry)
        {
            if (has_vertex)
            {
                glAttachShader(m_program, vShader);
                // just flagged for deletion
                glDeleteShader(vShader);
            }

            if (has_fragment)
            {
                glAttachShader(m_program, fShader);
                // just flagged for deletion
                glDeleteShader(fShader);
            }

            if (has_geometry)
            {
                glAttachShader(m_program, gShader);
                // just flagged for deletion
                glDeleteShader(gShader);
            }

            if (USE_BINARY && GLAD_GL_VERSION_4_1) // core since 4.1
            {
                const MFGLenum &ppnames = *getMFProgramParameterNames();
                const MFUInt32 &ppvalues = *getMFProgramParameterValues();
                for (UInt32 i = 0; i < ppnames.size(); ++i)
                {
                    if (i < ppvalues.size())
                        glProgramParameteri(m_program, ppnames[i], ppvalues[i]);
                }
                glProgramParameteri(m_program, GL_PROGRAM_BINARY_RETRIEVABLE_HINT, GL_TRUE);
            }
            else
            {
                updateProgramParameters();
            }

            glLinkProgram(m_program);

            OSGGLint success = 0;
            glGetProgramiv(m_program, GL_LINK_STATUS, &success);
            char *debug = NULL;
            OSGGLint debugLength;
            glGetProgramiv(m_program, GL_INFO_LOG_LENGTH, &debugLength);
            if (debugLength > 0)
            {
                debug = new char[debugLength];
                debug[0] = 0;
                glGetProgramInfoLog(m_program, debugLength, &debugLength, debug);
            }

            if (success)
            {
                if (_userCreateProgramCallback != NULL)
                    _userCreateProgramCallback(m_program);

                if (debug != NULL && debug[0] != 0)
                {
                    FWARNING(("SHLChunk: link status: %s\n", debug));
                }
                //else
                //{
                //    FWARNING(("SHLChunk: linked shader\n"));
                //}

                // write binary shader.
                if (!compileError && USE_BINARY && GLAD_GL_VERSION_4_1)// core since 4.1
                {
                    OSGGLsizei len = 0;
                    // ok for ATI we have to retreive the binary length without this the call to getProgramBinary() doesn't work!
                    glGetProgramiv(m_program, GL_PROGRAM_BINARY_LENGTH, &len);

                    if (len == 0)
                        len = 1048576; // 1 MB fallback
                    UInt8 *binary = new UInt8[len];
                    OSGGLenum binaryFormat = 0;
                    OSGGLsizei rlen = 0;
                    glGetProgramBinary(m_program, len, &rlen, &binaryFormat, binary);
                    binary_length = rlen;
                    if (rlen > 0 && glGetError() == GL_NO_ERROR)
                    {
                        FILE *f = osg::fopen(filename.c_str(), "wb");

                        if (f != NULL)
                        {
                            fwrite(&binaryFormat, sizeof(binaryFormat), 1, f);
                            // crypt shader
                            crypt(binary, rlen);
                            fwrite(binary, rlen, 1, f);
                            fclose(f);
                        }
                    }
                    else
                    {
                        FWARNING(("SHLChunk: couldn't get binary shader program!\n"));
                    }
                    delete[] binary;
                }

                // after linking we can detach the shader objects should save some
                // memory. ok for ATI we have to this after writing the binary shader code,
                // doing it before getProgramBinary fails!
                if (has_vertex)
                    glDetachShader(m_program, vShader);
                if (has_fragment)
                    glDetachShader(m_program, fShader);
                if (has_geometry)
                    glDetachShader(m_program, gShader);
            }
            else
            {
                if (debug != NULL && debug[0] != 0)
                {
                    FFATAL(("Couldn't link vertex and fragment program!\n%s\n", debug));
#ifdef DEBUG_COMPILE_ERROR
                    std::ofstream outF("ShaderLinkingError.glsl", std::ios::out|std::ios::binary);
                    outF << fragmentProg << std::endl;
                    outF.close();
#endif
                }
                else
                {
                    #ifdef DEBUG_COMPILE_ERROR
                    std::ofstream outF("ShaderLinkingError.glsl", std::ios::out|std::ios::binary);
                    outF << fragmentProg << std::endl;
                    outF.close();
#endif
                    FFATAL(("Couldn't link vertex and fragment program!\n"));
                }
                glUseProgram(0); // remove binding
                glDeleteProgram(m_program);
                m_program = 0;
                m_programChanged = true;
                m_programParameterChanged = true;

            }
            if (debug != NULL)
                delete[] debug;
        }
        else
        {
            glUseProgram(0); // remove binding
            glDeleteProgram(m_program);

            m_program = 0;
            m_programChanged = true;
            m_programParameterChanged = true;
        }
    }
    if (m_program != 0)
    {
        updateProgramParameters();
        // update all parameter locations
        updateParameterLocations(*getMFParameters());
        // update all parameters. Force it since everything has changed
        updateParameters(*getMFParameters(), true, true);

        glUseProgram(m_program);

        // update the standard uniform blocks
        GLuint viewportBlockIndex = glGetUniformBlockIndex(m_program, "OSGViewport");
        if (viewportBlockIndex != GL_INVALID_INDEX)
            glUniformBlockBinding(m_program, viewportBlockIndex, UBO_VIEWPORT);

        GLuint cameraBlockIndex = glGetUniformBlockIndex(m_program, "OSGCamera");
        if (cameraBlockIndex != GL_INVALID_INDEX)
            glUniformBlockBinding(m_program, cameraBlockIndex, UBO_CAMERA);

        GLuint geometryBlockIndex = glGetUniformBlockIndex(m_program, "OSGGeometry");
        if (geometryBlockIndex != GL_INVALID_INDEX)
            glUniformBlockBinding(m_program, geometryBlockIndex, UBO_OBJECT);

        GLuint lightBlockIndex = glGetUniformBlockIndex(m_program, "OSGLightSources");
        if (lightBlockIndex != GL_INVALID_INDEX)
            glUniformBlockBinding(m_program, lightBlockIndex, UBO_LIGHTS);

        if (GLAD_GL_VERSION_4_3 && GLAD_GL_ARB_bindless_texture) // light textures are only supported when bindless is available
        {
            GLuint lightTexturesBlockIndex = glGetUniformBlockIndex(m_program, "OSGLightSourceTextures");
            if (lightTexturesBlockIndex != GL_INVALID_INDEX)
                glUniformBlockBinding(m_program, lightTexturesBlockIndex, UBO_LIGHTTEXTURES);
        }

        GLuint lensFlareBlockIndex = glGetUniformBlockIndex(m_program, "OSGLensFlare");
        if(lensFlareBlockIndex != GL_INVALID_INDEX)
            glUniformBlockBinding(m_program, lensFlareBlockIndex, UBO_LENSFLARES);

        GLuint lensFlareRingsBlockIndex = glGetUniformBlockIndex(m_program, "OSGLensFlareRings");
        if(lensFlareRingsBlockIndex != GL_INVALID_INDEX)
            glUniformBlockBinding(m_program, lensFlareRingsBlockIndex, UBO_LENSFLARE_RINGS);

        GLuint lensFlareGlowsBlockIndex = glGetUniformBlockIndex(m_program, "OSGLensFlareGlows");
        if(lensFlareGlowsBlockIndex != GL_INVALID_INDEX)
            glUniformBlockBinding(m_program, lensFlareGlowsBlockIndex, UBO_LENSFLARE_GLOWS);

        GLuint lensFlareStarsBlockIndex = glGetUniformBlockIndex(m_program, "OSGLensFlareStars");
        if(lensFlareStarsBlockIndex != GL_INVALID_INDEX)
            glUniformBlockBinding(m_program, lensFlareStarsBlockIndex, UBO_LENSFLARE_STARS);

        GLuint lensFlareGhostsBlockIndex = glGetUniformBlockIndex(m_program, "OSGLensFlareGhosts");
        if(lensFlareGhostsBlockIndex != GL_INVALID_INDEX)
            glUniformBlockBinding(m_program, lensFlareGhostsBlockIndex, UBO_LENSFLARE_GHOSTS);

        GLuint lensFlareStreaksBlockIndex = glGetUniformBlockIndex(m_program, "OSGLensFlareStreaks");
        if(lensFlareStreaksBlockIndex != GL_INVALID_INDEX)
            glUniformBlockBinding(m_program, lensFlareStreaksBlockIndex, UBO_LENSFLARE_STREAKS);

        GLuint lensFlareTextureElementsBlockIndex = glGetUniformBlockIndex(m_program, "OSGLensFlareTextureElements");
        if(lensFlareTextureElementsBlockIndex != GL_INVALID_INDEX)
            glUniformBlockBinding(m_program, lensFlareTextureElementsBlockIndex, UBO_LENSFLARE_TEXTURE_ELEMENTS);

        GLuint sampleBlockIndex = glGetUniformBlockIndex(m_program, "OSGRandomSamples");
        if (sampleBlockIndex != GL_INVALID_INDEX)
            glUniformBlockBinding(m_program, sampleBlockIndex, UBO_RANDOM);

        GLuint environmentBlockIndex = glGetUniformBlockIndex(m_program, "OSGEnvironment");
        if (environmentBlockIndex != GL_INVALID_INDEX)
            glUniformBlockBinding(m_program, environmentBlockIndex, UBO_ENVIRONMENT);

        GLuint deformerBlockIndex = glGetUniformBlockIndex(m_program, "OSGDeformer");
        if (deformerBlockIndex != GL_INVALID_INDEX)
            glUniformBlockBinding(m_program, deformerBlockIndex, UBO_DEFORMER);

        GLuint scatterVolumesBlockIndex = glGetUniformBlockIndex(m_program, "OSGScatterVolumes");
        if (scatterVolumesBlockIndex != GL_INVALID_INDEX)
            glUniformBlockBinding(m_program, scatterVolumesBlockIndex, UBO_SCATTER_VOLUME);

        GLuint scatterVolumeInstancesBlockIndex = glGetUniformBlockIndex(m_program, "OSGScatterVolumeInstances");
        if (scatterVolumeInstancesBlockIndex != GL_INVALID_INDEX)
            glUniformBlockBinding(m_program, scatterVolumeInstancesBlockIndex, UBO_SCATTER_VOLUME_INSTANCE);

        GLuint VDBVolumesBlockIndex = glGetUniformBlockIndex(m_program, "OSGVDBVolumes");
        if (VDBVolumesBlockIndex != GL_INVALID_INDEX)
            glUniformBlockBinding(m_program, VDBVolumesBlockIndex, UBO_VDB_VOLUME);

        GLuint VDBVolumeInstancesBlockIndex = glGetUniformBlockIndex(m_program, "OSGOpenVDBVolumeInstances");
        if (VDBVolumeInstancesBlockIndex != GL_INVALID_INDEX)
            glUniformBlockBinding(m_program, VDBVolumeInstancesBlockIndex, UBO_VDB_VOLUME_INSTANCE);

        GLuint volumeLightIndicesBlockIndex = glGetUniformBlockIndex(m_program, "OSGVolumeLightIndices");
        if (volumeLightIndicesBlockIndex != GL_INVALID_INDEX)
            glUniformBlockBinding(m_program, volumeLightIndicesBlockIndex, UBO_VOLUME_LIGHT_INDICES);

        GLint shadowMapLocation = glGetUniformLocation(m_program, "OSGShadowMaps");
        if (shadowMapLocation != -1)
            glUniform1i(shadowMapLocation, TEX_UNIT_SHADOW_MAPS);

        GLint cubeShadowMapLocation = glGetUniformLocation(m_program, "OSGShadowCubeMaps");
        if (cubeShadowMapLocation != -1)
            glUniform1i(cubeShadowMapLocation, TEX_UNIT_SHADOW_CUBE_MAPS);

        GLint ambientOcclussionLocation = glGetUniformLocation(m_program, "OSGAmbientOcclusion");
        if (ambientOcclussionLocation != -1)
            glUniform1i(ambientOcclussionLocation, TEX_UNIT_AMBIENT_OCCLUSION);
        
        GLint geometryLightLocation = glGetUniformLocation(m_program, "OSGGeometryLightsTexture");
        if (geometryLightLocation != -1)
            glUniform1i(geometryLightLocation, TEX_UNIT_GEOMETRY_LIGHTS);

        GLint lightMapLocation = glGetUniformLocation(m_program, "OSGLightMaps");
        if (lightMapLocation != -1)
            glUniform1i(lightMapLocation, TEX_UNIT_LIGHTMAPS);

        GLint lightMapLocation1 = glGetUniformLocation(m_program, "OSGLightMaps1");
        if (lightMapLocation1 != -1)
            glUniform1i(lightMapLocation1, TEX_UNIT_LIGHTMAPS1);

        glUseProgram(0);
        m_programChanged = false;
        m_useBindlessShadowTextures = getUseBindlessShadowTextures();
        m_stereoViewRendering = getStereoViewRenderingEnabled();
        m_lensMatchedShadingRendering = getLensMatchedShadingEnabled();
        m_useFog = getFogEnabled();
    }
    auto t1 = oneapi::tbb::tick_count::now();
#ifndef VR_RELEASE_BUILD
    std::cout << "Shader compile took " << (t1-t0).seconds() * 1000.0 << " ms";
    std::cout << " String memory: " << (fragmentProg.size() + geoProg.size() + vertexProg.size()) / 1024 << " KBytes";
    std::cout << " Binary memory: " << (binary_length) / 1024 << " KBytes" << std::endl;
#endif
}

void SHLChunk::updateParameterLocation(OSGGLuint program,
    const ShaderParameterPtr &parameter)
{
    if (program == 0)
    {
        FNOTICE(("SHLChunk::updateParameterLocation: program == 0!\n"
            "This frame will probably be rendered wrong!\n"));
        return;
    }
    if(parameter->getTypeId() == ShaderParameter::SHPTypeSubroutine)
    {
        parameter->setLocation(glGetSubroutineUniformLocation(program, GL_FRAGMENT_SHADER, parameter->getName().c_str()));
    }
    else 
    {
        // as the location won't change after linking we can store them for speedup.
        parameter->setLocation(glGetUniformLocation(program, parameter->getName().c_str()));
    }

   
}

void SHLChunk::updateParameterLocations(const MFShaderParameterPtr &parameters)
{
    if (m_program == 0 || (!m_programChanged && !m_programParameterChanged))
        return;
    //std::cout << "updateParameterLocations " << m_program << " m_programChanged: " << m_programChanged << " m_programParameterChanged: " << m_programParameterChanged << std::endl;
    if (parameters.empty())
        return;

    for (UInt32 i = 0; i < parameters.size(); ++i)
    {
        ShaderParameterPtr parameter = parameters[i];
         if(parameter->getTypeId() == ShaderParameter::SHPTypeSubroutine)
        {
            parameter->setLocation(glGetSubroutineUniformLocation(m_program, GL_FRAGMENT_SHADER, parameter->getName().c_str()));
        }
        else
        {
            // as the location won't change after linking we can store them for speedup.
            parameter->setLocation(glGetUniformLocation(m_program, parameter->getName().c_str()));
        }
    }
}

void SHLChunk::updateParameters(
    const MFShaderParameterPtr &parameters,
    bool useProgram, bool force,
    bool keepProgramActive)
{

    if (!GLAD_GL_VERSION_4_0)
        return;

    if (m_program == 0 || !(m_programChanged || m_programParameterChanged))
    {
        return;
    }
    //std::cout << "updateParameters: " << m_program << " m_programChanged: " << m_programChanged << " m_programParameterChanged: " << m_programParameterChanged << std::endl;

    const MFString &blockNames = *getMFProgramParameterBlockNames();
    if (parameters.empty() && blockNames.empty())
    {
        m_programParameterChanged = false;
        return;
    }
    glUseProgram(m_program);

    _subroutines.clear();
    int activeSubroutineLocationCount = 0;
    glGetProgramStageiv(m_program, GL_FRAGMENT_SHADER, GL_ACTIVE_SUBROUTINE_UNIFORM_LOCATIONS, &activeSubroutineLocationCount);
    if(activeSubroutineLocationCount > 0)
    {
        _subroutines.resize(activeSubroutineLocationCount, 0);
    } 
    // set program parameters.
    const MFUInt32 &blockBindings = *getMFProgramParameterBlockBindings();
    for (UInt32 i = 0; i < blockNames.size(); ++i)
    {
        if (i < blockBindings.size())
        {
            GLuint blockIndex = glGetUniformBlockIndex(m_program, blockNames[i].c_str());
            if (blockIndex != GL_INVALID_INDEX)
                glUniformBlockBinding(m_program, blockIndex, blockBindings[i]);
        }
    }
    for (UInt32 i = 0; i < parameters.size(); ++i)
    {
        ShaderParameterPtr parameter = parameters[i];

        if (!force)
        {
            if (!parameter->hasChanged())
                continue;
            parameter->resetChanged();
        }
        //printf("seeting parameter: '%s'\n", parameter->getName().c_str());

        switch (parameter->getTypeId())
        {
        case ShaderParameter::SHPTypeBool:
        {
            ShaderParameterBoolPtr p = ShaderParameterBoolPtr::dcast(parameter);

            //printf("setting: %s %d\n", p->getName().c_str(), p->getValue());
            if (p->getLocation() == -2)
                updateParameterLocation(m_program, p);
            if (p->getLocation() >= 0)
                glUniform1i(p->getLocation(), OSGGLint(p->getValue()));
#if OUTPUT_PARAMETER_WARNING
            // ok I added my own simple parser to get all uniforms unused uniforms
            // are optimized away from the glsl compiler so we would get here a wrong warning!
            else if (!hasUniform(p->getName()))
                FWARNING(("Unknown parameter '%s'!\n",
                    p->getName().c_str()));
#endif
        }
        break;
        case ShaderParameter::SHPTypeInt:
        {
            ShaderParameterIntPtr p = ShaderParameterIntPtr::dcast(
                parameter);

            //printf("setting: %s %d\n", p->getName().c_str(), p->getValue());
            if (p->getLocation() == -2)
                updateParameterLocation(m_program, p);
            if (p->getLocation() >= 0)
                glUniform1i(p->getLocation(), p->getValue());
#if OUTPUT_PARAMETER_WARNING
            else if (!hasUniform(p->getName()))
                FWARNING(("Unknown parameter '%s'!\n",
                    p->getName().c_str()));
#endif
        }
        break;
        case ShaderParameter::SHPTypeUInt:
        {
            ShaderParameterUIntPtr p = ShaderParameterUIntPtr::dcast(
                parameter);

            //printf("setting: %s %d\n", p->getName().c_str(), p->getValue());
            if (p->getLocation() == -2)
                updateParameterLocation(m_program, p);
            if (p->getLocation() >= 0)
                glUniform1ui(p->getLocation(), p->getValue());
#if OUTPUT_PARAMETER_WARNING
            else if (!hasUniform(p->getName()))
                FWARNING(("Unknown parameter '%s'!\n",
                    p->getName().c_str()));
#endif
        }
        break;
        case ShaderParameter::SHPTypeReal:
        {
            ShaderParameterRealPtr p = ShaderParameterRealPtr::dcast(parameter);

            //printf("setting: %s %f\n", p->getName().c_str(), p->getValue());
            if (p->getLocation() == -2)
                updateParameterLocation(m_program, p);
            if (p->getLocation() >= 0)
                glUniform1f(p->getLocation(), p->getValue());
#if OUTPUT_PARAMETER_WARNING
            else if (!hasUniform(p->getName()))
                FWARNING(("Unknown parameter '%s'!\n",
                    p->getName().c_str()));
#endif
        }
        break;
        case ShaderParameter::SHPTypeVec2f:
        {
            ShaderParameterVec2fPtr p = ShaderParameterVec2fPtr::dcast(parameter);

            if (p->getLocation() == -2)
                updateParameterLocation(m_program, p);
            if (p->getLocation() >= 0)
                glUniform2fv(p->getLocation(), 1,
                    const_cast<OSGGLfloat *>(
                        p->getValue().getValues()));
#if OUTPUT_PARAMETER_WARNING
            else if (!hasUniform(p->getName()))
                FWARNING(("Unknown parameter '%s'!\n",
                    p->getName().c_str()));
#endif
        }
        break;
        case ShaderParameter::SHPTypeVec3f:
        {
            ShaderParameterVec3fPtr p = ShaderParameterVec3fPtr::dcast(parameter);

            if (p->getLocation() == -2)
                updateParameterLocation(m_program, p);
            if (p->getLocation() >= 0)
                glUniform3fv(p->getLocation(), 1, const_cast<OSGGLfloat *>(p->getValue().getValues()));
#if OUTPUT_PARAMETER_WARNING
            else if (!hasUniform(p->getName()))
                FWARNING(("Unknown parameter '%s'!\n",
                    p->getName().c_str()));
#endif
        }
        break;
        case ShaderParameter::SHPTypeVec4f:
        {
            ShaderParameterVec4fPtr p = ShaderParameterVec4fPtr::dcast(parameter);

            if (p->getLocation() == -2)
                updateParameterLocation(m_program, p);
            if (p->getLocation() >= 0)
                glUniform4fv(p->getLocation(),
                    1,
                    const_cast<OSGGLfloat *>(
                        p->getValue().getValues()));
#if OUTPUT_PARAMETER_WARNING
            else if (!hasUniform(p->getName()))
                FWARNING(("Unknown parameter '%s'!\n",
                    p->getName().c_str()));
#endif
        }
        break;
        case ShaderParameter::SHPTypeMatrix:
        {
            ShaderParameterMatrixPtr p = ShaderParameterMatrixPtr::dcast(parameter);
            if (p->getLocation() == -2)
                updateParameterLocation(m_program, p);
            if (p->getLocation() >= 0)
                glUniformMatrix4fv(p->getLocation(), 1,
                    GL_FALSE,
                    const_cast<OSGGLfloat *>(
                        p->getValue().getValues()));
#if OUTPUT_PARAMETER_WARNING
            else if (!hasUniform(p->getName()))
                FWARNING(("Unknown parameter '%s'!\n",
                    p->getName().c_str()));
#endif
        }
        break;
        // arrays
        case ShaderParameter::SHPTypeMInt:
        {
            ShaderParameterMIntPtr p = ShaderParameterMIntPtr::dcast(parameter);

            //printf("setting: %s %d\n", p->getName().c_str(), p->getValue());
            if (p->getLocation() == -2)
                updateParameterLocation(m_program, p);
            if (p->getLocation() >= 0 && !p->getMFValue()->empty())
                glUniform1iv(p->getLocation(), p->getMFValue()->getSize(), const_cast<OSGGLint*>(&p->getValue(0)));
#if OUTPUT_PARAMETER_WARNING
            else if (!hasUniform(p->getName()))
                FWARNING(("Unknown parameter '%s'!\n",
                    p->getName().c_str()));
#endif
        }
        break;
        case ShaderParameter::SHPTypeMUInt:
        {
            ShaderParameterMUIntPtr p = ShaderParameterMUIntPtr::dcast(parameter);

            //printf("setting: %s %d\n", p->getName().c_str(), p->getValue());
            if (p->getLocation() == -2)
                updateParameterLocation(m_program, p);
            if (p->getLocation() >= 0 && !p->getMFValue()->empty())
                glUniform1uiv(p->getLocation(), p->getMFValue()->getSize(), const_cast<OSGGLuint*>(&p->getValue(0)));
#if OUTPUT_PARAMETER_WARNING
            else if (!hasUniform(p->getName()))
                FWARNING(("Unknown parameter '%s'!\n",
                    p->getName().c_str()));
#endif
        }
        break;
        case ShaderParameter::SHPTypeMReal:
        {
            ShaderParameterMRealPtr p = ShaderParameterMRealPtr::dcast(parameter);
            // get "glUniform1fvARB" function pointer
            if (p->getLocation() == -2)
                updateParameterLocation(m_program, p);
            if (p->getLocation() >= 0 && !p->getMFValue()->empty())
                glUniform1fv(p->getLocation(),
                    p->getMFValue()->getSize(),
                    const_cast<OSGGLfloat *>(&p->getValue(0)));
#if OUTPUT_PARAMETER_WARNING
            else if (!hasUniform(p->getName()))
                FWARNING(("Unknown parameter '%s'!\n",
                    p->getName().c_str()));
#endif
        }
        break;
        case ShaderParameter::SHPTypeMVec2f:
        {
            ShaderParameterMVec2fPtr p = ShaderParameterMVec2fPtr::dcast(parameter);

            if (p->getLocation() == -2)
                updateParameterLocation(m_program, p);

            if (p->getLocation() >= 0 && !p->getMFValue()->empty())
                glUniform2fv(p->getLocation(),
                    p->getMFValue()->getSize(),
                    const_cast<OSGGLfloat *>(
                        p->getValue(0).getValues()));
#if OUTPUT_PARAMETER_WARNING
            else if (!hasUniform(p->getName()))
                FWARNING(("Unknown parameter '%s'!\n",
                    p->getName().c_str()));
#endif
        }
        break;
        case ShaderParameter::SHPTypeMVec3f:
        {
            ShaderParameterMVec3fPtr p = ShaderParameterMVec3fPtr::dcast(parameter);

            if (p->getLocation() == -2)
                updateParameterLocation(m_program, p);
            if (p->getLocation() >= 0 && !p->getMFValue()->empty())
                glUniform3fv(p->getLocation(),
                    p->getMFValue()->getSize(),
                    const_cast<OSGGLfloat *>(
                        p->getValue(0).getValues()));
#if OUTPUT_PARAMETER_WARNING
            else if (!hasUniform(p->getName()))
                FWARNING(("Unknown parameter '%s'!\n",
                    p->getName().c_str()));
#endif
        }
        break;
        case ShaderParameter::SHPTypeMVec4f:
        {
            ShaderParameterMVec4fPtr p = ShaderParameterMVec4fPtr::dcast(parameter);
            // get "glUniform4fvARB" function pointer
            if (p->getLocation() == -2)
                updateParameterLocation(m_program, p);
            if (p->getLocation() >= 0 && !p->getMFValue()->empty())
                glUniform4fv(p->getLocation(),
                    p->getMFValue()->getSize(),
                    const_cast<OSGGLfloat *>(
                        p->getValue(0).getValues()));
#if OUTPUT_PARAMETER_WARNING
            else if (!hasUniform(p->getName()))
                FWARNING(("Unknown parameter '%s'!\n",
                    p->getName().c_str()));
#endif
        }
        break;
        case ShaderParameter::SHPTypeMMatrix:
        {
            ShaderParameterMMatrixPtr p = ShaderParameterMMatrixPtr::dcast(parameter);

            if (p->getLocation() == -2)
                updateParameterLocation(m_program, p);
            if (p->getLocation() >= 0 && !p->getMFValue()->empty())
                glUniformMatrix4fv(p->getLocation(),
                    p->getMFValue()->getSize(),
                    GL_FALSE,
                    const_cast<OSGGLfloat *>(
                        p->getValue(0).getValues()));
#if OUTPUT_PARAMETER_WARNING
            else if (!hasUniform(p->getName()))
                FWARNING(("Unknown parameter '%s'!\n",
                    p->getName().c_str()));
#endif
        }
        break;
        case ShaderParameter::SHPTypeHandle:
        {
            if (GLAD_GL_ARB_bindless_texture)
            {
                ShaderParameterHandlePtr p = ShaderParameterHandlePtr::dcast(parameter);
                if (p->getLocation() == -2)
                    updateParameterLocation(m_program, p);
                if (p->getLocation() >= 0)
                    glUniformHandleui64ARB(p->getLocation(), p->getValue());
#if OUTPUT_PARAMETER_WARNING
                else if (!hasUniform(p->getName()))
                    FWARNING(("Unknown parameter '%s'!\n",
                        p->getName().c_str()));
#endif
            }
        }
        break;
        case ShaderParameter::SHPTypeSubroutine:
        {
            if(GLAD_GL_ARB_shader_subroutine)
            {
                ShaderParameterSubroutinePtr p = ShaderParameterSubroutinePtr::dcast(parameter);
                if (p->getLocation() == -2)
                    updateParameterLocation(m_program, p);
                if (p->getLocation() >= 0 && p->getLocation() < _subroutines.size())
                {
                    GLuint subroutineIndex = glGetSubroutineIndex(m_program, GL_FRAGMENT_SHADER, p->getValue().c_str());
                    if(subroutineIndex != GL_INVALID_INDEX)
                        _subroutines[p->getLocation()] = subroutineIndex;
                }
#if OUTPUT_PARAMETER_WARNING
                else if (!hasUniform(p->getName()))
                    FWARNING(("Unknown parameter '%s'!\n",
                        p->getName().c_str()));
#endif
            }
        }
        break;
        case ShaderParameter::SHPTypeMHandle:
        {
            if (GLAD_GL_ARB_bindless_texture)
            {
                ShaderParameterMHandlePtr p = ShaderParameterMHandlePtr::dcast(parameter);
                if (p->getLocation() == -2)
                    updateParameterLocation(m_program, p);
                if (p->getLocation() >= 0 && !p->getMFValue()->empty())
                    glUniformHandleui64vARB(p->getLocation(),
                        p->getMFValue()->getSize(),
                        const_cast<OSGGLuint64*>(&p->getValue(0)));
#if OUTPUT_PARAMETER_WARNING
                else if (!hasUniform(p->getName()))
                    FWARNING(("Unknown parameter '%s'!\n",
                        p->getName().c_str()));
#endif
            }
        }
        break;
        default:
#if OUTPUT_PARAMETER_WARNING
            FWARNING(("Parameter '%s' has unknown type %d!\n",
                parameter->getName().c_str(),
                parameter->getTypeId()));
#endif
            break;
        }
        parameter->resetChanged();
    }

    if (!keepProgramActive)
    {
        glUseProgram(0);
    }
    m_programParameterChanged = false;
}

void SHLChunk::updateProgramParameters()
{
    if (!GLAD_GL_VERSION_4_0)
        return;

    if (m_program == 0 || !(m_programChanged || m_programParameterChanged))
    {
        return;
    }
    // std::cout << "Update Program Parameters" << m_program  << " m_programChanged: " << m_programChanged  << " m_programParameterChanged: " << m_programParameterChanged  << std::endl;
     // set program parameters.
    const MFGLenum &ppnames = *getMFProgramParameterNames();
    const MFUInt32 &ppvalues = *getMFProgramParameterValues();
    for (UInt32 i = 0; i < ppnames.size(); ++i)
    {
        if (i < ppvalues.size()) {
            glProgramParameteri(m_program, ppnames[i], ppvalues[i]);
        }
    }
}


void SHLChunk::addParameterCallback(const char *name, parametercbfp fp)
{
    if (name == NULL || fp == NULL)
        return;

    //    std::string szName(name);

    osgparametercbfp nullfp = NULL;

    setUniformParameter(name, 0);
    _userParameterCallbacks.insert(
        UserParameterCallbacksMap::value_type(name, std::make_pair(fp, nullfp)));
}

void SHLChunk::addParameterCallback(const char *name, osgparametercbfp fp)
{
    if (name == NULL || fp == NULL)
        return;

    //    std::string szName(name);

    parametercbfp nullfp = NULL;

    setUniformParameter(name, 0);
    _userParameterCallbacks.insert(
        UserParameterCallbacksMap::value_type(name, std::make_pair(nullfp, fp)));
}

void SHLChunk::setParameterCallback(parametercbfp fp)
{
    _userParametersCallback = fp;
}

SHLChunk::parametercbfp SHLChunk::getParameterCallback(void)
{
    return _userParametersCallback;
}

void SHLChunk::setCreateProgramCallback(programcbfp fp)
{
    _userCreateProgramCallback = fp;
}

SHLChunk::programcbfp SHLChunk::getCreateProgramCallback(void)
{
    return _userCreateProgramCallback;
}

void SHLChunk::setDeleteProgramCallback(programcbfp fp)
{
    _userDeleteProgramCallback = fp;
}

SHLChunk::programcbfp SHLChunk::getDeleteProgramCallback(void)
{
    return _userDeleteProgramCallback;
}

void SHLChunk::addProgramParameter(OSGGLenum name, UInt32 value)
{
    editMFProgramParameterNames()->push_back(name);
    editMFProgramParameterValues()->push_back(value);
}

void SHLChunk::subProgramParameter(OSGGLenum name)
{
    MFGLenum &ppnames = *editMFProgramParameterNames();
    MFUInt32 &ppvalues = *editMFProgramParameterValues();

    for (UInt32 i = 0; i < ppnames.size(); ++i)
    {
        if (ppnames[i] == name && i < ppvalues.size())
        {
            ppnames.erase(ppnames.begin() + i);
            ppvalues.erase(ppvalues.begin() + i);
            break;
        }
    }
}

void SHLChunk::setProgramParameter(OSGGLenum name, UInt32 value)
{
    // remove old one.
    subProgramParameter(name);
    // add new one.
    addProgramParameter(name, value);
}

void SHLChunk::addProgramParameterBlockBinding(const char* name, UInt32 value)
{
    editMFProgramParameterBlockNames()->push_back(std::string(name));
    editMFProgramParameterBlockBindings()->push_back(value);
}

void SHLChunk::subProgramParameterBlockBinding(const char* name)
{
    MFString &ppnames = *editMFProgramParameterBlockNames();
    MFUInt32 &ppvalues = *editMFProgramParameterBlockBindings();
    std::string sName = name;
    for (UInt32 i = 0; i < ppnames.size(); ++i)
    {
        if (ppnames[i] == sName && i < ppvalues.size())
        {
            ppnames.erase(ppnames.begin() + i);
            ppvalues.erase(ppvalues.begin() + i);
            break;
        }
    }
}

void SHLChunk::setProgramParameterBlockBinding(const char* name, UInt32 value)
{
    subProgramParameterBlockBinding(name);
    addProgramParameterBlockBinding(name, value);
}

UInt32 SHLChunk::getProgramParameter(OSGGLenum name)
{
    const MFGLenum &ppnames = *getMFProgramParameterNames();
    const MFUInt32 &ppvalues = *getMFProgramParameterValues();

    for (UInt32 i = 0; i < ppnames.size(); ++i)
    {
        if (ppnames[i] == name && i < ppvalues.size())
            return ppvalues[i];
    }
    FWARNING(("SHLChunk::getProgramParameter : Couldn't find program parameter %u!\n",
        name));
    return 0;
}

std::vector<std::pair<OSGGLenum, UInt32> > SHLChunk::getProgramParameters(void)
{
    std::vector<std::pair<OSGGLenum, UInt32> > parameters;

    const MFGLenum &ppnames = *getMFProgramParameterNames();
    const MFUInt32 &ppvalues = *getMFProgramParameterValues();

    for (UInt32 i = 0; i < ppnames.size(); ++i)
    {
        if (i < ppvalues.size())
            parameters.push_back(std::make_pair(ppnames[i], ppvalues[i]));
    }

    return parameters;
}

void SHLChunk::clearProgramParameters(void)
{
    editMFProgramParameterNames()->clear();
    editMFProgramParameterValues()->clear();
}

GLuint SHLChunk::getProgramId()
{
    if (m_program == 0 || m_programChanged ||
        m_programParameterChanged ||
        (m_stereoViewRendering != getStereoViewRenderingEnabled()) ||
        (m_lensMatchedShadingRendering != getLensMatchedShadingEnabled()))
    {
        updateProgram();
        updateProgramParameters();
        updateParameters(*getMFParameters(), true, true);
    }
    return m_program;
}

OSGGLint SHLChunk::getUniformLocation(const char* name)
{
    if (m_program == 0)
        return -1;
    return glGetUniformLocation(m_program, name);
}

OSGGLint SHLChunk::getUniformBlockIndex(const char* name)
{
    if (m_program == 0)
        return -1;
    return glGetUniformBlockIndex(m_program, name);
}


void SHLChunk::setClusterId(Int32 id)
{
    _clusterId = id;
}

void SHLChunk::setUniformBlockSize(Int32 blockSize)
{
    s_uniformBlockSize = blockSize;
}

void SHLChunk::setStereoViewRenderingEnabled(bool on)
{
    s_hasStereoViewRenderingSupport = on;
}

void SHLChunk::setLensMatchedShadingEnabled(bool on)
{
    s_hasLensMatchedShadingSupport = on;
}

void SHLChunk::setUseBindlessShadowTextures(bool on)
{
    s_useBindlessShadowTextures = ( GLAD_GL_VERSION_4_3 && GLAD_GL_ARB_bindless_texture) ? on : false;
}


void SHLChunk::setGPUType(bool isIntel, bool isAMD, bool isNvidia)
{
    s_isAMD = isAMD;
    s_isIntel = isIntel;
    s_isNvidia = isNvidia;
}


void SHLChunk::updateClusterId(const ShaderParameterPtr &parameter,
    DrawActionBase *action, OSGGLuint program)
{
    if (parameter->getFlags() & ShaderParameter::SHPFlagUpdate)
        return;

    if (parameter->getLocation() == -2)
        updateParameterLocation(program, parameter);
    if (parameter->getLocation() >= 0)
        glUniform1i(parameter->getLocation(), OSGGLint(_clusterId));
}

/*------------------------------ State ------------------------------------*/

void SHLChunk::requestParameterUpdate()
{
    m_programParameterChanged = true;
}

void SHLChunk::update(DrawActionBase *action)
{

}

void SHLChunk::activate(DrawActionBase *action, UInt32 idx)
{
    if (!GLAD_GL_VERSION_4_0)
        return;

    //std::cout << "updateParameters: " << m_program << " m_programChanged: " << m_programChanged << " m_programParameterChanged: " << m_programParameterChanged << std::endl;
    if (m_program == 0 || m_programChanged || m_programParameterChanged || recompileRequired())
    {
        updateProgram();
        updateProgramParameters();
        updateParameters(*getMFParameters(), true, true);
    }



    if (getMinShadingSamples() > 0.0f)
        glMinSampleShading(getMinShadingSamples());

    glUseProgram(m_program);
    
    if(!_subroutines.empty())
        glUniformSubroutinesuiv(GL_FRAGMENT_SHADER, _subroutines.size(), _subroutines.data());

    if (getPointSize())
    {
        enableGLFlag(GL_PROGRAM_POINT_SIZE);
        enableGLFlag(GL_POINT_SPRITE);
    }
    // set the current active program and geometry Info location
    s_currentProgram = m_program;
}

void SHLChunk::deactivate(DrawActionBase *action, UInt32 idx)
{
    if (!GLAD_GL_VERSION_4_0)
        return;

    glUseProgram(0); // since everything is a shader now, there is not really much need to unbind a program since another bind will come in any case

    if (getMinShadingSamples() > 0.0f) // reset
    {
        if (s_isNvidia)
            glMinSampleShading(0.0f); // NVIDIA hase some sort of adaptive code giving good quality by only a slight performance drop
        else// if(s_isAMD || s_isIntel)
            glMinSampleShading(0.25f); // on AMD 0.25f has shown to give results similar to nvidia
    }

    if (getPointSize())
    {
        disableGLFlag(GL_PROGRAM_POINT_SIZE);
        disableGLFlag(GL_POINT_SPRITE);
    }

    s_currentProgram = 0;
    s_currentGeometryInfoUniform = -1;
}

/*-------------------------- Comparison -----------------------------------*/

bool SHLChunk::operator < (const StateChunk &other) const
{
    return this < &other;
}

bool SHLChunk::operator == (const StateChunk &other) const
{
    SHLChunk const *tother = dynamic_cast<SHLChunk const*>(&other);

    if (!tother)
        return false;

    if (getVertexProgram() != tother->getVertexProgram() ||
        getFragmentProgram() != tother->getFragmentProgram() ||
        getMFParameters()->size() != tother->getMFParameters()->size())
        return false;

    return true;
}

bool SHLChunk::operator != (const StateChunk &other) const
{
    return !(*this == other);
}

/*!
* \brief
* \param
* \return
*/
std::string SHLChunk::getShaderVersionString()
{
    std::string versionString;
    if (GLAD_GL_VERSION_4_6)
    {
        versionString = "#version 460 core\n";
        versionString += "#define GLAD_GL_VERSION_46\n";
        versionString += "#define GLAD_GL_VERSION_43\n"; // we need both defines

    }
    else if (GLAD_GL_VERSION_4_3)
    {
        versionString = "#version 430 core\n";
        versionString += "#define GLAD_GL_VERSION_43\n";

    }
    else if (GLAD_GL_VERSION_4_0)
    {
        if (GLAD_GL_MESA)
        {
            versionString = "#version 330 core\n"; //return "#version 430 core\n";
            versionString += "#define GLAD_GL_VERSION_33\n";
        }
        else
        {
            versionString = "#version 400 core\n"; //return "#version 430 core\n";
            versionString += "#define GLAD_GL_VERSION_40\n";
        }
    }
    else
    {
        return "";
    }
    return versionString;
}

/*!
* \brief
* \param
* \return
*/
std::string SHLChunk::getShaderStructUniformString(bool includeLensFlares)
{
    // NOTE: The defines must match the LightFlags defined in vrGlobalEnums
    std::string structString =
        "#define DirectionalLightType 0\n"
        "#define SpotLightType 1\n"
        "#define PointLightType 2\n"
        "#define DiskAreaLightType 3\n"
        "#define RectangularAreaLightType 4\n"
        "#define SphereAreaLightType 5\n"
        "#define RayLightType 6\n"
        "#define LightEnable (1 << 0)\n"
        "#define LocalLight (1 << 1)\n"
        "#define DeltaLight (1 << 2)\n"
        "#define GeometryLight (1 << 3)\n"
        "#define IlluminateScene (1 << 4)\n"
        "#define IlluminateShadowMaterial (1 << 5)\n"
        "#define CastShadowOnShadowMaterial (1 << 6)\n"
        "#define UseTextureImportanceSampling (1 << 7)\n"
        "#define SpotLight (1 << 8)\n"
        "#define DirectWithPhotons (1 << 9)\n"
        "#define EmitCausticPhotons (1 << 10)\n"
        "#define UseIESProfile (1 << 11)\n"
        "#define UseLightTexture (1 << 12)\n"
        "#define IlluminateVolumes (1 << 15)\n"

        // Volume Eval Flags
        "#define EnableVolumeShadow (1 << 0)\n"
        "#define EnableSelfShadow (1 << 1)\n"
        "#define EnableFalloff (1 << 2)\n"
        "#define EnableNoise (1 << 3)\n"
        "#define UseGlobalNoise (1 << 4)\n"
        "#define EnableVDBTraversalOptimization (1 << 5)\n"
        "#define EnableVDBTrilinearSampling (1 << 6)\n"

        "struct OSGLightSource\n"
        "{\n"
        "    mat4  worldMatrix; // the world transform matrix. \n"
        //"    mat4  invWorldMatrix; // the inverse world transform matrix. \n"
        "    mat4  lightMatrix; // the light matrix for shadow mapping. it is basically the world matrix combined with a projection matrix\n"
        "    vec3  diffuseColor; // diffuse color of the light \n"
        "    int   useColorTexture; // use color map\n"
        "    vec3  glossyColor; // glossy color of the light \n"
        "    int   flags; // flags: 1<<1: use ies profile. 1<<2: illuminate shadow material. 1 << 3 cast shadows on shadow material\n"
        "    vec3  attenuation; // attenuation: x = constant, y = linear, z = quadratic/realistic \n"
        "    float specularVisibility; // specular visibility factor, should be either 0 or 1\n"
        "    vec2  spotCone; // the spotlight cone description: x = cos(innerCone), y = cos(outerCone) . \n"
        "    vec2  nearFar; // near/far clipping used for shadowmap rendering \n"
        "    int   lightType; // int specifying the lightsource type     \n"
        "    float shadowIntensity; // shadow intensity scale\n"
        "    float shadowMaterialShadowIntensity; // shadow intensity scale\n"
        "    float shadowFilterRadius; // shadow map filter radius\n"
        "    int   shadowMapIndex; // index referring to the shadow map\n"
        "    int   shadowMapResolution; // resolution of the shadow map\n"
        "    float flip;\n"
        "    int   hasShadowMap; // flag to indicate if a bindless texture is present\n"
       "#ifdef GLAD_GL_ARB_bindless_shadow_texture\n"
        "    sampler2DShadow    shadowTexture; // shadow map when using bindless textures\n"
        "    samplerCubeShadow  shadowCubeTexture; // shadow cubemap when using bindless textures\n"
        "#else\n"
        "    int  padding0; // filling the texture slots\n"
        "    int  padding1;\n"
        "    int  padding2;\n"
        "    int  padding3;\n"
        "#endif \n" 
        "#ifdef GLAD_GL_ARB_bindless_texture\n"
        "    sampler2DArray     prefilteredTexture;\n"
        "#else\n"
        "    int  padding4;\n"
        "    int  padding5;\n"
        "#endif \n"
        " int pad0;\n"
        " int pad1;\n"
        "};\n";
        
#if VOLUME_SCATTER_AVAILABLE
    structString +=
        "struct OSGVolumeInstance\n"
        "{\n"
        "    mat4   worldToObjectMatrix; \n"
        "    vec3   minBox; \n"
        "    int    volumeIndex; \n"
        "    vec3   maxBox; \n"
        "    int    isRotated; \n"
        "    vec3   halfScatterDim; \n"
        "    int    pad0; \n"
        "};\n";

    structString +=
        "struct OSGVolume\n"
        "{\n"
        "    vec3   sigmaS; // the extinction factor\n"
        "    float  g; // heyney-greenstrein anisotropy value\n"
        "    vec3   sigmaT; // the scatter factor \n"
        "    uint   quality;\n"
        "    int    flags;\n"
        "    int    pad0;\n"
        "    int    pad1;\n"
        "    int    pad2;\n"
        "};\n";

    structString +=
        "struct OSGScatterVolume\n"
        "{\n"
        "    OSGVolume volumeProperties; \n"
        "    vec3   noiseFrequency; // base frequency of noise\n"
        "    float  noiseLacunarity;\n"
        "    vec3   falloffOffset;\n"
        "    float  noiseTime; // time in seconds for 4D noise\n"
        "    vec3   falloffScale;\n"
        "    float  noiseGainExponent;\n"
        "    float  gradientCoefficient; \n"
        "    int    noiseOctaves;\n"
        "    float  falloffRateExponent;\n"
        "    int    pad0;\n"
        "};\n";

    structString +=
        "struct OSGVDBVolume\n"
        "{\n"
        "    OSGVolume volumeProperties; \n"
        "    int     numChannels;\n"
        "    int    scatterAndAbsorptionChannel;\n"
        "    int    emissionChannel;\n"
        "    float   sigmaE;\n"
        "    uint   vdbBufferOffset;\n"
        "    float   emissionChannelMin;\n"
        "    float   emissionChannelMax;\n"
        "    int     emissionChannelTexLayer;\n"
        "    uvec4  channelOffsets[MAX_VDB_VOLUME_CHANNELS / 4];\n"
        "};\n";
#endif

        structString +=
        "struct OSGFog\n"
        "{\n"
        "    vec4 colorDistance; \n"
        "    vec4 colorHeight; \n"
        "    vec4 noiseSizeDistance;    // Distance fog noise size: x,y,z,unused \n"
        "    vec4 noiseOffsetDistance;    // Distance fog noise offset: x,y,z,unused \n"
        "    vec4 noiseSizeHeight;      // Height fog noise size: x,y,z,unused \n"
        "    vec4 noiseOffsetHeight;      // Height fog noise offset: x,y,z,unused \n"
        "    vec2 densityNoiseDistance; // Distance fog: density, noise. Noise == 0 disables noise \n"
        "    vec2 densityNoiseHeight;   // Height fog: density, noise. Noise == 0 disables noise \n"
        "    vec2 rangeDistance; \n"
        "    vec2 rangeHeight; \n"
        "    int  activeDistance; \n"
        "    int  activeHeight; \n"
        "    int  falloffDistance;            // realistic,linear \n"
        "    int  falloffHeight;              // realistic,linear \n"
        "    float blendHeight; \n"
        "    int   reserved0; \n"
        "    int   reserved1; \n"
        "    int   reserved2; \n"
            "};\n";

    if (GLAD_GL_VERSION_4_3 && GLAD_GL_ARB_bindless_texture)
    {
        structString +=
            "struct OSGLightSourceTexture\n"
            "{\n"
            "    mat4  lightMatrix; // the light projection matrix. \n"
            "    mat4  invWorldMatrix; // the inverse world transform matrix. \n"
            "    vec2  rotate; // texture rotation\n"
            "    vec2  repeat; // texture repeat \n"
            "    vec2  offset; // texture offset \n"
            "    vec2  reserved; // alignment purposes \n"
            "    sampler2D colorTexture; // color texture sampler\n"
            "    sampler2D iesProfile; // ies profile texture sampler\n"
            "};\n";
    }
    if(includeLensFlares)
    {
    structString +=
        "struct OSGLensFlareGhost\n"
        "{\n"
        "    vec4 tint; //The alpha value of the tint is the intensity of the ghost\n"
        "    vec3 chromaticAberration; //Aberration for each color. To get a small rainbow border use -0.1f, 0.0f, 0.1f\n"
        "    int  isChromaticAberrationActive; //Flag for turning chromatic aberration on and off\n"
        "    int  shape; //Enum for the different shapes Not decided yet\n"
        "    float distance; //Distance factor multiplied with the distance of light source to the image center to determine \n"
        "                           //the overall distance to the light source\n"
        "    float size;\n"
        "    float thickness;\n"
        "    float edgeIntensity;\n"
        "    float blurring;\n"
        "    float rotationDegrees;\n"
        "    int   id;\n"
        "};\n"
        "struct OSGLensFlareStar\n"
        "{\n"
        "    vec4 tint; //The alpha value of the tint is the intensity of the star\n"
        "    float size;\n"
        "    float intensityScaleFactor; //The scale factor that is multiplied with the intenity with repect to the distance of the image center\n"
        "    int useScaling; //Flag indicating if the intensity is scaled according to the distance of the star to the image center\n"
        "    int shape; //Shape of th aperture determining the look of the star\n"
        "    int numberOfBursts;\n"
        "    float thickness;\n"
        "    float rotationDegrees;\n"
        "    float blurring;\n"
        "    int id;\n"
        "    int padding1;\n"
        "    int padding2;\n"
        "    int padding3;\n"
        "};\n"
        "struct OSGLensFlareGlow\n"
        "{\n"
        "    vec4 tint;\n"
        "    float size;\n"
        "    float intensityScaleFactor; //The scale factor that is multiplied with the intenity with repect to the distance of the image center\n"
        "    float blurring;\n"
        "    int useScaling; //Flag indicating if the intensity is scaled according to the distance of the star to the image center\n"
        "    int shape;\n"
        "    float rotationDegrees;\n"
        "    int id;\n"
        "    int padding1;\n"
        "};\n"
        "struct OSGLensFlareRing\n"
        "{\n"
        "    vec4 tint;\n"
        "    vec3 chromaticAberration; //Aberration for each color. To get a small rainbow border use -0.1f, 0.0f, 0.1f\n"
        "    int isChromaticAberrationActive; //Flag for turning chromatic aberration on and off\n"
        "    int useScaling; //Flag indicating if the intensity is scaled according to the distance of the star to the image center\n"
        "    float intensityScaleFactor; //The scale factor that is multiplied with the intenity with repect to the distance of the image center\n"
        "    float size;\n"
        "    float blurring;\n"
        "    float thickness;\n"
        "    int shape;\n"
        "    float rotationDegrees;\n"
        "    int numberOfStreaks;\n"
        "    float streakWidth;\n"
        "    int id;\n"
        "    int padding1;\n"
        "    int padding2;\n"
        "};\n"
        "struct OSGLensFlareStreak\n"
        "{\n"
        "    vec4 tint;\n"
        "    vec2 offset;\n"
        "    float size;\n"
        "    float blurring;\n"
        "    float rotationDegrees;\n"
        "    float width;\n"
        "    int id;\n"
        "    int padding1;\n"
        "};\n"
        "struct OSGLensFlareTextureElement\n"
        "{\n"
        "    vec4 tint;\n"
        "    vec2 aspect;\n"
        "    float distance;\n"
        "    float size;\n"
        "    vec3 chromaticAberration;\n"
        "    float rotationDegrees;\n"
        "    float isGhost;\n"
        "    int id;\n"
        "    int padding1;\n"
        "    int padding2;\n"
        "};\n";
    }
    return structString;
}

/*!
* \brief
* \param
* \return
*/
std::string SHLChunk::getShaderViewportUniformString()
{
    std::string uniformString = "// viewport data, binding id 0\n";

    if (GLAD_GL_VERSION_4_3)
        uniformString += "layout(std140, binding = 0) uniform OSGViewport\n";
    else
        uniformString += "layout(std140) uniform OSGViewport\n";

    uniformString += "{\n"
        "    mat4 rec709toRCS; // colorspace conversion matrix\n"
        "    vec4 region; // left, right, bottom, top\n"
        "    vec4 pixelRegion; // left, right, bottom, top\n"
        "    vec2 pixelResolution; // pixelwidth, pixelheight\n"
        "    vec2 aspect;\n"
        "    float superSamplingScale;\n"
        "    float displayLuminance;\n"
        "    float physicalScaleFactor;\n"
        "    int   sampleId;\n"
        "    vec3 separateBakeScale;\n"
        "    uint useBaking;\n"
        "    float sceneUnitScale;\n"
        "    int maxVolumeSteps;\n"
        "    int colorspace;\n"
        "    float       pad2;\n"
        "} viewport;\n";
    uniformString +=
        "const uint BakeComponent_Shadows=1 << 0;\n"
        "const uint BakeComponent_BaseIllumination=1 << 1;\n"
        "const uint BakeComponent_SeparateIllumination=1 << 2;\n";
    return uniformString;
}
/*!
* \brief
* \param
* \return
*/
std::string SHLChunk::getShaderCameraUniformString()
{
    std::string uniformString = "// camera matrices and other attributes, binding id 1\n";

    if (GLAD_GL_VERSION_4_3)
        uniformString += "layout(std140, binding = 1) uniform OSGCamera\n";
    else
        uniformString += "layout(std140) uniform OSGCamera\n";

    uniformString += "{\n"
        "    mat4 projectionMatrix[2]; // projection matrix for left and right eye\n"
        "    mat4 invProjectionMatrix[2]; // inverse projection matrix for left and right eye\n"
        "    mat4 viewProjectionMatrix[2]; // view projection matrix for left and right eye\n"
        "    mat4 viewMatrix[2]; // view matrix for left and right eye\n"
        "    mat4 invViewMatrix[2]; // inverse view matrix for left and right eye\n"
        "    mat4 unjitteredViewProjectionMatrix[2]; // current frame view projection matrix for left and right eye\n"
        "    mat4 prevViewProjectionMatrix[2]; // previous frame view projection matrix for left and right eye\n"
        "    vec4 screenCenter[2];\n"
        "    vec4 region; // left, right, bottom, top\n"
        "    vec2 aspect; // camera pixel aspect\n"
        "    vec2 resolution; // pixel resolution\n"
        "    vec2 nearfar; // near and far plane\n"
        "    int  headlightActive; // use the headlight or not\n"
        "    int  eye; // the eye currently rendered\n"
        "    OSGLightSource  headlight;\n"
        "    OSGFog fog; \n"
        "    float frameTime; // the current frame time\n"
        "    float previousFrameTime; // the previous frame time\n"
        "    float depthBias;\n"
        "    int frameIdx; // current frame index \n"
        "} camera;\n";
    return uniformString;
}
/*!
* \brief
* \param
* \return
*/
std::string SHLChunk::getShaderObjectUniformString()
{
    std::string uniformString = "// geometry matrices and other attributes, binding id 2\n";
#ifdef USE_BINDLESS_LIGHTTEXTURE
    uniformString += "struct lightmapData{\n"
        "#ifdef GLAD_GL_ARB_bindless_texture\n"
        "   sampler2D lightmap;\n"
        "#else \n"
        "   int lightmapPad0; \n"
        "   int lightmapPad1; \n"
        "#endif\n"
        "   int lightmapInfo;\n"
        "   int padding;\n"
    "};\n";
#else
    uniformString += "struct lightmapData{\n"
        "   int lightmapInfo;\n"
        "   int padding0;\n"
        "   int padding1;\n"
        "   int padding2;\n"
        "};\n";
#endif

    if (GLAD_GL_VERSION_4_3)
        uniformString += "layout(std140, binding = 2) uniform OSGGeometry\n";
    else
        uniformString += "layout(std140) uniform OSGGeometry\n";

    uniformString += "{\n"
        "    mat4 worldMatrix; // object to world matrix\n"
        "    mat4 prevWorldMatrix; // object to world matrix\n"
        "    mat4 invWorldMatrix; // inverse object to world matrix\n"
        "    mat4 transInvWorldMatrix; // transposed inverse world matrix aka normal matrix\n"
        "    mat4 objectToTextureMatrix; // object to texture space matrix\n"
        "    mat4 textureToObjectMatrix; // texture to object space matrix\n"
        "    vec4 clippingPlanes[6]; // clipping planes\n"
        "    ivec4 activeLightIndices[16]; // index buffer, 16 byte aligned but to get it tightly packed it is an ivec4, allowing 64 local lights\n"
        "    int  clippingPlaneCount; // number of clipping planes\n"
        "    int  activeLightCount; // number of active lightsources\n"
        "    int  deformerActive; // whether to use deformer data\n"
        "    int  numLightmaps; //number of lightmaps used\n"
        "    lightmapData lightmap[4]; //lightmap data\n"
        "    float updateTime; // the time of the current update\n"
        "    float alpha; // visible in alpha multiplier \n"
        "    float pad1; \n"
        "    float pad2; \n"
        //"    int unused[2]; // for future use\n"
        "} object;\n";

    //constants for lightmapInfo 
#ifndef USE_BINDLESS_LIGHTTEXTURE
    uniformString += "uniform sampler2D OSGLightMaps;\n";
    uniformString += "uniform sampler2D OSGLightMaps1;\n";
#endif
    uniformString +=  "const int lightmap_info_lightAndShadows=1;\n"
        "const int lightmap_info_additionalLight=2;\n";

    // deformer matrices for skinning
    if (GLAD_GL_VERSION_4_3)
        uniformString += "layout(std140, binding = 7) uniform OSGDeformer\n";
    else
        uniformString += "layout(std140) uniform OSGDeformer\n";
    uniformString += "{\n"
        "    mat4 matrix[NUMDEFORMERMATRICES]; // the deformer matrix array\n"
        "} deformerMatrices;\n";

#if VOLUME_SCATTER_AVAILABLE
    if (GLAD_GL_VERSION_4_3)
        uniformString += "layout(std140, binding = 8) uniform OSGScatterVolumes\n";
    else
        uniformString += "layout(std140) uniform OSGScatterVolumes\n";
    uniformString += "{\n"
        "    OSGScatterVolume  volumes[MAX_SCATTER_VOLUME_MATERIALS]; // the scatter volume array\n"
        "} scatterVolumes;\n";

    if (GLAD_GL_VERSION_4_3)
        uniformString += "layout(std140, binding = 9) uniform OSGVDBVolumes\n";
    else
        uniformString += "layout(std140) uniform OSGVDBVolumes\n";
    uniformString += "{\n"
        "    OSGVDBVolume       volumes[MAX_VDB_VOLUME_MATERIALS]; // the scatter volume array\n"
        "} openVDBVolumes;\n";
    if (GLAD_GL_VERSION_4_3)
        uniformString += "layout(std140, binding = 10) uniform OSGScatterVolumeInstances\n";
    else
        uniformString += "layout(std140) uniform OSGScatterVolumeInstances\n";
    uniformString += "{\n"
        "    int                instance_count; \n"
        "    int                padding0; \n"
        "    int                padding1; \n"
        "    int                padding2; \n"
        "    OSGVolumeInstance  instances[MAX_SCATTER_VOLUME_INSTANCES]; \n"
        "} scatterVolumeInstances;\n";
    if (GLAD_GL_VERSION_4_3)
        uniformString += "layout(std140, binding = 11) uniform OSGOpenVDBVolumeInstances\n";
    else
        uniformString += "layout(std140) uniform OSGOpenVDBVolumeInstances\n";
    uniformString += "{\n"
        "    int                instance_count; \n"
        "    int                padding0; \n"
        "    int                padding1; \n"
        "    int                padding2; \n"
        "    OSGVolumeInstance  instances[MAX_VDB_VOLUME_INSTANCES]; \n"
        "} openVDBVolumeInstances;\n";

    if (GLAD_GL_VERSION_4_3)
        uniformString += "layout(std140, binding = 12) uniform OSGVolumeLightIndices\n";
    else
        uniformString += "layout(std140) uniform OSGVolumeLightIndices\n";
    uniformString += "{\n"
        "    int                numLights; \n"
        "    int                numVolumeOnlyLights; \n"
        "    int                padding1; \n"
        "    int                padding2; \n"
        "    uvec4              indices[MAX_VOLUME_LIGHTS / 4]; \n"
        "} volumeLightIndices;\n";
#endif
    return uniformString;
}

/*!
* \brief
* \param
* \return
*/
std::string SHLChunk::getShaderLightUniformString()
{
    std::string uniformString = "// global lightsource buffer,  binding id 3\n";
    if (GLAD_GL_VERSION_4_3)
        uniformString += "layout(std140, binding = 3) uniform OSGLightSources\n";
    else
        uniformString += "layout(std140) uniform OSGLightSources\n";

    uniformString += "{\n"
        "    int             activeGlobalLightCount; // number of active global lightsources\n"
        "    int             activeGeometryLightCount; // number of active geometry lightsources\n"
        "   int padding0;\n"
        "   int padding1;\n "
        "    OSGLightSource  light[NUMLIGHTS]; // lightsources \n"
        "} lights;\n";


        uniformString +=  "#ifndef GLAD_GL_ARB_bindless_shadow_texture\n"
                        "uniform sampler2DArrayShadow OSGShadowMaps;\n"
                        "uniform samplerCubeArrayShadow OSGShadowCubeMaps;\n"
                        "#endif\n";
  
  
    if (GLAD_GL_VERSION_4_3 && GLAD_GL_ARB_bindless_texture)
    {
        uniformString += "layout(std140, binding = 5) uniform OSGLightSourceTextures\n"
            "{\n"
            "    OSGLightSourceTexture textures[NUMLIGHTS];\n"
            "} lightTextures;\n";
    }
    return uniformString;
}
/*!
* \brief
* \param
* \return
*/
std::string SHLChunk::getShaderLensFlareUniformString()
{
    std::string uniformString = "// global lensflare buffer,  binding id 9\n";
    if (GLAD_GL_VERSION_4_3)
        uniformString += "layout(std140, binding = 6) uniform OSGLensFlare\n";
    else
        uniformString += "layout(std140) uniform OSGLensFlare\n";

    uniformString += "{\n"
        "    mat4 worldMatrix;\n"
        "    mat4 lightMatrix;\n"
        "    vec4 color;\n"
        "    int castsLensFlare;\n"
        "    float radius;\n"
        "    float angularIntensity;\n"
        "    int lightIndex;\n"
        "    float useLightProperties;\n"
        "    float intensity;\n"
        "    int padding1;\n"
        "    int padding2;\n"     
        "} lensFlare;\n";

    if (GLAD_GL_VERSION_4_3)
        uniformString += "layout(std140, binding = 7) uniform OSGLensFlareRings\n";
    else
        uniformString += "layout(std140) uniform OSGLensFlareRings\n";

    uniformString +=    
        "{\n"
        "    int numberOfRings;\n"
        "    int padding1;\n"
        "    int padding2;\n"
        "    int padding3;\n"
        "    OSGLensFlareRing rings[MAX_LENSFLARE_STARS];\n"
        "} lensFlareRings;\n";


    if (GLAD_GL_VERSION_4_3)
        uniformString += "layout(std140, binding = 8) uniform OSGLensFlareGlows\n";
    else
        uniformString += "layout(std140) uniform OSGLensFlareGlows\n";

    uniformString +=    
        "{\n"
        "    int numberOfGlows;\n"
        "    int padding1;\n"
        "    int padding2;\n"
        "    int padding3;\n"
        "    OSGLensFlareGlow glows[MAX_LENSFLARE_STARS];\n"
        "} lensFlareGlows;\n";

    if (GLAD_GL_VERSION_4_3)
        uniformString += "layout(std140, binding = 9) uniform OSGLensFlareStars\n";
    else
        uniformString += "layout(std140) uniform OSGLensFlareStars\n";

    uniformString +=
        "{\n"
        "    int numberOfStars;\n"
        "    int padding1;\n"
        "    int padding2;\n"
        "    int padding3;\n"
        "    OSGLensFlareStar stars[MAX_LENSFLARE_STARS];\n"
        "} lensFlareStars;\n";

    if (GLAD_GL_VERSION_4_3)
        uniformString += "layout(std140, binding = 10) uniform OSGLensFlareGhosts\n";
    else
        uniformString += "layout(std140) uniform OSGLensFlareGhosts\n";

    uniformString +=
        "{\n"
        "    int numberOfGhosts;\n"
        "    int padding1;\n"
        "    int padding2;\n"
        "    int padding3;\n"
        "    OSGLensFlareGhost ghosts[MAX_LENSFLARE_GHOSTS];\n"
        "} lensFlareGhosts;\n";

    if (GLAD_GL_VERSION_4_3)
        uniformString += "layout(std140, binding = 11) uniform OSGLensFlareStreaks\n";
    else
        uniformString += "layout(std140) uniform OSGLensFlareStreaks\n";

    uniformString +=
        "{\n"
        "    int numberOfStreaks;\n"
        "    int padding1;\n"
        "    int padding2;\n"
        "    int padding3;\n"
        "    OSGLensFlareStreak streaks[MAX_LENSFLARE_STARS];\n"
        "} lensFlareStreaks;\n";

    if (GLAD_GL_VERSION_4_3)
        uniformString += "layout(std140, binding = 12) uniform OSGLensFlareTextureElements\n";
    else
        uniformString += "layout(std140) uniform OSGLensFlareTextureElements\n";

    uniformString +=
        "{\n"
        "    OSGLensFlareTextureElement textureElements[MAX_LENSFLARE_GHOSTS];\n"
        "} lensFlareTextureElements;\n";

    return uniformString;
}
/*!
* \brief
* \param
* \return
*/
std::string SHLChunk::getShaderEnvironmentUniformString()
{
    std::string uniformString = "// global environment buffer,  binding id 6\n";

    if (GLAD_GL_VERSION_4_3)
        uniformString += "layout(std140, binding = 6) uniform OSGEnvironment\n";
    else
        uniformString += "layout(std140) uniform OSGEnvironment\n";

    uniformString += "{\n"
        "    mat4  envTransform; // environment transformation matrix\n"
        "    mat4  invEnvTransform; // inverse environment transformation matrix\n"
        "    mat4  envColorMatrix;  // environment color correction matrix\n"
        "    vec4  averageEnvColor; // averate environment color\n"
        "    float exposure;  // exposure setting\n"
        "    float saturation; // saturation\n"
        "    float whitepoint;  // whitepoint\n"
        "    float environmentSize;  // the size of the environment\n"
        "	 vec3 sunDirection; // sun direction for skylight environment \n"
        "	 float sunAngle; // sun angle for skylight environment \n"
        "	 vec3 sunColor; // sun color for skylight environment \n"
        "	 int  lightCount; // number of lights \n"
        "    uint uDistCdfSize; // uDistribution CDF size\n"
        "    uint vDistCdfSize; // vDistribution CDF size\n"
        "	 int  padding0;\n"
        "	 int  padding1;\n"
        "    OSGLightSource  lights[NUMLIGHTS]; // environment lightsources \n"
        "} environment;\n"
        "uniform samplerCube envMap;  // specular environment Cubemap\n"
        "uniform samplerCube ambientEnvMap;  // ambient environment Cubemap (for when shadow lightsources are enabled)\n"
        "uniform sampler2D irradianceMap;  // Pre-computed convoluted irradiance cube-map\n"
        "uniform sampler3D filteredEnvMap;  // Pre-computed convoluted environment cube-map\n"
        "uniform sampler2D brdfMap;  // Pre-computed BRDF look-up map\n"
        "uniform sampler2D sheenBrdfMap;  // Pre-computed sheen BRDF look-up map\n";
        //"uniform sampler3D filteredEnvMap; // filtered map for diffuse and glossy lookup\n";
    return uniformString;
}

/*!
* \brief
* \param
* \return
*/
std::string SHLChunk::getShaderRandomSamplesUniformString()
{
    std::string uniformString = "// global random samples buffer,  binding id 4\n";

    if (GLAD_GL_VERSION_4_3)
        uniformString += "layout(std140, binding = 4) uniform OSGRandomSamples\n";
    else
        uniformString += "layout(std140) uniform OSGRandomSamples\n";

    uniformString += "{\n"
        "    vec4 value[4096]; // random samples \n"
        "} random;\n";
    return uniformString;
}

/*!
* \brief Returns the shader string for the screen space ambient occlussion texture uniform.
* \param
* \return
*/
std::string SHLChunk::getShaderAmbientOcclussionUniformString()
{
    static std::string uniformString =  "// global screen space ambient occlussion texture\n"
                                        "uniform sampler2D OSGAmbientOcclusion;\n";
    return uniformString;
}

/*!
* \brief
* \param
* \return
*/
std::string SHLChunk::getShaderUniformString()
{

    //TODO
    std::string temp = "uniform sampler2D OSGVolumetricAttenuationTexture;\nuniform sampler2D OSGVolumetricInscatterTexture;\n";
    static std::string shaderUniformString = getShaderStructUniformString() + getShaderViewportUniformString() + getShaderCameraUniformString() + getShaderEnvironmentUniformString() +
        getShaderRandomSamplesUniformString() + getShaderObjectUniformString() + getShaderAmbientOcclussionUniformString() + temp;
    
    return shaderUniformString;
}

std::string SHLChunk::getShaderDefinesString(ProgramType programType, bool singleViewOnly)
{
    std::string defines;
    if (GLAD_GL_VERSION_4_3 && GLAD_GL_ARB_bindless_texture)
    {
        defines += "#extension GL_ARB_bindless_texture : require\n";
    }
    if (GLAD_GL_NV_viewport_array2)
    {
        defines += "#extension GL_NV_viewport_array2 : enable\n";
    }
    if (programType == kGeometryProgram)
    {
        if (GLAD_GL_NV_geometry_shader_passthrough)
        {
            defines += "#extension GL_NV_geometry_shader_passthrough : require\n";
        }
    }
    if (GLAD_GL_NV_viewport_array2 && s_hasStereoViewRenderingSupport && !singleViewOnly)
    {
        if (programType == kVertexProgram || (programType == kGeometryProgram))
        {
            defines += "#extension GL_NV_stereo_view_rendering : require\n";
        }
    }
    
    if (GLAD_GL_VERSION_4_3 && GLAD_GL_ARB_bindless_texture)
    {
        defines += "#define GLAD_GL_ARB_bindless_texture\n";
        if(getUseBindlessShadowTextures())
            defines += "#define GLAD_GL_ARB_bindless_shadow_texture\n";
        defines += "layout(bindless_sampler) uniform;\n";
#ifdef USE_BINDLESS_LIGHTTEXTURE
        defines += "#define USE_BINDLESS_LIGHTTEXTURE;\n";
#endif

    }
    if( getFogEnabled())
    {
        defines += "#define USE_FOG\n";
    }
    if (GLAD_GL_NV_viewport_array2)
    {
        defines += "#define GLAD_GL_NV_viewport_array2\n";
    }

    if (GLAD_GL_NV_viewport_array2 && s_hasLensMatchedShadingSupport && !singleViewOnly)
    {
        defines += "#define USE_LENS_MATCHED_SHADING\n";
    }
    if (programType == kGeometryProgram)
    {
        if (GLAD_GL_NV_geometry_shader_passthrough)
        {
            defines += "#define GLAD_GL_NV_geometry_shader_passthrough\n";
        }

    }
    if (GLAD_GL_NV_viewport_array2 && s_hasStereoViewRenderingSupport && !singleViewOnly)
    {
        if (programType == kVertexProgram || (programType == kGeometryProgram))
        {
            defines += "#define GLAD_GL_NV_stereo_view_rendering\n";
            defines += "layout(secondary_view_offset = 1) out highp int gl_Layer;\n";
        }
        else if (programType == kFragmentProgram)
        {
            defines += "#define GLAD_GL_NV_stereo_view_rendering\n";
            defines += "in highp int gl_Layer;\n";
            defines += "int eye = gl_Layer;\n";
        }
    }
    else
    {
        defines += "const int eye = 0;\n";
    }

    if (s_uniformBlockSize < 65535)
    {
        defines += "#define NUMLIGHTS 64\n";
        defines += "#define NUMDEFORMERMATRICES 256\n";
        defines += "#define MAX_SCATTER_VOLUME_INSTANCES 8\n";
        defines += "#define MAX_SCATTER_VOLUME_MATERIALS 8\n";
        defines += "#define MAX_VDB_VOLUME_INSTANCES 21\n";
        defines += "#define MAX_VDB_VOLUME_MATERIALS 21\n";
        defines += "#define MAX_VDB_VOLUME_CHANNELS 8\n";
        defines += "#define MAX_VOLUME_LIGHTS 16\n";
        defines += "#define MAX_LENSFLARE_GHOSTS 128\n";
        defines += "#define MAX_LENSFLARE_STARS 64\n";
    }
    else
    {
        defines += "#define NUMLIGHTS 128\n";
        defines += "#define NUMDEFORMERMATRICES 1024\n";
        defines += "#define MAX_SCATTER_VOLUME_INSTANCES 8\n";
        defines += "#define MAX_SCATTER_VOLUME_MATERIALS 8\n";
        defines += "#define MAX_VDB_VOLUME_INSTANCES 21\n";
        defines += "#define MAX_VDB_VOLUME_MATERIALS 21\n";
        defines += "#define MAX_VDB_VOLUME_CHANNELS 8\n";
        defines += "#define MAX_VOLUME_LIGHTS 16\n";
        defines += "#define MAX_LENSFLARE_GHOSTS 128\n";
        defines += "#define MAX_LENSFLARE_STARS 64\n";
    }

    if (s_isAMD)
    {
        defines += "#define AMD_GPU\n";
    }
    else if (s_isIntel)
    {
        defines += "#define INTEL_GPU\n";
    }
    else if( s_isNvidia)
    {
         defines += "#define NVIDIA_GPU\n";
    }
    defines += "#define LUMINANCE_R 0.212671\n";
    defines += "#define LUMINANCE_G 0.715160\n";
    defines += "#define LUMINANCE_B 0.072169\n";

#ifdef __APPLE__
    defines += "#define MAC\n";
#endif
    return defines;
}

/*!
* \brief
* \param
* \return
*/
std::string SHLChunk::getShaderHeaderString(ProgramType programType, bool includeVersion, bool includeLights, bool includeLensFlares)
{
    static std::string lensFlaresWithLights = getShaderStructUniformString(true) + getShaderViewportUniformString() + 
                                    getShaderCameraUniformString() + getShaderRandomSamplesUniformString() + getShaderAmbientOcclussionUniformString() + 
                                    getShaderLightUniformString() + getShaderLensFlareUniformString();

    static std::string lensFlaresWithoutLights = getShaderStructUniformString(true) + getShaderViewportUniformString() + 
                                    getShaderCameraUniformString() + getShaderRandomSamplesUniformString() + getShaderAmbientOcclussionUniformString() + 
                                    getShaderLensFlareUniformString();

    static std::string noLensFlaresWithLights = getShaderUniformString() + getShaderLightUniformString();
    static std::string noLensFlaresNoLights = getShaderUniformString();

    if(includeLensFlares)
    {
        if (includeLights)
            return lensFlaresWithLights;
        return lensFlaresWithoutLights;
    }
    else
    {
        if (includeLights)
            return noLensFlaresWithLights;
        return noLensFlaresNoLights;
    }
    
#if 0
    std::string shaderHeaderString;
    if(includeLensFlares)
    {
        shaderHeaderString = getShaderStructUniformString(includeLensFlares) + getShaderViewportUniformString() + getShaderCameraUniformString() + getShaderRandomSamplesUniformString() + getShaderAmbientOcclussionUniformString();
    }
    else
    {
        shaderHeaderString = getShaderUniformString();
    }

    if (includeLights)
        shaderHeaderString += getShaderLightUniformString();

    if (includeLensFlares)
        shaderHeaderString += getShaderLensFlareUniformString();
        
    return shaderHeaderString;
#endif
}

void SHLChunk::dumpShaderSource(const std::string name) const
{
//#if 0
#ifndef VR_RELEASE_BUILD
    if (!getVertexProgram().empty())
    {
        std::string vertexProg = getShaderVersionString() + getShaderDefinesString(kVertexProgram, m_singleViewRenderingOnly) + getVertexProgram();
        std::string vertexFile = name + "_vertex.glsl";
        std::ofstream outStream(vertexFile.c_str(), std::ios::out|std::ios::binary);
        outStream << vertexProg << std::endl;
        outStream.close();
    }
    if (!getGeometryProgram().empty())
    {
        std::string geoProg = getShaderVersionString() + getShaderDefinesString(kGeometryProgram, m_singleViewRenderingOnly) + getGeometryProgram();
        std::string geoFile = name + "_geometry.glsl";
        std::ofstream outStream(geoFile.c_str(), std::ios::out|std::ios::binary);
        outStream << geoProg << std::endl;
        outStream.close();
    }
    if (!getFragmentProgram().empty())
    {
        std::string fragmentProg = getShaderVersionString() + getShaderDefinesString(kFragmentProgram, m_singleViewRenderingOnly) + getFragmentProgram();
        std::string fragmentFile = name + "_fragment.glsl";
        std::ofstream outStream(fragmentFile.c_str(), std::ios::out|std::ios::binary);
        outStream << fragmentProg << std::endl;
        outStream.close();
    }
#endif
//#endif
}

/*------------------------------------------------------------------------*/
/*                              cvs id's                                  */

#ifdef OSG_SGI_CC
#pragma set woff 1174
#endif

#ifdef OSG_LINUX_ICC
#pragma warning( disable : 177 )
#endif

namespace
{
    static Char8 cvsid_cpp[] = "@(#)$Id: OSGSHLChunk.cpp,v 1.68 2008/12/12 11:28:45 a-m-z Exp $";
    static Char8 cvsid_hpp[] = OSGSHLCHUNKBASE_HEADER_CVSID;
    static Char8 cvsid_inl[] = OSGSHLCHUNKBASE_INLINE_CVSID;

    static Char8 cvsid_fields_hpp[] = OSGSHLCHUNKFIELDS_HEADER_CVSID;
}

#ifdef __sgi
#pragma reset woff 1174
#endif
