/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *               Copyright (C) 2000-2002 by the OpenSG Forum                 *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

#ifndef _OSGSHLCHUNK_H_
#define _OSGSHLCHUNK_H_
#pragma once

#include <OSGConfig.h>

#include <string>
#include <vector>
#include <map>
#include <set>
#include <string>

#include "OSGDrawActionBase.h"
#include "OSGWindow.h"

#include <OSGSHLChunkBase.h>

// Uniform Buffer Ids
#ifndef UBO_VIEWPORT
#define UBO_VIEWPORT 0
#endif

#ifndef UBO_CAMERA
#define UBO_CAMERA 1
#endif

#ifndef UBO_OBJECT 
#define UBO_OBJECT 2
#endif

#ifndef UBO_LIGHTS
#define UBO_LIGHTS 3
#endif

#ifndef UBO_RANDOM
#define UBO_RANDOM 4
#endif

#ifndef UBO_LIGHTTEXTURES
#define UBO_LIGHTTEXTURES 5
#endif

#ifndef UBO_ENVIRONMENT
#define UBO_ENVIRONMENT 6
#endif

#ifndef UBO_DEFORMER
#define UBO_DEFORMER 7
#endif

#ifndef UBO_SCATTER_VOLUME
#define UBO_SCATTER_VOLUME 8
#endif

#ifndef UBO_VDB_VOLUME
#define UBO_VDB_VOLUME 9
#endif

#ifndef UBO_SCATTER_VOLUME_INSTANCE
#define UBO_SCATTER_VOLUME_INSTANCE 10
#endif

#ifndef UBO_VDB_VOLUME_INSTANCE
#define UBO_VDB_VOLUME_INSTANCE 11
#endif

#ifndef UBO_VOLUME_LIGHT_INDICES
#define UBO_VOLUME_LIGHT_INDICES 12
#endif


#ifndef UBO_LENSFLARES
#define UBO_LENSFLARES 6
#endif

#ifndef UBO_LENSFLARE_RINGS
#define UBO_LENSFLARE_RINGS 7
#endif

#ifndef UBO_LENSFLARE_GLOWS
#define UBO_LENSFLARE_GLOWS 8
#endif

#ifndef UBO_LENSFLARE_STARS
#define UBO_LENSFLARE_STARS 9
#endif

#ifndef UBO_LENSFLARE_GHOSTS
#define UBO_LENSFLARE_GHOSTS 10
#endif

#ifndef UBO_LENSFLARE_STREAKS
#define UBO_LENSFLARE_STREAKS 11
#endif

#ifndef UBO_LENSFLARE_TEXTURE_ELEMENTS
#define UBO_LENSFLARE_TEXTURE_ELEMENTS 12
#endif

#ifndef SSBO_MDL_RO_DATA_SEGMENT
#define SSBO_MDL_RO_DATA_SEGMENT 13
#endif

#ifndef SSBO_MDL_ARGUMENT_BLOCK
#define SSBO_MDL_ARGUMENT_BLOCK 14
#endif

#ifndef SSBO_VDB_DATA_BLOCK
#define SSBO_VDB_DATA_BLOCK 15
#endif

#ifndef MAX_OPENGL_ENVIRONMENT_LIGHTS
#define MAX_OPENGL_ENVIRONMENT_LIGHTS 128
#endif

// should match the scatter volume instance define in vrVolumeTypes.h
#ifndef MAX_OPENGL_SCATTER_VOLUME_INSTANCES
#define MAX_OPENGL_SCATTER_VOLUME_INSTANCES 8
#endif

// should match the vdb volume instance define in vrVolumeTypes.h
#ifndef MAX_OPENGL_VDB_VOLUME_INSTANCES
#define MAX_OPENGL_VDB_VOLUME_INSTANCES 21
#endif

#ifndef MAX_OPENGL_SCATTER_VOLUME_MATERIALS
#define MAX_OPENGL_SCATTER_VOLUME_MATERIALS 8
#endif

#ifndef MAX_OPENGL_VDB_VOLUME_MATERIALS
#define MAX_OPENGL_VDB_VOLUME_MATERIALS 21
#endif

#ifndef MAX_OPENGL_VDB_VOLUME_CHANNELS
#define MAX_OPENGL_VDB_VOLUME_CHANNELS 8
#endif

#ifndef MAX_OPENGL_VOLUME_LIGHTS
#define MAX_OPENGL_VOLUME_LIGHTS 16
#endif

#ifndef MAX_OPENGL_LIGHTS_SMALL
#define MAX_OPENGL_LIGHTS_SMALL 64
#endif

#ifndef MAX_OPENGL_LIGHTS
#define MAX_OPENGL_LIGHTS 128
#endif

#ifndef MAX_OPENGL_LOCAL_LIGHTS
#define MAX_OPENGL_LOCAL_LIGHTS 64
#endif

#ifndef MAX_DEFORMER_MATRICES
#define MAX_DEFORMER_MATRICES 1024
#endif
#ifndef MAX_DEFORMER_MATRICES_SMALL
#define MAX_DEFORMER_MATRICES_SMALL 256
#endif

#ifndef MAX_LIGHTMAP_TEXTURES_PER_NODE
#define MAX_LIGHTMAP_TEXTURES_PER_NODE 4
#endif

#ifndef MAX_OPENGL_CLIPPING_PLANES
#define MAX_OPENGL_CLIPPING_PLANES 6
#endif

typedef struct
{
    osg::Matrix		 worldMatrix; // the world transform matrix. Light direction = normalize( mat3(worldMatrix) * vec3(0.0, 0.0, -1.0)), position = worldMatrix[4].xyz; ??
    osg::Matrix		 lightMatrix; // the light matrix for shadow mapping. it is basically the world matrix combined with a projection matrix
    osg::Vec3f       diffuseColor; // diffuse color of the light
    int              useColorTexture; // flag for evaluating a color texture
    osg::Vec3f       glossyColor; // glossy color of the light
    int              flags; // flags for some evaluation terms
    osg::Vec3f       attenuation; // attenuation: x = constant, y = linear, z = quadratic/realistic, w = undefined, used for padding
    float            specularVisibility; // specular visibility factor, should be either 0 or 1
    osg::Vec2f       spotCone; // the spotlight cone description: x = cos(innerCone), y = cos(outerCone) . Calculate falloff as smoothstep(cosOuter, cosInner, dot( -L, light.direction));
    osg::Vec2f       nearFar; // near/far clipping used for shadowmap rendering
    int              lightType; // int specifying the lightsource type. 
    float            shadowIntensity; // shadow intensity scale
    float            shadowMaterialShadowIntensity; // shadow intensity on the shadow material
    float            shadowFilterRadius; // radius to be used for shadow map filtering
    int              shadowMapIndex; // index refering to the shadow map
    int              shadowMapResolution; //resolution of the shadow map
    float            flip;
    int              hasShadowMap; // flag indicating if a bindless shadowmap is present
    osg::UInt64      shadowTexture; // shadow map when using bindless textures
    osg::UInt64      shadowCubeTexture; // shadow cubemap when using bindless textures*/
    osg::UInt64      prefilteredTexture;  //
    int              pad0;
    int              pad1;
} vrGLLightSource;

typedef struct
{
    osg::Matrix		 lightMatrix; // the light projection matrix 
    osg::Matrix		 invWorldMatrix; // the inverse world transform matrix. 
    osg::Vec2f       rotate; // rotate value for the color texture
    osg::Vec2f       repeat; // repeat value for the color texture
    osg::Vec2f       offset; // offset value for the color texture
    osg::Vec2f       reserved; // alignment purposes
    osg::UInt64      texture; // address refering to the color texture
    osg::UInt64      iesProfile; // address refering to an ies profile texture

} vrGLLightTexture;

typedef struct
{
    osg::Matrix worldToObjectMatrix;
    osg::Vec3f minBox;
    int volumeIndex;
    osg::Vec3f maxBox;
    int isRotated;
    osg::Vec3f halfScatterDim;
    int pad0;
} vrGLVolumeInstance;

typedef struct
{
    osg::Vec3f   sigmaS;
    float        g;
    osg::Vec3f   sigmaT;
    unsigned int quality;
    int          flags;
    int          pad0;
    int          pad1;
    int          pad2;
} vrGLVolume;

typedef struct
{
    vrGLVolume volumeProperties;
    osg::Vec3f noiseFrequency;
    float      noiseLacunarity;
    osg::Vec3f falloffOffset;
    float      noiseTime;
    osg::Vec3f falloffScale;
    float      noiseGainExponent;
    float      gradientCoefficient;
    int        noiseOctaves;
    float      falloffRateExponent;
    int        pad0;
} vrGLScatterVolume;

typedef struct
{
    vrGLVolume volumeProperties;
    int        numChannels;
    int        scatterAndAbsorptionChannel;
    int        emissionChannel;
    float      sigmaE;
    unsigned int vdbBufferOffset;
    float     emissionChannelMin;
    float     emissionChannelMax;
    int       emissionChannelTexLayer;
    unsigned int channelOffsets[MAX_OPENGL_VDB_VOLUME_CHANNELS];
} vrGLOpenVDBVolume;


typedef struct
{
    osg::Vec4f colorDistance;
    osg::Vec4f colorHeight;
    osg::Vec4f noiseSizeDistance;    // Distance  noise size: x,y,z,unused
    osg::Vec4f noiseOffsetDistance;    // Distance  noise offset: x,y,z,unused
    osg::Vec4f noiseSizeHeight;      // Height  noise size: x,y,z,unused
    osg::Vec4f noiseOffsetHeight;      // Height  noise offset: x,y,z,unused
    osg::Vec2f densityNoiseDistance; // Distance : density, noise. Noise == 0 disables noise
    osg::Vec2f densityNoiseHeight;   // Height : density, noise. Noise == 0 disables noise
    osg::Vec2f rangeDistance;
    osg::Vec2f rangeHeight;
    int  activeDistance;
    int  activeHeight;
    int  falloffDistance;            // realistic,linear
    int  falloffHeight;              // realistic,linear
    
    float blendHeight;
    int   reserved0;
    int   reserved1;
    int   reserved2;
} vrGLFog;

typedef struct
{
    osg::Matrix projectionMatrix[2]; // projection matrices for left and right eye
    osg::Matrix invProjectionMatrix[2]; // projection matrices for left and right eye
    osg::Matrix viewProjectionMatrix[2]; // combined view projection matrices for left and right eye
    osg::Matrix viewMatrix[2]; // view matrix for left and right eye
    osg::Matrix invViewMatrix[2]; // inverse view matrix for left and right eye
    osg::Matrix unjitteredViewProjectionMatrix[2]; // current frame view projection matrix for left and right eye
    osg::Matrix prevViewProjectionMatrix[2]; // previous frame view projection matrix for left and right eye
    osg::Vec4f  screenCenter[2]; 
    osg::Vec4f  region; // left, right, bottom, top
    osg::Vec2f  aspect; // aspect ratio
    osg::Vec2f  resolution; // pixel resolution
    osg::Vec2f  nearFar; // near and far plane
    int  headlightActive; // use the headlight or not
    int  eye; // eye index
    vrGLLightSource  headlight;
    vrGLFog          fog;
    float       frameTime; // the current frame time
    float       previousFrameTime; // the previous frame time
    float       depthBias; // bias scale for the depth value
    int frameIdx; // current frame index
} vrGLCamera;

typedef struct
{
    osg::Matrix rec709toRCS; // colorspace matrix
    osg::Vec4f region; // left, right, bottom, top
    osg::Vec4f pixelRegion; // left, right, bottom, top
    osg::Vec2f pixelResolution; // pixelwidth, pixelheight
    osg::Vec2f aspect; // aspect ratio
    float      superSamplingScale;
    float      displayLuminance;
    float      physicalScaleFactor;
    int        sampleId;
    osg::Vec3f separateBakeScale; // must start at 16 byte boundary
    unsigned   int useBaking;
    float      sceneUnitScale;
    int        maxVolumeSteps;
    int        colorspace; // the rendering colorspace
    float      pad2;
}vrGLViewport;

typedef struct
{
    osg::Vec4f sample[4096]; //random samples
} vrGLRandomSamples;

typedef struct
{
    int                                global_lights_count;
    int                                geometry_lights_count;
    int                                padding1;
    int                                padding2;
    vrGLLightSource                    light[MAX_OPENGL_LIGHTS_SMALL]; // currently limit the number of lights to 8
} vrGLLightSourcesSmall;

typedef struct
{
    int                                global_lights_count;
    int                                geometry_lights_count;
    int                                padding1;
    int                                padding2;
    vrGLLightSource                    light[MAX_OPENGL_LIGHTS]; // currently limit the number of lights to 8
} vrGLLightSources;

typedef struct
{
    vrGLLightTexture                   texture[MAX_OPENGL_LIGHTS];
} vrGLLightSourceTextures;

typedef struct
{
    vrGLScatterVolume   volumes[MAX_OPENGL_SCATTER_VOLUME_MATERIALS];
} vrGLScatterVolumes;

typedef struct
{
    vrGLOpenVDBVolume   volumes[MAX_OPENGL_VDB_VOLUME_MATERIALS];
} vrGLOpenVDBVolumes;

typedef struct
{
    int                 instance_count;
    int                 padding0;
    int                 padding1;
    int                 padding2;
    vrGLVolumeInstance  instances[MAX_OPENGL_SCATTER_VOLUME_INSTANCES];
} vrGLScatterVolumeInstances;

typedef struct
{
    int                 instance_count;
    int                 padding0;
    int                 padding1;
    int                 padding2;
    vrGLVolumeInstance  instances[MAX_OPENGL_VDB_VOLUME_INSTANCES];
} vrGLOpenVDBVolumeInstances;

typedef struct
{
    int                 numLights;
    int                 numVolumeOnlyLights;
    int                 padding1;
    int                 padding2;
    unsigned int        indices[MAX_OPENGL_VOLUME_LIGHTS];
} vrGLVolumeLightIndices;

typedef struct
{
    osg::Matrix  m_envTransform; // environment transformation matrix
    osg::Matrix  m_invEnvTransform; // environment transformation matrix
    osg::Matrix  m_envColorMatrix;  // environment color correction matrix
    osg::Vec4f	 m_averageEnvColor; // averate environment color
    float		 m_exposure;  // exposure setting
    float		 m_saturation;
    float		 m_whitepoint;  // whitepoint
    float		 m_environmentSize;  // the size of the environment
    osg::Vec3f	 m_sunDirection; // sun direction for skylight environment
    float	     m_sunAngle; // sun angle for skylight environment
    osg::Vec3f	 m_sunColor; // sun color for skylight environment
    int			 m_numLights; // number of explicit light sources to evaluate
    unsigned int m_uDistCdfSize; // uDistribution CDF size
    unsigned int m_vDistCdfSize; // vDistribution CDF size
    int          padding0;
    int          padding1;
    vrGLLightSource	m_lights[MAX_OPENGL_ENVIRONMENT_LIGHTS]; // environment lights
} vrGLEnvironment;




OSG_BEGIN_NAMESPACE

/*! \brief SHLChunk class. See \ref
           PageKernelSHLChunk for a description.
*/

class OSG_SYSTEMLIB_DLLMAPPING SHLChunk : public SHLChunkBase
{
private:

    typedef SHLChunkBase Inherited;

    /*==========================  PUBLIC  =================================*/
public:
    enum ProgramType
    {
        kVertexProgram = 0,
        kGeometryProgram = 1,
        kFragmentProgram = 2
    };

    /*---------------------------------------------------------------------*/
    /*! \name                 Chunk Class Access                           */
    /*! \{                                                                 */

    virtual const StateChunkClass * getClass(void) const;

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name              Static Chunk Class Access                       */
    /*! \{                                                                 */

    inline static        UInt32            getStaticClassId(void);
    inline static  const StateChunkClass * getStaticClass(void);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                      Sync                                    */
    /*! \{                                                                 */

    virtual void changed(BitVector  whichField,
        UInt32     origin);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                     Output                                   */
    /*! \{                                                                 */

    virtual void dump(UInt32     uiIndent = 0,
        const BitVector  bvFlags = 0) const;

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                       Paramerters                            */
    /*! \{                                                                 */

    static void updateParameterLocation(OSGGLuint program,
        const ShaderParameterPtr &parameter);

    void updateParameterLocations(const MFShaderParameterPtr &parameters);

    void updateParameters(const MFShaderParameterPtr &parameters,
        bool useProgram = true, bool force = false,
        bool keepProgramActive = false);

    void updateProgramParameters();

    void requestParameterUpdate();

    OSGGLuint   getProgramId();

    OSGGLint    getUniformLocation(const char* name);
    OSGGLint    getUniformBlockIndex(const char* name);

    bool getStereoViewRendering() const;
    void setStereoViewRendering(const bool);

    bool getLensMatchedShadingRendering() const;
    void setLensMatchedShadingRendering(const bool on);

    bool getSingleViewRenderingOnly() const;
    void setSingleViewRenderingOnly(const bool);


    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                       State                                  */
    /*! \{                                                                 */

    virtual void update(DrawActionBase * action);

    virtual void activate(DrawActionBase * action, UInt32 index = 0);

    virtual void deactivate(DrawActionBase * action, UInt32 index = 0);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                    Comparison                                */
    /*! \{                                                                 */
    virtual bool   operator <  (const StateChunk &other) const;

    virtual bool   operator == (const StateChunk &other) const;
    virtual bool   operator != (const StateChunk &other) const;

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                    Parameter Callbacks                       */
    /*! \{                                                                 */

    typedef OSGGLint(OSG_APIENTRY * OSGGLGETUNIFORMLOCATIONARBPROC)
        (OSGGLuint programObj, const char *name);

    typedef void(*parametercbfp) (OSGGLGETUNIFORMLOCATIONARBPROC getUniformLocation,
        DrawActionBase *action, OSGGLuint program);

    typedef void(*osgparametercbfp) (const ShaderParameterPtr &parameter,
        DrawActionBase *action, OSGGLuint program);

    void addParameterCallback(const char *name, parametercbfp fp);
    void addParameterCallback(const char *name, osgparametercbfp fp);

    static void setParameterCallback(parametercbfp fp);
    static parametercbfp getParameterCallback(void);

    typedef void(*programcbfp) (OSGGLuint program);

    static void setCreateProgramCallback(programcbfp fp);
    static programcbfp getCreateProgramCallback(void);

    static void setDeleteProgramCallback(programcbfp fp);
    static programcbfp getDeleteProgramCallback(void);

    static std::string getShaderVersionString();
    static std::string getShaderDefinesString(ProgramType programType, bool singleViewOnly);
    static std::string getShaderStructUniformString(bool includeLensFlares = false);
    static std::string getShaderViewportUniformString();
    static std::string getShaderCameraUniformString();
    static std::string getShaderObjectUniformString();
    static std::string getShaderLightUniformString();
    static std::string getShaderLensFlareUniformString();
    static std::string getShaderEnvironmentUniformString();
    static std::string getShaderRandomSamplesUniformString();
    static std::string getShaderAmbientOcclussionUniformString();
    static std::string getShaderUniformString();
    static std::string getShaderHeaderString(ProgramType programType, bool includeVersion = true, bool includeLights = false, bool includeLensFlares = false);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                    Program Parameter                         */
    /*! \{                                                                 */

    void addProgramParameter(OSGGLenum name, UInt32 value);
    void subProgramParameter(OSGGLenum name);
    void setProgramParameter(OSGGLenum name, UInt32 value);
    UInt32 getProgramParameter(OSGGLenum name);
    std::vector<std::pair<OSGGLenum, UInt32> > getProgramParameters(void);
    void clearProgramParameters(void);

    void addProgramParameterBlockBinding(const char* name, UInt32 binding);
    void subProgramParameterBlockBinding(const char* name);
    void setProgramParameterBlockBinding(const char* name, UInt32 binding);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                    Parameter funcs                           */
    /*! \{                                                                 */


    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                    Cluster                                   */
    /*! \{                                                                 */

    static void setClusterId(Int32 id);
    static void setUniformBlockSize(Int32 blockSize);
    static Int32 getUniformBlockSize();
    static void setStereoViewRenderingEnabled(bool on);
    static bool getStereoViewRenderingEnabled();
    static void setLensMatchedShadingEnabled(bool on);
    static bool getLensMatchedShadingEnabled();
    static void setUseBindlessShadowTextures(bool on);
    static bool getUseBindlessShadowTextures();
    static bool getFogEnabled();
    static void setFogEnabled(const bool on);

    static void setGPUType(bool isIntel, bool isAMD, bool isNvidia);

    void dumpShaderSource(const std::string name) const;
    /*! \}                                                                 */
    /*=========================  PROTECTED  ===============================*/
protected:

    // Variables should all be in SHLChunkBase.

    void onCreate(const SHLChunk *source = NULL);
    void onDestroy(void);

    /*---------------------------------------------------------------------*/
    /*! \name                  Constructors                                */
    /*! \{                                                                 */

    SHLChunk(void);
    SHLChunk(const SHLChunk &source);

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name                   Destructors                                */
    /*! \{                                                                 */

    virtual ~SHLChunk(void);

    /*! \}                                                                 */

    /*==========================  PRIVATE  ================================*/
private:

    friend class FieldContainer;
    friend class SHLChunkBase;

    // class. Used for indexing in State
    static StateChunkClass _class;

    /*---------------------------------------------------------------------*/
    /*! \name            OpenGL Extension Handling                         */
    /*! \{                                                                 */


    static Int32 _clusterId;

    static bool  _unknown_parameter_warning;

    /*! \}                                                                 */

    static void initMethod(void);

    // prohibit default functions (move to 'public' if you need one)
    void operator =(const SHLChunk &source);

    void parseUniformVariables(void);
    bool hasUniform(const std::string &name);

    void updateProgram();
    bool recompileRequired() const;

    static void updateClusterId(const ShaderParameterPtr &parameter,
        DrawActionBase *action, OSGGLuint program);

    typedef std::vector<
        std::pair<std::pair<parametercbfp,
        osgparametercbfp>,
        ShaderParameterPtr          > > OSGParametersCallbacks;
    typedef std::map<std::string, std::pair<parametercbfp, osgparametercbfp> >
        UserParameterCallbacksMap;

    OSGParametersCallbacks     _osgParametersCallbacks;
    UInt32                     _oldParameterSize;
    UInt32                      m_vertexHash;
    UInt32                      m_geometryHash;
    UInt32                      m_fragmentHash;

    std::set<std::string>       _uniforms;
    std::vector<osg::UInt32>    _subroutines;

    osg::UInt32                 m_program;
    bool                        m_programChanged;
    bool                        m_programParameterChanged;
    bool                        m_stereoViewRendering;
    bool                        m_lensMatchedShadingRendering;
    bool                        m_singleViewRenderingOnly;
    bool                        m_useBindlessShadowTextures;
    bool                        m_useFog;

    static osg::UInt32          s_currentProgram;
    static osg::Int32           s_currentGeometryInfoUniform;

    UserParameterCallbacksMap _userParameterCallbacks;
    static parametercbfp      _userParametersCallback;

    static programcbfp        _userCreateProgramCallback;
    static programcbfp        _userDeleteProgramCallback;

    static osg::Int32          s_uniformBlockSize; // the maximum size of a uniform buffer block
    static bool                s_hasStereoViewRenderingSupport;
    static bool                s_hasLensMatchedShadingSupport;
    static bool                s_useBindlessShadowTextures;
    static bool                s_useFog;
    static bool                s_useVolumeScattering;
    static bool                s_isIntel;
    static bool                s_isAMD;
    static bool                s_isNvidia;
};

typedef SHLChunk *SHLChunkP;

OSG_END_NAMESPACE

#include <OSGSHLChunkBase.inl>
#include <OSGSHLChunk.inl>

#define OSGSHLCHUNK_HEADER_CVSID "@(#)$Id: OSGSHLChunk.h,v 1.38 2008/12/12 11:28:45 a-m-z Exp $"

#endif /* _OSGCCHUNK_H_ */
