/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *               Copyright (C) 2000-2002 by the OpenSG Forum                 *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

#include <stdlib.h>
#include <stdio.h>

#include <OSGConfig.h>

#include "OSGShaderParameterChunk.h"

#include <OSGShaderParameter.h>
#include <OSGShaderParameterBool.h>
#include <OSGShaderParameterInt.h>
#include <OSGShaderParameterUInt.h>
#include <OSGShaderParameterReal.h>
#include <OSGShaderParameterVec2f.h>
#include <OSGShaderParameterVec3f.h>
#include <OSGShaderParameterVec4f.h>
#include <OSGShaderParameterMatrix.h>
#include <OSGShaderParameterHandle.h>
#include <OSGShaderParameterMHandle.h>
#include <OSGShaderParameterSubroutine.h>

#include <OSGShaderParameterMInt.h>
#include <OSGShaderParameterMUInt.h>
#include <OSGShaderParameterMReal.h>
#include <OSGShaderParameterMVec2f.h>
#include <OSGShaderParameterMVec3f.h>
#include <OSGShaderParameterMVec4f.h>
#include <OSGShaderParameterMMatrix.h>


OSG_USING_NAMESPACE

/***************************************************************************\
 *                            Description                                  *
\***************************************************************************/

/*! \class osg::ShaderParameterChunk

*/

/***************************************************************************\
 *                           Class variables                               *
\***************************************************************************/

/***************************************************************************\
 *                           Class methods                                 *
\***************************************************************************/

void ShaderParameterChunk::initMethod (void)
{
}


/***************************************************************************\
 *                           Instance methods                              *
\***************************************************************************/

/*-------------------------------------------------------------------------*\
 -  private                                                                 -
\*-------------------------------------------------------------------------*/

/*----------------------- constructors & destructors ----------------------*/

ShaderParameterChunk::ShaderParameterChunk(void) :
    Inherited(),
    _parameter_access(NULL),
    _cleared_parameters(false)
{
}

ShaderParameterChunk::ShaderParameterChunk(const ShaderParameterChunk &source) :
    Inherited(source),
    _parameter_access(source._parameter_access),
    _cleared_parameters(source._cleared_parameters)
{
}

ShaderParameterChunk::~ShaderParameterChunk(void)
{
}

void ShaderParameterChunk::onCreate(const ShaderParameterChunk *source)
{
    Inherited::onCreate(source);

    // ignore prototypes.
    if(GlobalSystemState == Startup)
        return;

    _parameter_access = new ShaderParameterAccess(*editMFParameters());
}

void ShaderParameterChunk::onDestroy(void)
{
    Inherited::onDestroy();

    clearUniformParameters();

    if(_parameter_access != NULL)
        delete _parameter_access;
}

void ShaderParameterChunk::setLocalParameters(const bool local)
{
    clearUniformParameters();

    if (_parameter_access != NULL)
        delete _parameter_access;
   
    _parameter_access = new ShaderParameterAccess(*editMFParameters(), local);
}

/*----------------------------- class specific ----------------------------*/

void ShaderParameterChunk::changed(BitVector whichField, UInt32 origin)
{
    Inherited::changed(whichField, origin);
}

void ShaderParameterChunk::dump(      UInt32    , 
                         const BitVector ) const
{
    SLOG << "Dump ShaderParameterChunk NI" << std::endl;
}

/*---------------------------------- Access -------------------------------*/


/*------------------------------------ Set --------------------------------*/

bool ShaderParameterChunk::setUniformParameter(const char *name, bool value)
{
    return _parameter_access->setParameter<ShaderParameterBool>(name, value);
}

bool ShaderParameterChunk::setUniformParameter(const char *name, Int32 value)
{
    return _parameter_access->setParameter<ShaderParameterInt>(name, value);
}

bool ShaderParameterChunk::setUniformParameter(const char *name, UInt32 value)
{
    return _parameter_access->setParameter<ShaderParameterUInt>(name, value);
}

bool ShaderParameterChunk::setUniformParameter(const char *name, Real32 value)
{
    return _parameter_access->setParameter<ShaderParameterReal>(name, value);
}

bool ShaderParameterChunk::setUniformParameter(const char *name, const Vec2f &value)
{
    return _parameter_access->setParameter<ShaderParameterVec2f>(name, value);
}

bool ShaderParameterChunk::setUniformParameter(const char *name, const Vec3f &value)
{
    return _parameter_access->setParameter<ShaderParameterVec3f>(name, value);
}

bool ShaderParameterChunk::setUniformParameter(const char *name, const Vec4f &value)
{
    return _parameter_access->setParameter<ShaderParameterVec4f>(name, value);
}

bool ShaderParameterChunk::setUniformParameter(const char *name, const Matrix &value)
{
    return _parameter_access->setParameter<ShaderParameterMatrix>(name, value);
}

bool ShaderParameterChunk::setUniformParameterHandle(const char *name, UInt64& value)
{
    return _parameter_access->setParameter<ShaderParameterHandle>(name, value);
}

bool ShaderParameterChunk::setUniformParameterSubroutine(const char *name, const char* value)
{
    return _parameter_access->setParameter<ShaderParameterSubroutine>(name, value);
}


// arrays

bool ShaderParameterChunk::setUniformParameter(const char *name, const MFInt32 &value)
{
    return _parameter_access->setMParameter<ShaderParameterMInt>(name, value);
}

bool ShaderParameterChunk::setUniformParameter(const char* name, const MFUInt32& value)
{
    return _parameter_access->setMParameter<ShaderParameterMUInt>(name, value);
}

bool ShaderParameterChunk::setUniformParameter(const char *name, const MFReal32 &value)
{
    return _parameter_access->setMParameter<ShaderParameterMReal>(name, value);
}

bool ShaderParameterChunk::setUniformParameter(const char *name, const MFVec2f &value)
{
    return _parameter_access->setMParameter<ShaderParameterMVec2f>(name, value);
}

bool ShaderParameterChunk::setUniformParameter(const char *name, const MFVec3f &value)
{
    return _parameter_access->setMParameter<ShaderParameterMVec3f>(name, value);
}

bool ShaderParameterChunk::setUniformParameter(const char *name, const MFVec4f &value)
{
    return _parameter_access->setMParameter<ShaderParameterMVec4f>(name, value);
}

bool ShaderParameterChunk::setUniformParameter(const char *name, const MFMatrix &value)
{
    return _parameter_access->setMParameter<ShaderParameterMMatrix>(name, value);
}

bool ShaderParameterChunk::setUniformParameterMHandle(const char* name, const MFUInt64& value)
{
    return _parameter_access->setMParameter<ShaderParameterMHandle>(name, value);
}

/*------------------------------------ Get --------------------------------*/

bool ShaderParameterChunk::getUniformParameter(const char *name, bool &value)
{
    return _parameter_access->getParameter<ShaderParameterBool>(name, value);
}

bool ShaderParameterChunk::getUniformParameter(const char *name, Int32 &value)
{
    return _parameter_access->getParameter<ShaderParameterInt>(name, value);
}

bool ShaderParameterChunk::getUniformParameter(const char *name, Real32 &value)
{
    return _parameter_access->getParameter<ShaderParameterReal>(name, value);
}

bool ShaderParameterChunk::getUniformParameter(const char *name, Vec2f &value)
{
    return _parameter_access->getParameter<ShaderParameterVec2f>(name, value);
}

bool ShaderParameterChunk::getUniformParameter(const char *name, Vec3f &value)
{
    return _parameter_access->getParameter<ShaderParameterVec3f>(name, value);
}

bool ShaderParameterChunk::getUniformParameter(const char *name, Vec4f &value)
{
    return _parameter_access->getParameter<ShaderParameterVec4f>(name, value);
}

bool ShaderParameterChunk::getUniformParameter(const char *name, Matrix &value)
{
    return _parameter_access->getParameter<ShaderParameterMatrix>(name, value);
}

bool ShaderParameterChunk::subUniformParameter(const char *name)
{
    return _parameter_access->subParameter(name);
}

bool osg::ShaderParameterChunk::hasShaderParameter(std::string const& name) const
{
    for ( const osg::ShaderParameterPtr & parameter : _mfParameters )
    {
        if ( parameter == osg::NullFC )
            continue;

        if ( parameter->getName() == name )
            return true;
    }
    return false;
}

int osg::ShaderParameterChunk::findShaderParameter(osg::ShaderParameterPtr const& searchParameter) const
{    
    for (size_t i = 0; i < _mfParameters.size(); ++i )
    {
        osg::ShaderParameterPtr parameter = _mfParameters[i];
        if (parameter == osg::NullFC)
            continue;

        if (parameter == searchParameter)
            return i;
    }
    return -1;
}

void ShaderParameterChunk::addShaderParameter(osg::ShaderParameterPtr const& parameter)
{
    insertShaderParameter(parameter, -1);
}

void ShaderParameterChunk::insertShaderParameter(osg::ShaderParameterPtr const& parameter, int index)
{
    addRefCP(parameter);
    if ((index < 0) || (index >= _mfParameters.size()))
    {
        _mfParameters.push_back(parameter);
    }
    else
    {
        _mfParameters.insert(_mfParameters.begin()+index, parameter);
    }
    if (_parameter_access != NULL)
        _parameter_access->updateMap();
}

void ShaderParameterChunk::removeShaderParameter(osg::ShaderParameterPtr const& parameter)
{
    const auto iter = _mfParameters.find(parameter);
    if ( iter != _mfParameters.end() )
    {
        _mfParameters.erase(iter);
        subRefCP(parameter);
    }

    if (_parameter_access != NULL)
        _parameter_access->updateMap();
}


osg::ShaderParameterPtr osg::ShaderParameterChunk::findShaderParameter(std::string const& name) const
{
    return _parameter_access->getParameterFC(name.c_str());
}

void ShaderParameterChunk::clearUniformParameters(void)
{
    MFShaderParameterPtr &parameters = *editMFParameters();
    UInt32 size = parameters.size();

    for(UInt32 i=0;i<size;++i)
        subRefCP(parameters[i]);

    parameters.clear();

    if(_parameter_access != NULL)
        _parameter_access->updateMap();

    _cleared_parameters = true;
}

// arrays

bool ShaderParameterChunk::getUniformParameter(const char *name, MFInt32 &value)
{
    return _parameter_access->getMParameter<ShaderParameterMInt>(name, value);
}

bool ShaderParameterChunk::getUniformParameter(const char *name, MFReal32 &value)
{
    return _parameter_access->getMParameter<ShaderParameterMReal>(name, value);
}

bool ShaderParameterChunk::getUniformParameter(const char *name, MFVec2f &value)
{
    return _parameter_access->getMParameter<ShaderParameterMVec2f>(name, value);
}

bool ShaderParameterChunk::getUniformParameter(const char *name, MFVec3f &value)
{
    return _parameter_access->getMParameter<ShaderParameterMVec3f>(name, value);
}

bool ShaderParameterChunk::getUniformParameter(const char *name, MFVec4f &value)
{
    return _parameter_access->getMParameter<ShaderParameterMVec4f>(name, value);
}

bool ShaderParameterChunk::getUniformParameter(const char *name, MFMatrix &value)
{
    return _parameter_access->getMParameter<ShaderParameterMMatrix>(name, value);
}

/*------------------------------------------------------------------------*/
/*                              cvs id's                                  */

#ifdef OSG_SGI_CC
#pragma set woff 1174
#endif

#ifdef OSG_LINUX_ICC
#pragma warning( disable : 177 )
#endif

namespace
{
    static Char8 cvsid_cpp       [] = "@(#)$Id: OSGShaderParameterChunk.cpp,v 1.6 2008/06/09 07:30:32 vossg Exp $";
    static Char8 cvsid_hpp       [] = OSGSHADERPARAMETERCHUNKBASE_HEADER_CVSID;
    static Char8 cvsid_inl       [] = OSGSHADERPARAMETERCHUNKBASE_INLINE_CVSID;

    static Char8 cvsid_fields_hpp[] = OSGSHADERPARAMETERCHUNKFIELDS_HEADER_CVSID;
}

#ifdef __sgi
#pragma reset woff 1174
#endif
