/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright (C) 2000-2002 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

#include <stdlib.h>
#include <stdio.h>

#include "OSGConfig.h"

#include <OSGGL.h>

#include "OSGDrawActionBase.h"
#include "OSGWindow.h"

#include "OSGBlendChunk.h"

OSG_USING_NAMESPACE


/***************************************************************************\
 *                            Description                                  *
\***************************************************************************/

/*! \class osg::BlendChunk
    \ingroup GrpSystemState

See \ref PageSystemBlendChunk for a description. 

This chunk wraps glBlendFunc() (osg::BlendChunk::_sfSrcFactor,
osg::BlendChunk::_sfDestFactor), glBlendEquation() or glBlendEquationEXT()
(whichever is supported) (osg::BlendChunk::_sfEquation), glBlendColor() 
(osg::BlendChunk::_sfColor) and glAlphaFunc() (osg::BlendChunk::_sfAlphaFunc,
osg::BlendChunk::_sfAlphaValue) including glEnable(GL_ALPHA_TEST).

*/

/***************************************************************************\
 *                           Class variables                               *
\***************************************************************************/

StateChunkClass BlendChunk::_class("Blend");


/***************************************************************************\
 *                           Class methods                                 *
\***************************************************************************/

void BlendChunk::initMethod (void)
{
    Inherited::initMethod();
}

/***************************************************************************\
 *                           Instance methods                              *
\***************************************************************************/

/*-------------------------------------------------------------------------*\
 -  private                                                                 -
\*-------------------------------------------------------------------------*/

/*------------- constructors & destructors --------------------------------*/

BlendChunk::BlendChunk(void) :
    Inherited()
{
}

BlendChunk::BlendChunk(const BlendChunk &source) :
    Inherited(source)
{
}

BlendChunk::~BlendChunk(void)
{
}

/*------------------------- Chunk Class Access ---------------------------*/

const StateChunkClass *BlendChunk::getClass(void) const
{
    return &_class;
}

/*------------------------------- Sync -----------------------------------*/

void BlendChunk::changed(BitVector whichField, UInt32 origin)
{
    Inherited::changed(whichField, origin);
}

/*------------------------------ Output ----------------------------------*/

void BlendChunk::dump(      UInt32    OSG_CHECK_ARG(uiIndent),
                      const BitVector OSG_CHECK_ARG(bvFlags )) const
{
    SLOG << "Dump BlendChunk NI" << std::endl;
}

/*------------------------------ State ------------------------------------*/

void BlendChunk::activate(DrawActionBase *action, UInt32)
{
    OSGGLenum src   = _sfSrcFactor.getValue();
    OSGGLenum dest  = _sfDestFactor.getValue();
    OSGGLenum asrc  = _sfAlphaSrcFactor.getValue();
    OSGGLenum adest = _sfAlphaDestFactor.getValue();

    if(src != GL_ONE || dest != GL_ZERO ||
       (asrc  != OSG_GL_UNUSED && asrc  != GL_ONE) || 
       (adest != OSG_GL_UNUSED && adest != GL_ZERO)
      )
    {
        if(asrc != OSG_GL_UNUSED || adest != OSG_GL_UNUSED)
        {
            if(asrc == OSG_GL_UNUSED || adest == OSG_GL_UNUSED)
            {
                FWARNING(("BlendChunk::activate: only one of alpha src and"
                          " alpha dest is set. Ignored.\n"));
                glBlendFunc(src, dest);
            }
            else
            {
                glBlendFuncSeparate(src, dest, asrc, adest);
            }
        }
        else
            glBlendFunc(src, dest);

        // This is not nice, but efficient
        // As the OpenGL constants are fixed it should be safe
        if((src   >= GL_CONSTANT_COLOR && 
            src   <= GL_ONE_MINUS_CONSTANT_ALPHA ) ||
           (dest  >= GL_CONSTANT_COLOR && 
            dest  <= GL_ONE_MINUS_CONSTANT_ALPHA ) ||
           (asrc  >= GL_CONSTANT_COLOR && 
            asrc  <= GL_ONE_MINUS_CONSTANT_ALPHA ) ||
           (adest >= GL_CONSTANT_COLOR && 
            adest <= GL_ONE_MINUS_CONSTANT_ALPHA )
          )
        {

            glBlendColor(_sfColor.getValue().red(),
                        _sfColor.getValue().green(),
                        _sfColor.getValue().blue(),
                        _sfColor.getValue().alpha());

        }
        pushGLFlags();
        enableGLFlag(GL_BLEND);
    }


    if(_sfEquation.getValue() != GL_NONE)
    {
        glBlendEquation(_sfEquation.getValue());
    }
    else
    {
        glBlendEquation(GL_FUNC_ADD);
    }



}

void BlendChunk::deactivate(DrawActionBase *action, UInt32 )
{
    OSGGLenum src    = _sfSrcFactor.getValue();
    OSGGLenum dest   = _sfDestFactor.getValue();
    OSGGLenum asrc   = _sfAlphaSrcFactor.getValue();
    OSGGLenum adest  = _sfAlphaDestFactor.getValue();

    if(src != GL_ONE || dest != GL_ZERO ||
       (asrc  != OSG_GL_UNUSED && asrc  != GL_ONE) || 
       (adest != OSG_GL_UNUSED && adest != GL_ZERO)
      )
    {
        popGLFlags();
        //disableGLFlag(GL_BLEND);
    }
    
    
    if(_sfEquation.getValue() != GL_NONE)
    {
        glBlendEquation(GL_FUNC_ADD);
    }
}

/*-------------------------- Comparison -----------------------------------*/

bool BlendChunk::isTransparent(void) const
{
    return getDestFactor() != GL_ZERO;
}


bool BlendChunk::operator < (const StateChunk &other) const
{
    return this < &other;
}

bool BlendChunk::operator == (const StateChunk &other) const
{
    BlendChunk const *tother = dynamic_cast<BlendChunk const*>(&other);

    if(!tother)
        return false;

    if(tother == this)
        return true;

    if(getSrcFactor()  != tother->getSrcFactor()  ||
       getDestFactor() != tother->getDestFactor() ||
       getEquation()   != tother->getEquation()   ||
       getColor()      != tother->getColor()      ||
       getAlphaFunc()  != tother->getAlphaFunc()  ||
       getAlphaValue() != tother->getAlphaValue()   )
        return false;

    return true;
}

bool BlendChunk::operator != (const StateChunk &other) const
{
    return ! (*this == other);
}


/*------------------------------------------------------------------------*/
/*                              cvs id's                                  */

#ifdef OSG_SGI_CC
#pragma set woff 1174
#endif

#ifdef OSG_LINUX_ICC
#pragma warning( disable : 177 )
#endif

namespace
{
    static Char8 cvsid_cpp       [] = "@(#)$Id: FCTemplate_cpp.h,v 1.13 2002/06/01 10:37:25 vossg Exp $";
    static Char8 cvsid_hpp       [] = OSGBLENDCHUNK_HEADER_CVSID;
    static Char8 cvsid_inl       [] = OSGBLENDCHUNK_INLINE_CVSID;

    static Char8 cvsid_fields_hpp[] = OSGBLENDCHUNKFIELDS_HEADER_CVSID;
}

#ifdef __sgi
#pragma reset woff 1174
#endif


