/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright (C) 2000-2002 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

#include <stdlib.h>
#include <stdio.h>

#include <OSGConfig.h>

#include <OSGGL.h>
#include <OSGGLU.h>
#include <OSGGLEXT.h>

#include "OSGDrawActionBase.h"

#include "OSGCubeTextureChunk.h"
#include "OSGRenderAction.h"
#include "OSGImage.h"

OSG_USING_NAMESPACE

/***************************************************************************\
 *                            Description                                  *
\***************************************************************************/

/*! \class osg::CubeTextureChunk
    \ingroup GrpSystemState

See \ref PageSystemCubeTextureChunk for a description.

The CubeTexture is defined by 6 separate texture images. As it is derived from 
the standard texture chunk it already has an image, which is used for the 
negative Z direction. The other 5 images are stored in variables that are named
according to the direction they represent (osg::CubeTextureChunk::_sfPosZImage,
osg::CubeTextureChunk::_sfNegYImage, osg::CubeTextureChunk::_sfPosYImage, 
osg::CubeTextureChunk::_sfNegXImage, osg::CubeTextureChunk::_sfPosXImage).

*/


/***************************************************************************\
 *                           Instance methods                              *
\***************************************************************************/

/*-------------------------------------------------------------------------*\
 -  public                                                                 -
\*-------------------------------------------------------------------------*/

/*------------- constructors & destructors --------------------------------*/

CubeTextureChunk::CubeTextureChunk(void) :
    Inherited()
{
}

CubeTextureChunk::CubeTextureChunk(const CubeTextureChunk &source) :
    Inherited(source)
{
}

CubeTextureChunk::~CubeTextureChunk(void)
{
}

/*------------------------- Chunk Class Access ---------------------------*/

const StateChunkClass *CubeTextureChunk::getClass(void) const
{
    return TextureChunk::getClass();
}

/*------------------------------- Sync -----------------------------------*/

void CubeTextureChunk::changed(BitVector whichField, UInt32 origin)
{
    if (whichField & PosXImageFieldMask)
    {

        m_imageChanged = true;
    }
    if (whichField & PosYImageFieldMask)
    {

        m_imageChanged = true;
    }
    if(whichField & PosZImageFieldMask)
    {
        m_imageChanged = true;
    }
    if(whichField & NegXImageFieldMask)
    {
        m_imageChanged = true;
    }
    
   
    if(whichField & NegYImageFieldMask)
    {
        m_imageChanged = true;
    }
    Inherited::changed(whichField, origin);
}

/*----------------------------- onCreate --------------------------------*/

void CubeTextureChunk::onCreate(const CubeTextureChunk *)
{
    if(GlobalSystemState == Startup)
        return;
}

void CubeTextureChunk::onDestroy(void)
{
    Inherited::onDestroy();

    if(_sfPosZImage.getValue() != NullFC)
        subRefCP(_sfPosZImage.getValue());

    if(_sfPosXImage.getValue() != NullFC)
        subRefCP(_sfPosXImage.getValue());

    if(_sfNegXImage.getValue() != NullFC)
        subRefCP(_sfNegXImage.getValue());

    if(_sfPosYImage.getValue() != NullFC)
        subRefCP(_sfPosYImage.getValue());

    if(_sfNegYImage.getValue() != NullFC)
        subRefCP(_sfNegYImage.getValue());

    
}

/*------------------------------ Output ----------------------------------*/

void CubeTextureChunk::dump(      UInt32    , 
                             const BitVector ) const
{
    SLOG << "Dump CubeTextureChunk NI" << std::endl;
}


void CubeTextureChunk::prepareTexture(Window *win, UInt32 id)
{
    if (!m_imageChanged)
        return;

    handleTexture(win, id,
        GL_TEXTURE_CUBE_MAP,
        GL_TEXTURE_CUBE_MAP,
        GL_TEXTURE_CUBE_MAP_NEGATIVE_Z,
        getImage());

    handleTexture(win, id,
        GL_TEXTURE_CUBE_MAP,
        GL_TEXTURE_CUBE_MAP, //GL_NONE,
        GL_TEXTURE_CUBE_MAP_POSITIVE_Z,
        getPosZImage());
    handleTexture(win, id,
        GL_TEXTURE_CUBE_MAP,
        GL_TEXTURE_CUBE_MAP, //GL_NONE
        GL_TEXTURE_CUBE_MAP_NEGATIVE_Y,
        getNegYImage());
    handleTexture(win, id,
        GL_TEXTURE_CUBE_MAP,
        GL_TEXTURE_CUBE_MAP, //GL_NONE, 
        GL_TEXTURE_CUBE_MAP_POSITIVE_Y,
         getPosYImage());
    handleTexture(win, id,
        GL_TEXTURE_CUBE_MAP,
        GL_TEXTURE_CUBE_MAP, //GL_NONE, 
        GL_TEXTURE_CUBE_MAP_NEGATIVE_X,
        getNegXImage());

    handleTexture(win, id,
        GL_TEXTURE_CUBE_MAP,
        GL_TEXTURE_CUBE_MAP, //GL_NONE, 
        GL_TEXTURE_CUBE_MAP_POSITIVE_X,
        getPosXImage());

    m_imageChanged = false;
}
/*------------------------------ State ------------------------------------*/

void CubeTextureChunk::activate( DrawActionBase *action, UInt32 idx )
{   
    Window *win = action->getWindow();

    glActiveTexture(GL_TEXTURE0 + idx);

    if (m_textureId == 0)
    {
        glGenTextures(1, &m_textureId);
        if (m_textureId == 0)
            return;
    }
    prepareTexture(win, m_textureId);

    glBindTexture(GL_TEXTURE_CUBE_MAP, m_textureId);

}

void CubeTextureChunk::deactivate(DrawActionBase *action, UInt32 idx)
{
    Window *win = action->getWindow();   

    glActiveTexture(GL_TEXTURE0 + idx);
    glBindTexture(GL_TEXTURE_CUBE_MAP, 0);

    glErr("CubeTextureChunk::deactivate");
}

/*-------------------------- Comparison -----------------------------------*/
bool CubeTextureChunk::operator < (const StateChunk &other) const
{
    return this < &other;
}

bool CubeTextureChunk::operator == (const StateChunk &other) const
{
    CubeTextureChunk const *tother = 
            dynamic_cast<CubeTextureChunk const*>(&other);

    if(!tother)
        return false;

    if(tother == this)
        return true;

    return  TextureChunk::operator==(other) &&
            getPosZImage()  == tother->getPosZImage() &&
            getPosYImage()  == tother->getPosYImage() &&
            getNegYImage()  == tother->getNegYImage() &&
            getPosXImage()  == tother->getPosXImage() &&
            getNegXImage()  == tother->getNegXImage();
}

bool CubeTextureChunk::operator != (const StateChunk &other) const
{
    return ! (*this == other);
}

/*-------------------------------------------------------------------------*/
/*                              cvs id's                                   */

#ifdef __sgi
#pragma set woff 1174
#endif

#ifdef OSG_LINUX_ICC
#pragma warning( disable : 177 )
#endif

namespace
{
    static Char8 cvsid_cpp[] = "@(#)$Id: $";
    static Char8 cvsid_hpp[] = OSGCUBETEXTURECHUNK_HEADER_CVSID;
    static Char8 cvsid_inl[] = OSGCUBETEXTURECHUNK_INLINE_CVSID;
}
