/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright (C) 2000-2002 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

#include <stdlib.h>
#include <stdio.h>

#include "OSGConfig.h"

#include <OSGGL.h>
#include <OSGGLU.h>

#include "OSGMaterialChunk.h"

OSG_USING_NAMESPACE

/***************************************************************************\
 *                            Description                                  *
\***************************************************************************/

/*! \class osg::MaterialChunk
    \ingroup GrpSystemState

See \ref PageSystemMaterialChunk for details.

This chunk wraps glMaterial() (osg::MaterialChunk::_sfDiffuse,
osg::MaterialChunk::_sfSpecular, osg::MaterialChunk::_sfEmission,
osg::MaterialChunk::_sfShininess, osg::MaterialChunk::_sfAmbient),
glEnable(GL_LIGHTING)(osg::MaterialChunk::_sfLit) and glColorMaterial()
(osg::MaterialChunk::_sfColorMaterial).

*/

/***************************************************************************\
 *                           Class variables                               *
\***************************************************************************/

StateChunkClass MaterialChunk::_class("Material");

/***************************************************************************\
 *                           Class methods                                 *
\***************************************************************************/

void MaterialChunk::initMethod (void)
{
}

/***************************************************************************\
 *                           Instance methods                              *
\***************************************************************************/

/*-------------------------------------------------------------------------*\
 -  private                                                                 -
\*-------------------------------------------------------------------------*/

MaterialChunk::MaterialChunk(void) :
    Inherited(),
    _shlChunk(NullFC)
{
}

MaterialChunk::MaterialChunk(const MaterialChunk &source) :
    Inherited(source),
    _shlChunk(source._shlChunk)
{
}

MaterialChunk::~MaterialChunk(void)
{
    subRefCP(_shlChunk);
}

/*------------------------- Chunk Class Access ---------------------------*/

const StateChunkClass *MaterialChunk::getClass(void) const
{
    return &_class;
}

/*------------------------------- Sync -----------------------------------*/

void MaterialChunk::changed(BitVector whichField, UInt32 origin)
{
    if (_shlChunk == NullFC)
        initChunk();

    // TODO: make this more efficient....maybe....mmhhhh...who cares ;)
    if (_shlChunk != NullFC)
    {
        beginEditCP(_shlChunk, SHLChunk::ParametersFieldMask);
        {
            _shlChunk->setUniformParameter("diffuseColor", Vec4f(getDiffuse()[0], getDiffuse()[1], getDiffuse()[2], getDiffuse()[3]));
            _shlChunk->setUniformParameter("specularColor", Vec4f(getSpecular()[0], getSpecular()[1], getSpecular()[2], getSpecular()[3]));
            _shlChunk->setUniformParameter("emissiveColor", Vec4f(getEmission()[0], getEmission()[1], getEmission()[2], getEmission()[3]));
            _shlChunk->setUniformParameter("shininess", getShininess());
            _shlChunk->setUniformParameter("isLit", getLit());
            _shlChunk->setUniformParameter("textureMap", 0);
        }
        endEditCP(_shlChunk, SHLChunk::ParametersFieldMask);
    }
    Inherited::changed(whichField, origin);
}

/*------------------------------ Output ----------------------------------*/

void MaterialChunk::dump(      UInt32    OSG_CHECK_ARG(uiIndent),
                         const BitVector OSG_CHECK_ARG(bvFlags )) const
{
    SLOG << "Dump MaterialChunk NI" << std::endl;
}

/*------------------------------ State ------------------------------------*/

void MaterialChunk::activate(DrawActionBase * action, UInt32 idx)
{
    if (_shlChunk == NullFC)
    {
        initChunk();
        if (_shlChunk != NullFC)
        {
            beginEditCP(_shlChunk, SHLChunk::ParametersFieldMask);
            {
                _shlChunk->setUniformParameter("diffuseColor", Vec4f(getDiffuse()[0], getDiffuse()[1], getDiffuse()[2], getDiffuse()[3]));
                _shlChunk->setUniformParameter("specularColor", Vec4f(getSpecular()[0], getSpecular()[1], getSpecular()[2], getSpecular()[3]));
                _shlChunk->setUniformParameter("emissiveColor", Vec4f(getEmission()[0], getEmission()[1], getEmission()[2], getEmission()[3]));
                _shlChunk->setUniformParameter("shininess", getShininess());
                _shlChunk->setUniformParameter("isLit", getLit());
                _shlChunk->setUniformParameter("textureMap", 0);
            }
            endEditCP(_shlChunk, SHLChunk::ParametersFieldMask);
        }
    }
    if (_shlChunk != NullFC)
    {
        _shlChunk->activate(action, idx);
    }
}

void MaterialChunk::deactivate(DrawActionBase * action, UInt32 idx)
{
    if (_shlChunk != NullFC)
    {
        _shlChunk->deactivate(action, idx);
    }
}


/*-------------------------- Comparison -----------------------------------*/


bool MaterialChunk::isTransparent(void) const
{
    return(getDiffuse()[3] <(1. - Eps));
}


bool MaterialChunk::operator <(const StateChunk &other) const
{
    return this < &other;
}

bool MaterialChunk::operator ==(const StateChunk &other) const
{
    MaterialChunk const *tother = dynamic_cast<MaterialChunk const*>(&other);

    if(!tother)
        return false;

    if(tother == this)
        return true;

    if(!getAmbient  ().equals(tother->getAmbient  (), Eps) ||
       !getDiffuse  ().equals(tother->getDiffuse  (), Eps) ||
       !getSpecular ().equals(tother->getSpecular (), Eps) ||
       !getEmission ().equals(tother->getEmission (), Eps) ||
        getShininess()     != tother->getShininess()       ||
        getLit()           != tother->getLit()             ||
        getColorMaterial() != tother->getColorMaterial()
      )
        return false;

    return true;
}

bool MaterialChunk::operator !=(const StateChunk &other) const
{
    return !(*this == other);
}

void MaterialChunk::initChunk()
{
if (_shlChunk == NullFC)
{
    _shlChunk = SHLChunk::create();
    std::string vp = SHLChunk::getShaderHeaderString(osg::SHLChunk::kVertexProgram, true, false) +
        "layout (location = 0) in vec3 inPosition; \n"
        "layout (location = 1) in vec3 inNormal; \n"
        "layout (location = 2) in vec2 inTexCoord; \n"
        "out VertexData  \n"
        "{             \n"
        "    vec4 wP;  \n"
        "    vec3 wN; \n"
        "    vec2 texCoord; \n"
        "} vs_out;    \n"
        "void main()  \n"
        "{             \n"
        "   vs_out.texCoord = inTexCoord; \n"
        "   vec4 wP = object.worldMatrix * vec4(inPosition.xyz, 1.0); \n"
        "	vs_out.wN =  mat3(object.transInvWorldMatrix) * inNormal; \n"
        "	vs_out.wP = wP; \n"
        "   gl_Position = camera.viewProjectionMatrix[0] * wP; \n"
        "#ifdef GLAD_GL_NV_stereo_view_rendering \n"
        "   gl_SecondaryPositionNV = camera.viewProjectionMatrix[1] * wP; \n"
        "   gl_Layer = 0; \n"
        "#endif \n"
		"#ifdef USE_LENS_MATCHED_SHADING \n"
		"	gl_ViewportMask[0] = 15; \n"
		"#ifdef GLAD_GL_NV_stereo_view_rendering\n"
		"	gl_SecondaryViewportMaskNV[0] = 15;\n"
		"#endif \n"
		"#endif \n"
        "}\n";

    std::string fp = SHLChunk::getShaderHeaderString(osg::SHLChunk::kFragmentProgram, true, false) +
        "uniform vec4 diffuseColor; \n"
        "uniform vec4 specularColor; \n"
        "uniform vec4 emissiveColor; \n"
        "uniform float shininess;  \n"
        "uniform bool isLit; \n"
        "uniform sampler2D textureMap;\n"
        "in VertexData  \n"
        "{\n"
        "    vec4 wP; \n"
        "    vec3 wN; \n"
        "    vec2 texCoord; \n"
        "} fs_in;\n"
        "layout (location = 0) out vec4 colorBufferRGBA;\n"
        "void main()  \n"
        "{  \n"
        "   if( isLit)\n"
        "   { \n"
        "       vec3 eP = (camera.viewMatrix[eye] * fs_in.wP).xyz; \n"
        "       vec3 eN = mat3(camera.viewMatrix[eye]) *  fs_in.wN; \n"
        "	    vec3 N = normalize(eN); \n"
        "	    vec3 V = -normalize(eP);  \n"
        "       N = faceforward( N, V, -N); \n"
        "       float NdotV =  abs( dot(N, V)); \n"
        "       float spec = pow( NdotV, shininess);\n"
        "       vec4 diffuse = diffuseColor * NdotV; \n"
        "       if( textureSize(textureMap, 0).x > 0 && textureSize(textureMap, 0).y > 0) \n"
        "           diffuse *= texture(textureMap, fs_in.texCoord);\n"
        "       vec4 specular = specularColor * spec;  \n"
        "       colorBufferRGBA = vec4( emissiveColor.rgb + diffuse.rgb + specular.rgb, diffuse.a);\n"
        "   }\n"
        "   else \n"
        "   { \n"
        "       float displayLuminance = 1.0; \n" // viewport.displayLuminance
        "       if( textureSize(textureMap, 0).x > 0 && textureSize(textureMap, 0).y > 0) \n"
        "           colorBufferRGBA = vec4( displayLuminance, displayLuminance, displayLuminance, 1.0) * diffuseColor * texture(textureMap, fs_in.texCoord); \n"
        "       else \n"
        "           colorBufferRGBA = vec4( displayLuminance, displayLuminance, displayLuminance, 1.0) * diffuseColor; \n"
        "   } \n"
        "} \n";

    beginEditCP(_shlChunk, osg::FieldBits::AllFields);
    _shlChunk->setVertexProgram(vp);
    _shlChunk->setGeometryProgram("");
    _shlChunk->setFragmentProgram(fp);
    _shlChunk->clearUniformParameters();
    endEditCP(_shlChunk, osg::FieldBits::AllFields);

    addRefCP(_shlChunk);
}
}


/*------------------------------------------------------------------------*/
/*                              cvs id's                                  */

#ifdef OSG_SGI_CC
#pragma set woff 1174
#endif

#ifdef OSG_LINUX_ICC
#pragma warning( disable : 177 )
#endif

namespace
{
    static Char8 cvsid_cpp       [] = "@(#)$Id: FCTemplate_cpp.h,v 1.13 2002/06/01 10:37:25 vossg Exp $";
    static Char8 cvsid_hpp       [] = OSGMATERIALCHUNK_HEADER_CVSID;
    static Char8 cvsid_inl       [] = OSGMATERIALCHUNK_INLINE_CVSID;

    static Char8 cvsid_fields_hpp[] = OSGMATERIALCHUNKFIELDS_HEADER_CVSID;
}

#ifdef __sgi
#pragma reset woff 1174
#endif


