/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright (C) 2000-2002 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
 \*---------------------------------------------------------------------------*/
 /*---------------------------------------------------------------------------*\
  *                                License                                    *
  *                                                                           *
  * This library is free software; you can redistribute it and/or modify it   *
  * under the terms of the GNU Library General Public License as published    *
  * by the Free Software Foundation, version 2.                               *
  *                                                                           *
  * This library is distributed in the hope that it will be useful, but       *
  * WITHOUT ANY WARRANTY; without even the implied warranty of                *
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
  * Library General Public License for more details.                          *
  *                                                                           *
  * You should have received a copy of the GNU Library General Public         *
  * License along with this library; if not, write to the Free Software       *
  * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
  *                                                                           *
  \*---------------------------------------------------------------------------*/
  /*---------------------------------------------------------------------------*\
   *                                Changes                                    *
   *                                                                           *
   *                                                                           *
   *                                                                           *
   *                                                                           *
   *                                                                           *
   *                                                                           *
   \*---------------------------------------------------------------------------*/

   //---------------------------------------------------------------------------
   //  Includes
   //---------------------------------------------------------------------------

#include <stdlib.h>
#include <stdio.h>

#include "OSGConfig.h"

#include <OSGGL.h>
#include <OSGGLU.h>
#include <OSGGLEXT.h>
#include <OSGImage.h>

#include "OSGDrawActionBase.h"
#include "OSGRenderAction.h"
#include "OSGTextureTransformChunk.h"
#include "OSGTextureChunk.h"

#ifndef TMHACK
#define TMHACK 1	// hehe :)
#endif

OSG_USING_NAMESPACE


/***************************************************************************\
 *                            Description                                  *
 \***************************************************************************/

 /*! \class osg::TextureChunk
     \ingroup GrpSystemState

     See \ref PageSystemTextureChunk for a description.

     This chunk wraps glTexImage[123]D (osg::TextureChunk::_sfImage,
     osg::TextureChunk::_sfInternalFormat, osg::TextureChunk::_sfExternalFormat),
     glTexParameter (osg::TextureChunk::_sfMinFilter,
     osg::TextureChunk::_sfMagFilter, osg::TextureChunk::_sfWrapS,
     osg::TextureChunk::_sfWrapT, osg::TextureChunk::_sfWrapR), glTexEnv
     (osg::TextureChunk::_sfEnvMode, osg::TextureChunk::_sfEnvColor,
     osg::TextureChunk::_sfPriority). The ARB combine extension is also supported,
     where available (osg::TextureChunk::_sfEnvCombineRGB,
     osg::TextureChunk::_sfEnvScaleRGB, osg::TextureChunk::_sfEnvSource0RGB,
     osg::TextureChunk::_sfEnvSource1RGB, osg::TextureChunk::_sfEnvSource2RGB,
     osg::TextureChunk::_sfEnvOperand0RGB, osg::TextureChunk::_sfEnvOperand1RGB,
     osg::TextureChunk::_sfEnvOperand2RGB,
     osg::TextureChunk::_sfEnvCombineAlpha,   osg::TextureChunk::_sfEnvScaleAlpha,
     osg::TextureChunk::_sfEnvSource0Alpha, osg::TextureChunk::_sfEnvSource1Alpha,
     osg::TextureChunk::_sfEnvSource2Alpha, osg::TextureChunk::_sfEnvOperand0Alpha,
     osg::TextureChunk::_sfEnvOperand1Alpha,
     osg::TextureChunk::_sfEnvOperand2Alpha). It is possible to enable the point
     sprite coordinate replacement  (osg::TextureChunk::_sfPointSprite), see \ref
     PageSystemPointChunk for details. The two parameters
     osg::TextureChunk::_sfScale and osg::TextureChunk::_sfFrame specify details
     about the texture.

     On hardware that supports it (i.e. NVidia boards) the texture shader
     extension(s) are also available.

     */

     /***************************************************************************\
      *                           Class variables                               *
      \***************************************************************************/

    char TextureChunk::cvsid[] = "@(#)$Id: OSGTextureChunk.cpp,v 1.41 2002/06/17 12:27:14 jbehr Exp $";

StateChunkClass TextureChunk::_class("Texture", osgMaxTexImages);


std::vector<bool>   TextureChunk::_needTexMat;
std::vector<Matrix> TextureChunk::_lastTexMat;

/***************************************************************************\
 *                           Class methods                                 *
 \***************************************************************************/

 /*-------------------------------------------------------------------------*\
  -  private                                                                -
  \*-------------------------------------------------------------------------*/

void TextureChunk::initMethod(void)
{
    // leads to a crash with msvc 9.0, looks like _needTexMat and
    // _lastTexMat are not yet initialized at the call time of initMethod()!
    //_needTexMat.resize(4, false);
    //_lastTexMat.resize(4, Matrix::identity());
}

bool TextureChunk::activeMatrix(Matrix &texMat, UInt16 texture)
{
    if (texture < _needTexMat.size())
    {
        texMat = _lastTexMat[texture];
        return _needTexMat[texture];
    }
    else
        return false;
}

/***************************************************************************\
 *                           Instance methods                              *
 \***************************************************************************/

 /*-------------------------------------------------------------------------*\
  -  private                                                                 -
  \*-------------------------------------------------------------------------*/


  /*------------- constructors & destructors --------------------------------*/

TextureChunk::TextureChunk(void) :
    m_textureId(0),
    m_imageChanged(true),
    Inherited()
{
    if (_needTexMat.empty())
        _needTexMat.resize(4, false);

    if (_lastTexMat.empty())
        _lastTexMat.resize(4, Matrix::identity());

    Window::registerConstant(GL_MAX_TEXTURE_UNITS);
    Window::registerConstant(GL_MAX_TEXTURE_IMAGE_UNITS);
}

TextureChunk::TextureChunk(const TextureChunk &source) :
    m_textureId(0),
    m_imageChanged(true),
    Inherited(source)
{
}

TextureChunk::~TextureChunk(void)
{
}

/*------------------------- Chunk Class Access ---------------------------*/

const StateChunkClass *TextureChunk::getClass(void) const
{
    return &_class;
}

/*------------------------------- Sync -----------------------------------*/

/*! React to field changes.
    Note: this function also handles CubeTexture changes, make sure to keep
    it consistent with the cubeTexture specifics
    */

void TextureChunk::changed(BitVector whichField, UInt32 origin)
{

    // Only filter changed? Mipmaps need reinit.
    if ((whichField & ~(MinFilterFieldMask | MagFilterFieldMask)) == 0)
    {
        if ((getMinFilter() != GL_NEAREST) &&
            (getMinFilter() != GL_LINEAR))
        {
            triggerReInit();
        }
        else // switching to GL_NEAREST or GL_LINEAR only requires refresh
        {
            imageContentChanged();
        }
    } // Only priority changed? Refresh is fine.
    else if ((whichField & ~(PriorityFieldMask | FrameFieldMask)) == 0)
    {
        imageContentChanged();
    } // Only dirty rectangle changed? Refresh is fine.
    else if ((whichField & ~(DirtyMinXFieldMask | DirtyMaxXFieldMask |
        DirtyMinYFieldMask | DirtyMaxYFieldMask |
        DirtyMinZFieldMask | DirtyMaxZFieldMask)) == 0)
    {
        triggerRefresh();
        m_imageChanged = true;
    }
    else // Play it safe, do a reinit
    {
        triggerReInit();
        m_imageChanged = true;
    }

    if (whichField & ImageFieldMask)
    {
        m_imageChanged = true;
    }

    Inherited::changed(whichField, origin);
}

bool TextureChunk::isTransparent(void) const
{
    // Even if the texture has alpha, the Blending makes the sorting
    // important, thus textures per se are not transparent
    return false;
}


/*----------------------------- onCreate --------------------------------*/

void TextureChunk::onCreate(const TextureChunk *)
{
    if (GlobalSystemState == Startup)
        return;
}

void TextureChunk::onDestroy(void)
{
    Inherited::onDestroy();

    if (_sfImage.getValue() != NullFC)
    {
        TextureChunkPtr thisPtr(*this);
        _sfImage.getValue()->subParent(thisPtr);

        subRefCP(_sfImage.getValue());
    }

    if (m_textureId != 0)
    {
        if (osgMakeCurrent())
            glDeleteTextures(1, &m_textureId);

        m_textureId = 0;
        m_imageChanged = true;
    }
}

/*------------------------------ Output ----------------------------------*/

void TextureChunk::dump(UInt32    OSG_CHECK_ARG(uiIndent),
    const BitVector OSG_CHECK_ARG(bvFlags)) const
{
    SLOG << "Dump TextureChunk NI" << std::endl;
}

void TextureChunk::fixFormats(GLenum paramtarget, GLenum& internalFormat, GLenum& externalFormat, Int32 numChannels) const
{
    //std::cout << "fixFormats: " << numChannels << " internal format: " << std::hex << internalFormat << std::endl;
    int	swizzle[4] = { GL_RED, GL_GREEN, GL_BLUE, GL_ALPHA };


    if (internalFormat == GL_NONE)
    {
        switch (externalFormat)
        {
        case GL_BGR:
            internalFormat = GL_RGB;
            break;
        case GL_BGRA:
            internalFormat = GL_RGBA;
            break;
        case GL_INTENSITY:
        {
            internalFormat = GL_RED;
            externalFormat = GL_RED;
            swizzle[0] = GL_RED;
            swizzle[1] = GL_RED;
            swizzle[2] = GL_RED;
            swizzle[3] = GL_RED;
            break;
        }
        case GL_INTENSITY16F_ARB:
        {
            internalFormat = GL_R16F;
            externalFormat = GL_R16F;
            swizzle[0] = GL_RED;
            swizzle[1] = GL_RED;
            swizzle[2] = GL_RED;
            swizzle[3] = GL_RED;
            break;
        }
        case GL_INTENSITY32F_ARB:
        {
            internalFormat = GL_R32F;
            externalFormat = GL_R32F;
            swizzle[0] = GL_RED;
            swizzle[1] = GL_RED;
            swizzle[2] = GL_RED;
            swizzle[3] = GL_RED;
            break;
        }
        case GL_LUMINANCE:
        {
            internalFormat = GL_RED;
            externalFormat = GL_RED;
            swizzle[0] = GL_RED;
            swizzle[1] = GL_RED;
            swizzle[2] = GL_RED;
            swizzle[3] = GL_ONE;
            break;
        }
        case GL_LUMINANCE16F_ARB:
        {
            internalFormat = GL_R16F;
            externalFormat = GL_R16F;
            swizzle[0] = GL_RED;
            swizzle[1] = GL_RED;
            swizzle[2] = GL_RED;
            swizzle[3] = GL_ONE;
            break;
        }
        case GL_LUMINANCE32F_ARB:
        {
            internalFormat = GL_R32F;
            externalFormat = GL_R32F;
            swizzle[0] = GL_RED;
            swizzle[1] = GL_RED;
            swizzle[2] = GL_RED;
            swizzle[3] = GL_ONE;
            break;
        }
        case GL_SLUMINANCE:
        {
            swizzle[0] = GL_RED;
            swizzle[1] = GL_RED;
            swizzle[2] = GL_RED;
            swizzle[3] = GL_ONE;
            internalFormat = GL_SRGB8;
            externalFormat = GL_RED;
            break;
        }
        case GL_ALPHA:
        {
            internalFormat = GL_RED;
            externalFormat = GL_RED;
            swizzle[0] = GL_RED;
            swizzle[1] = GL_RED;
            swizzle[2] = GL_RED;
            swizzle[3] = GL_RED;
            break;
        }
        case GL_LUMINANCE_ALPHA:
        {
            internalFormat = GL_RG;
            externalFormat = GL_RG;
            swizzle[0] = GL_RED;
            swizzle[1] = GL_RED;
            swizzle[2] = GL_RED;
            swizzle[3] = GL_GREEN;
            break;
        }
        case GL_SLUMINANCE_ALPHA:
        {
            swizzle[0] = GL_RED;
            swizzle[1] = GL_RED;
            swizzle[2] = GL_RED;
            swizzle[3] = GL_GREEN;
            internalFormat = GL_SRGB8;
            externalFormat = GL_RG;
            break;
        }
        case GL_LUMINANCE_ALPHA16F_ARB:
        {
            swizzle[0] = GL_RED;
            swizzle[1] = GL_RED;
            swizzle[2] = GL_RED;
            swizzle[3] = GL_GREEN;
            internalFormat = GL_RG16F;
            externalFormat = GL_RG16F;
            break;
        }
        case GL_LUMINANCE_ALPHA32F_ARB:
        {
            swizzle[0] = GL_RED;
            swizzle[1] = GL_RED;
            swizzle[2] = GL_RED;
            swizzle[3] = GL_GREEN;
            internalFormat = GL_RG32F;
            externalFormat = GL_RG32F;
            break;
        }
        default:

            internalFormat = externalFormat;
            break;
        };
    }
    else
    {
        // since the opengl core profile does not support luminance/intensity formats anymore we need to set the swizzle parameter instead
        switch (internalFormat)
        {
        case GL_INTENSITY:
        {
            internalFormat = GL_RED;
            swizzle[0] = GL_RED;
            swizzle[1] = GL_RED;
            swizzle[2] = GL_RED;
            swizzle[3] = GL_RED;
            break;
        }
        case GL_INTENSITY16F_ARB:
        {
            internalFormat = GL_R16F;
            swizzle[0] = GL_RED;
            swizzle[1] = GL_RED;
            swizzle[2] = GL_RED;
            swizzle[3] = GL_RED;
            break;
        }
        case GL_INTENSITY32F_ARB:
        {
            internalFormat = GL_R32F;
            swizzle[0] = GL_RED;
            swizzle[1] = GL_RED;
            swizzle[2] = GL_RED;
            swizzle[3] = GL_RED;
            break;
        }
        case GL_LUMINANCE:
        {
            internalFormat = GL_RED;
            swizzle[0] = GL_RED;
            swizzle[1] = GL_RED;
            swizzle[2] = GL_RED;
            swizzle[3] = GL_ONE;
            break;
        }
        case GL_LUMINANCE16F_ARB:
        {
            internalFormat = GL_R16F;
            swizzle[0] = GL_RED;
            swizzle[1] = GL_RED;
            swizzle[2] = GL_RED;
            swizzle[3] = GL_ONE;
            break;
        }
        case GL_LUMINANCE32F_ARB:
        {
            internalFormat = GL_R32F;
            swizzle[0] = GL_RED;
            swizzle[1] = GL_RED;
            swizzle[2] = GL_RED;
            swizzle[3] = GL_ONE;
            break;
        }
        case GL_SLUMINANCE:
        {
            swizzle[0] = GL_RED;
            swizzle[1] = GL_RED;
            swizzle[2] = GL_RED;
            swizzle[3] = GL_ONE;
            internalFormat = GL_SRGB8;
            break;
        }
        case GL_ALPHA:
        {
            internalFormat = GL_RED;
            swizzle[0] = GL_RED;
            swizzle[1] = GL_RED;
            swizzle[2] = GL_RED;
            swizzle[3] = GL_RED;
            break;
        }
        case GL_LUMINANCE_ALPHA:
        {
            internalFormat = GL_RG;
            swizzle[0] = GL_RED;
            swizzle[1] = GL_RED;
            swizzle[2] = GL_RED;
            swizzle[3] = (numChannels > 1) ? GL_GREEN : GL_ONE;
            break;
        }
        case GL_SLUMINANCE_ALPHA:
        {
            internalFormat = GL_SRGB8;
            swizzle[0] = GL_RED;
            swizzle[1] = GL_RED;
            swizzle[2] = GL_RED;
            swizzle[3] = (numChannels > 1) ? GL_GREEN : GL_ONE;
            break;
        }
        case GL_LUMINANCE_ALPHA16F_ARB:
        {
            internalFormat = GL_RG16F;
            swizzle[0] = GL_RED;
            swizzle[1] = GL_RED;
            swizzle[2] = GL_RED;
            swizzle[3] = (numChannels > 1) ? GL_GREEN : GL_ONE;
            break;
        }
        case GL_LUMINANCE_ALPHA32F_ARB:
        {
            internalFormat = GL_RG32F;
            swizzle[0] = GL_RED;
            swizzle[1] = GL_RED;
            swizzle[2] = GL_RED;
            swizzle[3] = (numChannels > 1) ? GL_GREEN : GL_ONE;
            break;
        }
        default:
            break;
        };

        switch (externalFormat)
        {
        case GL_INTENSITY:
        {
            externalFormat = GL_RED;
            break;
        }
        case GL_INTENSITY16F_ARB:
        {
            externalFormat = GL_R16F;
            break;
        }
        case GL_INTENSITY32F_ARB:
        {
            externalFormat = GL_R32F;
            break;
        }
        case GL_LUMINANCE:
        {
            externalFormat = GL_RED;
            break;
        }
        case GL_LUMINANCE16F_ARB:
        {
            externalFormat = GL_R16F;
            break;
        }
        case GL_LUMINANCE32F_ARB:
        {
            externalFormat = GL_R32F;
            break;
        }
        case GL_SLUMINANCE:
        {
            externalFormat = GL_RED;
            break;
        }
        case GL_ALPHA:
        {
            externalFormat = GL_RED;
            break;
        }
        case GL_LUMINANCE_ALPHA:
        {
            externalFormat = GL_RG;
            break;
        }
        case GL_SLUMINANCE_ALPHA:
        {
            externalFormat = GL_RG;
            break;
        }
        case GL_LUMINANCE_ALPHA16F_ARB:
        {
            externalFormat = GL_RG16F;
            break;
        }
        case GL_LUMINANCE_ALPHA32F_ARB:
        {
            externalFormat = GL_RG32F;
            break;
        }
        default:
            break;
        };
    }
    // set the swizzle
    glTexParameteriv(paramtarget, GL_TEXTURE_SWIZZLE_RGBA, swizzle);
}

/*------------------------------ State ------------------------------------*/

/*! Texture handler. Create/update a single texture.
    Also used by derived CubeMap chunk.
    */

void TextureChunk::handleTexture(Window *win, UInt32 id,
    OSGGLenum bindtarget,
    OSGGLenum paramtarget,
    OSGGLenum imgtarget,
    ImagePtr img2, int side)
{
    if (img2 == NullFC || !img2->getDimension()) // no image ?
        return;

    ImagePtr img = img2;

    bool needMipmaps = getMinFilter() == GL_NEAREST_MIPMAP_NEAREST ||
        getMinFilter() == GL_LINEAR_MIPMAP_NEAREST ||
        getMinFilter() == GL_NEAREST_MIPMAP_LINEAR ||
        getMinFilter() == GL_LINEAR_MIPMAP_LINEAR;

    // as we're not allocating anything here, the same code can be used
    // for reinitialization
    if (!img || !img->getDimension()) // no image ?
        return;

    glErr("TextureChunk::initialize precheck");

    FDEBUG(("texture (re-)initialize\n"));

    // set the image
    OSGGLenum internalFormat = getInternalFormat();
    OSGGLenum externalFormat = img->getPixelFormat();
    OSGGLenum type = img->getDataType();
    UInt32 width = img->getWidth();
    UInt32 height = img->getHeight();
    UInt32 depth = img->getDepth();
    bool   compressedData = img->hasCompressedData();

    glBindTexture(bindtarget, id);

    if (paramtarget != GL_NONE)
    {
        // set the parameters
        glTexParameteri(paramtarget, GL_TEXTURE_MIN_FILTER, getMinFilter());
        glTexParameteri(paramtarget, GL_TEXTURE_MAG_FILTER, getMagFilter());
        glTexParameteri(paramtarget, GL_TEXTURE_WRAP_S, getWrapS());
        glTexParameterf(paramtarget, GL_TEXTURE_LOD_BIAS, getLodBias());
        if (paramtarget == GL_TEXTURE_2D ||
            paramtarget == GL_TEXTURE_3D ||
            paramtarget == GL_TEXTURE_CUBE_MAP ||
            paramtarget == GL_TEXTURE_RECTANGLE
            )
            glTexParameteri(paramtarget, GL_TEXTURE_WRAP_T, getWrapT());
        if (paramtarget == GL_TEXTURE_3D ||
            paramtarget == GL_TEXTURE_CUBE_MAP)
            glTexParameteri(paramtarget, GL_TEXTURE_WRAP_R, getWrapR());

        glTexParameterf(paramtarget, GL_TEXTURE_MAX_ANISOTROPY_EXT, getAnisotropy());

        glTexParameterfv(paramtarget, GL_TEXTURE_BORDER_COLOR, static_cast<const OSGGLfloat*>(getBorderColor().getValuesRGBA()));
        glErr("TextureChunk::initialize params");
    }
    bool doScale = getScale(); // scale the texture to 2^?
    UInt32 frame = getFrame();

    fixFormats(paramtarget, internalFormat, externalFormat, img->getComponents());

    if (imgtarget == GL_TEXTURE_RECTANGLE && needMipmaps)
    {
        SWARNING << "TextureChunk::initialize: Can't do mipmaps"
            << "with GL_TEXTURE_RECTANGLE target! Ignored"
            << std::endl;
        needMipmaps = false;
    }


    // got here needing mipmaps?
    void * data = const_cast<void *>(static_cast<const void *>(img->getData(0, frame, side)));;
    UInt32 datasize = (img->getSideCount() > 1) ? img->getRawSideSize() : img->getRawFrameSize();

    // A image without data is quite handy if you need the
    // texture only on the graphics card. So don't check for data here
    if (compressedData)
    {
        switch (imgtarget)
        {
        case GL_TEXTURE_1D:

            glCompressedTexImage1D(GL_TEXTURE_1D, 0, internalFormat, width, getBorderWidth(), datasize, data);
            if (needMipmaps)
                glGenerateMipmap(GL_TEXTURE_1D);
            break;
        case GL_TEXTURE_2D:
            glCompressedTexImage2D(imgtarget, 0, internalFormat,
                width, height, getBorderWidth(),
                datasize, data);
            if (needMipmaps)
                glGenerateMipmap(GL_TEXTURE_2D);
            break;
        case GL_TEXTURE_CUBE_MAP_POSITIVE_X:
        case GL_TEXTURE_CUBE_MAP_NEGATIVE_X:
        case GL_TEXTURE_CUBE_MAP_POSITIVE_Y:
        case GL_TEXTURE_CUBE_MAP_NEGATIVE_Y:
        case GL_TEXTURE_CUBE_MAP_POSITIVE_Z:
        case GL_TEXTURE_CUBE_MAP_NEGATIVE_Z:
            glCompressedTexImage2D(imgtarget, 0, internalFormat,
                width, height, getBorderWidth(),
                datasize, data);
            if (needMipmaps && imgtarget == GL_TEXTURE_CUBE_MAP_POSITIVE_X)
                glGenerateMipmap(GL_TEXTURE_CUBE_MAP);
            break;
        case GL_TEXTURE_RECTANGLE:
            glCompressedTexImage2D(GL_TEXTURE_RECTANGLE, 0, internalFormat,
                width, height, getBorderWidth(),
                datasize, data);
            break;
        case GL_TEXTURE_3D:
            glCompressedTexImage3D(GL_TEXTURE_3D, 0, internalFormat,
                width, height, depth, getBorderWidth(),
                datasize, data);
            if (needMipmaps)
                glGenerateMipmap(GL_TEXTURE_3D);
            break;
        default:
            SFATAL << "TextureChunk::initialize3: unknown target "
                << imgtarget << "!!!" << std::endl;
        }
    }
    else
    {
        switch (imgtarget)
        {
        case GL_TEXTURE_1D:
            glTexImage1D(GL_TEXTURE_1D, 0, internalFormat,
                width, getBorderWidth(),
                externalFormat, type,
                data);
            if (needMipmaps)
                glGenerateMipmap(GL_TEXTURE_1D);
            break;
        case GL_TEXTURE_2D:
            glTexImage2D(imgtarget, 0, internalFormat,
                width, height, getBorderWidth(),
                externalFormat, type,
                data);
            if (needMipmaps)
                glGenerateMipmap(GL_TEXTURE_2D);
            break;
        case GL_TEXTURE_CUBE_MAP_POSITIVE_X:
        case GL_TEXTURE_CUBE_MAP_NEGATIVE_X:
        case GL_TEXTURE_CUBE_MAP_POSITIVE_Y:
        case GL_TEXTURE_CUBE_MAP_NEGATIVE_Y:
        case GL_TEXTURE_CUBE_MAP_POSITIVE_Z:
        case GL_TEXTURE_CUBE_MAP_NEGATIVE_Z:
            glTexImage2D(imgtarget, 0, internalFormat,
                width, height, getBorderWidth(),
                externalFormat, type,
                data);
            if (needMipmaps && imgtarget == GL_TEXTURE_CUBE_MAP_POSITIVE_X)
                glGenerateMipmap(GL_TEXTURE_CUBE_MAP);
            break;
        case GL_TEXTURE_RECTANGLE:
            glTexImage2D(GL_TEXTURE_RECTANGLE, 0, internalFormat,
                width, height, getBorderWidth(),
                externalFormat, type,
                data);
            break;
        case GL_TEXTURE_3D:
            glTexImage3D(GL_TEXTURE_3D, 0, internalFormat,
                width, height, depth, getBorderWidth(),
                externalFormat, type,
                data);
            if (needMipmaps)
                glGenerateMipmap(GL_TEXTURE_3D);
            break;
        default:
            SFATAL << "TextureChunk::initialize3: unknown target "
                << imgtarget << "!!!" << std::endl;
        }
    }


    if (data != img->getData(0, frame, side))
        free(data);
}


void TextureChunk::prepareTexture(Window *win, UInt32 id)
{
    if (!m_imageChanged)
        return;

    OSGGLenum target;

    ImagePtr img = getImage();

    if (img != NullFC)
    {
        if (img->getSideCount() == 1)
        {
            target = getTarget();
            if (target == GL_NONE)
            {
                if (img->getDepth() > 1)
                {
                    target = GL_TEXTURE_3D;
                }
                else if (img->getHeight() > 1)
                    target = GL_TEXTURE_2D;
                else
                    target = GL_TEXTURE_1D;
            }
            handleTexture(win, id, target, target, target, img);
            m_imageChanged = false;
        }
        else // Got a cubemap...
        {
            handleTexture(win, id,
                GL_TEXTURE_CUBE_MAP,
                GL_TEXTURE_CUBE_MAP,
                GL_TEXTURE_CUBE_MAP_NEGATIVE_Z,
                getImage(), 5);
            // Have to use initialize mode here, otherwise the
            // texture is destroyed for every side      
            handleTexture(win, id,
                GL_TEXTURE_CUBE_MAP,
                GL_TEXTURE_CUBE_MAP, //GL_NONE,
                GL_TEXTURE_CUBE_MAP_POSITIVE_Z,
                getImage(), 4);
            handleTexture(win, id,
                GL_TEXTURE_CUBE_MAP,
                GL_TEXTURE_CUBE_MAP, //GL_NONE,
                GL_TEXTURE_CUBE_MAP_NEGATIVE_Y,
                getImage(), 3);
            handleTexture(win, id,
                GL_TEXTURE_CUBE_MAP,
                GL_TEXTURE_CUBE_MAP, //GL_NONE,
                GL_TEXTURE_CUBE_MAP_POSITIVE_Y,
                getImage(), 2);
            handleTexture(win, id,
                GL_TEXTURE_CUBE_MAP,
                GL_TEXTURE_CUBE_MAP, //GL_NONE,
                GL_TEXTURE_CUBE_MAP_NEGATIVE_X,
                getImage(), 1);
            handleTexture(win, id,
                GL_TEXTURE_CUBE_MAP,
                GL_TEXTURE_CUBE_MAP, //GL_NONE,g
                GL_TEXTURE_CUBE_MAP_POSITIVE_X,
                getImage(), 0);
            m_imageChanged = false;
        }
    }
}



void TextureChunk::activate(DrawActionBase *action, UInt32 idx)
{
    Window *win = action->getWindow();

    glActiveTexture(GL_TEXTURE0 + idx);

    if (m_textureId == 0)
    {
        glGenTextures(1, &m_textureId);
        if (m_textureId == 0)
            return;
    }
    prepareTexture(win, m_textureId);

   
    ImagePtr img = getImage();
    OSGGLenum target = getTarget();

    if (img == NullFC || !img->getDimension()) // no image ?
        return;

    glErr("TextureChunk::activate precheck");

    if (img->getSideCount() == 1)
    {
        if (target == GL_NONE)
        {
            if (img->getDepth() > 1)
            {
                target = GL_TEXTURE_3D;
            }
            else if (img->getHeight() > 1)
            {
                target = GL_TEXTURE_2D;
            }
            else
            {
                target = GL_TEXTURE_1D;
            }
        }
    }
    else
    {
        target = GL_TEXTURE_CUBE_MAP;
    }
    glBindTexture(target, m_textureId);

    if (idx >= _needTexMat.size())
    {
        _needTexMat.resize(idx + 1, false);
        _lastTexMat.resize(idx + 1, Matrix::identity());
    }
    _needTexMat[idx] = false;
    _lastTexMat[idx].setIdentity();

    glErr("TextureChunk::activate");
}

void TextureChunk::deactivate(DrawActionBase *action, UInt32 idx)
{
    Window *win = action->getWindow();
    ImagePtr img = getImage();
    OSGGLenum target = getTarget();

    if (img == NullFC || !img->getDimension())
        return;

    glErr("TextureChunk::deactivate precheck");
    glActiveTexture(GL_TEXTURE0 + idx);

    if (img->getSideCount() == 1)
    {
        if (target == GL_NONE)
        {
            if (img->getDepth() > 1)
            {
                target = GL_TEXTURE_3D;
            }
            else if (img->getHeight() > 1)
                target = GL_TEXTURE_2D;
            else
                target = GL_TEXTURE_1D;
        }
    }
    else
    {
        target = GL_TEXTURE_CUBE_MAP;
    }
    glBindTexture(target, 0); // this would be the clean way but probably not the most efficient
    if (idx >= _needTexMat.size())
    {
        _needTexMat.resize(idx + 1, false);
        _lastTexMat.resize(idx + 1, Matrix::identity());
    }
    _needTexMat[idx] = false;
    _lastTexMat[idx].setIdentity();

    glErr("TextureChunk::deactivate");
}

osg::UInt32 TextureChunk::getTextureId(Window *win)
{
    if (m_textureId == 0)
    {
        glGenTextures(1, &m_textureId);
        if (m_textureId == 0)
            return 0;
    }
    prepareTexture(win, m_textureId);

    return m_textureId;
}

/*-------------------------- Comparison -----------------------------------*/


bool TextureChunk::operator < (const StateChunk &other) const
{
    return this < &other;
}

bool TextureChunk::operator == (const StateChunk &other) const
{
    TextureChunk const *tother = dynamic_cast<TextureChunk const*>(&other);

    if (!tother)
        return false;

    if (tother == this)
        return true;

    bool returnValue =
        getImage() == tother->getImage() &&
        getMinFilter() == tother->getMinFilter() &&
        getMagFilter() == tother->getMagFilter() &&
        getWrapS() == tother->getWrapS() &&
        getWrapT() == tother->getWrapT() &&
        getWrapR() == tother->getWrapR() &&
        getNPOTMatrixScale() == tother->getNPOTMatrixScale();

    if (returnValue == true && getEnvMode() == GL_COMBINE_EXT)
    {
        returnValue =
            getEnvCombineRGB() == tother->getEnvCombineRGB() &&

            getEnvSource0RGB() == tother->getEnvSource0RGB() &&
            getEnvSource1RGB() == tother->getEnvSource1RGB() &&
            getEnvSource2RGB() == tother->getEnvSource2RGB() &&

            getEnvOperand0RGB() == tother->getEnvOperand0RGB() &&
            getEnvOperand1RGB() == tother->getEnvOperand1RGB() &&
            getEnvOperand2RGB() == tother->getEnvOperand2RGB() &&

            getEnvCombineAlpha() == tother->getEnvCombineAlpha() &&

            getEnvSource0Alpha() == tother->getEnvSource0Alpha() &&
            getEnvSource1Alpha() == tother->getEnvSource1Alpha() &&
            getEnvSource2Alpha() == tother->getEnvSource2Alpha() &&

            getEnvOperand0Alpha() == tother->getEnvOperand0Alpha() &&
            getEnvOperand1Alpha() == tother->getEnvOperand1Alpha() &&
            getEnvOperand2Alpha() == tother->getEnvOperand2Alpha();

        returnValue &=
            ((getEnvScaleRGB() - tother->getEnvScaleRGB()) < Eps) &&
            ((tother->getEnvScaleRGB() - getEnvScaleRGB()) < Eps) &&
            ((getEnvScaleAlpha() - tother->getEnvScaleAlpha()) < Eps) &&
            ((tother->getEnvScaleAlpha() - getEnvScaleAlpha()) < Eps);
    }

    return returnValue;
}

bool TextureChunk::operator != (const StateChunk &other) const
{
    return !(*this == other);
}


/*------------------------------------------------------------------------*/
/*                              cvs id's                                  */

#ifdef OSG_SGI_CC
#pragma set woff 1174
#endif

#ifdef OSG_LINUX_ICC
#pragma warning( disable : 177 )
#endif

namespace
{
    static Char8 cvsid_cpp[] = "@(#)$Id: FCTemplate_cpp.h,v 1.13 2002/06/01 10:37:25 vossg Exp $";
    static Char8 cvsid_hpp[] = OSGTEXTURECHUNK_HEADER_CVSID;
    static Char8 cvsid_inl[] = OSGTEXTURECHUNK_INLINE_CVSID;

    static Char8 cvsid_fields_hpp[] = OSGTEXTURECHUNKFIELDS_HEADER_CVSID;
}

#ifdef __sgi
#pragma reset woff 1174
#endif

