/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright (C) 2000-2002 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/
//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------
#include <stdlib.h>
#include <stdio.h>
#include <iostream>
#include <sstream>

#include <OSGConfig.h>
#include <OSGGL.h>

#include <OSGNodePtr.h>
#include <OSGImage.h>

#include <OSGTextTXFFace.h>
#include <OSGTextTXFGlyph.h>
#include <OSGTextLayoutParam.h>
#include <OSGTextLayoutResult.h>
#include <OSGRemoteAspect.h>
#include <OSGViewport.h>
#include "OSGWindow.h"
#include "OSGSimpleStatisticsForeground.h"

#include "OSGStatisticsDefaultFont.h"

OSG_USING_NAMESPACE

/*! \class osg::SimpleStatisticsForeground
    \ingroup GrpSystemWindowForegroundsStatistics

SimpleStatisticsForeground displays the Statistics info as simple text lines.
See \ref PageSystemWindowForegroundStatisticsSimple for a description.

The format string for the given elements are stored in the _mfFormats Field,
the size and color used for all lines in _sfSize and _sfColor.
*/

/*----------------------- constructors & destructors ----------------------*/
SimpleStatisticsForeground::SimpleStatisticsForeground(void) :
    Inherited(), _face(0), _texchunk(), 
    m_vertexArrayId(0),
    m_vertexBufferId(0),
    m_viewportBufferId(0),
    m_shlChunk(NullFC),
    m_colorUniformLocation(-1),
    m_textureMapUniformLocation(-1),
    m_textScaleUniformLocation(-1),
    m_useTextureUniformLocation(-1),
    m_textPixelOffsetUniformLocation(-1),
    m_textOffsetUniformLocation(-1)
{
}

SimpleStatisticsForeground::SimpleStatisticsForeground(const SimpleStatisticsForeground &source) :
Inherited(), _face(source._face), _texchunk(source._texchunk), m_shlChunk(source.m_shlChunk),
    m_vertexArrayId(0),
    m_vertexBufferId(0),
    m_viewportBufferId(0),
    m_colorUniformLocation(-1),
    m_textureMapUniformLocation(-1),
    m_textScaleUniformLocation(-1),
    m_useTextureUniformLocation(-1),
    m_textPixelOffsetUniformLocation(-1),
    m_textOffsetUniformLocation(-1)
{
    if (_face != 0)
        addRefP(_face);
    if (_texchunk != NullFC)
        addRefCP(_texchunk);
    if (m_shlChunk != NullFC)
        addRefCP(m_shlChunk);

}

/* */
SimpleStatisticsForeground::~SimpleStatisticsForeground(void)
{
   
}

void SimpleStatisticsForeground::onCreate(const SimpleStatisticsForeground *source)
{
    Inherited::onCreate(source);

    // ignore prototypes.
    if (GlobalSystemState == Startup)
        return;
}

void SimpleStatisticsForeground::onDestroy(void)
{
    Inherited::onDestroy();

    if (_face != 0)
        subRefP(_face);
    if (_texchunk != NullFC)
        subRefCP(_texchunk);
    if (m_shlChunk != NullFC)
        subRefCP(m_shlChunk);

    if (osgMakeCurrent())
    {
        if (m_vertexArrayId != 0)
        {
            glDeleteVertexArrays(1, &m_vertexArrayId);
            m_vertexArrayId = 0;
        }

        if (m_vertexBufferId != 0)
        {
            glDeleteBuffers(1, &m_vertexBufferId);
            m_vertexBufferId = 0;
        }
        if (m_viewportBufferId != 0)
        {
            glDeleteBuffers(1, &m_viewportBufferId);
            m_viewportBufferId = 0;
        }
    }
    m_vertices.clear();
}


/*----------------------------- class specific ----------------------------*/
void SimpleStatisticsForeground::initMethod(void)
{
}

/* */
void SimpleStatisticsForeground::changed(BitVector whichField, UInt32 origin)
{
    Inherited::changed(whichField, origin);
}

/* */
void SimpleStatisticsForeground::dump(UInt32, const BitVector) const
{
    SLOG << "Dump SimpleStatisticsForeground NI" << std::endl;
}

/*! Convenience function to add an element and format.
*/
void SimpleStatisticsForeground::addElement(StatElemDescBase &desc,
                                            const char *format)
{
    editMFElementIDs()->push_back(desc.getID());
    editMFFormats()->push_back(format ? format : "");
}

/*! Convenience function to add an element and format.
*/
void SimpleStatisticsForeground::addElement(Int32 id, const char *format)
{
    editMFElementIDs()->push_back(id);
    editMFFormats()->push_back(format ? format : "");
}

/*! Convenience function to add a line of text.
*/
void SimpleStatisticsForeground::addText( const char *text )
{
    addElement( -1, text );
}

/*! Convenience function to clear all elements.
*/
void SimpleStatisticsForeground::clearElems( void )
{
    editMFElementIDs()->clear();
    editMFFormats()->clear();
    editCollector().clearElems();
}

/*! Initialize the text used. It is compiled into the library as
    StatisticsDefaultFontData and used as a TXF font.
*/
void SimpleStatisticsForeground::initText(const std::string &family, Real32 size)
{
    // Cleanup
    if (_face != 0)
    {
        subRefP(_face);
        _face = 0;
    }
    if (_texchunk != NullFC)
    {
        subRefCP(_texchunk);
        _texchunk = NullFC;
    }

    if (m_shlChunk != NullFC)
    {
        subRefCP(m_shlChunk);
        m_shlChunk = NullFC;
    }
    // Create the font
    if (family.empty() == false)
    {
        TextTXFParam param;
        param.size = static_cast<UInt32>(size);
        _face = TextTXFFace::create(family, TextFace::STYLE_PLAIN, param);
        if (_face != 0)
        {
            _texchunk = TextureChunk::create();
            beginEditCP(_texchunk, osg::FieldBits::AllFields);
            {
                ImagePtr texture = _face->getTexture();
                _texchunk->setImage(texture);
                _texchunk->setWrapS(GL_CLAMP_TO_EDGE);
                _texchunk->setWrapT(GL_CLAMP_TO_EDGE);
                _texchunk->setMinFilter(GL_NEAREST);
                _texchunk->setMagFilter(GL_NEAREST);
            }
            endEditCP(_texchunk, osg::FieldBits::AllFields);
        }
    }

    // We failed to create the font - fallback to the default font
    if (_face == 0)
    {
        _face = getStatisticsDefaultFont();
        _texchunk = getStatisticsDefaultFontTexture();
    }

    if (m_shlChunk == NullFC)
    {
        std::string vp = SHLChunk::getShaderViewportUniformString();
        vp += "layout (location = 0) in vec2 inPosition; \n"
            "layout (location = 1) in vec2 inTexCoord; \n"
            "out vec2 texCoord; \n"
            "uniform vec2 textOffset; \n"
            //"uniform vec2 textSize; \n"
            "uniform float textScale; \n"
            "uniform vec2 textPixelOffset; \n"
            "void main(void) \n"
            "{\n"
            "vec2 normalizedPosition = ((inPosition * textScale) + textPixelOffset + textOffset)  / viewport.pixelResolution; \n"
            "vec2 orthoScale = vec2(2.0 / (viewport.region[1] - viewport.region[0]), 2.0 / (viewport.region[3] - viewport.region[2]));\n"
            "vec2 orthoOffset = vec2(-(viewport.region[0] + viewport.region[1]) / (viewport.region[1] - viewport.region[0]), (viewport.region[2] + viewport.region[3]) / (viewport.region[3] - viewport.region[2]));\n"
            "vec2 devicePosition = (orthoScale * normalizedPosition) + orthoOffset;\n"
            "gl_Position = vec4(devicePosition, 0.0, 1.0);\n"
            "texCoord = inTexCoord.xy; \n"
            "}\n";

        std::string fp = "uniform vec4 color; \n"
            "uniform sampler2D textureMap;\n"
            "uniform bool useTexture; \n"
            "in vec2 texCoord; \n"
            "layout (location = 0) out vec4 colorBufferRGBA;\n"
            "void main()  \n"
            "{  \n"
            "   if(useTexture) \n"
            "       colorBufferRGBA = vec4( color.rgb, texture(textureMap, texCoord).a); \n"
            "   else\n"
            "       colorBufferRGBA = color; \n"
            "} \n";

        m_shlChunk = SHLChunk::create();
        beginEditCP(m_shlChunk, osg::FieldBits::AllFields);
        m_shlChunk->setSingleViewRenderingOnly(true);
        m_shlChunk->setVertexProgram(vp);
        m_shlChunk->setGeometryProgram("");
        m_shlChunk->setFragmentProgram(fp);
        m_shlChunk->clearUniformParameters();
        endEditCP(m_shlChunk, osg::FieldBits::AllFields);

        //addRefCP(m_shlChunk);

        m_colorUniformLocation = -1;
        m_textureMapUniformLocation = -1;
        m_textScaleUniformLocation = -1;
        m_useTextureUniformLocation = -1;
        m_textPixelOffsetUniformLocation = -1;
        m_textOffsetUniformLocation = -1;
    }


    // Increment reference counters
    addRefP(_face);
    addRefCP(_texchunk);
    addRefCP(m_shlChunk);
  
}

/*! Draw the statistics lines.
*/
void SimpleStatisticsForeground::draw(DrawActionBase *action, Viewport *port)
{
    if (getActive() == false)
        return;
	
    if (_face == 0 || _face->getParam().size != static_cast<UInt32>(getSize()))
        initText(getFamily(), getSize());
    
    if (!getCollector().getNumOfElems() && !getMFElementIDs()->size())
        return; // nothing to do

    Real32  pw = Real32(port->getPixelWidth ());
    Real32  ph = Real32(port->getPixelHeight());

    if (pw < 1 || ph < 1)
        return;

    Window* win = action->getWindow();
    if (win == NULL)
        return;


    if (m_vertexArrayId == 0)
        glGenVertexArrays(1, &m_vertexArrayId);

   
    vrGLViewport viewportData;
    viewportData.rec709toRCS.setIdentity();
    // to adjust for split stereo we need to adjust the viewport region by using the inverse
    if (port->getRight() > 1.0f || port->getTop() > 1.0f)
    {
        osg::Vec4f region = osg::Vec4f(port->getPixelLeft() / pw, (port->getPixelRight() + 1)/ pw, port->getPixelBottom() / ph, (port->getPixelTop() + 1) / ph);
        viewportData.region = osg::Vec4f(region[0], 1.0f / region[1], region[2], 1.0f / region[3]);
    }
    else
        viewportData.region = osg::Vec4f(port->getLeft(), 1.0f / port->getRight(), port->getBottom(), 1.0f / port->getTop());
    
    viewportData.pixelRegion = osg::Vec4f(port->getPixelLeft(), port->getPixelRight(), port->getPixelBottom(), port->getPixelTop());
    viewportData.pixelResolution = osg::Vec2f(port->getPixelWidth(), port->getPixelHeight());
    viewportData.aspect = osg::Vec2f(1.0f, 1.0f); // square assumption as for now

    if (m_viewportBufferId == 0)
    {
        glGenBuffers(1, &m_viewportBufferId);
        glBindBuffer(GL_UNIFORM_BUFFER, m_viewportBufferId);
        glBufferData(GL_UNIFORM_BUFFER, sizeof(vrGLViewport), &viewportData, GL_DYNAMIC_DRAW);
        glBindBuffer(GL_UNIFORM_BUFFER, 0);
    }
    else
    {
        glBindBuffer(GL_UNIFORM_BUFFER, m_viewportBufferId);
        glBufferSubData(GL_UNIFORM_BUFFER, 0, sizeof(vrGLViewport), &viewportData);
        glBindBuffer(GL_UNIFORM_BUFFER, 0);
    }

    pushGLFlags();
    // set blending functions
    disableGLFlag(GL_DEPTH_TEST);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    enableGLFlag(GL_BLEND);

    // retrieve text
    std::vector < std::string > stat;
    StatCollector   *col = const_cast<StatCollector *>(&getCollector());
    StatElem        *el;

    if(getMFElementIDs()->size() != 0)
    {
        for(UInt32 i = 0; i < getMFElementIDs()->size(); ++i)
        {
            Int32 id(getElementIDs(i));
            el = ((id >= 0) ? col->getElem(id) : 0);

            stat.resize(stat.size() + 1);
            std::vector < std::string >::iterator str = stat.end() - 1;

            const char  *format = NULL;
            if(i < getMFFormats()->size() && getFormats(i).length())
            {
                format = getFormats(i).c_str();
            }

            if (el)
            {
                el->putToString(*str, format);
            }
            else
                *str = format;
        }
    }
    else    // fallback, show all elements
    {
        for(UInt32 i = 0; i < col->getNumOfElems(); ++i)
        {
            el = col->getElem(i, false);
            if(el)
            {
                std::string desc(el->getDesc()->getName().str()), eltext;

                el->putToString(eltext);
                desc = desc + " : " + eltext;

                stat.resize(stat.size() + 1);
                std::vector < std::string >::iterator str = stat.end() - 1;
                *str = desc;
            }
        }
    }
    
    // split each line into columns on tab characters to allow proper alignment
    std::vector< std::vector< std::string > > statGrid;
    int row = 0;
    for (const std::string &line : stat)
    {
        std::stringstream ss(line);
        std::string item;
        int column = 0;
        while (std::getline(ss, item, '\t'))
        {
            if (statGrid.size() <= column)
            {
                // current line contains more columns then previous lines
                statGrid.push_back(std::vector<std::string>());
                // fill the new column with empty items for all previous lines
                for (int r = 0; r < row; ++r)
                {
                    statGrid.back().push_back(std::string());
                }
            }
            
            // add the item for current line
            statGrid[column].push_back(item);
            
            ++column;
        }
        
        // if current line has fewer columns add empty strings to any remaining columns
        for (; column < statGrid.size(); ++column)
        {
            statGrid[column].push_back(std::string());
        }
        
        ++row;
    }
    
    TextLayoutParam layoutParam;
#ifdef WIN32
    layoutParam.spacing = 1.1;
#else
    layoutParam.spacing = 1.7;
#endif
    
    layoutParam.majorAlignment = TextLayoutParam::ALIGN_BEGIN;
    layoutParam.minorAlignment = TextLayoutParam::ALIGN_BEGIN;

    Real32 scale = 1 / _face->getScale();
    UInt32 numGlyphs = 0;
    
    // Do layout of columns
    std::vector<TextLayoutResult> columnLayouts;
    std::vector<Real32> columnOffset;
    // offset for first column
    columnOffset.push_back(0);
    
    for (const std::vector<std::string> &column : statGrid)
    {
        columnLayouts.push_back(TextLayoutResult());
        _face->layout(column, layoutParam, columnLayouts.back());
        columnOffset.push_back(columnOffset.back() + columnLayouts.back().textBounds.x() * scale + getTextMargin().x());
        numGlyphs += columnLayouts.back().getNumGlyphs();
    }
    
    Real32 size = _face->getParam().size;
    Real32 textWidth = columnOffset.back() + size + getTextMargin().x() * 2.0f;
    Real32 textHeight = columnLayouts.back().textBounds.y() * scale + size + getTextMargin().y() * 2.0f;
    
#if 0
    // Let's do some simple form of layout management
    Real32 orthoX = 0, orthoY = ph;

    switch ( getHorizontalAlign() )
    {
        case Right:
            orthoX = pw - textWidth;
            break;
        case Middle:
            orthoX = (pw - textWidth) * 0.5;
            break;
        case Left:
        default:
            break;
    }

    switch ( getVerticalAlign() )
    {
        case Bottom:
            orthoY = textHeight;
            break;
        case Center:
            orthoY = (ph - textHeight) * 0.5 + textHeight;
            break;
        case Top:
        default:
            break;
    }
#endif


    GLuint shaderProgram = m_shlChunk->getProgramId();
    if (shaderProgram == 0)
        return;

    if (m_colorUniformLocation == -1 || m_textureMapUniformLocation == -1 || m_textScaleUniformLocation == -1 ||
        m_useTextureUniformLocation == -1 || m_textPixelOffsetUniformLocation == -1 || m_textOffsetUniformLocation == -1)
    {
        m_colorUniformLocation = glGetUniformLocation(shaderProgram, "color");
        m_textureMapUniformLocation = glGetUniformLocation(shaderProgram, "textureMap");
        m_textScaleUniformLocation = glGetUniformLocation(shaderProgram, "textScale");
        m_useTextureUniformLocation = glGetUniformLocation(shaderProgram, "useTexture");
        m_textPixelOffsetUniformLocation = glGetUniformLocation(shaderProgram, "textPixelOffset");
        m_textOffsetUniformLocation = glGetUniformLocation(shaderProgram, "textOffset");
    }
   
    glBindVertexArray(m_vertexArrayId);
    createBuffer(columnLayouts, numGlyphs, textWidth, textHeight); // create the buffer
  
    glBindBufferBase(GL_UNIFORM_BUFFER, UBO_VIEWPORT, m_viewportBufferId);
    
    _texchunk->activate(action);
    m_shlChunk->activate(action);

    glUniform4f(m_colorUniformLocation, getBgColor()[0], getBgColor()[1], getBgColor()[2], getBgColor()[3]);
    glUniform1i(m_textureMapUniformLocation, 0);
    glUniform1f(m_textScaleUniformLocation, 1.0f);
    glUniform1i(m_useTextureUniformLocation, false);
    glUniform2f(m_textPixelOffsetUniformLocation, 0.0f, 0.0f);
    glUniform2f(m_textOffsetUniformLocation, 0.0f, 0.0f);

    glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);

    glUniform4f(m_colorUniformLocation, getBorderColor()[0], getBorderColor()[1], getBorderColor()[2], getBorderColor()[3]);


    glDrawArrays(GL_LINE_LOOP, 4, 4);
    
    glUniform1i(m_useTextureUniformLocation, true);
    glUniform1f(m_textScaleUniformLocation, scale);

    UInt32 drawnGlyphs = 0;
    for (size_t i = 0; i < columnLayouts.size(); ++i)
    {
        glUniform4f(m_colorUniformLocation, getShadowColor()[0], getShadowColor()[1], getShadowColor()[2], getShadowColor()[3]);
        glUniform1i(m_textureMapUniformLocation, 0);
        glUniform2f(m_textPixelOffsetUniformLocation, getShadowOffset()[0] + columnOffset[i], getShadowOffset()[1]);
#ifdef WIN32
        glUniform2f(m_textOffsetUniformLocation, 0.5 * size + getTextMargin().x(), -0.5 * size - getTextMargin().y());
#else
        glUniform2f(m_textOffsetUniformLocation, 0.5 * size + getTextMargin().x(), -0.5 * size);
#endif
        glDrawArrays(GL_TRIANGLES, 8 + drawnGlyphs * 6, static_cast<GLuint>(columnLayouts[i].getNumGlyphs() * 6));

        // draw the text
        glUniform4f(m_colorUniformLocation, getColor()[0], getColor()[1], getColor()[2], 1.0f);
        glUniform2f(m_textPixelOffsetUniformLocation, columnOffset[i], 0.0f);
        glDrawArrays(GL_TRIANGLES, 8 + drawnGlyphs * 6, static_cast<GLuint>(columnLayouts[i].getNumGlyphs() * 6));

        drawnGlyphs += columnLayouts[i].getNumGlyphs();
    }

    m_shlChunk->deactivate(action);
    _texchunk->deactivate(action);
    
    glBindVertexArray(0);
    glDisableVertexAttribArray(0);
    glDisableVertexAttribArray(1);
    glBindBuffer(GL_ARRAY_BUFFER, 0);
    glBindBufferBase(GL_UNIFORM_BUFFER, 0, 0);
   

    // restore to default
    enableGLFlag(GL_DEPTH_TEST);
    disableGLFlag(GL_BLEND);
    popGLFlags();
}

void SimpleStatisticsForeground::createBuffer(const std::vector<TextLayoutResult> &layoutResults, const osg::UInt32 numGlyphs, const osg::Real32 width, const osg::Real32 height)
{
    if (m_vertices.size() < 8 + numGlyphs * 6)
        m_vertices.reserve(8 + numGlyphs * 6); // need to paint triangles, so reserve enough for them

    m_vertices.clear(); // clear but keep the memory

    //  background
    VBOVertex bgLL, bgLR, bgTR, bgTL;
    bgLL.m_pos = osg::Vec2f(0.0f, -height);
    bgLL.m_texCoord = osg::Vec2f(0.0f, 0.0f);

    bgLR.m_pos = osg::Vec2f(width, -height);
    bgLR.m_texCoord = osg::Vec2f(1.0f, 0.0f);

    bgTR.m_pos = osg::Vec2f(width, 0.0f);
    bgTR.m_texCoord = osg::Vec2f(1.0f, 1.0f);

    bgTL.m_pos = osg::Vec2f( 0.0f, 0.0f);
    bgTL.m_texCoord = osg::Vec2f(0.0f, 1.0f);

    m_vertices.push_back(bgLL);
    m_vertices.push_back(bgLR);
    m_vertices.push_back(bgTL);
    m_vertices.push_back(bgTR);

    //  border
    bgLL.m_pos = osg::Vec2f(getBorderOffset().x(), -height + 1 + getBorderOffset().y());
    bgLR.m_pos = osg::Vec2f(width - 1 - getBorderOffset().x(), -height + 1 + getBorderOffset().y());
    bgTR.m_pos = osg::Vec2f(width - 1 - getBorderOffset().x(), -1 - getBorderOffset().y());
    bgTL.m_pos = osg::Vec2f(getBorderOffset().x(), -1 - getBorderOffset().y());

    m_vertices.push_back(bgLL);
    m_vertices.push_back(bgLR);
    m_vertices.push_back(bgTR);
    m_vertices.push_back(bgTL);
    
    for (int column = 0; column < layoutResults.size(); ++column)
    {
        const TextLayoutResult &layout = layoutResults[column];
        for (int i = 0; i < layout.getNumGlyphs(); ++i)
        {
            const TextTXFGlyph &glyph = _face->getTXFGlyph(layout.indices[i]);
            float width = glyph.getWidth();
            float height = glyph.getHeight();
            // No need to draw invisible glyphs
            if ((width <= 0.f) || (height <= 0.f))
                continue;

            // Calculate coordinates
            Vec2f pos = layout.positions[i];
            Real32 posLeft = pos.x();
            Real32 posTop = pos.y();
            Real32 posRight = pos.x() + width;
            Real32 posBottom = pos.y() - height;

            Real32 texCoordLeft = glyph.getTexCoord(TextTXFGlyph::COORD_LEFT);
            Real32 texCoordTop = glyph.getTexCoord(TextTXFGlyph::COORD_TOP);
            Real32 texCoordRight = glyph.getTexCoord(TextTXFGlyph::COORD_RIGHT);
            Real32 texCoordBottom = glyph.getTexCoord(TextTXFGlyph::COORD_BOTTOM);

            VBOVertex vertexLL;
            vertexLL.m_pos = osg::Vec2f(posLeft, posBottom);
            vertexLL.m_texCoord = osg::Vec2f(texCoordLeft, texCoordBottom);
            
            VBOVertex vertexLR;
            vertexLR.m_pos = osg::Vec2f(posRight, posBottom);
            vertexLR.m_texCoord = osg::Vec2f(texCoordRight, texCoordBottom);

            VBOVertex vertexTL;
            vertexTL.m_pos = osg::Vec2f(posLeft, posTop);
            vertexTL.m_texCoord = osg::Vec2f(texCoordLeft, texCoordTop);

            VBOVertex vertexTR;
            vertexTR.m_pos = osg::Vec2f(posRight, posTop);
            vertexTR.m_texCoord = osg::Vec2f(texCoordRight, texCoordTop);

            m_vertices.push_back(vertexLL);
            m_vertices.push_back(vertexLR);
            m_vertices.push_back(vertexTR);
            m_vertices.push_back(vertexLL);
            m_vertices.push_back(vertexTR);
            m_vertices.push_back(vertexTL);
        }
    }
    if (m_vertexBufferId == 0)
        glGenBuffers(1, &m_vertexBufferId);

    glBindBuffer(GL_ARRAY_BUFFER, m_vertexBufferId);
    glBufferData(GL_ARRAY_BUFFER, m_vertices.size() * sizeof(VBOVertex), &(m_vertices[0]), GL_STREAM_DRAW);
    
    glVertexAttribPointer(0, 2, GL_FLOAT, GL_FALSE, sizeof(VBOVertex), (GLvoid*)offsetof(VBOVertex, m_pos));
    glEnableVertexAttribArray(0);
    
    glVertexAttribPointer(1, 2, GL_FLOAT, GL_FALSE, sizeof(VBOVertex), (GLvoid*)offsetof(VBOVertex, m_texCoord));
    glEnableVertexAttribArray(1);
}

/*-------------------------------------------------------------------------*/
/*                              cvs id's                                   */
#ifdef __sgi
#pragma set woff 1174
#endif
#ifdef OSG_LINUX_ICC
#pragma warning(disable : 177)
#endif
namespace
{
static char cvsid_cpp[] = "@(#)$Id: OSGSimpleStatisticsForeground.cpp,v 1.11 2002/08/07 04:04:13 vossg Exp $";
static char cvsid_hpp[] = OSGSIMPLESTATISTICSFOREGROUND_HEADER_CVSID;
static char cvsid_inl[] = OSGSIMPLESTATISTICSFOREGROUND_INLINE_CVSID;
}
