/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright (C) 2000-2002 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

#ifdef _MSC_VER
# pragma warning (disable: 4786)
#endif

#include "OSGTextTXFFace.h"
#include "OSGTextTXFGlyph.h"
#include "OSGTextLayoutParam.h"
#include "OSGTextLayoutResult.h"
#include "OSGTextFaceFactory.h"

#include <fstream>
#include <functional>
#include <algorithm>
#include <set>
#ifdef __sgi
# include <assert.h>
#else
# include <cassert>
#endif


using namespace std;


OSG_BEGIN_NAMESPACE


//----------------------------------------------------------------------
// Static Class Variable implementations
// Author: pdaehne
//----------------------------------------------------------------------
TextTXFGlyph TextTXFFace::_emptyGlyph;


//----------------------------------------------------------------------
// Nested struct
// Author: jincheng li
//----------------------------------------------------------------------
struct TextTXFFace::LayoutData
{
    UInt32 gap;
    UInt32 width;

    UInt32 xpos;
    UInt32 ypos;
    UInt32 hrow;

    UInt32 height;

    // A flag indicates if the incremental algorithm
    // works on this instance.
    bool   incremental;

    // This container is not part of the data needed 
    // by the incremental lay out algorithm. But it is
    // just put here to avoid exposing too many details
    // in the header.
    std::set<GeometryPtr> textGeoms;
};


//----------------------------------------------------------------------
// Constructor
// Author: pdaehne
//----------------------------------------------------------------------
TextTXFFace::TextTXFFace()
    : TextFace(), _scale(), _param(),
    _texture(NullFC), _glyphMap(), _layoutData(new LayoutData)
{
    _layoutData->incremental = false;
}


//----------------------------------------------------------------------
// Destructor
// Author: pdaehne
//----------------------------------------------------------------------
TextTXFFace::~TextTXFFace()
{
    // Delete all glyphs in the glyph cache
    GlyphMap::iterator it;
    for (it = _glyphMap.begin(); it != _glyphMap.end(); ++it)
    {
        assert(it->second != 0);
        delete it->second;
    }

    if (_layoutData)
    {
        // Release all the shared ownership of the geometries we keep track of.
        for (auto iter = _layoutData->textGeoms.begin(); iter != _layoutData->textGeoms.end(); ++iter)
        {
            if (NullFC != *iter)
                subRefCP(*iter);
        }
        delete _layoutData;
    }

    // Delete the texture
    if (NullFC != _texture)
        subRefCP(_texture);
}


//----------------------------------------------------------------------
// Returns information about a glyph.
// Author: pdaehne
//----------------------------------------------------------------------
const TextGlyph &TextTXFFace::getGlyph(TextGlyph::Index glyphIndex)
{
    return getTXFGlyph(glyphIndex);
}


//----------------------------------------------------------------------
// Returns information about a glyph.
// Author: pdaehne
//----------------------------------------------------------------------
const TextTXFGlyph &TextTXFFace::getTXFGlyph(TextGlyph::Index glyphIndex)
{
    // Try to find the glyph in the map of glyphs
    GlyphMap::const_iterator it = _glyphMap.find(glyphIndex);
    if (it != _glyphMap.end())
    {
        assert(it->second != 0);
        return *(it->second);
    }

    // We did not find the glyph in the map of glyphs,
    // so try to convert uppercase letters to lowercase
    // letters and vice versa
    if (glyphIndex > 255)
        glyphIndex = TextGlyph::INVALID_INDEX;
    else if (isupper(glyphIndex))
        glyphIndex = tolower(glyphIndex);
    else if (islower(glyphIndex))
        glyphIndex = toupper(glyphIndex);
    else
        glyphIndex = TextGlyph::INVALID_INDEX;
    it = _glyphMap.find(glyphIndex);
    if (it != _glyphMap.end())
    {
        assert(it->second != 0);
        return *(it->second);
    }

    return _emptyGlyph;
}


//----------------------------------------------------------------------
// Fills a geometry with a new text
// Author: afischle, pdaehne
//----------------------------------------------------------------------
void TextTXFFace::fillGeo(GeometryPtr &geoPtr, const TextLayoutResult &layoutResult, Real32 scale,
                          Vec2f offset, Color3f color)
{
    // cast the field containers down to the needed type and create them
    // when they have the wrong type
    GeoPositions3fPtr posPtr = GeoPositions3fPtr::dcast(geoPtr->getPositions());
    if (posPtr != NullFC)
       posPtr->clear();

    // Clear out any existing data and then add to the geom
    GeoNormals3fPtr normalPtr = GeoNormals3fPtr::dcast(geoPtr->getNormals());
    if (normalPtr != NullFC)
       normalPtr->clear();

    GeoTexCoords2fPtr texPtr = GeoTexCoords2fPtr::dcast(geoPtr->getTexCoords());
    if (texPtr != NullFC)
       texPtr->clear();

    GeoColors3fPtr colorPtr = GeoColors3fPtr::dcast(geoPtr->getColors());
    if (NullFC != colorPtr)
       colorPtr->clear();

    GeoPLengthsUI32Ptr lensPtr = GeoPLengthsUI32Ptr::dcast(geoPtr->getLengths());
    if (lensPtr != NullFC)
       lensPtr->clear();

    GeoPTypesUI8Ptr typesPtr = GeoPTypesUI8Ptr::dcast(geoPtr->getTypes());
    if (typesPtr != NullFC)
       typesPtr->clear();

    GeoIndicesUI32Ptr indicesPtr = GeoIndicesUI32Ptr::dcast(geoPtr->getIndices());
    if (indicesPtr != NullFC)
       indicesPtr->clear();

    geoPtr->setSecondaryColors(NullFC);
    geoPtr->setTexCoords1(NullFC);
    geoPtr->setTexCoords2(NullFC);
    geoPtr->setTexCoords3(NullFC);
    geoPtr->editMFIndexMapping()->clear();

    addToGeom(geoPtr,layoutResult,scale,offset,color);
}

void TextTXFFace::addToGeom(GeometryPtr &geoPtr, const TextLayoutResult &layoutResult, Real32 scale,
                            Vec2f offset, Color3f color)
{
    // We need to keep track of all the text geometry because once the texture is resized, 
    // the texture coordinates for each geometry must be updated as well.
    auto result = _layoutData->textGeoms.insert(geoPtr);
    // If this is a newly added geometry, take a shared ownership of it.
    if (result.second)
        addRefCP(geoPtr);

    beginEditCP(geoPtr, osg::FieldBits::AllFields);

    // cast the field containers down to the needed type and create them
    // when they have the wrong type
    GeoIndicesUI32Ptr  indexPtr = GeoIndicesUI32Ptr::dcast(geoPtr->getIndices());
    GeoPositions3fPtr posPtr = GeoPositions3fPtr::dcast(geoPtr->getPositions());
    GeoNormals3fPtr normalPtr = GeoNormals3fPtr::dcast(geoPtr->getNormals());
    GeoTexCoords2fPtr texPtr = GeoTexCoords2fPtr::dcast(geoPtr->getTexCoords());
    GeoColors3fPtr colorPtr = GeoColors3fPtr::dcast(geoPtr->getColors());
    GeoPLengthsUI32Ptr lensPtr = GeoPLengthsUI32Ptr::dcast(geoPtr->getLengths());
    GeoPTypesUI8Ptr typesPtr = GeoPTypesUI8Ptr::dcast(geoPtr->getTypes());

    // Create color buffer: If Null container AND color is set && we have not potentially added text before
    if ((NullFC == colorPtr) && (color != osg::Color3f(-1,-1,-1)) &&
        ((NullFC == posPtr) && (NullFC == texPtr)) )
    {
       colorPtr = GeoColors3f::create();
       geoPtr->setColors(colorPtr);
    }
    bool use_colors(NullFC != colorPtr);

    if (indexPtr == NullFC)
    {
        indexPtr = GeoIndicesUI32::create();
        geoPtr->setIndices(indexPtr);
    }

    if (posPtr == NullFC)
    {
        posPtr = GeoPositions3f::create();
        geoPtr->setPositions(posPtr);
    }

    if (normalPtr == NullFC)
    {
        normalPtr = GeoNormals3f::create();
        geoPtr->setNormals(normalPtr);
    }

    if (texPtr == NullFC)
    {
        texPtr = GeoTexCoords2f::create();
        geoPtr->setTexCoords(texPtr);
    }

    if (lensPtr == NullFC)
    {
        lensPtr = GeoPLengthsUI32::create();
        geoPtr->setLengths(lensPtr);
    }

    if (typesPtr == NullFC)
    {
        typesPtr = GeoPTypesUI8::create();
        geoPtr->setTypes(typesPtr);
    }

    UInt32 numGlyphs = layoutResult.getNumGlyphs();
    if (numGlyphs == 0)
    {
        endEditCP(geoPtr, osg::FieldBits::AllFields);
        return;
    }

    beginEditCP(indexPtr, GeoIndicesUI32::GeoPropDataFieldMask);
    beginEditCP(posPtr, GeoPositions3f::GeoPropDataFieldMask);
    beginEditCP(normalPtr, GeoNormals3f::GeoPropDataFieldMask);
    beginEditCP(texPtr, GeoTexCoords2f::GeoPropDataFieldMask);
    if(NullFC != colorPtr)
    {  
        beginEditCP(colorPtr, GeoIndicesUI32::GeoPropDataFieldMask); 
    }
    beginEditCP(lensPtr, GeoPLengthsUI32::GeoPropDataFieldMask);
    beginEditCP(typesPtr, GeoPTypesUI8::GeoPropDataFieldMask);

    osg::Vec3f normal(0.f, 0.f, 1.f);        // normal to use for each glyph

    typesPtr->push_back(OSG_GL_TRIANGLES);
    unsigned num_glyphs_added(0);
    for (UInt32 i = 0; i < numGlyphs; ++i)
    {
        TextGlyph::Index glyphIndex = layoutResult.indices[i];
        const TextTXFGlyph &glyph = getTXFGlyph(glyphIndex);
        Real32 width = glyph.getWidth();
        Real32 height = glyph.getHeight();

        // No need to draw invisible glyphs
        if ((width <= 0.f) || (height <= 0.f))
        {
            //printf("osg::TextTXFFace::addToGeom: invisible glyph! id:%d w:%f h:%f\n", glyph.getGlyphIndex(), width, height);
            continue;
        }
        else
        {
            num_glyphs_added += 1;
        }
        UInt32 baseindex = posPtr->size();

        indexPtr->push_back(baseindex);
        indexPtr->push_back(baseindex + 1);
        indexPtr->push_back(baseindex + 2);
        indexPtr->push_back(baseindex);
        indexPtr->push_back(baseindex + 2);
        indexPtr->push_back(baseindex + 3);

        // Calculate coordinates
        Vec2f pos = layoutResult.positions[i];
        Real32 posLeft = (pos.x() * scale) + offset.x();
        Real32 posTop = (pos.y() * scale) + offset.y();
        Real32 posRight = ((pos.x() + width) * scale) + offset.x();
        Real32 posBottom = ((pos.y() - height) * scale) + offset.y();

        // Calculate texture coordinates
        Real32 texCoordLeft = glyph.getTexCoord(TextTXFGlyph::COORD_LEFT);
        Real32 texCoordTop = glyph.getTexCoord(TextTXFGlyph::COORD_TOP);
        Real32 texCoordRight = glyph.getTexCoord(TextTXFGlyph::COORD_RIGHT);
        Real32 texCoordBottom = glyph.getTexCoord(TextTXFGlyph::COORD_BOTTOM);

        // lower left corner
        posPtr->push_back(Vec3f(posLeft, posBottom, 0.f));
        texPtr->push_back(Vec2f(texCoordLeft, texCoordBottom));
        normalPtr->push_back(normal);
        if(use_colors) 
            colorPtr->push_back(color);

        // lower right corner
        posPtr->push_back(Vec3f(posRight, posBottom, 0.f));
        texPtr->push_back(Vec2f(texCoordRight, texCoordBottom));
        normalPtr->push_back(normal);
        if(use_colors) 
            colorPtr->push_back(color);

        // upper right corner
        posPtr->push_back(Vec3f(posRight, posTop, 0.f));
        texPtr->push_back(Vec2f(texCoordRight, texCoordTop));
        normalPtr->push_back(normal);
        if(use_colors) 
            colorPtr->push_back(color);

        // upper left corner
        posPtr->push_back(Vec3f(posLeft, posTop, 0.f));
        texPtr->push_back(Vec2f(texCoordLeft, texCoordTop));
        normalPtr->push_back(normal);
        if(use_colors) 
            colorPtr->push_back(color);
    }

    lensPtr->push_back(num_glyphs_added*6);

    endEditCP(indexPtr, GeoIndicesUI32::GeoPropDataFieldMask);
    endEditCP(typesPtr, GeoPTypesUI8::GeoPropDataFieldMask);
    endEditCP(lensPtr, GeoPLengthsUI32::GeoPropDataFieldMask);
    endEditCP(texPtr, GeoTexCoords2f::GeoPropDataFieldMask);
    endEditCP(normalPtr, GeoNormals3f::GeoPropDataFieldMask);
    endEditCP(posPtr, GeoPositions3f::GeoPropDataFieldMask);

    geoPtr->editMFIndexMapping()->clear();

    if(NullFC != colorPtr)
    { 
        geoPtr->editMFIndexMapping()->push_back(Geometry::MapPosition | Geometry::MapNormal | Geometry::MapTexCoords | Geometry::MapColor);
        endEditCP(colorPtr, GeoIndicesUI32::GeoPropDataFieldMask); 
    }
    else
    {
        geoPtr->editMFIndexMapping()->push_back(Geometry::MapPosition | Geometry::MapNormal | Geometry::MapTexCoords);
    }
    endEditCP(geoPtr, osg::FieldBits::AllFields);
}


//----------------------------------------------------------------------
// Creates a new text geometry
// Author: pdaehne
//----------------------------------------------------------------------
GeometryPtr TextTXFFace::makeGeo(const TextLayoutResult &layoutResult, Real32 scale,
                                 Vec2f offset, Color3f color)
{
    GeometryPtr geo = Geometry::create();
    fillGeo(geo, layoutResult, scale, offset, color);
    return geo;
}


//----------------------------------------------------------------------
// Creates a new node with a text geometry
// Author: pdaehne
//----------------------------------------------------------------------
NodePtr TextTXFFace::makeNode(const TextLayoutResult &layoutResult, Real32 scale,
                              Vec2f offset, Color3f color)
{
    GeometryPtr geo = makeGeo(layoutResult, scale, offset, color);
    NodePtr node = Node::create();
    beginEditCP(node, Node::CoreFieldMask);
    node->setCore(geo);
    endEditCP(node, Node::CoreFieldMask);
    return node;
}


//----------------------------------------------------------------------
// Tries to create a TXF face
// Author: pdaehne
//----------------------------------------------------------------------
TextTXFFace *TextTXFFace::create(const string &family, Style style, const TextTXFParam &param)
{ return TextFaceFactory::the().createTXFFace(family, style, param); }


//----------------------------------------------------------------------
// Reads a long value in network byte order from the input stream
// Author: pdaehne
//----------------------------------------------------------------------
static UInt32 readLong(istream &is, bool swap)
{
    UInt8 bytes[4];
    is.read(reinterpret_cast<istream::char_type*>(bytes), 4);
    return swap ?
        (bytes[3] << 24) |
        (bytes[2] << 16) |
        (bytes[1] << 8) |
        bytes[0]
                :
        (bytes[0] << 24) |
        (bytes[1] << 16) |
        (bytes[2] << 8) |
        bytes[3];
}


//----------------------------------------------------------------------
// Reads a short value in network byte order from the input stream
// Author: pdaehne
//----------------------------------------------------------------------
static UInt16 readShort(istream &is, bool swap)
{
    UInt8 bytes[2];
    is.read(reinterpret_cast<istream::char_type*>(bytes), 2);
    return swap ?
        (bytes[1] << 8) |
        bytes[0]
                :
        (bytes[0] << 8) |
        bytes[1];
}


//----------------------------------------------------------------------
// Reads a TXF face from an input stream
// Author: pdaehne
//----------------------------------------------------------------------
TextTXFFace *TextTXFFace::createFromStream(istream &is, const string &family, Style style)
{
    // Check the magic bytes
    istream::char_type magicBytes[4];
    is.read(magicBytes, 4);
    if ((is.good() == false) || (strncmp(magicBytes, "\xfftxf", 4) != 0))
        return 0;

    // Check endianess
    UInt32 endianness = readLong(is, false);
    bool swap;
    if (endianness == 0x12345678)
        swap = false;
    else if (endianness == 0x78563412)
        swap = true;
    else
        return 0;

    TextTXFFace *face = new TextTXFFace();
    face->_family = family;
    face->_style = style;

    // Read header
    UInt32 format = readLong(is, swap);
    UInt32 textureWidth = readLong(is, swap);
    UInt32 textureHeight = readLong(is, swap);
    Int32 max_ascent = readLong(is, swap);
    Int32 max_descent = readLong(is, swap);
    if (max_descent < 0)
        max_descent = -max_descent;
    UInt32 num_glyphs = readLong(is, swap);
    if (is.good() == false)
    {
        subRefP(face);
        return 0;
    }

    // Determine parameters
    face->_param.size = max_ascent + max_descent;
    face->_param.gap = 0; // There is no way to determine the gap
    face->_param.textureWidth = textureWidth;

    // Determine the scale factor
    face->_scale = 1.f / static_cast<Real32>(max_ascent + max_descent);

    // Determine ascent
    face->_horiAscent = static_cast<Real32>(max_ascent) * face->_scale;
    face->_vertAscent = -0.5f;

    // Determine descent
    face->_horiDescent = static_cast<Real32>(-max_descent) * face->_scale;
    face->_vertDescent = 0.5f;

    // Parse glyph information
    Int32 vertBearingY = -(max_ascent + max_descent) / 20;
    if (vertBearingY > -1)
        vertBearingY = -1;
    Real32 vertAdvanceOffset = static_cast<Real32>(vertBearingY) * 2.f * face->_scale;
    wstring characters;
    characters.reserve(num_glyphs);
    UInt32 i;
    for (i = 0; i < num_glyphs; ++i)
    {
        // Create the new glyph object
        TextTXFGlyph *glyph = new TextTXFGlyph();
        UInt16 glyphIndex = readShort(is, swap);
        characters.append(static_cast<wchar_t>(glyphIndex), 1);
        glyph->_glyphIndex = glyphIndex;
        glyph->_scale = face->_scale;
        glyph->_width = static_cast<unsigned char>(is.get());
        glyph->_height = static_cast<unsigned char>(is.get());
        glyph->_horiBearingX = static_cast<signed char>(is.get());
        glyph->_horiBearingY = static_cast<signed char>(is.get()) + glyph->_height;
        glyph->_horiAdvance = (static_cast<signed char>(is.get())) * face->_scale;
        is.ignore(); // padding
        glyph->_x = readShort(is, swap);
        glyph->_y = readShort(is, swap);

        // There is no vertical layout information in a TXF file,
        // therefore we have to guess reasonable values
        glyph->_vertBearingX = -static_cast<Int32>(glyph->_width >> 1);
        if (glyphIndex == 32)
        {
            glyph->_vertBearingY = -glyph->_horiBearingX;
            glyph->_vertAdvance = -glyph->_horiAdvance;
        }
        else
        {
            glyph->_vertBearingY = vertBearingY;
            glyph->_vertAdvance = -static_cast<Real32>(glyph->_height) * face->_scale + vertAdvanceOffset;
        }

        glyph->calculateCoordinates(textureWidth, textureHeight);
        face->_glyphMap[glyphIndex] = glyph;

        if (is.good() == false)
        {
            subRefP(face);
            return 0;
        }
    }
    face->_param.setCharacters(characters);

    // Create the texture
    face->_texture = Image::create();
    addRefCP(face->_texture);
    beginEditCP(face->_texture, osg::FieldBits::AllFields);
    face->_texture->set(Image::OSG_A_PF, textureWidth, textureHeight);
    face->_texture->clear();

    // Parse texture
    switch (format)
    {
        case 0: // TXF_FORMAT_BYTE
            {
                UInt32 size = textureWidth * textureHeight;
                assert(face->_texture->getSize() == size);
                is.read(reinterpret_cast<istream::char_type*>(
                            face->_texture->editData()), size);
                endEditCP(face->_texture, osg::FieldBits::AllFields);
                if (is.good() == false)
                {
                    subRefP(face);
                    return 0;
                }
            }
            break;
        case 1: // TXF_FORMAT_BITMAP
            {
                UInt32 stride = (textureWidth + 7) >> 3;
                UInt32 size = stride * textureHeight;
                UInt8 *buffer = new UInt8[size];
                is.read(reinterpret_cast<istream::char_type*>(buffer), size);
                if (is.good() == false)
                {
                    delete [] buffer;
                    endEditCP(face->_texture, osg::FieldBits::AllFields);
                    subRefP(face);
                    return 0;
                }
                assert(face->_texture->getSize() == textureWidth * textureHeight);
                UInt8 *dst = face->_texture->editData();
                UInt32 x, y;
                for (y = 0; y < textureHeight; ++y)
                    for (x = 0; x < textureWidth; ++x)
                        dst[y * textureWidth + x] = buffer[y * stride + (x >> 3)] & (1 << (x & 7)) ? 255 : 0;
                delete [] buffer;
                endEditCP(face->_texture, osg::FieldBits::AllFields);
            }
            break;
        default:
            endEditCP(face->_texture, osg::FieldBits::AllFields);
            subRefP(face);
            return 0;
    }

    return face;
}


//----------------------------------------------------------------------
// Reads a TXF face from a file
// Author: pdaehne
//----------------------------------------------------------------------
TextTXFFace *TextTXFFace::createFromFile(const string &filename)
{
    // Open the file
    ifstream is;
    if(osgUseUtf8())
        is.open(Utf8ToUtf16(filename.c_str()).c_str(), ios_base::in | ios_base::binary);
    else
        is.open(filename.c_str(), ios_base::in | ios_base::binary);

    if (is.good() == false)
        return 0;

    // Remove the directory and the suffix from the filename and use the
    // remaining filename as the family name
    string::size_type pos = filename.find_last_of("/\\");
    string family = pos != string::npos ? filename.substr(pos + 1) : filename;
    pos = family.rfind('.');
    if (pos != string::npos)
        family.erase(pos);

    // Parse the file
    TextTXFFace *face = createFromStream(is, family);
    if (face == 0)
        return 0;

    return face;
}


//----------------------------------------------------------------------
// Writes a long value in network byte order to the output stream
// Author: pdaehne
//----------------------------------------------------------------------
static void writeLong(ostream &os, UInt32 value)
{
    UInt8 bytes[4];
    bytes[0] = static_cast<UInt8>((value >> 24) & 0xff);
    bytes[1] = static_cast<UInt8>((value >> 16) & 0xff);
    bytes[2] = static_cast<UInt8>((value >> 8) & 0xff);
    bytes[3] = static_cast<UInt8>(value & 0xff);
    os.write(reinterpret_cast<ostream::char_type*>(bytes), 4);
}


//----------------------------------------------------------------------
// Writes a short value in network byte order to the output stream
// Author: pdaehne
//----------------------------------------------------------------------
static void writeShort(ostream &os, UInt16 value)
{
    UInt8 bytes[2];
    bytes[0] = static_cast<UInt8>((value >> 8) & 0xff);
    bytes[1] = static_cast<UInt8>(value & 0xff);
    os.write(reinterpret_cast<ostream::char_type*>(bytes), 2);
}


//----------------------------------------------------------------------
// Writes a TXF face to an output stream
// Author: pdaehne
//----------------------------------------------------------------------
bool TextTXFFace::writeToStream(ostream &os) const
{
    // Write the magic bytes
    const ostream::char_type *magicBytes = "\xfftxf";
    os.write(magicBytes, 4);

    // Write the header
    assert(_texture != NullFC);
    writeLong(os, 0x12345678); // endianness
    writeLong(os, 0); // format
    writeLong(os, _texture->getWidth());
    writeLong(os, _texture->getHeight());
    writeLong(os, static_cast<UInt32>(_horiAscent / _scale));
    writeLong(os, static_cast<UInt32>(_horiDescent / _scale));
    writeLong(os, _glyphMap.size());
    if (os.good() == false)
        return false;

    // Write glyph information
    GlyphMap::const_iterator it;
    for (it = _glyphMap.begin(); it != _glyphMap.end(); ++it)
    {
        assert(it->second != 0);
        writeShort(os, static_cast<UInt16>(it->second->getGlyphIndex()));
        os.put(it->second->getPixmapWidth());
        os.put(it->second->getPixmapHeight());
        os.put(it->second->getPixmapHoriBearingX());
        os.put(it->second->getPixmapHoriBearingY() - it->second->getPixmapHeight());
        os.put(static_cast<ostream::char_type>(it->second->getHoriAdvance() / _scale));
        os.put(0); // padding
        writeShort(os, it->second->getX());
        writeShort(os, it->second->getY());
        if (os.good() == false)
            return false;
    }

    // Write texture
    assert(_texture->getSize() == static_cast<UInt32>(_texture->getWidth() * _texture->getHeight()));
    os.write(reinterpret_cast<const ostream::char_type*>(_texture->getData()), _texture->getWidth() * _texture->getHeight());

    return os.good();
}


//----------------------------------------------------------------------
// Writes a TXF face to a file
// Author: pdaehne
//----------------------------------------------------------------------
bool TextTXFFace::writeToFile(const string &filename) const
{
    ofstream os(filename.c_str(), ios_base::out | ios_base::trunc | ios_base::binary);
    if (os.good() == false)
        return false;
    return writeToStream(os);
}


//----------------------------------------------------------------------
// Lays out one line of text.
// Author: pdaehne
//----------------------------------------------------------------------
void TextTXFFace::layout(const string &utf8Text,
                         const TextLayoutParam &param,
                         TextLayoutResult &result)
{
    TextFace::layout(utf8Text, param, result);
}


//----------------------------------------------------------------------
// Lays out one line of text
// Author: pdaehne
//----------------------------------------------------------------------
void TextTXFFace::layout(const wstring &text,
                         const TextLayoutParam &param,
                         TextLayoutResult &result)
{
    // First we try to update map and texture
    tryToUpdate(text);

    // Initialize return values
    result.clear();

    // Do the layout depending on the direction
    Vec2f currPos;
    size_t i, len = text.length();
    result.indices.reserve(len);
    result.positions.reserve(len);
    vector<UInt32> spaceIndices;
    bool justify = param.getLength(0) > 0.f;
    for (i = 0; i < len; ++i)
    {
        // Get glyph
        const TextGlyph &glyph = getGlyph(text[i]);
        //if(glyph.getGlyphIndex() == -1)
        //    printf("osg::TextTXFFace::layout no glyph for %d 0x%x\n", text[i], text[i]);
        if ((justify == true) && (text[i] == ' '))
            spaceIndices.push_back(result.indices.size());

        // Calculate position
        Vec2f pos;
        if (param.horizontal == true)
        {
            if (param.leftToRight == true)
            {
                pos = currPos;
                pos[0] += glyph.getHoriBearingX();
                pos[1] += glyph.getHoriBearingY();
                currPos[0] += glyph.getHoriAdvance();
            }
            else // leftToRight == false
            {
                currPos[0] -= glyph.getHoriAdvance();
                pos = currPos;
                pos[0] += glyph.getHoriBearingX();
                pos[1] += glyph.getHoriBearingY();
            }
        }
        else // horizontal == false
        {
            if (param.topToBottom == true)
            {
                pos = currPos;
                pos[0] += glyph.getVertBearingX();
                pos[1] += glyph.getVertBearingY();
                currPos[1] += glyph.getVertAdvance();
            }
            else // topToBottom == false
            {
                currPos[1] -= glyph.getVertAdvance();
                pos = currPos;
                pos[0] += glyph.getVertBearingX();
                pos[1] += glyph.getVertBearingY();
            }
        }

        result.indices.push_back(glyph.getGlyphIndex());
        result.positions.push_back(pos);
    }

    // Justify the line
    if (justify == true)
        justifyLine(param, spaceIndices, currPos, result);

    // Adjust the origin depending on the major and the minor alignment
    adjustLineOrigin(param, currPos, result);

    // Determine text bounds / line bounds
    if (param.horizontal == true)
        result.textBounds.setValues(osgabs(currPos.x()), _horiAscent - _horiDescent);
    else
        result.textBounds.setValues(_vertDescent - _vertAscent, osgabs(currPos.y()));
    result.lineBounds.push_back(result.textBounds);
}


//----------------------------------------------------------------------
// Lays out multiple lines of text
// Author: pdaehne
//----------------------------------------------------------------------
void TextTXFFace::layout(const vector<string> &lines,
                         const TextLayoutParam &param,
                         TextLayoutResult &result)
{
    TextFace::layout(lines, param, result);
}


//----------------------------------------------------------------------
// Lays out multiple lines of text
// Author: pdaehne
//----------------------------------------------------------------------
void TextTXFFace::layout(const vector<wstring> &lines,
                         const TextLayoutParam &param,
                         TextLayoutResult &result)
{
    // Construct a temp long string that contains all chars 
    // and try to update the texture. This saves some overheads.
    wstring allChars;
    for (auto iter = lines.begin(); iter != lines.end(); ++iter)
    {
        allChars.append(*iter);
    }
    tryToUpdate(allChars);
    TextFace::layout(lines, param, result);
}


//----------------------------------------------------------------------
// Calculates the positions of the glyphs on the texture
// Author: pdaehne
//----------------------------------------------------------------------
void TextTXFFace::prepareTexture(const TextTXFParam &param)
{
#if 1
    // Init data
    initLayoutData(param);
    const UInt32 textureWidth = _layoutData->width;

    // Lay out glyphs using the core algorithm.
    std::vector<TextTXFGlyph*> glyphs;
    for (auto iter = param.getCharacters().begin(); iter != param.getCharacters().end(); ++iter)
    {
        auto git = _glyphMap.find(*iter);
        if (_glyphMap.end() != git)
            glyphs.push_back(git->second);
    }
    layoutWithFixedWidth(glyphs, _layoutData);
    const UInt32 textureHeight = _layoutData->height;

    // Create the texture
    if (NullFC == _texture)
    {
        _texture = Image::create();
        addRefCP(_texture);
    }
    beginEditCP(_texture, osg::FieldBits::AllFields);
    {
        _texture->set(Image::OSG_A_PF, textureWidth, textureHeight);
        _texture->clear();
    }
    endEditCP(_texture, osg::FieldBits::AllFields);

    // Normalize the coordinates of all glyphs
    for (auto iter = _glyphMap.begin(); iter != _glyphMap.end(); ++iter)
        iter->second->calculateCoordinates(textureWidth, textureHeight);

#else
    // Sort characters by height and calculate the area necessary
    // to keep all characters
    typedef multimap<UInt32, TextTXFGlyph*, greater<UInt32> > HeightMap;
    HeightMap heightMap;
    UInt32 area = 0, maxWidth = 0;
    GlyphMap::iterator gmIt;
    for (gmIt = _glyphMap.begin(); gmIt != _glyphMap.end(); ++gmIt)
    {
        TextTXFGlyph *glyph = gmIt->second;
        heightMap.insert(HeightMap::value_type(glyph->getPixmapHeight(), glyph));
        if (maxWidth < glyph->getPixmapWidth())
            maxWidth = glyph->getPixmapWidth();
        area += (glyph->getPixmapWidth() + param.gap) * (glyph->getPixmapHeight() + param.gap);
    }

    UInt32 textureSize;
    if (param.textureWidth == 0)
    {
        // Try to make a good guess about the optimal width of the texture
        textureSize = static_cast<UInt32>(ceil(sqrt(static_cast<Real32>(area))));
    }
    else
        textureSize = param.textureWidth;
    maxWidth += param.gap << 1;
    if (textureSize < maxWidth)
        textureSize = maxWidth;
    UInt32 textureWidth = osgnextpower2(textureSize);

    // Calculate the positions of the glyphs in the texture
    HeightMap::iterator hmIt = heightMap.begin();
    UInt32 xpos = param.gap, ypos = param.gap, heightOfRow = 0;
    while (hmIt != heightMap.end())
    {
        HeightMap::iterator hmIt3;
        // Does the next glyph fit into the line?
        if (xpos + hmIt->second->getPixmapWidth() + param.gap > textureWidth)
        {
            // No, so lets try to find another glyph
            HeightMap::iterator hmIt2 = hmIt;
            for (++hmIt2; hmIt2 != heightMap.end(); ++hmIt2)
            {
                if (xpos + hmIt2->second->getPixmapWidth() + param.gap <= textureWidth)
                    break;
            }
            if (hmIt2 == heightMap.end())
            {
                // There is no other glyph
                xpos = param.gap;
                ypos += heightOfRow + param.gap;
                heightOfRow = 0;
                hmIt3 = hmIt;
                ++hmIt;
            }
            else // There is another glyph that fits
                hmIt3 = hmIt2;
        }
        else
        {
            // Yes, the glyph fits into the line
            hmIt3 = hmIt;
            ++hmIt;
        }
        hmIt3->second->_x = xpos;
        hmIt3->second->_y = ypos;
        xpos += hmIt3->second->getPixmapWidth() + param.gap;
        if (hmIt3->second->getPixmapHeight() > heightOfRow)
            heightOfRow = hmIt3->second->getPixmapHeight();
        heightMap.erase(hmIt3);
    }
    ypos += heightOfRow;

    // Calculate the height of the texture
    UInt32 textureHeight = osgnextpower2(static_cast<UInt32>(ypos));

    // Create the texture
    _texture = Image::create();
    addRefCP(_texture);
    beginEditCP(_texture, osg::FieldBits::AllFields);
    {
        _texture->set(Image::OSG_A_PF, textureWidth, textureHeight);
        _texture->clear();
    }
    endEditCP(_texture, osg::FieldBits::AllFields);

    // Calculate the coordinates of all glyphs
    for (gmIt = _glyphMap.begin(); gmIt != _glyphMap.end(); ++gmIt)
        gmIt->second->calculateCoordinates(textureWidth, textureHeight);
#endif
}


//----------------------------------------------------------------------
// Give a chance to update the map and texture with any missing glyphs 
// in the text to be shown.
// Author: jincheng li
//----------------------------------------------------------------------
void TextTXFFace::tryToUpdate(const std::wstring &text)
{
    bool needUpdate = false;
    for (auto iter = text.begin(); iter != text.end(); ++iter)
    {
        if (_glyphMap.find(*iter) == _glyphMap.end())
        {
            needUpdate = true;
            break;
        }
    }

    if (needUpdate)
    {
        // Need to do this check because if createFromStream() is called, initLayoutData()
        // will not be called and layout data will be invalid as well but _texture will not
        // be NullFC. So we disable the ability to add new glyphs in that case.
        if (_layoutData->incremental || (NullFC == _texture))
        {
            const bool doMipMap = (NullFC != _texture)? _texture->getMipMapCount() > 1 : true;
#if 1
            const std::wstring &newChars = text; // not limited to have only new chars.
            TextFaceFactory::the().updateTXFFace(this, newChars);

            if (doMipMap)
            {
                // Because the texture is updated mipmap also needs to be regenerated.
                beginEditCP(_texture, osg::FieldBits::AllFields);
                _texture->createMipmap();
                endEditCP(_texture, osg::FieldBits::AllFields);
            }

            ////_texture->write("C:\\Users\\jinchel\\Desktop\\tmpfiles\\New folder\\chars.tiff");

#else
            TextTXFParam newParam = _param;
            newParam.setCharacters(std::wstring(text).append(_param.getCharacters()));
            TextWIN32TXFFace* newFace = static_cast<TextWIN32TXFFace*>(s_updateBackend.createTXFFace(_family, _style, newParam));
            if (newFace)
            {
                _param = newFace->_param;
                _glyphMap = std::move(newFace->_glyphMap);
                beginEditCP(_texture, osg::FieldBits::AllFields);
                _texture->set(newFace->_texture);
                _texture->createMipmap();
                endEditCP(_texture, osg::FieldBits::AllFields);

                delete newFace;
            }
#endif
        }
    }
}


//----------------------------------------------------------------------
// Initialize the data for calling layoutNewGlyphsInTexture() incrementally.
// It requires _glyphMap to have all glyphs stored.
// Author: jincheng li
//----------------------------------------------------------------------
void TextTXFFace::initLayoutData(const TextTXFParam &param)
{
    // Try to figure out the width of the texture
    UInt32 area = 0, maxWidth = 0;
    for (auto gmIt = _glyphMap.begin(); gmIt != _glyphMap.end(); ++gmIt)
    {
        TextTXFGlyph *glyph = gmIt->second;
        if (maxWidth < glyph->getPixmapWidth())
            maxWidth = glyph->getPixmapWidth();
        area += (glyph->getPixmapWidth() + param.gap) * (glyph->getPixmapHeight() + param.gap);
    }

    UInt32 textureSize;
    if (param.textureWidth == 0)
    {
        // Try to make a good guess about the optimal width of the texture
        textureSize = static_cast<UInt32>(ceil(sqrt(static_cast<Real32>(area))));
    }
    else
        textureSize = param.textureWidth;
    maxWidth += param.gap << 1;
    if (textureSize < maxWidth)
        textureSize = maxWidth;
    UInt32 textureWidth = osgnextpower2(textureSize);

    // And in case there is too few chars used to initialize the texture, make sure the width is no shorter than 1024.
    textureWidth = std::max(textureWidth, UInt32(1024));

    // Initialize layout data
    _layoutData->gap = param.gap;
    _layoutData->width = textureWidth;
    _layoutData->xpos = param.gap;
    _layoutData->ypos = param.gap;
    _layoutData->hrow = 0;
    _layoutData->incremental = true;
}


//----------------------------------------------------------------------
// Lay out new glyphs into the internal texture and adjust the texture 
// if necessary. Glyphs already laid out in the texture will NOT change
// their positions.
// initLayoutData() needs to be called before any use of this method.
// Author: jincheng li
//----------------------------------------------------------------------
void TextTXFFace::layoutNewGlyphsInTexture(const std::vector<TextTXFGlyph*> &newGlyphs)
{
    // Call the core layout algorithm
    layoutWithFixedWidth(newGlyphs, _layoutData);

    // Create the texture on the first time this method is called
    if (NullFC == _texture)
    {
        _texture = Image::create();
        addRefCP(_texture);
        beginEditCP(_texture, osg::FieldBits::AllFields);
        {
            _texture->set(Image::OSG_A_PF, _layoutData->width, _layoutData->height);
            _texture->clear();
        }
        endEditCP(_texture, osg::FieldBits::AllFields);

        printf("TextTXFFace texture resize to %i * %i.\n", _texture->getWidth(), _texture->getHeight()); fflush(stdout);
    }

    // Get the current height of the texture
    const Int32 currHeight = _texture->getHeight();
    // If the current texture is large enough
    if (currHeight >= _layoutData->height)
    {
        // Only need to normalize the coordinates of the new glyphs
        for (auto iter = newGlyphs.begin(); iter != newGlyphs.end(); ++iter)
            (*iter)->calculateCoordinates(_texture->getWidth(), _texture->getHeight());
    } 
    // Otherwise we need to enlarge the texture
    else
    {
        // Determine the new height of the texture.
        // Make it two rows larger than the height needed.
        const Int32 newHeight = _layoutData->height + 2 * (_layoutData->hrow + _layoutData->gap);
        const Int32 oldHeight = _texture->getHeight();

        // Enlarge the texture while keep its old content.
        std::vector<UInt8> oldContent(_texture->getSize(false, false, false));
        memcpy(oldContent.data(), _texture->getData(0, 0, 0), oldContent.size());

        beginEditCP(_texture, osg::FieldBits::AllFields);
        _texture->set(Image::OSG_A_PF, _layoutData->width, newHeight);
        _texture->clear();
        memcpy(_texture->editData(), oldContent.data(), oldContent.size());
        endEditCP(_texture, osg::FieldBits::AllFields);

        printf("TextTXFFace texture resize to %i * %i.\n", _texture->getWidth(), _texture->getHeight()); fflush(stdout);

        // Since the height of the texture changed, all glyphs need to normalize their coordinates
        for (auto iter = _glyphMap.begin(); iter != _glyphMap.end(); ++iter)
            iter->second->calculateCoordinates(_texture->getWidth(), _texture->getHeight());

        // We also need to update the texture coordinates of all the text geometries.
        auto iter = _layoutData->textGeoms.begin();
        while (_layoutData->textGeoms.end() != iter)
        {
            // Since one ref count was increased when it was 
            // created, refcount==1 means it is not used anywhere
            // and can be deleted.
            if ((*iter).getRefCount() == 1)
            {
                subRefCP(*iter);
                iter = _layoutData->textGeoms.erase(iter);
            }
            else
            {
                //beginEditCP(geom, osg::FieldBits::AllFields);
                GeoTexCoords2fPtr texPtr = GeoTexCoords2fPtr::dcast((*iter)->getTexCoords());
                if (NullFC != texPtr)
                {
                    const float scaleV = float(oldHeight) / float(newHeight);
                    beginEditCP(texPtr, osg::FieldBits::AllFields);
                    for (int i=0; i<texPtr->size(); ++i)
                    {
                        Vec2f uv = texPtr->getValue(i);
                        uv[1] *= scaleV;
                        texPtr->setValue(uv, i);
                    }
                    endEditCP(texPtr, osg::FieldBits::AllFields);
                }
                //endEditCP(geom, osg::FieldBits::AllFields);
                ++iter;
            }
        }
    }
}

//----------------------------------------------------------------------
// Implements the core layout algorithm. It adds a list of new glyphs to 
// a texture with fixed width and gap value. It will increase the texture
// height as necessary as one of the outputs.
// Author: jincheng li
//----------------------------------------------------------------------
void TextTXFFace::layoutWithFixedWidth(const std::vector<TextTXFGlyph*> &gls, LayoutData *data)
{
    // make a copy of the input vector
    std::vector<TextTXFGlyph*> glyphs(gls);

    // Sort characters by height in ascending order
    std::sort(glyphs.begin(), glyphs.end(), 
        [](const TextTXFGlyph *g1, const TextTXFGlyph *g2)
    {
        return g1->getPixmapHeight() < g2->getPixmapHeight();
    });

    // Loop all the new glyphs
    while (!glyphs.empty())
    {
        // Mark the one with currently the largest height as the current glyph.
        auto gIter = glyphs.rbegin();

        // If this glyph does not fit into the current line
        if (data->xpos + data->gap + (*gIter)->getPixmapWidth() > data->width)
        {
            // Try to find another glyph that fits
            auto aIter = gIter+1;
            for (; aIter != glyphs.rend(); ++aIter)
            {
                if (data->xpos + data->gap + (*aIter)->getPixmapWidth() <= data->width)
                    break;
            }
            // No other glyph can fit into the current line. So start a new line.
            if (glyphs.rend() == aIter)
            {
                data->xpos  = data->gap;
                data->ypos += data->hrow + data->gap;
                data->hrow  = 0;
            } 
            // Find a new glyph that fits and mark it as the current.
            else
                gIter = aIter;
        } 
        
        // Record the current position into the current glyph
        (*gIter)->_x = data->xpos;
        (*gIter)->_y = data->ypos;

        // Move the x cursor and update the record of the height of the current row.
        data->xpos += (*gIter)->getPixmapWidth() + data->gap;
        if ((*gIter)->getPixmapHeight() > data->hrow)
            data->hrow = (*gIter)->getPixmapHeight();

        // Remove the current glyph.
        glyphs.erase((gIter+1).base());
    }

    // Calculate the total height of the texture.
    data->height = data->ypos + data->hrow;
    //data->height = osgnextpower2(static_cast<UInt32>(data->height));
}


OSG_END_NAMESPACE


/*------------------------------------------------------------------------*/
/*                              cvs id's                                  */

#ifdef OSG_SGI_CC
#pragma set woff 1174
#endif

#ifdef OSG_LINUX_ICC
#pragma warning( disable : 177 )
#endif

namespace
{
    static osg::Char8 cvsid_cpp[] = "@(#)$Id: OSGTextTXFFace.cpp,v 1.10 2008/06/09 07:30:42 vossg Exp $";
    static osg::Char8 cvsid_hpp[] = OSGTEXTTXFFACE_HEADER_CVSID;
    static osg::Char8 cvsid_inl[] = OSGTEXTTXFFACE_INLINE_CVSID;
}

#ifdef __sgi
#pragma reset woff 1174
#endif
